﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using EffectCombiner.Primitives.Generation.Extensions;
using EffectCombiner.Primitives.Generation.Usage;
using Workflow.Core;
using ShaderGenerator.GLSL;
using EffectDefinitions;
using System.Globalization;

namespace EffectCombiner.Primitives.Generation
{
    public class EffectInputPlug : InputPlug<PlugValue>
    {
        public EffectBlockElementBase BlockElement { get; private set; }
        public Plug BlockDefinitionPlug { get; private set; }

        public EffectInputPlug(Plug blockDefinitionPlug, EffectBlockElementBase blockElement)
        {
            if (blockElement == null)
                throw new ArgumentNullException("blockElement");
            if (blockDefinitionPlug == null)
                throw new ArgumentNullException("blockDefinitionPlug");

            BlockElement = blockElement;
            BlockDefinitionPlug = blockDefinitionPlug;
            Name = string.Format("{0} {1}", blockDefinitionPlug.Type, blockDefinitionPlug.Target);
        }

        public override string ToString()
        {
            return Name + (RemoteOutputPlug != null ? " (plugged)" : string.Empty);
        }
    }

    public class EffectOutputPlug : OutputPlug<PlugValue>
    {
        public EffectBlockElementBase BlockElement { get; private set; }
        public Plug Plug { get; private set; }

        public EffectOutputPlug(Plug plug, EffectBlockElementBase blockElement)
        {
            if (blockElement == null)
                throw new ArgumentNullException("blockElement");
            if (plug == null)
                throw new ArgumentNullException("plug");

            BlockElement = blockElement;
            Plug = plug;
            Name = string.Format("{0} {1}", plug.Type, plug.Target);
        }

        public override string ToString()
        {
            return Name;
        }
    }

    public class EffectWorkflowItem : WorkflowItem<PlugValue>
    {
        public EffectBlockElementBase BlockElement { get; private set; }

        private GlslGenerationContext glslGenerationContext;

        public EffectWorkflowItem(EffectBlockElementBase blockElement)
        {
            if (blockElement == null)
                throw new ArgumentNullException("blockElement");

            BlockElement = blockElement;
        }

        public void SetGenerationContext(GlslGenerationContext context)
        {
            if (context == null)
                throw new ArgumentNullException("context");
            glslGenerationContext = context;
        }

        protected override void OnInputsPolled()
        {
            base.OnInputsPolled();

            if (glslGenerationContext == null)
                return;

            if (BlockElement is ConstantBlockElement)
            {
                GenerateConstant((ConstantBlockElement)BlockElement);
                return;
            }

            if (BlockElement is CommentBlockElement)
            {
                GenerateComment((CommentBlockElement)BlockElement);
                return;
            }

            ShaderTyping.FunctionDefinition funcDef;

            if (BlockElement is PolymorphicBlockElement)
            {
                funcDef = ((PolymorphicBlockElement)BlockElement).UsableFunctionDefinitions[0];
            }
            else
            {
                funcDef = BlockElement.BlockDefinition.CurrentFunctionDefinition;
            }

            var inputPlugs = BlockElement.WorkflowItem.InputPlugs
                .Cast<EffectInputPlug>()
                .ToArray();

            var possibleInputPlugs = DeterminePossibleInputPlugs(inputPlugs);

            var inputValues = new List<PlugValue>();

            var inputParams = funcDef.Parameters.Where(x => x.IsAnInput).ToArray();
            if (IsGraphInput == false)
            {
                var plugInputValues = GetInputValues();

                foreach (var param in inputParams)
                {
                    var plugs = possibleInputPlugs
                        .Where(p => p.BlockDefinitionPlug.BaseTarget == param.Name)
                        .ToArray();

                    if (plugs.Length == 0)
                    {
                        var plug = inputPlugs.FirstOrDefault(p => p.BlockDefinitionPlug.Target == param.Name);
                        inputValues.Add(CreateInputValue(plugInputValues, plug));
                    }
                    else
                    {
                        inputValues.AddRange(plugs.Select(p => CreateInputValue(plugInputValues, p)));
                    }
                }
            }

            var outputValues = new GlslGenerator(glslGenerationContext, Identifier).ProduceOutputs(funcDef, inputValues.ToArray());

            var outputParams = funcDef.Parameters.Where(p => p.IsAnOutput).ToList();
            if (funcDef.ReturnType.IsVoid == false)
                outputParams.Insert(0, funcDef.GetReturnAsParameter());

            for (var i = 0; i < outputValues.Length; i++)
            {
                var param = outputParams[i];

                var plugs = OutputPlugs
                    .Cast<EffectOutputPlug>()
                    .Where(p => p.Plug.BaseTarget == param.Name);

                foreach (var plug in plugs)
                {
                    var fields = PlugValue.GetTargetFields(plug.Plug.Target);
                    var expression = string.Format("{0}{1}", outputValues[i].Target, fields);

                    SetOutputValue(plug, new PlugValue(
                        plug.Plug.Type,
                        expression,
                        outputValues[i].Target));
                }
            }
        }

        private List<EffectInputPlug> DeterminePossibleInputPlugs(EffectInputPlug[] inputPlugs)
        {
            var groupedPossibleInputPlugs = inputPlugs
                .Where(p => p.RemoteOutputPlug != null) // only the connected ones
                .GroupBy(p => new // group by BaseTarget and by Type (group is done purely by BaseTarget, but Type is useful for future)
                {
                    p.BlockDefinitionPlug.BaseTarget,
                    Type = p.BlockDefinitionPlug is SubPlug
                        ? ((SubPlug) p.BlockDefinitionPlug).RelatedPlug.Type
                        : p.BlockDefinitionPlug.Type,
                })
                .ToArray();

            var possibleInputPlugs = new List<EffectInputPlug>();

            foreach (var group in groupedPossibleInputPlugs)
            {
                // first make sur to add the found connected plugs
                possibleInputPlugs.AddRange(group);

                // extract plugs where their BlockDefinition plug is a SubPlug
                var subPlugged = group
                    .Where(p => p.BlockDefinitionPlug is SubPlug)
                    .ToArray();

                if (subPlugged.Length == 0)
                    continue; // if no SubPlugs, no need to go any further

                var shaderType = GlslTypingUtility.GetShaderTypeFromGlslTypeName(group.Key.Type.TypeString);
                var dimX = GlslTypingUtility.GetDimensionX(shaderType);
                var dimY = GlslTypingUtility.GetDimensionY(shaderType);

                // compute the number of fields of the type
                var fieldCounts = dimX * dimY;
                var fieldSlots = new bool[fieldCounts];

                foreach (var plug in subPlugged)
                {
                    // iterate over the field indices of the SubPlug
                    foreach (var idx in ((SubPlug)plug.BlockDefinitionPlug).FieldIndices)
                        fieldSlots[idx] = true; // and set the slot as occupied
                }

                // check if all slot are occupied
                if (fieldSlots.All(s => s))
                    continue; // early break, all required plugs are already there

                var allSubPlugged = inputPlugs
                    .Where(p => p.BlockDefinitionPlug is SubPlug) // the ones with SubPlug
                    .Where(p => p.BlockDefinitionPlug.BaseTarget == group.Key.BaseTarget) // that match the parameter name
                    .ToArray();

                foreach (var plug in allSubPlugged)
                {
                    // get the SubPlug field indices
                    var indices = ((SubPlug)plug.BlockDefinitionPlug).FieldIndices;

                    // check whether the SubPlug can fit or not in the available slots
                    if (indices.All(i => fieldSlots[i] == false))
                    {
                        // add the plug
                        possibleInputPlugs.Add(plug);

                        // and mark the slots as occupied
                        foreach (var idx in indices)
                            fieldSlots[idx] = true;
                    }

                    // check whether all slots are occupied
                    if (fieldSlots.All(s => s))
                        break; // early break, all required plugs are already there
                }

                // ensure that all slots are occupied (if any one is not occupied, then raise an exception)
                if (fieldSlots.Any(s => s == false))
                {
                    var blockName = BlockElement.BlockDefinition.Name;
                    var blockGuid = BlockElement.BlockDefinition.Guid;
                    var funcName = BlockElement.BlockDefinition.CurrentFunctionDefinition.Name;
                    var paramName = group.Key.BaseTarget;
                    var typeName = group.Key.Type.TypeString;
                    var slotsText = string.Join(", ", from f in fieldSlots
                                                      where f == false
                                                      select f.ToString(CultureInfo.InvariantCulture));

                    throw new InvalidOperationException(string.Format(Messages.EXCEPTION_FIELDS_NOT_SATISFIED,
                        blockName, blockGuid, funcName, paramName, typeName, slotsText));
                }
            }

            return possibleInputPlugs;
        }

        private static PlugValue CreateInputValue(PlugValue[] plugInputValues, EffectInputPlug plug)
        {
            var ipv = plugInputValues[plug.Index];

            string expression;
            if (ipv != null)
                expression = ipv.Expression;
            else
            {
                var shaderType = GlslTypingUtility.GetShaderTypeFromGlslTypeName(plug.BlockDefinitionPlug.Type.TypeString);
                expression = GlslTypingUtility.GetValueExpression(shaderType, plug.BlockDefinitionPlug.DefaultValues);
            }

            return new PlugValue(
                plug.BlockDefinitionPlug.Type,
                expression,
                plug.BlockDefinitionPlug.Target);
        }

        private void GenerateConstant(ConstantBlockElement blockElement)
        {
            var outputPlugs = BlockElement.BlockDefinition.OutputPlugs;

            var shaderType = blockElement.BlockDefinition.OutputPlugs[0].Type.TypeString;
            var glslShaderType = GlslTypingUtility.GetShaderTypeFromGlslTypeName(shaderType);
            var expression = GlslTypingUtility.GetValueExpression(glslShaderType, blockElement.Values);

            for (var i = 0; i < outputPlugs.Length; i++)
                SetOutputValue(i, new PlugValue(outputPlugs[i].Type, expression, outputPlugs[i].Target));
        }

        private void GenerateComment(CommentBlockElement blockElement)
        {
            var outputPlugs = BlockElement.BlockDefinition.OutputPlugs;

            var shaderType = blockElement.BlockDefinition.OutputPlugs[0].Type.TypeString;
            var glslShaderType = GlslTypingUtility.GetShaderTypeFromGlslTypeName(shaderType);
            var expression = GlslTypingUtility.GetValueExpression(glslShaderType, blockElement.Values);

            for (var i = 0; i < outputPlugs.Length; i++)
                SetOutputValue(i, new PlugValue(outputPlugs[i].Type, expression, outputPlugs[i].Target));
        }

        protected override bool AreAllInputPlugsConnected()
        {
            var blockDef = BlockElement.BlockDefinition as EffectBlockDefinition;
            if (blockDef == null)
                return base.AreAllInputPlugsConnected();

            var groups = InputPlugs.GetGroupedEffectInputPlugs(blockDef.InputPlugs);
            if (groups.Length == 0)
                return base.AreAllInputPlugsConnected();

            if (blockDef.CurrentFunctionDefinition == null)
                return base.AreAllInputPlugsConnected();

            var inputParameters = blockDef.CurrentFunctionDefinition.Parameters
                .Where(p => p.IsAnInput)
                .Cast<ParameterDefinition>()
                .ToArray();

            var i = -1;
            foreach (var group in groups)
            {
                i++;

                var effectPlug = group.Plug as EffectInputPlug;
                if (effectPlug == null)
                {
                    // improbable case
                    return false; // should throw an exception ?
                }


                if (group.SubPlugs.Length == 0)
                {
                    if (effectPlug.BlockDefinitionPlug.DefaultValues == null)
                    {
                        if (group.Plug.RemoteOutputPlug == null)
                            return false;
                    }
                    continue;
                }

                if (string.IsNullOrWhiteSpace(inputParameters[i].Usage))
                {
                    // improbable case
                    return false; // should throw an exception ?
                }

                var usageDescriptor = UsageManager.GetUsageByName(inputParameters[i].Usage);
                if (usageDescriptor == null)
                    continue; // should throw an exception ?

                var effectSubPlugs = group.SubPlugs.OfType<EffectInputPlug>().ToArray();
                if (group.SubPlugs.Length != effectSubPlugs.Length)
                {
                    // improbable case
                    return false; // should throw an exception ?
                }

                if (usageDescriptor.AreInputsProperlyConnected(effectPlug, effectSubPlugs) == false)
                    return false;
            }

            return true;
        }

        public override string ToString()
        {
            return BlockElement.BlockDefinition.Name;
        }
    }
}
