﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Drawing;
using System.Linq;
using Blocks.Core;
using EffectDefinitions;

namespace EffectCombiner.Primitives.Generation
{
    public abstract class EffectBlockElementBase : BlockElementBase
    {
        public EffectWorkflowItem WorkflowItem { get; private set; }

        public BlockRenderInfo BlockRenderInfo { get; private set; }
        public IBlockHitTester BlockHitTester { get; private set; }

        protected EffectBlockElementBase(
            BlockDefinition blockDefinition,
            BlockRenderInfo blockRenderInfo)
            : base(blockDefinition)
        {
            if (blockRenderInfo == null)
                throw new ArgumentNullException("blockRenderInfo");

            BlockRenderInfo = blockRenderInfo;

            WorkflowItem = new EffectWorkflowItem(this);

            // blockDefinition.Changed += make sure to recompute the block size and force re-render
        }

        public void SetHitTester(IBlockHitTester blockHitTester)
        {
            if (BlockHitTester != null)
                throw new ArgumentException(); // TODO: message

            if (blockHitTester == null)
                throw new ArgumentNullException("blockHitTester");

            BlockHitTester = blockHitTester;
        }

        public bool SetComputedSize(double width, double height)
        {
            return Resize(width, height);
        }

        public virtual void OnBeforeRendering()
        {
            BlockRenderInfo.IsGraphInput = WorkflowItem.IsGraphInput;
            BlockRenderInfo.IsGraphOutput = WorkflowItem.IsGraphOutput;
            BlockRenderInfo.PluggedInputs = WorkflowItem.InputPlugs.Select(x => x.RemoteOutputPlug != null).ToArray();
            BlockRenderInfo.PluggedOutputs = WorkflowItem.OutputPlugs.Select(x => x.RemoteInputPlugs.Any()).ToArray();
        }

        public virtual void OnAfterRendering()
        {
        }

        public virtual bool HitTest(double x, double y)
        {
            return BlockHitTester.HitTest(x - Left, y - Top);
        }

        public virtual PlugHitTestResult PlugHitTest(double x, double y, out int plugIndex)
        {
            return BlockHitTester.PlugHitTest(x - Left, y - Top, out plugIndex);
        }

        public EffectBlockElementBase[] GetInputBlocks()
        {
            var workflows = WorkflowItem.GetInputWorkflowItems();

            if (workflows == null)
                return new EffectBlockElementBase[0];

            return workflows
                .Where(w => w != null)
                .Cast<EffectWorkflowItem>()
                .Select(w => w.BlockElement)
                .ToArray();
        }

        public RectangleF GetBounds()
        {
            return new RectangleF((float)Left, (float)Top, (float)Width, (float)Height);
        }
    }
}
