﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using Blocks.Core;
using EffectCombiner.Primitives.Blocks;
using EffectCombiner.Primitives.Generation;
using Renderer2D.Core;
using Globals = EffectCombiner.Primitives.Globals;

namespace EffectCombiner.Editor
{
    public class WorkspaceManager : IWorkspaceManager
    {
        private readonly RendererManager rendererManager;

        public CompositionBlockManager BlockManager { get; private set; }
        public MouseInputManager MouseInputManager { get; private set; }
        public KeyboardInputManager KeyboardInputManager { get; private set; }
        public WorkspaceRendering WorkspaceRendering { get; private set; }
        public PreviewController PreviewController { get; private set; }

        public IPoint WorkspacePosition { get; private set; }
        public void UpdateWorkspacePosition(Point point)
        {
            WorkspacePosition = point;
        }

        public ISize ViewportSize { get; private set; }

        public double Zoom { get { return zoomMatrix.M11; } }
        private IMatrix zoomMatrix = Matrix.Identity;

        public void ZoomAt(double deltaZoom, double x, double y)
        {
            zoomMatrix = zoomMatrix.Multiply(Matrix.Scale(deltaZoom, deltaZoom, x, y));
        }

        public WorkspaceManager(RendererManager rendererManager)
        {
            if (rendererManager == null)
                throw new ArgumentNullException("rendererManager");

            this.WorkspacePosition = new Point();

            this.rendererManager = rendererManager;

            this.BlockManager = new CompositionBlockManager(this);
            this.WorkspaceRendering = new WorkspaceRendering(this);
            this.MouseInputManager = new MouseInputManager(this);
            this.KeyboardInputManager = new KeyboardInputManager(this);
            this.PreviewController = new PreviewController(this);

            this.SetupBlockManager();

            this.rendererManager.RenderSurfaceCreated += this.RenderSurfaceCreated;
            this.BlockManager.SelectionChanged += this.BlockManager_SelectionChanged;

            this.PreviewController.ConnectionChanged += this.PreviewerConnectionChanged;

            var uniformManager = Primitives.Generation.Globals.UniformManager;

            uniformManager.UniformDataChanged += this.UniformDataChanged;
            uniformManager.UniformDataInitialized += this.UniformDataInitialized;
        }

        /// <summary>
        /// ここで定数パネルを出したり引っ込めたりする。
        /// </summary>
        /// <param name="sender">送信元</param>
        /// <param name="e">イベント</param>
        private void BlockManager_SelectionChanged(object sender, EventArgs e)
        {
            IBlockElementEditPanel targetEditPanel = null;
            BlockElementBase targetBlockElement = null;

            IBlockElementEditPanel[] editPanels = new IBlockElementEditPanel[] { Globals.CommentTextPanel, Globals.ConstantValuePanel, Globals.EffectBlockElementPanel };

            // 編集パネルと編集対象のブロックエレメントをどれにするか決める
            foreach (var panel in editPanels)
            {
                // 編集可能なブロックエレメントを取得
                var targetBlockElements = this.BlockManager.SelectedItems.Where(x => panel.CanEditBlockElement(x)).ToArray();

                // 編集可能なブロックエレメントが 2 つ以上あるときは編集不可
                if (targetBlockElements.Length >= 2)
                {
                    break;
                }

                // 編集可能なブロックエレメントが 1 つだけあるときはそれを編集対象にする
                if (targetBlockElements.Length == 1)
                {
                    targetEditPanel = panel;
                    targetBlockElement = targetBlockElements[0];
                    break;
                }
            }

            // 編集パネルにブロックエレメントを設定する
            foreach (var panel in editPanels)
            {
                if (panel == targetEditPanel)
                {
                    panel.SetBlockElement(targetBlockElement);
                }
                else
                {
                    panel.SetBlockElement(null);
                }
            }
        }

        public void UpdateTransform()
        {
            var translationMatrix = Matrix.Translation(
                WorkspacePosition.X,
                WorkspacePosition.Y);

            var rotationMatrix = Matrix.Rotation(20.0);

            var matrix = translationMatrix
                //.Multiply(rotationMatrix)
                .Multiply(zoomMatrix);

            if (rendererManager.RenderSurface != null)
                rendererManager.RenderSurface.Renderer.Transform = matrix;
        }

        private void UpdateViewportSize()
        {
            if (rendererManager.RenderSurface != null)
                ViewportSize = new Size(rendererManager.RenderSurface.Width, rendererManager.RenderSurface.Height);
        }

        public void InvalidateRender()
        {
            if (rendererManager.RenderSurface != null)
                rendererManager.RenderSurface.Invalidate();
        }

        private void SetupBlockManager()
        {
            this.BlockManager.RequestRender += this.BlockManagerRequestRendering;

            this.BlockManager.BlockAdded += this.BlockManagerBlockAdded;
            this.BlockManager.BlockReplaced += this.BlockManagerBlockReplaced;

            this.BlockManager.SelectionChanged += this.BlockManagerSelectionChanged;
            this.BlockManager.DraggingChanged += this.BlockManagerDraggedSelectionChanged;

            this.BlockManager.BlockElementPositionChanged += this.BlockElementPositionChanged;
            this.BlockManager.BlockElementSizeChanged += this.BlockElementSizeChanged;
            this.BlockManager.BlockElementZOrderChanged += this.BlockElementZOrderChanged;

            this.BlockManager.BlockElementConstantValuesChanged += this.BlockElementConstantValueChanged;
        }

        private void RenderSurfaceCreated(object sender, EventArgs e)
        {
            rendererManager.RenderSurface.SetWorkspaceManager(this);
            UpdateViewportSize();

            rendererManager.RenderSurface.Disposed += RenderSurfaceDisposed;
            rendererManager.RenderSurface.Resize += RenderSurfaceResize;

            this.PreviewController.Renderer = rendererManager.RenderSurface.Renderer;
        }

        private void RenderSurfaceDisposed(object sender, EventArgs e)
        {
            rendererManager.RenderSurface.Disposed -= RenderSurfaceDisposed;
            rendererManager.RenderSurface.Resize -= RenderSurfaceResize;

            this.PreviewController.Renderer = null;
        }

        private void RenderSurfaceResize(object sender, EventArgs e)
        {
            UpdateViewportSize();
        }

        /// <summary>
        /// 描画が要求されたときの処理を行います。
        /// </summary>
        /// <param name="sender">イベントの発生元</param>
        /// <param name="e">イベント情報</param>
        private void BlockManagerRequestRendering(object sender, EventArgs e)
        {
            this.InvalidateRender();
        }

        /// <summary>
        /// ブロックエレメントが追加されたときの処理を行います。
        /// </summary>
        /// <param name="sender">イベントの発生元</param>
        /// <param name="e">イベント情報</param>
        private void BlockManagerBlockAdded(object sender, BlockElementEventArgs e)
        {
            if (e.BlockElement is EffectBlockElementBase)
            {
                this.PreviewController.UpdatePreviewImage((EffectBlockElementBase)e.BlockElement, false);
            }
        }

        /// <summary>
        /// ブロックエレメントが置換されたときの処理を行います。
        /// </summary>
        /// <param name="sender">イベントの発生元</param>
        /// <param name="e">イベント情報</param>
        private void BlockManagerBlockReplaced(object sender, BlockElementReplacedEventArgs e)
        {
            if (e.NewBlockElement is EffectBlockElementBase)
            {
                this.PreviewController.UpdatePreviewImage((EffectBlockElementBase)e.NewBlockElement, true);
            }
        }

        /// <summary>
        /// ブロックエレメントの位置が変わったときの処理を行います。
        /// </summary>
        /// <param name="sender">イベントの発生元</param>
        /// <param name="e">イベント情報</param>
        private void BlockElementPositionChanged(object sender, BlockElementPositionChangedEventArgs e)
        {
            this.InvalidateRender();
        }

        /// <summary>
        /// ブロックエレメントのサイズが変わったときの処理を行います。
        /// </summary>
        /// <param name="sender">イベントの発生元</param>
        /// <param name="e">イベント情報</param>
        private void BlockElementSizeChanged(object sender, BlockElementSizeChangedEventArgs e)
        {
            this.InvalidateRender();
        }

        /// <summary>
        /// ブロックエレメントの Z オーダーが変わったときの処理を行います。
        /// </summary>
        /// <param name="sender">イベントの発生元</param>
        /// <param name="e">イベント情報</param>
        private void BlockElementZOrderChanged(object sender, BlockElementZOrderChangedEventArgs e)
        {
            this.InvalidateRender();
        }

        /// <summary>
        /// ブロックエレメントの選択状態が変わったときの処理を行います。
        /// </summary>
        /// <param name="sender">イベントの発生元</param>
        /// <param name="e">イベント情報</param>
        private void BlockManagerSelectionChanged(object sender, EventArgs e)
        {
            this.InvalidateRender();
        }

        /// <summary>
        /// ブロックエレメントのドラッグ状態が変わったときの処理を行います。
        /// </summary>
        /// <param name="sender">イベントの発生元</param>
        /// <param name="e">イベント情報</param>
        private void BlockManagerDraggedSelectionChanged(object sender, EventArgs e)
        {
            this.InvalidateRender();
        }

        /// <summary>
        /// ブロックエレメントの constant 値が変更されたときの処理を行います。
        /// </summary>
        /// <param name="sender">イベントの発生元</param>
        /// <param name="e">イベント情報</param>
        private void BlockElementConstantValueChanged(object sender, ConstantValuesChangedEventArgs e)
        {
            this.PreviewController.UpdatePreviewImage(e.BlockElement, true);

            this.InvalidateRender();
        }

        /// <summary>
        /// プレビューアとの接続状態が変わったときの処理を行います。
        /// </summary>
        /// <param name="sender">イベントの発生元</param>
        /// <param name="e">イベント情報</param>
        private void PreviewerConnectionChanged(object sender, EventArgs e)
        {
            if (this.PreviewController.IsConnected == false)
            {
                return;
            }

            // Previewer に uniform 変数値を設定
            this.PreviewController.SetUniformValues(Globals.WorkspaceManager.BlockManager, Primitives.Generation.Globals.UniformManager.Uniforms.ToArray());

            // 最終出力プレビューを更新
            this.PreviewController.UpdateOutputPreview(this.BlockManager);
        }

        /// <summary>
        /// uniform 変数の値が変更されたときの処理を行います。
        /// </summary>
        /// <param name="sender">イベントの発生元</param>
        /// <param name="e">イベント情報</param>
        private void UniformDataChanged(object sender, UniformDataChangedEventArgs e)
        {
            // ブロックエレメントに uniform 変数値を設定
            this.UpdateUniformBlockElementData(e.UniformName, e.NewValue);

            var value = new KeyValuePair<string, UniformData>(e.UniformName, e.NewValue);

            // Previewer に uniform 変数値を設定
            this.PreviewController.SetUniformValues(this.BlockManager, new[] { value });
        }

        /// <summary>
        /// uniform 変数の値が初期化されたときの処理を行います。
        /// </summary>
        /// <param name="sender">イベントの発生元</param>
        /// <param name="e">イベント情報</param>
        private void UniformDataInitialized(object sender, EventArgs e)
        {
            var uniformManager = Primitives.Generation.Globals.UniformManager;

            // ブロックエレメントに uniform 変数値を設定
            foreach (var uniformItem in uniformManager.Uniforms)
            {
                this.UpdateUniformBlockElementData(uniformItem.Key, uniformItem.Value);
            }

            // Previewer に uniform 変数値を設定
            this.PreviewController.SetUniformValues(Globals.WorkspaceManager.BlockManager, uniformManager.Uniforms.ToArray());

            // 最終出力プレビューを更新
            this.PreviewController.UpdateOutputPreview(this.BlockManager);
        }

        /// <summary>
        /// uniform 変数値をブロックエレメントに適用します。
        /// </summary>
        /// <param name="name">uniform 変数名</param>
        /// <param name="value">uniform 変数値</param>
        private void UpdateUniformBlockElementData(string name, UniformData value)
        {
            // 指定された uniform 変数を使用している全てのブロックエレメントについて
            // uniform 変数値をセットする
            foreach (var blockElement in this.BlockManager.BlockElements)
            {
                var userElement = blockElement as UserDefinitionBlockElement;

                if (userElement == null)
                {
                    continue;
                }

                var uniformDefinition = userElement.BlockDefinition.Uniform;

                if (uniformDefinition == null)
                {
                    continue;
                }

                if (uniformDefinition.Name == name)
                {
                    userElement.UniformValue = value;
                    userElement.UpdateData();
                }
            }
        }
    }
}
