﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Drawing;
using System.Linq;
using System.Text;
using System.Windows.Forms;
using System.Threading;
using EffectCombiner.Core.Extensions;
using System.Threading.Tasks;
using EffectCombiner.Core;
using EffectCombiner.Primitives;

namespace EffectCombiner.Editor
{
    public partial class WaitForm : Form
    {
        private readonly CancellationTokenSource cancellationTokenSource;
        private readonly ProgressionReporter progressionReporter;
        private readonly EventWaitHandle ready;

        private readonly SynchronizationContext syncContext;

        public WaitForm()
        {
            InitializeComponent();

            syncContext = SynchronizationContext.Current;

            var localizationSubscription = Globals.Localization.RegisterLocalization(() =>
                {
                    Text = Localization.Controls.FORM_WAIT_PLEASE_WAIT;
                    btnCancel.Text = Localization.Controls.BUTTON_CANCEL;
                });

            FormClosed += (ss, ee) => localizationSubscription.Dispose();

            Disposed += FormDisposed;

            btnCancel.Enabled = false;
        }

        public WaitForm(CancellationTokenSource cancellationTokenSource, ProgressionReporter progressionReporter, EventWaitHandle ready)
            : this()
        {
            if (ready == null)
                throw new ArgumentNullException("ready");

            this.cancellationTokenSource = cancellationTokenSource;
            this.progressionReporter = progressionReporter;
            this.ready = ready;

            if (cancellationTokenSource != null && cancellationTokenSource.Token.CanBeCanceled)
            {
                btnCancel.Enabled = true;
                btnCancel.Click += CancelButtonClick;
            }

            if (progressionReporter != null)
                progressionReporter.ProgressionChanged += ProgressionChanged;
        }

        protected override void OnHandleCreated(EventArgs e)
        {
            base.OnHandleCreated(e);
            ready.Set();
        }

        protected override void OnTextChanged(EventArgs e)
        {
            base.OnTextChanged(e);
            if (progressionReporter == null)
                lblInfo.Text = Text;
        }

        public void SafeClose()
        {
            syncContext.RunSync(Close);
        }

        private void FormDisposed(object sender, EventArgs e)
        {
            btnCancel.Click -= CancelButtonClick;

            if (progressionReporter != null)
                progressionReporter.ProgressionChanged -= ProgressionChanged;
        }

        private void CancelButtonClick(object sender, EventArgs e)
        {
            if (progressionReporter != null)
                progressionReporter.ProgressionChanged -= ProgressionChanged;
            btnCancel.Enabled = false;
            lblInfo.Text = Localization.Controls.CANCELING;

            cancellationTokenSource.Cancel();
        }

        private void ProgressionChanged(object sender, EventArgs e)
        {
            UpdateProgression();
        }

        private void UpdateProgression()
        {
            syncContext.RunSync(() =>
                {
                    if (progressionReporter.Maximum > 0)
                        lblProgress.Width = progressionReporter.Current * pnlProgress.Width / progressionReporter.Maximum;
                    lblInfo.Text = Compact(progressionReporter.Message, lblInfo, EllipsisFormat.Middle);
                });
        }

        #region Found on CodeProject

        [Flags]
        public enum EllipsisFormat
        {
            // Text is not modified.
            None = 0,
            // Text is trimmed at the end of the string. An ellipsis (...)
            // is drawn in place of remaining text.
            End = 1,
            // Text is trimmed at the beginning of the string.
            // An ellipsis (...) is drawn in place of remaining text.
            Start = 2,
            // Text is trimmed in the middle of the string.
            // An ellipsis (...) is drawn in place of remaining text.
            Middle = 3,
            // Preserve as much as possible of the drive and filename information.
            // Must be combined with alignment information.
            Path = 4,
            // Text is trimmed at a word boundary.
            // Must be combined with alignment information.
            Word = 8
        }

        private const string EllipsisChars = "...";

        public static string Compact(string text, Control ctrl, EllipsisFormat options)
        {
            var s = TextRenderer.MeasureText(text, ctrl.Font);

            // control is large enough to display the whole text
            if (s.Width <= ctrl.Width)
                return text;

            var len = 0;
            var seg = text.Length;
            var fit = "";

            // find the longest string that fits into
            // the control boundaries using bisection method
            while (seg > 1)
            {
                seg -= seg / 2;

                var left = len + seg;
                var right = text.Length;

                if (left > right)
                    continue;

                if ((EllipsisFormat.Middle & options) == EllipsisFormat.Middle)
                {
                    right -= left / 2;
                    left -= left / 2;
                }
                else if ((EllipsisFormat.Start & options) != 0)
                {
                    right -= left;
                    left = 0;
                }

                // build and measure a candidate string with ellipsis
                var tst = text.Substring(0, left) + EllipsisChars + text.Substring(right);

                s = TextRenderer.MeasureText(tst, ctrl.Font);

                // candidate string fits into control boundaries,
                // try a longer string
                // stop when seg <= 1
                if (s.Width <= ctrl.Width)
                {
                    len += seg;
                    fit = tst;
                }
            }

            if (len == 0) // string can't fit into control
                return EllipsisChars;

            return fit;
        }

        #endregion // Found on CodeProject
    }
}
