﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Reflection;
using EffectCombiner.Data;
using EffectCombiner.Core;
using System.Threading;
using EffectCombiner.Core.Extensions;

namespace EffectCombiner.Editor
{
    public enum ReportCategory
    {
        Application,
        Project,
        Options,
        WorkflowProcessorComposition,
        WorkflowConverterComposition,
        Renderer,
        RendererComposition,
        Definition,
        Communication,
        ShaderGeneration,
    }

    public class EventReport : EventReportBase
    {
        public ReportCategory Category { get; private set; }

        public EventReport(
            Func<string> messageFunc,
            ReportLevel level,
            ReportCategory category,
            string source,
            Func<string> commentFunc,
            Exception exception)
            : base(messageFunc, level, source, commentFunc, exception)
        {
            Category = category;
        }
    }

    public static class Reporting
    {
        private static EventsReportForm reportForm;
        private static SynchronizationContext syncContext;

        public static void Initialize()
        {
            syncContext = SynchronizationContext.Current;
        }

        /// <summary>
        /// レポートを追加する
        /// </summary>
        /// <param name="eventReport">追加するイベントの情報</param>
        /// <param name="showForm">イベントログを表示するか</param>
        public static void Report(EventReport eventReport, bool showForm = true)
        {
            syncContext.RunSync(() =>
                {
                    CheckForm();
                    reportForm.Report(eventReport);
                    AnalyzeException(eventReport);
                    if (showForm == true)
                    {
                        reportForm.Show();
                    }
                });
        }

        public static void ShowReportForm()
        {
            syncContext.RunSync(() =>
                {
                    CheckForm();
                    reportForm.Show();
                    reportForm.BringToFront();
                });
        }

        private static void CheckForm()
        {
            if (reportForm == null || reportForm.IsDisposed)
                reportForm = new EventsReportForm();
        }

        private static void AnalyzeException(EventReport eventReport)
        {
            if (eventReport == null || eventReport.Exception == null)
                return;

            var reflectionTypeLoadException = eventReport.Exception as ReflectionTypeLoadException;
            if (reflectionTypeLoadException == null)
                return;

            foreach (var loaderError in reflectionTypeLoadException.LoaderExceptions)
            {
                Report(new EventReport(
                           () => "LoaderException unwrap",
                           eventReport.Level,
                           eventReport.Category,
                           eventReport.Source,
                           null,
                           loaderError));
            }
        }
    }

    public static class ReportingExtensions
    {
        public static EventReport ToEventReport(this EffectDefinitions.Error error)
        {
            var level = ReportLevel.Error;
            if (error.Severity == EffectDefinitions.ErrorSeverity.Warning)
                level = ReportLevel.Warning;

            return new EventReport(
                () => error.Message, level,
                ReportCategory.Definition,
                error.Location != null ? error.Location.ToString() : null,
                () => string.Format(Localization.Messages.NATURE_OF_THE_ERROR, error.Type),
                error.Exception);
        }
    }

    /*
    public class DataLayerCompositionExceptionReporter : EffectMaker.Data.IExceptionReporter
    {
        private readonly ReportCategory categ;

        public DataLayerCompositionExceptionReporter(ReportCategory category)
        {
            categ = category;
        }

        public void Report(Exception error)
        {
            Reporting.Report(new EventReport(
                error.Message,
                ReportLevel.Error,
                categ,
                null,
                null,
                error));
        }
    }
    */

    public class ReportingExceptionReporter : IExceptionReporter
    {
        public ReportCategory ReportCategory { get; private set; }

        public ReportingExceptionReporter(ReportCategory reportCategory)
        {
            ReportCategory = reportCategory;
        }

        public void Report(Exception error)
        {
            Reporting.Report(new EventReport(
                () => error.Message,
                ReportLevel.Error,
                ReportCategory,
                null,
                null,
                error));
        }
    }

    public class WorkflowDataEventReporter : IWorkflowDataEventReporter
    {
        public void Report(WorkflowDataEventReport eventReport)
        {
            Reporting.Report(new EventReport(
                eventReport.Message,
                eventReport.IsError ? ReportLevel.Error : ReportLevel.Warning,
                ReportCategory.Project,
                null,
                null,
                eventReport.Exception));
        }
    }
}
