﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Drawing;
using System.Linq;
using System.Reflection;
using System.Text;
using System.Windows.Forms;
using EffectCombiner.Core;
using EffectCombiner.Editor.OptionPanes;
using EffectCombiner.Primitives;

namespace EffectCombiner.Editor
{
    public partial class OptionsForm : Form
    {
        public OptionsForm()
        {
            InitializeComponent();

            btnAccept.Click += ButtonAcceptClick;
            btnCancel.Click += ButtonCancelClick;
            trvPaneNodes.AfterSelect += TreeViewPaneNodesAfterSelect;

            optionPanes = LoadOptionPanes();
            BuildOptionPanesTree();

            var localizationSubscription = Globals.Localization.RegisterLocalization(() =>
                {
                    Text = Localization.Controls.FORM_OPTIONS_TITLE;
                    btnAccept.Text = Localization.Controls.BUTTON_OK;
                    btnCancel.Text = Localization.Controls.BUTTON_CANCEL;
                });

            FormClosed += (ss, ee) =>
                {
                    localizationSubscription.Dispose();
                    UnloadOptionPanes();
                };
        }

        protected override void OnLoad(EventArgs e)
        {
            base.OnLoad(e);
            OptionsExtensions.ApplyRecordedFormWindowState(Globals.Options.EnvironmentSettings.OptionsWindow, this);
        }

        protected override void OnFormClosing(FormClosingEventArgs e)
        {
            base.OnFormClosing(e);

            if (DialogResult == DialogResult.OK)
            {
                foreach (var pane in optionPanes)
                    pane.OnAccept();
                OptionsManager.Save();
            }
            else
            {
                foreach (var pane in optionPanes)
                    pane.OnCancel();
            }

            OptionsExtensions.RecordFormWindowState(Globals.Options.EnvironmentSettings.OptionsWindow, this);
        }

        private void ButtonAcceptClick(object sender, EventArgs e)
        {
            DialogResult = DialogResult.OK;
            Close();
        }

        private void ButtonCancelClick(object sender, EventArgs e)
        {
            DialogResult = DialogResult.Cancel;
            Close();
        }

        private void TreeViewPaneNodesAfterSelect(object sender, TreeViewEventArgs e)
        {
            //using (new DrawingSuspender(pnlRoot))
            {
                pnlPaneContainer.Controls.Clear();
                var optionPane = e.Node as OptionPaneTreeViewItem;
                if (optionPane != null)
                {
                    var control = optionPane.OptionPane as Control;
                    if (control != null)
                    {
                        control.Width = pnlPaneContainer.Width;
                        pnlPaneContainer.Controls.Add(control);
                    }
                }
            }
            //pnlRoot.Update();
        }

        private readonly IOptionPane[] optionPanes;

        private void BuildOptionPanesTree()
        {
            var nodes = optionPanes
                .Select(p => new OptionPaneTreeViewItem(p))
                .ToArray();

            var hasChildren = false;

            using (new DrawingSuspender(trvPaneNodes))
            {
                foreach (var node in nodes)
                {
                    if (node.OptionPane.ChildOf == null)
                        trvPaneNodes.Nodes.Add(node);
                    else
                    {
                        var parent = nodes.Single(item =>
                            item.OptionPane.Identifier == node.OptionPane.ChildOf);
                        parent.Nodes.Add(node);
                        hasChildren = true;
                    }
                }

                trvPaneNodes.ShowRootLines = hasChildren;
                if (hasChildren)
                    trvPaneNodes.ExpandAll();

                foreach (var pane in optionPanes)
                    pane.OnInitialize();
            }
        }

        private IOptionPane[] LoadOptionPanes()
        {
            var asm = Assembly.GetExecutingAssembly();

            var controlType = typeof(Control);
            var optionPaneInterfaceType = typeof(IOptionPane);

            var list = new List<IOptionPane>();

            foreach (var t in asm.GetTypes())
            {
                var type = t;

                var match = true;
                match &= type.IsInterface == false;
                match &= controlType.IsAssignableFrom(type);
                match &= optionPaneInterfaceType.IsAssignableFrom(type);

                if (match == false)
                    continue;

                IOptionPane instance;

                try
                {
                    instance = (IOptionPane)Activator.CreateInstance(type);
                }
                catch (Exception ex)
                {
                    Reporting.Report(new EventReport(
                        () => string.Format(Localization.Messages.IMPOSSIBLE_TO_INSTANCIATE_PANE, type.FullName),
                        ReportLevel.Error, ReportCategory.Application, null, null, ex));
                    continue;
                }

                var ctrl = (Control)instance;
                ctrl.Anchor = AnchorStyles.Left | AnchorStyles.Top | AnchorStyles.Right;
                ctrl.Height = ctrl.Controls.Cast<Control>().Max(c => c.Bottom);

                list.Add(instance);
            }

            return list
                .OrderBy(GetDisplayOrder)
                .ThenBy(p => p.DisplayName)
                .ToArray();
        }

        private void UnloadOptionPanes()
        {
            foreach (var node in trvPaneNodes.Nodes.Cast<OptionPaneTreeViewItem>())
                UnloadOptionPane(node);
        }

        private void UnloadOptionPane(OptionPaneTreeViewItem item)
        {
            foreach (var child in item.Nodes.Cast<OptionPaneTreeViewItem>())
                UnloadOptionPane(child);
            item.Dispose();
        }

        private static int GetDisplayOrder(object instance)
        {
            var attr = instance.GetType()
                .GetCustomAttributes(typeof(DisplayOrderAttribute), true)
                .Cast<DisplayOrderAttribute>()
                .FirstOrDefault();

            return attr != null ? attr.Value : 0;
        }
    }

    public class OptionPaneTreeViewItem : TreeNode, IDisposable
    {
        public IOptionPane OptionPane { get; private set; }

        public OptionPaneTreeViewItem(IOptionPane optionPane)
        {
            if (optionPane == null)
                throw new ArgumentNullException("optionPane");

            OptionPane = optionPane;
            Text = optionPane.DisplayName;

            Globals.Localization.CultureChanged += LocalizationCultureChanged;
        }

        public void Dispose()
        {
            Globals.Localization.CultureChanged -= LocalizationCultureChanged;
            OptionPane.OnTerminate();
        }

        private void LocalizationCultureChanged(object sender, EventArgs e)
        {
            Text = OptionPane.DisplayName;
        }
    }
}
