﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Drawing;
using System.Data;
using System.Linq;
using System.Text;
using System.Windows.Forms;
using EffectCombiner.Primitives;

namespace EffectCombiner.Editor.OptionPanes
{
    public partial class ShaderGenerationPane : UserControl, IOptionPane
    {
        public ShaderGenerationPane()
        {
            InitializeComponent();

            btnBrowseShaderFilename.Click += ButtonBrowseShaderFilenameClick;

            var localizationSubscription = Globals.Localization.RegisterLocalization(() =>
                {
                    chkDumpShaderToFile.Text = Localization.Controls.OPTION_PANE_SHADERGEN_DUMP_TO_FILE;
                    lblShaderFilename.Text = Localization.Controls.OPTION_PANE_SHADERGEN_SHADER_FILENAME_LABEL;
                    lblDefinitionsShaderFileName.Text = Localization.Controls.OPTION_PANE_SHADERGEN_DEFINITIONS_SHADER_FILENAME_LABEL;
                    chkShowShaderWindow.Text = Localization.Controls.OPTION_PANE_SHADERGEN_PREVIEW_GENERATED_SHADER_CHECKBOX;
                    chkTryCompileShader.Text = Localization.Controls.OPTION_PANE_SHADERGEN_COMPILE_ON_THE_FLY_CHECKBOX;
                    chkAllowShaderGenOnInvalidGraph.Text = Localization.Controls.OPTION_PANE_SHADER_GEN_ALLOW_SHADER_GEN_ON_INVALID_GRAPH;
                });

            Disposed += (ss, ee) => localizationSubscription.Dispose();
        }

        private void ButtonBrowseShaderFilenameClick(object sender, EventArgs e)
        {
            var dlg = new SaveFileDialog
            {
                CheckFileExists = false,
                CheckPathExists = true,
                CreatePrompt = false,
                OverwritePrompt = true,
                ShowHelp = true,
                Title = Localization.Messages.DLG_GENERATED_SHADER_DUMP_FILE_TITLE,
            };

            if (dlg.ShowDialog() == DialogResult.OK)
                txtShaderFilename.Text = dlg.FileName;
        }

        private void BrowseDefinitionsShaderFileNameClick(object sender, EventArgs e)
        {
            var dlg = new OpenFileDialog
            {
                CheckFileExists = true,
                CheckPathExists = true,
                ShowHelp = true,
                Filter = string.Format("{0}|*.glsl;*.vsh;*.gsh;*.fsh|{1}|*.*",
                Localization.Messages.DLG_FILTER_SHADER_FILES,
                Localization.Messages.DLG_FILTER_ALL_FILES),
                Title = Localization.Messages.DLG_DEFINITION_SHADER_FILE_TITLE,
            };

            if (dlg.ShowDialog() == DialogResult.OK)
            {
                txtDefinitionsShaderFileName.Text = dlg.FileName;
            }
        }

        public string Identifier
        {
            get { return "ShaderGenerationPane"; }
        }

        public string ChildOf
        {
            get { return null; }
        }

        public string DisplayName
        {
            get { return Localization.Controls.OPTION_PANE_NAME_SHADER_GENERATION; }
        }

        public void OnInitialize()
        {
            if (Globals.Options.ShaderGeneration == null)
                return;

            chkDumpShaderToFile.Checked = Globals.Options.ShaderGeneration.WriteShaderFile;
            txtShaderFilename.Text = Globals.Options.ShaderGeneration.OutputShaderFilename;

            lblShaderFilename.Enabled = Globals.Options.ShaderGeneration.WriteShaderFile;
            txtShaderFilename.Enabled = Globals.Options.ShaderGeneration.WriteShaderFile;
            btnBrowseShaderFilename.Enabled = Globals.Options.ShaderGeneration.WriteShaderFile;

            // TODO: for now only one file is supported, it may become a file list in the future
            txtDefinitionsShaderFileName.Text =
                Globals.Options.ShaderGeneration.LinkDefinitionsShaderFileNames != null &&
                Globals.Options.ShaderGeneration.LinkDefinitionsShaderFileNames.Length > 0
                ? Globals.Options.ShaderGeneration.LinkDefinitionsShaderFileNames[0].Path
                : string.Empty;

            chkShowShaderWindow.Checked = Globals.Options.ShaderGeneration.ShowShaderCodeWindow;
            chkTryCompileShader.Checked = Globals.Options.ShaderGeneration.TryCompileShader;
            chkAllowShaderGenOnInvalidGraph.Checked = Globals.Options.ShaderGeneration.AllowShaderGenerationOnNotProperlyConnectedGraph;
        }

        public void OnTerminate()
        {
        }

        public void OnAccept()
        {
            if (Globals.Options.ShaderGeneration == null)
                Globals.Options.ShaderGeneration = new ShaderGeneration();

            Globals.Options.ShaderGeneration.WriteShaderFile = chkDumpShaderToFile.Checked;
            Globals.Options.ShaderGeneration.OutputShaderFilename = txtShaderFilename.Text;

            // TODO: for now only one file is supported, it may become a file list in the future
            Globals.Options.ShaderGeneration.LinkDefinitionsShaderFileNames = new[]
                {
                    new DesactivablePath
                    {
                        IsActive = true,
                        Path = txtDefinitionsShaderFileName.Text,
                    },
                };

            Globals.Options.ShaderGeneration.ShowShaderCodeWindow = chkShowShaderWindow.Checked;
            Globals.Options.ShaderGeneration.TryCompileShader = chkTryCompileShader.Checked;
            Globals.Options.ShaderGeneration.AllowShaderGenerationOnNotProperlyConnectedGraph = chkAllowShaderGenOnInvalidGraph.Checked;
        }

        public void OnCancel()
        {
        }

        private void chkDumpShaderToFile_CheckedChanged(object sender, EventArgs e)
        {
            lblShaderFilename.Enabled = chkDumpShaderToFile.Checked;
            txtShaderFilename.Enabled = chkDumpShaderToFile.Checked;
            btnBrowseShaderFilename.Enabled = chkDumpShaderToFile.Checked;
        }
    }
}
