﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Drawing;
using System.Data;
using System.Linq;
using System.Text;
using System.Windows.Forms;
using EffectCombiner.Primitives;
using System.Globalization;

namespace EffectCombiner.Editor.OptionPanes
{
    public partial class LanguagePane : UserControl, IOptionPane
    {
        public LanguagePane()
        {
            InitializeComponent();

            lstAvailableLanguages.SelectedIndexChanged += lstAvailableLanguages_SelectedIndexChanged;
        }

        public string Identifier
        {
            get { return "LanguagePane"; }
        }

        public string ChildOf
        {
            get { return null; }
        }

        public string DisplayName
        {
            get { return Localization.Controls.OPTION_PANE_NAME_LANGUAGE; }
        }

        private CultureInfo validatedLanguage;

        public void OnInitialize()
        {
            validatedLanguage = Globals.Localization.CurrentUICulture;

            var availableLanguages = Globals.Localization.DetectAvailableLanguages();

            lstAvailableLanguages.Items.Add(new LanguageListBoxItem(CultureInfo.InvariantCulture));
            foreach (var culture in availableLanguages)
                lstAvailableLanguages.Items.Add(new LanguageListBoxItem(culture));

            var index = 0;

            if (Globals.Localization.CurrentUICulture != null)
            {
                index = lstAvailableLanguages.Items
                    .Cast<LanguageListBoxItem>()
                    .ToList()
                    .FindIndex(item => item.Culture == Globals.Localization.CurrentUICulture);

                if (index < 0)
                    index = 0;
            }

            lstAvailableLanguages.SelectedIndex = index;
        }

        private void lstAvailableLanguages_SelectedIndexChanged(object sender, EventArgs e)
        {
            var item = lstAvailableLanguages.SelectedItem as LanguageListBoxItem;
            if (item == null)
                return;

            Globals.Localization.ChangeLanguage(item.Culture);
        }

        public void OnTerminate()
        {
        }

        public void OnAccept()
        {
            validatedLanguage = Globals.Localization.CurrentUICulture;
            Globals.Options.LanguageSettings.CurrentLanguage = validatedLanguage != null ? validatedLanguage.Name : null;
            Globals.Localization.ChangeLanguage(validatedLanguage);
        }

        public void OnCancel()
        {
            Globals.Localization.ChangeLanguage(validatedLanguage);
        }

        private class LanguageListBoxItem
        {
            public CultureInfo Culture { get; private set; }

            public LanguageListBoxItem(CultureInfo culture)
            {
                if (culture == null)
                    throw new ArgumentNullException("culture");

                Culture = culture;
            }

            public override string ToString()
            {
                return Culture == CultureInfo.InvariantCulture
                    ? "(Default)" // (Default) text must not be translated
                    : Capitalize(Culture.NativeName);
            }
        }

        private static string Capitalize(string str)
        {
            if (string.IsNullOrWhiteSpace(str))
                return str;

            return string.Format("{0}{1}", char.ToUpperInvariant(str.First()), new string(str.Skip(1).ToArray()));
        }
    }
}
