﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Reflection;
using System.Threading;

using EffectCombiner.Primitives;
using EffectCombiner.Primitives.Blocks;
using ShaderGenerator.CafeCompiler;

namespace EffectCombiner.Editor
{
    public class GenerationManager : IDisposable
    {
        public GenerationManager(CompositionBlockManager blockManager)
        {
            if (blockManager == null)
                throw new ArgumentNullException("blockManager");

            BlockManager = blockManager;

            BlockManager.ShaderCodeProduced += BlockManagerShaderCodeProduced;
            synchronizationContext = SynchronizationContext.Current;
        }

        public CompositionBlockManager BlockManager { get; private set; }

        private readonly SynchronizationContext synchronizationContext;

        private bool disposed = false;
        public void Dispose()
        {
            if (disposed == false)
            {
                BlockManager.ShaderCodeProduced -= BlockManagerShaderCodeProduced;
                disposed = true;
            }
        }

        private void BlockManagerShaderCodeProduced(object sender, CompositionBlockManager.ShaderProducedEventArgs e)
        {
            var wentThroughCompilation = false;
            var compilationResult = CompilationResultCode.Success;

            if (e.IsNewlyGenerated == false)
            {
                ShowShaderCode(e.SourceCode, null, compilationResult, wentThroughCompilation);
                return;
            }

            IEnumerable<string> errors = null;

            if (Compiler.IsAvailable && (e.IsBuildForced || Globals.Options.ShaderGeneration.TryCompileShader))
            {
                compilationResult = TryToCompile(e.SourceCode, out errors);
                wentThroughCompilation = true;
            }

            if (Globals.Options.ShaderGeneration.WriteShaderFile &&
                string.IsNullOrWhiteSpace(Globals.Options.ShaderGeneration.OutputShaderFilename) == false)
            {
                try
                {
                    WriteGeneratedShaderToFile(e.SourceCode, Globals.Options.ShaderGeneration.OutputShaderFilename);
                }
                catch { }
            }

            if (Globals.Options.ShaderGeneration.ShowShaderCodeWindow)
            {
                synchronizationContext.Post(_ =>
                    ShowShaderCode(e.SourceCode, errors, compilationResult, wentThroughCompilation), null);
            }

            if (wentThroughCompilation)
            {
                ShowCompilationResultQuickInfo(compilationResult);
            }
        }

        private void ShowShaderCode(string shaderCode, IEnumerable<string> errors,
            CompilationResultCode compilationResult,
            bool wentThroughCompilation)
        {
            var form = new ShaderCodeForm(shaderCode, errors, compilationResult, wentThroughCompilation);
            form.Show();
        }

        private static CompilationResultCode TryToCompile(string sourceCode, out IEnumerable<string> errors)
        {
            var cafeCompiler = new Compiler();

            string compiledShader;
            return cafeCompiler.CompileInMemory(sourceCode, out compiledShader, false, out errors);
        }

        public static void WriteGeneratedShaderToFile(string sourceCode, string fileName)
        {
            if (System.IO.Path.IsPathRooted(fileName) == false)
            {
                var path = System.IO.Path.GetDirectoryName(Assembly.GetEntryAssembly().Location);
                fileName = string.Format("{0}\\{1}", path, fileName);
            }

            System.IO.File.WriteAllText(fileName, sourceCode);
        }

        private static void ShowCompilationResultQuickInfo(CompilationResultCode compilationResult)
        {
            if (compilationResult.ToOutcome() == CompilationOutcome.CompilationSucceeded)
                Globals.QuickInfo.SetQuickInfo(Localization.Controls.QUICK_INFO_SHADER_BUILD_OK, true, VisualResourceSet.ShaderOKQuickInfoColor);
            else if (compilationResult.ToOutcome() == CompilationOutcome.CompilationFailed)
                Globals.QuickInfo.SetQuickInfo(Localization.Controls.QUICK_INFO_SHADER_BUILD_FAILED, false, VisualResourceSet.ShaderNotOKQuickInfoColor);
            else
                Globals.QuickInfo.SetQuickInfo(string.Format(Localization.Controls.QUICK_INFO_COULD_NOT_RUN_COMPILER,
                    compilationResult.ToMessage()), false, VisualResourceSet.CompilerErrorQuickInfoColor);
        }
    }
}
