﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.IO;
using System.Linq;
using EffectCombiner.Editor.Log;
using EffectCombiner.Generator;

namespace EffectCombiner.Editor.CommandLine
{
    /// <summary>
    /// コマンドラインでシェーダコードを生成します。
    /// </summary>
    public class CommandLineShaderCodeGenerator
    {
        /// <summary>
        /// シェーダコードを生成します。
        /// </summary>
        /// <param name="param">生成パラメータ</param>
        /// <returns></returns>
        public int Generate(CommandLineParams param)
        {
            string[] shaderEnvironmentPath;

            // パラメータの内容をチェックする
            {
                shaderEnvironmentPath = param.ShaderEnvironmentPath.Split(new char[] { ';' }, StringSplitOptions.RemoveEmptyEntries);
                shaderEnvironmentPath = shaderEnvironmentPath.Select(x => x.Trim()).ToArray();

                // シェーダの環境パスがあるかチェック
                foreach (string path in shaderEnvironmentPath)
                {
                    if (Directory.Exists(path) == false)
                    {
                        Logger.Log("Console", LogLevels.Error, Properties.Resources.DirectoryNotFound, path);

                        return (int)CommandLineApplication.ExitCodeKind.InvalidArguments;
                    }
                }

                string outputDir = Path.GetDirectoryName(param.OutputFile);

                // 出力フォルダがあるかチェック
                if (Directory.Exists(outputDir) == false)
                {
                    Logger.Log("Console", LogLevels.Error, Properties.Resources.DirectoryNotFound, outputDir);

                    return (int)CommandLineApplication.ExitCodeKind.InvalidArguments;
                }

                // 入力ファイルがあるかチェック
                if (File.Exists(param.InputFile) == false)
                {
                    Logger.Log("Console", LogLevels.Error, Properties.Resources.FileNotFound, param.InputFile);

                    return (int)CommandLineApplication.ExitCodeKind.InvalidArguments;
                }
            }

            Logger.Log("Console", LogLevels.Information, "Start generating shader code.");

            // シェーダコードを生成する
            try
            {
                ShaderGenSettings genSettings = new ShaderGenSettings();

                foreach (string path in shaderEnvironmentPath)
                {
                    genSettings.EffectDefinitionsPaths.Add(path, SearchOption.AllDirectories, "*.glsl", "*.edml");
                }

                ShaderGen shaderGen = new ShaderGen(genSettings);
                shaderGen.Initialize();

                string shaderMainName = "main";
                string[] codes = shaderGen.ProduceShaderCodes(param.InputFile, shaderMainName);

                if (codes == null || codes.Length == 0)
                {
                    Logger.Log("Console", LogLevels.Error, "The EffectCombiner can not generate any shader from {0}", param.InputFile);

                    return (int)CommandLineApplication.ExitCodeKind.WarningOnConvertProcess;
                }

                GenerationManager.WriteGeneratedShaderToFile(codes[0], param.OutputFile);
            }
            catch (Exception e)
            {
                Logger.LogException(e);

                return (int)CommandLineApplication.ExitCodeKind.WarningOnConvertProcess;
            }

            Logger.Log("Console", LogLevels.Information, "Generating shader code is successfully.");

            return (int)CommandLineApplication.ExitCodeKind.NoErrors;
        }
    }
}
