﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections;
using System.Collections.Generic;
using System.Drawing;
using System.Linq;
using System.Text;
using System.Windows.Forms;
using System.Threading;

using EffectCombiner.Core.Extensions;
using EffectDefinitions;
using EffectCombiner.Primitives;

namespace EffectCombiner.Editor
{
    public enum InOutStatus
    {
        Input,
        Output,
        Neutral,
    }

    public class BlockTreeNode : TreeNode
    {
        public BlockDefinition BlockDefinition { get; private set; }

        public BlockTreeNode(BlockDefinition blockDefinition)
        {
            if (blockDefinition == null)
                throw new ArgumentNullException("blockDefinition");

            BlockDefinition = blockDefinition;
            Text = blockDefinition.Name;
        }
    }

    /// <summary>
    /// ListViewで使用される、blockごとのitem
    /// </summary>
    public class BlockListViewItem : ListViewItem, IDisposable
    {
        public int CreationIndex { get; private set; }
        public BlockDefinition BlockDefinition { get; private set; }
        public InOutStatus InOutStatus { get; private set; }

        public BlockListViewItem(int creationIndex, BlockDefinition blockDefinition)
        {
            if (blockDefinition == null)
                throw new ArgumentNullException("blockDefinition");

            BlockDefinition = blockDefinition;
            CreationIndex = creationIndex;

            SetInOutStatus(blockDefinition);

            // ListViewのカラムの順番に関係するので、このSubItemsの順番を変えた場合は、ListViewのカラムの順番も変更すること.
            SubItems[0].Text = string.Join(", ", (blockDefinition.Tags ?? new string[0]).OrderBy(x => x));
            SubItems/*[1]*/.Add(blockDefinition.Name);
            SubItems/*[2]*/.Add(blockDefinition.Description);
            SubItems/*[3]*/.Add(GetInOutStatusText());
            SubItems/*[4]*/.Add(blockDefinition.Source != null ? blockDefinition.Source.Uri.ToString() : string.Empty);

            Globals.Localization.CultureChanged += LocalizationCultureChanged;

            // タグのSubItemだけ背景色を変更する処理をListViewで行うので、
            // Font, ForeColor, BackColorを使用しない設定にする.
            UseItemStyleForSubItems = false;
        }

        public void Dispose()
        {
            Globals.Localization.CultureChanged -= LocalizationCultureChanged;
        }

        private void LocalizationCultureChanged(object sender, EventArgs e)
        {
            SubItems[3].Text = GetInOutStatusText();
        }

        private void SetInOutStatus(BlockDefinition definition)
        {
            var inCount = definition.InputPlugs.Length;
            var outCount = definition.OutputPlugs.Length;

            InOutStatus = InOutStatus.Neutral;

            if (outCount == 0 && inCount != 0)
                InOutStatus = InOutStatus.Output;
            else if (inCount == 0 && outCount != 0)
                InOutStatus = InOutStatus.Input;
        }

        private string GetInOutStatusText()
        {
            if (InOutStatus == InOutStatus.Input)
                return Localization.Messages.TEXT_BLOCK_LIST_INPUT;
            if (InOutStatus == InOutStatus.Output)
                return Localization.Messages.TEXT_BLOCK_LIST_OUTPUT;

            return string.Empty;
        }

        public void UpdateTags()
        {
            var tags = string.Join(", ", (BlockDefinition.Tags ?? new string[0]).OrderBy(x => x));
            SubItems[2].Text = tags;
        }
    }

    /// <summary>
    /// ListViewの項目の並び替えに使用するクラス
    /// </summary>
    public class BlockListViewItemComparer : IComparer
    {
        /// <summary>
        /// The _column.
        /// </summary>
        private int column;

        /// <summary>
        /// The _order.
        /// </summary>
        private SortOrder order;

        /// <summary>
        /// The _mode.
        /// </summary>
        private ComparerMode mode;

        /// <summary>
        /// The _column modes.
        /// </summary>
        private ComparerMode[] columnModes;

        /// <summary>
        /// ListViewItemComparerクラスのコンストラクタ
        /// </summary>
        /// <param name="col">並び替える列の番号</param>
        /// <param name="ord">昇順か降順か</param>
        /// <param name="cmod">並び替えの方法</param>
        public BlockListViewItemComparer(
            int col,
            SortOrder ord,
            ComparerMode cmod)
        {
            this.column = col;
            this.order = ord;
            this.mode = cmod;
        }

        /// <summary>
        /// デフォルトコンストラクタ
        /// </summary>
        public BlockListViewItemComparer()
        {
            this.column = 0;
            this.order = SortOrder.Ascending;
            this.mode = ComparerMode.String;
        }

        /// <summary>
        /// 比較する方法
        /// </summary>
        public enum ComparerMode
        {
            /// <summary>
            /// 文字列として比較
            /// </summary>
            String,

            /// <summary>
            /// 数値（Int32型）として比較
            /// </summary>
            Integer,

            /// <summary>
            /// 日時（DataTime型）として比較
            /// </summary>
            DateTime
        }

        /// <summary>
        /// 並び替えるListView列の番号
        /// </summary>
        public int Column
        {
            get
            {
                return this.column;
            }

            set
            {
                // 現在と同じ列の時は、昇順降順を切り替える
                if (this.column == value)
                {
                    if (this.order == SortOrder.Ascending)
                    {
                        this.order = SortOrder.Descending;
                    }
                    else
                    {
                        this.order = SortOrder.Ascending;
                    }
                }
                else
                {
                    this.order = SortOrder.Ascending;
                }

                this.column = value;
            }
        }

        /// <summary>
        /// 昇順か降順か
        /// </summary>
        public SortOrder Order
        {
            get
            {
                return this.order;
            }

            set
            {
                this.order = value;
            }
        }

        /// <summary>
        /// 並び替えの方法
        /// </summary>
        public ComparerMode Mode
        {
            get
            {
                return this.mode;
            }

            set
            {
                this.mode = value;
            }
        }

        /// <summary>
        /// 列ごとの並び替えの方法
        /// </summary>
        public ComparerMode[] ColumnModes
        {
            set
            {
                this.columnModes = value;
            }
        }

        /// <summary>
        /// 比較関数
        /// </summary>
        /// <param name="x">
        /// The x.
        /// </param>
        /// <param name="y">
        /// The y.
        /// </param>
        /// <returns>
        /// xがyより小さいときはマイナスの数、大きいときはプラスの数、
        /// 同じときは0を返す
        /// </returns>
        public int Compare(object x, object y)
        {
            int tmpCol = this.column;

            int result = 0;

            // ListViewItemの取得
            var itemx = (ListViewItem)x;
            var itemy = (ListViewItem)y;

            // 並べ替えの方法を決定
            if (this.columnModes != null && this.columnModes.Length > tmpCol)
            {
                this.mode = this.columnModes[tmpCol];
            }

            // 並び替えの方法別に、xとyを比較する
            switch (this.mode)
            {
                case ComparerMode.String:
                    // 文字列をとして比較
                    result = string.CompareOrdinal(
                        itemx.SubItems[tmpCol].Text,
                        itemy.SubItems[tmpCol].Text);
                    break;
                case ComparerMode.Integer:
                    // Int32に変換して比較
                    // .NET Framework 2.0からは、TryParseメソッドを使うこともできる
                    result = int.Parse(itemx.SubItems[tmpCol].Text).CompareTo(
                        int.Parse(itemy.SubItems[tmpCol].Text));
                    break;
                case ComparerMode.DateTime:
                    // DateTimeに変換して比較
                    // .NET Framework 2.0からは、TryParseメソッドを使うこともできる
                    result = DateTime.Compare(
                        DateTime.Parse(itemx.SubItems[tmpCol].Text),
                        DateTime.Parse(itemy.SubItems[tmpCol].Text));
                    break;
            }

            // 降順の時は結果を+-逆にする
            if (this.order == SortOrder.Descending)
            {
                result = -result;
            }

            // 結果を返す
            return result;
        }
    }
}
