﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

//=============================================================================
// include
//=============================================================================
#include "NintendoNormalMapFilterScripting.h"

using namespace nn::gfx::tool::nps;

//-----------------------------------------------------------------------------
// 無名名前空間を開始します。
namespace
{

//=============================================================================
// constants
//=============================================================================
const DescriptorEnumID OperationEnumIds[] =
{
    nnNormalMapFilterEnumConvertFromHeightMap,
    nnNormalMapFilterEnumNormalize,
    nnNormalMapFilterEnumScaleSlope,
    nnNormalMapFilterEnumCombineFile,
    nnNormalMapFilterEnumCombineLayer,
};

const DescriptorEnumID FilterTypeEnumIds[] =
{
    nnNormalMapFilterEnumFilterType4,
    nnNormalMapFilterEnumFilterType3x3,
    nnNormalMapFilterEnumFilterType5x5,
    nnNormalMapFilterEnumFilterType7x7,
    nnNormalMapFilterEnumFilterType9x9,
    nnNormalMapFilterEnumFilterType13x13,
    nnNormalMapFilterEnumFilterType17x17,
    nnNormalMapFilterEnumFilterType21x21,
    nnNormalMapFilterEnumFilterType25x25,
};

const DescriptorKeyID CombineLayerNameKeyIds[] =
{
    nnNormalMapFilterKeyLayerName0,
    nnNormalMapFilterKeyLayerName1,
    nnNormalMapFilterKeyLayerName2,
    nnNormalMapFilterKeyLayerName3,
    nnNormalMapFilterKeyLayerName4,
};

const DescriptorKeyID CombineLayerScaleKeyIds[] =
{
    nnNormalMapFilterKeyLayerScale1,
    nnNormalMapFilterKeyLayerScale2,
    nnNormalMapFilterKeyLayerScale3,
    nnNormalMapFilterKeyLayerScale4,
};

//-----------------------------------------------------------------------------
//! @brief リード記述子から合成するレイヤーの名前を取得します。
//!
//! @param[in,out] globals グローバルデータです。
//! @param[in] keyId 記述子のキー ID です。
//! @param[in] desc リード記述子です。
//! @param[in] utf8ToSjis 文字コードを UTF-8 から Shift-JIS に変換するなら true です。
//-----------------------------------------------------------------------------
void GetCombineLayerName(
    GPtr globals,
    const DescriptorKeyID keyId,
    const PIReadDescriptor& desc,
    const bool utf8ToSjis
)
{
    int layerIdx = 0;
    for (const DescriptorKeyID id : CombineLayerNameKeyIds)
    {
        if (id == keyId)
        {
            char* nameBuf = gParams->m_CombineLayerNames[layerIdx];
            const size_t nameBufSize = sizeof(gParams->m_CombineLayerNames[layerIdx]);
            memset(nameBuf, 0x00, nameBufSize);
            RGetStringParam(globals, nameBuf, nameBufSize, desc, utf8ToSjis);
            break;
        }
        ++layerIdx;
    }
}

//-----------------------------------------------------------------------------
//! @brief リード記述子から合成するレイヤーのスケール値を取得します。
//!
//! @param[in,out] globals グローバルデータです。
//! @param[in] keyId 記述子のキー ID です。
//! @param[in] desc リード記述子です。
//-----------------------------------------------------------------------------
void GetCombineLayerScale(
    GPtr globals,
    const DescriptorKeyID keyId,
    const PIReadDescriptor& desc
)
{
    int layerIdx = 1;
    for (const DescriptorKeyID id : CombineLayerScaleKeyIds)
    {
        if (id == keyId)
        {
            int32 scale;
            PIGetInt(desc, &scale);
            gParams->m_CombineLayerScales[layerIdx] = scale;
            break;
        }
        ++layerIdx;
    }
}

//-----------------------------------------------------------------------------
//! @brief 指定されたレイヤー合成パラメーターを必要に応じて修正します。
//!
//! @param[in,out] globals グローバルデータです。
//-----------------------------------------------------------------------------
void AdjustCombineLayerParams(GPtr globals)
{
    //-----------------------------------------------------------------------------
    // 指定されたレイヤーが存在しなければレイヤー名をクリアします。
    const RStringArray& layerNames = *globals->m_pLayerNameArray;
    int combineCount = 0;
    for (int layerIdx = 0; layerIdx < RParameters::CombineLayerCountMax; ++layerIdx)
    {
        const char* nameBuf = gParams->m_CombineLayerNames[layerIdx];
        if (RFindValueInArray(layerNames, std::string(nameBuf)) == -1)
        {
            gParams->ClearCombineLayerName(layerIdx);
        }
        if (layerIdx >= 1 && nameBuf[0] != '\0')
        {
            ++combineCount;
        }
    }

    //-----------------------------------------------------------------------------
    // 合成可能なレイヤーが 1 つも存在しなければ
    // アクティブレイヤーとその 1 つ上のレイヤーを合成するようにします。
    if (combineCount == 0                                               &&
        globals->m_ActiveLayerIdx != -1                                 &&
        gParams->m_CombineLayerNames[0] == RParameters::ActiveLayerName)
    {
        const int upperLayerIdx = globals->m_ActiveLayerIdx - 1;
        if (0 <= upperLayerIdx && upperLayerIdx < layerNames.size())
        {
            strncpy_s(gParams->m_CombineLayerNames[1], layerNames[upperLayerIdx].c_str(), _TRUNCATE);
            //RNoteTrace("set default combine layer: %d", upperLayerIdx);
        }
    }
}

//-----------------------------------------------------------------------------
// 無名名前空間を終了します。
} // unnamed namespace

//-----------------------------------------------------------------------------
//! @brief スクリプトパラメーターをリードします。
//!
//! @param[in,out] globals グローバルデータです。
//-----------------------------------------------------------------------------
OSErr ReadScriptParameters(GPtr globals)
{
    //-----------------------------------------------------------------------------
    // リード記述子からパラメーターをリードします。
    DescriptorKeyIDArray KeyIdsToRead =
    {
        nnNormalMapFilterKeyOperation,
        nnNormalMapFilterKeyPositiveZ,
        nnNormalMapFilterKeyHeightScale,
        nnNormalMapFilterKeyFilterType,
        nnNormalMapFilterKeyEdgeWrap,
        nnNormalMapFilterKeyMultiplyAlpha,
        nnNormalMapFilterKeySlopeScale,
        nnNormalMapFilterKeyOtherPath,
        nnNormalMapFilterKeyCombineScale,
        nnNormalMapFilterKeyLayerName0,
        nnNormalMapFilterKeyLayerName1,
        nnNormalMapFilterKeyLayerName2,
        nnNormalMapFilterKeyLayerName3,
        nnNormalMapFilterKeyLayerName4,
        nnNormalMapFilterKeyLayerScale1,
        nnNormalMapFilterKeyLayerScale2,
        nnNormalMapFilterKeyLayerScale3,
        nnNormalMapFilterKeyLayerScale4,
        NULLID
    };

    OSErr error = noErr;
    if (DescriptorAvailable(nullptr))
    {
        PIReadDescriptor token = OpenReader(KeyIdsToRead);
        //RNoteTrace("read token: %p", token); // 一度も Write されてなければ 0
        if (token)
        {
            // Photoshop CC 2014 以降では PIGetText で取得した値が UTF-8 になっているので、Shift-JIS に変換します。
            const bool utf8ToSjis = RIsPhotoshopVersionGreaterEqual(PhotoshopVersionMajorCc2014, 0);

            // レイヤー合成関連のパラメーターをクリアします。
            gParams->ClearCombineLayerParams();

            DescriptorKeyID key = NULLID;
            DescriptorTypeID type = NULLID;
            int32 flags = 0;
            while (PIGetKey(token, &key, &type, &flags))
            {
                //RNoteTrace("key: %s", RGetStringFrom32BitId(key).c_str());
                switch (key)
                {
                case nnNormalMapFilterKeyOperation:
                    {
                        DescriptorEnumID opeId;
                        error = PIGetEnum(token, &opeId);
                        for (int iOpe = 0; iOpe < RParameters::Operation_Count; ++iOpe)
                        {
                            if (OperationEnumIds[iOpe] == opeId)
                            {
                                gParams->m_Operation = static_cast<RParameters::Operation>(iOpe);
                                break;
                            }
                        }
                        break;
                    }

                case nnNormalMapFilterKeyPositiveZ:
                    {
                        Boolean positiveZ;
                        error = PIGetBool(token, &positiveZ);
                        gParams->m_PositiveZ = (positiveZ == TRUE);
                        break;
                    }

                case nnNormalMapFilterKeyHeightScale:
                    {
                        int32 scale;
                        error = PIGetInt(token, &scale);
                        gParams->m_HeightScale = scale;
                        break;
                    }

                case nnNormalMapFilterKeyFilterType:
                    {
                        DescriptorEnumID filterTypeId;
                        error = PIGetEnum(token, &filterTypeId);
                        for (int iFilterType = 0; iFilterType < RParameters::FilterType_Count; ++iFilterType)
                        {
                            if (FilterTypeEnumIds[iFilterType] == filterTypeId)
                            {
                                gParams->m_FilterType = static_cast<RParameters::FilterType>(iFilterType);
                                break;
                            }
                        }
                        break;
                    }

                case nnNormalMapFilterKeyEdgeWrap:
                    {
                        Boolean edgeWrap;
                        error = PIGetBool(token, &edgeWrap);
                        gParams->m_EdgeWrap = (edgeWrap == TRUE);
                        break;
                    }

                case nnNormalMapFilterKeyMultiplyAlpha:
                    {
                        Boolean alphaMul;
                        error = PIGetBool(token, &alphaMul);
                        gParams->m_MultiplyAlpha = (alphaMul == TRUE);
                        break;
                    }

                case nnNormalMapFilterKeySlopeScale:
                    {
                        int32 scale;
                        error = PIGetInt(token, &scale);
                        gParams->m_SlopeScale = scale;
                        break;
                    }

                case nnNormalMapFilterKeyOtherPath:
                    RGetStringParam(globals, gParams->m_OtherPath, sizeof(gParams->m_OtherPath), token, utf8ToSjis);
                    break;

                case nnNormalMapFilterKeyCombineScale:
                    {
                        int32 scale;
                        error = PIGetInt(token, &scale);
                        gParams->m_CombineScale = scale;
                        break;
                    }

                case nnNormalMapFilterKeyLayerName0:
                case nnNormalMapFilterKeyLayerName1:
                case nnNormalMapFilterKeyLayerName2:
                case nnNormalMapFilterKeyLayerName3:
                case nnNormalMapFilterKeyLayerName4:
                    GetCombineLayerName(globals, key, token, utf8ToSjis);
                    break;

                case nnNormalMapFilterKeyLayerScale1:
                case nnNormalMapFilterKeyLayerScale2:
                case nnNormalMapFilterKeyLayerScale3:
                case nnNormalMapFilterKeyLayerScale4:
                    GetCombineLayerScale(globals, key, token);
                    break;

                default:
                    break;
                }
            }
            error = CloseReader(&token);
            if (error == errMissingParameter)
            {
                // KeyIdsToRead のキーが一部なかった場合 → エラーではない
                error = noErr;
            }
        }
        gQueryForParameters = PlayDialog();
    }

    //-----------------------------------------------------------------------------
    // 指定されたレイヤー合成パラメーターを必要に応じて修正します。
    AdjustCombineLayerParams(globals);

    return error;
} // NOLINT(impl/function_size)

//-----------------------------------------------------------------------------
//! @brief スクリプトパラメーターをライトします。
//!
//! @param[in,out] globals グローバルデータです。
//-----------------------------------------------------------------------------
OSErr WriteScriptParameters(GPtr globals)
{
    OSErr error = noErr;

    if (DescriptorAvailable(nullptr))
    {
        PIWriteDescriptor token = OpenWriter();
        if (token)
        {
            // common
            PIPutEnum(token, nnNormalMapFilterKeyOperation,
                nnNormalMapFilterTypeOperation, OperationEnumIds[gParams->m_Operation]);
            if (gParams->m_Operation != RParameters::Operation_ConvertFromHeightMap)
            {
                PIPutBool(token, nnNormalMapFilterKeyPositiveZ, gParams->m_PositiveZ);
            }

            // convert from height map
            if (gParams->m_Operation == RParameters::Operation_ConvertFromHeightMap)
            {
                PIPutInt(token, nnNormalMapFilterKeyHeightScale, gParams->m_HeightScale);

                PIPutEnum(token, nnNormalMapFilterKeyFilterType,
                    nnNormalMapFilterTypeFilterType, FilterTypeEnumIds[gParams->m_FilterType]);

                PIPutBool(token, nnNormalMapFilterKeyEdgeWrap, gParams->m_EdgeWrap);

                PIPutBool(token, nnNormalMapFilterKeyMultiplyAlpha, gParams->m_MultiplyAlpha);
            }

            // scale slope
            if (gParams->m_Operation == RParameters::Operation_ScaleSclope)
            {
                PIPutInt(token, nnNormalMapFilterKeySlopeScale, gParams->m_SlopeScale);
            }

            // ファイル合成
            if (gParams->m_Operation == RParameters::Operation_CombineFile)
            {
                Handle hOtherPath = PICString2Handle(gParams->m_OtherPath);
                error = PIPutText(token, nnNormalMapFilterKeyOtherPath, hOtherPath);
                PIDisposeHandle(hOtherPath);
                PIPutInt(token, nnNormalMapFilterKeyCombineScale, gParams->m_CombineScale);
            }

            // レイヤー合成
            if (gParams->m_Operation == RParameters::Operation_CombineLayer)
            {
                for (int layerIdx = 0; layerIdx < RParameters::CombineLayerCountMax; ++layerIdx)
                {
                    const char* nameBuf = gParams->m_CombineLayerNames[layerIdx];
                    if (nameBuf[0] != '\0')
                    {
                        Handle hLayerName = PICString2Handle(nameBuf);
                        error = PIPutText(token, CombineLayerNameKeyIds[layerIdx], hLayerName);
                        PIDisposeHandle(hLayerName);
                        if (layerIdx >= 1)
                        {
                            PIPutInt(token, CombineLayerScaleKeyIds[layerIdx - 1],
                                gParams->m_CombineLayerScales[layerIdx]);
                        }
                    }
                }
            }

            error = CloseWriter(&token);
        }
    }

    return error;
}

