﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

//=============================================================================
// include
//=============================================================================
#include "NpsCommon.h"

//=============================================================================
// nps ネームスペースを開始します。
//=============================================================================
namespace nn {
namespace gfx {
namespace tool {
namespace nps {

//=============================================================================
//! @brief プレビュー表示のクラスです。
//=============================================================================
class RPreview
{
public:
    static const int BORDER_W = 1; //!< 画像の周りの縁の幅です。
    static const int TIMER_DRAG = 8; //!< パラメータのスライダドラッグ中の更新までの秒数 / 0.05 です。

    bool m_DisplaysRgb; //!< RGB 成分を表示するなら true です。
    bool m_DisplaysAlpha; //!< RGB 成分を表示するなら true です。
    bool m_IsDragging; //!< ドラッグしてスクロール中なら true です。
    bool m_IsShowingOriginal; //!< 元画像を表示中なら true です。
    int m_UpdateTimerCount; //!< 更新までのタイマーのカウンタです。

    RECT m_MaxRect; //!< 縁を含まない最大画像領域です。クライアント座標での矩形です。
    RECT m_ItemRect; //!< 拡大後の表示領域です。クライアント座標での矩形です。
    int m_OriginalW; //!< 元画像の幅です。
    int m_OriginalH; //!< 元画像の高さです。
    int m_Zoom; //!< 拡大率です。0 以下の場合 2^(-1 + zoom) 倍となります。
    int m_MinZoom; //!< 拡大率の最小値です。
    int m_MaxZoom; //!< 拡大率の最大値です。
    int m_Ix; //!< 現在の水平方向のスクロール値です。
    int m_Iy; //!< 現在の垂直方向のスクロール値です。
    int m_MaxIx; //!< 水平方向の最大スクロール値です。
    int m_MaxIy; //!< 水平方向の最大スクロール値です。

    // windows
    HBITMAP m_hWinBmp; //!< ビットマップハンドルです。
    void* m_pWinBmpPixels; //!< ビットマップのピクセルデータです。
    int m_ZoomInId; //!< ズームインボタンの ID です。
    int m_ZoomOutId; //!< ズームアウトボタンの ID です。
    int m_ZoomTextId; //!< 拡大率テキストの ID です。

public:
    //! コンストラクタです。
    RPreview()
    : m_DisplaysRgb(true),
      m_DisplaysAlpha(false),
      m_IsDragging(false),
      m_IsShowingOriginal(false),
      m_UpdateTimerCount(0),
      m_hWinBmp(NULL),
      m_pWinBmpPixels(NULL),
      m_ZoomInId(0),
      m_ZoomOutId(0),
      m_ZoomTextId(0)
    {
    }

    //! デストラクタです。
    virtual ~RPreview()
    {
        //RNoteTrace("~RPreview()");
        FreeMemory();
    }

    //! メモリを解放します。
    void FreeMemory()
    {
        //RFreeAndClearArray(m_DataBuf);
        if (m_hWinBmp != NULL)
        {
            DeleteObject(m_hWinBmp);
            m_hWinBmp = NULL;
            m_pWinBmpPixels = NULL;
        }
    }

    //! @brief メモリを初期化します。
    //!
    //! @param[in] hDlg ダイアログハンドルです。
    //! @param[in] totalRect 縁を含む最大画像領域です。クライアント座標での矩形を指定します。
    //! @param[in] originalW 元画像の幅です。
    //! @param[in] originalH 元画像の高さです。
    //!
    void InitMemory(
        HWND hDlg,
        const RECT& totalRect,
        const int originalW,
        const int originalH
    );

    //! @brief コントロール ID を設定します。
    //!
    //! @param[in] zoomInId ズームインボタンの ID です。
    //! @param[in] zoomOutId ズームアウトボタンの ID です。
    //! @param[in] zoomTextId 拡大率テキストの ID です。
    //!
    void SetControlId(
        const int zoomInId,
        const int zoomOutId,
        const int zoomTextId
    )
    {
        m_ZoomInId   = zoomInId;
        m_ZoomOutId  = zoomOutId;
        m_ZoomTextId = zoomTextId;
    }

    //! 拡大後の表示領域の幅を返します。
    const int GetItemW() const { return (m_ItemRect.right  - m_ItemRect.left); }

    //! 拡大後の表示領域の高さを返します。
    const int GetItemH() const { return (m_ItemRect.bottom - m_ItemRect.top ); }

    //! 拡大後の幅を返します。
    int GetZoomedW(const int srcW) const
    {
        return (m_Zoom >= 1) ? srcW * m_Zoom : RMax(srcW >> (1 - m_Zoom), 1);
    }

    //! 拡大後の高さを返します。
    int GetZoomedH(const int srcH) const
    {
        return (m_Zoom >= 1) ? srcH * m_Zoom : RMax(srcH >> (1 - m_Zoom), 1);
    }

    //! @brief 拡大率を変更します。
    //!
    //! @param[in] hDlg ダイアログハンドルです。
    //! @param[in] zoomStep 拡大率の増分です。縮小なら負の値を指定します。
    //!
    void ChangeZoom(HWND hDlg, const int zoomStep);

    //! @brief スクロール位置を変更します。
    //!
    //! @param[in] hDlg ダイアログハンドルです。
    //! @param[in] ix 水平方向のスクロール値です。
    //! @param[in] iy 垂直方向のスクロール値です。
    //!
    void ChangeScroll(HWND hDlg, const int ix, const int iy);

    //! 画像の中央が表示されるようにスクロール位置を設定します。
    void CenterScroll();

    //! スクロール可能なら true を返します。
    bool IsScrollEnable() const { return (m_MaxIx > 0 || m_MaxIy > 0); }

    //! スクロール値をクランプします。
    void ClampScrollValue()
    {
        m_Ix = RClampValue(static_cast<int>(0), m_MaxIx, m_Ix);
        m_Iy = RClampValue(static_cast<int>(0), m_MaxIy, m_Iy);
    }

    //! @brief Windows ビットマップを更新します。
    //!
    //! @param[in] hDlg ダイアログハンドルです。
    //! @param[in] pSrcBitmap ビットマップデータです。
    //!
    void UpdateWinBmp(HWND hDlg, const void* pSrcBitmap);

    //! @brief ペイントします。
    //!
    //! @param[in] hDlg ダイアログハンドルです。
    //!
    void Paint(HWND hDlg) const;

    //! @brief コントロールの状態を更新します。
    //!
    //! @param[in] hDlg ダイアログハンドルです。
    //!
    void UpdateControl(HWND hDlg) const;

protected:
    //! 拡大後の表示領域とスクロールの範囲を設定します。
    void SetItemRect();

    //! @brief 領域を更新リージョンに追加します。
    //!
    //! @param[in] hDlg ダイアログハンドルです。
    //!
    void InvalidateArea(HWND hDlg) const;

    //! @brief 表示領域のまわりの領域をダイアログの背景色でフィルします。
    //!
    //! @param[in] hDC デバイスコンテキストハンドルです。
    //!
    void FillAroundItem(HDC& hDC) const;

    //! @brief Windows ビットマップを初期化します。
    //!
    //! @param[in] hDlg ダイアログハンドルです。
    //!
    void InitWinBmp(HWND hDlg);
};

//=============================================================================
//! @brief ローカルプログレスのクラスです。
//=============================================================================
class RLocalProgress : public RProgress
{
public:
    //! @brief コンストラクタです。
    //!
    //! @param[in] total 処理全体のカウント数です。
    //! @param[in] dialogId プログレスダイアログのダイアログ ID です。
    //! @param[in] hParent 親ウィンドウのウィンドウハンドルです。
    //! @param[in] progressId プログレスバーの ID です
    //! @param[in] textId テキストコントロールの ID です。使用しない場合は 0 を指定します。
    //! @param[in] text ダイアログに表示するテキストです。NULL ならリソースで指定されたテキストを表示します。
    //!
    RLocalProgress(
        const int total,
        const int dialogId,
        HWND hParent,
        const int progressId,
        const int textId = 0,
        const char* text = NULL
    );

    //! デストラクタです。
    virtual ~RLocalProgress()
    {
        End();
    }

    //! ダイアログを終了します。
    void End();

    //! @brief 更新します。
    //!
    //! @param[in] step 現在の進行カウントに加算する値です。
    //!
    //! @return 処理を続行するなら true、中断するなら false を返します。
    //!
    virtual bool Update(const int step = 1);

    //! プログレス開始時のクロックをリセットします。
    void ResetStartClock()
    {
        m_StartClock = std::clock();
    }

    //! ダイアログプロシージャです。
    static INT_PTR CALLBACK DialogProc(HWND hDlg, UINT uMsg, WPARAM wParam, LPARAM lParam);

protected:
    //! @brief 親ウィンドウの有効状態を設定します。
    //!
    //! @param[in] enables 有効なら true を指定します。
    //!
    void EnableParent(const bool enables) const
    {
        if (m_hParent != NULL && IsWindow(m_hParent))
        {
            EnableWindow(m_hParent, enables);
        }
    }

protected:
    HWND m_hDialog; //!< ダイアログハンドルです。
    HWND m_hParent; //!< 親ウィンドウのウィンドウハンドルです。
    int m_ProgressId; //!< プログレスバーのコントロール ID です。
    std::clock_t m_StartClock; //!< プログレス開始時のクロックです。
    bool m_IsDialogShown; //!< ダイアログ表示済みなら true です。
};

//=============================================================================
// nps ネームスペースを終了します。
//=============================================================================
} // namespace nps
} // namespace tool
} // namespace gfx
} // namespace nn

