﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

// SetBitmapDataToPhotoshop GetBitmapDataFromPhotoshop

// SetAddInfoForWrite GetImageStatusForWrite

// SetFormatResourceData GetFormatResourceData

// RUpdatePreviewDataBuf

//=============================================================================
// include
//=============================================================================
#ifndef NPS_CLONE_SOURCE
    #include "Globals.h"
    #include "NpsVersion.h"
    #include "NpsFormatUI.h"
#endif

//=============================================================================
// nps ネームスペースを開始します。
//=============================================================================
namespace nn {
namespace gfx {
namespace tool {
namespace nps {

//-----------------------------------------------------------------------------
//! @brief テクスチャフォーマットに対応したデフォルトのヒント情報を取得します。
//-----------------------------------------------------------------------------
std::string RGetDefaultHint(const FtxFormat format)
{
    return (format == FtxFormat_Snorm_Bc5 || format == FtxFormat_Snorm_Eac_11_11) ?
        HintValueNormal : "";
}

//-----------------------------------------------------------------------------
//! @brief 復帰情報を設定します。
//-----------------------------------------------------------------------------
void RSetRevertInfo(GPtr globals)
{
    const int REVERT_INFO_SIZE = 4; // dummy

    //-----------------------------------------------------------------------------
    // alloc
    if (gStuff->revertInfo == NULL)
    {
        gStuff->revertInfo = gStuff->handleProcs->newProc(REVERT_INFO_SIZE);
    }

    //-----------------------------------------------------------------------------
    // set
    if (gStuff->revertInfo != NULL)
    {
        char* info = gStuff->handleProcs->lockProc(gStuff->revertInfo, false);
        memset(info, 0xff, REVERT_INFO_SIZE);
        gStuff->handleProcs->unlockProc(gStuff->revertInfo);
    }
}

//-----------------------------------------------------------------------------
//! @brief ライト時に Photoshop 上に存在するアルファチャンネル数を返します。
//!        レイヤーの透明度プレーンはカウントしません。
//-----------------------------------------------------------------------------
int RGetPsAlphaChanCount(GPtr globals)
{
    int planeSize = gStuff->planes;
    if (gStuff->transparencyPlane > 0)
    {
        // レイヤーがある場合、gStuff->planes には
        // transparency plane もカウントされているので減らす
        if (gStuff->imageMode == plugInModeGrayScale ||
            gStuff->imageMode == plugInModeRGBColor)
        {
            --planeSize;
        }
    }

    if (gStuff->imageMode == plugInModeGrayScale)
    {
        return (planeSize >= 2) ? planeSize - 1 : 0;
    }
    else if (gStuff->imageMode == plugInModeIndexedColor)
    {
        return (planeSize >= 2) ? planeSize - 1 : 0;
    }
    else if (gStuff->imageMode == plugInModeRGBColor)
    {
        return (planeSize >= 4) ? planeSize - 3 : 0;
    }
    else
    {
        return 0;
    }
}

//-----------------------------------------------------------------------------
//! @brief ライト時に Photoshop から取得するアルファプレーンのインデックスを返します。
//!        背景がなくレイヤーのみの画像の場合、アルファチャンネルがなければ
//!        レイヤーの透明度プレーンを返します。
//-----------------------------------------------------------------------------
int RGetPsAlphaPlaneIndex(const GPtr globals)
{
    if (gStuff->transparencyPlane > 0) // 背景がなくレイヤーのみ
    {
        if (gStuff->imageMode == plugInModeGrayScale ||
            gStuff->imageMode == plugInModeRGBColor)
        {
            if (gStuff->transparencyPlane + 1 < gStuff->planes) // アルファあり
            {
                return gStuff->transparencyPlane + 1;
            }
            else // アルファなし
            {
                return gStuff->transparencyPlane;
            }
        }
    }

    if (gStuff->imageMode == plugInModeGrayScale ||
        gStuff->imageMode == plugInModeIndexedColor)
    {
        return (gStuff->planes >= 2) ? 1 : 0;
    }
    else
    {
        return (gStuff->planes >= 4) ? 3 : 0;
    }
}

//-----------------------------------------------------------------------------
//! @brief インデックスカラーモードでカラーテーブルに透明色があれば true を返します。
//!
//! @param[in] globals グローバルデータです。
//!
//! @return インデックスカラーモードでカラーテーブルに透明色があれば true を返します。
//-----------------------------------------------------------------------------
static bool IsPsTransparentIndexUsed(GPtr globals)
{
    return (gStuff->imageMode == plugInModeIndexedColor &&
            gStuff->transparentIndex <= 0xff);
}

//-----------------------------------------------------------------------------
//! @brief 現在の設定におけるテクスチャの実データサイズを取得します。
//!        中間ファイルに出力されるデータはタイリングされるので、
//!        この実データサイズより大きくなる場合があります。
//-----------------------------------------------------------------------------
size_t RAddInfo::GetDataSize(const int imageW, const int imageH, const int imageD) const
{
    const bool isAstc = RIsAstcTextureFormat(m_Format);
    const size_t bpp = RGetBitsPerPixel(m_Format);
    int minW;
    int minH;
    RGetMinimumWH(minW, minH, m_Format);

    size_t dataSize = 0;
    for (int level = 0; level < m_MipLevel; ++level)
    {
        const int curW = RMax(imageW >> level, 1);
        const int curH = RMax(imageH >> level, 1);
        const int curD = (m_Dimension == FtxDimension_3d) ?
            RMax(imageD >> level, 1) : imageD;
        if (isAstc)
        {
            const size_t blockCountX = (curW + minW - 1) / minW;
            const size_t blockCountY = (curH + minH - 1) / minH;
            dataSize += AstcBlockBytes * blockCountX * blockCountY * curD;
        }
        else
        {
            const int dstW = RAlignValue(curW, minW);
            const int dstH = RAlignValue(curH, minH);
            dataSize += bpp * dstW * dstH * curD / 8;
        }
        if (curW == 1 && curH == 1 &&
            (curD == 1 || m_Dimension != FtxDimension_3d))
        {
            break;
        }
    }
    return dataSize;
}

//-----------------------------------------------------------------------------
//! @brief RGBA データから行バッファを設定します。
//!        カラーインデックスは非対応です。
//!
//! @param[out] pDst 格納先の行バッファです。
//! @param[in] pSrc RGBA データです。
//! @param[in] globals グローバルデータです。
//! @param[in] pixelCount ピクセル数です。
//-----------------------------------------------------------------------------
static void SetRowBufFromRgba(
    uint8_t* pDst,
    const uint8_t* pSrc,
    const GPtr globals,
    const int pixelCount
)
{
    if (gStuff->depth == 8)
    {
        if (gStuff->imageMode == plugInModeGrayScale)
        {
            for (int iPix = 0; iPix < pixelCount; ++iPix)
            {
                *pDst++ = pSrc[0];
                if (gStuff->planes == 2) // IA
                {
                    *pDst++ = pSrc[3];
                }
                pSrc += R_RGBA_BYTES;
            }
        }
        else if (gStuff->imageMode == plugInModeIndexedColor)
        {
            // 非対応
        }
        else // RGBColor
        {
            if (gStuff->planes == 3) // RGB
            {
                for (int iPix = 0; iPix < pixelCount; ++iPix)
                {
                    *pDst++ = pSrc[0];
                    *pDst++ = pSrc[1];
                    *pDst++ = pSrc[2];
                    pSrc += R_RGBA_BYTES;
                }
            }
            else // RGBA
            {
                memcpy(pDst, pSrc, pixelCount * R_RGBA_BYTES);
            }
        }
    }
    else if (gStuff->depth == 32)
    {
        const float* pSrcF32 = reinterpret_cast<const float*>(pSrc);
        float* pDstF32 = reinterpret_cast<float*>(pDst);
        if (gStuff->imageMode == plugInModeGrayScale)
        {
            for (int iPix = 0; iPix < pixelCount; ++iPix)
            {
                *pDstF32++ = pSrcF32[0];
                if (gStuff->planes == 2) // IA
                {
                    *pDstF32++ = pSrcF32[3];
                }
                pSrcF32 += R_RGBA_COUNT;
            }
        }
        else // RGBColor
        {
            if (gStuff->planes == 3) // RGB
            {
                for (int iPix = 0; iPix < pixelCount; ++iPix)
                {
                    *pDstF32++ = pSrcF32[0];
                    *pDstF32++ = pSrcF32[1];
                    *pDstF32++ = pSrcF32[2];
                    pSrcF32 += R_RGBA_COUNT;
                }
            }
            else // RGBA
            {
                memcpy(pDstF32, pSrcF32, pixelCount * R_RGBA_FLOAT_BYTES);
            }
        }
    }
}

//-----------------------------------------------------------------------------
//! @brief 行バッファから RGBA データを設定します。
//!
//! @param[out] pDst 格納先の RGBA データです。
//! @param[in] pSrc 行バッファです。
//! @param[in] globals グローバルデータです。
//! @param[in] pixelCount ピクセル数です。
//! @param[in] alphaPlaneIdx 行バッファ中のアルファプレーンのインデックスです。
//-----------------------------------------------------------------------------
static void GetRgbaFromRowBuf(
    uint8_t* pDst,
    const uint8_t* pSrc,
    const GPtr globals,
    const int pixelCount,
    const int alphaPlaneIdx
)
{
    if (gStuff->depth == 8)
    {
        if (gStuff->imageMode == plugInModeGrayScale)
        {
            for (int iPix = 0; iPix < pixelCount; ++iPix)
            {
                pDst[0] = pDst[1] = pDst[2] = pSrc[0];
                pDst[3] = (alphaPlaneIdx != 0) ? pSrc[alphaPlaneIdx] : 0xff;
                pDst += R_RGBA_BYTES;
                pSrc += gStuff->colBytes;
            }
        }
        else if (gStuff->imageMode == plugInModeIndexedColor)
        {
            for (int iPix = 0; iPix < pixelCount; ++iPix)
            {
                const int idx = pSrc[0];
                *pDst++ = gStuff->redLUT  [idx];
                *pDst++ = gStuff->greenLUT[idx];
                *pDst++ = gStuff->blueLUT [idx];
                if (alphaPlaneIdx != 0)
                {
                    *pDst++ = pSrc[alphaPlaneIdx];
                }
                else // use transaprent color of LUT
                {
                    *pDst++ = (idx == gStuff->transparentIndex) ? 0x00 : 0xff;
                }
                pSrc += gStuff->colBytes;
            }
        }
        else // RGBColor
        {
            for (int iPix = 0; iPix < pixelCount; ++iPix)
            {
                *pDst++ = pSrc[0];
                *pDst++ = pSrc[1];
                *pDst++ = pSrc[2];
                *pDst++ = (alphaPlaneIdx != 0) ? pSrc[alphaPlaneIdx] : 0xff;
                pSrc += gStuff->colBytes;
            }
        }
    }
    else if (gStuff->depth == 32)
    {
        const float* pSrcF32 = reinterpret_cast<const float*>(pSrc);
        float* pDstF32 = reinterpret_cast<float*>(pDst);
        if (gStuff->imageMode == plugInModeGrayScale)
        {
            for (int iPix = 0; iPix < pixelCount; ++iPix)
            {
                pDstF32[0] = pDstF32[1] = pDstF32[2] = pSrcF32[0];
                pDstF32[3] = (alphaPlaneIdx != 0) ? pSrcF32[alphaPlaneIdx] : 1.0f;
                pDstF32 += R_RGBA_COUNT;
                pSrcF32 += gStuff->colBytes / sizeof(float);
            }
        }
        else // RGBColor
        {
            for (int iPix = 0; iPix < pixelCount; ++iPix)
            {
                *pDstF32++ = pSrcF32[0];
                *pDstF32++ = pSrcF32[1];
                *pDstF32++ = pSrcF32[2];
                *pDstF32++ = (alphaPlaneIdx != 0) ? pSrcF32[alphaPlaneIdx] : 1.0f;
                pSrcF32 += gStuff->colBytes / sizeof(float);
            }
        }
    }
}

//-----------------------------------------------------------------------------
//! @brief ビットマップデータを Photoshop に設定します。
//!        カラーインデックスは非対応です。
//-----------------------------------------------------------------------------
void SetBitmapDataToPhotoshop(GPtr globals, const void* pBitmapData)
{
    gStuff->loPlane    = 0;
    gStuff->hiPlane    = gStuff->planes - 1;
    gStuff->colBytes   = gStuff->planes * gStuff->depth / 8;
    gStuff->rowBytes   = gStuff->colBytes * gStuff->imageSize32.h;
    gStuff->planeBytes = gStuff->depth / 8;
    gStuff->theRect32.left  = 0;
    gStuff->theRect32.right = gStuff->imageSize32.h;
    uint8_t* rowBuf = new uint8_t[gStuff->rowBytes];
    gStuff->data = rowBuf;

    const int srcRowBytes = gStuff->imageSize32.h * 4 * gStuff->depth / 8;
    const uint8_t* pSrc = reinterpret_cast<const uint8_t*>(pBitmapData);
    RProgress progress(gStuff->imageSize32.v, gStuff->progressProc);
    for (int iy = 0; iy < gStuff->imageSize32.v; ++iy)
    {
        SetRowBufFromRgba(rowBuf, pSrc, globals, gStuff->imageSize32.h);
        gStuff->theRect32.top    = iy;
        gStuff->theRect32.bottom = iy + 1;
        gResult = AdvanceState(); // memory to Photoshop
        if (gResult != noErr)
        {
            break;
        }
        pSrc += srcRowBytes;
        progress.Update();
    }
    delete[] rowBuf;
}

//-----------------------------------------------------------------------------
//! @brief ビットマップデータを Photoshop から取得します。
//-----------------------------------------------------------------------------
void GetBitmapDataFromPhotoshop(void* pBitmapData, GPtr globals)
{
    //-----------------------------------------------------------------------------
    // ピクセルの透明度が不透明でない場合にコンポジットチャンネルに
    // 合成する色を指定します。
    //gStuff->transparencyMatting = 0; // 0:合成なし（デフォルト）, 1:黒, 2:灰色, 3:白

    //-----------------------------------------------------------------------------
    // 取得するプレーン数を決定します。
    int planesMax = (gStuff->imageMode == plugInModeRGBColor) ? 4 : 2;
    if (gStuff->transparencyPlane > 0)
    {
        // レイヤーがある場合、RGB チャンネルの次に transparency plane が入るので
        // 最大プレーン数を 1 増やします。
        if (gStuff->imageMode == plugInModeGrayScale ||
            gStuff->imageMode == plugInModeRGBColor)
        {
            ++planesMax;
        }
    }
    const int planesGet = (gStuff->planes <= planesMax) ?
        gStuff->planes : planesMax;

    //-----------------------------------------------------------------------------
    // Photoshop から取得します。
    gStuff->loPlane    = 0;
    gStuff->hiPlane    = static_cast<short>(planesGet - 1);
    gStuff->colBytes   = static_cast<short>(planesGet * gStuff->depth / 8);
    gStuff->rowBytes   = gStuff->colBytes * gStuff->imageSize32.h;
    gStuff->planeBytes = gStuff->depth / 8;
    gStuff->theRect32.left  = 0;
    gStuff->theRect32.right = gStuff->imageSize32.h;
    uint8_t* rowBuf = new uint8_t[gStuff->rowBytes];
    gStuff->data = rowBuf;

    const int alphaPlaneIdx = RGetPsAlphaPlaneIndex(globals);
    const int dstRowBytes = gStuff->imageSize32.h * 4 * gStuff->depth / 8;
    uint8_t* pDst = reinterpret_cast<uint8_t*>(pBitmapData);
    for (int iy = 0; iy < gStuff->imageSize32.v; ++iy)
    {
        gStuff->theRect32.top    = iy;
        gStuff->theRect32.bottom = iy + 1;
        gResult = AdvanceState(); // Photoshop to memory
        if (gResult != noErr)
        {
            break;
        }
        GetRgbaFromRowBuf(pDst, rowBuf, globals, gStuff->imageSize32.h, alphaPlaneIdx);
        pDst += dstRowBytes;
    }
    delete[] rowBuf;
}

//-----------------------------------------------------------------------------
//! @brief 隣接する 4 ピクセルを取得します。
//!
//! @param[out] adjs 隣接する 4 ピクセルの RGBA データを格納します。
//! @param[in] pBitmapData ビットマップデータです。
//! @param[in] ix 中央のピクセルの X 座標です。
//! @param[in] iy 中央のピクセルの Y 座標です。
//! @param[in] imageW 画像の幅です。
//! @param[in] imageH 画像の高さです。
//! @param[in] isDiagonal 上下左右のピクセルを取得するなら false、斜め 4 方向のピクセルを取得するなら true を指定します。
//!
//! @return アルファが 0 でない隣接ピクセルがあれば true を返します。
//-----------------------------------------------------------------------------
template <typename T>
bool GetAdjacent4Pixels(
    T adjs[4][R_RGBA_COUNT],
    const T* pBitmapData,
    const int ix,
    const int iy,
    const int imageW,
    const int imageH,
    const bool isDiagonal
)
{
    const int POS_COUNT = 4;

    //   0
    // 1 * 2
    //   3
    static const int poss0[POS_COUNT][2] =
    {
        {  0, -1 }, // 0
        { -1,  0 }, // 1
        {  1,  0 }, // 2
        {  0,  1 }, // 3
    };

    // 0   1
    //   *
    // 2   3
    static const int poss1[POS_COUNT][2] =
    {
        { -1, -1 }, // 0
        {  1, -1 }, // 1
        { -1,  1 }, // 2
        {  1,  1 }, // 3
    };

    const int pixelBytes = sizeof(T) * R_RGBA_COUNT;
    const int rowOfs = imageW * R_RGBA_COUNT;
    const int (*pPoss)[2] = (!isDiagonal) ? poss0 : poss1;

    if (0 < ix && ix < imageW - 1 &&
        0 < iy && iy < imageH - 1)
    {
        // 中央のピクセルが画像の端でない場合
        for (int iPos = 0; iPos < POS_COUNT; ++iPos)
        {
            const int px = ix + pPoss[iPos][0];
            const int py = iy + pPoss[iPos][1];
            memcpy(adjs[iPos], pBitmapData + px * R_RGBA_COUNT + py * rowOfs, pixelBytes);
        }
    }
    else
    {
        // 中央のピクセルが画像の端の場合
        memset(adjs, 0x00, pixelBytes * POS_COUNT);
        for (int iPos = 0; iPos < POS_COUNT; ++iPos)
        {
            const int px = ix + pPoss[iPos][0];
            const int py = iy + pPoss[iPos][1];
            if (0 <= px && px < imageW &&
                0 <= py && py < imageH)
            {
                memcpy(adjs[iPos], pBitmapData + px * R_RGBA_COUNT + py * rowOfs, pixelBytes);
            }
        }
    }

    return (
        adjs[0][3] != 0 ||
        adjs[1][3] != 0 ||
        adjs[2][3] != 0 ||
        adjs[3][3] != 0);
}

//-----------------------------------------------------------------------------
//! @brief 透明なピクセルの RGB 値を補整します（テンプレート）。
//!
//! @param[in,out] pBitmapData ビットマップデータです。
//! @param[in] imageW 画像の幅です。
//! @param[in] imageH 画像の高さです。
//! @param[in] clamps 値を [0, 255] の範囲にクランプするなら true を指定します。
//-----------------------------------------------------------------------------
template <typename T>
void AdjustTransparentPixelRgbSub(
    T* pBitmapData,
    const int imageW,
    const int imageH,
    const bool clamps
)
{
    T* pCur = pBitmapData;
    T adjs[4][R_RGBA_COUNT];
    for (int iy = 0; iy < imageH; ++iy)
    {
        for (int ix = 0; ix < imageW; ++ix)
        {
            if (pCur[3] == 0x00)
            {
                for (int iTry = 0; iTry < 2; ++iTry)
                {
                    const bool isDiagonal = (iTry == 1);
                    if (GetAdjacent4Pixels(adjs, pBitmapData, ix, iy, imageW, imageH, isDiagonal))
                    {
                        const float alphaSum = static_cast<float>(
                            adjs[0][3] +
                            adjs[1][3] +
                            adjs[2][3] +
                            adjs[3][3]);
                        const float alphaSumInv = 1.0f / alphaSum;
                        if (clamps)
                        {
                            for (int iRgb = 0; iRgb < R_RGB_COUNT; ++iRgb)
                            {
                                const float sum =
                                    adjs[0][iRgb] * static_cast<float>(adjs[0][3]) +
                                    adjs[1][iRgb] * static_cast<float>(adjs[1][3]) +
                                    adjs[2][iRgb] * static_cast<float>(adjs[2][3]) +
                                    adjs[3][iRgb] * static_cast<float>(adjs[3][3]);
                                const float average = sum * alphaSumInv;
                                pCur[iRgb] = static_cast<T>(RClampValue(0x00, 0xff, RRound(average)));
                            }
                        }
                        else
                        {
                            for (int iRgb = 0; iRgb < R_RGB_COUNT; ++iRgb)
                            {
                                const float sum =
                                    adjs[0][iRgb] * static_cast<float>(adjs[0][3]) +
                                    adjs[1][iRgb] * static_cast<float>(adjs[1][3]) +
                                    adjs[2][iRgb] * static_cast<float>(adjs[2][3]) +
                                    adjs[3][iRgb] * static_cast<float>(adjs[3][3]);
                                const float average = sum * alphaSumInv;
                                pCur[iRgb] = static_cast<T>(average);
                            }
                        }
                        break;
                    }
                }
            }
            pCur += R_RGBA_COUNT;
        }
    }
}

//-----------------------------------------------------------------------------
//! @brief 透明なピクセルの RGB 値を補整します。
//!        レイヤーのピクセルの透明度をアルファとしてセーブする場合に、
//!        バイリニア補間による透明ピクセルの白色のにじみを軽減するための処理です。
//-----------------------------------------------------------------------------
void AdjustTransparentPixelRgb(void* pBitmapData, GPtr globals)
{
    //-----------------------------------------------------------------------------
    // インデックスカラーモードの場合、RGB 値はカラーテーブルの色が反映されるので補整しません。
    // アルファチャンネルが存在する場合は補整しません。
    if (gStuff->imageMode == plugInModeIndexedColor ||
        RGetPsAlphaChanCount(globals) != 0)
    {
        return;
    }

    //-----------------------------------------------------------------------------
    // 透明な（アルファが 0）ピクセルの RGB 値に、
    // 隣接する透明でない（アルファ > 0）ピクセルの RGB 値の平均を設定します。
    const int imageW = gStuff->imageSize32.h;
    const int imageH = gStuff->imageSize32.v;
    //RTimeMeasure tm1;
    if (gStuff->depth == 8)
    {
        AdjustTransparentPixelRgbSub(reinterpret_cast<uint8_t*>(pBitmapData), imageW, imageH, true);
    }
    else if (gStuff->depth == 32)
    {
        AdjustTransparentPixelRgbSub(reinterpret_cast<float*>(pBitmapData), imageW, imageH, false);
    }
    //RNoteTrace("adjust xpa rgb: %.2f ms", tm1.GetMilliSec());
}

//-----------------------------------------------------------------------------
//! @brief ファイルサイズを取得します。
//-----------------------------------------------------------------------------
size_t RGetFileSize(const intptr_t dataFork)
{
    #ifdef _M_IX86
    return GetFileSize(reinterpret_cast<HANDLE>(dataFork), NULL);
    #else
    size_t fileSize = 0;
    GetFileSizeEx(reinterpret_cast<HANDLE>(dataFork),
        reinterpret_cast<PLARGE_INTEGER>(&fileSize));
    return fileSize;
    #endif
}

//-----------------------------------------------------------------------------
//! @brief 指定したバイト数リードします。
//-----------------------------------------------------------------------------
OSErr RReadSome(const intptr_t dataFork, void* pBuffer, const size_t size)
{
    const DWORD MAX_READ_BYTES = 0x40000000; // 1GB
    uint8_t* pDst = reinterpret_cast<uint8_t*>(pBuffer);
    size_t remainBytes = size;
    while (remainBytes > 0)
    {
        const DWORD readBytes = (remainBytes >= MAX_READ_BYTES) ?
            MAX_READ_BYTES : static_cast<DWORD>(remainBytes);
        DWORD retBytes = 0;
        if (!ReadFile(reinterpret_cast<HANDLE>(dataFork), pDst, readBytes, &retBytes, NULL) ||
            retBytes != readBytes)
        {
            return readErr;
        }
        pDst += retBytes;
        remainBytes -= retBytes;
    }
    return noErr;
}

//-----------------------------------------------------------------------------
//! @brief 指定したバイト数ライトします。
//-----------------------------------------------------------------------------
OSErr RWriteSome(const intptr_t dataFork, const void* pBuffer, const size_t size)
{
    const DWORD MAX_WRITE_BYTES = 0x40000000; // 1GB
    const uint8_t* pSrc = reinterpret_cast<const uint8_t*>(pBuffer);
    size_t remainBytes = size;
    while (remainBytes > 0)
    {
        const DWORD writeBytes = (remainBytes >= MAX_WRITE_BYTES) ?
            MAX_WRITE_BYTES : static_cast<DWORD>(remainBytes);
        DWORD retBytes = 0;
        if (!WriteFile(reinterpret_cast<HANDLE>(dataFork), pSrc, writeBytes, &retBytes, NULL) ||
            retBytes != writeBytes)
        {
            return writErr;
        }
        pSrc += retBytes;
        remainBytes -= retBytes;
    }
    return noErr;
}

//-----------------------------------------------------------------------------
//! @brief ファイル形式プラグイン用のリソースデータをドキュメントに設定します。
//!        リード時とライト時に呼ばれます。
//-----------------------------------------------------------------------------
void SetFormatResourceData(GPtr globals, const bool atWriteFlag)
{
    //-----------------------------------------------------------------------------
    // リード時に付加情報がなければリソースデータを設定しません。
    if (!atWriteFlag && g_AddInfo.m_Format == FtxFormat_Invalid)
    {
        return;
    }

    //-----------------------------------------------------------------------------
    // ライト時は以前のリソースデータをすべて削除します。
    if (atWriteFlag)
    {
        RDeleteResourceDataAll(globals);

        // テクスチャエクスポートプラグインから呼ばれた場合は、
        // PSD ファイルにリソースデータが含まれないようにするため、
        // リソースデータを設定しません。
        if (g_AddInfo.m_IsExportTexture)
        {
            return;
        }
    }

    //-----------------------------------------------------------------------------
    // RD version
    RAddRDBlock(globals, RD_TAG_RD_VER, RGetNumberString(RD_VER_CURRENT));

    //-----------------------------------------------------------------------------
    // プリセット名（コンフィグ名）
    RAddRDBlock(globals, RD_TAG_DCC_PRESET, g_AddInfo.m_ConfigName);

    //-----------------------------------------------------------------------------
    // ヒント情報
    RAddRDBlock(globals, RD_TAG_HINT, g_AddInfo.m_Hint);

    //-----------------------------------------------------------------------------
    // リニア変換フラグ
    RAddRDBlock(globals, RD_TAG_LINEAR, RGetOptStringFromLinearFlag(g_AddInfo.m_LinearFlag));

    //-----------------------------------------------------------------------------
    // dimension
    RAddRDBlock(globals, RD_TAG_DIMENSION, RGetTextureDimensionString(g_AddInfo.m_Dimension));

    //-----------------------------------------------------------------------------
    // format
    RAddRDBlock(globals, RD_TAG_FORMAT, RGetTextureFormatString(g_AddInfo.m_Format));

    //-----------------------------------------------------------------------------
    // weighted compress
    RAddRDBlock(globals, RD_TAG_WEIGHT_CMPR, RGetNumberString(g_AddInfo.m_WeightedCompress ? 1 : 0));

    //-----------------------------------------------------------------------------
    // 初期スウィズル値
    RAddRDBlock(globals, RD_TAG_INI_SWIZZLE, RGetNumberString(g_AddInfo.m_InitialSwizzle));

    //-----------------------------------------------------------------------------
    // ミップマップのレベル数
    if (g_AddInfo.m_MipLevel >= 2)
    {
        RAddRDBlock(globals, RD_TAG_MIP_LEVEL, RGetNumberString(g_AddInfo.m_MipLevel));
    }

    //-----------------------------------------------------------------------------
    // ミップマップ生成フィルタ
    RAddRDBlock(globals, RD_TAG_MIP_GEN_FILTER, RGetMipGenFilterString(g_AddInfo.m_MipGenFilter));

    //-----------------------------------------------------------------------------
    // comp sel
    RAddRDBlock(globals, RD_TAG_COMP_SEL, RGetCompSelOptionString(g_AddInfo.m_CompSel));

    //-----------------------------------------------------------------------------
    // original paths
    if (!g_AddInfo.m_OriginalPaths.empty())
    {
        RAddRDBlock(globals, RD_TAG_ORG_PATHS, RJoinStrings(g_AddInfo.m_OriginalPaths, ";"));
    }

    //-----------------------------------------------------------------------------
    // create info
    if (!g_AddInfo.m_CreateInfo.empty())
    {
        RAddRDBlock(globals, RD_TAG_CREATE_INFO, g_AddInfo.m_CreateInfo);
    }

    //-----------------------------------------------------------------------------
    // user data
    if (g_AddInfo.m_pPackedUserData != NULL)
    {
        RAddRDBlock(globals, RD_TAG_USER_DATA, g_AddInfo.m_pPackedUserData, g_AddInfo.m_PackedUserDataSize);
    }

    //-----------------------------------------------------------------------------
    // 編集用コメント
    if (!g_AddInfo.m_CommentLabel.empty())
    {
        RAddRDBlock(globals, RD_TAG_COMMENT_LABEL, g_AddInfo.m_CommentLabel);
    }

    if (!g_AddInfo.m_CommentCol.empty())
    {
        RAddRDBlock(globals, RD_TAG_COMMENT_COLOR, g_AddInfo.m_CommentCol);
    }

    if (!g_AddInfo.m_CommentText.empty())
    {
        RAddRDBlock(globals, RD_TAG_COMMENT_TEXT, g_AddInfo.m_CommentText);
    }

    //-----------------------------------------------------------------------------
    // tool data
    if (!g_AddInfo.m_ToolData.empty())
    {
        RAddRDBlock(globals, RD_TAG_TOOL_DATA, g_AddInfo.m_ToolData);
    }

    if (!g_AddInfo.m_UserToolData.empty())
    {
        RAddRDBlock(globals, RD_TAG_USER_TOOL_DATA, g_AddInfo.m_UserToolData);
    }
}

//-----------------------------------------------------------------------------
//! @brief プリセット名（コンフィグ名）のリソースデータを解析します。
//-----------------------------------------------------------------------------
static void ParseRDDccPreset(GPtr globals, const uint8_t* dataBuf, const uint32_t dataSize)
{
    g_AddInfo.m_ConfigName = (dataSize != 0) ?
        std::string(reinterpret_cast<const char*>(dataBuf), dataSize) : std::string();
    g_AddInfo.m_IsConfigNameGot = true;
}

//-----------------------------------------------------------------------------
//! @brief ヒント情報のリソースデータを解析します。
//-----------------------------------------------------------------------------
static void ParseRDHint(GPtr globals, const uint8_t* dataBuf, const uint32_t dataSize)
{
    g_AddInfo.m_Hint = (dataSize != 0) ?
        std::string(reinterpret_cast<const char*>(dataBuf), dataSize) : std::string();
}

//-----------------------------------------------------------------------------
//! @brief リニア変換フラグのリソースデータを解析します。
//-----------------------------------------------------------------------------
static void ParseRDLinearFlag(GPtr globals, const uint8_t* dataBuf, const uint32_t dataSize)
{
    const int linearFlag = RGetLinearFlagFromOptString(
        std::string(reinterpret_cast<const char*>(dataBuf), dataSize));
    if (linearFlag != RLinearFlag::LINEAR_INVALID)
    {
        g_AddInfo.m_LinearFlag = linearFlag;
    }
}

//-----------------------------------------------------------------------------
//! @brief 次元のリソースデータを解析します。
//-----------------------------------------------------------------------------
static void ParseRDDimension(GPtr globals, const uint8_t* dataBuf, const uint32_t dataSize)
{
    g_ImageStatus.m_LastDimension = RGetTextureDimensionFromString(
        std::string(reinterpret_cast<const char*>(dataBuf), dataSize));
}

//-----------------------------------------------------------------------------
//! @brief フォーマットのリソースデータを解析します。
//-----------------------------------------------------------------------------
static void ParseRDFormat(GPtr globals, const uint8_t* dataBuf, const uint32_t dataSize)
{
    g_ImageStatus.m_LastFormat = RGetTextureFormatFromString(
        std::string(reinterpret_cast<const char*>(dataBuf), dataSize));
}

//-----------------------------------------------------------------------------
//! @brief 重み付け圧縮のリソースデータを解析します。
//-----------------------------------------------------------------------------
static void ParseRDWeightedCompress(GPtr globals, const uint8_t* dataBuf, const uint32_t dataSize)
{
    g_AddInfo.m_WeightedCompress = (dataSize > 0 && dataBuf[0] != '0');
}

//-----------------------------------------------------------------------------
//! @brief 初期スウィズル値のリソースデータを解析します。
//-----------------------------------------------------------------------------
static void ParseRDInitialSwizzle(GPtr globals, const uint8_t* dataBuf, const uint32_t dataSize)
{
    const int initialSwizzle = atol(std::string(reinterpret_cast<const char*>(dataBuf), dataSize).c_str());
    if (RAddInfo::SWIZZLE_MIN <= initialSwizzle && initialSwizzle <= RAddInfo::SWIZZLE_MAX)
    {
        g_AddInfo.m_InitialSwizzle = initialSwizzle;
    }
}

//-----------------------------------------------------------------------------
//! @brief ミップマップのレベル数のリソースデータを解析します。
//-----------------------------------------------------------------------------
static void ParseRDMipLevel(GPtr globals, const uint8_t* dataBuf, const uint32_t dataSize)
{
    const int mipLevel = atol(std::string(reinterpret_cast<const char*>(dataBuf), dataSize).c_str());
    if (mipLevel > 0 && mipLevel <= RImage::MipCountMax)
    {
        g_ImageStatus.m_LastMipLevel = mipLevel;
    }
}

//-----------------------------------------------------------------------------
//! @brief ミップマップ生成フィルタのリソースデータを解析します。
//-----------------------------------------------------------------------------
static void ParseRDMipGenFilter(GPtr globals, const uint8_t* dataBuf, const uint32_t dataSize)
{
    g_AddInfo.m_MipGenFilter = RGetMipGenFilterFromString(
        std::string(reinterpret_cast<const char*>(dataBuf), dataSize), true);
}

//-----------------------------------------------------------------------------
//! @brief 成分選択のリソースデータを解析します。
//-----------------------------------------------------------------------------
static void ParseRDCompSel(GPtr globals, const uint8_t* dataBuf, const uint32_t dataSize)
{
    g_AddInfo.m_CompSel = RGetCompSelFromOptionString(
        std::string(reinterpret_cast<const char*>(dataBuf), dataSize));
}

//-----------------------------------------------------------------------------
//! @brief オリジナルパス配列のリソースデータを解析します。
//-----------------------------------------------------------------------------
static void ParseRDOriginalPaths(GPtr globals, const uint8_t* dataBuf, const uint32_t dataSize)
{
    RSeparateString(g_AddInfo.m_OriginalPaths,
        std::string(reinterpret_cast<const char*>(dataBuf), dataSize), ";");
}

//-----------------------------------------------------------------------------
//! @brief 作成情報のリソースデータを解析します。
//-----------------------------------------------------------------------------
static void ParseRDCreateInfo(GPtr globals, const uint8_t* dataBuf, const uint32_t dataSize)
{
    g_AddInfo.m_CreateInfo =
        std::string(reinterpret_cast<const char*>(dataBuf), dataSize);
}

//-----------------------------------------------------------------------------
//! @brief ユーザーデータのリソースデータを解析します。
//-----------------------------------------------------------------------------
static void ParseRDUserData(GPtr globals, const uint8_t* dataBuf, const uint32_t dataSize)
{
    g_AddInfo.m_pPackedUserData = new uint8_t[dataSize];
    g_AddInfo.m_PackedUserDataSize = dataSize;
    memcpy(g_AddInfo.m_pPackedUserData, dataBuf, dataSize);
}

//-----------------------------------------------------------------------------
//! @brief 編集用コメントラベルのリソースデータを解析します。
//-----------------------------------------------------------------------------
static void ParseRDCommentLabel(GPtr globals, const uint8_t* dataBuf, const uint32_t dataSize)
{
    g_AddInfo.m_CommentLabel =
        std::string(reinterpret_cast<const char*>(dataBuf), dataSize);
}

//-----------------------------------------------------------------------------
//! @brief 編集用コメントカラー文字列のリソースデータを解析します。
//-----------------------------------------------------------------------------
static void ParseRDCommentCol(GPtr globals, const uint8_t* dataBuf, const uint32_t dataSize)
{
    g_AddInfo.m_CommentCol =
        std::string(reinterpret_cast<const char*>(dataBuf), dataSize);
}

//-----------------------------------------------------------------------------
//! @brief 編集用コメント文のリソースデータを解析します。
//-----------------------------------------------------------------------------
static void ParseRDCommentText(GPtr globals, const uint8_t* dataBuf, const uint32_t dataSize)
{
    g_AddInfo.m_CommentText =
        std::string(reinterpret_cast<const char*>(dataBuf), dataSize);
    //RNoteTrace("comment: [%s]", g_AddInfo.m_CommentText.c_str());
}

//-----------------------------------------------------------------------------
//! @brief ツールデータのリソースデータを解析します。
//-----------------------------------------------------------------------------
static void ParseRDToolData(GPtr globals, const uint8_t* dataBuf, const uint32_t dataSize)
{
    g_AddInfo.m_ToolData =
        std::string(reinterpret_cast<const char*>(dataBuf), dataSize);
}

//-----------------------------------------------------------------------------
//! @brief ユーザーツールデータのリソースデータを解析します。
//-----------------------------------------------------------------------------
static void ParseRDUserToolData(GPtr globals, const uint8_t* dataBuf, const uint32_t dataSize)
{
    g_AddInfo.m_UserToolData =
        std::string(reinterpret_cast<const char*>(dataBuf), dataSize);
}

//-----------------------------------------------------------------------------
//! @brief ファイル形式プラグイン用のリソースデータをドキュメントから取得します。
//!        見積もり時に呼ばれます。
//!
//! @param[in,out] globals グローバルデータです。
//-----------------------------------------------------------------------------
static void GetFormatResourceData(GPtr globals)
{
    static const RDParse parseTable[] =
    {
        { RD_TAG_DCC_PRESET    , ParseRDDccPreset        },
        { RD_TAG_HINT          , ParseRDHint             },
        { RD_TAG_LINEAR        , ParseRDLinearFlag       },
        { RD_TAG_DIMENSION     , ParseRDDimension        },
        { RD_TAG_FORMAT        , ParseRDFormat           },
        { RD_TAG_WEIGHT_CMPR   , ParseRDWeightedCompress },
        { RD_TAG_INI_SWIZZLE   , ParseRDInitialSwizzle   },
        { RD_TAG_MIP_LEVEL     , ParseRDMipLevel         },
        { RD_TAG_MIP_GEN_FILTER, ParseRDMipGenFilter     },
        { RD_TAG_COMP_SEL      , ParseRDCompSel          },
        { RD_TAG_ORG_PATHS     , ParseRDOriginalPaths    },
        { RD_TAG_CREATE_INFO   , ParseRDCreateInfo       },
        { RD_TAG_USER_DATA     , ParseRDUserData         },
        { RD_TAG_COMMENT_LABEL , ParseRDCommentLabel     },
        { RD_TAG_COMMENT_COLOR , ParseRDCommentCol       },
        { RD_TAG_COMMENT_TEXT  , ParseRDCommentText      },
        { RD_TAG_TOOL_DATA     , ParseRDToolData         },
        { RD_TAG_USER_TOOL_DATA, ParseRDUserToolData     },
    };
    const uint32_t parseSize = sizeof(parseTable) / sizeof(RDParse);

    //-----------------------------------------------------------------------------
    // 最初にバージョンを取得します (先頭にあるとは限らないため先に取得します）。
    g_ImageStatus.m_RdVersion = RGetResourceDataVersion(globals);

    //-----------------------------------------------------------------------------
    // 他のリソースデータを取得します。
    RDoRDParseTable(globals, parseTable, parseSize);
}

//-----------------------------------------------------------------------------
//! @brief ドキュメントの最終的なファイルパスを返します。
//!        別名で保存した場合に別名で保存したファイルパスを取得できます。
//-----------------------------------------------------------------------------
std::string RGetFinalFilePath(GPtr globals)
{
    int32 major, miner, fix;
    PIGetHostVersion(major, miner, fix);

    if (major >= 11) // CS4
    {
        #if (PS_API_VERSION >= 1100)
        const uint16* finalSpec = gStuff->finalSpec;
        #else
        const uint16* finalSpec = *reinterpret_cast<const uint16**>(&gStuff->reserved[24]);
        #endif
        return RGetShiftJisFromUnicode(reinterpret_cast<const wchar_t*>(finalSpec));
    }
    else
    {
        // fileSpec2 は D:\img\ftx\~psA25D.tmp のように
        // フォルダは最終と同じだがファイル名と拡張子が異なる
        std::string path = RGetShiftJisFromUnicode(
            reinterpret_cast<const wchar_t*>(gStuff->fileSpec2->mReference));

        // ファイル名を取得します。
        //Handle complex;
        //GetComplex('titl', 0, &complex); // 1 度も保存されていなければ「名称未設定 1」
        //Str255 psStr;
        //PIHandle2PString(complex, psStr);
        //PIDisposeHandle(complex);
        //const std::string name = RGetStringFromStr255(psStr);

        const std::string docPath = reinterpret_cast<SPPlatformFileSpecification*>(
            gStuff->documentInfo->fileSpec)->path; // 1 度も保存されていなければ空文字
        const std::string name = (!docPath.empty()) ?
            RGetFileNameFromFilePath(docPath) : "untitled";

        return RGetFolderFromFilePath(path) + "\\" + name;
    }
}

//-----------------------------------------------------------------------------
//! @brief ビットマップデータの色をチェックします。
//!
//! @param[in,out] globals グローバルデータです。
//-----------------------------------------------------------------------------
static void CheckBitmapDataColor(GPtr globals)
{
    const bool isRealNumber = (gStuff->depth != 8);

    //-----------------------------------------------------------------------------
    // gray
    g_ImageStatus.m_IsAllGray = true;
    if (gStuff->imageMode != plugInModeGrayScale)
    {
        g_ImageStatus.m_IsAllGray = RIsAllGrayBitmap(
            globals->m_pBitmapData, gStuff->imageSize32.h, gStuff->imageSize32.v, isRealNumber);
    }
    //RNoteTrace("is all gray: %d", g_ImageStatus.m_IsAllGray);

    //-----------------------------------------------------------------------------
    // xpa mode
    g_ImageStatus.m_XpaMode = RImage::OPA;
    if (RGetPsAlphaPlaneIndex(globals) != 0 || IsPsTransparentIndexUsed(globals))
    {
        g_ImageStatus.m_XpaMode = RGetBitmapTransparencyMode(
            globals->m_pBitmapData, gStuff->imageSize32.h, gStuff->imageSize32.v, isRealNumber);
    }
    //RNoteTrace("xpa mode: %d", g_ImageStatus.m_XpaMode);
}

//-----------------------------------------------------------------------------
//! @brief 画像状態をもとに推奨フォーマットを設定します。
//!
//! @param[in,out] globals グローバルデータです。
//-----------------------------------------------------------------------------
static void SetRecommendFormat(GPtr globals)
{
    const bool isRealNumber = (gStuff->depth != 8);

    if (g_ImageStatus.m_LastFormat != FtxFormat_Invalid &&
        RIsConvertibleTextureFormat(g_ImageStatus.m_LastFormat, isRealNumber))
    {
        //-----------------------------------------------------------------------------
        // 前回のフォーマットでライトできるならそれを推奨します。
        g_ImageStatus.m_RecomFormat = g_ImageStatus.m_LastFormat;
        g_ImageStatus.m_IsFormatDecided = true;
    }
    else if (!isRealNumber)
    {
        if (gStuff->imageMode == plugInModeGrayScale)
        {
            //-----------------------------------------------------------------------------
            // グレースケールモードの場合
            g_ImageStatus.m_RecomFormat = (g_ImageStatus.m_XpaMode == RImage::OPA) ?
                FtxFormat_Unorm_Bc4 : FtxFormat_Unorm_Bc5;
        }
        else
        {
            //-----------------------------------------------------------------------------
            // RGB カラーモードまたはインデックスカラーモードの場合
            if (g_ImageStatus.m_IsAllGray)
            {
                g_ImageStatus.m_RecomFormat = (g_ImageStatus.m_XpaMode == RImage::OPA) ?
                    FtxFormat_Unorm_Bc4 : FtxFormat_Unorm_Bc5;
            }
            else
            {
                switch (g_ImageStatus.m_XpaMode)
                {
                    default:
                    case RImage::OPA:
                        g_ImageStatus.m_RecomFormat = FtxFormat_Unorm_Bc1;
                        break;
                    case RImage::MASK:
                        g_ImageStatus.m_RecomFormat = FtxFormat_Unorm_Bc1;
                        break;
                    case RImage::XLU:
                        g_ImageStatus.m_RecomFormat = FtxFormat_Unorm_Bc3;
                        break;
                }
            }
        }
    }
    else
    {
        g_ImageStatus.m_RecomFormat = FtxFormat_Float_16_16_16_16;
    }
}

//-----------------------------------------------------------------------------
//! @brief ライトのために画像状態を取得します。
//!
//! @param[in,out] globals グローバルデータです。
//-----------------------------------------------------------------------------
static void GetImageStatusForWrite(GPtr globals)
{
    //RMsgBox("save file: %s", GetSaveFilePathTest(globals).c_str());

    //-----------------------------------------------------------------------------
    // 各次元が有効かどうかのフラグを設定します。
    g_ImageStatus.m_Is1DEnable = true;
    g_ImageStatus.m_Is2DEnable = true;

    g_ImageStatus.m_3DFaceD = (gStuff->imageSize32.v >= gStuff->imageSize32.h &&
        (gStuff->imageSize32.v % gStuff->imageSize32.h) == 0) ?
        gStuff->imageSize32.v / gStuff->imageSize32.h : 0;
    g_ImageStatus.m_3DFaceW = gStuff->imageSize32.h;
    g_ImageStatus.m_3DFaceH = g_ImageStatus.m_3DFaceW;
    g_ImageStatus.m_Is3DEnable = (g_ImageStatus.m_3DFaceD >= 1);

    g_ImageStatus.m_IsCubeHCEnable = RImage::IsCubeHCWH(gStuff->imageSize32.h, gStuff->imageSize32.v);
    g_ImageStatus.m_IsCubeVCEnable = RImage::IsCubeVCWH(gStuff->imageSize32.h, gStuff->imageSize32.v);
    //RNoteTrace("enable: %d %d %d %d %d", g_ImageStatus.m_Is1DEnable, g_ImageStatus.m_Is2DEnable, g_ImageStatus.m_Is3DEnable, g_ImageStatus.m_IsCubeHCEnable, g_ImageStatus.m_IsCubeVCEnable);

    //-----------------------------------------------------------------------------
    // キューブマップの場合の幅と高さを設定します。
    g_ImageStatus.m_CubeFaceW = gStuff->imageSize32.h;
    g_ImageStatus.m_CubeFaceH = gStuff->imageSize32.v;
    if (g_ImageStatus.m_IsCubeHCEnable)
    {
        g_ImageStatus.m_CubeFaceW /= RImage::CUBE_HC_COUNT_H;
        g_ImageStatus.m_CubeFaceH /= RImage::CUBE_HC_COUNT_V;
    }
    else if (g_ImageStatus.m_IsCubeVCEnable)
    {
        g_ImageStatus.m_CubeFaceW /= RImage::CUBE_VC_COUNT_H;
        g_ImageStatus.m_CubeFaceH /= RImage::CUBE_VC_COUNT_V;
    }

    //-----------------------------------------------------------------------------
    // リソースデータ（フォーマット、ミップマップのレベル数など）をドキュメントから取得します。
    GetFormatResourceData(globals);

    //-----------------------------------------------------------------------------
    // ビットマップデータの色をチェックします。
    CheckBitmapDataColor(globals);

    //-----------------------------------------------------------------------------
    // 推奨フォーマットを設定します。
    SetRecommendFormat(globals);
}

//-----------------------------------------------------------------------------
//! @brief ライトのために付加情報を設定します。
//-----------------------------------------------------------------------------
void SetAddInfoForWrite(GPtr globals)
{
    //-----------------------------------------------------------------------------
    // 画像状態を取得します。
    GetImageStatusForWrite(globals);
    if (gResult != noErr)
    {
        return;
    }

    //-----------------------------------------------------------------------------
    // 画像状態から次元、フォーマットなどを取得します。
    g_AddInfo.m_Dimension = g_ImageStatus.m_LastDimension;
    if ((g_AddInfo.m_Dimension == FtxDimension_1d      && !g_ImageStatus.m_Is1DEnable  ) ||
        (g_AddInfo.m_Dimension == FtxDimension_2d      && !g_ImageStatus.m_Is2DEnable  ) ||
        (g_AddInfo.m_Dimension == FtxDimension_3d      && !g_ImageStatus.m_Is3DEnable  ) ||
        (g_AddInfo.m_Dimension == FtxDimension_CubeMap && !g_ImageStatus.IsCubeEnable()) ||
        (g_AddInfo.m_Dimension == FtxDimension_1dArray && !g_ImageStatus.m_Is3DEnable  ) ||
        (g_AddInfo.m_Dimension == FtxDimension_2dArray && !g_ImageStatus.m_Is3DEnable  ))
    {
        if      (g_ImageStatus.m_Is2DEnable  ) g_AddInfo.m_Dimension = FtxDimension_2d;
        else if (g_ImageStatus.IsCubeEnable()) g_AddInfo.m_Dimension = FtxDimension_CubeMap;
        else if (g_ImageStatus.m_Is3DEnable  ) g_AddInfo.m_Dimension = FtxDimension_3d;
        else if (g_ImageStatus.m_Is1DEnable  ) g_AddInfo.m_Dimension = FtxDimension_1d;
    }

    g_AddInfo.m_Format = g_ImageStatus.m_RecomFormat;
    //g_AddInfo.m_MipLevel = g_ImageStatus.m_LastMipLevel;
    g_AddInfo.m_MipLevel = (
        g_ImageStatus.m_LastFormat == FtxFormat_Invalid ||
        g_ImageStatus.m_LastMipLevel >= 2) ?
        RImage::MipCountMax : 1; // 1 x 1 のレベルまで作成します。
    if (g_ImageStatus.m_LastFormat == FtxFormat_Invalid)
    {
        g_AddInfo.m_CompSel = RGetDefaultCompSel(g_AddInfo.m_Format, g_AddInfo.m_Hint);
    }
}

//-----------------------------------------------------------------------------
//! @brief プレビューの矩形領域を白色でフィルします。
//-----------------------------------------------------------------------------
static void FillPreviewRectArea(
    GPtr globals,
    const int dstIx,
    const int dstIy,
    const int dstW,
    const int dstH
)
{
    uint8_t* dst = globals->m_pPreviewBitmap + (dstIx + dstIy * gStuff->imageSize32.h) * R_RGBA_BYTES;
    for (int iy = 0; iy < dstH; ++iy)
    {
        memset(dst, 0xff, dstW * R_RGBA_BYTES);
        dst += gStuff->imageSize32.h * R_RGBA_BYTES;
    }
}

//-----------------------------------------------------------------------------
//! @brief プレビューのキューブマップの空き領域を白色でフィルします。
//-----------------------------------------------------------------------------
static void FillPreviewCubeMapEmptyArea(GPtr globals)
{
    // TODO: 垂直十字に対応

    int dstIx = 0;
    for (int level = 0; level < 1; ++level) // 現在はミップマップに非対応
    {
        const int curW = RMax(g_ImageStatus.m_CubeFaceW >> level, 1);
        const int curH = RMax(g_ImageStatus.m_CubeFaceH >> level, 1);
        if (g_ImageStatus.m_IsCubeHCEnable)
        {
            // A * CC
            // ******
            // B * DD
            const int dstIx2 = dstIx + curW * 2;
            FillPreviewRectArea(globals, dstIx ,        0, curW    , curH); // A
            FillPreviewRectArea(globals, dstIx , curH * 2, curW    , curH); // B
            FillPreviewRectArea(globals, dstIx2,        0, curW * 2, curH); // CC
            FillPreviewRectArea(globals, dstIx2, curH * 2, curW * 2, curH); // DD
            dstIx += curW * RImage::CUBE_HC_COUNT_H;
        }
        else if (g_ImageStatus.m_IsCubeVCEnable)
        {

        }
    }
}

//-----------------------------------------------------------------------------
//! @brief プレビューのデータバッファを更新します。
//-----------------------------------------------------------------------------
void RUpdatePreviewDataBuf(GPtr globals)
{
    if (globals->m_pPreviewBitmap == NULL)
    {
        return;
    }
    //RTrace("Update Preview Buffer\n");

    //-----------------------------------------------------------------------------
    // update option
    globals->m_PreviewFormat = g_AddInfo.m_Format;

    globals->m_PreviewDataSetFlag = true;

    //-----------------------------------------------------------------------------
    // set preview bitmap
    //switch (g_AddInfo.m_Format)
    //{
    //}

    //-----------------------------------------------------------------------------
    // キューブマップの空白部分を白（アルファ 0xff）に
    if (g_AddInfo.m_Dimension == FtxDimension_CubeMap)
    {
        FillPreviewCubeMapEmptyArea(globals);
    }
}

//=============================================================================
// nps ネームスペースを終了します。
//=============================================================================
} // namespace nps
} // namespace tool
} // namespace gfx
} // namespace nn

