﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

// variables constants
// PluginMain InitGlobals
// DoFilterFile
// DoReadStart DoReadContinue DecodeFtx
// DoOptionsStart
// DoEstimateStart
// DoWriteStart OutputFtx

//=============================================================================
// include
//=============================================================================
#include "NintendoFtx.h"
#include "NintendoFtxScripting.h"
#include "NpsFormatUi.h"

using namespace nn::gfx::tool::nps;

//=============================================================================
// constants
//=============================================================================

//-----------------------------------------------------------------------------
// ftx
const std::string FTXA_EXTENSION = "ftxa"; //!< アスキー形式の出力ファイルのデフォルトの拡張子です。
const std::string FTXB_EXTENSION = "ftxb"; //!< バイナリー形式の出力ファイルのデフォルトの拡張子です。
const std::string FTX_NO_FILE_INFO = "<none>"; //!< 入力 ftx に <file_info> がなかった場合の文字列です。

//-----------------------------------------------------------------------------
//! @brief グローバルデータを初期化します。
//!        プラグインが最初に使用されたときに１度だけ呼ばれます。
//!
//! @param[in,out] globalPtr グローバルデータのポインタです。
//-----------------------------------------------------------------------------
static void InitGlobals(Ptr globalPtr)
{
    //RNoteTrace("init globals");

    //-----------------------------------------------------------------------------
    // copy for use macro (g????)
    //GPtr globals = (GPtr)globalPtr;

    //-----------------------------------------------------------------------------
    // call constructor (name must be "globals" for use macro (g????))
    GPtr globals = new (globalPtr) Globals;
    R_UNUSED_VARIABLE(globals);
}

//-----------------------------------------------------------------------------
//! @brief 画像リソースを破棄します。
//!
//! @param[in,out] globals グローバルデータです。
//-----------------------------------------------------------------------------
static void DisposeImageResources(GPtr globals)
{
    if (gStuff->imageRsrcData)
    {
        PIDisposeHandle(gStuff->imageRsrcData);
        gStuff->imageRsrcData = NULL;
        gStuff->imageRsrcSize = 0;
    }
}

//-----------------------------------------------------------------------------
//! @brief リードの準備をします。
//!
//! @param[in,out] globals グローバルデータです。
//-----------------------------------------------------------------------------
static void DoReadPrepare(GPtr globals)
{
    gStuff->maxData = 0;
}

//-----------------------------------------------------------------------------
//! @brief LA (Luminance + Alpha) 形式に適したフォーマットなら true を返します。
//!
//! @param[in] format テクスチャーフォーマットです。
//!
//! @return LA 形式に適したフォーマットなら true を返します。
//-----------------------------------------------------------------------------
static bool IsFormatForLuminanceAlpha(const FtxFormat format)
{
    return (format == FtxFormat_Unorm_4_4       ||
            format == FtxFormat_Unorm_8_8       ||
            format == FtxFormat_Unorm_Bc5       ||
            format == FtxFormat_Unorm_Eac_11_11);
}

//-----------------------------------------------------------------------------
//! @brief ユーザーデータ群をパックします。
//!        この関数内でパックしたデータのメモリーを確保して返します。
//!
//! @param[out] packedUserDataSize パックしたデータのサイズを格納します。
//! @param[in] userDatas ユーザーデータ配列です。
//!
//! @return パックしたデータへのポインタを返します。
//!         ユーザーデータがなければ NULL を返します。
//-----------------------------------------------------------------------------
static uint8_t* PackUserDataArray(uint32_t& packedUserDataSize, const RUserDataArray& userDatas)
{
    //-----------------------------------------------------------------------------
    // check count
    const int userDataCount = static_cast<int>(userDatas.size());
    if (userDataCount == 0)
    {
        packedUserDataSize = 0;
        return NULL;
    }

    //-----------------------------------------------------------------------------
    // 各ユーザーデータの内容を oss に出力します。
    std::ostringstream oss;
    for (int iData = 0; iData < userDataCount; ++iData)
    {
        //-----------------------------------------------------------------------------
        // uint32_t nameSize;        // ends は含みません
        // char name[nameSize]; // ends は含みません
        const RUserData& userData = userDatas[iData];
        //RNoteTrace("ud: %d: %s", iData, userData.m_Name.c_str());
        const uint32_t nameSize = static_cast<uint32_t>(userData.m_Name.size());
        oss.write(reinterpret_cast<const char*>(&nameSize), sizeof(nameSize));
        if (nameSize != 0)
        {
            oss.write(userData.m_Name.c_str(), nameSize);
        }

        //-----------------------------------------------------------------------------
        // uint32_t type;
        oss.write(reinterpret_cast<const char*>(&userData.m_Type), sizeof(userData.m_Type));

        //-----------------------------------------------------------------------------
        // uint32_t contentSize;
        // uint8_t content[contentSize];
        if (userData.m_Type == RUserData::STREAM)
        {
            const uint32_t contentSize = static_cast<uint32_t>(userData.m_StreamValues.size());
            oss.write(reinterpret_cast<const char*>(&contentSize), sizeof(contentSize));
            if (contentSize != 0)
            {
                oss.write(reinterpret_cast<const char*>(&userData.m_StreamValues[0]),
                    userData.m_StreamValues.size());
            }
        }
        else
        {
            std::ostringstream contentOss;
            if (userData.m_Type == RUserData::FLOAT || userData.m_Type == RUserData::INT)
            {
                // uint32_t valueCount;
                // char text[contentSize - 4]; // ends は含みません
                contentOss.write(reinterpret_cast<const char*>(&userData.m_NumberValueCount),
                    sizeof(userData.m_NumberValueCount));
                contentOss << userData.m_NumberValuesText;
            }
            else // STRING, WSTRING
            {
                // uint32_t stringCount;
                // {
                //     uint32_t stringSize;          // ends は含みません
                //     char string[stringSize]; // ends は含みません
                // } x stringCount
                const uint32_t stringCount = static_cast<uint32_t>(userData.m_StringValues.size());
                contentOss.write(reinterpret_cast<const char*>(&stringCount), sizeof(stringCount));
                for (uint32_t iString = 0; iString < stringCount; ++iString)
                {
                    const std::string& str = userData.m_StringValues[iString];
                    const uint32_t stringSize = static_cast<uint32_t>(str.size());
                    contentOss.write(reinterpret_cast<const char*>(&stringSize), sizeof(stringSize));
                    contentOss.write(str.c_str(), str.size());
                }
            }
            const std::string contentStr = contentOss.str();
            const uint32_t contentSize = static_cast<uint32_t>(contentStr.size());
            oss.write(reinterpret_cast<const char*>(&contentSize), sizeof(contentSize));
            if (contentSize != 0)
            {
                oss.write(contentStr.c_str(), contentStr.size());
            }
        }
    }

    //-----------------------------------------------------------------------------
    // メモリーを確保して oss からコピーします。
    const std::string userDataStr = oss.str();
    packedUserDataSize = 4 + static_cast<uint32_t>(userDataStr.size());
    uint8_t* pPacked = new uint8_t[packedUserDataSize];
    memcpy(pPacked + 0, &userDataCount, sizeof(userDataCount));
    memcpy(pPacked + 4, userDataStr.c_str(), userDataStr.size());
    //RNoteTrace("packed ud size: %d", packedUserDataSize);
    //RNoteDumpMemory(pPacked, packedUserDataSize, "packed ud");

    return pPacked;
}

//-----------------------------------------------------------------------------
//! @brief ftx ファイルの編集用コメントを取得します。
//!
//! @param[out] pCommentLabel 編集用コメントラベルへのポインターです。
//! @param[out] pCommentCol 編集用コメントカラー文字列へのポインターです。
//! @param[out] pCommentText 編集用コメント文へのポインターです。
//! @param[in] textureElem <texture> XML 要素です。
//-----------------------------------------------------------------------------
static void GetFtxComment(
    std::string* pCommentLabel,
    std::string* pCommentCol,
    std::string* pCommentText,
    const RXMLElement* textureElem
)
{
    const RXMLElement* commentElem = textureElem->FindElement("comment", false);
    if (commentElem != nullptr)
    {
        *pCommentLabel = RDecodeXmlString(
            RGetShiftJisFromUtf8(commentElem->GetAttribute("label", "", false))); // 4.0.0 で追加
        *pCommentCol = commentElem->GetAttribute("color");
        *pCommentText = RDecodeXmlString(
            RGetShiftJisFromUtf8(commentElem->GetAttribute("text")));
    }
}

//-----------------------------------------------------------------------------
//! @brief ftx ファイルのツールデータとユーザーツールデータを取得します。
//!
//! @param[out] toolData <tool_data> 要素の内容を格納します。
//! @param[out] userToolData <user_tool_data> 要素の内容を格納します。
//! @param[in] xmlStr ftx ファイルの XML 部分の文字列です。
//-----------------------------------------------------------------------------
static void GetFtxToolData(
    std::string& toolData,
    std::string& userToolData,
    const std::string& xmlStr)
{
    //-----------------------------------------------------------------------------
    // get tool data
    const std::string TOOL_DATA_BEGIN = "<tool_data";
    const std::string TOOL_DATA_END = "</tool_data>";
    const size_t iToolData = xmlStr.find(TOOL_DATA_BEGIN);
    if (iToolData != std::string::npos)
    {
        const size_t iEndToolData = xmlStr.find(TOOL_DATA_END, iToolData);
        if (iEndToolData != std::string::npos)
        {
            toolData = RGetShiftJisFromUtf8(xmlStr.substr(
                iToolData, iEndToolData + TOOL_DATA_END.size() - iToolData));
            //RNoteTrace("tool data: [%s]", toolData.c_str());
        }
    }

    //-----------------------------------------------------------------------------
    // get user tool data
    const std::string USER_TOOL_DATA_BEGIN = "<user_tool_data";
    const std::string USER_TOOL_DATA_END = "</user_tool_data>";
    const size_t iUserToolData = xmlStr.find(USER_TOOL_DATA_BEGIN);
    if (iUserToolData != std::string::npos)
    {
        const size_t iEndUserToolData = xmlStr.find(USER_TOOL_DATA_END, iUserToolData);
        if (iEndUserToolData != std::string::npos)
        {
            userToolData = RGetShiftJisFromUtf8(xmlStr.substr(
                iUserToolData, iEndUserToolData + USER_TOOL_DATA_END.size() - iUserToolData));
            //RNoteTrace("user tool data: [%s]", userToolData.c_str());
        }
    }
}

//-----------------------------------------------------------------------------
//! @brief XML 要素のエラー表示関数です。
//!
//! @param[in] message エラーメッセージです。
//! @param[in] pUserData ユーザーデータのポインタです。
//-----------------------------------------------------------------------------
static void XmlErrorFunc(const char* message, void* pUserData)
{
    GPtr globals = reinterpret_cast<GPtr>(pUserData);
    if (!gStuff->openForPreview) // プレビューではエラー表示しません。
    {
        RShowError(globals, message);
        //RNoteTrace(message);
    }
}

//-----------------------------------------------------------------------------
//! @brief ftx ファイルのアスキー形式のデータ列を解析します。
//!
//! @param[out] pDst データ列のデータを格納します。
//! @param[in] dstBytes 取得するデータの最大サイズ（バイト数）です。
//! @param[in] streamElem データ列要素です。
//-----------------------------------------------------------------------------
static void ParseFtxAsciiStream(
    void* pDst,
    const size_t dstBytes,
    const RXMLElement* streamElem
)
{
    std::istringstream iss(streamElem->text);
    const std::string streamType = streamElem->GetAttribute("type");
    if (streamType == "byte")
    {
        uint8_t* pDstU8 = reinterpret_cast<uint8_t*>(pDst);
        const uint8_t* pDstEndU8 = pDstU8 + dstBytes;
        while (iss && pDstU8 < pDstEndU8)
        {
            std::string value;
            iss >> value;
            if (!value.empty())
            {
                *pDstU8++ = static_cast<uint8_t>(strtol(value.c_str(), NULL, 16));
            }
        }
    }
    else if (streamType == "float")
    {
        float* pDstF32 = reinterpret_cast<float*>(pDst);
        const float* pDstEndF32 = pDstF32 + (dstBytes / sizeof(float));
        while (iss && pDstF32 < pDstEndF32)
        {
            std::string value;
            iss >> value;
            if (!value.empty())
            {
                *pDstF32++ = static_cast<float>(atof(value.c_str()));
            }
        }
    }
}

//-----------------------------------------------------------------------------
//! @brief ftx ファイルの元画像データを解析して、ピクセルデータを取得します。
//!
//! @param[out] pHasAlpha A 成分存在フラグを格納します。
//! @param[in] globals グローバルデータです。
//! @param[in] ftxPath ftx ファイルのパスです。
//! @param[in] imageW 画像の幅です。
//! @param[in] imageH 画像の高さです。
//! @param[in] imageD 画像の奥行きです。
//! @param[in] originalsElem 元画像配列要素です。
//! @param[in] streamsElem データ列配列要素です（バイナリー形式なら nullptr）。
//! @param[in] binTop 中間ファイルのバイナリー部分の先頭へのポインターです。
//!                   （アスキー形式なら nullptr）
//-----------------------------------------------------------------------------
static void ParseFtxOriginalImages(
    bool* pHasAlpha,
    const GPtr globals,
    const std::string& ftxPath,
    const int imageW,
    const int imageH,
    const int imageD,
    const RXMLElement* originalsElem,
    const RXMLElement* streamsElem,
    const void* binTop
)
{
    *pHasAlpha = false;
    const bool isFloat = (gStuff->depth != 8);
    const size_t pixelBytes = (isFloat) ? R_RGBA_FLOAT_BYTES : R_RGBA_BYTES;
    g_AddInfo.m_OriginalPaths.resize(imageD);
    for (int iz = 0; iz < imageD; ++iz)
    {
        //-----------------------------------------------------------------------------
        // データ列のデータを取得します。
        const RXMLElement* originalElem = &(originalsElem->nodes[iz]);
        int dstIdx = atoi(originalElem->GetAttribute("slice_index").c_str());
        uint8_t* pDst = globals->m_pBitmapData;
        if (g_AddInfo.m_Dimension == FtxDimension_CubeMap)
        {
            const std::string faceName = originalElem->GetAttribute("face");
            const int faceIdx =
                (faceName == "negative_x") ? RImage::CUBE_FACE_NX :
                (faceName == "positive_y") ? RImage::CUBE_FACE_PY :
                (faceName == "negative_y") ? RImage::CUBE_FACE_NY :
                (faceName == "positive_z") ? RImage::CUBE_FACE_PZ :
                (faceName == "negative_z") ? RImage::CUBE_FACE_NZ :
                RImage::CUBE_FACE_PX;
            const int* facePos = RImage::GetCubeHCFacePos(faceIdx);
            pDst += pixelBytes * (facePos[0] * imageW +
                     facePos[1] * imageH * gStuff->imageSize32.h);
            dstIdx += faceIdx;
        }
        else
        {
            pDst += pixelBytes * RMin(dstIdx, imageD - 1) * imageW * imageH;
        }

        // オリジナルパスを取得してフルパスに変換します。
        std::string originalPath = RDecodeXmlString(
            RGetShiftJisFromUtf8(originalElem->GetAttribute("original_path")));
        if (!RIsFullFilePath(originalPath))
        {
            originalPath = RGetFullFilePath(
                RGetFolderFromFilePath(ftxPath) + "/" + originalPath, true);
        }
        g_AddInfo.m_OriginalPaths[dstIdx] = originalPath;

        const std::string originalFormat = originalElem->GetAttribute("format");
        size_t originalDataSize;
        std::istringstream(originalElem->GetAttribute("size")) >> originalDataSize;
        uint8_t* pStreamData = new uint8_t[originalDataSize];
        const int streamIdx = atoi(originalElem->GetAttribute("stream_index").c_str());
        if (binTop == nullptr)
        {
            ParseFtxAsciiStream(pStreamData, originalDataSize, &(streamsElem->nodes[streamIdx]));
        }
        else
        {
            const RBinDataStream* binStream = RBinDataStream::Get(binTop, streamIdx);
            memcpy(pStreamData, binStream->m_Data, originalDataSize);
        }

        //-----------------------------------------------------------------------------
        // データ列のデータをリード用バッファーにコピーします。
        const uint8_t* pSrc = pStreamData;
        if (originalFormat == "rgba8" || originalFormat == "rgba32f")
        {
            for (int iy = 0; iy < imageH; ++iy)
            {
                memcpy(pDst, pSrc, pixelBytes * imageW);
                pSrc += pixelBytes * imageW;
                pDst += pixelBytes * gStuff->imageSize32.h;
            }
            *pHasAlpha = true;
        }
        else if (originalFormat == "rgb8")
        {
            for (int iy = 0; iy < imageH; ++iy)
            {
                for (int ix = 0; ix < imageW; ++ix)
                {
                    pDst[0] = pSrc[0];
                    pDst[1] = pSrc[1];
                    pDst[2] = pSrc[2];
                    pDst[3] = 0xff;
                    pSrc += R_RGB_COUNT;
                    pDst += R_RGBA_COUNT;
                }
                pDst += (gStuff->imageSize32.h - imageW) * R_RGBA_COUNT;
            }
        }
        else if (originalFormat == "rgb32f")
        {
            const float* pSrcF32 = reinterpret_cast<const float*>(pStreamData);
            float* pDstF32 = reinterpret_cast<float*>(pDst);
            for (int iy = 0; iy < imageH; ++iy)
            {
                for (int ix = 0; ix < imageW; ++ix)
                {
                    pDstF32[0] = pSrcF32[0];
                    pDstF32[1] = pSrcF32[1];
                    pDstF32[2] = pSrcF32[2];
                    pDstF32[3] = 1.0f;
                    pSrcF32 += R_RGB_COUNT;
                    pDstF32 += R_RGBA_COUNT;
                }
                pDstF32 += (gStuff->imageSize32.h - imageW) * R_RGBA_COUNT;
            }
        }
        delete[] pStreamData;
    }
}

//-----------------------------------------------------------------------------
//! @brief ftx ファイルのエンコードされたピクセルデータをデコードして
//!        プレーンなピクセルデータを取得します。
//!
//! @param[in,out] pHasAlpha A 成分存在フラグへのポインターです。
//!                          条件によっては値を変更します。
//! @param[in] globals グローバルデータです。
//! @param[in] imageW 画像の幅です。
//! @param[in] imageH 画像の高さです。
//! @param[in] imageD 画像の奥行きです。
//! @param[in] infoElem テクスチャー情報要素です。
//! @param[in] streamsElem データ列配列要素です（バイナリー形式なら nullptr）。
//! @param[in] binTop 中間ファイルのバイナリー部分の先頭へのポインターです。
//!                   （アスキー形式なら nullptr）
//-----------------------------------------------------------------------------
static bool GetFtxDecodedPixels(
    bool* pHasAlpha,
    const GPtr globals,
    const int imageW,
    const int imageH,
    const int imageD,
    const RXMLElement* infoElem,
    const RXMLElement* streamsElem,
    const void* binTop
)
{
    //-----------------------------------------------------------------------------
    // テクスチャーエンコーダーを初期化します。
    RTexEncoder encoder;
    if (!encoder.Initialize())
    {
        return false;
    }

    //-----------------------------------------------------------------------------
    // エンコードされたピクセルデータを取得します。
    size_t encodedDataSize;
    std::istringstream(infoElem->GetAttribute("size")) >> encodedDataSize;
    uint8_t* pEncodedData = new uint8_t[encodedDataSize];
    const int streamIdx = atoi(infoElem->GetAttribute("stream_index").c_str());
    if (binTop == nullptr)
    {
        ParseFtxAsciiStream(pEncodedData, encodedDataSize, &(streamsElem->nodes[streamIdx]));
    }
    else
    {
        const RBinDataStream* binStream = RBinDataStream::Get(binTop, streamIdx);
        memcpy(pEncodedData, binStream->m_Data, encodedDataSize);
    }

    //-----------------------------------------------------------------------------
    // エンコードされたピクセルデータの最上位レベルをデコードします。
    const bool isFloat = (gStuff->depth != 8);
    const size_t pixelBytes = (isFloat) ? R_RGBA_FLOAT_BYTES : R_RGBA_BYTES;
    const size_t rowBytes = pixelBytes * imageW;
    const size_t faceBytes = rowBytes * imageH;
    const size_t decodedDataSize = faceBytes * imageD;
    uint8_t* pDecodedData = new uint8_t[decodedDataSize];

    const bool isSigned = RIsSignedTextureFormat(g_AddInfo.m_Format);
    const FtxFormat dstFormat = (isFloat ) ? FtxFormat_Float_32_32_32_32 :
                                (isSigned) ? FtxFormat_Snorm_8_8_8_8     :
                                FtxFormat_Unorm_8_8_8_8;
    const bool isSucceeded = encoder.ConvertFormat(pDecodedData, pEncodedData,
        RGetUnicodeFromShiftJis(RGetTextureFormatString(dstFormat)).c_str(),
        RGetUnicodeFromShiftJis(RGetTextureFormatString(g_AddInfo.m_Format)).c_str(),
        L"", nn::gfx::tool::texenc::EncodeFlag_ReverseRgba,
        RTexEncoder::GetImageDimension(g_AddInfo.m_Dimension), imageW, imageH, imageD, 1);
    delete[] pEncodedData;

    //-----------------------------------------------------------------------------
    // 必要に応じて符号付き 8 ビットから符号なし 8 ビットに変換します。
    if (isSucceeded && dstFormat == FtxFormat_Snorm_8_8_8_8)
    {
        uint8_t* pValue = pDecodedData;
        for (size_t valueIdx = 0; valueIdx < decodedDataSize; ++valueIdx)
        {
            const uint8_t value = *pValue;
            *pValue++ = static_cast<uint8_t>(
                ((value & 0x80) == 0) ? value + 0x80 : value - 0x80);
        }
    }

    //-----------------------------------------------------------------------------
    // 必要に応じて成分選択に従って RGBA 成分をスワップします。
    if (isSucceeded)
    {
        const int srcCompCount = RGetComponentCount(g_AddInfo.m_Format);
        const bool isHintForNormal = (g_AddInfo.m_Hint == HintValueNormal);
        const bool isAutoLa = IsFormatForLuminanceAlpha(g_AddInfo.m_Format) && !isHintForNormal;
        const bool isAstcNormal = RIsAstcTextureFormat(g_AddInfo.m_Format) && isHintForNormal;
        if ((srcCompCount == 1 && g_AddInfo.m_CompSel == FtxCompSelRrr1) ||
            (isAutoLa          && g_AddInfo.m_CompSel == FtxCompSelRrrg) ||
            (isAstcNormal      && g_AddInfo.m_CompSel == FtxCompSelRa01))
        {
            *pHasAlpha = SwapPixelsRgbaComponent(pDecodedData, decodedDataSize,
                isFloat, g_AddInfo.m_CompSel);
        }
    }

    //-----------------------------------------------------------------------------
    // デコードしたピクセルデータをリード用バッファーにコピーします。
    if (isSucceeded)
    {
        const uint8_t* pSrc = pDecodedData;
        for (int iz = 0; iz < imageD; ++iz)
        {
            if (g_AddInfo.m_Dimension == FtxDimension_CubeMap)
            {
                // 画像のピクセルデータは +X, -X, +Y, -Y, -Z, +Z の順に格納されているので
                // -Z と +Z を入れ替えます。
                int faceIdx = iz % RImage::CUBE_FACE_COUNT;
                faceIdx = (faceIdx == RImage::CUBE_FACE_PZ) ? RImage::CUBE_FACE_NZ :
                          (faceIdx == RImage::CUBE_FACE_NZ) ? RImage::CUBE_FACE_PZ :
                          faceIdx;
                const int* facePos = RImage::GetCubeHCFacePos(faceIdx);
                uint8_t* pDst = globals->m_pBitmapData + pixelBytes *
                    (facePos[0] * imageW + facePos[1] * imageH * gStuff->imageSize32.h);
                for (int iy = 0; iy < imageH; ++iy)
                {
                    memcpy(pDst, pSrc, rowBytes);
                    pDst += pixelBytes * gStuff->imageSize32.h;
                    pSrc += rowBytes;
                }
            }
            else
            {
                memcpy(globals->m_pBitmapData + faceBytes * iz, pSrc, faceBytes);
                pSrc += faceBytes;
            }
        }
    }
    delete[] pDecodedData;
    return isSucceeded;
}

//-----------------------------------------------------------------------------
//! @brief ftx ファイルのデータをデコードします。
//!
//! @param[in,out] globals グローバルデータです。
//! @param[in] fileBuf ファイルをリードしたバッファです。
//! @param[in] fileSize ファイルのサイズです。
//!
//! @return 処理成功なら true を返します。
//-----------------------------------------------------------------------------
static bool DecodeFtx(GPtr globals, const uint8_t* fileBuf, const size_t fileSize)
{
    //-----------------------------------------------------------------------------
    // get ftx path
    const std::string ftxPath = RGetUnixFilePath(RGetShiftJisFromUnicode(
        reinterpret_cast<const wchar_t*>(gStuff->fileSpec2->mReference)));
    const bool isBinary = (RGetExtensionFromFilePath(ftxPath) == FTXB_EXTENSION);

    //-----------------------------------------------------------------------------
    // parse XML
    const size_t binOfs = (isBinary) ?
        RGetBinaryOffset(fileBuf, fileSize) : fileSize;
    const std::string xmlStr(reinterpret_cast<const char*>(fileBuf), binOfs);
    //RNoteTrace("binOfs: %d %x\n", isBinary, binOfs);

    RXMLElement xml;
    xml.SetErrorFunc(XmlErrorFunc, globals);
    xml.LoadDocument(NULL, xmlStr);

    //-----------------------------------------------------------------------------
    // get create info
    const RXMLElement* intermediateElem = xml.FindElement("nw4f_3dif");
    if (intermediateElem == NULL)
    {
        return false;
    }
    const int intermediateVer = RGet3dIntermediateFileVersionAsInt(intermediateElem->GetAttribute("version"));
    const RXMLElement* fileInfoElem = intermediateElem->FindElement("file_info", false);
    if (fileInfoElem != NULL)
    {
        const RXMLElement* createElem = fileInfoElem->FindElement("create", false);
        if (createElem != NULL)
        {
            g_AddInfo.m_CreateInfo = "\t" +
                std::string("<create tool_name=\"") + RGetShiftJisFromUtf8(createElem->GetAttribute("tool_name")) +
                "\" tool_version=\"" + createElem->GetAttribute("tool_version") +
                "\"" + R_ENDL;
            const std::string srcPath = createElem->GetAttribute("src_path", "", false); // 2.2.0 で追加
            g_AddInfo.m_CreateInfo += "\t\t" + std::string("src_path=\"") + RGetShiftJisFromUtf8(srcPath) +
                "\"" + R_ENDL;
            g_AddInfo.m_CreateInfo += "\t" + std::string("/>") + R_ENDL;
        }
    }
    else
    {
        g_AddInfo.m_CreateInfo = FTX_NO_FILE_INFO;
    }

    //-----------------------------------------------------------------------------
    // get texture info
    const RXMLElement* textureElem = intermediateElem->FindElement("texture");
    if (textureElem == NULL)
    {
        return false;
    }
    const RXMLElement* infoElem = textureElem->FindElement("texture_info");
    if (infoElem == NULL)
    {
        return false;
    }
    const std::string dimensionStr = infoElem->GetAttribute("dimension");
    g_AddInfo.m_Dimension = RGetTextureDimensionFromString(dimensionStr);
    g_AddInfo.m_Format    = RGetTextureFormatFromString(infoElem->GetAttribute("quantize_type"));
    int imageW            = atoi(infoElem->GetAttribute("width").c_str());
    int imageH            = atoi(infoElem->GetAttribute("height").c_str());
    int imageD            = atoi(infoElem->GetAttribute("depth").c_str());
    g_AddInfo.m_MipLevel  = atoi(infoElem->GetAttribute("mip_level").c_str());

    const std::string compSel = infoElem->GetAttribute("comp_sel", "", false); // 1.0.0 で追加
    g_AddInfo.m_CompSel = (!compSel.empty()) ?
        RGetCompSelFromString(compSel) : RGetDefaultCompSel(g_AddInfo.m_Format, "");
    //RNoteTrace("comp_sel: %s", RGetCompSelString(g_AddInfo.m_CompSel).c_str());

    const std::string weightedCompress = infoElem->GetAttribute("weighted_compress", "", false); // 1.0.0 で追加
    g_AddInfo.m_WeightedCompress = (!weightedCompress.empty()) ?
        (weightedCompress == "true") : false;

    g_AddInfo.m_ConfigName = RGetShiftJisFromUtf8(infoElem->GetAttribute("dcc_preset", "", false)); // 3.0.0 で追加

    const std::string defaultHint = RGetDefaultHint(g_AddInfo.m_Format);
    g_AddInfo.m_Hint = infoElem->GetAttribute("hint", defaultHint.c_str(), false); // 2.2.0 で追加

    const std::string linearStr = infoElem->GetAttribute("linear", "", false); // 2.2.0 で追加
    g_AddInfo.m_LinearFlag = (!linearStr.empty()) ?
        RGetLinearFlagFromString(linearStr) : RLinearFlag::LINEAR_NONE;

    const std::string initialSwizzle = infoElem->GetAttribute("initial_swizzle", "", false); // 2.2.0 で追加
    g_AddInfo.m_InitialSwizzle = (!initialSwizzle.empty()) ? atoi(initialSwizzle.c_str()) : 0;

    bool hasAlpha = (RGetComponentCount(g_AddInfo.m_Format) >= 4);
    bool isFloat = RIsRealNumberTextureFormat(g_AddInfo.m_Format);

    //-----------------------------------------------------------------------------
    // 1 次元の場合、texture_info の height は 1 なので、
    // 元画像データが存在すれば幅と高さは original_image から取得します。
    const RXMLElement* originalsElem = textureElem->FindElement("original_image_array", false);
    if (originalsElem != nullptr)
    {
        const RXMLElement* firstOriginalElem = &(originalsElem->nodes[0]);
        imageW = atoi(firstOriginalElem->GetAttribute("width").c_str());
        imageH = atoi(firstOriginalElem->GetAttribute("height").c_str());
        const std::string firstOriginalFormat = firstOriginalElem->GetAttribute("format");
        isFloat = (firstOriginalFormat == "rgba32f" || firstOriginalFormat == "rgb32f");
    }

    //-----------------------------------------------------------------------------
    // 次元が 1d、2d、cube 以外ならエラーとします。
    if (g_AddInfo.m_Dimension != FtxDimension_1d   &&
        g_AddInfo.m_Dimension != FtxDimension_2d   &&
        !RIsCubMapDimension(g_AddInfo.m_Dimension))
    {
        if (!gStuff->openForPreview)
        {
            RShowError(globals, (dimensionStr + " texture is not supported").c_str());
            return false;
        }
        else // プレビューでは 1 枚目のみ 2D として表示します。
        {
            g_AddInfo.m_Dimension = FtxDimension_2d;
            imageD = 1;
        }
    }

    //-----------------------------------------------------------------------------
    // cube 配列ならエラーとします。
    if (g_AddInfo.m_Dimension == FtxDimension_CubeMapArray)
    {
        if (!gStuff->openForPreview)
        {
            RShowError(globals, "cube array texture is not supported");
            return false;
        }
        else // プレビューでは最初のキューブのみ表示します。
        {
            g_AddInfo.m_Dimension = FtxDimension_CubeMap;
            imageD = RImage::CUBE_FACE_COUNT;
        }
    }

    //-----------------------------------------------------------------------------
    // set image size
    gStuff->imageSize32.h = imageW;
    gStuff->imageSize32.v = imageH;
    if (g_AddInfo.m_Dimension == FtxDimension_CubeMap)
    {
        gStuff->imageSize32.h *= RImage::CUBE_HC_COUNT_H;
        gStuff->imageSize32.v *= RImage::CUBE_HC_COUNT_V;
    }
    else
    {
        gStuff->imageSize32.v *= imageD;
    }

    //-----------------------------------------------------------------------------
    // ピクセルデータのバッファーを確保します。
    const size_t pixelBytes = (isFloat) ? R_RGBA_FLOAT_BYTES : R_RGBA_BYTES;
    const size_t bitmapDataSize = pixelBytes * gStuff->imageSize32.h * gStuff->imageSize32.v;
    globals->m_pBitmapData = new uint8_t[bitmapDataSize];
    if (g_AddInfo.m_Dimension == FtxDimension_CubeMap)
    {
        // キューブマップなら先に白でフィルしておきます。
        FillPixelsValue(globals->m_pBitmapData, bitmapDataSize, 0xff, 1.0f, isFloat);
    }
    gStuff->depth = (isFloat) ? 32 : 8;

    //-----------------------------------------------------------------------------
    // ピクセルデータを取得します。
    const RXMLElement* streamsElem = (isBinary) ? nullptr : textureElem->FindElement("stream_array");
    const void* binTop = (isBinary) ? fileBuf + binOfs : nullptr;
    if (originalsElem != nullptr)
    {
        ParseFtxOriginalImages(&hasAlpha, globals, ftxPath, imageW, imageH, imageD,
            originalsElem, streamsElem, binTop);
    }
    else
    {
        if (!GetFtxDecodedPixels(&hasAlpha, globals, imageW, imageH, imageD,
            infoElem, streamsElem, binTop))
        {
            FillPixelsValue(globals->m_pBitmapData, bitmapDataSize, 0xff, 1.0f, isFloat);
        }
    }

    //-----------------------------------------------------------------------------
    // get user data
    // 中間ファイル 1.0.0 以前のユーザーデータは取得しません。
    RUserDataArray userDatas;
    const RXMLElement* userDatasElem = textureElem->FindElement("user_data_array", false);
    if (intermediateVer >= 101 && userDatasElem != NULL)
    {
        for (size_t iData = 0; iData < userDatasElem->nodes.size(); ++iData)
        {
            const RXMLElement* userDataElem = &(userDatasElem->nodes[iData]);
            userDatas.push_back(RUserData(userDataElem, streamsElem, binTop));
        }
        g_AddInfo.m_pPackedUserData = PackUserDataArray(g_AddInfo.m_PackedUserDataSize, userDatas);
    }

    //-----------------------------------------------------------------------------
    // 編集用コメントを取得します。
    GetFtxComment(&g_AddInfo.m_CommentLabel, &g_AddInfo.m_CommentCol, &g_AddInfo.m_CommentText, textureElem);

    //-----------------------------------------------------------------------------
    // get tool data
    GetFtxToolData(g_AddInfo.m_ToolData, g_AddInfo.m_UserToolData, xmlStr);

    //-----------------------------------------------------------------------------
    // set image mode
    gStuff->imageMode = plugInModeRGBColor;
    gStuff->planes = (hasAlpha) ? 4 : 3;

    //-----------------------------------------------------------------------------
    // test
    //for (size_t iPath = 0; iPath < g_AddInfo.m_OriginalPaths.size(); ++iPath) RNoteTrace(g_AddInfo.m_OriginalPaths[iPath].c_str());

    return true;
} // NOLINT(impl/function_size)

//-----------------------------------------------------------------------------
//! @brief リードを開始します。
//!        PiPL の FormatFlags が fmtCanCreateThumbnail なら
//!        サムネイル作成時（gStuff->openForPreview = TRUE）にも呼ばれます。
//!
//! @param[in,out] globals グローバルデータです。
//-----------------------------------------------------------------------------
static void DoReadStart(GPtr globals)
{
    //RNoteTrace("NintendoFtx: read start: %s: %d", RGetStringFromStr255(gStuff->fileSpec->name).c_str(), gStuff->openForPreview);

    //-----------------------------------------------------------------------------
    // init memory
    gStuff->imageRsrcData = NULL;
    globals->m_pAddInfo = new RAddInfo();
    globals->m_pBitmapData = NULL;
    uint8_t* fileBuf = NULL;

    //-----------------------------------------------------------------------------
    // read script param
    gResult = ReadScriptParamsOnRead(globals); // override params here
    if (gResult != noErr)
    {
        goto ErrorHandle;
    }

    //-----------------------------------------------------------------------------
    // ファイルサイズを取得します。
    const size_t fileSize = RGetFileSize(gStuff->dataFork);
    if (fileSize == 0)
    {
        gResult = userCanceledErr;
        goto ErrorHandle;
    }

    //-----------------------------------------------------------------------------
    // ファイルをすべてメモリーにリードします。
    fileBuf = new uint8_t[fileSize];
    gResult = RReadSome(gStuff->dataFork, fileBuf, fileSize);
    if (gResult != noErr)
    {
        goto ErrorHandle;
    }

    //-----------------------------------------------------------------------------
    // decode ftx
    if (!DecodeFtx(globals, fileBuf, fileSize))
    {
        gResult = userCanceledErr;
    }
    if (gResult != noErr)
    {
        goto ErrorHandle;
    }
    RFreeAndClearArray(fileBuf);

    //-----------------------------------------------------------------------------
    // create empty resource
    gStuff->imageRsrcData = PINewHandle(0);
    gStuff->imageRsrcSize = 0;

    //-----------------------------------------------------------------------------
    // set revert info
    RSetRevertInfo(globals);

    //-----------------------------------------------------------------------------
    // end
    return;

    //-----------------------------------------------------------------------------
    // error handle
ErrorHandle:
    globals->FreeBitmapData();
    globals->FreeAddInfo();
    RFreeAndClearArray(fileBuf);
    DisposeImageResources(globals);
}

//-----------------------------------------------------------------------------
//! @brief リードを継続します。
//!        AdvanceState は read start で呼ぶとエラー（formatBadParameters）になるので
//!        この関数が必要です。
//!
//! @param[in,out] globals グローバルデータです。
//-----------------------------------------------------------------------------
static void DoReadContinue(GPtr globals)
{
    //RMsgBox("NintendoFtx: read continue");

    //-----------------------------------------------------------------------------
    // dispose image resource if exist
    DisposeImageResources(globals);

    //-----------------------------------------------------------------------------
    // set bitmap data to Photoshop
    SetBitmapDataToPhotoshop(globals, globals->m_pBitmapData);

    //-----------------------------------------------------------------------------
    // set format resource data (for read)
    if (!gStuff->openForPreview)
    {
        SetFormatResourceData(globals, false);
    }

    //-----------------------------------------------------------------------------
    // don't read ICC profile
    gStuff->canUseICCProfiles = FALSE;

    //-----------------------------------------------------------------------------
    // end
    gStuff->data = NULL; // no continue
}

//-----------------------------------------------------------------------------
//! @brief リードを完了します。
//!        read start や read continue でエラーの場合は呼ばれません。
//!
//! @param[in,out] globals グローバルデータです。
//-----------------------------------------------------------------------------
static void DoReadFinish(GPtr globals)
{
    //RMsgBox("NintendoFtx: read finish");

    //-----------------------------------------------------------------------------
    // dispose image resource if exist
    DisposeImageResources(globals);

    //-----------------------------------------------------------------------------
    // write script param
    gResult = WriteScriptParamsOnRead(globals);

    //-----------------------------------------------------------------------------
    // free memory
    globals->FreeBitmapData();
    globals->FreeAddInfo();
}

//-----------------------------------------------------------------------------
//! @brief オプションの準備をします。
//!
//! @param[in,out] globals グローバルデータです。
//-----------------------------------------------------------------------------
static void DoOptionsPrepare(GPtr globals)
{
    //RMsgBox("option prepare: %d", PlayDialog());

    //-----------------------------------------------------------------------------
    // init max data
    gStuff->maxData = 0;

    //-----------------------------------------------------------------------------
    // set query for parameters
    gQueryForParameters = TRUE; // show option dialog
}

//-----------------------------------------------------------------------------
//! @brief オプションを開始します。
//!        option 関連は上書き保存のときは呼ばれません。
//!
//! @param[in,out] globals グローバルデータです。
//-----------------------------------------------------------------------------
static void DoOptionsStart(GPtr globals)
{
    //RMsgBox("option start: %s", getenv("PROCESSOR_ARCHITECTURE"));
    //RNoteDumpMemory(gStuff, sizeof(FormatRecord), "reserved");
    //RMsgBox("play info: %d", gStuff->descriptorParameters->playInfo);

    //-----------------------------------------------------------------------------
    // set revert info
    RSetRevertInfo(globals);

    //-----------------------------------------------------------------------------
    gStuff->data = NULL;    // no continue
}

//-----------------------------------------------------------------------------
//! @brief オプションを継続します。
//!
//! @param[in,out] globals グローバルデータです。
//-----------------------------------------------------------------------------
static void DoOptionsContinue(GPtr globals)
{
    R_UNUSED_VARIABLE(globals);
}

//-----------------------------------------------------------------------------
//! @brief オプションを完了します。
//!
//! @param[in,out] globals グローバルデータです。
//-----------------------------------------------------------------------------
static void DoOptionsFinish(GPtr globals)
{
    R_UNUSED_VARIABLE(globals);
}

//-----------------------------------------------------------------------------
//! @brief 見積もりの準備をします。
//!
//! @param[in,out] globals グローバルデータです。
//-----------------------------------------------------------------------------
static void DoEstimatePrepare(GPtr globals)
{
    gStuff->maxData = 0;
}

//-----------------------------------------------------------------------------
//! @brief コンフィグファイルで重み付け圧縮が有効なら true を返します。
//!
//! @param[in] configName コンフィグ名です。
//!
//! @return 重み付け圧縮が有効なら true を返します。
//-----------------------------------------------------------------------------
static bool IsWeightedCompressEnabled(const std::string& configName)
{
    const std::string configFolderPath = RGetConfigFolderPath(true);
    const std::string configPath = RGetConfigPathFromConfigName(configName, configFolderPath);
    const std::string configStr = RReadConfigFile(configPath);
    return RGetEnablesWeightedCompress(configStr);
}

//-----------------------------------------------------------------------------
//! @brief 見積もりを開始します。
//!        ここでオプションダイアログを表示します。
//!
//! @param[in,out] globals グローバルデータです。
//-----------------------------------------------------------------------------
static void DoEstimateStart(GPtr globals)
{
    //RMsgBox("estimate start: %d planes, xpa_plane = %d, xpa_index = %d", gStuff->planes, gStuff->transparencyPlane, gStuff->transparentIndex);
    //RMsgBox("overwrite: %d", !gQueryForParameters);
    //RMsgBox("dialog: %d", PlayDialog()); // アクションから実行以外は 1 ？
    #if 0
    // planes は .r ファイルの FormatMaxChannels 以下になっています。
    RMsgBox("estimate start: %d x %d, %d %d %d", gStuff->imageSize32.h, gStuff->imageSize32.v,
        gStuff->depth, gStuff->planes, gStuff->imageRsrcSize);
    #endif

    //-----------------------------------------------------------------------------
    // 3D テクスチャーコンバーターを初期化します（DLL をロードします）。
    globals->m_pTexCvtr = new RTexCvtr();
    RTexCvtr& texCvtr = *globals->m_pTexCvtr;
    if (!texCvtr.Initialize(globals))
    {
        gResult = userCanceledErr;
        globals->FreeTexConverter();
        gStuff->data = NULL; // no continue
        return;
    }

    //-----------------------------------------------------------------------------
    // init
    globals->m_pBitmapData = NULL;
    globals->m_pAddInfo = new RAddInfo();
    globals->m_pImageStatus = new RImageStatus();

    //-----------------------------------------------------------------------------
    // check alpha size
    if (RGetPsAlphaChanCount(globals) >= 2 && PlayDialog())
    {
        RShowWarning(globals, "Multi alpha channel exists\n\nOnly first alpha channel is saved");
    }

    //-----------------------------------------------------------------------------
    // ビットマップデータを Photoshop から取得します。
    const size_t colBytes = (gStuff->depth == 8) ? R_RGBA_BYTES : R_RGBA_FLOAT_BYTES;
    const size_t bitmapDataSize = colBytes * gStuff->imageSize32.h * gStuff->imageSize32.v;
    globals->m_pBitmapData = new uint8_t[bitmapDataSize];
    GetBitmapDataFromPhotoshop(globals->m_pBitmapData, globals);
    //if (RGetEnvVar("NINTENDO_PHOTOSHOP_NO_ADJ_XPA_RGB") != "1")
    //{
    //    AdjustTransparentPixelRgb(globals->m_pBitmapData, globals); // added (2013/02/21)
    //}

    //-----------------------------------------------------------------------------
    // set add info
    SetAddInfoForWrite(globals);
    if (gResult != noErr)
    {
        goto ErrorHandle;
    }

    //-----------------------------------------------------------------------------
    // レジストリからコンフィグ名とプレビューフラグをリードします。
    ReadRegistryParameters(globals);

    //-----------------------------------------------------------------------------
    // read script param
    if (!PlayDialog())
    {
        gResult = ReadScriptParamsOnWrite(globals);
        if (gResult != noErr)
        {
            goto ErrorHandle;
        }
    }

    //-----------------------------------------------------------------------------
    // check can save last format
    // 前回の次元、フォーマットで保存できなければ必ずダイアログ表示
    if (g_ImageStatus.m_LastFormat  == FtxFormat_Invalid             ||
        g_AddInfo.m_Dimension       != g_ImageStatus.m_LastDimension ||
        g_AddInfo.m_Format          != g_ImageStatus.m_LastFormat    ||
        (g_AddInfo.m_MipLevel == 1) != (g_ImageStatus.m_LastMipLevel == 1)) // ミップマップの有無のみ比較します。
    {
        gQueryForParameters = TRUE;
    }

    //-----------------------------------------------------------------------------
    // コンフィグファイルで重み付け圧縮が無効になっていれば、
    // 重み付け圧縮を OFF にしてダイアログを表示します。
    if ((RIsBc123TextureFormat(g_AddInfo.m_Format) || RIsEtcTextureFormat(g_AddInfo.m_Format)) &&
        g_AddInfo.m_WeightedCompress &&
        !IsWeightedCompressEnabled(g_AddInfo.m_ConfigName))
    {
        //RNoteTrace("disable weighted compress");
        g_AddInfo.m_WeightedCompress = false;
        gQueryForParameters = TRUE;
    }

    //-----------------------------------------------------------------------------
    // do format dialog
    if (gQueryForParameters)
    {
        gQueryForParameters = FALSE;
        if (PlayDialog())
        {
            // ファイル形式オプションダイアログを表示します。
            if (!RDoFormatDialog(globals))
            {
                //RMsgBox("format dialog cancel: %x", (uint32_t)hDllInstance);
                gResult = userCanceledErr;
                goto ErrorHandle;
            }
        }
    }

    //-----------------------------------------------------------------------------
    // レジストリにコンフィグ名とプレビューフラグをライトします。
    WriteRegistryParameters(globals);

    //-----------------------------------------------------------------------------
    // ファイルをライトする際のデータサイズの最小・最大値を設定します。
    // 実際にライトするサイズより小さくても構いません。
    {
        const int pixelCount = gStuff->imageSize32.h * gStuff->imageSize32.v;
        gStuff->minDataBytes = pixelCount * R_RGB_BYTES;
        gStuff->maxDataBytes = pixelCount * R_RGBA_FLOAT_BYTES;
    }

    //-----------------------------------------------------------------------------
    // end
    gStuff->data = NULL; // no continue
    return;

    //-----------------------------------------------------------------------------
    // error handle
ErrorHandle:
    globals->FreeTexConverter();
    globals->FreeBitmapData();
    globals->FreeAddInfo();
    globals->FreeImageStatus();

    gStuff->data = NULL; // no continue
}

//-----------------------------------------------------------------------------
//! @brief 見積もりを継続します。
//!
//! @param[in,out] globals グローバルデータです。
//-----------------------------------------------------------------------------
static void DoEstimateContinue(GPtr globals)
{
    R_UNUSED_VARIABLE(globals);
}

//-----------------------------------------------------------------------------
//! @brief 見積もりを完了します。
//!
//! @param[in,out] globals グローバルデータです。
//-----------------------------------------------------------------------------
static void DoEstimateFinish(GPtr globals)
{
    R_UNUSED_VARIABLE(globals);
}

//-----------------------------------------------------------------------------
//! @brief ライトの準備をします。
//!
//! @param[in,out] globals グローバルデータです。
//-----------------------------------------------------------------------------
static void DoWritePrepare(GPtr globals)
{
    gStuff->maxData = 0;
}

//-----------------------------------------------------------------------------
//! @brief ftx ファイルの文字列から作成情報を取得します。
//!
//! @param[in] ftxStr ftx ファイルの文字列です。
//!
//! @return 作成情報の文字列を返します。
//-----------------------------------------------------------------------------
static std::string GetCreateInfoFromFtx(const std::string& ftxStr)
{
    const size_t iStart = ftxStr.find("\t<create");
    if (iStart != std::string::npos)
    {
        const size_t iEnd = ftxStr.find("/>", iStart);
        if (iEnd != std::string::npos)
        {
            return ftxStr.substr(iStart, iEnd + 2 - iStart) + R_ENDL;
        }
    }
    return "";
}

//-----------------------------------------------------------------------------
//! @brief GPU エンコーディングモードを取得します。
//!
//! @param[in] configStr コンフィグファイルをリードした文字列です。
//!
//! @return GPU エンコーディングモードを返します。
//-----------------------------------------------------------------------------
static std::string GetGpuEncodingMode(const std::string& configStr)
{
    std::string gpuEncoding;
    if (RFindConfigVariable(configStr, "gpu_encoding") != std::string::npos)
    {
        gpuEncoding = RTrimString(RGetConfigStringValue(configStr, "gpu_encoding", ""));
            // 現在の実装ではダブルクォートで囲われていなくても = 以降の文字列を取得するので、
            // bool 型でも true / false が取得されます。
        if (!gpuEncoding.empty())
        {
            if (gpuEncoding != "true" && gpuEncoding != "false" && gpuEncoding != "auto")
            {
                gpuEncoding = "";
            }
        }
        else
        {
            gpuEncoding = RGetConfigBooleanValue(configStr, "gpu_encoding", true) ? "true" : "false";
        }
    }
    return gpuEncoding;
}

//-----------------------------------------------------------------------------
//! @brief 画像編集オプションを設定します。
//!
//! @param[in,out] options ファイルオプション文字列配列です。
//! @param[in] addInfo 付加情報です。
//! @param[in] adjustsTransparentRgb 透明ピクセルの RGB 成分を調整するなら true です。
//-----------------------------------------------------------------------------
static void SetEditingOptions(
    RStringArray& options,
    const RAddInfo& addInfo,
    const bool adjustsTransparentRgb
)
{
    //-----------------------------------------------------------------------------
    // 切り抜き
    if (!addInfo.m_CropRect.empty())
    {
        RStringArray tokens;
        if (RTokenizeString(tokens, addInfo.m_CropRect, ",") == 4)
        {
            const int x = atoi(tokens[0].c_str());
            const int y = atoi(tokens[1].c_str());
            const int w = atoi(tokens[2].c_str());
            const int h = atoi(tokens[3].c_str());
            if (x >= 0 && y >= 0 && w >= 1 && h >= 1)
            {
                options.push_back("--crop=\"" + addInfo.m_CropRect + "\"");
            }
        }
    }

    //-----------------------------------------------------------------------------
    // サイズ変更
    if (!addInfo.m_ResizeWH.empty())
    {
        RStringArray tokens;
        if (RTokenizeString(tokens, addInfo.m_ResizeWH, "x") == 2)
        {
            const int rw = atoi(tokens[0].c_str());
            const int rh = atoi(tokens[1].c_str());
            if (rw >= 1 && rh >= 1)
            {
                options.push_back("--resize-w=\"" + RGetNumberString(rw) + "\"");
                options.push_back("--resize-h=\"" + RGetNumberString(rh) + "\"");
                if (!addInfo.m_ResizeFilter.empty())
                {
                    options.push_back("--resize-filter=\"" + addInfo.m_ResizeFilter + "\"");
                }
                options.push_back("--resize-gamma=\"" + std::string(RBoolStr(addInfo.m_ResizesInLinear)) + "\"");
            }
        }
    }

    //-----------------------------------------------------------------------------
    // 透明ピクセルの RGB 成分調整
    options.push_back("--adjust-transparent-rgb=\"" + std::string(RBoolStr(adjustsTransparentRgb)) + "\"");
}

//-----------------------------------------------------------------------------
//! @brief ftx ファイルを出力します。
//!
//! @param[in,out] globals グローバルデータです。
//!
//! @return 処理成功なら true を返します。
//-----------------------------------------------------------------------------
static bool OutputFtx(GPtr globals)
{
    RTexCvtr& texCvtr = *globals->m_pTexCvtr;

    //-----------------------------------------------------------------------------
    // コンフィグファイルをリードします。
    const std::string configFolderPath = RGetConfigFolderPath(true);
    const std::string configPath = RGetConfigPathFromConfigName(g_AddInfo.m_ConfigName, configFolderPath);
    const std::string configStr = RReadConfigFile(configPath);

    //-----------------------------------------------------------------------------
    // コンフィグファイルから各設定を取得します。
    std::string projectRootPath = RTrimString(RGetConfigStringValue(configStr, "project_root", ""));
    projectRootPath = (!projectRootPath.empty()) ?
        RGetFullFilePath(projectRootPath, true) : "";
    const bool disablesOriginalImage = RGetConfigBooleanValue(configStr, "disable_original_image", false);
    const std::string gpuEncoding = GetGpuEncodingMode(configStr);

    //-----------------------------------------------------------------------------
    // get input paths
    RStringArray inputPaths;
    const std::string inputPath = (g_AddInfo.OverwritesInputPath()) ?
        std::string(g_AddInfo.m_OverwriteInputPath) : RGetDocumentFilePath(globals);
    inputPaths.push_back(RGetUnixFilePath(inputPath));

    //-----------------------------------------------------------------------------
    // get original paths
    RStringArray originalPaths = g_AddInfo.m_OriginalPaths;
    if (originalPaths.empty())
    {
        originalPaths = inputPaths;
    }
    //RNoteTrace("original paths:\n%s", RJoinStrings(originalPaths, ", ").c_str());

    //-----------------------------------------------------------------------------
    // get output path
    std::string ftxPath = RGetUnixFilePath(RGetFinalFilePath(globals));
    const std::string ext = RGetExtensionFromFilePath(ftxPath);
    if (ext != FTXA_EXTENSION && ext != FTXB_EXTENSION)
    {
        ftxPath = RGetNoExtensionFilePath(ftxPath) + "." + FTXB_EXTENSION;
    }
    //RNoteTrace("ftx path: %s", ftxPath.c_str());

    //-----------------------------------------------------------------------------
    // test
    #if 0
    const int bitmapW = 4;
    const int bitmapH = 3;
    const int bitmapD = 1;
    static const uint8_t bitmapBuf[bitmapW * bitmapH * bitmapD][4] =
    {
        // RGBW
        // CMYK
        // rgbw
        { 0xff, 0x00, 0x00, 0xff }, { 0x00, 0xff, 0x00, 0xff }, { 0x00, 0x00, 0xff, 0xff }, { 0xff, 0xff, 0xff, 0xff },
        { 0x00, 0xff, 0xff, 0xff }, { 0xff, 0x00, 0xff, 0xff }, { 0xff, 0xff, 0x00, 0xff }, { 0x00, 0x00, 0x00, 0xff },
        { 0x80, 0x00, 0x00, 0xff }, { 0x00, 0x80, 0x00, 0xff }, { 0x00, 0x00, 0x80, 0xff }, { 0x80, 0x80, 0x80, 0xff },
    };
    #endif

    //-----------------------------------------------------------------------------
    // create bitmap buffer
    int bitmapW = gStuff->imageSize32.h;
    int bitmapH = gStuff->imageSize32.v;
    int bitmapD = 1; // 水平十字／垂直十字キューブマップの場合も 1 を指定します。

    if (g_AddInfo.m_Dimension == FtxDimension_3d      ||
        g_AddInfo.m_Dimension == FtxDimension_1dArray ||
        g_AddInfo.m_Dimension == FtxDimension_2dArray)
    {
        bitmapW = g_ImageStatus.m_3DFaceW;
        bitmapH = g_ImageStatus.m_3DFaceH;
        bitmapD = g_ImageStatus.m_3DFaceD;
    }

    const bool isFloat = (gStuff->depth == 32);
    const size_t colBytes = (isFloat) ? R_RGBA_FLOAT_BYTES : R_RGBA_BYTES;
    const size_t faceBytes   = colBytes * bitmapW * bitmapH;
    const size_t bitmapBytes = faceBytes * bitmapD;
    uint8_t* pBitmapBuf = new uint8_t[bitmapBytes];
    memcpy(pBitmapBuf, globals->m_pBitmapData, bitmapBytes);

    //-----------------------------------------------------------------------------
    // set file options
    RStringArray options;
    options.push_back("--output=\"" + ftxPath + "\"");

    if (!projectRootPath.empty())
    {
        options.push_back("--project-root=\"" + projectRootPath + "\"");
    }
    if (disablesOriginalImage)
    {
        options.push_back("--disable-original-image");
    }
    if (!g_AddInfo.m_ConfigName.empty())
    {
        options.push_back("--dcc-preset=\"" + g_AddInfo.m_ConfigName + "\""); // マージ ON でも常に優先
    }

    if (g_AddInfo.Merges())
    {
        options.push_back("--merge=\"" + RGetUnixFilePath(g_AddInfo.m_MergePath) + "\"");
    }
    else
    {
        options.push_back("--hint=\"" + g_AddInfo.m_Hint + "\"");
        if (g_AddInfo.m_LinearFlag != RLinearFlag::LINEAR_NONE)
        {
            options.push_back("--linear=\"" + RGetOptStringFromLinearFlag(g_AddInfo.m_LinearFlag) + "\"");
        }
        options.push_back("--dimension=\"" + RGetTextureDimensionString(g_AddInfo.m_Dimension) + "\"");
        options.push_back("--format=\"" + RGetTextureFormatString(g_AddInfo.m_Format) + "\"");
        options.push_back("--weighted-compress=\"" + std::string(RBoolStr(g_AddInfo.m_WeightedCompress)) + "\"");
        options.push_back("--swizzle=\"" + RGetNumberString(g_AddInfo.m_InitialSwizzle) + "\"");
        options.push_back("--mip-level=\"" + RGetNumberString(g_AddInfo.m_MipLevel) + "\"");
        options.push_back("--mip-gen-filter=\"" + RGetMipGenFilterString(g_AddInfo.m_MipGenFilter) + "\"");
        options.push_back("--comp-sel=\"" + RGetCompSelOptionString(g_AddInfo.m_CompSel) + "\"");
    }

    if (!gpuEncoding.empty())
    {
        options.push_back("--gpu-encoding=\"" + gpuEncoding + "\"");
    }

    if (!g_AddInfo.m_CommentText.empty())
    {
        options.push_back("--comment=\"" + g_AddInfo.m_CommentText + "\""); // マージ ON でも常に優先
    }

    const bool adjustsTransparentRgb = (gStuff->imageMode != plugInModeIndexedColor &&
        RGetPsAlphaChanCount(globals) == 0);
    SetEditingOptions(options, g_AddInfo, adjustsTransparentRgb);

    // テクスチャーエクスポートプラグインでテンポラリーの ftx ファイルを出力して、
    // それを TGA ファイルに変換する場合、サイズエラー判定をスキップします。
    if (g_AddInfo.m_IsExportTexture && g_AddInfo.m_Hint == "export_texture_tga_only_special")
    {
        options.push_back("--disable-size-error");
    }

    if (!g_AddInfo.m_OutputOriginalPath.empty())
    {
        // 元画像の画像ファイルを同時に出力します。
        options.push_back("--output-original=\"" + g_AddInfo.m_OutputOriginalPath + "\"");
        if (!RGetConfigBooleanValue(configStr, "tga_rle", true))
        {
            options.push_back("--no-tga-rle");
        }
    }

    //for (size_t iOp = 0; iOp < options.size(); ++iOp) RNoteTrace("op%02d: %s", iOp, options[iOp].c_str());

    //-----------------------------------------------------------------------------
    // コンバーターにビットマップデータをリードさせます。
    const bool hasAlpha = (RGetPsAlphaChanCount(globals) != 0 ||
        g_ImageStatus.m_XpaMode != RImage::OPA);
    bool success = texCvtr.ReadBitmapData(pBitmapBuf, bitmapW, bitmapH, bitmapD, hasAlpha, isFloat,
        RUnicodeCArray(inputPaths).GetCArray(),
        RUnicodeCArray(originalPaths).GetCArray(),
        RUnicodeCArray(options).GetCArray());
    delete[] pBitmapBuf;
    if (!success)
    {
        RShowError(globals, (std::string("Cannot read bitmap data\n") + texCvtr.GetErrorShiftJisString()).c_str());
        return false;
    }

    //-----------------------------------------------------------------------------
    // set create info
    if (!g_AddInfo.m_CreateInfo.empty())
    {
        success = texCvtr.SetCreateInfo(RGetUnicodeFromShiftJis(g_AddInfo.m_CreateInfo).c_str());
        //RNoteTrace("set create info: %d\n", success);
    }

    //-----------------------------------------------------------------------------
    // set user data
    if (g_AddInfo.m_pPackedUserData != NULL && !g_AddInfo.Merges())
    {
        //RUserDataArray userDatas;
        //UnpackUserDataArray(userDatas, g_AddInfo.m_pPackedUserData, g_AddInfo.m_PackedUserDataSize);
        success = texCvtr.SetUserData(g_AddInfo.m_pPackedUserData, g_AddInfo.m_PackedUserDataSize);
        //RNoteTrace("set user data: %d\n", success);
    }

    //-----------------------------------------------------------------------------
    // 編集用コメントラベル、編集用コメントカラー文字列を設定します。
    if (!g_AddInfo.m_CommentLabel.empty())
    {
        success = texCvtr.SetCommentLabel(g_AddInfo.m_CommentLabel.c_str(), g_AddInfo.m_CommentLabel.size());
    }
    if (!g_AddInfo.m_CommentCol.empty())
    {
        success = texCvtr.SetCommentColor(g_AddInfo.m_CommentCol.c_str(), g_AddInfo.m_CommentCol.size());
    }

    //-----------------------------------------------------------------------------
    // set tool data
    if (!g_AddInfo.m_ToolData.empty() && !g_AddInfo.Merges())
    {
        success = texCvtr.SetToolData(g_AddInfo.m_ToolData.c_str(), g_AddInfo.m_ToolData.size());
    }

    if (!g_AddInfo.m_UserToolData.empty() && !g_AddInfo.Merges())
    {
        success = texCvtr.SetUserToolData(g_AddInfo.m_UserToolData.c_str(), g_AddInfo.m_UserToolData.size());
    }

    //-----------------------------------------------------------------------------
    // convert to data
    const void* pData;
    size_t dataSize;
    if (!texCvtr.ConvertToData(&pData, &dataSize))
    {
        RShowError(globals, (std::string("Cannot convert to data\n") + texCvtr.GetErrorShiftJisString()).c_str());
        return false;
    }

    //-----------------------------------------------------------------------------
    // write to file
    gResult = RWriteSome(gStuff->dataFork, pData, dataSize);

    //-----------------------------------------------------------------------------
    // 上書きでなければ作成情報を ftx ファイルから取得します。
    if (g_AddInfo.m_CreateInfo.empty() ||
        inputPaths[0] != ftxPath)
    {
        const size_t HEAD_MAX_SIZE = 1024;
        const size_t headSize = RMin(dataSize, HEAD_MAX_SIZE);
        g_AddInfo.m_CreateInfo = GetCreateInfoFromFtx(
            std::string(reinterpret_cast<const char*>(pData), headSize));
        //RNoteTrace("update crate info: %s\r\n%s\r\n%s", g_AddInfo.m_CreateInfo.c_str(), inputPaths[0].c_str(), ftxPath.c_str());
    }

    //-----------------------------------------------------------------------------
    // 3D テクスチャーコンバーターのメモリーをクリアします。
    texCvtr.Clear();

    //-----------------------------------------------------------------------------
    // nvtt の DLL が Photoshop 終了までアンロードされないようにします。
    if (RGetEnvVar("NINTENDO_PHOTOSHOP_PIN_NVTT_DLL") != "0")
    {
        texCvtr.PinNvttDll();
    }

    return true;
} // NOLINT(impl/function_size)

//-----------------------------------------------------------------------------
//! @brief ライトを開始します。
//!
//! @param[in,out] globals グローバルデータです。
//-----------------------------------------------------------------------------
static void DoWriteStart(GPtr globals)
{
    //RMsgBox("write start: %s", RGetStringFromStr255(gStuff->fileSpec->name).c_str()); // file name (temporary)

    //-----------------------------------------------------------------------------
    // ftx ファイルを出力します。
    RWaitCursor* pWaitCursor = (!g_AddInfo.m_IsExportTexture) ? new RWaitCursor() : nullptr;
    if (!OutputFtx(globals))
    {
        gResult = userCanceledErr;
    }
    if (pWaitCursor != nullptr)
    {
        delete pWaitCursor;
    }
    if (gResult != noErr)
    {
        goto ErrorHandle;
    }

    //-----------------------------------------------------------------------------
    // set format resource data (write)
    SetFormatResourceData(globals, true);

    //-----------------------------------------------------------------------------
    // don't write ICC profile
    gStuff->canUseICCProfiles = FALSE;

    //-----------------------------------------------------------------------------
    // end
    gStuff->data = NULL; // no continue
    return;

    //-----------------------------------------------------------------------------
    // error handle
ErrorHandle:
    globals->FreeTexConverter();
    globals->FreeBitmapData();
    globals->FreeAddInfo();
    globals->FreeImageStatus();

    gStuff->data = NULL; // no continue
}

//-----------------------------------------------------------------------------
//! @brief ライトを継続します。
//!
//! @param[in,out] globals グローバルデータです。
//-----------------------------------------------------------------------------
static void DoWriteContinue(GPtr globals)
{
    R_UNUSED_VARIABLE(globals);
}

//-----------------------------------------------------------------------------
//! @brief ライトを完了します。
//!
//! @param[in,out] globals グローバルデータです。
//-----------------------------------------------------------------------------
static void DoWriteFinish(GPtr globals)
{
    //RMsgBox("write finish");
    gResult = WriteScriptParamsOnWrite(globals); // should be different for read/write

    //-----------------------------------------------------------------------------
    // free memory
    globals->FreeTexConverter();
    globals->FreeBitmapData();
    globals->FreeAddInfo();
    globals->FreeImageStatus();
}

//-----------------------------------------------------------------------------
//! @brief ファイルをこのプラグインで処理するかフィルタします。
//!
//! @param[in] globals グローバルデータです。
//-----------------------------------------------------------------------------
static void DoFilterFile(GPtr globals)
{
    //RNoteTrace("NintendoFtx: filter file: %s", gStuff->fileSpec->name + 1);

    //-----------------------------------------------------------------------------
    // check result
    if (gResult != noErr)
    {
        return;
    }
}

//-----------------------------------------------------------------------------
//! @brief プラグインの処理セレクタの名前データです（デバッグ用）。
//-----------------------------------------------------------------------------
static const char* s_SelectorNames[] =
{
    "about",                // 0
    "read prepare",         // 1
    "read start",           // 2
    "read continue",        // 3
    "read finish",          // 4
    "options prepare",      // 5
    "options start",        // 6
    "options continue",     // 7
    "options finish",       // 8
    "estimate prepare",     // 9
    "estimate start",       // 10
    "estimate continue",    // 11
    "estimate finish",      // 12
    "write prepare",        // 13
    "write start",          // 14
    "write continue",       // 15
    "write finish",         // 16
    "filter file"           // 17
};

//-----------------------------------------------------------------------------
//! @brief プラグインのメイン関数です。
//-----------------------------------------------------------------------------
#if (PS_API_VERSION >= 1100)
DLLExport MACPASCAL void PluginMain(
    const int16 selector,
    FormatRecordPtr formatParamBlock,
    intptr_t* data,
    int16* result
)
#else
MACPASCAL void PluginMain(
    const int16 selector,
    FormatRecord* formatParamBlock,
    int32* data,
    int16* result
)
#endif
{
    //RNoteTrace("pm: %-17s: data = %p", s_SelectorNames[selector], data);
    //RCheckMemoryStatus(s_SelectorNames[selector]);

    //-----------------------------------------------------------------------------
    // do each proc
    if (selector == formatSelectorAbout)
    {
        //-----------------------------------------------------------------------------
        // do about
        AboutRecordPtr aboutRecord = reinterpret_cast<AboutRecordPtr>(formatParamBlock);
        sSPBasic = aboutRecord->sSPBasic;
        RDoAboutDialog(aboutRecord);
    }
    else
    {
        //-----------------------------------------------------------------------------
        // alloc global
        sSPBasic = formatParamBlock->sSPBasic;

        Ptr globalPtr = NULL;
        GPtr globals = NULL;

        // alloc & init globals
        #if (PS_API_VERSION >= 1100)
        globalPtr = AllocateGlobals(result,
            formatParamBlock, formatParamBlock->handleProcs,
            sizeof(Globals), data, InitGlobals);
        #else
        globalPtr = AllocateGlobals((uint32_t)result,
            reinterpret_cast<uint32_t>(formatParamBlock), formatParamBlock->handleProcs,
            sizeof(Globals), data, InitGlobals);
        #endif

        if (globalPtr == NULL)
        {
            *result = memFullErr;
            return;
        }

        globals = reinterpret_cast<GPtr>(globalPtr);

        //-----------------------------------------------------------------------------
        // 32bit の座標を使用します。
        gStuff->PluginUsing32BitCoordinates = 1;

        //-----------------------------------------------------------------------------
        // dispatch selector
        try
        {
            switch (selector)
            {
                case formatSelectorReadPrepare:
                    DoReadPrepare(globals);
                    break;
                case formatSelectorReadStart:
                    DoReadStart(globals);
                    break;
                case formatSelectorReadContinue:
                    DoReadContinue(globals);
                    break;
                case formatSelectorReadFinish:
                    DoReadFinish(globals);
                    break;

                case formatSelectorOptionsPrepare:
                    DoOptionsPrepare(globals);
                    break;
                case formatSelectorOptionsStart:
                    DoOptionsStart(globals);
                    break;
                case formatSelectorOptionsContinue:
                    DoOptionsContinue(globals);
                    break;
                case formatSelectorOptionsFinish:
                    DoOptionsFinish(globals);
                    break;

                case formatSelectorEstimatePrepare:
                    DoEstimatePrepare(globals);
                    break;
                case formatSelectorEstimateStart:
                    DoEstimateStart(globals);
                    break;
                case formatSelectorEstimateContinue:
                    DoEstimateContinue(globals);
                    break;
                case formatSelectorEstimateFinish:
                    DoEstimateFinish(globals);
                    break;

                case formatSelectorWritePrepare:
                    DoWritePrepare(globals);
                    break;
                case formatSelectorWriteStart:
                    DoWriteStart(globals);
                    break;
                case formatSelectorWriteContinue:
                    DoWriteContinue(globals);
                    break;
                case formatSelectorWriteFinish:
                    DoWriteFinish(globals);
                    break;

                case formatSelectorFilterFile:
                    DoFilterFile(globals);
                    break;

                default:
                    break;
            }
        }
        catch (int16 inError)
        {
            *result = inError;
        }
        catch (...)
        {
            *result = -1; // program error
        }

        //-----------------------------------------------------------------------------
        // unlock data
        if (reinterpret_cast<Handle>(*data) != NULL)
        {
            PIUnlockHandle(reinterpret_cast<Handle>(*data));
        }
    }

    if (selector == formatSelectorAbout          ||
        selector == formatSelectorWriteFinish    ||
        selector == formatSelectorReadFinish     ||
        selector == formatSelectorOptionsFinish  ||
        selector == formatSelectorEstimateFinish ||
        selector == formatSelectorFilterFile     ||
        *result != noErr)
    {
        PIUSuitesRelease();
    }
} // NOLINT(impl/function_size)

