﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

// DoOptionDialog OptionDialogProc InitOptionDialog UpdateOptionDialog
// UpdatePreviewBitmap SaveSettings LoadSettings
// DoAboutDialog

//=============================================================================
// include
//=============================================================================
#include "NintendoDistanceField.h"
#include "NintendoDistanceFieldUi.h"
#include "../NintendoFtx/NpsVersion.h"
#include "../NintendoFtx/NpsPreview.h"
#include "Resource.h"
#include <commctrl.h> // for slider & tooltip

using namespace nn::gfx::tool::nps;

//=============================================================================
// variables
//=============================================================================

//-----------------------------------------------------------------------------
// preview
static RPreview s_Preview; //!< プレビュー表示です。
static volatile bool s_UpdatingPreview = false; //!< プレビュー更新中なら true です。

//-----------------------------------------------------------------------------
//! @brief プレビュー画像を更新します。
//!
//! @param[in] globals グローバルデータです。
//! @param[in] hDlg ダイアログハンドルです。
//-----------------------------------------------------------------------------
static void UpdatePreviewBitmap(GPtr globals, HWND hDlg)
{
    //-----------------------------------------------------------------------------
    // プレビューが無効なら何もしません。
    s_Preview.m_UpdateTimerCount = 0;
    if (!globals->m_DisplaysPreview)
    {
        return;
    }

    //-----------------------------------------------------------------------------
    // プレビュー更新中なら何もしません。
    if (s_UpdatingPreview)
    {
        //RNoteTrace("skip update preview: %d", gParams->m_MaxDistance);
        return;
    }
    s_UpdatingPreview = true;

    //-----------------------------------------------------------------------------
    // フィルター後のビットマップデータを更新します。
    bool dstChanged = false;
    if (globals->m_PreviewParams != *gParams)
    {
        //RNoteTrace("update preview: %d", gParams->m_MaxDistance);

        //-----------------------------------------------------------------------------
        // フィルタ処理します。
        const bool isRough = !gParams->m_EdgeWrap;
        RProgress* pProgress = NULL;
        if (!isRough)
        {
            // スライダーを無効にします。
            // 無効にしないとスライダーをドラッグしてこの関数が呼ばれた際に、
            // マウスを動かすとさらにスライダーが移動してしまいます。
            EnableWindow(GetDlgItem(hDlg, IDC_MAX_DISTANCE_SLIDER), FALSE);

            pProgress = new RLocalProgress(1, IDD_PROGRESS, hDlg,
                IDC_PROGRESS_BAR, IDC_PROGRESS_TEXT, "Calculating Preview ...");
        }

        sPSUIHooks->SetCursor(kPICursorWatch);
        bool success = DoFilter(globals, globals->m_pDstBitmapData, globals->m_pBinaryData,
            gBigDoc->imageSize32.h, gBigDoc->imageSize32.v, gBigDoc->imageSize32.h, isRough,
            pProgress);
        sPSUIHooks->SetCursor(kPICursorArrow);

        if (pProgress != NULL)
        {
            delete pProgress;

            // スライダーを有効にします。
            EnableWindow(GetDlgItem(hDlg, IDC_MAX_DISTANCE_SLIDER), TRUE);
        }

        globals->m_IsPreviewFiltered = success && !isRough;
        if (success)
        {
            globals->m_PreviewParams = *gParams;
            dstChanged = true;
        }
        else
        {
            memset(globals->m_pDstBitmapData, 0x00,
                gBigDoc->imageSize32.h * gBigDoc->imageSize32.v * R_RGBA_COUNT);
            dstChanged = true;
        }
    }

    //-----------------------------------------------------------------------------
    // プレビュー画像を更新します。
    if (dstChanged)
    {
        //RTimeMeasure tm1;
        s_Preview.UpdateWinBmp(hDlg, globals->m_pDstBitmapData);
        //RNoteTrace("up win bmp: %6.2f", tm1.GetMilliSec());
    }

    s_UpdatingPreview = false;
}

//-----------------------------------------------------------------------------
//! @brief スライダーのつまみの位置を設定します。
//!        値がスライダーの範囲外なら範囲を拡張します。
//!
//! @param[in] slider スライダーのウィンドウハンドルです。
//! @param[in] pos つまみの位置です。
//-----------------------------------------------------------------------------
static void SetSliderPosAndRange(HWND slider, const int pos)
{
    if (pos < static_cast<int>(SendMessage(slider, TBM_GETRANGEMIN, 0, 0)))
    {
        SendMessage(slider, TBM_SETRANGEMIN, FALSE, pos);
    }
    if (pos > static_cast<int>(SendMessage(slider, TBM_GETRANGEMAX, 0, 0)))
    {
        SendMessage(slider, TBM_SETRANGEMAX, FALSE, pos);
    }
    SendMessage(slider, TBM_SETPOS, TRUE, pos);
}

//-----------------------------------------------------------------------------
//! @brief オプションダイアログを更新します。
//!
//! @param[in,out] globals グローバルデータです。
//! @param[in] hDlg ダイアログハンドルです。
//-----------------------------------------------------------------------------
static void UpdateOptionDialog(GPtr globals, HWND hDlg)
{
    //-----------------------------------------------------------------------------
    // option
    SetDlgItemInt(hDlg, IDC_MAX_DISTANCE, gParams->m_MaxDistance, FALSE);

    CheckDlgButton(hDlg, IDC_EDGE_WRAP,
        (gParams->m_EdgeWrap) ? BST_CHECKED : BST_UNCHECKED);

    CheckDlgButton(hDlg, IDC_NORMALIZE,
        (gParams->m_Normalize) ? BST_CHECKED : BST_UNCHECKED);

    //-----------------------------------------------------------------------------
    // preview
    s_Preview.UpdateControl(hDlg);
}

//-----------------------------------------------------------------------------
//! @brief オプションダイアログを初期化します。
//!
//! @param[in,out] globals グローバルデータです。
//! @param[in] hDlg ダイアログハンドルです。
//-----------------------------------------------------------------------------
static void InitOptionDialog(GPtr globals, HWND hDlg)
{
    //-----------------------------------------------------------------------------
    // center dialog
    CenterDialog(hDlg);

    //-----------------------------------------------------------------------------
    // max distance
    const int INITIAL_DISTANCE_MAX = 64;
    HWND maxDistanceSlider = GetDlgItem(hDlg, IDC_MAX_DISTANCE_SLIDER);
    SendMessage(maxDistanceSlider, TBM_SETRANGE   , TRUE,
        MAKELPARAM(RParameters::DISTANCE_MIN, INITIAL_DISTANCE_MAX));
    SendMessage(maxDistanceSlider, TBM_SETPAGESIZE, TRUE, 10);
    //SetSliderPosAndRange(maxDistanceSlider, gParams->m_MaxDistance);

    //-----------------------------------------------------------------------------
    // preview
    static const int PREVIEW_W = 256 + 208 + 2;
    static const int PREVIEW_H = 256 + 2;

    RECT previewRect;
    GetWindowRect(GetDlgItem(hDlg, IDC_PREVIEW_AREA), &previewRect);
    ScreenToClient(hDlg, reinterpret_cast<LPPOINT>(&previewRect.left));
    previewRect.right  = previewRect.left + PREVIEW_W;
    previewRect.bottom = previewRect.top  + PREVIEW_H;

    s_Preview.InitMemory(hDlg, previewRect, gBigDoc->imageSize32.h, gBigDoc->imageSize32.v);
    s_Preview.CenterScroll();
    s_Preview.SetControlId(IDC_ZOOM_IN, IDC_ZOOM_OUT, IDC_ZOOM_VALUE);
    s_UpdatingPreview = false;

    //-----------------------------------------------------------------------------
    // update dialog
    UpdateOptionDialog(globals, hDlg);

    //-----------------------------------------------------------------------------
    // focus OK button
    SetFocus(GetDlgItem(hDlg, IDOK));

    //-----------------------------------------------------------------------------
    // 各 UI にツールチップを追加します。
    const bool isJP = RIsJapaneseUI();
    RAddToolTip(hDlg, IDOK, (isJP) ? "フィルタを実行します" : "Applies the filter");
    RAddToolTip(hDlg, IDCANCEL, (isJP) ? "キャンセルします" : "Cancels");

    RAddToolTip(hDlg, IDC_ZOOM_OUT, (isJP) ? "プレビュー画像を縮小します" : "Zooms out on the preview image");
    RAddToolTip(hDlg, IDC_ZOOM_IN, (isJP) ? "プレビュー画像を拡大します" : "Zooms in on the preview image");

    const char* pMaxDistanceTip = (isJP) ? "白黒境界からの最大距離（ピクセル）を指定します" :
        "Specifies the max distance from the white and black border";
    RAddToolTip(hDlg, IDC_MAX_DISTANCE_LBL   , pMaxDistanceTip);
    RAddToolTip(hDlg, IDC_MAX_DISTANCE       , pMaxDistanceTip);
    RAddToolTip(hDlg, IDC_MAX_DISTANCE_SLIDER, pMaxDistanceTip);
    RAddToolTip(hDlg, IDC_EDGE_WRAP, (isJP) ?
        "画像が繰り返しているとして端のピクセルを処理するなら ON にします。\n"
        "画像の端のピクセルを引き伸ばして処理するなら OFF にします" :
        "Processes edge pixels in the case of a repeating height map if ON is selected. \n"
        "Extends the edge pixels of an image if OFF is selected");
    RAddToolTip(hDlg, IDC_NORMALIZE, (isJP) ? "結果を 0～255 に正規化するなら ON にします" :
            "Normalizes the result in [0,255] if ON is selected");
}

//-----------------------------------------------------------------------------
//! @brief オプションダイアログのカーソルを設定します。
//!
//! @param[in] globals グローバルデータです。
//! @param[in] pos カーソル座標です。
//-----------------------------------------------------------------------------
static void SetOptionDialogCursor(GPtr globals, const POINTS& pos)
{
    if (globals->m_DisplaysPreview &&
        (s_Preview.m_IsDragging        ||
         s_Preview.m_IsShowingOriginal ||
         RIsPointInRect(s_Preview.m_ItemRect, pos)))
    {
        sPSUIHooks->SetCursor(kPICursorHand);
    }
    else
    {
        sPSUIHooks->SetCursor(kPICursorArrow);
    }
}

//-----------------------------------------------------------------------------
//! @brief プレビューの拡大率を変更します。
//!
//! @param[in] globals グローバルデータです。
//! @param[in] hDlg ダイアログハンドルです。
//! @param[in] zoomStep 拡大率の増分です。縮小なら負の値を指定します。
//-----------------------------------------------------------------------------
static void ChangePreviewZoom(GPtr globals, HWND hDlg, const int zoomStep)
{
    const int newZoom = s_Preview.m_Zoom + zoomStep;
    if (s_Preview.m_MinZoom <= newZoom && newZoom <= s_Preview.m_MaxZoom)
    {
        s_Preview.ChangeZoom(hDlg, zoomStep);
    }
    R_UNUSED_VARIABLE(globals);
}

//-----------------------------------------------------------------------------
//! @brief オプションダイアログのダイアログプロシージャです。
//!
//! @param[in] hDlg ダイアログハンドルです。
//! @param[in] uMsg メッセージです。
//! @param[in] wParam メッセージの付加情報です。
//! @param[in] lParam メッセージの付加情報です。
//!
//! @return 処理した場合は TRUE を返します。
//-----------------------------------------------------------------------------
static BOOL WINAPI OptionDialogProc(HWND hDlg, UINT uMsg, WPARAM wParam, LPARAM lParam)
{
    static GPtr globals = NULL; // ダイアログ初期化時に保持するので static にする必要があります。
    static RParameters paramsBak; // キャンセル時にパラメータを元に戻すために保持します。
    static POINTS dragStartPos; // ドラッグ開始時の座標です。
    static int dragStartIx; // ドラッグ開始時のプレビューの水平方向のスクロール値です。
    static int dragStartIy; // ドラッグ開始時のプレビューの垂直方向のスクロール値です。
    const int UPDATE_TIMER_ID = 1000; // プレビュー更新用タイマーの ID です。

    char strBuf[MAX_PATH] = { 0 }; // エディットテキストの値を取得するためのバッファです。

    switch (uMsg)
    {
        case  WM_INITDIALOG:
            globals = reinterpret_cast<GPtr>(lParam);
            paramsBak = *gParams;
            InitOptionDialog(globals, hDlg);
            if (gResult != noErr)
            {
                return FALSE;
            }
            SetTimer(hDlg, UPDATE_TIMER_ID, 50, NULL); // 50 [msec] ごとに WM_TIMER を発生させます。
            //break; // break せずに WM_PAINT の処理もします。

        case WM_PAINT:
            if (globals->m_DisplaysPreview)
            {
                //RNoteTrace("paint");
                s_Preview.Paint(hDlg);
            }
            return FALSE;

        case WM_DESTROY:
            KillTimer(hDlg, UPDATE_TIMER_ID);
            s_Preview.FreeMemory();
            break;

        case WM_TIMER:
            if (wParam == UPDATE_TIMER_ID)
            {
                if (s_Preview.m_UpdateTimerCount > 0)
                {
                    //RNoteTrace("timer: %d", s_Preview.m_UpdateTimerCount);
                    --s_Preview.m_UpdateTimerCount;
                    if (s_Preview.m_UpdateTimerCount == 0)
                    {
                        //RNoteTrace("update by timer");
                        UpdatePreviewBitmap(globals, hDlg);
                    }
                }
            }
            break;

        case WM_COMMAND:
        {
            const int id     = LOWORD(wParam);
            const int notify = HIWORD(wParam);
            switch (id)
            {
                case IDOK:
                    EndDialog(hDlg, id);
                    break;

                case IDCANCEL:
                    *gParams = paramsBak;
                    EndDialog(hDlg, id);
                    break;

                case IDC_MAX_DISTANCE: // SetDlgItemInt で設定された際も呼ばれます。
                    if (notify == EN_CHANGE || notify == EN_KILLFOCUS)
                    {
                        GetWindowText(GetDlgItem(hDlg, id), strBuf, sizeof(strBuf));
                        if (strBuf[0] != 0)
                        {
                            gParams->m_MaxDistance = atoi(strBuf);
                            //RNoteTrace("max distance: %d\n", gParams->m_MaxDistance);
                            const int clamped = RClampValue(RParameters::DISTANCE_MIN, RParameters::DISTANCE_MAX, gParams->m_MaxDistance);
                            if (gParams->m_MaxDistance != clamped)
                            {
                                gParams->m_MaxDistance = clamped;
                                SetDlgItemInt(hDlg, IDC_MAX_DISTANCE, gParams->m_MaxDistance, FALSE);
                            }
                            SetSliderPosAndRange(GetDlgItem(hDlg, IDC_MAX_DISTANCE_SLIDER), gParams->m_MaxDistance);
                            if (s_Preview.m_UpdateTimerCount == 0)
                            {
                                UpdatePreviewBitmap(globals, hDlg);
                            }
                        }
                    }
                    break;

                case IDC_EDGE_WRAP:
                    gParams->m_EdgeWrap = (IsDlgButtonChecked(hDlg, id) == BST_CHECKED);
                    UpdatePreviewBitmap(globals, hDlg);
                    break;

                case IDC_NORMALIZE:
                    gParams->m_Normalize = (IsDlgButtonChecked(hDlg, id) == BST_CHECKED);
                    UpdatePreviewBitmap(globals, hDlg);
                    break;

                case IDC_ZOOM_IN:
                    ChangePreviewZoom(globals, hDlg, 1);
                    break;

                case IDC_ZOOM_OUT:
                    ChangePreviewZoom(globals, hDlg, -1);
                    break;

                case IDM_LOCAL_HELP:
                    RShowPluginHelp(NPS_PLUGIN_HELP_URL);
                    break;

                case IDM_HELP_INDEX:
                    RShowPluginHelp(NPS_HELP_INDEX_URL);
                    break;
            }
            break;
        }

        case WM_HSCROLL:
        {
            const int code = LOWORD(wParam);
            //const int pos = HIWORD(wParam); // SB_THUMBPOSITION と SB_THUMBTRACK のみ有効
            const int id = GetWindowLong(reinterpret_cast<HWND>(lParam), GWL_ID);
            if (id == IDC_MAX_DISTANCE_SLIDER)
            {
                //RNoteTrace("hscroll: %d %d", code, HIWORD(wParam));
                if (code != SB_ENDSCROLL)
                {
                    s_Preview.m_UpdateTimerCount = (code == SB_THUMBTRACK) ?
                        RPreview::TIMER_DRAG : 0;
                    int editId;
                    switch (id)
                    {
                        default:
                        case IDC_MAX_DISTANCE_SLIDER : editId = IDC_MAX_DISTANCE; break;
                    }
                    const int posVal = static_cast<int>(SendDlgItemMessage(hDlg, id, TBM_GETPOS, 0, 0));
                    SetDlgItemInt(hDlg, editId, posVal, FALSE);
                }
            }
            break;
        }

        case WM_LBUTTONDOWN:
        {
            POINTS pos = MAKEPOINTS(lParam);
            if (globals->m_DisplaysPreview &&
                RIsPointInRect(s_Preview.m_ItemRect, pos))
            {
                // フォーカスをプレビューに移してホイールで拡大縮小ができるようにします。
                SetFocus(GetDlgItem(hDlg, IDC_PREVIEW_AREA));
                s_Preview.m_IsDragging = s_Preview.IsScrollEnable();
                s_Preview.m_IsShowingOriginal = true;
                s_Preview.UpdateWinBmp(hDlg, globals->m_pSrcBitmapData);

                SetOptionDialogCursor(globals, pos);
                if (s_Preview.m_IsDragging)
                {
                    // start drag
                    dragStartPos = pos;
                    dragStartIx = s_Preview.m_Ix;
                    dragStartIy = s_Preview.m_Iy;
                    SetCapture(hDlg);
                    return FALSE;
                }
            }
            break;
        }

        case WM_MOUSEMOVE:
        {
            POINTS pos = MAKEPOINTS(lParam);
            if (s_Preview.m_IsDragging)
            {
                s_Preview.ChangeScroll(hDlg,
                    dragStartIx - (pos.x - dragStartPos.x),
                    dragStartIy - (pos.y - dragStartPos.y));
            }
            SetOptionDialogCursor(globals, pos);
            break;
        }

        case WM_LBUTTONUP:
        {
            POINTS pos = MAKEPOINTS(lParam);
            if (s_Preview.m_IsDragging)
            {
                s_Preview.m_IsDragging = false;
                ReleaseCapture();
            }
            if (s_Preview.m_IsShowingOriginal)
            {
                s_Preview.m_IsShowingOriginal = false;
                s_Preview.UpdateWinBmp(hDlg, globals->m_pDstBitmapData);
            }
            SetOptionDialogCursor(globals, pos);
            break;
        }

        case WM_MOUSEWHEEL:
        {
            POINTS pos = MAKEPOINTS(lParam); // スクリーン座標
            POINT pnt; // クライアント座標に変換するために POINT (LONG x 2) を使用します。
            pnt.x = pos.x;
            pnt.y = pos.y;
            ScreenToClient(hDlg, &pnt);
            pos.x = static_cast<short>(pnt.x);
            pos.y = static_cast<short>(pnt.y);
            const short delta = static_cast<short>(HIWORD(wParam));
            if (globals->m_DisplaysPreview)
            {
                if (delta > 0)
                {
                    ChangePreviewZoom(globals, hDlg, 1);
                }
                else if (delta < 0)
                {
                    ChangePreviewZoom(globals, hDlg, -1);
                }
            }
            SetOptionDialogCursor(globals, pos);
            //RNoteTrace("wheel: %d %d\n", pos.x, pos.y);
            break;
        }

        default:
            return FALSE;
    }
    return TRUE;
}

//-----------------------------------------------------------------------------
//! @brief オプションダイアログを表示します。
//!
//! @param[in,out] globals グローバルデータです。
//!
//! @return OK ボタンがクリックされたら true、
//!         Cancel ボタンがクリックされたら false を返します。
//-----------------------------------------------------------------------------
bool DoOptionDialog(GPtr globals)
{
    HINSTANCE instance = GetDLLInstance();
    //RNoteTrace("dll path: %s", RGetModuleFileName(instance).c_str());
    PlatformData* platform = reinterpret_cast<PlatformData*>(gStuff->platformData);
    INT_PTR result = DialogBoxParam(instance,
        MAKEINTRESOURCE(IDD_OPTION), reinterpret_cast<HWND>(platform->hwnd),
        reinterpret_cast<DLGPROC>(OptionDialogProc), reinterpret_cast<LPARAM>(globals));
    return (result == IDOK);
}

//-----------------------------------------------------------------------------
//! @brief アバウトダイアログを表示します。
//!
//! @param[in] about アバウトレコードです。
//-----------------------------------------------------------------------------
void DoAboutDialog(AboutRecordPtr about)
{
    HINSTANCE instance = GetDLLInstance();
    PlatformData* platform = reinterpret_cast<PlatformData*>(about->platformData);
    RAboutParam aboutParam =
    {
        NPS_ABOUT_TITLE, NPS_ABOUT_MESSAGE,
        IDC_ABOUT_TEXT, IDC_HELP_INDEX, NPS_HELP_INDEX_URL
    };
    DialogBoxParam(instance,
        MAKEINTRESOURCE(IDD_ABOUT), reinterpret_cast<HWND>(platform->hwnd),
        reinterpret_cast<DLGPROC>(RAboutDialogProc), reinterpret_cast<LPARAM>(&aboutParam));
}

