﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

// PluginMain InitGlobals ValidateParameters InitParameters
// DoParameters DoPrepare DoStart DoContinue DoFinish
// DoFilter
// ReadBitmap WriteBitmap
// CalculateDistanceRoundRobin
// CalculateDistanceChamfer
// CalculateDistanceDeadReckoning

//=============================================================================
// include
//=============================================================================
#include "NintendoDistanceField.h"
#include "NintendoDistanceFieldScripting.h"
#include "NintendoDistanceFieldUI.h"

using namespace nn::gfx::tool::nps;

//-----------------------------------------------------------------------------
//! @brief パラメータを初期化します。
//!        フィルタプラグインでは Photoshop 起動時に一度だけ呼ばれます。
//!        パラメータ構造体を変更した際は Photoshop を起動しなおす必要があります。
//!
//! @param[in,out] globals グローバルデータです。
//-----------------------------------------------------------------------------
static void InitParameters(GPtr globals)
{
    gParams->m_MaxDistance = 10;
    gParams->m_EdgeWrap = false;
    gParams->m_Normalize = true;
}

//-----------------------------------------------------------------------------
//! @brief パラメータを有効にします。
//!        確保されていなければ確保して初期化します。
//!
//! @param[in,out] globals グローバルデータです。
//-----------------------------------------------------------------------------
static void ValidateParameters(GPtr globals)
{
    if (gStuff->parameters == NULL)
    {
        gStuff->parameters = PINewHandle(sizeof(RParameters));
        if (gStuff->parameters == NULL)
        {
            gResult = memFullErr;
            return;
        }
        InitParameters(globals);
    }
}

//-----------------------------------------------------------------------------
//! @brief グローバルデータを初期化します。
//!        プラグインが最初に使用されたときに 1 度だけ呼ばれます。
//!
//! @param[in,out] globalPtr グローバルデータのポインタです。
//-----------------------------------------------------------------------------
static void InitGlobals(Ptr globalPtr)
{
    //RMsgBox("init globals");

    //-----------------------------------------------------------------------------
    // copy for use macro
    //GPtr globals = (GPtr)globalPtr;

    //-----------------------------------------------------------------------------
    // call constructor (name must be "globals" for use macro (g????))
    GPtr globals = new (globalPtr) Globals;

    //-----------------------------------------------------------------------------
    // validate parameters
    ValidateParameters(globals);

    //-----------------------------------------------------------------------------
    // init element
    gQueryForParameters = FALSE; // DoParameters が呼ばれるまでは FALSE

    globals->m_pSrcBitmapData = NULL;
    globals->m_pDstBitmapData = NULL;
    globals->m_pBinaryData    = NULL;
    globals->m_pDistanceData  = NULL;

    globals->m_DisplaysPreview = true;
}

//-----------------------------------------------------------------------------
//! @brief パラメータの処理をします。
//!        フィルタの再実行を行った場合は呼ばれません。
//!        アクションから実行する際にダイアログアイコンが OFF なら呼ばれません。
//!
//! @param[in,out] globals グローバルデータです。
//-----------------------------------------------------------------------------
static void DoParameters(GPtr globals)
{
    //RMsgBox("do parameters");

    //-----------------------------------------------------------------------------
    // validate parameters
    ValidateParameters(globals);

    //-----------------------------------------------------------------------------
    // set query for parameters
    gQueryForParameters = true; // show option dialog
}

//-----------------------------------------------------------------------------
//! @brief フィルタ処理の準備をします。
//!
//! @param[in,out] globals グローバルデータです。
//-----------------------------------------------------------------------------
static void DoPrepare(GPtr globals)
{
    //RMsgBox("do prepare: %d %d", gStuff->bufferSpace, gStuff->maxSpace);
    // デフォルトで bufferSpace は 0、maxSpace は環境設定の最大使用メモリの値

    //-----------------------------------------------------------------------------
    // validate parameters
    ValidateParameters(globals);
}

//-----------------------------------------------------------------------------
//! @brief ビットマップを Photoshop からリードします。
//!
//! @param[in,out] globals グローバルデータです。
//-----------------------------------------------------------------------------
static void ReadBitmap(GPtr globals)
{
    //-----------------------------------------------------------------------------
    // alloc bitmap
    const int pixelCount = gBigDoc->imageSize32.h * gBigDoc->imageSize32.v;
    globals->m_pSrcBitmapData = new uint8_t[pixelCount * R_RGBA_COUNT];
    globals->m_pDstBitmapData = new uint8_t[pixelCount * R_RGBA_COUNT];

    //-----------------------------------------------------------------------------
    // alloc channel buffer
    uint8_t* pChanBuf = new uint8_t[pixelCount];

    //-----------------------------------------------------------------------------
    // set read parameter
    VRect bounds;
    bounds.left   = 0;
    bounds.top    = 0;
    bounds.right  = gBigDoc->imageSize32.h;
    bounds.bottom = gBigDoc->imageSize32.v;

    PixelMemoryDesc memDesc;
    memDesc.data      = pChanBuf;
    memDesc.rowBits   = gBigDoc->imageSize32.h * globals->m_ColorChanTop->depth;
    memDesc.colBits   = globals->m_ColorChanTop->depth;
    memDesc.bitOffset = 0;
    memDesc.depth     = globals->m_ColorChanTop->depth;

    //-----------------------------------------------------------------------------
    // カラーチャンネルをリードします。
    int iRgb = 0;
    const ReadChannelDesc* chan = globals->m_ColorChanTop;
    while (chan != NULL && iRgb < 3)
    {
        if (RGetChannelWriteFlag(chan))
        {
            if (sPSChannelProcs->ReadPixelsFromLevel(chan->port, 0, &bounds, &memDesc))
            {
                gResult = errPlugInHostInsufficient;
                goto CleanUp;
            }
            RSetOneChannelToPixelsPadding(
                globals->m_pSrcBitmapData + iRgb, pChanBuf,
                gBigDoc->imageSize32.h, gBigDoc->imageSize32.v,
                gBigDoc->imageSize32.h, gBigDoc->imageSize32.v);
            ++iRgb;
        }
        chan = chan->next;
    }

    //-----------------------------------------------------------------------------
    // アルファチャンネルのみ選択されている場合は、アルファチャンネルを R 成分としてリードします。
    if (iRgb == 0 && globals->m_SelAlphaChanCount != 0)
    {
        chan = globals->m_AlphaChanTop;
        while (chan != NULL)
        {
            if (RGetChannelWriteFlag(chan))
            {
                if (sPSChannelProcs->ReadPixelsFromLevel(chan->port, 0, &bounds, &memDesc))
                {
                    gResult = errPlugInHostInsufficient;
                    goto CleanUp;
                }
                RSetOneChannelToPixelsPadding(
                    globals->m_pSrcBitmapData + 0, pChanBuf,
                    gBigDoc->imageSize32.h, gBigDoc->imageSize32.v,
                    gBigDoc->imageSize32.h, gBigDoc->imageSize32.v);
                iRgb = 1;
                break;
            }
            chan = chan->next;
        }
    }

    //-----------------------------------------------------------------------------
    // RGB の 3 成分がリードされていなければ R 成分を GB 成分にコピーします。
    if (iRgb < 3)
    {
        uint8_t* pDst = globals->m_pSrcBitmapData;
        for (int iPix = 0; iPix < pixelCount; ++iPix)
        {
            *(pDst + 1) = *(pDst + 2) = *pDst;
            pDst += R_RGBA_COUNT;
        }
    }

    //-----------------------------------------------------------------------------
    // A 成分を 0xff で埋めます。
    uint8_t* pDst = globals->m_pSrcBitmapData + 3;
    for (int iPix = 0; iPix < pixelCount; ++iPix)
    {
        *pDst = 0xff;
        pDst += R_RGBA_COUNT;
    }

    //-----------------------------------------------------------------------------
    // free channel buffer
CleanUp:
    delete[] pChanBuf;
}

//-----------------------------------------------------------------------------
//! @brief ビットマップを Photoshop にライトします。
//!
//! @param[in,out] globals グローバルデータです。
//-----------------------------------------------------------------------------
static void WriteBitmap(GPtr globals)
{
    //-----------------------------------------------------------------------------
    // alloc channel buffer
    const int pixelCount = gBigDoc->imageSize32.h * gBigDoc->imageSize32.v;
    uint8_t* pChanBuf = new uint8_t[pixelCount];

    //-----------------------------------------------------------------------------
    // set write parameter
    VRect bounds;
    bounds.left   = 0;
    bounds.top    = 0;
    bounds.right  = gBigDoc->imageSize32.h;
    bounds.bottom = gBigDoc->imageSize32.v;

    PixelMemoryDesc memDesc;
    memDesc.data      = pChanBuf;
    memDesc.rowBits   = gBigDoc->imageSize32.h * globals->m_ColorChanTop->depth;
    memDesc.colBits   = globals->m_ColorChanTop->depth;
    memDesc.bitOffset = 0;
    memDesc.depth     = globals->m_ColorChanTop->depth;

    //-----------------------------------------------------------------------------
    // カラーチャンネルをライトします。
    int iRgb = 0;
    const ReadChannelDesc* chan = globals->m_ColorChanTop;
    while (chan != NULL && iRgb < 3)
    {
        if (RGetChannelWriteFlag(chan))
        {
            RGetOneChannelFromPixels(
                pChanBuf, globals->m_pDstBitmapData + iRgb,
                gBigDoc->imageSize32.h, gBigDoc->imageSize32.v, gBigDoc->imageSize32.h);
            if (sPSChannelProcs->WritePixelsToBaseLevel(chan->writePort, &bounds, &memDesc))
            {
                gResult = errPlugInHostInsufficient;
                goto CleanUp;
            }
            ++iRgb;
        }
        chan = chan->next;
    }

    //-----------------------------------------------------------------------------
    // アルファチャンネルのみ選択されている場合は、R 成分をアルファチャンネルとしてライトします。
    if (iRgb == 0 && globals->m_SelAlphaChanCount != 0)
    {
        chan = globals->m_AlphaChanTop;
        while (chan != NULL)
        {
            if (RGetChannelWriteFlag(chan))
            {
                RGetOneChannelFromPixels(
                    pChanBuf, globals->m_pDstBitmapData + 0,
                    gBigDoc->imageSize32.h, gBigDoc->imageSize32.v, gBigDoc->imageSize32.h);
                if (sPSChannelProcs->WritePixelsToBaseLevel(chan->writePort, &bounds, &memDesc))
                {
                    gResult = errPlugInHostInsufficient;
                    goto CleanUp;
                }
                break;
            }
            chan = chan->next;
        }
    }

    //-----------------------------------------------------------------------------
    // free channel buffer
CleanUp:
    delete[] pChanBuf;
}

//-----------------------------------------------------------------------------
//! @brief ラップした座標を計算します。
//!
//! @param[in] pos 座標です。
//! @param[in] length 長さです。
//!
//! @return ラップした座標を返します。
//-----------------------------------------------------------------------------
static inline int CalcWrapPos(int pos, int length)
{
    while (pos < 0)
    {
        pos += length;
    }
    while (pos >= length)
    {
        pos -= length;
    }
    return pos;
}

//-----------------------------------------------------------------------------
//! @brief ラップを考慮してビットマップの値を取得します。
//!
//! @param[in] pBitmap ビットマップを格納したバッファです。
//! @param[in] curW ビットマップの幅です。
//! @param[in] curH ビットマップの高さです。
//! @param[in] x 対象ピクセルの X 座標です。
//! @param[in] y 対象ピクセルの Y 座標です。
//! @param[in] edgeWrap 画像が繰り返しているとして端のピクセルを処理するなら true です。
//!
//! @return ビットマップの値を取得します。
//-----------------------------------------------------------------------------
template <typename T>
static inline T GetBitmapValue(
    const T* pBitmap,
    const int curW,
    const int curH,
    const int x,
    const int y,
    const bool edgeWrap
)
{
    int sx;
    int sy;
    if (!edgeWrap)
    {
        sx = RClampValue(0, curW - 1, x);
        sy = RClampValue(0, curH - 1, y);
    }
    else
    {
        sx = CalcWrapPos(x, curW);
        sy = CalcWrapPos(y, curH);
    }
    return pBitmap[sx + sy * curW];
}

//-----------------------------------------------------------------------------
//! @brief Chamfer 法で白黒境界までの距離を計算します。
//!
//! @param[out] pDistanceData 白黒境界までの距離を格納するバッファです。
//! @param[in] pBinaryData 2 値化した値を格納したバッファです。
//! @param[in] curW 処理するビットマップの幅です。
//! @param[in] curH 処理するビットマップの高さです。
//! @param[in] distanceSub 距離から引く値です。
//! @param[in] edgeWrap 画像が繰り返しているとして端のピクセルを処理するなら true です。
//!
//! @return 処理成功なら true、中止されたら false を返します。
//-----------------------------------------------------------------------------
#if 0
static bool CalculateDistanceChamfer(
    float* pDistanceData,
    const uint8_t* pBinaryData,
    const int curW,
    const int curH,
    const float distanceSub,
    const bool edgeWrap
)
{
    float* pDistanceData = ;
    const bool edgeWrap = gParams->m_EdgeWrap;

    //-----------------------------------------------------------------------------
    // 距離を大きな値で初期化します。
    float* pDistance = pDistanceData;
    for (int iy = 0; iy < curH; ++iy)
    {
        for (int ix = 0; ix < curW; ++ix)
        {
            *pDistance++ = 1.0e10f;
        }
    }

    //-----------------------------------------------------------------------------
    // 上下左右で異なる色のピクセルに隣接するなら距離を 1 にします。
    const uint8_t* pBin = pBinaryData;
    pDistance = pDistanceData;
    for (int iy = 0; iy < curH; ++iy)
    {
        for (int ix = 0; ix < curW; ++ix)
        {
            const uint8_t bin = *pBin++;
            if (GetBitmapValue(pBinaryData, curW, curH, ix - 1, iy    , edgeWrap) != bin ||
                GetBitmapValue(pBinaryData, curW, curH, ix + 1, iy    , edgeWrap) != bin ||
                GetBitmapValue(pBinaryData, curW, curH, ix    , iy - 1, edgeWrap) != bin ||
                GetBitmapValue(pBinaryData, curW, curH, ix    , iy + 1, edgeWrap) != bin)
            {
                *pDistance = 1.0f - distanceSub;
            }
            ++pDistance;
        }
    }

    //-----------------------------------------------------------------------------
    // 前方へのパスを実行します。
    // 元の論文では d1 = 3, d2 = 4 ですが、
    // 結果が総当たりに近いのは d1 = 1, d2 = √2 です。
    //const float d1 = 3.0f;
    //const float d2 = 4.0f;
    const float d1 = 1.0f;
    const float d2 = sqrtf(2.0f);

    pDistance = pDistanceData;
    for (int iy = 0; iy < curH; ++iy)
    {
        for (int ix = 0; ix < curW; ++ix)
        {
            const float val0 = GetBitmapValue(pDistanceData, curW, curH, ix - 1, iy - 1, edgeWrap);
            const float val1 = GetBitmapValue(pDistanceData, curW, curH, ix    , iy - 1, edgeWrap);
            const float val2 = GetBitmapValue(pDistanceData, curW, curH, ix + 1, iy - 1, edgeWrap);
            const float val3 = GetBitmapValue(pDistanceData, curW, curH, ix - 1, iy    , edgeWrap);
            if (val0 + d2 < *pDistance)
            {
                *pDistance = val0 + d2;
            }
            if (val1 + d1 < *pDistance)
            {
                *pDistance = val1 + d1;
            }
            if (val2 + d2 < *pDistance)
            {
                *pDistance = val2 + d2;
            }
            if (val3 + d1 < *pDistance)
            {
                *pDistance = val3 + d1;
            }
            ++pDistance;
        }
    }

    //-----------------------------------------------------------------------------
    // 後方へのパスを実行します。
    pDistance = pDistanceData + curW * curH - 1;
    for (int iy = curH - 1; iy >= 0; --iy)
    {
        for (int ix = curW - 1; ix >= 0; --ix)
        {
            const float val0 = GetBitmapValue(pDistanceData, curW, curH, ix + 1, iy    , edgeWrap);
            const float val1 = GetBitmapValue(pDistanceData, curW, curH, ix - 1, iy + 1, edgeWrap);
            const float val2 = GetBitmapValue(pDistanceData, curW, curH, ix    , iy + 1, edgeWrap);
            const float val3 = GetBitmapValue(pDistanceData, curW, curH, ix + 1, iy + 1, edgeWrap);
            if (val0 + d1 < *pDistance)
            {
                *pDistance = val0 + d1;
            }
            if (val1 + d2 < *pDistance)
            {
                *pDistance = val1 + d2;
            }
            if (val2 + d1 < *pDistance)
            {
                *pDistance = val2 + d1;
            }
            if (val3 + d2 < *pDistance)
            {
                *pDistance = val3 + d2;
            }
            --pDistance;
        }
    }

    //-----------------------------------------------------------------------------
    // 符号を設定します。
    pBin = pBinaryData;
    pDistance = pDistanceData;
    for (int iy = 0; iy < curH; ++iy)
    {
        for (int ix = 0; ix < curW; ++ix)
        {
            const float distance = *pDistance;
            *pDistance++ = (*pBin++ != 0) ? distance : -distance;
        }
    }
    return true;
}
#endif

//-----------------------------------------------------------------------------
//! @brief Dead Reckoning 法で白黒境界までの距離を計算します。
//!        Chamfer 法より曲線がきれいに表示できます。
//!
//! @param[out] pDistanceData 白黒境界までの距離を格納するバッファです。
//! @param[in] pBinaryData 2 値化した値を格納したバッファです。
//! @param[in] curW 処理するビットマップの幅です。
//! @param[in] curH 処理するビットマップの高さです。
//! @param[in] distanceSub 距離から引く値です。
//! @param[in] edgeWrap 画像が繰り返しているとして端のピクセルを処理するなら true です。
//!
//! @return 処理成功なら true、中止されたら false を返します。
//-----------------------------------------------------------------------------
static bool CalculateDistanceDeadReckoning(
    float* pDistanceData,
    const uint8_t* pBinaryData,
    const int curW,
    const int curH,
    const float distanceSub,
    const bool edgeWrap
)
{
    //-----------------------------------------------------------------------------
    // 各ピクセルに対する白黒境界上の点を格納するメモリを確保します。
    int* pBorderXs = new int[curW * curH];
    int* pBorderYs = new int[curW * curH];

    //-----------------------------------------------------------------------------
    // 距離を大きな値で初期化します。
    float* pDistance = pDistanceData;
    int* pX = pBorderXs;
    int* pY = pBorderYs;
    for (int iy = 0; iy < curH; ++iy)
    {
        for (int ix = 0; ix < curW; ++ix)
        {
            *pDistance++ = 1.0e10f;
            *pX++ = -1;
            *pY++ = -1;
        }
    }

    //-----------------------------------------------------------------------------
    // 上下左右で異なる色のピクセルに隣接するなら距離を 0 にします。
    const uint8_t* pBin = pBinaryData;
    pDistance = pDistanceData;
    pX = pBorderXs;
    pY = pBorderYs;
    for (int iy = 0; iy < curH; ++iy)
    {
        for (int ix = 0; ix < curW; ++ix)
        {
            const uint8_t bin = *pBin++;
            if (GetBitmapValue(pBinaryData, curW, curH, ix - 1, iy    , edgeWrap) != bin ||
                GetBitmapValue(pBinaryData, curW, curH, ix + 1, iy    , edgeWrap) != bin ||
                GetBitmapValue(pBinaryData, curW, curH, ix    , iy - 1, edgeWrap) != bin ||
                GetBitmapValue(pBinaryData, curW, curH, ix    , iy + 1, edgeWrap) != bin)
            {
                *pDistance = 0.0f;
                *pX = ix;
                *pY = iy;
            }
            ++pDistance;
            ++pX;
            ++pY;
        }
    }

    //-----------------------------------------------------------------------------
    // 最初のパスを実行します。
    const float d1 = 1.0f;
    const float d2 = sqrtf(2.0f);

    pDistance = pDistanceData;
    pX = pBorderXs;
    pY = pBorderYs;
    for (int iy = 0; iy < curH; ++iy)
    {
        for (int ix = 0; ix < curW; ++ix)
        {
            const float val0 = GetBitmapValue(pDistanceData, curW, curH, ix - 1, iy - 1, edgeWrap);
            const float val1 = GetBitmapValue(pDistanceData, curW, curH, ix    , iy - 1, edgeWrap);
            const float val2 = GetBitmapValue(pDistanceData, curW, curH, ix + 1, iy - 1, edgeWrap);
            const float val3 = GetBitmapValue(pDistanceData, curW, curH, ix - 1, iy    , edgeWrap);
            if (val0 + d2 < *pDistance)
            {
                *pX = GetBitmapValue(pBorderXs, curW, curH, ix - 1, iy - 1, edgeWrap);
                *pY = GetBitmapValue(pBorderYs, curW, curH, ix - 1, iy - 1, edgeWrap);
                const float dx = static_cast<float>(ix - *pX);
                const float dy = static_cast<float>(iy - *pY);
                *pDistance = sqrtf(dx * dx + dy * dy);
            }
            if (val1 + d1 < *pDistance)
            {
                *pX = GetBitmapValue(pBorderXs, curW, curH, ix    , iy - 1, edgeWrap);
                *pY = GetBitmapValue(pBorderYs, curW, curH, ix    , iy - 1, edgeWrap);
                const float dx = static_cast<float>(ix - *pX);
                const float dy = static_cast<float>(iy - *pY);
                *pDistance = sqrtf(dx * dx + dy * dy);
            }
            if (val2 + d2 < *pDistance)
            {
                *pX = GetBitmapValue(pBorderXs, curW, curH, ix + 1, iy - 1, edgeWrap);
                *pY = GetBitmapValue(pBorderYs, curW, curH, ix + 1, iy - 1, edgeWrap);
                const float dx = static_cast<float>(ix - *pX);
                const float dy = static_cast<float>(iy - *pY);
                *pDistance = sqrtf(dx * dx + dy * dy);
            }
            if (val3 + d1 < *pDistance)
            {
                *pX = GetBitmapValue(pBorderXs, curW, curH, ix - 1, iy    , edgeWrap);
                *pY = GetBitmapValue(pBorderYs, curW, curH, ix - 1, iy    , edgeWrap);
                const float dx = static_cast<float>(ix - *pX);
                const float dy = static_cast<float>(iy - *pY);
                *pDistance = sqrtf(dx * dx + dy * dy);
            }
            ++pDistance;
            ++pX;
            ++pY;
        }
    }

    //-----------------------------------------------------------------------------
    // 最後のパスを実行します。
    pDistance = pDistanceData + curW * curH - 1;
    pX = pBorderXs + curW * curH - 1;
    pY = pBorderYs + curW * curH - 1;
    for (int iy = curH - 1; iy >= 0; --iy)
    {
        for (int ix = curW - 1; ix >= 0; --ix)
        {
            const float val0 = GetBitmapValue(pDistanceData, curW, curH, ix + 1, iy    , edgeWrap);
            const float val1 = GetBitmapValue(pDistanceData, curW, curH, ix - 1, iy + 1, edgeWrap);
            const float val2 = GetBitmapValue(pDistanceData, curW, curH, ix    , iy + 1, edgeWrap);
            const float val3 = GetBitmapValue(pDistanceData, curW, curH, ix + 1, iy + 1, edgeWrap);
            if (val0 + d1 < *pDistance)
            {
                *pX = GetBitmapValue(pBorderXs, curW, curH, ix + 1, iy    , edgeWrap);
                *pY = GetBitmapValue(pBorderYs, curW, curH, ix + 1, iy    , edgeWrap);
                const float dx = static_cast<float>(ix - *pX);
                const float dy = static_cast<float>(iy - *pY);
                *pDistance = sqrtf(dx * dx + dy * dy);
            }
            if (val1 + d2 < *pDistance)
            {
                *pX = GetBitmapValue(pBorderXs, curW, curH, ix - 1, iy + 1, edgeWrap);
                *pY = GetBitmapValue(pBorderYs, curW, curH, ix - 1, iy + 1, edgeWrap);
                const float dx = static_cast<float>(ix - *pX);
                const float dy = static_cast<float>(iy - *pY);
                *pDistance = sqrtf(dx * dx + dy * dy);
            }
            if (val2 + d1 < *pDistance)
            {
                *pX = GetBitmapValue(pBorderXs, curW, curH, ix    , iy + 1, edgeWrap);
                *pY = GetBitmapValue(pBorderYs, curW, curH, ix    , iy + 1, edgeWrap);
                const float dx = static_cast<float>(ix - *pX);
                const float dy = static_cast<float>(iy - *pY);
                *pDistance = sqrtf(dx * dx + dy * dy);
            }
            if (val3 + d2 < *pDistance)
            {
                *pX = GetBitmapValue(pBorderXs, curW, curH, ix + 1, iy + 1, edgeWrap);
                *pY = GetBitmapValue(pBorderYs, curW, curH, ix + 1, iy + 1, edgeWrap);
                const float dx = static_cast<float>(ix - *pX);
                const float dy = static_cast<float>(iy - *pY);
                *pDistance = sqrtf(dx * dx + dy * dy);
            }
            --pDistance;
            --pX;
            --pY;
        }
    }

    //-----------------------------------------------------------------------------
    // 符号を設定します。
    pBin = pBinaryData;
    pDistance = pDistanceData;
    for (int iy = 0; iy < curH; ++iy)
    {
        for (int ix = 0; ix < curW; ++ix)
        {
            const float distance = *pDistance + 1.0f - distanceSub;
            *pDistance++ = (*pBin++ != 0) ? distance : -distance;
        }
    }

    //-----------------------------------------------------------------------------
    // メモリを解放します。
    delete[] pBorderXs;
    delete[] pBorderYs;

    return true;
}

//-----------------------------------------------------------------------------
//! @brief 対象ピクセルから白黒境界までの距離を取得します。
//!
//! @param[in] pBinaryData 2 値化したデータを格納したバッファです。
//! @param[in] curW 処理するビットマップの幅です。
//! @param[in] curH 処理するビットマップの高さです。
//! @param[in] pointX 対象ピクセルの X 座標です。
//! @param[in] pointY 対象ピクセルの Y 座標です。
//! @param[in] maxDistance 白黒境界からの最大距離（ピクセル）です。
//! @param[in] distanceSub 距離から引く値です。
//! @param[in] edgeWrap 画像が繰り返しているとして端のピクセルを処理するなら true です。
//!
//! @return 白黒境界までの距離を返します。対象ピクセルが黒なら「-距離」を返します。
//!         maxDistance までに白黒境界がない場合の距離は maxDistance とします。
//-----------------------------------------------------------------------------
static float GetDistance(
    const uint8_t* pBinaryData,
    const int curW,
    const int curH,
    const int pointX,
    const int pointY,
    const int maxDistance,
    const float distanceSub,
    const bool edgeWrap
)
{
    int startX = pointX - maxDistance;
    int startY = pointY - maxDistance;
    int endX   = pointX + maxDistance;
    int endY   = pointY + maxDistance;
    if (!edgeWrap)
    {
        startX = (startX < 0) ? 0 : startX;
        startY = (startY < 0) ? 0 : startY;
        endX   = (endX >= curW) ? curW - 1 : endX;
        endY   = (endY >= curH) ? curH - 1 : endY;
    }

    float distanceSqMin = static_cast<float>(maxDistance);
    distanceSqMin *= distanceSqMin;

    const uint8_t baseBin = pBinaryData[pointX + pointY * curW];
    for (int iy = startY; iy <= endY; ++iy)
    {
        const int sy = (edgeWrap) ? CalcWrapPos(iy, curH) : iy;
        for (int ix = startX; ix <= endX; ++ix)
        {
            const int sx = (edgeWrap) ? CalcWrapPos(ix, curW) : ix;
            if (pBinaryData[sx + sy * curW] != baseBin)
            {
                const float dx = static_cast<float>(ix - pointX);
                const float dy = static_cast<float>(iy - pointY);
                const float distanceSq = dx * dx + dy * dy;
                if (distanceSq < distanceSqMin)
                {
                    distanceSqMin = distanceSq;
                    if (distanceSqMin <= 1.0f)
                    {
                        iy = endY + 1;
                        break;
                    }
                }
            }
        }
    }

    const float distanceMin = sqrtf(distanceSqMin) - distanceSub;
    return (baseBin != 0) ? distanceMin : -distanceMin;
}

//-----------------------------------------------------------------------------
//! @brief 総当たりで白黒境界までの距離を計算します。
//!
//! @param[out] pDistanceData 白黒境界までの距離を格納するバッファです。
//! @param[in] pBinaryData R 成分を 2 値化した値を格納したバッファです。
//! @param[in] curW 処理するビットマップの幅です。
//! @param[in] curH 処理するビットマップの高さです。
//! @param[in] maxDistance 白黒境界からの最大距離（ピクセル）です。
//! @param[in] distanceSub 距離から引く値です。
//! @param[in] edgeWrap 画像が繰り返しているとして端のピクセルを処理するなら true です。
//! @param[in] usesRough 近似計算の結果を利用して高速化するなら true です。
//! @param[in,out] pProgress プログレスのポインタです。NULL なら使用しません。
//!
//! @return 処理成功なら true、中止されたら false を返します。
//-----------------------------------------------------------------------------
static bool CalculateDistanceRoundRobin(
    float* pDistanceData,
    const uint8_t* pBinaryData,
    const int curW,
    const int curH,
    const int maxDistance,
    const float distanceSub,
    const bool edgeWrap,
    const bool usesRough,
    RProgress* pProgress
)
{
    //-----------------------------------------------------------------------------
    // プログレスを初期化します。
    if (pProgress != NULL)
    {
        pProgress->Init(curH);
    }

    //-----------------------------------------------------------------------------
    // 近似計算で距離を計算します。
    if (usesRough)
    {
        CalculateDistanceDeadReckoning(pDistanceData,
            pBinaryData, curW, curH, distanceSub, edgeWrap);
    }

    //-----------------------------------------------------------------------------
    // 総当たりで距離を計算します。
    float* pDistance = pDistanceData;
    for (int iy = 0; iy < curH; ++iy)
    {
        if (pProgress != NULL)
        {
            if (!pProgress->Update())
            {
                return false;
            }
        }
        for (int ix = 0; ix < curW; ++ix)
        {
            int checkDistance = maxDistance;
            if (usesRough)
            {
                const float d = *pDistance; // 白黒境界のない画像の場合、絶対値が大きな値になっています。
                const float ad = RMin(abs(d), static_cast<float>(maxDistance)) + distanceSub;
                if (ad <= 1.0f)
                {
                    *pDistance++ = d;
                    continue;
                }
                const int adi = RRound(ad * 1.5f);
                if (checkDistance > adi)
                {
                    checkDistance = adi;
                }
            }
            *pDistance++ = GetDistance(pBinaryData, curW, curH, ix, iy,
                checkDistance, distanceSub, edgeWrap);
        }
    }
    return true;
}

//-----------------------------------------------------------------------------
//! @brief フィルタ処理を実行します。
//!
//! @param[in,out] globals グローバルデータです。
//! @param[out] pDstData フィルタ処理後のビットマップデータを格納するバッファです。
//! @param[in] pBinaryData R 成分を 2 値化した値を格納したバッファです。
//! @param[in] curW 処理するビットマップの幅です。
//! @param[in] curH 処理するビットマップの高さです。
//! @param[in] fullW ビットマップ全体の幅です。
//! @param[in] isRough 近似アルゴリズムで計算するなら true です。
//! @param[in,out] pProgress プログレスのポインタです。NULL なら使用しません。
//!
//! @return 処理成功なら true、中止されたら false を返します。
//-----------------------------------------------------------------------------
bool DoFilter(
    GPtr globals,
    uint8_t* pDstData,
    const uint8_t* pBinaryData,
    const int curW,
    const int curH,
    const int fullW,
    const bool isRough,
    RProgress* pProgress
)
{
    RTimeMeasure tm1;
    //RNoteTrace("do filter: %d\n", gParams->m_MaxDistance);

    //-----------------------------------------------------------------------------
    // 白黒境界までの距離を計算します。
    const float distanceSub = 0.5f;
        // ↑距離から引く値です。
        // 「黒黒白白白」のように並んでいる場合、
        // 「-2 -1 +1 +2 +3」だと -1 と +1 の間だけ間隔が大きいので、
        // 距離の絶対値から 0.5 を引いて「-1.5 -0.5 +0.5 +1.5 +2.5」とします。
    bool success = true;
    if (!isRough)
    {
        //const bool usesRough = false;
        const bool usesRough = true;
        success = CalculateDistanceRoundRobin(globals->m_pDistanceData,
            pBinaryData, curW, curH,
            gParams->m_MaxDistance, distanceSub, gParams->m_EdgeWrap, usesRough, pProgress);
    }
    else
    {
        //success = CalculateDistanceChamfer(globals->m_pDistanceData,
        //  pBinaryData, curW, curH, distanceSub, gParams->m_EdgeWrap);
        success = CalculateDistanceDeadReckoning(globals->m_pDistanceData,
            pBinaryData, curW, curH, distanceSub, gParams->m_EdgeWrap);
    }

    if (!success)
    {
        return false;
    }

    //-----------------------------------------------------------------------------
    // 距離をカラーに変換します。
    const float distanceMul = 0.5f / (gParams->m_MaxDistance - distanceSub);
    float* pDistance = globals->m_pDistanceData;
    uint8_t* pDst = pDstData;
    if (!gParams->m_Normalize) // 正規化なし
    {
        for (int iy = 0; iy < curH; ++iy)
        {
            for (int ix = 0; ix < curW; ++ix)
            {
                const float distance = *pDistance++;
                const float fv = distance * distanceMul + 0.5f;
                pDst[0] = pDst[1] = pDst[2] = static_cast<uint8_t>(
                    RClampValue(0x00, 0xff, RRound(fv * 255.0f)));
                pDst += R_RGBA_COUNT;
            }
            pDst += (fullW - curW) * R_RGBA_COUNT;
        }
    }
    else // 正規化あり
    {
        // 距離を 0 ～ 1 の値に変換して、最小・最大値を求めます。
        float fvMin = 1.0f;
        float fvMax = 0.0f;
        for (int iy = 0; iy < curH; ++iy)
        {
            for (int ix = 0; ix < curW; ++ix)
            {
                const float distance = *pDistance;
                const float fv = RClampValue(0.0f, 1.0f, distance * distanceMul + 0.5f);
                if (fv < fvMin)
                {
                    fvMin = fv;
                }
                if (fv > fvMax)
                {
                    fvMax = fv;
                }
                *pDistance++ = fv;
            }
        }

        // 正規化してカラーを設定します。
        if (fvMin == fvMax)
        {
            fvMin = 0.0f;
        }
        const float fvMul = (fvMax > fvMin) ? 1.0f / (fvMax - fvMin) : 1.0f;
        pDistance = globals->m_pDistanceData;
        for (int iy = 0; iy < curH; ++iy)
        {
            for (int ix = 0; ix < curW; ++ix)
            {
                float fv = *pDistance++;
                fv = (fv - fvMin) * fvMul;
                pDst[0] = pDst[1] = pDst[2] = static_cast<uint8_t>(
                    RClampValue(0x00, 0xff, RRound(fv * 255.0f)));
                pDst += R_RGBA_COUNT;
            }
            pDst += (fullW - curW) * R_RGBA_COUNT;
        }
    }
    //RNoteTrace("do filter: %.2f ms", tm1.GetMilliSec());
    return true;
}

//-----------------------------------------------------------------------------
//! @brief チャンネルの情報を取得します。
//!
//! @param[in,out] globals グローバルデータです。
//-----------------------------------------------------------------------------
static void GetChannelInfo(GPtr globals)
{
    // レイヤ選択時は選択されたレイヤのチャンネルが取得できます。
    globals->m_ColorChanTop = gDocInfo->targetCompositeChannels;
    globals->m_AlphaChanTop = gDocInfo->alphaChannels; // アルファチャンネルがなければ NULL
    globals->m_TransChanTop = gDocInfo->targetTransparency; // レイヤが選択されてなければ NULL
    globals->m_ColorChanCount = RGetChannelCount(globals->m_ColorChanTop);
    globals->m_AlphaChanCount = RGetChannelCount(globals->m_AlphaChanTop);
    globals->m_TransChanCount = RGetChannelCount(globals->m_TransChanTop);
    globals->m_SelColorChanCount = RGetSelectedChannelCount(globals->m_ColorChanTop);
    globals->m_SelAlphaChanCount = RGetSelectedChannelCount(globals->m_AlphaChanTop);
    globals->m_SelTransChanCount = RGetSelectedChannelCount(globals->m_TransChanTop);
    globals->m_UsesAlphaChan = (globals->m_SelAlphaChanCount > 0);
    globals->m_UsesTransChan =
        (!globals->m_UsesAlphaChan && globals->m_SelTransChanCount > 0);
    globals->m_FiltersAlpha =
        (globals->m_UsesAlphaChan || globals->m_UsesTransChan);
    //RNoteTrace("color chan: %d / %d\n", globals->m_SelColorChanCount, globals->m_ColorChanCount);
    //RNoteTrace("alpha chan: %d / %d\n", globals->m_SelAlphaChanCount, globals->m_AlphaChanCount);
    //RNoteTrace("trans chan: %d / %d\n", globals->m_SelTransChanCount, globals->m_TransChanCount);
}

//-----------------------------------------------------------------------------
//! @brief フィルタ処理を開始します。
//!
//! @param[in,out] globals グローバルデータです。
//-----------------------------------------------------------------------------
static void DoStart(GPtr globals)
{
    //RMsgBox("do start: %d %d", gStuff->inLayerPlanes, gStuff->inNonLayerPlanes);

    //-----------------------------------------------------------------------------
    // validate parameters
    ValidateParameters(globals);

    //-----------------------------------------------------------------------------
    // init buffer
    globals->m_pSrcBitmapData = NULL;
    globals->m_pDstBitmapData = NULL;
    globals->m_pBinaryData    = NULL;
    globals->m_pDistanceData  = NULL;

    //-----------------------------------------------------------------------------
    // get channel info
    GetChannelInfo(globals);

    //-----------------------------------------------------------------------------
    // check mode & channels
    const bool isGrayScale = (globals->m_ColorChanTop->channelType == ctBlack);
    if (!isGrayScale)
    {
        if (globals->m_SelColorChanCount != 0 &&
            globals->m_SelColorChanCount != 3)
        {
            RShowError(globals, "Please select RGB channels");
            gResult = userCanceledErr;
            return;
        }
    }

    if (globals->m_SelColorChanCount == 0 &&
        globals->m_SelAlphaChanCount >= 2)
    {
        RShowError(globals, "Please select only one alpha channel");
        gResult = userCanceledErr;
        return;
    }

    //-----------------------------------------------------------------------------
    // read script parameter
    gResult = ReadScriptParameters(globals);
    if (gResult != noErr)
    {
        goto CleanUp;
    }

    //-----------------------------------------------------------------------------
    // read bitmap
    ReadBitmap(globals);
    if (gResult != noErr)
    {
        goto CleanUp;
    }

    //-----------------------------------------------------------------------------
    // R 成分を 2 値化した値を計算します。
    const int SOLID_VAL = 128;
    globals->m_pBinaryData = new uint8_t[gBigDoc->imageSize32.h * gBigDoc->imageSize32.v];
    const uint8_t* pSrc = globals->m_pSrcBitmapData;
    uint8_t* pBinary = globals->m_pBinaryData;
    for (int iy = 0; iy < gBigDoc->imageSize32.v; ++iy)
    {
        for (int ix = 0; ix < gBigDoc->imageSize32.h; ++ix)
        {
            *pBinary++ = (*pSrc >= SOLID_VAL) ? 1 : 0;
            pSrc += R_RGBA_COUNT;
        }
    }

    //-----------------------------------------------------------------------------
    // 白黒境界までの距離を格納するバッファを確保します。
    globals->m_pDistanceData = new float[gBigDoc->imageSize32.h * gBigDoc->imageSize32.v];

    //-----------------------------------------------------------------------------
    // プレビュー関連のパラメータを初期化します。
    globals->m_IsPreviewFiltered = false;
    // オプションダイアログ初期化時にエディットテキストのコールバックで更新されるように
    // UI では設定できない値をプレビュー用パラメータに設定します。
    memset(&globals->m_PreviewParams, 0x00, sizeof(globals->m_PreviewParams));
    globals->m_PreviewParams.m_MaxDistance = -1;

    //-----------------------------------------------------------------------------
    // do UI
    if (gQueryForParameters)
    {
        if (!DoOptionDialog(globals))
        {
            gResult = userCanceledErr;
            goto CleanUp;
        }
        gQueryForParameters = FALSE;
    }

    //-----------------------------------------------------------------------------
    // プレビューで計算済みでなければフィルタ処理します。
    if (!globals->m_IsPreviewFiltered)
    {
        //RNoteTrace("not previewed");

        //-----------------------------------------------------------------------------
        // do filter
        const bool isRough = false;
        RProgress progress(1, gStuff->progressProc, gStuff->abortProc);
        if (!DoFilter(globals, globals->m_pDstBitmapData, globals->m_pBinaryData,
            gBigDoc->imageSize32.h, gBigDoc->imageSize32.v, gBigDoc->imageSize32.h, isRough,
            &progress))
        {
            gResult = userCanceledErr;
        }
    }

    //-----------------------------------------------------------------------------
    // write bitmap
    if (gResult == noErr)
    {
        WriteBitmap(globals);
    }

    //-----------------------------------------------------------------------------
    // end
CleanUp:
    RFreeAndClearArray(globals->m_pSrcBitmapData);
    RFreeAndClearArray(globals->m_pDstBitmapData);
    RFreeAndClearArray(globals->m_pBinaryData);
    RFreeAndClearArray(globals->m_pDistanceData);

    PISetRect(&gStuff->outRect, 0, 0, 0, 0); // no continue
    memset(&gBigDoc->outRect32, 0x00, sizeof(gBigDoc->outRect32)); // no continue
}

//-----------------------------------------------------------------------------
//! @brief フィルタ処理を継続します。
//!        gStuff->inRect、gStuff->outRect、gStuff->maskRect
//!        のいずれかひとつが空でなければ呼ばれます。
//!
//! @param[in,out] globals グローバルデータです。
//-----------------------------------------------------------------------------
static void DoContinue(GPtr globals)
{
    //RNoteTrace("do continue");
    R_UNUSED_VARIABLE(globals);
}

//-----------------------------------------------------------------------------
//! @brief フィルタ処理を終了します。
//!
//! @param[in,out] globals グローバルデータです。
//-----------------------------------------------------------------------------
static void DoFinish(GPtr globals)
{
    //RNoteTrace("do finish");

    //-----------------------------------------------------------------------------
    // write script parameter
    gResult = WriteScriptParameters(globals);
}

//-----------------------------------------------------------------------------
//! @brief プラグインのメイン関数です。
//-----------------------------------------------------------------------------
#if (PS_API_VERSION >= 1100)
DLLExport MACPASCAL void PluginMain(
    const int16 selector,
    FilterRecordPtr filterParamBlock,
    intptr_t* data,
    int16* result
)
#else
MACPASCAL void PluginMain(
    const int16 selector,
    void* filterParamBlock,
    int32* data,
    int16* result
)
#endif
{
    //-----------------------------------------------------------------------------
    // do each proc
    if (selector == filterSelectorAbout)
    {
        //-----------------------------------------------------------------------------
        // do about
        AboutRecordPtr aboutRecord = reinterpret_cast<AboutRecordPtr>(filterParamBlock);
        sSPBasic = aboutRecord->sSPBasic;
        DoAboutDialog(aboutRecord);
    }
    else
    {
        //-----------------------------------------------------------------------------
        // alloc global
        FilterRecord* filterRecord = reinterpret_cast<FilterRecord*>(filterParamBlock);
        sSPBasic = filterRecord->sSPBasic;

        Ptr globalPtr = NULL; // Pointer for global structure
        GPtr globals = NULL; // actual globals

        // alloc & init globals
        #if (PS_API_VERSION >= 1100)
        globalPtr = AllocateGlobals(result,
            filterParamBlock, filterParamBlock->handleProcs,
            sizeof(Globals), data, InitGlobals);
        #else
        globalPtr = AllocateGlobals(reinterpret_cast<uint32_t>(result),
            reinterpret_cast<uint32_t>(filterRecord), filterRecord->handleProcs,
            sizeof(Globals), data, InitGlobals);
        #endif

        if (globalPtr == NULL)
        {
            *result = memFullErr;
            return;
        }

        globals = reinterpret_cast<GPtr>(globalPtr);

        //-----------------------------------------------------------------------------
        // 32bit の座標を使用します。
        gBigDoc->PluginUsing32BitCoordinates = 1;

        //-----------------------------------------------------------------------------
        // dispatch selector
        try
        {
            switch (selector)
            {
                case filterSelectorParameters:
                    DoParameters(globals);
                    break;
                case filterSelectorPrepare:
                    DoPrepare(globals);
                    break;
                case filterSelectorStart:
                    DoStart(globals);
                    break;
                case filterSelectorContinue:
                    DoContinue(globals);
                    break;
                case filterSelectorFinish:
                    DoFinish(globals);
                    break;
            }
        }
        catch (int16 inError)
        {
            *result = inError;
        }
        catch (...)
        {
            *result = -1; // program error
        }

        //-----------------------------------------------------------------------------
        // unlock data
        if (reinterpret_cast<Handle>(*data) != NULL)
        {
            PIUnlockHandle(reinterpret_cast<Handle>(*data));
        }
    }

    //-----------------------------------------------------------------------------
    // release suites
    if (selector == filterSelectorAbout  ||
        selector == filterSelectorFinish ||
        *result != noErr)
    {
        PIUSuitesRelease();
    }
}

