﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

// YCamera YLight YFog YEnvObjs

//=============================================================================
// include
//=============================================================================
#include "Export.h"

//=============================================================================
//! @brief Maya 用のカメラのクラスです。
//=============================================================================
class YCamera : public nn::gfx::tool::dcc::RCamera
{
public:
    //! @brief Maya 上のカメラの種類を表す列挙型です。
    enum Kind
    {
        KIND_BASIC, //!< 基本カメラです。
        KIND_AIM,   //!< カメラおよびエイムカメラです。
        KIND_AIM_UP //!< カメラ、エイム、アップカメラです。
    };

    //-----------------------------------------------------------------------------
    // dag path
    MDagPath m_XformPath; //!< transform ノードの DAG パスです。
    MDagPath m_ShapePath; //!< シェイプノードの DAG パスです。
    MDagPath m_LookAtPath; //!< lookAt ノードの DAG パスです。
    MDagPath m_AimPath; //!< エイムロケータの DAG パスです。
    MDagPath m_UpPath; //!< アップロケータの DAG パスです。

    //-----------------------------------------------------------------------------
    // internal attr
    Kind m_Kind; //!< Maya 上のカメラの種類です。
    std::string m_OrgName; //!< オリジナルの名前（transform ノード名）です。
    MTransformationMatrix::RotationOrder m_RotateOrder; //!< transform ノードの回転順序です。
    MFnCamera::FilmFit m_FilmFit; //!< レゾリューションゲート適合です。

    //-----------------------------------------------------------------------------
    // anim
    MPlug m_AnimPlugs[PARAM_COUNT]; //!< アニメーションの各パラメーターに対応する Maya のプラグ配列です。
    ChannelInput m_AnimChans[PARAM_COUNT]; //!< アニメーションの各パラメーターのチャンネル入力情報配列です。
    YAnimCurve m_Anims[PARAM_COUNT]; //!< アニメーションの各パラメーターのアニメーションカーブ配列です。

public:
    //! コンストラクタです（引数なし）。
    YCamera()
    {
    }

    //! @brief コンストラクタです（transform ノード指定）。
    //!
    //! @param[in,out] rscene シーンです。
    //! @param[in] xformPath transform ノードの DAG パスです。
    //! @param[in] yopt エクスポートオプションです。
    //! @param[out] pStatus 処理結果を格納します。
    //!
    YCamera(
        nn::gfx::tool::dcc::RScene& rscene,
        const MDagPath& xformPath,
        const YExpOpt& yopt,
        MStatus* pStatus = NULL
    );

    //! @brief ユーザーデータを取得します。
    //!
    //! @param[in,out] pScene シーンデータへのポインターです。
    //!
    void GetUserData(nn::gfx::tool::dcc::RScene* pScene)
    {
        ::GetUserData(m_UserDatas, *pScene, m_XformPath.node());
    }

    //! @brief <camera_anim> 要素を出力します。
    //!
    //! @param[in,out] os 出力ストリームです。
    //! @param[in,out] dataStreams データ列配列です。
    //! @param[in] tabCount <camera_anim> 要素のインデントに必要なタブの数です。
    //! @param[in] index <camera_anim_array> 内でのインデックスです。
    //!
    void OutAnim(
        std::ostream& os,
        nn::gfx::tool::dcc::RDataStreamArray& dataStreams,
        const int tabCount,
        const int index
    ) const;

    //! @brief パラメータに対応する Maya のアトリビュート名を返します。
    //!
    //! @param[in] paramIdx パラメータのインデックスです。
    //!
    //! @return パラメータに対応する Maya のアトリビュート名を返します。
    //!
    static const char* GetAttrName(const int paramIdx);
};

//! @brief Maya 用のカメラ配列の定義です。
typedef std::vector<YCamera> YCameraArray;

//! @brief Maya 用のカメラのポインタ配列の定義です。
typedef std::vector<YCamera*> YCameraPtrArray;

//=============================================================================
//! @brief Maya 用のライトのクラスです。
//=============================================================================
class YLight : public nn::gfx::tool::dcc::RLight
{
public:
    //-----------------------------------------------------------------------------
    // dag path
    MDagPath m_XformPath; //!< transform ノードの DAG パスです。
    MDagPath m_ShapePath; //!< シェイプノードの DAG パスです。

    //-----------------------------------------------------------------------------
    // internal attr
    std::string m_OrgName; //!< オリジナルの名前（transform ノード名）です。
    bool m_EmitsDiffuse; //!< ディフューズを放出するなら true です。
    bool m_EmitsSpecular; //!< スペキュラを放出するなら true です。

    //-----------------------------------------------------------------------------
    // anim
    MPlug m_AnimPlugs[PARAM_COUNT]; //!< アニメーションの各パラメーターに対応する Maya のプラグ配列です。
    ChannelInput m_AnimChans[PARAM_COUNT]; //!< アニメーションの各パラメーターのチャンネル入力情報配列です。
    YAnimCurve m_Anims[PARAM_COUNT]; //!< アニメーションの各パラメーターのアニメーションカーブ配列です。

public:
    //! @brief コンストラクタです（transform ノード指定）。
    //!
    //! @param[in,out] rscene シーンです。
    //! @param[in] xformPath transform ノードの DAG パスです。
    //! @param[in] yopt エクスポートオプションです。
    //! @param[out] pStatus 処理結果を格納します。
    //!
    YLight(
        nn::gfx::tool::dcc::RScene& rscene,
        const MDagPath& xformPath,
        const YExpOpt& yopt,
        MStatus* pStatus = NULL
    );

    //! @brief ユーザーデータを取得します。
    //!
    //! @param[in,out] pScene シーンデータへのポインターです。
    //!
    void GetUserData(nn::gfx::tool::dcc::RScene* pScene)
    {
        ::GetUserData(m_UserDatas, *pScene, m_XformPath.node());
    }

    //! @brief <light_anim> 要素を出力します。
    //!
    //! @param[in,out] os 出力ストリームです。
    //! @param[in,out] dataStreams データ列配列です。
    //! @param[in] tabCount <light_anim> 要素のインデントに必要なタブの数です。
    //! @param[in] index <light_anim_array> 内でのインデックスです。
    //!
    void OutAnim(
        std::ostream& os,
        nn::gfx::tool::dcc::RDataStreamArray& dataStreams,
        const int tabCount,
        const int index
    ) const;

    //! @brief 方向を返します。
    //!        ただし、ディレクショナルライト以外は -Z 方向を返します。
    //!
    //! @param[in] snapToZero 0 に近い値を 0 にするなら true を指定します。
    //!
    //! @return 方向を返します。
    //!
    nn::gfx::tool::dcc::RVec3 GetDirection(const bool snapToZero) const;

    //! @brief 目標の位置を返します。
    //!        ただし、スポットライト以外は原点を返します。
    //!
    //! @param[in] position ライトの位置です。
    //! @param[in] magnify 移動値に掛ける倍率です。
    //! @param[in] snapToZero 0 に近い値を 0 にするなら true を指定します。
    //!
    //! @return 目標の位置を返します。
    //!
    nn::gfx::tool::dcc::RVec3 GetAim(
        const nn::gfx::tool::dcc::RVec3& position,
        const double magnify,
        const bool snapToZero
    ) const;

    //! @brief 距離減衰を取得します。
    //!
    //! @param[out] startDist 距離減衰の開始距離を格納します。
    //! @param[out] endDist 距離減衰の終了距離を格納します。
    //! @param[in] magnify 移動値に掛ける倍率です。
    //! @param[in] snapToZero 0 に近い値を 0 にするなら true を指定します。
    //!
    void GetDistAttn(
        float& startDist,
        float& endDist,
        const double magnify,
        const bool snapToZero
    ) const;

    //! @brief 角度減衰を取得します。
    //!        ただし、スポットライト以外は (0, 0) を返します。
    //!
    //! @param[out] startAngle 角度減衰の開始角度を格納します。
    //! @param[out] endAngle 角度減衰の終了角度を格納します。
    //! @param[in] snapToZero 0 に近い値を 0 にするなら true を指定します。
    //!
    void GetAngleAttn(float& startAngle, float& endAngle, const bool snapToZero) const;

    //! @brief パラメータに対応する Maya のアトリビュート名を返します。
    //!
    //! @param[in] paramIdx パラメータのインデックスです。
    //!
    //! @return パラメータに対応する Maya のアトリビュート名を返します。
    //!
    static const char* GetAttrName(const int paramIdx);
};

//! @brief Maya 用のライト配列の定義です。
typedef std::vector<YLight> YLightArray;

//! @brief Maya 用のライトのポインタ配列の定義です。
typedef std::vector<YLight*> YLightPtrArray;

//=============================================================================
//! @brief Maya 用のフォグのクラスです。
//=============================================================================
class YFog : public nn::gfx::tool::dcc::RFog
{
public:
    //-----------------------------------------------------------------------------
    // dag path
    MDagPath m_XformPath; //!< transform ノードの DAG パスです。
    MDagPath m_ShapePath; //!< シェイプノードの DAG パスです。

    //-----------------------------------------------------------------------------
    // internal attr
    std::string m_OrgName; //!< オリジナルの名前（transform ノード名）です。
    MObject m_SgObj; //!< シェーディンググループのオブジェクトです。
    MObject m_MatObj; //!< envFog ノードのオブジェクトです。

    //-----------------------------------------------------------------------------
    // anim
    MPlug m_AnimPlugs[PARAM_COUNT]; //!< アニメーションの各パラメーターに対応する Maya のプラグ配列です。
    ChannelInput m_AnimChans[PARAM_COUNT]; //!< アニメーションの各パラメーターのチャンネル入力情報配列です。
    YAnimCurve m_Anims[PARAM_COUNT]; //!< アニメーションの各パラメーターのアニメーションカーブ配列です。

public:
    //! @brief コンストラクタです（transform ノード指定）。
    //!
    //! @param[in,out] rscene シーンです。
    //! @param[in] xformPath transform ノードの DAG パスです。
    //! @param[in] yopt エクスポートオプションです。
    //! @param[out] pStatus 処理結果を格納します。
    //!
    YFog(
        nn::gfx::tool::dcc::RScene& rscene,
        const MDagPath& xformPath,
        const YExpOpt& yopt,
        MStatus* pStatus = NULL
    );

    //! @brief ユーザーデータを取得します。
    //!
    //! @param[in,out] pScene シーンデータへのポインターです。
    //!
    void GetUserData(nn::gfx::tool::dcc::RScene* pScene)
    {
        ::GetUserData(m_UserDatas, *pScene, m_SgObj);
    }

    //! @brief <fog_anim> 要素を出力します。
    //!
    //! @param[in,out] os 出力ストリームです。
    //! @param[in,out] dataStreams データ列配列です。
    //! @param[in] tabCount <fog_anim> 要素のインデントに必要なタブの数です。
    //! @param[in] index <fog_anim_array> 内でのインデックスです。
    //!
    void OutAnim(
        std::ostream& os,
        nn::gfx::tool::dcc::RDataStreamArray& dataStreams,
        const int tabCount,
        const int index
    ) const;

    //! @brief 距離減衰を取得します。
    //!
    //! @param[out] startDist 距離減衰の開始距離を格納します。
    //! @param[out] endDist 距離減衰の終了距離を格納します。
    //! @param[in] magnify 移動値に掛ける倍率です。
    //! @param[in] snapToZero 0 に近い値を 0 にするなら true を指定します。
    //!
    void GetDistAttn(
        float& startDist,
        float& endDist,
        const double magnify,
        const bool snapToZero
    ) const;

    //! @brief カラーを返します。
    //!
    //! @param[in] snapToZero 0 に近い値を 0 にするなら true を指定します。
    //!
    //! @return カラーを返します。
    //!
    nn::gfx::tool::dcc::RVec3 GetColor(const bool snapToZero) const;

    //! @brief パラメータに対応する Maya のアトリビュート名を返します。
    //!
    //! @param[in] paramIdx パラメータのインデックスです。
    //!
    //! @return パラメータに対応する Maya のアトリビュート名を返します。
    //!
    static const char* GetAttrName(const int paramIdx);
};

//! @brief Maya 用のフォグ配列の定義です。
typedef std::vector<YFog> YFogArray;

//! @brief Maya 用のフォグのポインタ配列の定義です。
typedef std::vector<YFog*> YFogPtrArray;

//=============================================================================
//! @brief 環境オブジェクト群のクラスです。
//=============================================================================
class YEnvObjs
{
public:
    //-----------------------------------------------------------------------------
    // camera
    YCameraArray m_YCameras; //!< カメラ配列です。

    //! @brief 出力用カメラのポインタ配列です。
    //!        m_YCameras 内の要素へのポインタです。
    //!        カメラ名のアルファベット順にソートされます。
    YCameraPtrArray m_pOutYCameras;

    //-----------------------------------------------------------------------------
    // light
    YLightArray m_YLights; //!< ライト配列です。

    //! @brief 出力用ライトのポインタ配列です。
    //!        m_YLights 内の要素へのポインタです。
    //!        ライト名のアルファベット順にソートされます。
    YLightPtrArray m_pOutYLights;

    //-----------------------------------------------------------------------------
    // fog
    YFogArray m_YFogs; //!< フォグ配列です。

    //! @brief 出力用フォグのポインタ配列です。
    //!        m_YFogs 内の要素へのポインタです。
    //!        フォグ名のアルファベット順にソートされます。
    YFogPtrArray m_pOutYFogs;
};

//=============================================================================
// グローバル関数の定義です。
//=============================================================================

//-----------------------------------------------------------------------------
// camera

//! @brief カメラのシェイプノードの DAG パスを返します。
//!
//! @param[in] xformPath カメラの transform ノードの DAG パスです。
//! @param[out] pStatus 処理結果を格納します。
//!
//! @return カメラのシェイプノードの DAG パスを返します。
//!         存在しなかった場合は xformPath をそのまま返します。
//!
MDagPath GetCameraShapePath(const MDagPath& xformPath, MStatus* pStatus = NULL);

//! @brief 立体表示用のステレオカメラの子の transform ノードなら true を返します。
//!
//! @param[in] xformPath カメラの transform ノードの DAG パスです。
//!
//! @return 立体表示用のステレオカメラの子の transform ノードなら true を返します。
//!
bool IsStereoCameraChild(const MDagPath& xformPath);

//-----------------------------------------------------------------------------
// anim

//! @brief 全サブフレームにおける環境オブジェクトのアニメーション値を取得します。
//!        現在のフレームを変化させるたびに呼ばれます。
//!
//! @param[in,out] yenvObjs 環境オブジェクト群です。
//! @param[in] yopt エクスポートオプションです。
//!
void GetEnvObjFullAnimValue(YEnvObjs& yenvObjs, const YExpOpt& yopt);

//! @brief 全サブフレームにおける環境オブジェクトのアニメーション値を分析します。
//!        回転の角度が連続的になるように補正します。
//!        アニメーション値が定数か判定します。
//!
//! @param[in,out] yenvObjs 環境オブジェクト群です。
//! @param[in] yopt エクスポートオプションです。
//!
void AnalyzeEnvObjFullAnim(YEnvObjs& yenvObjs, const YExpOpt& yopt);

//! @brief 環境オブジェクトのキーアニメーションを取得します。
//!
//! @param[in,out] yenvObjs 環境オブジェクト群です。
//! @param[in] yopt エクスポートオプションです。
//!
//! @return 処理結果を返します。
//!
MStatus GetEnvObjKeyAnim(YEnvObjs& yenvObjs, const YExpOpt& yopt);

//-----------------------------------------------------------------------------
// check

//! @brief 環境オブジェクトの数が制限を超えていないか判定します。
//!
//! @return 処理結果を返します。
//!
MStatus CheckEnvObjCount(const YEnvObjs& yenvObjs, const YExpOpt& yopt);

//-----------------------------------------------------------------------------
// name

//! @brief 環境オブジェクトが各カテゴリ内（カメラ、ライト、フォグ）で
//!        ユニークな名前を持つように名前を変更します。
//!
//! @param[in,out] yenvObjs 環境オブジェクト群です。
//! @param[in,out] rscene シーンです。
//! @param[in] yopt エクスポートオプションです。
//!
void SetUniqueEnvObjNames(YEnvObjs& yenvObjs, nn::gfx::tool::dcc::RScene& rscene, const YExpOpt& yopt);

//-----------------------------------------------------------------------------
// sort

//! @brief 環境オブジェクトを中間ファイルに出力する順（名前順）にソートします。
//!
//! @param[in,out] yenvObjs 環境オブジェクト群です。
//!
void SortEnvObjByName(YEnvObjs& yenvObjs);

//-----------------------------------------------------------------------------
// ユーザーデータ関連

//! @brief 環境オブジェクト群のユーザーデータを取得します。
//!
//! @param[in,out] pYEnvObjs 環境オブジェクト群へのポインターです。
//! @param[in,out] pScene シーンデータへのポインターです。
//!
void GetEnvObjUserData(YEnvObjs* pYEnvObjs, nn::gfx::tool::dcc::RScene* pScene);

//-----------------------------------------------------------------------------
// out

//! @brief fsn ファイル（シーンアニメーション）を出力します。
//!
//! @param[in,out] os 出力ストリームです。
//! @param[in,out] dataStreams データ列配列です。
//! @param[in] yenvObjs 環境オブジェクト群です。
//!
//! @return 処理結果を返します。
//!
MStatus OutputFsnFile(
    std::ostream& os,
    nn::gfx::tool::dcc::RDataStreamArray& dataStreams,
    const YEnvObjs& yenvObjs
);

