﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

//=============================================================================
// include
//=============================================================================
#include "LodModel.h"

#include <algorithm>

#include <maya/MGlobal.h>
#include <maya/MDoubleArray.h>
#include <maya/MFnDagNode.h>
#include <maya/MFnMesh.h>
#include <maya/MFnSkinCluster.h>
#include <maya/MItDag.h>
#include <maya/MItMeshPolygon.h>
#include <maya/MItDependencyGraph.h>
#include <maya/MSelectionList.h>

#include "LodCommon.h"

using std::cerr;
using std::endl;

//=============================================================================
// lod ネームスペースを開始します。
//=============================================================================
namespace lod
{

//-----------------------------------------------------------------------------
//! @brief MIntArray から値を検索してインデックスを返します。
//!        見つからなければ -1 を返します。
//-----------------------------------------------------------------------------
int LodFindValueInArray(const MIntArray& array, const int value, const int startIdx = 0)
{
    const int count = array.length();
    for (int index = startIdx; index < count; ++index)
    {
        if (array[index] == value)
        {
            return index;
        }
    }
    return -1;
}

bool Shape::IsUsedBone(const Bone* bone) const
{
    for (size_t i = 0 ; i < bones.size() ; ++i)
    {
        if (bones[i]->GetName() == bone->GetName()) return true;
    }

    return false;
}

MString Shape::GetMeshName() const
{
    return dag.partialPathName();
}

MString Shape::GetMaterialName() const
{
    return MFnDependencyNode(material).name();
}


Model::Model()
{
}

Model::~Model()
{
    {
        std::vector<Bone*>::iterator it = m_bones.begin();
        for ( ; it != m_bones.end() ; ++it) delete *it;
    }

    {
        std::vector<Shape*>::iterator it = m_shapes.begin();
        for ( ; it != m_shapes.end() ; ++it) delete *it;
    }

}

MString Model::GetName() const
{
    return GetNodeName(m_dagRoot);
}

//! マテリアルの MObject 比較用です。
static bool LessMaterial(const MObject& lhs,const MObject& rhs)
{
    // 名前での比較です。
    MFnDependencyNode matL(lhs);
    MFnDependencyNode matR(rhs);

    return ( strcmp(matL.name().asChar(),matR.name().asChar()) < 0);
}


//! Bone* ソート用です。
static bool LessBone(const Bone *lhs,const Bone *rhs)
{
    // DAG 階層の比較結果を返します。
    return (strcmp(lhs->dag.fullPathName().asChar(),rhs->dag.fullPathName().asChar()) < 0);
}

//! Shape* ソート用です。
static bool LessShape(const Shape *lhs,const Shape *rhs)
{
    // まず、DAG 階層を比較します。
    // 差がある場合にはその結果を返します。
    int dagPathDiff = strcmp(lhs->dag.fullPathName().asChar(),rhs->dag.fullPathName().asChar());

    if (dagPathDiff  < 0 )
    {
        return true;
    }
    else if (dagPathDiff > 0)
    {
        return false;
    }
    else
    {
        // DAG パスに差がない同一メッシュ内の場合はマテリアルを比較します。
        return LessMaterial(lhs->material,rhs->material);
    }
}



void Model::SetRootNode(const MDagPath& dagRoot)
{
    m_dagRoot = dagRoot;

    MItDag dagIter(MItDag::kBreadthFirst, MFn::kTransform);
    for (dagIter.reset(dagRoot); !dagIter.isDone() ; dagIter.next())
    {
        MDagPath dag;
        dagIter.getPath(dag);
        if (dag.hasFn(MFn::kTransform) && this->IsExportableNode(dag))
        {
            this->RegisterNode(dag);
        }
    }

    // 出力順にソートしておきます
    std::sort(m_bones.begin(),m_bones.end(),LessBone);
    std::sort(m_shapes.begin(),m_shapes.end(),LessShape);
}

//!
void Model::RegisterNode(const MDagPath& dag)
{
    MStatus stat;

    m_exportableNodes.append(dag);

    if (dag.hasFn(MFn::kMesh))
    {
        MFnMesh meshFn(dag);
        if (meshFn.isIntermediateObject()) return;

        // メッシュをマテリアル単位で分解してシェイプ情報化します。
        MObjectArray shaders;
        MIntArray faceMaterialIds;
        meshFn.getConnectedShaders(0,shaders,faceMaterialIds);

        for (unsigned i = 0 ; i < shaders.length() ; ++i)
        {
            MIntArray faceIdsInMaterial;
            for (unsigned f = 0 ; f < faceMaterialIds.length() ; ++f)
            {
                if (static_cast<int>(i) == faceMaterialIds[f])
                {
                    faceIdsInMaterial.append(f);
                }
            }

            MObject faceComp;
            GetMeshComponentFromIndices(faceComp,faceIdsInMaterial,MFn::kMeshPolygonComponent);

            this->RegisterShape(dag, faceComp, shaders[i]);
        }
    }

}

//! １シェイプ情報を登録します。
void Model::RegisterShape(const MDagPath& dag,const MObject& faceComp,const MObject& sg)
{
    MStatus stat;

    Shape* pShape = new Shape();
    pShape->dag = dag;
    pShape->m_MeshPath = dag;
    if (pShape->m_MeshPath.node().apiType() == MFn::kTransform)
    {
        pShape->m_MeshPath.extendToShape();
    }
    pShape->faceComp = faceComp;
    pShape->material = GetMaterialFromShadingEngine(sg);
    pShape->skinMode = Shape::cNoSkin;
    pShape->m_VertexSkinningCount = 0;

    m_shapes.push_back(pShape);

    // 有効な UV セットを収集します。
    this->SetShapeUVSetNames(pShape);

    // 有効なカラーセットを収集します。
    this->SetShapeColorSetNames(pShape);

    MFnMesh meshFn(dag);

    MObject faceComp_ = faceComp;
    MItMeshPolygon polyIt(dag,faceComp_);


    // スキンモデルの場合はスキン情報を収集します。
    MObject skin = GetRelativeSkinCluster(pShape->m_MeshPath.node());
    if (skin.hasFn(MFn::kSkinClusterFilter))
    {
        // 少なくとも RigidSkin or SmoothSkin なので Rigid としておきます。
        pShape->skinMode = Shape::cRigidSkin;

        MFnSkinCluster skinFn(skin);

        MDagPathArray influences;
        skinFn.influenceObjects(influences);

        // 構成頂点 ID を収集します。
        MIntArray vtxIds;
        for ( ; !polyIt.isDone() ; polyIt.next())
        {
            MIntArray vtxIdsInFace;
            polyIt.getVertices(vtxIdsInFace);
            for (unsigned v = 0; v < vtxIdsInFace.length(); ++v)
            {
                int vid = vtxIdsInFace[v];
                if (!FindInMArray(vtxIds,vid))
                {
                    vtxIds.append(vid);
                }
            }
        }

        // コンポーネントは順番を保持しないので、昇順にソートしておきます。
        SortArrayAscending(vtxIds);

        MObject vertexComp;
        GetMeshComponentFromIndices(vertexComp, vtxIds, MFn::kMeshVertComponent);

        // スキン情報関するウェイト情報を取得します。

        unsigned numInfluences;
        MDoubleArray weights;
        stat = skinFn.getWeights(dag, vertexComp, weights, numInfluences);
        if (!CheckStatusOK(stat,"getWeights")) return;

        const int numVertices = vtxIds.length();

        // このマテリアルで使用されるインフルエンスと、シェイプのスムース状況を調査します。
        // ２インフルエンス以上使用する頂点が１つでもあればスムーススキンです。
        MDagPathArray infsUsedInThisMat;

        for (int v = 0 ; v < numVertices ; ++v)
        {
            int vid = vtxIds[v];

            int infCountPerVtx = 0;
            for (unsigned i = 0 ; i < numInfluences ; ++i)
            {
                double w = weights[numInfluences * v + i];
                if (w > 0.0)
                {
                    // 使用インフルエンスに未登録なら登録します。
                    if (!FindInMArray(infsUsedInThisMat,influences[i]))
                    {
                        infsUsedInThisMat.append(influences[i]);
                    }

                    // 使用インフルエンス数が 2 以上ならスムーススキンシェイプです。
                    if (++infCountPerVtx > 1)
                    {
                        pShape->skinMode = Shape::cSmoothSkin;
                    }

                //  TRACE("%s.vtx[%d]: weight(%lf) infCount(%d)\n",dag.partialPathName().asChar(),vid,w,infCountPerVtx);

                }
            }

            // エラー原因選択用に複数インフルエンスから影響のある頂点IDを保持しておきます。
            if (infCountPerVtx > 1)
            {
                pShape->smoothVtxIds.append(vid);
            }

            if (infCountPerVtx > pShape->m_VertexSkinningCount)
            {
                pShape->m_VertexSkinningCount = infCountPerVtx;
            }
        }

        for (unsigned i = 0 ; i < influences.length() ; ++i)
        {
            const MDagPath& dagInf = influences[i];

            Bone *pBone = this->GetBoneByDag(dagInf);
            if (!pBone)
            {
                pBone = new Bone();
                pBone->dag = dagInf;

                // モデルのボーンには頂点での使用の有無に関わらず登録します (エクスポータの <bone> 仕様)
                m_bones.push_back(pBone);
            }

            // この <shape> で使用している bone の場合
            if ( FindInMArray(infsUsedInThisMat,dagInf) )
            {
                // rigid/smooth はこのシェイプの影響を受けます。
                switch (pShape->skinMode)
                {
                case Shape::cSmoothSkin:
                    pBone->useSmoothSkin = true;
            //      TRACE("bone(%s) is smooth : shape(%s,%s)\n",pBone->dag.partialPathName().asChar(),
            //                      meshFn.partialPathName().asChar(), MFnDependencyNode(pShape->material).name().asChar());
                    break;

                case Shape::cRigidSkin:
                    pBone->useRigidSkin = true;
                    break;
                default:
                    break;
                }

                // シェイプの使用ボーンに登録します
                pShape->bones.push_back(pBone);
            }

        //  TRACE("%s : smooth(%d) rigid(%d)\n",pBone->dag.partialPathName().asChar(),pBone->useSmoothSkin,pBone->useRigidSkin);

            // この Bone に影響を受ける頂点を収集しておきます。
            MIntArray affectedVtxIdsInThisShape;

            MSelectionList points;
            MFloatArray infWeights;
            skinFn.getPointsAffectedByInfluence(dagInf, points, infWeights);

            for (unsigned s = 0 ; s < points.length() ; ++s)
            {
                MDagPath dummy;
                MObject vtxComp;
                points.getDagPath(s,dummy,vtxComp);

                MIntArray affectedVtxIds;
                GetIndicesFromMeshComponent(affectedVtxIds,vtxComp);

                for (unsigned v = 0; v < affectedVtxIds.length() ; ++v)
                {
                    int vid = affectedVtxIds[v];

                    if ( FindInMArray(vtxIds,vid) )
                    {
                        affectedVtxIdsInThisShape.append(vid);
                    }
                }

            //  TRACE("%s <-> %s \n",pShape->GetMeshName().asChar(), dag.partialPathName().asChar());
            }

            pShape->boneToVtxIds.insert( std::make_pair(pBone,affectedVtxIdsInThisShape) );
        }
    }

} // NOLINT(impl/function_size)

static void GetFileNodesForPlug(MObjectArray& fileNodes,MPlug plug)
{
    MItDependencyGraph dgIter(plug,
                                MFn::kFileTexture,
                                MItDependencyGraph::kUpstream,
                                MItDependencyGraph::kBreadthFirst,
                                MItDependencyGraph::kNodeLevel);

    for ( ; !dgIter.isDone() ; dgIter.next() )
    {
        MObject obj = dgIter.currentItem();
        if (obj.apiType() == MFn::kFileTexture)
        {
            if (!FindInMArray(fileNodes,obj))
            {
                fileNodes.append(obj);
            }
        }
    }
}

//-----------------------------------------------------------------------------
//! @brief UV セットファミリ名からインスタンスに対応した UV セット名を取得します。
//!        UV セットがインスタンスごとに設定されている場合、
//!        mesh ノードのインスタンス番号に対応する UV セット名を取得します。
//!
//! @param[in] uvSetFamilyName UV セットファミリ名です。
//! @param[in] meshPath mesh ノードの DAG パスです。
//!
//! @return インスタンスに対応した UV セット名を返します。
//!         存在しなければ空文字を返します。
//-----------------------------------------------------------------------------
static MString GetInstanceUvSetName(const MDagPath& meshPath, const MString& uvSetFamilyName)
{
    MFnMesh meshFn(meshPath);
    if (!meshFn.isUVSetPerInstance(uvSetFamilyName))
    {
        return uvSetFamilyName;
    }
    else // UV セットがインスタンスごとに設定されている場合
    {
        MStringArray members;
        meshFn.getUVSetsInFamily(uvSetFamilyName, members);
        const int memberCount = static_cast<int>(members.length());
        // UV セットファミリ中の UV セットのうち、
        // mesh ノードのインスタンス番号に対応したものを取得します。
        for (int iMember = 0; iMember < memberCount; ++iMember)
        {
            MIntArray instances;
            meshFn.getAssociatedUVSetInstances(members[iMember], instances);
            if (LodFindValueInArray(instances, meshPath.instanceNumber()) != -1)
            {
                //cerr << "instance uv set: " << meshPath.partialPathName() << ": " << members[iMember] << " " << members << " " << instances << R_ENDL;
                return members[iMember];
            }
        }
        return MString();
    }
}

//-----------------------------------------------------------------------------
//! @brief テクスチャノードに対応する UV セット名を取得します。
//!        UV セットがインスタンス間で共有されない場合、
//!        インスタンスに対応した UV セット名を返します。
//-----------------------------------------------------------------------------
static MString GetUvSetForFileNode(const MDagPath& meshPath, const MObject& fileObj)
{
    MStatus status;

    //-----------------------------------------------------------------------------
    // デフォルトの UV セット名を取得します。
    MString uvSetName = "map1";
    MPlug uvSetPlug = MFnDagNode(meshPath).findPlug("uvSet");
    if (uvSetPlug.numElements() > 0)
    {
        MPlug namePlug = uvSetPlug.elementByLogicalIndex(0, &status);
        if (status)
        {
            namePlug.child(0).getValue(uvSetName);
        }
    }

    //-----------------------------------------------------------------------------
    // file ノードの uvCoord に接続された place2dTexture ノードを取得します。
    MObject placeObj;
    MPlugArray plugArray;
    MFnDependencyNode(fileObj).findPlug("uvCoord").connectedTo(plugArray, true, false);
    if (plugArray.length() != 0 &&
        plugArray[0].node().apiType() == MFn::kPlace2dTexture)
    {
        placeObj = plugArray[0].node();
    }

    //-----------------------------------------------------------------------------
    // uvChooser ノードを取得します。
    // place2dTexture ノードがあれば place2dTexture ノードの uvCoord、
    // なければ file ノードの uvCoord プラグの入力を調べます。
    MObject chooserObj;
    MObject uvTargetObj = (!placeObj.isNull()) ? placeObj : fileObj;
    MFnDependencyNode(uvTargetObj).findPlug("uvCoord").connectedTo(plugArray, true, false);
    if (plugArray.length() > 0 &&
        plugArray[0].node().apiType() == MFn::kUvChooser)
    {
        chooserObj = plugArray[0].node();
    }

    //-----------------------------------------------------------------------------
    // uvChooser ノードから UV セット名を取得します。
    if (!chooserObj.isNull())
    {
        MPlug uvSetsPlug = MFnDependencyNode(chooserObj).findPlug("uvSets");
        MIntArray idxs;
        uvSetsPlug.getExistingArrayAttributeIndices(idxs);
        const int uvSetsCount = static_cast<int>(idxs.length());
        for (int iElem = 0; iElem < uvSetsCount; ++iElem)
        {
            MPlug uvnPlug = uvSetsPlug.elementByLogicalIndex(idxs[iElem], &status);
            if (!status)
            {
                break;
            }
            uvnPlug.connectedTo(plugArray, true, false);
            if (plugArray.length() == 0)
            {
                continue;
            }
            if (plugArray[0].node() == meshPath.node())
            {
                uvnPlug.getValue(uvSetName);
                break;
            }
        }
    }

    //-----------------------------------------------------------------------------
    // インスタンス単位の UV セットの場合のインデックスをカットします。
    const int iParenthesis = uvSetName.index('(');
    if (iParenthesis > 0)
    {
        uvSetName = uvSetName.substring(0, iParenthesis - 1);
    }

    //-----------------------------------------------------------------------------
    // インスタンスに対応した UV セット名を取得します。
    uvSetName = GetInstanceUvSetName(meshPath, uvSetName);

    //MGlobal::displayInfo("uvset: " + meshPath.partialPathName() + ": " + MFnDependencyNode(fileObj).name() + ": " + uvSetName);

    return uvSetName;
}

//static MString GetUvSetForFileNode(const MString& meshName,const MString& fileNodeName)
//{
//  MString cmd = "getAttr(`uvLink -q -texture " + fileNodeName + " -qo " + meshName + "`)";
//
//  MStringArray uvSets;
//  MGlobal::executeCommand(cmd,uvSets);
//
//  return uvSets[0];
//}
// ↑
// uvLink -q が返すプラグのノード名はユニークな DAG パスでない（mesh ノード名のみ）なので、
// 同名の mesh ノードが複数ある場合に正しく動作しません。
// たとえば、Base|pPlane1 が map1、LOD1|pPlane1 が map2 の場合、
// getAttr pPlaneShape1.uvSet[0].uvSetName は配列 { "map1", "map2" } を返します。

void Model::SetShapeUVSetNames(Shape* pShape) const
{
    // サンプラごとのテクスチャに対する UV Set 取得
    MFnDependencyNode materialNodeFn(pShape->material);
    MObjectArray fileNodes;

    // color
    GetFileNodesForPlug(fileNodes, materialNodeFn.findPlug("color"));

    // transparency
    GetFileNodesForPlug(fileNodes, materialNodeFn.findPlug("transparency"));

    // ambient
    GetFileNodesForPlug(fileNodes, materialNodeFn.findPlug("ambientColor"));

    // incandescence
    GetFileNodesForPlug(fileNodes, materialNodeFn.findPlug("incandescence"));

    // normal
    GetFileNodesForPlug(fileNodes, materialNodeFn.findPlug("normalCamera"));

    for (unsigned i = 0 ; i < fileNodes.length() ; ++i)
    {
        const MString uvSet = GetUvSetForFileNode(pShape->m_MeshPath, fileNodes[i]);
        if (!FindInMArray(pShape->uvSets, uvSet))
        {
            pShape->uvSets.append(uvSet);
        }
    }

    //-----------------------------------------------------------------------------
    // 未登録の UV セットで、名前が nw4f_uv または nw4f_fix で始まるものを取得します。
    MFnMesh meshFn(pShape->m_MeshPath);
    MStringArray uvSetFamilyNames;
    meshFn.getUVSetFamilyNames(uvSetFamilyNames);
    for (unsigned iUvSet = 0; iUvSet < uvSetFamilyNames.length(); ++iUvSet)
    {
        const MString& uvSetFamilyName = uvSetFamilyNames[iUvSet];
        if (IsUvSetNameForNintendo(uvSetFamilyName))
        {
            const MString uvSetName = GetInstanceUvSetName(pShape->m_MeshPath, uvSetFamilyName);
            if (uvSetName != "" && !FindInMArray(pShape->uvSets, uvSetName))
            {
                pShape->uvSets.append(uvSetName);
            }
        }
    }
}


static void ParseColorSetType(Shape::ColorSet* pColorSet, const MString& suffix)
{
    const int ComponentCountRa = 5; // RA の 2 成分を出力する場合の成分数です。

    Shape::ColorSet& colorSet = *pColorSet;
    colorSet.isRa = false;

    MString compCountStr;
    MString typeStr;
    if (suffix.length() == 1)
    {
        typeStr = suffix;
    }
    else if (suffix.length() == 2)
    {
        compCountStr = suffix.substring(0, 0);
        typeStr = suffix.substring(1, 1);

        // サイズが [1,5] の整数か確認し、異なれば無効にします。
        if (compCountStr.isInt())
        {
            colorSet.compCount = compCountStr.asInt();
            if (colorSet.compCount == ComponentCountRa)
            {
                colorSet.compCount = 2;
                colorSet.isRa = true;
            }
            if (colorSet.compCount < 1 || 4 < colorSet.compCount)
            {
                colorSet.compCount = -1;
            }
        }
    }

    colorSet.type =
        (typeStr == "i") ? Shape::ColorSet::cInt  :
        (typeStr == "u") ? Shape::ColorSet::cUint :
        Shape::ColorSet::cFloat;
}

//-----------------------------------------------------------------------------
//! @brief 無視する（出力しない）カラーセットなら true を返します。
//!
//! @param[in] colSetName カラーセット名です。
//!
//! @return 無視する（出力しない）カラーセットなら true を返します。
//-----------------------------------------------------------------------------
static bool IgnoresColorSet(const MString& colSetName)
{
    return (colSetName.indexW("nw_ignore") == 0);
}

//-----------------------------------------------------------------------------
//! @brief シェイプのカラーセット名配列を設定します。
//-----------------------------------------------------------------------------
void Model::SetShapeColorSetNames(Shape* pShape) const
{
    static const MString COLOR_PREFIX = "nw4f_color";

    MFnMesh meshFn(pShape->dag);

    MStringArray colorSetFamilyNames;
    meshFn.getColorSetFamilyNames(colorSetFamilyNames);

    for (unsigned i = 0 ; i < colorSetFamilyNames.length() ; ++i)
    {
        const MString& colorSetFamilyName = colorSetFamilyNames[i];
        MString colorSetName;
        if (!meshFn.isColorSetPerInstance(colorSetFamilyName))
        {
            colorSetName = colorSetFamilyName;
        }
        else
        {
            MStringArray colorSetNames;
            meshFn.getColorSetsInFamily(colorSetFamilyName,colorSetNames);
            for (unsigned c = 0 ; c < colorSetNames.length() ; ++c)
            {
                MIntArray instances;
                meshFn.getAssociatedColorSetInstances(colorSetNames[c],instances);
                if (FindInMArray<MIntArray,int>(instances,pShape->dag.instanceNumber()) )
                {
                    colorSetName = colorSetNames[c];
                    break;
                }
            }
        }


        if (colorSetName.indexW(COLOR_PREFIX) == 0)
        {
            Shape::ColorSet colorSet;
            colorSet.name = colorSetName;

            MString stripPrefix = colorSetFamilyName.substring(COLOR_PREFIX.length(),colorSetName.length());

            // nw4f_color の次が [0,7] の数字でなければ無効
            MString colorIndexStr = stripPrefix.substring(0,0);
            if (!colorIndexStr.isInt()) continue;

            int colorIndex = colorIndexStr.asInt();
            if (colorIndex < 0 || 7 < colorIndex) continue;

            // _ で区切った最後の文字がタイプ決定文字ルールかを判断
            MStringArray buf;
            stripPrefix.split('_',buf);
            ParseColorSetType(&colorSet, buf[buf.length() - 1]);

            // サイズ無指定、不正な場合はカラーセットのタイプから判断します。
            colorSet.repr = meshFn.getColorRepresentation(colorSetName);
            if (colorSet.compCount == -1)
            {
                colorSet.compCount = (colorSet.repr == MFnMesh::kRGB) ? 3 : 4;
            }

            if (colorIndex >= pShape->colorSets.size())
            {
                pShape->colorSets.resize(colorIndex + 1);
            }

            pShape->colorSets[colorIndex] = colorSet;
        }

    }

    // 空の場合はカレントカラーセットを 0 番目に
    if (pShape->colorSets.size() == 0)
    {
        MString colorSetName;
        meshFn.getCurrentColorSetName(colorSetName);
        if (!IgnoresColorSet(colorSetName))
        {
            Shape::ColorSet colorSet;
            colorSet.name = colorSetName;

            MStringArray buf;
            colorSetName.split('_',buf);
            ParseColorSetType(&colorSet, buf[buf.length() - 1]);

            colorSet.repr = meshFn.getColorRepresentation(colorSetName);
            if (colorSet.compCount == -1)
            {
                colorSet.compCount = (colorSet.repr == MFnMesh::kRGB) ? 3 : 4;
            }

            pShape->colorSets.push_back(colorSet);
        }
    }

}



//! 指定されたシェイプ情報と一致するシェイプを取得。
//! 異なるレベル間の一致シェイプを探す用途を想定。
Shape* Model::FindCorrespondingShape(const Shape* other) const
{
    const MString& nodeName = MayaNode(other->dag.node()).GetName();

    for (size_t i = 0 ; i < m_shapes.size() ; ++i)
    {
        if ( MayaNode(m_shapes[i]->dag.node()).GetName() == nodeName )
        {
            if ( other->material == m_shapes[i]->material)
            {
                return m_shapes[i];
            }
        }
    }

    return NULL;
}


Bone* Model::GetBoneByDag(const MDagPath& dag) const
{
    for (size_t i = 0 ; i < m_bones.size() ; ++i)
    {
    //  TRACE("%s <-> %s\n",m_bones[i]->dag.partialPathName().asChar(),dag.partialPathName().asChar());
        if (m_bones[i]->dag == dag)
        {
            return m_bones[i];
        }
    }

    return NULL;
}

//! 指定されたシェイプ情報と一致するボーンを取得。
//! 異なる LOD 間の一致ボーンを探す用途を想定。
Bone* Model::FindCorrespondingBone(const Bone *other) const
{
    const MString& nodeName = MayaNode(other->dag.node()).GetName();

    for (size_t i = 0 ; i < m_bones.size() ; ++i)
    {
        if ( MayaNode(m_bones[i]->dag.node()).GetName() == nodeName )
        {
            return m_bones[i];
        }
    }

    return NULL;
}



int Model::GetMaterials(MObjectArray& materials) const
{
    std::vector<MObject> matList;
    for (std::vector<Shape*>::const_iterator it = m_shapes.begin() ; it != m_shapes.end() ; ++it)
    {
        if (std::find(matList.begin(),matList.end(), (*it)->material) == matList.end())
        {
            matList.push_back((*it)->material);
        }
    }

    std::sort(matList.begin(),matList.end(),LessMaterial);

    for (std::vector<MObject>::const_iterator it = matList.begin(); it != matList.end() ; ++it)
    {
        materials.append(*it);
    }


    return materials.length();
}


bool Model::IsExportableNode(const MDagPath& dag) const
{
    MObject oNode = dag.node();

    MayaNode node(oNode);

    if (dag.node().hasFn(MFn::kGeometric) && !node.GetBoolAttr("primaryVisibility")) return false;
    if (node.GetBoolAttr("template")) return false;
    if (node.GetBoolAttr("intermediateObject") ) return false;
    if (node.GetBoolAttr("overrideEnabled") )
    {
        if (!node.GetBoolAttr("overrideVisibility")) return false;
        if (!node.GetBoolAttr("overrideShading")) return false;

    //  MPlug overrideDispTypePlug = dagFn.findPlug("overrideDisplayType");
    //  if (overrideDispTypePlug.asInt() == 1) return false; // override template

        int displayType;
        node.GetAttr("overrideDisplayType", displayType);
        if (displayType == 1) return false; // 1: override template
    }

    // 親を再帰チェック
    MFnDagNode dagFn(dag);
    MObject oParent = dagFn.parent( dagFn.instanceCount(true) - 1 );
    if (oParent.hasFn(MFn::kDagNode))
    {
        MFnDagNode parentDagFn(oParent);
        MDagPath parentDag;
        parentDagFn.getPath(parentDag);

        return IsExportableNode(parentDag);
    }

    return true;
}

//=============================================================================
// lod ネームスペースを終了します。
//=============================================================================
} // namespace lod

