﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

// RVertex RVtxAttrib RVtxBuffer RVtxInput
// RShapeT RSubmesh RPrimSet RPrimitive RPrimVtx
// RProgShape ROutShapeInfo ROutShapeVtxInfo ROutShapeResult
// RVtxMtx RVtxWeight
// RKeyShape RShapeUpdate

//=============================================================================
// include
//=============================================================================
#include "DccCommon.h"

//=============================================================================
// dcc ネームスペースを開始します。
//=============================================================================
namespace nn {
namespace gfx {
namespace tool {
namespace dcc {

//=============================================================================
// 処理選択用マクロです。
//=============================================================================

//#define R_RELOAD_SHAPE_MTX_PAL // 小さな行列パレットにロードしながら描画する場合に定義します。

//=============================================================================
// 定数の定義です。
//=============================================================================

//! 法線・接線・従法線の許容誤差です。XYZ 成分の誤差がすべてこの値未満なら同じとみなします。
const float R_VTX_NRM_TOLERANCE = 0.001f;

//! テクスチャ座標の許容誤差です。XY 成分の誤差がすべてこの値未満なら同じとみなします。
const float R_VTX_TEX_TOLERANCE = 1.0e-5f;

//! 法線・接線・従法線の長さが 0 の場合にこのベクトルで置き換えます。
const RVec3 R_ZERO_NRM_REPLACE = RVec3::kYAxis;

//=============================================================================
//! @brief 頂点属性のクラスです。
//=============================================================================
class RVtxAttrib
{
public:
    //! @brief 関連付けに利用するヒント情報を表す列挙型です。
    enum Hint
    {
        POSITION,
        NORMAL,
        TANGENT,
        BINORMAL,
        COLOR,
        UV,
        BLEND_INDEX,
        BLEND_WEIGHT,
        HINT_COUNT      //!< ヒント情報の総数です。
    };

    static const int HINT_INDEX_MAX = 254; //!< ヒント情報のインデックスの最大値です。

    //! @brief シェーダで扱う論理的な型を表す列挙型です。
    enum Type
    {
        INT  , INT2  , INT3  , INT4  ,
        UINT , UINT2 , UINT3 , UINT4 ,
        FLOAT, FLOAT2, FLOAT3, FLOAT4
    };

    //! @brief 量子化型を表す列挙型です。
    enum QuantizeType
    {
        NONE,
        UNORM_8, UINT_8, SNORM_8, SINT_8,
        UNORM_4_4,
        UNORM_16, UINT_16, SNORM_16, SINT_16, FLOAT_16,
        UNORM_8_8, UINT_8_8, SNORM_8_8, SINT_8_8,
        UINT_32, SINT_32, FLOAT_32,
        UNORM_16_16, UINT_16_16, SNORM_16_16, SINT_16_16, FLOAT_16_16,
        FLOAT_10_11_11,
        UNORM_8_8_8_8, UINT_8_8_8_8, SNORM_8_8_8_8, SINT_8_8_8_8,
        UNORM_10_10_10_2, UINT_10_10_10_2, SNORM_10_10_10_2, SINT_10_10_10_2,
        UINT_32_32, SINT_32_32, FLOAT_32_32,
        UNORM_16_16_16_16, UINT_16_16_16_16, SNORM_16_16_16_16, SINT_16_16_16_16, FLOAT_16_16_16_16,
        UINT_32_32_32, SINT_32_32_32, FLOAT_32_32_32, UINT_32_32_32_32, SINT_32_32_32_32, FLOAT_32_32_32_32
    };

    Hint m_Hint; //!< 関連付けに利用するヒント情報です。
    int m_HintIndex; //!< 関連付けに利用するヒント情報のインデックスです。
    std::string m_NameOverride; //!< 頂点属性名のオーバーライドです。
    Type m_Type; //!< シェーダで扱う論理的な型です。
    QuantizeType m_QuantizeType; //!< 量子化型です。
    int m_Count; //!< 頂点数です。
    int m_StreamIndex; //!< 頂点ストリームのインデックスです。

public:
    //! コンストラクタです。
    RVtxAttrib(
        const Hint hint,
        const int hintIndex,
        const Type type,
        const QuantizeType quantizeType,
        const int count,
        const int streamIndex
    )
    : m_Hint(hint),
      m_HintIndex(hintIndex),
      m_Type(type),
      m_QuantizeType(quantizeType),
      m_Count(count),
      m_StreamIndex(streamIndex)
    {
    }

    //! 頂点属性名を返します。
    std::string GetName() const;

    //! @brief 名前の文字列を返します（static 関数）。
    static std::string GetNameString(const Hint hint, const int hintIndex);

    //! @brief ヒント情報の文字列を返します（static 関数）。
    static std::string GetHintString(const Hint hint, const int hintIndex);

    //! @brief 出力します。
    //!
    //! @param[in,out] os 出力ストリームです。
    //! @param[in] tabCount <vtx_attrib> 要素のインデントに必要なタブの数です。
    //! @param[in] index <vtx_attrib_array> 内でのインデックスです。
    //!
    void Out(std::ostream& os, const int tabCount, const int index) const;

    //! 同値であれば true を返します。
    bool operator==(const RVtxAttrib& rhs) const
    {
        return (
            m_Hint         == rhs.m_Hint         &&
            m_Type         == rhs.m_Type         &&
            m_QuantizeType == rhs.m_QuantizeType &&
            m_Count        == rhs.m_Count        &&
            m_StreamIndex  == rhs.m_StreamIndex);
    }

    //! 同値でなければ true を返します。
    bool operator!=(const RVtxAttrib& rhs) const
    {
        return !(*this == rhs);
    }
};

//! @brief 頂点属性配列の定義です。
typedef std::vector<RVtxAttrib> RVtxAttribArray;

//=============================================================================
//! @brief 頂点入力のクラスです。
//=============================================================================
class RVtxInput
{
public:
    int m_AttribIndex; //!< シェーダに入力する頂点属性のインデックスです。

public:
    //! コンストラクタです。
    explicit RVtxInput(int attribIndex)
    : m_AttribIndex(attribIndex)
    {
    }

    //! @brief 出力します。
    //!
    //! @param[in,out] os 出力ストリームです。
    //! @param[in] tabCount <input> 要素のインデントに必要なタブの数です。
    //! @param[in] index <input_array> 内でのインデックスです。
    //!
    void Out(std::ostream& os, const int tabCount, const int index) const;

    //! 同値であれば true を返します。
    bool operator==(const RVtxInput& rhs) const
    {
        return (
            m_AttribIndex  == rhs.m_AttribIndex);
    }

    //! 同値でなければ true を返します。
    bool operator!=(const RVtxInput& rhs) const
    {
        return !(*this == rhs);
    }
};

//! @brief 頂点入力配列の定義です。
typedef std::vector<RVtxInput> RVtxInputArray;

//=============================================================================
//! @brief 頂点バッファのクラスです。
//=============================================================================
class RVtxBuffer
{
public:
    RVtxInputArray m_Inputs; //!< 頂点入力配列です。

public:
    //! コンストラクタです。
    RVtxBuffer()
    {
    }

    //! @brief 出力します。
    //!
    //! @param[in,out] os 出力ストリームです。
    //! @param[in] tabCount <vtx_buffer> 要素のインデントに必要なタブの数です。
    //! @param[in] index <vtx_buffer_array> 内でのインデックスです。
    //!
    void Out(std::ostream& os, const int tabCount, const int index) const;

    //! 同値であれば true を返します。
    bool operator==(const RVtxBuffer& rhs) const
    {
        return RIsSameArray(m_Inputs, rhs.m_Inputs);
    }

    //! 同値でなければ true を返します。
    bool operator!=(const RVtxBuffer& rhs) const
    {
        return !(*this == rhs);
    }
};

//! @brief 頂点バッファ配列の定義です。
typedef std::vector<RVtxBuffer> RVtxBufferArray;

//=============================================================================
//! @brief 頂点データのクラスです。
//=============================================================================
class RVertex
{
public:
    RVtxAttribArray m_VtxAttribs; //!< 頂点属性配列です。
    RVtxBufferArray m_VtxBuffers; //!< 頂点バッファ配列です。

public:
    //! コンストラクタです。
    RVertex()
    {
    }

    //! @brief 出力します。
    //!
    //! @param[in,out] os 出力ストリームです。
    //! @param[in] tabCount <vertex> 要素のインデントに必要なタブの数です。
    //! @param[in] index <vertex_array> 内でのインデックスです。
    //!
    void Out(std::ostream& os, const int tabCount, const int index) const;

    //! 同値であれば true を返します。
    bool operator==(const RVertex& rhs) const
    {
        return (
            RIsSameArray(m_VtxAttribs, rhs.m_VtxAttribs) &&
            RIsSameArray(m_VtxBuffers, rhs.m_VtxBuffers));
    }

    //! 同値でなければ true を返します。
    bool operator!=(const RVertex& rhs) const
    {
        return !(*this == rhs);
    }
};

//! @brief 頂点データ配列の定義です。
typedef std::vector<RVertex> RVertexArray;

//=============================================================================
//! @brief 頂点が参照するボーンインデックスとウェイト値のクラスです。
//=============================================================================
class RVtxWeight
{
public:
    enum { WEIGHT_MAX = 100 };

    int m_BoneIdx; //!< ボーンインデックスです。
    int m_Weight; //!< ウェイト値です。

public:
    //! コンストラクタです。
    RVtxWeight(const int boneIdx = 0, const int weight = WEIGHT_MAX)
    : m_BoneIdx(boneIdx),
      m_Weight(weight)
    {
    }

    //! 同値であれば true を返します。
    bool operator==(const RVtxWeight& rhs) const
    {
        return (m_BoneIdx == rhs.m_BoneIdx &&
                m_Weight  == rhs.m_Weight);
    }

    //! 同値でなければ true を返します。
    bool operator!=(const RVtxWeight& rhs) const
    {
        return (m_BoneIdx != rhs.m_BoneIdx ||
                m_Weight  != rhs.m_Weight);
    }
};

//! @brief 頂点が参照するボーンインデックスとウェイト値配列の定義です。
typedef std::vector<RVtxWeight> RVtxWeightArray;

//=============================================================================
//! @brief 頂点行列のクラスです。
//=============================================================================
class RVtxMtx
{
public:
    enum { DEFAULT_BONE_MAX = 8 };

    RVtxWeightArray m_VtxWeights; //!< ボーンウェイト値配列です。

public:
    //! コンストラクタです（引数なし）。
    RVtxMtx()
    {
    }

    //! コンストラクタです（ボーンインデックスとウェイトを指定）。
    RVtxMtx(const int boneIdx, const int weight = RVtxWeight::WEIGHT_MAX)
    {
        m_VtxWeights.push_back(RVtxWeight(boneIdx, weight));
    }

    int GetBoneCount() const { return static_cast<int>(m_VtxWeights.size()); }
    bool IsFull() const { return (GetBoneCount() == 1); }
    int GetBoneIndex(const int localBoneIdx) const
    {
        return m_VtxWeights[localBoneIdx].m_BoneIdx;
    }
    void SetBoneIndex(const int localBoneIdx, const int newBoneIdx)
    {
        m_VtxWeights[localBoneIdx].m_BoneIdx = newBoneIdx;
    }
    int GetWeight(const int localBoneIdx) const
    {
        return m_VtxWeights[localBoneIdx].m_Weight;
    }
    void SetWeight(const int localBoneIdx, const int newWeight)
    {
        m_VtxWeights[localBoneIdx].m_Weight = newWeight;
    }
    int GetWeightSum() const
    {
        int weightSum = 0;
        for (int ibone = 0; ibone < static_cast<int>(m_VtxWeights.size()); ++ibone)
        {
            weightSum += m_VtxWeights[ibone].m_Weight;
        }
        return weightSum;
    }
    void Append(const int boneIdx, const int weight)
    {
        m_VtxWeights.push_back(RVtxWeight(boneIdx, weight));
        SortByIndexLess();
    }

    void SortByIndexLess();
    void SortByWeightGreater();

    //! @brief 頂点行列のボーンウェイト値配列を調整します。
    //!
    //! @param[in] maxBoneCount 1 頂点が参照できる最大ボーン数です。
    //!
    void AdjustWeight(const int maxBoneCount = DEFAULT_BONE_MAX);

    //! @brief ボーンに対する行列インデックスの配列からスキニング頂点属性を取得します。
    //!
    //! @param[out] idxVecs ブレンドインデックスベクトル配列を格納します。
    //! @param[out] weightVecs ブレンドウェイト値ベクトル配列を格納します。
    //! @param[in] boneMtxIdxs ボーンに対する行列インデックスの配列です。
    //!
    #ifndef R_RELOAD_SHAPE_MTX_PAL
    void GetSkinningVtxAttrMtxIdxs(
        RIVec4* idxVecs,
        RVec4* weightVecs,
        const RIntArray& boneMtxIdxs
    ) const;
    #endif

    //! @brief 行列パレットからスキニング頂点属性を取得します。
    //!
    //! @param[out] idxVecs ブレンドインデックスベクトル配列を格納します。
    //! @param[out] weightVecs ブレンドウェイト値ベクトル配列を格納します。
    //! @param[in] mtxPal 行列パレットです。
    //! @param[in] smoothSkinFlag スムーススキニングなら true を指定します。
    //!
    #ifdef R_RELOAD_SHAPE_MTX_PAL
    void GetSkinningVtxAttrMtxPal(
        RIVec4* idxVecs,
        RVec4* weightVecs,
        const RIntArray& mtxPal,
        const bool smoothSkinFlag
    ) const;
    #endif

    void Print(std::ostream& os, const char* title = NULL) const;

    //! 同値であれば true を返します。
    bool operator==(const RVtxMtx& rhs) const
    {
        return (RIsSameArray(m_VtxWeights, rhs.m_VtxWeights));
    }

    //! 同値でなければ true を返します。
    bool operator!=(const RVtxMtx& rhs) const
    {
        return (!RIsSameArray(m_VtxWeights, rhs.m_VtxWeights));
    }
};

//! @brief 頂点行列配列の定義です。
typedef std::vector<RVtxMtx> RVtxMtxArray;

//-----------------------------------------------------------------------------
//! @brief 警告表示用関数の定義です。
//-----------------------------------------------------------------------------
typedef void (*RPolyWarningFuncPtr)(const std::string& msg, void* pParam);

//-----------------------------------------------------------------------------
//! @brief 進行状況表示用関数の定義です。
//-----------------------------------------------------------------------------
typedef void (*RPolyProgVoidFuncPtr)(void* pParam);

//=============================================================================
//! @brief シェイプ処理の進行状況を伝えるクラスです。
//=============================================================================
class RProgShape
{
public:
    // warning
    bool m_InvalidPolyWarningFlag;
    RPolyWarningFuncPtr m_WarningFunc;
    void* m_pWarningParam;

    // progress
    RPolyProgVoidFuncPtr m_OptimizePrimSetsFunc;
    void* m_pProgressParam;

public:
    //! コンストラクタです。
    RProgShape(
        RPolyWarningFuncPtr warningFunc = NULL,
        void* pWarningParam = NULL,
        RPolyProgVoidFuncPtr optimizePrimSetsFunc = NULL,
        void* pProgressParam = NULL)
    : m_InvalidPolyWarningFlag(false),
      m_WarningFunc(warningFunc),
      m_pWarningParam(pWarningParam),
      m_OptimizePrimSetsFunc(optimizePrimSetsFunc),
      m_pProgressParam(pProgressParam)
    {
    }

    void DoWarning(const std::string& msg) const
    {
        if (m_WarningFunc != NULL)
        {
            m_WarningFunc(msg, m_pWarningParam);
        }
    }

    void StartOptimizePrimSets() const
    {
        if (m_OptimizePrimSetsFunc != NULL)
        {
            m_OptimizePrimSetsFunc(m_pProgressParam);
        }
    }
};

//=============================================================================
//! @brief プリミティブの頂点情報のクラスです。
//=============================================================================
class RPrimVtx
{
public:
    static const int VTX_TAN_MAX = 4; //!< 1 頂点あたりの接線・従法線の最大数です。
    static const int VTX_COL_MAX = 8; //!< 1 頂点あたりの頂点カラーの最大数です。
    static const int VTX_TEX_MAX = 8; //!< 1 頂点あたりのテクスチャ座標の最大数です。
    static const int VTX_USR_MAX = 1; //!< 1 頂点あたりのユーザー頂点属性の最大数です。

    //! @brief 頂点属性の種類を表す列挙型です。
    enum VtxAttr
    {
        POS0,
        NRM0,
        TAN0,
        TAN1,
        TAN2,
        TAN3,
        BIN0,
        BIN1,
        BIN2,
        BIN3,
        COL0,
        COL1,
        COL2,
        COL3,
        COL4,
        COL5,
        COL6,
        COL7,
        TEX0,
        TEX1,
        TEX2,
        TEX3,
        TEX4,
        TEX5,
        TEX6,
        TEX7,
        IDX0,
        IDX1,
        WGT0,
        WGT1,
        USR0,
        VA_COUNT
    };

    static const uint32_t COMPARE_TAN1 = (1 << 0); //!< 同値判定の TAN1、BIN1 以降比較フラグです。
    static const uint32_t COMPARE_COL1 = (1 << 1); //!< 同値判定の COL1 以降比較フラグです。
    static const uint32_t COMPARE_TEX1 = (1 << 2); //!< 同値判定の TEX1 以降比較フラグです。
    static const uint32_t COMPARE_USR0 = (1 << 3); //!< 同値判定の USR0 以降比較フラグです。

    //! @brief 頂点属性の値の型を表す列挙型です。
    enum ValueType
    {
        VALUE_INT,
        VALUE_UINT,
        VALUE_FLOAT
    };

    int m_Attr[VA_COUNT]; //!< 頂点属性ごとのデータへのインデックスです。
    int m_Mtx; //!< 頂点行列のインデックスです。

public:
    //! コンストラクタです。
    RPrimVtx()
    {
        for (int iAttr = 0; iAttr < VA_COUNT; ++iAttr)
        {
            m_Attr[iAttr] = -1;
        }
        m_Mtx = -1;
    }

    //! @brief データへのインデックスの参照を返す添え字演算子です。
    //!
    //! @param[in] iAttr 頂点属性の種類です。
    //!
    //! @return データへのインデックスの参照を返します。
    //!
    int& operator[](const int iAttr)
    {
        return m_Attr[iAttr];
    }

    //! @brief データへのインデックスを返す添え字演算子です。
    //!
    //! @param[in] iAttr 頂点属性の種類です。
    //!
    //! @return データへのインデックスを返します。
    //!
    int operator[](const int iAttr) const
    {
        return m_Attr[iAttr];
    }

    //! 同値であれば true を返します。
    bool operator==(const RPrimVtx& rhs) const;

    //! 同値でなければ true を返します。
    bool operator!=(const RPrimVtx& rhs) const
    {
        return !(*this == rhs);
    }

    //! @brief 行列以外の頂点属性が同値であれば true を返します。
    //!
    //! @param[in] other 比較する他の頂点属性です。
    //! @param[in] compareFlags 比較する属性の種類を CompareFlag* の論理和で指定します。
    //!
    //! @return 同値であれば true を返します。
    //!
    bool IsSameAttr(const RPrimVtx& other, const uint32_t compareFlags) const;
};

//! @brief プリミティブの頂点情報配列の定義です。
typedef std::vector<RPrimVtx> RPrimVtxArray;

//=============================================================================
//! @brief プリミティブのクラスです。
//=============================================================================
class RPrimitive
{
public:
    //! @brief プリミティブタイプを表す列挙型です。
    enum PrimType
    {
        POINTS,
        LINES,
        LINE_STRIP,
        TRIANGLES,
        TRIANGLE_FAN,
        TRIANGLE_STRIP,
        OTHER
    };
    enum { VTX_SIZE_MAX = 0x7fffffff }; // 実質無制限です。

    // attr
    PrimType m_PrimType; //!< プリミティブタイプです。
    int m_VtxCount; //!< 頂点数です。
    RPrimVtxArray m_Vtxs; //!< 頂点情報配列です。
    RIntArray m_BoneIdxs; //!< プリミティブが使用するボーンインデックス配列です。配列の値は重複しません。

    RIntArray m_OutIdxs; //!< 出力する頂点インデックス配列です。

public:
    //! コンストラクタです（引数なし）。
    RPrimitive()
    : m_PrimType(OTHER),
      m_VtxCount(0)
    {
    }

    //! コンストラクタです（頂点数を指定）。
    explicit RPrimitive(const int vtxCount)
    : m_VtxCount(vtxCount),
      m_Vtxs(vtxCount)
    {
        if (m_VtxCount == 1)
        {
            m_PrimType = POINTS;
        }
        else if (m_VtxCount == 2)
        {
            m_PrimType = LINES;
        }
        else if (m_VtxCount == 3)
        {
            m_PrimType = TRIANGLES;
        }
        else
        {
            m_PrimType = TRIANGLE_FAN;
        }
    }

    void SetPrimVtx(const int ivtx, const RPrimVtx& vtx)
    {
        m_Vtxs[ivtx] = vtx;
    }

    void SetBoneIndexes(const RVtxMtxArray& vtxMtxs);
    int GetBoneCount() const { return static_cast<int>(m_BoneIdxs.size()); } // SetBoneIndexes 実行後に有効です。
    int GetBoneIndex(const int localBoneIdx) const // SetBoneIndexes 実行後に有効です。
    {
        return m_BoneIdxs[localBoneIdx];
    }

    int GetMtx(const int ivtx) const { return m_Vtxs[ivtx].m_Mtx; }

    //! ポリゴン数を取得します。
    int GetPolygonCount() const;

    //! 三角形数を取得します。
    int GetTriangleCount() const;

    std::string GetPrimitiveModeStr() const;

    static std::string RPrimitive::GetPrimitiveModeStr(const RPrimitive::PrimType type);
};

//! @brief プリミティブのポインタ配列の定義です。
typedef std::vector<RPrimitive*> RPrimPtrArray;

//! @brief プリミティブのポインタ配列のイテレータの定義です。
typedef RPrimPtrArray::iterator ItRPrimPtr;

//! @brief プリミティブのポインタ配列の const イテレータの定義です。
typedef RPrimPtrArray::const_iterator const_ItRPrimPtr;

//=============================================================================
//! @brief 同じ行列パレットを使用するプリミティブ群のクラスです。
//=============================================================================
class RPrimSet
{
public:
    RIntArray m_BoneIdxs; //!< プリミティブ群が使用するボーンインデックス配列です。配列の値は重複しません。
    RPrimPtrArray m_pPrimitives; //!< プリミティブのポインタ配列です。

    //! @brief 出力用ボーンインデックス配列です。
    //!        行列パレットを使用しない場合は m_BoneIdxs と同じです。
    //!        行列パレットを使用する場合、このプリミティブ群で使用されていないボーンを含むことがあります。
    RIntArray m_OutBoneIdxs;

public:
    //! コンストラクタです。
    RPrimSet()
    {
    }

    int GetBoneCount() const { return static_cast<int>(m_BoneIdxs.size()); }
    int GetBoneIndex(const int localBoneIdx) const
    {
        return m_BoneIdxs[localBoneIdx];
    }
    int GetOutBoneCount() const { return static_cast<int>(m_OutBoneIdxs.size()); }
    int GetOutBoneIdx(const int localBoneIdx) const
    {
        return m_OutBoneIdxs[localBoneIdx];
    }

    //! 頂点ロード数を取得します。
    int GetVtxCount() const;

    //! ポリゴン数を取得します。
    int GetPolygonCount() const;

    //! 三角形数を取得します。
    int GetTriangleCount() const;

    bool PrimTypeExists(const RPrimitive::PrimType type) const;
};

//! @brief プリミティブ群のポインタ配列の定義です。
typedef std::vector<RPrimSet*> RPrimSetPtrArray;

//! @brief プリミティブ群のポインタ配列のイテレータの定義です。
typedef RPrimSetPtrArray::iterator ItRPrimSetPtr;

//! @brief プリミティブ群のポインタ配列の const イテレータの定義です。
typedef RPrimSetPtrArray::const_iterator const_ItRPrimSetPtr;

//=============================================================================
//! @brief サブメッシュのクラスです。
//=============================================================================
class RSubmesh
{
public:
    int m_Offset;
    int m_Count;

public:
    //! コンストラクタです。
    RSubmesh(int offset, int count)
    : m_Offset(offset),
      m_Count(count)
    {
    }

    //! @brief 出力します。
    //!
    //! @param[in,out] os 出力ストリームです。
    //! @param[in] tabCount <submesh> 要素のインデントに必要なタブの数です。
    //! @param[in] index <submesh_array> 内でのインデックスです。
    //!
    void Out(std::ostream& os, const int tabCount, const int index) const;

    //! 同値であれば true を返します。
    bool operator==(const RSubmesh& rhs) const
    {
        return (
            m_Offset     == rhs.m_Offset &&
            m_Count      == rhs.m_Count);
    }

    //! 同値でなければ true を返します。
    bool operator!=(const RSubmesh& rhs) const
    {
        return !(*this == rhs);
    }
};

//! @brief サブメッシュ配列の定義です。
typedef std::vector<RSubmesh> RSubmeshArray;

//=============================================================================
//! @brief キーシェイプのクラスです。
//=============================================================================
class RKeyShape
{
public:
    std::string m_Name; //!< キーシェイプ名です。
    RStringArray m_AttribNames; //!< キーシェイプが使用する頂点属性名の配列です。

public:
    //! コンストラクタです。
    explicit RKeyShape(const std::string& name)
    : m_Name(name)
    {
    }

    //! @brief 出力します。
    //!
    //! @param[in,out] os 出力ストリームです。
    //! @param[in] tabCount <key_shape> 要素のインデントに必要なタブの数です。
    //! @param[in] index <key_shape_array> 内でのインデックスです。
    //!
    void Out(std::ostream& os, const int tabCount, const int index) const;
};

//! @brief キーシェイプ配列の定義です。
typedef std::vector<RKeyShape> RKeyShapeArray;

//=============================================================================
//! @brief シェイプの頂点属性出力情報のクラスです。
//=============================================================================
class ROutShapeVtxInfo
{
public:
    //! @brief 成分数です。0 以上 4 以下の値を指定します。
    //!        出力しない場合は 0 を指定します。
    //!        COL, USR のみ使用します。
    int m_CompCount;

    RPrimVtx::ValueType m_ValueType; //!< 値の型です。

    //! @brief 値の配列のポインタ配列です。
    //!        ベースシェイプが先頭で、ベースシェイプ以外のキーシェイプが
    //!        存在すればその後に格納します。
    //!        POS, NRM は RVec3Array* のみ指定可能です。
    //!        TAN, BIN は RVec4Array* のみ指定可能です。
    //!        COL は RVec4Array* のみ指定可能です。
    //!        TEX は RVec2Array* のみ指定可能です。
    //!        IDX と WGT は使用しません（NULL のままで可）。
    //!        USR は RVec4Array* のみ指定可能です。
    std::vector<const void*> m_pArrays;

public:
    //! @brief コンストラクタです。
    //!
    //! @param[in] compCount 成分数です。
    //! @param[in] valueType 値の型です。
    //! @param[in] pArray ベースシェイプの値の配列のポインタです。
    //!
    ROutShapeVtxInfo(
        const int compCount = 0,
        const RPrimVtx::ValueType valueType = RPrimVtx::VALUE_FLOAT,
        const void* pArray = NULL)
    : m_CompCount(compCount),
      m_ValueType(valueType)
    {
        m_pArrays.push_back(pArray);
    }
};

//=============================================================================
//! @brief シェイプ出力情報のクラスです。
//=============================================================================
class ROutShapeInfo
{
public:
    int m_Index;            //!< <shape> の index に出力する値です。
    std::string m_Name;     //!< <shape> の name に出力する名前です。
    std::string m_MatName;  //!< <shape_info> の mat_name に出力する名前です。
    std::string m_BoneName; //!< <shape_info> の bone_name に出力する名前です。
    ROrientedBB m_OrientedBB; //!< 有向バウンディングボックスです。現在出力しません。
    ROutShapeVtxInfo m_VtxInfos[RPrimVtx::VA_COUNT]; //!< シェイプの頂点属性出力情報配列です。

    //! @brief UV セットの頂点属性名配列へのポインタです。出力する UV セット数と同じ長さです。
    //!        NULL ならすべての UV セットの頂点属性名をヒント情報から自動決定します。
    //!        部分的に空文字であれば、空文字の UV セットの頂点属性名のみ自動決定します。
    const RStringArray* m_pUvAttribNames;

    //! @brief UV セットのヒント情報インデックス配列へのポインタです。出力する UV セット数と同じ長さです。
    //!        NULL なら uv0 から順番に割り当てます。
    const RIntArray* m_pUvHintIdxs;

    //! @brief キーシェイプの名前の配列です。
    //!        ベースシェイプが先頭で、ベースシェイプ以外のキーシェイプの名前を
    //!        その後に格納します。
    //!        ベースシェイプのみ出力する場合は空のままにします。
    RStringArray m_KeyNames;

public:
    //! コンストラクタです。
    ROutShapeInfo()
    : m_Index(0),
      m_pUvAttribNames(NULL),
      m_pUvHintIdxs(NULL)
    {
    }
};

//=============================================================================
//! @brief シェイプ出力結果情報のクラスです。
//=============================================================================
class ROutShapeResult
{
public:
    int m_TriangleCount; //!< 出力した三角形数です。
    int m_IndexCount; //!< 出力した頂点インデックス数です。
    int m_VertexCount; //!< 出力した頂点数です。
    int m_ProcessVertexCount; //!< 頂点シェーダが処理する頂点数です。
};

//=============================================================================
//! @brief シェイプのクラスです。
//=============================================================================
class RShape // RShapeT
{
public:
    //! @brief スキニングモードを表す列挙型です。
    enum SkinningMode
    {
        NO_SKINNING,    //!< スキニングなしです。
        RIGID,          //!< リジッドスキニングです。
        SMOOTH          //!< スムーススキニングです。
    };

    static const int RENDER_PRIORITY_MIN = -128; //!< 描画ソートに利用する優先度の最小値です。
    static const int RENDER_PRIORITY_MAX =  127; //!< 描画ソートに利用する優先度の最大値です。

    // attr
    SkinningMode m_SkinningMode; //!< スキニングモードです。
    int m_MtxPalCount; //!< 行列パレットサイズです。
    std::string m_OrgName; //!< デバッグ表示用の名前（ノード名 + マテリアル名）です。
    bool m_VtxAttrFlag[RPrimVtx::VA_COUNT]; //!< 頂点属性の有効フラグです。

    int m_OutBoneCountMax; //!< 全プリミティブ群での使用ボーン数の最大値です。
    RIVec4Array m_VtxBlendIdxs[RVtxMtx::DEFAULT_BONE_MAX / 4]; //!< 頂点のブレンドインデックス配列です。
    RVec4Array  m_VtxBlendWgts[RVtxMtx::DEFAULT_BONE_MAX / 4]; //!< 頂点のブレンドウェイト値配列です。

    RPrimPtrArray m_pPrimitives;  //!< プリミティブのポインタ配列です。
    RPrimSetPtrArray m_pPrimSets; //!< プリミティブセットのポインタ配列です。

    // info
    int m_OrgPrimSetCount; //!< 最適化前のプリミティブセット数です。
    int m_OutPrimSetCount; //!< 最適化後のプリミティブセット数です。
    int m_OrgMtxLoadCount; //!< 最適化前の行列ロード回数です。
    int m_OutMtxLoadCount; //!< 最適化後の行列ロード回数です。

    // progress
    RProgShape* m_ProgShapePtr; //!< シェイプ処理の進行状況です。

public:
    //! @brief コンストラクタです。
    //!
    //! @param[in] skinningMode スキニングモードです。
    //! @param[in] mtxPalCount 行列パレットサイズです。
    //! @param[in] orgName シェイプ名（デバッグ表示用）です。設定しなくても構いません。
    //! @param[in] progShapePtr シェイプ処理の進行状況を伝えるオブジェクトです。設定しなくても構いません。
    //!
    explicit RShape(
        const SkinningMode skinningMode,
        const int mtxPalCount,
        const char* orgName = NULL,
        RProgShape* progShapePtr = NULL);

    //! デストラクタです。
    virtual ~RShape() { FreeMemory(); }

    //! メモリを開放します。
    void FreeMemory();

    //! @brief ポイント群を追加します。
    //!        頂点属性がすべて等しいポイントがすでに存在する場合は追加しません。
    //!
    //! @param[in] rprim 追加する POINTS タイプのプリミティブです。
    //! @param[in] vtxMtxs 頂点行列配列です。
    //!
    //! @return 処理結果を返します。常に成功します。
    //!
    RStatus AppendPoints(const RPrimitive& rprim, const RVtxMtxArray& vtxMtxs);

    //! @brief ライン群を追加します。
    //!        頂点属性がすべて等しいラインがすでに存在する場合は追加しません。
    //!
    //! @param[in] rprim 追加する LINE_LOOP タイプのプリミティブです。
    //! @param[in] vtxMtxs 頂点行列配列です。
    //!
    //! @return 処理結果を返します。常に成功します。
    //!
    RStatus AppendLines(const RPrimitive& rprim, const RVtxMtxArray& vtxMtxs);

    //! @brief ポリゴンを追加します。
    //!
    //! @param[in] rprim 追加する TRIANGLE_FAN タイプのプリミティブです。
    //! @param[in] vtxMtxs 頂点行列配列です。
    //! @param[in] forceTriangulate 四角形以上を強制的に三角形分割して追加するなら true を指定します。
    //!                             false の場合、使用する行列数が行列パレットサイズを超えるなら三角形分割します。
    //! @param[in] triPattern 三角形分割パターンです。
    //!
    //! @return 処理結果を返します。現在は常に成功します。
    //!         三角形分割しても参照するボーン数が行列パレットサイズを超える場合は FAILURE を返します。
    //!
    RStatus AppendPolygon(
        const RPrimitive& rprim,
        const RVtxMtxArray& vtxMtxs,
        const bool forceTriangulate,
        const int triPattern
    );

    //! @brief 出力用データを設定して最適化します。
    //!
    //! @param[in] vtxMtxs 頂点行列の値の配列です。
    //! @param[in] boneMtxIdxs ボーンに対する行列パレットインデックスの配列です。
    //!                        スムーススキニングの場合はスムーススキニング用のインデックス配列、
    //!                        リジッドスキニングの場合はリジッドスキニング用のインデックス配列、
    //!                        リジッドボディの場合はすべて -1 の配列を指定します。
    //!
    //! @return 処理結果を返します。常に成功します。
    //!
    RStatus SetAndOptimize(const RVtxMtxArray& vtxMtxs, const RIntArray& boneMtxIdxs);

    //! 頂点ロード数を取得します。
    int GetVtxCount() const;

    //! ポリゴン数を取得します。
    int GetPolygonCount() const;

    //! 三角形数を取得します。
    int GetTriangleCount() const;

    //! プリミティブ数を取得します。
    int GetPrimCount() const;

    //! 特定のプリミティブが存在するか取得します。
    bool PrimTypeExists(const RPrimitive::PrimType type) const;

    //! @brief 出力します。
    //!
    //! @param[in,out] os 出力ストリームです。
    //! @param[in,out] vertices 頂点データ配列です。
    //! @param[in,out] dataStreams データ列配列です。
    //! @param[out] result 出力結果情報を格納します。
    //! @param[in] tabCount <shape> 要素のインデントに必要なタブの数です。
    //! @param[in] outInfo シェイプ出力情報のオブジェクトです。
    //!
    //! @return 処理結果を返します。失敗した場合の原因は RStatus::GetMessage で取得できます。
    //!
    RStatus Out(
        std::ostream& os,
        RVertexArray& vertices,
        RDataStreamArray& dataStreams,
        ROutShapeResult& result,
        const int tabCount,
        const ROutShapeInfo& outInfo
    ) const;

    //! @brief 標準シェーダの行列パレットサイズを取得します。
    //!
    //! @param[in] nonUniformScaleFlag XYZ のスケールが均等でないシェイプを
    //!                                正しくライティングするなら true を指定します。
    //! @param[in] maxReservedUniformRegs ユーザーが使用するユニフォームレジスタの数の最大値です。
    //!
    //! @return 行列パレットサイズを返します。
    //!
    static int GetStandardMtxPalCount(
        const bool nonUniformScaleFlag,
        const int maxReservedUniformRegs
    );

protected:
    RStatus SetPrimSets();

    void OptimizePrimSets();

    void SetOutBoneIndexes();

    void SetSkinningVtxAttr(const RVtxMtxArray& vtxMtxs, const RIntArray& boneMtxIdxs);

    int GetSkinningComponentCount(
        const int outBoneCountMax,
        const RVec4Array* outWeights
    ) const;
};

//=============================================================================
//! @brief シェイプアニメーションの頂点属性更新情報のクラスです。
//=============================================================================
class RShapeUpdate
{
public:
    bool m_UpdatesPos; //!< 頂点座標を更新するなら true です。
    bool m_UpdatesNrm; //!< 法線を更新するなら true です。
    bool m_UpdatesTan; //!< 接線を更新するなら true です。
    bool m_UpdatesBin; //!< 従法線を更新するなら true です。
    bool m_UpdatesCols[RPrimVtx::VTX_COL_MAX]; //!< 頂点カラーを更新するなら true です。

public:
    //! コンストラクタです。
    RShapeUpdate()
    {
        DisableAll();
    }

    //! すべて頂点カラー更新フラグに同じ値を設定します。
    void SetAllCols(const bool enable)
    {
        for (int iCol = 0; iCol < RPrimVtx::VTX_COL_MAX; ++iCol)
        {
            m_UpdatesCols[iCol] = enable;
        }
    }

    //! すべて属性の更新を無効にします。
    void DisableAll()
    {
        m_UpdatesPos =
        m_UpdatesNrm =
        m_UpdatesTan =
        m_UpdatesBin = false;
        SetAllCols(false);
    }

    //! いずれかの頂点カラーを更新するなら true を返します。
    bool UpdatesSomeCol() const
    {
        for (int iCol = 0; iCol < RPrimVtx::VTX_COL_MAX; ++iCol)
        {
            if (m_UpdatesCols[iCol])
            {
                return true;
            }
        }
        return false;
    }

    //! いずれかの属性を更新するなら true を返します。
    bool UpdatesSome() const
    {
        return (
            m_UpdatesPos ||
            m_UpdatesNrm ||
            m_UpdatesTan ||
            m_UpdatesBin ||
            UpdatesSomeCol());
    }
};

//=============================================================================
// グローバル関数の定義です。
//=============================================================================

//! @brief 最鋭角が最大になる三角形分割パターンを取得します。
//!
//! @param[in] poss 多角形の頂点座標配列です。
//!
//! @return 最鋭角が最大になる三角形分割パターンを返します。
//!
int RGetAngleTriangulationPattern(const RVec3Array& poss);

//! @brief 頂点カラーの同一判定のための許容値を取得します。
//!
//! @param[in] valueType 値の型です。
//!
float RGetVertexColorTolerance(const RPrimVtx::ValueType valueType);

//! @brief ユーザー頂点属性の同一判定のための許容値を取得します。
//!
//! @param[in] valueType 値の型です。
//!
float RGetVertexUserAttrTolerance(const RPrimVtx::ValueType valueType);

//=============================================================================
// dcc ネームスペースを終了します。
//=============================================================================
} // namespace dcc
} // namespace tool
} // namespace gfx
} // namespace nn

