﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

// RShapeCT AppendPolygon AppendLines AppendPoints TriangulateRPrim
// SetAndOptimize SetPrimSets OptimizePrimSets MergeSingleRPrim
// OutRShape AppendOneVtxStream AppendVtxStream AppendVec4VtxStream
// AdjustTangents

// RVertex RVtxAttrib RVtxBuffer RVtxInput
// RShape RPrimSet RPrimitive RPrimVtx
// RVtxMtx RVtxWeight
// RKeyShape

//=============================================================================
// include
//=============================================================================
#include "DccShape.h"
#include "DccOutput.h"

using namespace std;

//=============================================================================
// dcc ネームスペースを開始します。
//=============================================================================
namespace nn {
namespace gfx {
namespace tool {
namespace dcc {

//-----------------------------------------------------------------------------
//! 行列ロード状態を表す列挙型です。
#ifdef R_RELOAD_SHAPE_MTX_PAL
enum MtxLoad
{
    MTX_LOAD_NONE,
    MTX_LOAD_NOLOAD,
    MTX_LOAD_LOAD
};
#endif

//-----------------------------------------------------------------------------
//! @brief 頂点カラーの誤差の許容値を取得します。
//-----------------------------------------------------------------------------
float RGetVertexColorTolerance(const RPrimVtx::ValueType valueType)
{
    R_UNUSED_VARIABLE(valueType); // 現状 byte、ubyte 型がないので使用しません。
    return 1.0e-5f;
}

//-----------------------------------------------------------------------------
//! @brief ユーザー頂点属性の誤差の許容値を取得します。
//-----------------------------------------------------------------------------
float RGetVertexUserAttrTolerance(const RPrimVtx::ValueType valueType)
{
    R_UNUSED_VARIABLE(valueType); // 現状 byte、ubyte 型がないので使用しません。
    return 1.0e-5f;
}

//-----------------------------------------------------------------------------
//! @brief 頂点属性の名前の文字列を返します（static 関数）。
//-----------------------------------------------------------------------------
std::string RVtxAttrib::GetNameString(const Hint hint, const int hintIndex)
{
    static const char* const nameStrs[HINT_COUNT] =
    {
        "_p",
        "_n",
        "_t",
        "_b",
        "_c",
        "_u",
        "_i",
        "_w",
    };
    return nameStrs[hint] + RGetNumberString(hintIndex);
}

//-----------------------------------------------------------------------------
//! @brief 頂点属性のヒント情報の文字列を返します（static 関数）。
//-----------------------------------------------------------------------------
std::string RVtxAttrib::GetHintString(const Hint hint, const int hintIndex)
{
    static const char* const hintStrs[HINT_COUNT] =
    {
        "position",
        "normal",
        "tangent",
        "binormal",
        "color",
        "uv",
        "blendindex",
        "blendweight",
    };
    return hintStrs[hint] + RGetNumberString(hintIndex);
}

//-----------------------------------------------------------------------------
//! @brief 頂点属性名を返します。
//-----------------------------------------------------------------------------
std::string RVtxAttrib::GetName() const
{
    if (!m_NameOverride.empty())
    {
        return m_NameOverride;
    }
    else
    {
        return GetNameString(m_Hint, m_HintIndex);
    }
}

//-----------------------------------------------------------------------------
//! @brief 頂点属性を出力します。
//-----------------------------------------------------------------------------
void RVtxAttrib::Out(std::ostream& os, const int tabCount, const int index) const
{
    static const char* const typeStrs[] =
    {
        "int"  , "int2"  , "int3"  , "int4"  ,
        "uint" , "uint2" , "uint3" , "uint4" ,
        "float", "float2", "float3", "float4"
    };
    static const char* const quantizeTypeStrs[] =
    {
        "none",
        "unorm_8", "uint_8", "snorm_8", "sint_8",
        "unorm_4_4",
        "unorm_16", "uint_16", "snorm_16", "sint_16", "float_16",
        "unorm_8_8", "uint_8_8", "snorm_8_8", "sint_8_8",
        "uint_32", "sint_32", "float_32",
        "unorm_16_16", "uint_16_16", "snorm_16_16", "sint_16_16", "float_16_16",
        "float_10_11_11",
        "unorm_8_8_8_8", "uint_8_8_8_8", "snorm_8_8_8_8", "sint_8_8_8_8",
        "unorm_10_10_10_2", "uint_10_10_10_2", "snorm_10_10_10_2", "sint_10_10_10_2",
        "uint_32_32", "sint_32_32", "float_32_32",
        "unorm_16_16_16_16", "uint_16_16_16_16", "snorm_16_16_16_16", "sint_16_16_16_16", "float_16_16_16_16",
        "uint_32_32_32", "sint_32_32_32", "float_32_32_32", "uint_32_32_32_32", "sint_32_32_32_32", "float_32_32_32_32"
    };

    const int& tc = tabCount;
    os << RTab(tc) << "<vtx_attrib attrib_index=\"" << index
       << "\" name=\"" << GetName()
       << "\" hint=\"" << GetHintString(m_Hint, m_HintIndex) << "\"" << R_ENDL;
    os << RTab(tc + 1) << "type=\"" << typeStrs[m_Type] << "\"" << R_ENDL;
    os << RTab(tc + 1) << "quantize_type=\"" << quantizeTypeStrs[m_QuantizeType] << "\"" << R_ENDL;
    os << RTab(tc + 1) << "count=\"" << m_Count << "\"" << R_ENDL;
    os << RTab(tc + 1) << "stream_index=\"" << m_StreamIndex << "\"" << R_ENDL;
    os << RTab(tc) << "/>" << R_ENDL;
}

//-----------------------------------------------------------------------------
//! @brief 頂点入力を出力します。
//-----------------------------------------------------------------------------
void RVtxInput::Out(std::ostream& os, const int tabCount, const int index) const
{
    os << RTab(tabCount) << "<input index=\"" << index
       << "\" attrib_index=\"" << m_AttribIndex
       << "\" />" << R_ENDL;
}

//-----------------------------------------------------------------------------
//! @brief 頂点バッファを出力します。
//-----------------------------------------------------------------------------
void RVtxBuffer::Out(std::ostream& os, const int tabCount, const int index) const
{
    const int& tc = tabCount;
    os << RTab(tc) << "<vtx_buffer index=\"" << index << "\">" << R_ENDL;
    ROutArrayElement(os, tc + 1, m_Inputs, "input_array");
    os << RTab(tc) << "</vtx_buffer>" << R_ENDL;
}

//-----------------------------------------------------------------------------
//! @brief 頂点データを出力します。
//-----------------------------------------------------------------------------
void RVertex::Out(std::ostream& os, const int tabCount, const int index) const
{
    const int& tc = tabCount;
    os << RTab(tc) << "<vertex vertex_index=\"" << index << "\">" << R_ENDL;
    ROutArrayElement(os, tc + 1, m_VtxAttribs, "vtx_attrib_array");
    ROutArrayElement(os, tc + 1, m_VtxBuffers, "vtx_buffer_array");
    os << RTab(tc) << "</vertex>" << R_ENDL;
}

//-----------------------------------------------------------------------------
//! @brief 頂点行列をプリントします（デバッグ用）。
//-----------------------------------------------------------------------------
void RVtxMtx::Print(std::ostream& os, const char* title) const
{
    std::string str;
    if (title != NULL)
    {
        str += std::string(title) + ": ";
    }
    str += RGetNumberString(GetBoneCount()) + ": ";
    for (int ibone = 0; ibone < GetBoneCount(); ++ibone)
    {
        str += RGetNumberString(GetBoneIndex(ibone)) + " (" +
            RGetNumberString(GetWeight(ibone)) + ")  ";
    }
    os << str.c_str() << endl;
}

//-----------------------------------------------------------------------------
//! @brief 頂点行列のボーンウェイト値配列をボーンインデックスの小さい順にソートします。
//-----------------------------------------------------------------------------
void RVtxMtx::SortByIndexLess()
{
    //Print(cerr, "i before");
    const int boneCount = GetBoneCount();
    for (int ib0 = 0; ib0 < boneCount - 1; ++ib0)
    {
        RVtxWeight& vw0 = m_VtxWeights[ib0];
        for (int ib1 = ib0 + 1; ib1 < boneCount; ++ib1)
        {
            RVtxWeight& vw1 = m_VtxWeights[ib1];
            if (vw1.m_BoneIdx < vw0.m_BoneIdx)
            {
                RSwapValue(vw0, vw1);
            }
        }
    }
    //Print(cerr, "i after ");
}

//-----------------------------------------------------------------------------
//! @brief 頂点行列のボーンウェイト値配列をウェイト値の大きい順にソートします。
//-----------------------------------------------------------------------------
void RVtxMtx::SortByWeightGreater()
{
    //Print(cerr, "w before");
    const int boneCount = GetBoneCount();
    for (int ib0 = 0; ib0 < boneCount - 1; ++ib0)
    {
        RVtxWeight& vw0 = m_VtxWeights[ib0];
        for (int ib1 = ib0 + 1; ib1 < boneCount; ++ib1)
        {
            RVtxWeight& vw1 = m_VtxWeights[ib1];
            if (vw1.m_Weight > vw0.m_Weight)
            {
                RSwapValue(vw0, vw1);
            }
        }
    }
    //Print(cerr, "w after ");
}

//-----------------------------------------------------------------------------
//! @brief 頂点行列のボーンウェイト値配列を調整します。
//-----------------------------------------------------------------------------
void RVtxMtx::AdjustWeight(const int maxBoneCount)
{
    //-----------------------------------------------------------------------------
    // full weight
    const int boneCountBefore = GetBoneCount();
    if (boneCountBefore == 0)
    {
        return;
    }
    else if (boneCountBefore == 1)
    {
        SetWeight(0, RVtxWeight::WEIGHT_MAX);
        return;
    }

    //-----------------------------------------------------------------------------
    // 最大ボーン数を制限
    //Print(cerr, "adj before");
    if (boneCountBefore > maxBoneCount)
    {
        SortByWeightGreater();
        RVtxWeightArray newVtxWeights;
        for (int ibone = 0; ibone < maxBoneCount; ++ibone)
        {
            newVtxWeights.push_back(m_VtxWeights[ibone]);
        }
        m_VtxWeights = newVtxWeights;
        SortByIndexLess();
    }

    //-----------------------------------------------------------------------------
    // ウェイト値の合計を WEIGHT_MAX に
    const int weightSum = GetWeightSum();
    if (weightSum != RVtxWeight::WEIGHT_MAX)
    {
        const int boneCount = GetBoneCount();
        SortByWeightGreater();
        if (weightSum > RVtxWeight::WEIGHT_MAX)
        {
            // ウェイト値の大きい順から割り当てて最後を調整
            RVtxWeightArray newVtxWeights;
            int weightTotal = 0;
            for (int ibone = 0; ibone < boneCount; ++ibone)
            {
                int weight = GetWeight(ibone);
                if (weightTotal + weight > RVtxWeight::WEIGHT_MAX)
                {
                    weight = RVtxWeight::WEIGHT_MAX - weightTotal;
                    if (weight <= 0)
                    {
                        break;
                    }
                }
                newVtxWeights.push_back(
                    RVtxWeight(GetBoneIndex(ibone), weight));
                weightTotal += weight;
            }
            m_VtxWeights = newVtxWeights;
        }
        else
        {
            // ウェイト値の小さい順から割り当てて最後を調整
            int weightTotal = 0;
            for (int ibone = boneCount - 1; ibone > 0; --ibone)
            {
                weightTotal += GetWeight(ibone);
            }
            SetWeight(0, RVtxWeight::WEIGHT_MAX - weightTotal);
        }
        SortByIndexLess();
    }
    //Print(cerr, "adj after ");
}

//-----------------------------------------------------------------------------
//! @brief ボーンに対する行列インデックスの配列からスキニング頂点属性を取得します。
//-----------------------------------------------------------------------------
#ifndef R_RELOAD_SHAPE_MTX_PAL
void RVtxMtx::GetSkinningVtxAttrMtxIdxs(
    RIVec4* idxVecs,
    RVec4* weightVecs,
    const RIntArray& boneMtxIdxs
) const
{
    weightVecs[0] = weightVecs[1] = RVec4(0.0f, 0.0f, 0.0f, 0.0f);

    int boneCount = GetBoneCount();
    boneCount = (boneCount > DEFAULT_BONE_MAX) ? DEFAULT_BONE_MAX : boneCount;

    //-----------------------------------------------------------------------------
    // 頂点行列で使用されている順にブレンドインデックスを格納します。
    int maxW = -1;
    int idxAtMaxW = 0;
    for (int iLocalBone = 0; iLocalBone < boneCount; ++iLocalBone)
    {
        const int iAttr = iLocalBone / 4;
        const int iXyzw = iLocalBone & 3;
        RIVec4& idxVec = idxVecs[iAttr];
        idxVec[iXyzw] = boneMtxIdxs[GetBoneIndex(iLocalBone)];
        const int w = GetWeight(iLocalBone);
        weightVecs[iAttr][iXyzw] = static_cast<float>(w) / RVtxWeight::WEIGHT_MAX;
        if (w > maxW)
        {
            maxW = w;
            idxAtMaxW = idxVec[iXyzw];
        }
    }

    // ウェイトが 0 の部分のインデックスを頂点で最大のウェイトを持つインデックスにします。
    for (int iLocalBone = boneCount; iLocalBone < DEFAULT_BONE_MAX; ++iLocalBone)
    {
        idxVecs[iLocalBone / 4][iLocalBone & 3] = idxAtMaxW;
    }
}
#endif

//-----------------------------------------------------------------------------
//! @brief 行列パレットからスキニング頂点属性を取得します。
//-----------------------------------------------------------------------------
#ifdef R_RELOAD_SHAPE_MTX_PAL
void RVtxMtx::GetSkinningVtxAttrMtxPal(
    RIVec4* idxVecs,
    RVec4* weightVecs,
    const RIntArray& mtxPal,
    const bool smoothSkinFlag
) const
{
    idxVecs[0] = idxVecs[1] = RIVec4(0, 0, 0, 0);
    weightVecs[0] = weightVecs[1] = RVec4(0.0f, 0.0f, 0.0f, 0.0f);

    const int mtxPalCount = static_cast<int>(mtxPal.size());
    const bool fixedIdxFlag = (smoothSkinFlag && mtxPalCount <= 4);

    int boneCount = GetBoneCount();
    boneCount = (boneCount > DEFAULT_BONE_MAX) ? DEFAULT_BONE_MAX : boneCount;
    if (fixedIdxFlag)
    {
        for (int imp = 1; imp < mtxPalCount; ++imp)
        {
            idxVecs[0][imp] = imp;
        }
        for (int iLocalBone = 0; iLocalBone < boneCount; ++iLocalBone)
        {
            const int imp = RFindValueInArray(mtxPal, GetBoneIndex(iLocalBone));
            if (imp != -1)
            {
                const float weight = static_cast<float>(GetWeight(iLocalBone)) / RVtxWeight::WEIGHT_MAX;
                weightVecs[0][imp] = weight;
            }
        }
    }
    else
    {
        for (int iLocalBone = 0; iLocalBone < boneCount; ++iLocalBone)
        {
            const int imp = RFindValueInArray(mtxPal, GetBoneIndex(iLocalBone));
            if (imp != -1)
            {
                const float weight = static_cast<float>(GetWeight(iLocalBone)) / RVtxWeight::WEIGHT_MAX;
                const int iAttr = iLocalBone / 4;
                const int iXyzw = iLocalBone & 3;
                idxVecs[iAttr][iXyzw] = imp;
                weightVecs[iAttr][iXyzw] = weight;
            }
        }
    }
}
#endif

//-----------------------------------------------------------------------------
//! @brief 頂点属性が同値であれば true を返します。
//-----------------------------------------------------------------------------
bool RPrimVtx::operator==(const RPrimVtx& rhs) const
{
    if (m_Mtx != rhs.m_Mtx)
    {
        return false;
    }

    for (int iAttr = 0; iAttr < VA_COUNT; ++iAttr)
    {
        if (m_Attr[iAttr] != rhs[iAttr])
        {
            return false;
        }
    }
    return true;
}

//-----------------------------------------------------------------------------
//! @brief 行列以外の頂点属性が同値であれば true を返します。
//-----------------------------------------------------------------------------
bool RPrimVtx::IsSameAttr(const RPrimVtx& other, const uint32_t compareFlags) const
{
    if (
        m_Attr[POS0] != other[POS0] ||
        m_Attr[NRM0] != other[NRM0] ||
        m_Attr[TAN0] != other[TAN0] ||
        m_Attr[BIN0] != other[BIN0] ||
        m_Attr[COL0] != other[COL0] ||
        m_Attr[TEX0] != other[TEX0] ||
        m_Attr[IDX0] != other[IDX0] ||
        m_Attr[IDX1] != other[IDX1] ||
        m_Attr[WGT0] != other[WGT0] ||
        m_Attr[WGT1] != other[WGT1])
    {
        return false;
    }

    if (compareFlags == 0)
    {
        return true;
    }
    else
    {
        if ((compareFlags & COMPARE_TAN1) != 0)
        {
            for (int iAttr = TAN1; iAttr <= TAN3; ++iAttr)
            {
                if (m_Attr[iAttr] != other[iAttr]) return false;
            }
            for (int iAttr = BIN1; iAttr <= BIN3; ++iAttr)
            {
                if (m_Attr[iAttr] != other[iAttr]) return false;
            }
        }
        if ((compareFlags & COMPARE_COL1) != 0)
        {
            for (int iAttr = COL1; iAttr <= COL7; ++iAttr)
            {
                if (m_Attr[iAttr] != other[iAttr]) return false;
            }
        }
        if ((compareFlags & COMPARE_TEX1) != 0)
        {
            for (int iAttr = TEX1; iAttr <= TEX7; ++iAttr)
            {
                if (m_Attr[iAttr] != other[iAttr]) return false;
            }
        }
        if ((compareFlags & COMPARE_USR0) != 0)
        {
            if (m_Attr[USR0] != other[USR0]) return false;
        }
        return true;
    }
}

//-----------------------------------------------------------------------------
//! @brief プリミティブのポリゴン数（フェース数）を取得します。
//-----------------------------------------------------------------------------
int RPrimitive::GetPolygonCount() const
{
    if (m_PrimType == POINTS ||
        m_PrimType == LINES  ||
        m_PrimType == LINE_STRIP)
    {
        return 0;
    }
    else if (m_PrimType == TRIANGLES)
    {
        return m_VtxCount / 3;
    }
    else if (m_PrimType == TRIANGLE_STRIP)
    {
        return m_VtxCount - 2;
    }
    else
    {
        return 1; // TRIANGLE_FAN & OTHER
    }
}

//-----------------------------------------------------------------------------
//! @brief プリミティブの三角形数を取得します。
//-----------------------------------------------------------------------------
int RPrimitive::GetTriangleCount() const
{
    if (m_PrimType == POINTS ||
        m_PrimType == LINES  ||
        m_PrimType == LINE_STRIP)
    {
        return 0;
    }
    else if (m_PrimType == TRIANGLES)
    {
        return m_VtxCount / 3;
    }
    else if (m_PrimType == TRIANGLE_FAN)
    {
        return m_VtxCount - 2;
    }
    else if (m_PrimType == TRIANGLE_STRIP)
    {
        return m_VtxCount - 2;
    }
    else
    {
        return 1; // OTHER
    }
}

//-----------------------------------------------------------------------------
//! @brief プリミティブモードを表す文字列を取得します（static 関数版）。
//-----------------------------------------------------------------------------
std::string RPrimitive::GetPrimitiveModeStr(const RPrimitive::PrimType type)
{
    static const char* primStrs[] =
    {
        "points",
        "lines",
        "line_strip",
        "triangles",
        "triangle_fan",
        "triangle_strip",
        "other"
    };

    return primStrs[type];
}

//-----------------------------------------------------------------------------
//! @brief プリミティブモードを表す文字列を取得します。
//-----------------------------------------------------------------------------
std::string RPrimitive::GetPrimitiveModeStr() const
{
    return GetPrimitiveModeStr(m_PrimType);
}

//-----------------------------------------------------------------------------
//! @brief プリミティブが使用するボーンインデックス配列を設定します。
//!
//! @param[in] vtxMtxs 頂点行列配列です。
//-----------------------------------------------------------------------------
void RPrimitive::SetBoneIndexes(const RVtxMtxArray& vtxMtxs)
{
    m_BoneIdxs.clear();
    for (int iVtx = 0; iVtx < m_VtxCount; ++iVtx)
    {
        const int iVmt = GetMtx(iVtx);
        if (iVmt != -1)
        {
            const RVtxMtx& vmt = vtxMtxs[iVmt];
            const int boneCount = vmt.GetBoneCount();
            for (int ibone = 0; ibone < boneCount; ++ibone)
            {
                RFindAppendValue(m_BoneIdxs, vmt.GetBoneIndex(ibone));
            }
        }
    }
}

//-----------------------------------------------------------------------------
//! @brief プリミティブの頂点属性をコピーします。
//-----------------------------------------------------------------------------
static inline void CopyRPrimVtxAttr(
    RPrimitive& rprimDst,
    const int iVtxDst,
    const RPrimitive& rprimSrc,
    const int iVtxSrc
)
{
    rprimDst.m_Vtxs[iVtxDst] = rprimSrc.m_Vtxs[iVtxSrc];
}

//-----------------------------------------------------------------------------
//! @brief 三角形の 3 つの角のうち最鋭角となる角の内積を取得します。
//!
//! @param[in] poss 多角形の頂点座標配列です。
//! @param[in] idxs 三角形の 3 頂点の poss 内インデックス配列（長さ 3）です。
//!
//! @return 最鋭角となる角の内積を返します。
//-----------------------------------------------------------------------------
static inline float GetTriangleMaxDotProduct(const RVec3Array& poss, const int* idxs)
{
    const RVec3& p0 = poss[idxs[0]];
    const RVec3& p1 = poss[idxs[1]];
    const RVec3& p2 = poss[idxs[2]];
    const RVec3 v0 = (p1 - p0).Normal();
    const RVec3 v1 = (p2 - p1).Normal();
    const RVec3 v2 = (p0 - p2).Normal();
    const float maxDot = RMax(v0 * (-v2), v1 * (-v0));
    return RMax(maxDot, v2 * (-v1));
}

//-----------------------------------------------------------------------------
//! @brief トライアングルストリップ風に三角形分割する際の頂点インデックスを取得します。
//!
//! @param[in] iTri 三角形インデックスです。
//! @param[in] iVtx 三角形内の頂点インデックスです。
//! @param[in] orgCount 多角形の頂点数です。
//!
//! @brief 多角形内の頂点インデックスを返します。
//-----------------------------------------------------------------------------
static inline int GetStripTriangulationIndex(
    const int iTri,
    const int iVtx,
    const int orgCount
)
{
    int iOrg = iTri + iVtx;
    if (iOrg >= 2)
    {
        if (((iTri & 1) == 0) || iVtx == 0)
        {
            if (iOrg & 1)
            {
                iOrg = (iOrg >> 1) + 1;
            }
            else
            {
                iOrg = orgCount - (iOrg >> 1);
            }
        }
        else
        {
            if (iOrg & 1)
            {
                iOrg = orgCount - (iOrg >> 1);
            }
            else
            {
                iOrg = (iOrg >> 1) + 1;
            }
        }
    }
    return iOrg;
}

//-----------------------------------------------------------------------------
//! @brief 最鋭角が最大になる三角形分割パターンを取得します。
//-----------------------------------------------------------------------------
int RGetAngleTriangulationPattern(const RVec3Array& poss)
{
    int bestPattern = 0;
    const int orgCount = static_cast<int>(poss.size());
    //cerr << "org count: " << orgCount << endl;
    if (orgCount >= 4)
    {
        //-----------------------------------------------------------------------------
        // 多角形の各頂点からトライアングルストリップを開始した場合について比較します。
        const int triCount = orgCount - 2;
        float bestDot = 1.0e8f;
        const int patternCount = (orgCount == 4) ? 2 : orgCount;
        for (int iPattern = 0; iPattern < patternCount; ++iPattern)
        {
            //-----------------------------------------------------------------------------
            // 最鋭角（内積が最大）となる角の内積を求めます。
            float patternDot = -1.0e8f;
            for (int iTri = 0; iTri < triCount; ++iTri)
            {
                int idxs[3];
                idxs[0] = GetStripTriangulationIndex(iTri, 0, orgCount);
                idxs[1] = GetStripTriangulationIndex(iTri, 1, orgCount);
                idxs[2] = GetStripTriangulationIndex(iTri, 2, orgCount);
                if (iPattern != 0)
                {
                    idxs[0] = (idxs[0] + iPattern) % orgCount;
                    idxs[1] = (idxs[1] + iPattern) % orgCount;
                    idxs[2] = (idxs[2] + iPattern) % orgCount;
                }
                const float triDot = GetTriangleMaxDotProduct(poss, idxs);
                if (triDot > patternDot)
                {
                    patternDot = triDot;
                }
            }
            //cerr << "pattern dot: " << iPattern << ": " << patternDot << endl;

            //-----------------------------------------------------------------------------
            // 最鋭角が最大（内積が最小）となるパターンを取得します。
            // ほとんど同じ形状の多角形が並んでいる場合に
            // 誤差が原因で分割パターンがバラバラになるのを避けるため
            // パターン 0 の内積からは微小な値を引きます。
            if (patternDot < bestDot)
            {
                const float EP = 0.00001f;
                bestPattern = iPattern;
                bestDot = patternDot - ((iPattern == 0) ? EP : 0.0f);
            }
        }
        //cerr << "angle tri pattern: " << bestPattern << ": " << bestDot << endl;
    }
    return bestPattern;
}

//-----------------------------------------------------------------------------
//! @brief プリミティブを三角形分割します。
//!
//! @param[out] dst 三角形プリミティブを格納します。
//! @param[in] src 分割前のプリミティブです。
//! @param[in] iTri 三角形インデックスです。
//! @param[in] triPattern 三角形分割パターンです。
//-----------------------------------------------------------------------------
static void TriangulateRPrim(
    RPrimitive& dst,
    const RPrimitive& src,
    const int iTri,
    const int triPattern
)
{
    dst.m_PrimType = RPrimitive::TRIANGLES;

    const int orgCount = src.m_VtxCount;
    for (int iVtx = 0; iVtx < 3; ++iVtx)
    {
        #if 1
        // トライアングルストリップ風に分割します。
        int iOrg = GetStripTriangulationIndex(iTri, iVtx, orgCount);
        #else
        // トライアングルファン風に分割します。
        // 鋭角になりやすいので現在は未使用です。
        int iOrg = (iVtx == 0) ? 0 : iTri + iVtx;
        #endif
        if (triPattern != 0)
        {
            iOrg = (iOrg + triPattern) % orgCount;
        }
        CopyRPrimVtxAttr(dst, iVtx, src, iOrg);
    }
}

//-----------------------------------------------------------------------------
//! @brief プリミティブセットの頂点ロード数を取得します。
//-----------------------------------------------------------------------------
int RPrimSet::GetVtxCount() const
{
    int vtxCount = 0;

    const_ItRPrimPtr itPrim;
    for (itPrim = m_pPrimitives.begin(); itPrim != m_pPrimitives.end(); ++itPrim)
    {
        vtxCount += (*itPrim)->m_VtxCount;
    }

    return vtxCount;
}

//-----------------------------------------------------------------------------
//! @brief プリミティブセットのポリゴン数（フェース数）を取得します。
//-----------------------------------------------------------------------------
int RPrimSet::GetPolygonCount() const
{
    int polyCount = 0;

    const_ItRPrimPtr itPrim;
    for (itPrim = m_pPrimitives.begin(); itPrim != m_pPrimitives.end(); ++itPrim)
    {
        polyCount += (*itPrim)->GetPolygonCount();
    }

    return polyCount;
}

//-----------------------------------------------------------------------------
//! @brief プリミティブセットの三角形数を取得します。
//-----------------------------------------------------------------------------
int RPrimSet::GetTriangleCount() const
{
    int triCount = 0;

    const_ItRPrimPtr itPrim;
    for (itPrim = m_pPrimitives.begin(); itPrim != m_pPrimitives.end(); ++itPrim)
    {
        triCount += (*itPrim)->GetTriangleCount();
    }

    return triCount;
}

//-----------------------------------------------------------------------------
//! @brief プリミティブセットに特定のタイプのプリミティブが存在するなら true を返します。
//-----------------------------------------------------------------------------
bool RPrimSet::PrimTypeExists(const RPrimitive::PrimType type) const
{
    const_ItRPrimPtr itPrim;
    for (itPrim = m_pPrimitives.begin(); itPrim != m_pPrimitives.end(); ++itPrim)
    {
        if ((*itPrim)->m_PrimType == type)
        {
            return true;
        }
    }
    return false;
}

//-----------------------------------------------------------------------------
//! @brief プリミティブ配列を削除します。
//-----------------------------------------------------------------------------
static void DeleteRPrimArray(RPrimPtrArray& prims)
{
    ItRPrimPtr itPrim;
    for (itPrim = prims.begin(); itPrim != prims.end(); ++itPrim)
    {
        delete *itPrim;
    }
}

//-----------------------------------------------------------------------------
//! @brief プリミティブを頂点数の多い順にソートするための比較関数です。
//-----------------------------------------------------------------------------
static bool RPrimPtrVertexCountGreater(RPrimitive*& r1, RPrimitive*& r2)
{
    if (r1->m_VtxCount == r2->m_VtxCount)
    {
        // 頂点数が同じ場合は TriangleStrip を TRIANGLES より優先します。
        return (r1->m_PrimType > r2->m_PrimType);
    }
    else
    {
        return (r1->m_VtxCount > r2->m_VtxCount);
    }
}

//-----------------------------------------------------------------------------
//! @brief プリミティブをタイプ順にソートするための比較関数です。
//-----------------------------------------------------------------------------
static bool RPrimPtrVertexPrimTypeGreater(RPrimitive*& r1, RPrimitive*& r2)
{
    if (r1->m_PrimType == r2->m_PrimType)
    {
        // タイプが同じ場合は頂点数の多いほうを優先します。
        return (r1->m_VtxCount > r2->m_VtxCount);
    }
    else
    {
        return (r1->m_PrimType > r2->m_PrimType);
    }
}

//-----------------------------------------------------------------------------
//! @brief ボーンインデックス配列の値がすべて他の配列に含まれるなら true を返します。
//!
//! @param[in] dstBoneIdxs 含む側のボーンインデックス配列です。
//! @param[in] srcBoneIdxs 含まれる側のボーンインデックス配列です。
//!
//! @return srcBoneIdxs の値がすべて dstBoneIdxs に含まれていれば true を返します。
//-----------------------------------------------------------------------------
#ifdef R_RELOAD_SHAPE_MTX_PAL
static bool IsBoneIndexesIncluded(
    const RIntArray& dstBoneIdxs,
    const RIntArray& srcBoneIdxs
)
{
    const int srcBoneCount = static_cast<int>(srcBoneIdxs.size());
    for (int ibone = 0; ibone < srcBoneCount; ++ibone)
    {
        if (RFindValueInArray(dstBoneIdxs, srcBoneIdxs[ibone]) == -1)
        {
            return false;
        }
    }
    return true;
}
#endif

//-----------------------------------------------------------------------------
//! @brief 標準シェーダの行列パレットサイズを取得します。
//!
//! @param[in] nonUniformScaleFlag XYZ のスケールが均等でないシェイプを
//!                                正しくライティングするなら true を指定します。
//! @param[in] maxReservedUniformRegs ユーザーが使用するユニフォームレジスタの数の最大値です。
//!
//! @return 行列パレットサイズを返します。
//-----------------------------------------------------------------------------
int RShape::GetStandardMtxPalCount(
    const bool nonUniformScaleFlag,
    const int maxReservedUniformRegs
)
{
    const int BLOCK_MTX_MAX = 1365; // 65536 / (3 x 4 x sizeof(float)) = 1365.33...

    // TODO: ユニフォームレジスタ方式に対応するなら修正
    R_UNUSED_VARIABLE(nonUniformScaleFlag);
    R_UNUSED_VARIABLE(maxReservedUniformRegs);

    return BLOCK_MTX_MAX;
}

//-----------------------------------------------------------------------------
//! @brief シェイプのコンストラクタです。
//-----------------------------------------------------------------------------
RShape::RShape( // RShapeCT
    const SkinningMode skinningMode,
    const int mtxPalCount,
    const char* orgName,
    RProgShape* progShapePtr)
:   m_SkinningMode(skinningMode),
    m_MtxPalCount(mtxPalCount),
    m_ProgShapePtr(progShapePtr)
{
    //-----------------------------------------------------------------------------
    // original name (for debug)
    if (orgName != NULL)
    {
        m_OrgName = orgName;
    }

    //-----------------------------------------------------------------------------
    // init attr
    for (int iAttr = 0; iAttr < RPrimVtx::VA_COUNT; ++iAttr)
    {
        m_VtxAttrFlag[iAttr] = false;
    }
}

//-----------------------------------------------------------------------------
//! @brief シェイプのメモリを解放します。
//-----------------------------------------------------------------------------
void RShape::FreeMemory()
{
    //-----------------------------------------------------------------------------
    // free primitives
    ItRPrimPtr itPrim;
    for (itPrim = m_pPrimitives.begin(); itPrim != m_pPrimitives.end(); ++itPrim)
    {
        delete *itPrim;
    }
    m_pPrimitives.clear();

    //-----------------------------------------------------------------------------
    // free prim sets
    ItRPrimSetPtr psIter;
    for (psIter = m_pPrimSets.begin(); psIter != m_pPrimSets.end(); ++psIter)
    {
        delete *psIter;
    }
    m_pPrimSets.clear();
}

//-----------------------------------------------------------------------------
//! @brief シェイプにポイント群を追加します。
//! 頂点属性がすべて等しいポイントがすでに存在する場合は追加しません。
//!
//! @param[in] rprim 追加する Points タイプのプリミティブです。
//! @param[in] vtxMtxs 頂点行列配列です。
//!
//! @return 処理結果を返します。常に成功します。
//-----------------------------------------------------------------------------
RStatus RShape::AppendPoints(const RPrimitive& rprim, const RVtxMtxArray& vtxMtxs)
{
    for (int iVtx = 0; iVtx < rprim.m_VtxCount; ++iVtx)
    {
        const RPrimVtx& vtx = rprim.m_Vtxs[iVtx];
        bool foundFlag = false;
        const_ItRPrimPtr itPrim;
        for (itPrim = m_pPrimitives.begin(); itPrim != m_pPrimitives.end(); ++itPrim)
        {
            const RPrimitive& rprimOther = **itPrim;
            // すべて Points なので最初の頂点のみ比較
            if (rprimOther.m_Vtxs[0] == vtx)
            {
                foundFlag = true;
                break;
            }
        }
        if (!foundFlag)
        {
            RPrimitive* pNewPrim = new RPrimitive(1);
            pNewPrim->m_Vtxs[0] = vtx;
            pNewPrim->SetBoneIndexes(vtxMtxs);
            m_pPrimitives.push_back(pNewPrim);
        }
    }

    return RStatus::SUCCESS;
}

//-----------------------------------------------------------------------------
//! @brief シェイプにライン群を追加します。
//!        頂点属性がすべて等しいラインがすでに存在する場合は追加しません。
//-----------------------------------------------------------------------------
RStatus RShape::AppendLines(const RPrimitive& rprim, const RVtxMtxArray& vtxMtxs)
{
    for (int iVtx0 = 0; iVtx0 < rprim.m_VtxCount; ++iVtx0)
    {
        int iVtx1 = (iVtx0 < rprim.m_VtxCount - 1) ? iVtx0 + 1 : 0;
        const RPrimVtx& vtx0 = rprim.m_Vtxs[iVtx0];
        const RPrimVtx& vtx1 = rprim.m_Vtxs[iVtx1];

        bool foundFlag = false;
        const_ItRPrimPtr itPrim;
        for (itPrim = m_pPrimitives.begin(); itPrim != m_pPrimitives.end(); ++itPrim)
        {
            const RPrimitive& rprimOther = **itPrim;
            // すべて Lines なので最初の２頂点のみ比較
            if ((rprimOther.m_Vtxs[0] == vtx0 && rprimOther.m_Vtxs[1] == vtx1) ||
                (rprimOther.m_Vtxs[0] == vtx1 && rprimOther.m_Vtxs[1] == vtx0))
            {
                foundFlag = true;
                break;
            }
        }
        if (!foundFlag)
        {
            RPrimitive* pNewPrim = new RPrimitive(2);
            pNewPrim->m_Vtxs[0] = vtx0;
            pNewPrim->m_Vtxs[1] = vtx1;
            pNewPrim->SetBoneIndexes(vtxMtxs);
            m_pPrimitives.push_back(pNewPrim);
        }
    }

    return RStatus::SUCCESS;
}

//-----------------------------------------------------------------------------
//! @brief シェイプにポリゴンを追加します。
//-----------------------------------------------------------------------------
RStatus RShape::AppendPolygon(
    const RPrimitive& rprim,
    const RVtxMtxArray& vtxMtxs,
    const bool forceTriangulate,
    const int triPattern
)
{
    //-----------------------------------------------------------------------------
    // 三角形分割する必要があるか調べます。
    bool triangulateFlag = (rprim.m_VtxCount >= 4 && forceTriangulate);
    if (!triangulateFlag && rprim.m_VtxCount >= 4)
    {
        RPrimitive testPrim = rprim;
        testPrim.SetBoneIndexes(vtxMtxs);
        triangulateFlag = (testPrim.GetBoneCount() > m_MtxPalCount);
    }

    if (triangulateFlag)
    {
        //-----------------------------------------------------------------------------
        // 三角形分割して追加します。
        for (int iTri = 0; iTri < rprim.m_VtxCount - 2; ++iTri)
        {
            RPrimitive* pNewPrim = new RPrimitive(3);
            TriangulateRPrim(*pNewPrim, rprim, iTri, triPattern);
            pNewPrim->SetBoneIndexes(vtxMtxs);
            m_pPrimitives.push_back(pNewPrim);
            if (pNewPrim->GetBoneCount() > m_MtxPalCount)
            {
                // 行列パレットのサイズが 3 未満だと三角形分割しても表示できません。
                return RStatus::FAILURE;
            }
        }
    }
    else
    {
        //-----------------------------------------------------------------------------
        // そのまま追加します。
        RPrimitive* pNewPrim = new RPrimitive();
        *pNewPrim = rprim;
        pNewPrim->SetBoneIndexes(vtxMtxs);
        m_pPrimitives.push_back(pNewPrim);
    }

    return RStatus::SUCCESS;
}

//-----------------------------------------------------------------------------
//! @brief シェイプにプリミティブセット群を設定します。
//!
//! @return 処理結果を返します。常に成功します。
//-----------------------------------------------------------------------------
RStatus RShape::SetPrimSets()
{
    //-----------------------------------------------------------------------------
    // プリミティブ数をチェックします。
    if (m_pPrimitives.empty())
    {
        return RStatus::SUCCESS;
    }

    //-----------------------------------------------------------------------------
    // 頂点数の多い順にプリミティブをソートします。
    std::sort(m_pPrimitives.begin(), m_pPrimitives.end(), RPrimPtrVertexCountGreater);

#ifndef R_RELOAD_SHAPE_MTX_PAL
    //-----------------------------------------------------------------------------
    // すべてのプリミティブを単一のプリミティブセットにまとめます。
    RPrimSet* pPrimSet = new RPrimSet();
    if (m_SkinningMode == NO_SKINNING)
    {
        // スキニングなしの場合は使用するボーンが 1 つだけなので
        // 最初のプリミティブのボーンインデックス 1 つをプリミティブセットに設定します。
        pPrimSet->m_BoneIdxs.push_back(m_pPrimitives[0]->GetBoneIndex(0));
        for (ItRPrimPtr itPrim = m_pPrimitives.begin(); itPrim != m_pPrimitives.end(); ++itPrim)
        {
            pPrimSet->m_pPrimitives.push_back(*itPrim);
        }
    }
    else
    {
        // スキニングありの場合は各プリミティブが使用するボーンインデックス群を
        // 重複しないようにプリミティブセットに追加します。
        for (ItRPrimPtr itPrim = m_pPrimitives.begin(); itPrim != m_pPrimitives.end(); ++itPrim)
        {
            RPrimitive* pPrim = *itPrim;
            const int boneCount = pPrim->GetBoneCount();
            for (int iLocalBone = 0; iLocalBone < boneCount; ++iLocalBone)
            {
                const int boneIdx = pPrim->GetBoneIndex(iLocalBone);
                if (RFindValueInArray(pPrimSet->m_BoneIdxs, boneIdx) == -1)
                {
                    pPrimSet->m_BoneIdxs.push_back(boneIdx);
                }
            }
            pPrimSet->m_pPrimitives.push_back(pPrim);
        }
    }
    m_pPrimSets.push_back(pPrimSet);
#else
    //-----------------------------------------------------------------------------
    // loop for primitive
    ItRPrimPtr itPrim;
    for (itPrim = m_pPrimitives.begin(); itPrim != m_pPrimitives.end(); ++itPrim)
    {
        //-----------------------------------------------------------------------------
        // search existing prim set
        RPrimitive* primPtr = *itPrim;
        ItRPrimSetPtr psIter;
        for (psIter = m_pPrimSets.begin(); psIter != m_pPrimSets.end(); ++psIter)
        {
            RPrimSet* psPtr = *psIter;
            if (IsBoneIndexesIncluded(psPtr->m_BoneIdxs, primPtr->m_BoneIdxs))
            {
                // プリミティブのボーンインデックス配列がすべて
                // 既存のプリミティブセットに含まれるなら
                // 既存のプリミティブセットにプリミティブを追加します。
                psPtr->m_pPrimitives.push_back(primPtr);
                break;
            }
        }
        if (psIter == m_pPrimSets.end())
        {
            //-----------------------------------------------------------------------------
            // create new prim set
            RPrimSet* psPtr = new RPrimSet();
            const int boneCount = primPtr->GetBoneCount();
            for (int ibone = 0; ibone < boneCount; ++ibone)
            {
                const int boneIdx = primPtr->GetBoneIndex(ibone);
                if (RFindValueInArray(psPtr->m_BoneIdxs, boneIdx) == -1)
                {
                    psPtr->m_BoneIdxs.push_back(boneIdx);
                }
            }
            psPtr->m_pPrimitives.push_back(primPtr);
            m_pPrimSets.push_back(psPtr);
        }
    }
#endif

    return RStatus::SUCCESS;
}

//-----------------------------------------------------------------------------
//! @brief プリミティブセットをボーン数の大きい順にソートします。
//-----------------------------------------------------------------------------
#ifdef R_RELOAD_SHAPE_MTX_PAL
static bool PrimSetPtrBoneCountGreater(RPrimSet*& r1, RPrimSet*& r2)
{
    return (r1->GetBoneCount() > r2->GetBoneCount());
}

static void SortPrimSetsByBoneCountGreater(RPrimSetPtrArray& primSets)
{
    std::sort(primSets.begin(), primSets.end(), PrimSetPtrBoneCountGreater);
}
#endif

//-----------------------------------------------------------------------------
//! @brief プリミティブセットを結合した場合の最適化の評価値を取得します。
//-----------------------------------------------------------------------------
#ifdef R_RELOAD_SHAPE_MTX_PAL
static int GetPrimSetCombineScore(
    const RPrimSet& ps0,
    const RPrimSet& ps1,
    const int mtxPalCount
)
{
    const RIntArray& boneIdxs0 = ps0.m_BoneIdxs;
    const RIntArray& boneIdxs1 = ps1.m_BoneIdxs;
    const int boneCount0 = static_cast<int>(boneIdxs0.size());
    const int boneCount1 = static_cast<int>(boneIdxs1.size());

    int commonBoneCount = 0;
    int totalBoneCount = boneCount0;
    for (int ibone1 = 0; ibone1 < boneCount1; ++ibone1)
    {
        if (RFindValueInArray(boneIdxs0, boneIdxs1[ibone1]) == -1)
        {
            ++totalBoneCount;
        }
        else
        {
            ++commonBoneCount;
        }
    }
    if (totalBoneCount > mtxPalCount)
    {
        return -1;
    }

    const int appendBoneCount = totalBoneCount - commonBoneCount;
    const int boneCountMax = (boneCount0 > boneCount1) ? boneCount0 : boneCount1;
    return ((mtxPalCount + 1 - appendBoneCount) << 20) +
        (commonBoneCount << 10) +
        boneCountMax;
}
#endif

//-----------------------------------------------------------------------------
//! @brief 2 つのプリミティブセットを結合します。
//-----------------------------------------------------------------------------
static void CombineTwoPrimSets(RPrimSet& psDst, const RPrimSet& psSrc)
{
    //-----------------------------------------------------------------------------
    // merge bone indexes
    const int srcBoneCount = static_cast<int>(psSrc.m_BoneIdxs.size());
    for (int ibone = 0; ibone < srcBoneCount; ++ibone)
    {
        RFindAppendValue(psDst.m_BoneIdxs, psSrc.m_BoneIdxs[ibone]);
    }

    //-----------------------------------------------------------------------------
    // merge primitives
    const int srcPrimCount = static_cast<int>(psSrc.m_pPrimitives.size());
    for (int iPrim = 0; iPrim < srcPrimCount; ++iPrim)
    {
        psDst.m_pPrimitives.push_back(psSrc.m_pPrimitives[iPrim]);
    }
}

//-----------------------------------------------------------------------------
//! @brief プリミティブセット群をすべて結合して 1 つにします。
//!
//! @param[in,out] primSets プリミティブセット群です。
//-----------------------------------------------------------------------------
#ifndef R_RELOAD_SHAPE_MTX_PAL
static void CombineAllPrimSets(RPrimSetPtrArray& primSets)
{
    while (static_cast<int>(primSets.size()) >= 2)
    {
        CombineTwoPrimSets(*primSets[0], *primSets[1]);
        delete primSets[1];
        primSets.erase(primSets.begin() + 1);
    }
}
#endif

//-----------------------------------------------------------------------------
//! @brief プリミティブセット群の中で 2 つのプリミティブセットを結合します。
//! 優先度１）できるだけ追加されるボーンインデックス数を少なく
//! 優先度２）共通のボーンインデックスの多いものから
//! 優先度３）ボーンインデックス数の多いものから
//!
//! @param[in,out] primSets プリミティブセット群です。
//! @param[in] mtxPalCount 行列パレットサイズです。
//!
//! @return 結合した場合は true を返します。
//-----------------------------------------------------------------------------
#ifdef R_RELOAD_SHAPE_MTX_PAL
static bool CombinePrimSets(RPrimSetPtrArray& primSets, const int mtxPalCount)
{
    //-----------------------------------------------------------------------------
    // find best combination
    int iPsBest0 = 0, iPsBest1 = 0;
    int scoreBest = -1;
    const int primSetCount = static_cast<int>(primSets.size());
    for (int iPs0 = 0; iPs0 < primSetCount - 1; ++iPs0)
    {
        for (int iPs1 = iPs0 + 1; iPs1 < primSetCount; ++iPs1)
        {
            const int score = GetPrimSetCombineScore(
                *primSets[iPs0], *primSets[iPs1], mtxPalCount);
            if (score > scoreBest)
            {
                scoreBest = score;
                iPsBest0 = iPs0;
                iPsBest1 = iPs1;
            }
        }
    }

    //-----------------------------------------------------------------------------
    // combine
    if (scoreBest > 0)
    {
        //cerr << "combine prim sets: " << iPsBest0 << " " << iPsBest1 << endl;
        CombineTwoPrimSets(*primSets[iPsBest0], *primSets[iPsBest1]);
        delete primSets[iPsBest1];
        primSets.erase(primSets.begin() + iPsBest1);
        return true;
    }
    else
    {
        return false;
    }
}
#endif

//-----------------------------------------------------------------------------
//! @brief プリミティブセット群の行列ロード回数を取得します。
//!
//! @param[in] primSets プリミティブセット群です。
//! @param[in] mtxPalCount 行列パレットサイズです。
//!
//! @return 行列ロード回数を返します。
//-----------------------------------------------------------------------------
static int CountPrimSetsMtxLoad(const RPrimSetPtrArray& primSets, const int mtxPalCount)
{
    #ifndef R_RELOAD_SHAPE_MTX_PAL
    R_UNUSED_VARIABLE(primSets);
    R_UNUSED_VARIABLE(mtxPalCount);
    return 1;
    #else
    RIntArray mtxPal(mtxPalCount, -1);
    int loadCount = 0;
    const int primSetCount = static_cast<int>(primSets.size());
    for (int iPs = 0; iPs < primSetCount; ++iPs)
    {
        const RPrimSet& ps = *primSets[iPs];
        const int boneCount = ps.GetBoneCount();
        if (iPs == 0)
        {
            for (int ibone = 0; ibone < boneCount; ++ibone)
            {
                mtxPal[ibone] = ps.GetBoneIndex(ibone);
            }
            loadCount += boneCount;
        }
        else
        {
            RIntArray mtxFlags(mtxPalCount, MTX_LOAD_NONE);
            for (int ibone = 0; ibone < boneCount; ++ibone)
            {
                const int imp = RFindValueInArray(mtxPal, ps.GetBoneIndex(ibone));
                if (imp != -1)
                {
                    mtxFlags[imp] = MTX_LOAD_NOLOAD;
                }
            }
            for (int ibone = 0; ibone < boneCount; ++ibone)
            {
                const int boneIdx = ps.GetBoneIndex(ibone);
                const int imp = RFindValueInArray(mtxPal, boneIdx);
                if (imp == -1)
                {
                    const int impEmpty = RFindValueInArray(mtxFlags, static_cast<int>(MTX_LOAD_NONE));
                    if (impEmpty != -1)
                    {
                        mtxPal[impEmpty] = boneIdx;
                        mtxFlags[impEmpty] = MTX_LOAD_LOAD;
                        ++loadCount;
                    }
                }
            }
        }
    }
    return loadCount;
    #endif
}

//-----------------------------------------------------------------------------
//! @brief プリミティブセット群内の順番を交換します。
//!
//! @param[in,out] primSets プリミティブセット群です。
//! @param[in] iPs0 交換するプリミティブセットのインデックスです。
//! @param[in] iPs1 交換するプリミティブセットのインデックスです。
//-----------------------------------------------------------------------------
#ifdef R_RELOAD_SHAPE_MTX_PAL
static inline void SwapPrimSetData(
    RPrimSetPtrArray& primSets,
    const int iPs0,
    const int iPs1
)
{
    RPrimSet* psPtr0 = primSets[iPs0];
    primSets[iPs0] = primSets[iPs1];
    primSets[iPs1] = psPtr0;
}
#endif

//-----------------------------------------------------------------------------
//! @brief プリミティブセットを行列ロード回数が最小になるようにソートします。
//!
//! @param[in,out] primSets プリミティブセット群です。
//! @param[in] mtxPalCount 行列パレットサイズです。
//-----------------------------------------------------------------------------
#ifdef R_RELOAD_SHAPE_MTX_PAL
static void SortPrimSetsForReduceMtxLoad(RPrimSetPtrArray& primSets, const int mtxPalCount)
{
    const int primSetCount = static_cast<int>(primSets.size());
    for (int iPs0 = 0; iPs0 < primSetCount - 1; ++iPs0)
    {
        for (int iPs1 = iPs0 + 1; iPs1 < primSetCount; ++iPs1)
        {
            const int before = CountPrimSetsMtxLoad(primSets, mtxPalCount);
            SwapPrimSetData(primSets, iPs0, iPs1);
            if (CountPrimSetsMtxLoad(primSets, mtxPalCount) >= before)
            {
                SwapPrimSetData(primSets, iPs0, iPs1);
            }
        }
    }
}
#endif

//-----------------------------------------------------------------------------
//! @brief プリミティブセット内のプリミティブを頂点数が大きい順にソートします。
//!
//! @param[in,out] primSets プリミティブセット群です。
//-----------------------------------------------------------------------------
static void SortPrimitivesInSetsByVtxCountGreater(RPrimSetPtrArray& primSets)
{
    ItRPrimSetPtr psIter;
    for (psIter = primSets.begin(); psIter != primSets.end(); ++psIter)
    {
        RPrimPtrArray& rprims = (*psIter)->m_pPrimitives;
        std::sort(rprims.begin(), rprims.end(), RPrimPtrVertexCountGreater);
    }
}

//-----------------------------------------------------------------------------
//! @brief シェイプのプリミティブセット群を最適化します。
//-----------------------------------------------------------------------------
void RShape::OptimizePrimSets()
{
    // 進行状況を表示します。
    if (m_ProgShapePtr != NULL)
    {
        m_ProgShapePtr->StartOptimizePrimSets();
    }

    #ifndef R_RELOAD_SHAPE_MTX_PAL
        // プリミティブセットをすべて結合して 1 つにします。
        CombineAllPrimSets(m_pPrimSets);
    #else
        // プリミティブセットを行列パレットに収まる範囲でできるだけ結合します。
        for (;;)
        {
            SortPrimSetsByBoneCountGreater(m_pPrimSets);
            if (!CombinePrimSets(m_pPrimSets, m_MtxPalCount))
            {
                break;
            }
        }

        // プリミティブセットを行列ロード回数が最小になるようにソートします。
        SortPrimSetsForReduceMtxLoad(m_pPrimSets, m_MtxPalCount);
    #endif

    // プリミティブセット内のプリミティブを頂点数が大きい順にソートします。
    SortPrimitivesInSetsByVtxCountGreater(m_pPrimSets);
}

//-----------------------------------------------------------------------------
//! @brief プリミティブに他のプリミティブの頂点を 1 つ追加します。
//-----------------------------------------------------------------------------
static void AppendOneVertex(RPrimitive& dst, const RPrimitive& src, const int iVtx)
{
    ++dst.m_VtxCount;
    dst.m_Vtxs.push_back(src.m_Vtxs[iVtx]);
}

//-----------------------------------------------------------------------------
//! @brief プリミティブ群をタイプごとに単一のプリミティブにマージします。
//-----------------------------------------------------------------------------
static void MergeSingleRPrim(RPrimPtrArray& rprimsDst, const RPrimPtrArray& rprimsSrc)
{
    //-----------------------------------------------------------------------------
    // create rprim for triangles & lines & points
    RPrimitive* primPtrTris = new RPrimitive();
    primPtrTris->m_PrimType = RPrimitive::TRIANGLES;
    RPrimitive* primPtrLines = new RPrimitive();
    primPtrLines->m_PrimType = RPrimitive::LINES;
    RPrimitive* primPtrPoints = new RPrimitive();
    primPtrPoints->m_PrimType = RPrimitive::POINTS;

    //-----------------------------------------------------------------------------
    // loop for rprim
    int rprimsSrcCount = static_cast<int>(rprimsSrc.size());
    for (int irprim = 0; irprim < rprimsSrcCount; ++irprim)
    {
        const RPrimitive& rprim = *rprimsSrc[irprim];
        if (rprim.m_PrimType == RPrimitive::TRIANGLES)
        {
            //-----------------------------------------------------------------------------
            // 三角形を集める
            if (primPtrTris->m_VtxCount + rprim.m_VtxCount > RPrimitive::VTX_SIZE_MAX)
            {
                rprimsDst.push_back(primPtrTris);
                primPtrTris = new RPrimitive();
                primPtrTris->m_PrimType = RPrimitive::TRIANGLES;
            }
            for (int iVtx = 0; iVtx < rprim.m_VtxCount; ++iVtx)
            {
                AppendOneVertex(*primPtrTris, rprim, iVtx);
            }
        }
        else if (rprim.m_PrimType == RPrimitive::LINES)
        {
            //-----------------------------------------------------------------------------
            // ラインを集める
            if (primPtrLines->m_VtxCount + rprim.m_VtxCount > RPrimitive::VTX_SIZE_MAX)
            {
                rprimsDst.push_back(primPtrLines);
                primPtrLines = new RPrimitive();
                primPtrLines->m_PrimType = RPrimitive::LINES;
            }
            for (int iVtx = 0; iVtx < rprim.m_VtxCount; ++iVtx)
            {
                AppendOneVertex(*primPtrLines, rprim, iVtx);
            }
        }
        else if (rprim.m_PrimType == RPrimitive::POINTS)
        {
            //-----------------------------------------------------------------------------
            // 点を集める
            if (primPtrPoints->m_VtxCount + rprim.m_VtxCount > RPrimitive::VTX_SIZE_MAX)
            {
                rprimsDst.push_back(primPtrPoints);
                primPtrPoints = new RPrimitive();
                primPtrPoints->m_PrimType = RPrimitive::POINTS;
            }
            for (int iVtx = 0; iVtx < rprim.m_VtxCount; ++iVtx)
            {
                AppendOneVertex(*primPtrPoints, rprim, iVtx);
            }
        }
        else
        {
            //-----------------------------------------------------------------------------
            // triangle_strip / triangle_fan / line_strip / other はそのままコピー
            RPrimitive* pNewPrim = new RPrimitive();
            *pNewPrim = rprim;
            rprimsDst.push_back(pNewPrim);
        }
    }

    //-----------------------------------------------------------------------------
    // append rprim for triangles to dst prims
    if (primPtrTris->m_VtxCount > 0)
    {
        rprimsDst.push_back(primPtrTris);
    }
    else
    {
        delete primPtrTris;
    }

    //-----------------------------------------------------------------------------
    // append rprim for lines to dst prims
    if (primPtrLines->m_VtxCount > 0)
    {
        rprimsDst.push_back(primPtrLines);
    }
    else
    {
        delete primPtrLines;
    }

    //-----------------------------------------------------------------------------
    // append rprim for points to dst prims
    if (primPtrPoints->m_VtxCount > 0)
    {
        rprimsDst.push_back(primPtrPoints);
    }
    else
    {
        delete primPtrPoints;
    }

    //-----------------------------------------------------------------------------
    // m_PrimType でソート
    std::sort(rprimsDst.begin(), rprimsDst.end(), RPrimPtrVertexPrimTypeGreater);
}

//-----------------------------------------------------------------------------
//! @brief プリミティブセット群内のプリミティブ群をタイプごとに単一のプリミティブにマージします。
//!
//! @param[in,out] primSets プリミティブセット群です。
//! @param[in,out] prims プリミティブ群です。
//-----------------------------------------------------------------------------
static void MergeSinglePrimitivesInSets(
    RPrimSetPtrArray& primSets,
    RPrimPtrArray& prims
)
{
    //-----------------------------------------------------------------------------
    // loop for prim set
    RPrimPtrArray primsNew;
    const int primSetCount = static_cast<int>(primSets.size());
    for (int iPs = 0; iPs < primSetCount; ++iPs)
    {
        RPrimSet& ps = *primSets[iPs];
        RPrimPtrArray rprimsDst;
        MergeSingleRPrim(rprimsDst, ps.m_pPrimitives);
        ps.m_pPrimitives = rprimsDst;
        RAppendArrayToArray(primsNew, rprimsDst);
    }

    //-----------------------------------------------------------------------------
    // delete old rprim array & copy new
    DeleteRPrimArray(prims);
    prims = primsNew;
}

//-----------------------------------------------------------------------------
//! @brief シェイプのプリミティブセット群の出力用ボーンインデックス配列を設定します。
//-----------------------------------------------------------------------------
void RShape::SetOutBoneIndexes()
{
    //-----------------------------------------------------------------------------
    // 各プリミティブセットの出力用ボーンインデックス配列を設定します。
    #ifndef R_RELOAD_SHAPE_MTX_PAL
    const int primSetCount = static_cast<int>(m_pPrimSets.size());
    for (int iPs = 0; iPs < primSetCount; ++iPs)
    {
        // 行列パレットを使用しない場合はそのままコピーして、
        // ボーンインデックスの小さい順にソートします。
        RPrimSet& ps = *m_pPrimSets[iPs];
        ps.m_OutBoneIdxs = ps.m_BoneIdxs;
        std::sort(ps.m_OutBoneIdxs.begin(), ps.m_OutBoneIdxs.end());
    }
    #else
    RIntArray mtxPal(m_MtxPalCount, -1);
    const int primSetCount = static_cast<int>(m_pPrimSets.size());
    for (int iPs = 0; iPs < primSetCount; ++iPs)
    {
        //-----------------------------------------------------------------------------
        // set matrix memory
        RPrimSet& ps = *m_pPrimSets[iPs];
        const int boneCount = ps.GetBoneCount();
        if (iPs == 0)
        {
            for (int ibone = 0; ibone < boneCount; ++ibone)
            {
                mtxPal[ibone] = ps.GetBoneIndex(ibone);
            }
        }
        else
        {
            RIntArray mtxFlags(m_MtxPalCount, MTX_LOAD_NONE);
            for (int ibone = 0; ibone < boneCount; ++ibone)
            {
                const int imp = RFindValueInArray(mtxPal, ps.GetBoneIndex(ibone));
                if (imp != -1)
                {
                    mtxFlags[imp] = MTX_LOAD_NOLOAD;
                }
            }
            for (int ibone = 0; ibone < boneCount; ++ibone)
            {
                const int boneIdx = ps.GetBoneIndex(ibone);
                const int imp = RFindValueInArray(mtxPal, boneIdx);
                if (imp == -1)
                {
                    const int impEmpty = RFindValueInArray(mtxFlags, static_cast<int>(MTX_LOAD_NONE));
                    if (impEmpty != -1)
                    {
                        mtxPal[impEmpty] = boneIdx;
                        mtxFlags[impEmpty] = MTX_LOAD_LOAD;
                    }
                }
            }
        }

        //-----------------------------------------------------------------------------
        // set bone indexes
        ps.m_OutBoneIdxs.clear();
        for (int imp = 0; imp < m_MtxPalCount; ++imp)
        {
            const int boneIdx = mtxPal[imp];
            if (boneIdx == -1)
            {
                break;
            }
            ps.m_OutBoneIdxs.push_back(boneIdx);
        }
    }
    #endif

    //-----------------------------------------------------------------------------
    // 全プリミティブ群での使用ボーン数の最大値を取得します。
    m_OutBoneCountMax = 1;
    for (int iPs = 0; iPs < primSetCount; ++iPs)
    {
        const RPrimSet& ps = *m_pPrimSets[iPs];
        const int boneCount = ps.GetOutBoneCount();
        if (boneCount > m_OutBoneCountMax)
        {
            m_OutBoneCountMax = boneCount;
        }
    }
}

//-----------------------------------------------------------------------------
//! @brief シェイプのスキニング頂点属性を設定します。
//-----------------------------------------------------------------------------
void RShape::SetSkinningVtxAttr(const RVtxMtxArray& vtxMtxs, const RIntArray& boneMtxIdxs)
{
    if (m_VtxAttrFlag[RPrimVtx::IDX0])
    {
        const bool smoothSkinFlag = m_VtxAttrFlag[RPrimVtx::WGT0];

        // loop for prim sets
        const int primSetCount = static_cast<int>(m_pPrimSets.size());
        for (int iPs = 0; iPs < primSetCount; ++iPs)
        {
            const RPrimSet& ps = *m_pPrimSets[iPs];

            // loop for prims
            const int primCount = static_cast<int>(ps.m_pPrimitives.size());
            for (int iPrim = 0; iPrim < primCount; ++iPrim)
            {
                RPrimitive& prim = *ps.m_pPrimitives[iPrim];

                // loop for vertices
                const int vtxCount = static_cast<int>(prim.m_Vtxs.size());
                for (int iVtx = 0; iVtx < vtxCount; ++iVtx)
                {
                    RPrimVtx& vtx = prim.m_Vtxs[iVtx];
                    const RVtxMtx& vmt = vtxMtxs[vtx.m_Mtx];
                    RIVec4   idxVecs[RVtxMtx::DEFAULT_BONE_MAX / 4];
                    RVec4 weightVecs[RVtxMtx::DEFAULT_BONE_MAX / 4];
                    #ifndef R_RELOAD_SHAPE_MTX_PAL
                    vmt.GetSkinningVtxAttrMtxIdxs(idxVecs, weightVecs, boneMtxIdxs);
                    #else
                    vmt.GetSkinningVtxAttrMtxPal(idxVecs, weightVecs, ps.m_OutBoneIdxs, smoothSkinFlag);
                    R_UNUSED_VARIABLE(boneMtxIdxs);
                    #endif
                    vtx[RPrimVtx::IDX0] = RFindAppendValue(m_VtxBlendIdxs[0], idxVecs[0]);
                    if (smoothSkinFlag)
                    {
                        vtx[RPrimVtx::WGT0] = RFindAppendValue(m_VtxBlendWgts[0], weightVecs[0]);
                        if (m_OutBoneCountMax > 4)
                        {
                            vtx[RPrimVtx::IDX1] = RFindAppendValue(m_VtxBlendIdxs[1], idxVecs[1]);
                            vtx[RPrimVtx::WGT1] = RFindAppendValue(m_VtxBlendWgts[1], weightVecs[1]);
                        }
                    }
                }
            }
        }
    }
}

//-----------------------------------------------------------------------------
//! @brief シェイプの出力用データを設定して最適化します。
//!
//! @return 処理結果を返します。常に成功します。
//-----------------------------------------------------------------------------
RStatus RShape::SetAndOptimize(const RVtxMtxArray& vtxMtxs, const RIntArray& boneMtxIdxs)
{
    RStatus status;

    //-----------------------------------------------------------------------------
    // set prim sets
    status = SetPrimSets();
    if (!status)
    {
        return status;
    }

    //-----------------------------------------------------------------------------
    // optimize prim sets
    m_OrgPrimSetCount = static_cast<int>(m_pPrimSets.size());
    m_OrgMtxLoadCount = CountPrimSetsMtxLoad(m_pPrimSets, m_MtxPalCount);

    if (m_OrgPrimSetCount >= 2)
    {
        OptimizePrimSets();
    }

    m_OutPrimSetCount = static_cast<int>(m_pPrimSets.size());
    m_OutMtxLoadCount = CountPrimSetsMtxLoad(m_pPrimSets, m_MtxPalCount);
    //cerr << "shape mtx load: " << m_OrgMtxLoadCount << " -> " << m_OutMtxLoadCount << endl;

    //-----------------------------------------------------------------------------
    // merge single primitives in prim set
    MergeSinglePrimitivesInSets(m_pPrimSets, m_pPrimitives);

    //-----------------------------------------------------------------------------
    // set out bone indexes
    SetOutBoneIndexes();

    //-----------------------------------------------------------------------------
    // set skinning vtx attr
    SetSkinningVtxAttr(vtxMtxs, boneMtxIdxs);

    return RStatus::SUCCESS;
}

//-----------------------------------------------------------------------------
//! @brief シェイプの頂点ロード数を取得します。
//-----------------------------------------------------------------------------
int RShape::GetVtxCount() const
{
    int vtxCount = 0;
    const_ItRPrimSetPtr psIter;
    for (psIter = m_pPrimSets.begin(); psIter != m_pPrimSets.end(); ++psIter)
    {
        const RPrimSet& ps = **psIter;
        vtxCount += ps.GetVtxCount();
    }
    return vtxCount;
}

//-----------------------------------------------------------------------------
//! @brief シェイプのポリゴン数（フェース数）を取得します。
//-----------------------------------------------------------------------------
int RShape::GetPolygonCount() const
{
    int polyCount = 0;
    const_ItRPrimSetPtr psIter;
    for (psIter = m_pPrimSets.begin(); psIter != m_pPrimSets.end(); ++psIter)
    {
        const RPrimSet& ps = **psIter;
        polyCount += ps.GetPolygonCount();
    }
    return polyCount;
}

//-----------------------------------------------------------------------------
//! @brief シェイプの三角形数を取得します。
//-----------------------------------------------------------------------------
int RShape::GetTriangleCount() const
{
    int triCount = 0;
    const_ItRPrimSetPtr psIter;
    for (psIter = m_pPrimSets.begin(); psIter != m_pPrimSets.end(); ++psIter)
    {
        const RPrimSet& ps = **psIter;
        triCount += ps.GetTriangleCount();
    }
    return triCount;
}

//-----------------------------------------------------------------------------
//! @brief シェイプのプリミティブ数を取得します。
//-----------------------------------------------------------------------------
int RShape::GetPrimCount() const
{
    int primCount = 0;
    const_ItRPrimSetPtr psIter;
    for (psIter = m_pPrimSets.begin(); psIter != m_pPrimSets.end(); ++psIter)
    {
        const RPrimSet& ps = **psIter;
        primCount += static_cast<int>(ps.m_pPrimitives.size());
    }
    return primCount;
}

//-----------------------------------------------------------------------------
//! @brief シェイプに特定のタイプのプリミティブが存在するなら true を返します。
//-----------------------------------------------------------------------------
bool RShape::PrimTypeExists(const RPrimitive::PrimType type) const
{
    const_ItRPrimSetPtr psIter;
    for (psIter = m_pPrimSets.begin(); psIter != m_pPrimSets.end(); ++psIter)
    {
        const RPrimSet& ps = **psIter;
        if (ps.PrimTypeExists(type))
        {
            return true;
        }
    }
    return false;
}

//-----------------------------------------------------------------------------
//! @brief 頂点データ列を 1 つ追加します。
//!        テンプレート関数です。
//!
//! @param[in,out] dataStreams データ列配列です。
//! @param[in] array 頂点属性のデータです。
//! @param[in] vtxInfo シェイプの頂点属性出力情報です。
//!
//! @return 頂点属性のデータ数を返します。
//-----------------------------------------------------------------------------
template <typename T>
static int AppendOneVtxStream(
    RDataStreamArray& dataStreams,
    const std::vector<T>& array,
    const ROutShapeVtxInfo& vtxInfo
)
{
    //-----------------------------------------------------------------------------
    // check empty
    const int vecCount = static_cast<int>(array.size());
    if (vecCount == 0)
    {
        return 0;
    }

    //-----------------------------------------------------------------------------
    // quantize
    std::vector<T> outArray = array; // TODO: 出力時に量子化するなら変更

    //-----------------------------------------------------------------------------
    // out constant
    // TODO: 中間ファイルが固定頂点属性に対応したら実装
    //if (RIsConstantArray(outArray)) // 量子化後の値で判定
    //{
    //  scale = 1.0f;
    //  const T& vec = array[0]; // 量子化前の float 値を出力
    //  os << RTab(tc) << "<" << "VertexAttributeCtr Usage=\"" << ""
    //     << "\" Scale=\"" << scale << "\">";
    //  os << vec;
    //  os << "</" << "VertexAttributeCtr>" << R_ENDL;
    //  return 1;
    //}

    //-----------------------------------------------------------------------------
    // out stream
    dataStreams.push_back(RDataStream(array, vtxInfo.m_CompCount));

    return vecCount;
}

//-----------------------------------------------------------------------------
//! @brief 頂点データ列を追加して頂点データを設定します。
//! テンプレート関数です。
//!
//! @param[in,out] vtxData 頂点データです。
//! @param[in,out] dataStreams データ列配列です。
//! @param[in] array データ値の配列です。
//! @param[in] vtxInfo シェイプの頂点属性出力情報です。
//! @param[in] hint 頂点属性ヒント情報です。
//! @param[in] hintIndex 頂点属性ヒント情報のインデックスです。
//! @param[in] pNameOverride 頂点属性名のオーバーライドです。NULL ならヒント情報から自動決定します。
//-----------------------------------------------------------------------------
template <typename T>
static void AppendVtxStream(
    RVertex& vtxData,
    RDataStreamArray& dataStreams,
    const std::vector<T>& array,
    const ROutShapeVtxInfo& vtxInfo,
    const RVtxAttrib::Hint hint,
    const int hintIndex,
    const char* pNameOverride
)
{
    const int streamIdx = static_cast<int>(dataStreams.size());
    const int dataCount = AppendOneVtxStream(dataStreams, array, vtxInfo);

    const int attribIdx = static_cast<int>(vtxData.m_VtxAttribs.size());
    vtxData.m_VtxBuffers[0].m_Inputs.push_back(RVtxInput(attribIdx));

    vtxData.m_VtxAttribs.push_back(
        RVtxAttrib(hint, hintIndex, GetVtxAttribType(vtxInfo),
            RVtxAttrib::NONE, dataCount, streamIdx));
    if (pNameOverride != NULL)
    {
        vtxData.m_VtxAttribs.back().m_NameOverride = pNameOverride;
    }
}

//-----------------------------------------------------------------------------
//! @brief int 型の頂点データ列を追加して頂点データを設定します。
//!        頂点属性出力情報に応じてデータ値の配列の型を加工してから追加します。
//!
//! @param[in,out] vtxData 頂点データです。
//! @param[in,out] dataStreams データ列配列です。
//! @param[in] array データ値の配列です。
//! @param[in] vtxInfo シェイプの頂点属性出力情報です。
//! @param[in] hint 頂点属性ヒント情報です。
//! @param[in] hintIndex 頂点属性ヒント情報のインデックスです。
//! @param[in] pNameOverride 頂点属性名のオーバーライドです。NULL ならヒント情報から自動決定します。
//-----------------------------------------------------------------------------
static void AppendVec4VtxStream(
    RVertex& vtxData,
    RDataStreamArray& dataStreams,
    const RIVec4Array& array,
    const ROutShapeVtxInfo& vtxInfo,
    const RVtxAttrib::Hint hint,
    const int hintIndex,
    const char* pNameOverride
)
{
    // 成分数に応じて各 int 型配列を作成します。
    const int count = static_cast<int>(array.size());
    if (vtxInfo.m_CompCount == 1)
    {
        RIntArray vec1s;
        for (int index = 0; index < count; ++index)
        {
            vec1s.push_back(array[index].x);
        }
        AppendVtxStream(vtxData, dataStreams, vec1s, vtxInfo, hint, hintIndex, pNameOverride);
    }
    else if (vtxInfo.m_CompCount == 2)
    {
        RIVec2Array vec2s;
        for (int index = 0; index < count; ++index)
        {
            vec2s.push_back(RIVec2(array[index]));
        }
        AppendVtxStream(vtxData, dataStreams, vec2s, vtxInfo, hint, hintIndex, pNameOverride);
    }
    else if (vtxInfo.m_CompCount == 3)
    {
        RIVec3Array vec3s;
        for (int index = 0; index < count; ++index)
        {
            vec3s.push_back(RIVec3(array[index]));
        }
        AppendVtxStream(vtxData, dataStreams, vec3s, vtxInfo, hint, hintIndex, pNameOverride);
    }
    else // 4
    {
        AppendVtxStream(vtxData, dataStreams, array, vtxInfo, hint, hintIndex, pNameOverride);
    }
}

//-----------------------------------------------------------------------------
//! @brief float 型の頂点データ列を追加して頂点データを設定します。
//!        頂点属性出力情報に応じてデータ値の配列の型を加工してから追加します。
//!
//! @param[in,out] vtxData 頂点データです。
//! @param[in,out] dataStreams データ列配列です。
//! @param[in] array データ値の配列です。
//! @param[in] vtxInfo シェイプの頂点属性出力情報です。
//! @param[in] hint 頂点属性ヒント情報です。
//! @param[in] hintIndex 頂点属性ヒント情報のインデックスです。
//! @param[in] pNameOverride 頂点属性名のオーバーライドです。NULL ならヒント情報から自動決定します。
//-----------------------------------------------------------------------------
static void AppendVec4VtxStream(
    RVertex& vtxData,
    RDataStreamArray& dataStreams,
    const RVec4Array& array,
    const ROutShapeVtxInfo& vtxInfo,
    const RVtxAttrib::Hint hint,
    const int hintIndex,
    const char* pNameOverride
)
{
    const int count = static_cast<int>(array.size());

    //-----------------------------------------------------------------------------
    // 出力する値の型が int、uint なら int 型配列に変換して int 版の同名関数を呼びます。
    // 現状、頂点カラーのみこの条件に一致する場合があります。
    if (vtxInfo.m_ValueType == RPrimVtx::VALUE_INT  ||
        vtxInfo.m_ValueType == RPrimVtx::VALUE_UINT)
    {
        RIVec4Array ivecs;
        for (int index = 0; index < count; ++index)
        {
            const RVec4& fvec = array[index];
            const int x = RRound(fvec.x);
            const int y = RRound(fvec.y);
            const int z = RRound(fvec.z);
            const int w = RRound(fvec.w);
            ivecs.push_back(RIVec4(x, y, z, w));
        }
        AppendVec4VtxStream(vtxData, dataStreams, ivecs, vtxInfo, hint, hintIndex, pNameOverride);
        return;
    }

    //-----------------------------------------------------------------------------
    // 成分数に応じて各 float 型配列を作成します。
    if (vtxInfo.m_CompCount == 1)
    {
        RFloatArray vec1s;
        for (int index = 0; index < count; ++index)
        {
            vec1s.push_back(array[index].x);
        }
        AppendVtxStream(vtxData, dataStreams, vec1s, vtxInfo, hint, hintIndex, pNameOverride);
    }
    else if (vtxInfo.m_CompCount == 2)
    {
        RVec2Array vec2s;
        for (int index = 0; index < count; ++index)
        {
            vec2s.push_back(RVec2(array[index]));
        }
        AppendVtxStream(vtxData, dataStreams, vec2s, vtxInfo, hint, hintIndex, pNameOverride);
    }
    else if (vtxInfo.m_CompCount == 3)
    {
        RVec3Array vec3s;
        for (int index = 0; index < count; ++index)
        {
            vec3s.push_back(RVec3(array[index]));
        }
        AppendVtxStream(vtxData, dataStreams, vec3s, vtxInfo, hint, hintIndex, pNameOverride);
    }
    else // 4
    {
        AppendVtxStream(vtxData, dataStreams, array, vtxInfo, hint, hintIndex, pNameOverride);
    }
}

//-----------------------------------------------------------------------------
//! @brief スキニング用の頂点属性の成分数を取得します。
//!
//! @param[in] outBoneCountMax 全プリミティブ群での使用ボーン数の最大値です。
//! @param[in] outBlendWgts ブレンドウェイト値配列です。
//!
//! @return スキニング用の頂点属性の成分数を返します。
//-----------------------------------------------------------------------------
int RShape::GetSkinningComponentCount(
    const int outBoneCountMax,
    const RVec4Array* outBlendWgts
) const
{
    if (m_VtxAttrFlag[RPrimVtx::WGT0])
    {
        if (outBoneCountMax <= 2)
        {
            return outBoneCountMax;
        }
        else
        {
            const int attrMax = (outBoneCountMax > 4) ? 2 : 1;
            for (int iAttr = attrMax - 1; iAttr >= 0; --iAttr)
            {
                const RVec4Array& blendWgts = outBlendWgts[iAttr];
                const int baseCount = iAttr * 4;
                bool zFlag = false;
                bool yFlag = false;
                bool xFlag = false;
                const int weightCount = static_cast<int>(blendWgts.size());
                for (int iw = 0; iw < weightCount; ++iw)
                {
                    const RVec4& weight = blendWgts[iw];
                    if      (weight.w != 0.0f) return baseCount + 4;
                    else if (weight.z != 0.0f) zFlag = true;
                    else if (weight.y != 0.0f) yFlag = true;
                    else if (weight.x != 0.0f) xFlag = true;
                }
                if (zFlag) return baseCount + 3;
                if (yFlag) return baseCount + 2;
                if (xFlag) return baseCount + 1;
            }
            return 1; // ここに来ることはあり得ません。
        }
    }
    else
    {
        return m_VtxAttrFlag[RPrimVtx::IDX0] ? 1 : 0;
    }
}

//-----------------------------------------------------------------------------
//! @brief シェイプの頂点属性出力情報から頂点属性タイプを取得します。
//!
//! @param[in] vtxInfo シェイプの頂点属性出力情報です。
//!
//! @return 頂点属性タイプを返します。
//-----------------------------------------------------------------------------
static RVtxAttrib::Type GetVtxAttribType(const ROutShapeVtxInfo& vtxInfo)
{
    const int typeOfs = vtxInfo.m_CompCount - 1;
    switch (vtxInfo.m_ValueType)
    {
    case RPrimVtx::VALUE_INT:
        return static_cast<RVtxAttrib::Type>(RVtxAttrib::INT + typeOfs);

    case RPrimVtx::VALUE_UINT:
        return static_cast<RVtxAttrib::Type>(RVtxAttrib::UINT + typeOfs);

    default:
        return static_cast<RVtxAttrib::Type>(RVtxAttrib::FLOAT + typeOfs);
    }
}

//-----------------------------------------------------------------------------
//! @brief サブメッシュを出力します。
//-----------------------------------------------------------------------------
void RSubmesh::Out(std::ostream& os, const int tabCount, const int index) const
{
    os << RTab(tabCount) << "<submesh submesh_index=\"" << index
       << "\" offset=\"" << m_Offset
       << "\" count=\"" << m_Count
       << "\" />" << R_ENDL;
}

//-----------------------------------------------------------------------------
//! @brief キーシェイプを出力します。
//-----------------------------------------------------------------------------
void RKeyShape::Out(std::ostream& os, const int tabCount, const int index) const
{
    //-----------------------------------------------------------------------------
    // begin key shape
    const int& tc = tabCount;
    os << RTab(tc) << "<key_shape index=\"" << index
       << "\" name=\"" << RGetUtf8FromShiftJis(m_Name)
       << "\">" << R_ENDL;

    //-----------------------------------------------------------------------------
    // target attrib array
    const int targetAttribCount = static_cast<int>(m_AttribNames.size());
    os << RTab(tc + 1) << "<target_attrib_array length=\"" << targetAttribCount << "\">" << R_ENDL;
    for (int iAttrib = 0; iAttrib < targetAttribCount; ++iAttrib)
    {
        os << RTab(tc + 2) << "<target_attrib index=\"" << iAttrib
           << "\" attrib_name=\"" << m_AttribNames[iAttrib]
           << "\" />" << R_ENDL;
    }
    os << RTab(tc + 1) << "</target_attrib_array>" << R_ENDL;

    //-----------------------------------------------------------------------------
    // end key shape
    os << RTab(tc) << "</key_shape>" << R_ENDL;
}

//-----------------------------------------------------------------------------
//! @brief 接線・従法線・法線が互いに直交するようにデータ列の接線と従法線の値を調整します。
//!
//! @param[in,out] dataStreams データ列配列です。
//! @param[in] vtxData 頂点データです。
//-----------------------------------------------------------------------------
static void AdjustTangents(RDataStreamArray& dataStreams, const RVertex& vtxData)
{
    //-----------------------------------------------------------------------------
    // 法線・接線・従法線のデータ列インデックスを取得します。
    int iNrmStreamTop = -1;
    int iTanStreamTop = -1;
    int iBinStreamTop = -1;
    int nrmStreamCount = 0;
    int tanStreamCount = 0;
    int binStreamCount = 0;
    const RVtxAttribArray& attribs = vtxData.m_VtxAttribs;
    for (size_t iAttrib = 0; iAttrib < attribs.size(); ++iAttrib)
    {
        const RVtxAttrib& attrib = attribs[iAttrib];
        switch (attrib.m_Hint)
        {
        case RVtxAttrib::NORMAL:
            if (iNrmStreamTop == -1) iNrmStreamTop = attrib.m_StreamIndex;
            ++nrmStreamCount;
            break;
        case RVtxAttrib::TANGENT:
            if (iTanStreamTop == -1) iTanStreamTop = attrib.m_StreamIndex;
            ++tanStreamCount;
            break;
        case RVtxAttrib::BINORMAL:
            if (iBinStreamTop == -1) iBinStreamTop = attrib.m_StreamIndex;
            ++binStreamCount;
            break;
        default:
            break;
        }
    }
    //cerr << "adj bins: " << iNrmStreamTop << " " << iTanStreamTop << " " << iBinStreamTop << " " << nrmStreamCount << " " << tanStreamCount << " " << binStreamCount << endl;
    if (nrmStreamCount == 0 ||
        tanStreamCount == 0)
    {
        return;
    }

    //-----------------------------------------------------------------------------
    // 接線と従法線のデータ列の値を調整します。
    const int tanSetCount = RMax(tanStreamCount / nrmStreamCount, 1);
    for (int iStream = 0; iStream < tanStreamCount; ++iStream)
    {
        const int iNrmStream = RMin(iStream / tanSetCount, nrmStreamCount - 1);
        const RFloatArray& nrmVals = dataStreams[iNrmStreamTop + iNrmStream].m_FloatValues;
        RFloatArray&       tanVals = dataStreams[iTanStreamTop + iStream].m_FloatValues;
        RFloatArray&       binVals = (iStream < binStreamCount) ? dataStreams[iBinStreamTop + iStream].m_FloatValues : tanVals;
        const int nrmCount = static_cast<int>(nrmVals.size()) / R_XYZ_COUNT;
        const int tanCount = static_cast<int>(tanVals.size()) / R_XYZW_COUNT;
        const int binCount = static_cast<int>(binVals.size()) / R_XYZW_COUNT;
        //cerr << "adj tans: " << iNrmStreamTop + iNrmStream << ": " << iTanStreamTop + iStream << ": "  << iBinStreamTop + iStream << ": " << nrmCount << " " << tanCount << " " << binCount << endl;
        if (nrmCount == tanCount &&
            nrmCount == binCount)
        {
            int nrmOfs = 0;
            int tanOfs = 0;
            for (int iNrm = 0; iNrm < nrmCount; ++iNrm)
            {
                const RVec3 nrm(nrmVals[nrmOfs + 0], nrmVals[nrmOfs + 1], nrmVals[nrmOfs + 2]);
                const RVec3 tan(tanVals[tanOfs + 0], tanVals[tanOfs + 1], tanVals[tanOfs + 2]);
                //const RVec3 bin(binVals[tanOfs + 0], binVals[tanOfs + 1], binVals[tanOfs + 2]);
                const bool isRight = (tanVals[tanOfs + 3] >= 0.0f);
                RVec3 calcBin = (isRight) ? nrm ^ tan : tan ^ nrm;
                calcBin.Normalize();
                calcBin.SnapToZero();
                if (calcBin == RVec3::kZero) // 法線と接線が同じ方向
                {
                    calcBin = (!nrm.IsEquivalent(RVec3::kZAxis) && !nrm.IsEquivalent(RVec3::kZNegAxis)) ?
                        RVec3::kZNegAxis : RVec3::kXAxis;
                    RVec3 tmpTan = (isRight) ? calcBin ^ nrm : nrm ^ calcBin;
                    tmpTan.Normalize();
                    tmpTan.SnapToZero();
                    calcBin = (isRight) ? nrm ^ tmpTan : tmpTan ^ nrm;
                    calcBin.Normalize();
                    calcBin.SnapToZero();
                }
                RVec3 calcTan = (isRight) ? calcBin ^ nrm : nrm ^ calcBin;
                calcTan.Normalize();
                calcTan.SnapToZero();
                if (calcTan == RVec3::kZero)
                {
                    calcTan = (!nrm.IsEquivalent(RVec3::kXAxis) && !nrm.IsEquivalent(RVec3::kXNegAxis)) ?
                        RVec3::kXAxis : RVec3::kZAxis;;
                }
                tanVals[tanOfs + 0] = calcTan.x;
                tanVals[tanOfs + 1] = calcTan.y;
                tanVals[tanOfs + 2] = calcTan.z;
                if (iStream < binStreamCount)
                {
                    binVals[tanOfs + 0] = calcBin.x;
                    binVals[tanOfs + 1] = calcBin.y;
                    binVals[tanOfs + 2] = calcBin.z;
                }
                nrmOfs += R_XYZ_COUNT;
                tanOfs += R_XYZW_COUNT;
            }
        }
    }
}

//-----------------------------------------------------------------------------
//! @brief 各キーシェイプの頂点データ列を追加します。
//!        テンプレート関数です。
//!
//! @param[in,out] vtxData 頂点データです。
//! @param[in,out] dataStreams データ列配列です。
//! @param[in,out] keyShapes キーシェイプ配列です。
//! @param[in] vtxInfo シェイプの頂点属性出力情報です。
//! @param[in] outIdxs 出力する頂点属性のインデックス配列です。
//! @param[in] hint 頂点属性ヒント情報です。
//! @param[in] typeDummy 頂点属性の型をテンプレート関数に伝えるための使用されない引数です。
//-----------------------------------------------------------------------------
template <typename T>
static void AppendKeyShapesVtxStream(
    RVertex& vtxData,
    RDataStreamArray& dataStreams,
    RKeyShapeArray& keyShapes,
    const ROutShapeVtxInfo& vtxInfo,
    const RIntArray& outIdxs,
    const RVtxAttrib::Hint hint,
    const T& typeDummy
)
{
    const int keyCount = static_cast<int>(vtxInfo.m_pArrays.size());
    for (int iKey = 0; iKey < keyCount; ++iKey)
    {
        const std::vector<T>* pSrcValues =
            reinterpret_cast<const std::vector<T>*>(vtxInfo.m_pArrays[iKey]);

        const int outValueCount = static_cast<int>(outIdxs.size());
        std::vector<T> outValues(outValueCount);
        for (int iVal = 0; iVal < outValueCount; ++iVal)
        {
            outValues[iVal] = (*pSrcValues)[outIdxs[iVal]];
        }

        AppendVtxStream(vtxData, dataStreams,
            outValues, vtxInfo, hint, iKey, NULL);

        if (keyCount >= 2)
        {
            keyShapes[iKey].m_AttribNames.push_back(
                RVtxAttrib::GetNameString(hint, iKey));
        }
    }

    R_UNUSED_VARIABLE(typeDummy);
}

//-----------------------------------------------------------------------------
//! @brief マルチ頂点属性の各キーシェイプの頂点データ列を追加します。
//!
//! @param[in,out] vtxData 頂点データです。
//! @param[in,out] dataStreams データ列配列です。
//! @param[in,out] keyShapes キーシェイプ配列です。
//! @param[in] vtxInfos シェイプの頂点属性出力情報配列です。
//! @param[in] outIdxss 出力する頂点属性のインデックス配列の配列です。
//! @param[in] hint 頂点属性ヒント情報です。
//! @param[in] setCount マルチ頂点属性のセット数です。
//! @param[in] typeDummy 頂点属性の型をテンプレート関数に伝えるための使用されない引数です。
//-----------------------------------------------------------------------------
template <typename T>
static void AppendMultiKeyShapesVtxStream(
    RVertex& vtxData,
    RDataStreamArray& dataStreams,
    RKeyShapeArray& keyShapes,
    const ROutShapeVtxInfo* vtxInfos,
    const RIntArray* outIdxss,
    const RVtxAttrib::Hint hint,
    const int setCount,
    const T& typeDummy
)
{
    int hintIndex = 0;
    const int keyShapeCount = static_cast<int>(keyShapes.size());
    for (int iKey = 0; iKey < keyShapeCount; ++iKey)
    {
        for (int iSet = 0; iSet < setCount; ++iSet)
        {
            const ROutShapeVtxInfo& vtxInfo = vtxInfos[iSet];
            const int setKeyCount = static_cast<int>(vtxInfo.m_pArrays.size());
            if (iKey < setKeyCount)
            {
                const std::vector<T>* pSrcValues =
                    reinterpret_cast<const std::vector<T>*>(vtxInfo.m_pArrays[iKey]);

                const RIntArray& outIdxs = outIdxss[iSet];
                const int outValueCount = static_cast<int>(outIdxs.size());
                std::vector<T> outValues(outValueCount);
                for (int iVal = 0; iVal < outValueCount; ++iVal)
                {
                    outValues[iVal] = (*pSrcValues)[outIdxs[iVal]];
                }

                AppendVtxStream(vtxData, dataStreams,
                    outValues, vtxInfo, hint, hintIndex, NULL);

                if (setKeyCount >= 2)
                {
                    keyShapes[iKey].m_AttribNames.push_back(
                        RVtxAttrib::GetNameString(hint, hintIndex));
                }
                ++hintIndex;
            }
        }
    }

    R_UNUSED_VARIABLE(typeDummy);
}

//-----------------------------------------------------------------------------
//! @brief 頂点シェーダが処理する頂点数を返します。
//!
//! @param[in] idxs 頂点インデックス配列です。
//!
//! @return 頂点シェーダが処理する頂点数を返します。
//-----------------------------------------------------------------------------
static int GetProcessVertexCount(const RIntArray& idxs)
{
    const int VTX_CACHE_SIZE = 14;
    RIntArray cache(VTX_CACHE_SIZE, -1);
    int iFront = 0;

    int processCount = 0;
    const int idxCount = static_cast<int>(idxs.size());
    for (int iVtx = 0; iVtx < idxCount; ++iVtx)
    {
        const int idx = idxs[iVtx];
        if (RFindValueInArray(cache, idx) == -1)
        {
            ++processCount;
            cache[iFront++] = idx;
            if (iFront >= VTX_CACHE_SIZE)
            {
                iFront = 0;
            }
        }
    }
    return processCount;
}

//-----------------------------------------------------------------------------
//! @brief シェイプを出力します。
//-----------------------------------------------------------------------------
RStatus RShape::Out( // OutRShape
    std::ostream& os,
    RVertexArray& vertices,
    RDataStreamArray& dataStreams,
    ROutShapeResult& result,
    const int tabCount,
    const ROutShapeInfo& outInfo
) const
{
    const int& tc = tabCount;

    //-----------------------------------------------------------------------------
    // 各頂点属性の値の配列と比較フラグを取得します。
    uint32_t compareFlags = 0;

    const ROutShapeVtxInfo* outVtxInfos = outInfo.m_VtxInfos;
    const RVec3Array* pVtxPoss =
        reinterpret_cast<const RVec3Array*>(outVtxInfos[RPrimVtx::POS0].m_pArrays[0]);

    int tanSetCount = 0; // マルチ接線の数です。
    for (int iTanSet = 0; iTanSet < RPrimVtx::VTX_TAN_MAX; ++iTanSet)
    {
        if (!m_VtxAttrFlag[RPrimVtx::TAN0 + iTanSet])
        {
            break;
        }
        ++tanSetCount;
    }
    int binSetCount = 0; // マルチ従法線の数です。
    for (int iBinSet = 0; iBinSet < RPrimVtx::VTX_TAN_MAX; ++iBinSet)
    {
        if (!m_VtxAttrFlag[RPrimVtx::BIN0 + iBinSet])
        {
            break;
        }
        ++binSetCount;
    }
    if (tanSetCount >= 2 || binSetCount >= 2)
    {
        compareFlags |= RPrimVtx::COMPARE_TAN1;
    }

    const RVec4Array* pVtxColss[RPrimVtx::VTX_COL_MAX];
    int outVtxColCount = 0; // マルチ頂点カラーの数です。
    int maxBaseColSetIdx = 0; // ベースシェイプの最大カラーセットインデックスです。
    for (int iCol = 0; iCol < RPrimVtx::VTX_COL_MAX; ++iCol)
    {
        pVtxColss[iCol] =
            reinterpret_cast<const RVec4Array*>(outVtxInfos[RPrimVtx::COL0 + iCol].m_pArrays[0]);
        if (pVtxColss[iCol] != NULL)
        {
            ++outVtxColCount;
            maxBaseColSetIdx = iCol;
            if (iCol >= 1)
            {
                compareFlags |= RPrimVtx::COMPARE_COL1;
            }
        }
    }

    const RVec2Array* pVtxTexss[RPrimVtx::VTX_TEX_MAX];
    int outVtxTexCount = 0; // マルチ UV の数です。
    for (int iTex = 0; iTex < RPrimVtx::VTX_TEX_MAX; ++iTex)
    {
        pVtxTexss[iTex] =
            reinterpret_cast<const RVec2Array*>(outVtxInfos[RPrimVtx::TEX0 + iTex].m_pArrays[0]);
        if (pVtxTexss[iTex] == NULL)
        {
            break;
        }
        ++outVtxTexCount;
    }
    if (outVtxTexCount >= 2)
    {
        compareFlags |= RPrimVtx::COMPARE_TEX1;
    }

    const RVec4Array* pVtxUsrss[RPrimVtx::VTX_USR_MAX];
    int usrAttrCount = 0;
    for (int iUa = 0; iUa < RPrimVtx::VTX_USR_MAX; ++iUa)
    {
        pVtxUsrss[iUa] =
            reinterpret_cast<const RVec4Array*>(outVtxInfos[RPrimVtx::USR0 + iUa].m_pArrays[0]);
        if (pVtxUsrss[iUa] != NULL)
        {
            ++usrAttrCount;
        }
    }
    const bool outUsrFlag = (usrAttrCount > 0);
    if (outUsrFlag)
    {
        compareFlags |= RPrimVtx::COMPARE_USR0;
    }
    //cerr <<"compare flags:" << RGetNumberString(compareFlags, "%08x") << endl;

    //-----------------------------------------------------------------------------
    // 出力用の値の配列とインデックス配列を作成します。
    RPrimVtxArray primVtxs;
    // 頂点座標が最初に出現した primVtxs 内のインデックスです。
    // 検索を高速にするために使用します。
    RIntArray pvStartIdxs((*pVtxPoss).size(), -1);

    RIntArray outPosIdxs;
    RIntArray outNrmIdxs;
    RIntArray outTanIdxss[RPrimVtx::VTX_TAN_MAX];
    RIntArray outBinIdxss[RPrimVtx::VTX_TAN_MAX];
    RIntArray outColIdxss[RPrimVtx::VTX_COL_MAX];
    RVec2Array outTexss[RPrimVtx::VTX_TEX_MAX];
    RIVec4Array outBlendIdxss[RVtxMtx::DEFAULT_BONE_MAX / 4];
    RVec4Array  outBlendWgtss[RVtxMtx::DEFAULT_BONE_MAX / 4];
    RVec4Array outUsrss[RPrimVtx::VTX_USR_MAX];

    int iPvMax = 0;
    const int primSetCount = static_cast<int>(m_pPrimSets.size());
    for (int iPs = 0; iPs < primSetCount; ++iPs)
    {
        const RPrimSet& ps = *m_pPrimSets[iPs];
        const int primCount = static_cast<int>(ps.m_pPrimitives.size());
        for (int iPrim = 0; iPrim < primCount; ++iPrim)
        {
            RPrimitive& prim = *ps.m_pPrimitives[iPrim];
            const int vtxCount = static_cast<int>(prim.m_Vtxs.size());
            for (int iVtx = 0; iVtx < vtxCount; ++iVtx)
            {
                //-----------------------------------------------------------------------------
                // すでに追加済みの頂点を検索します。
                const RPrimVtx& vtx = prim.m_Vtxs[iVtx];
                const int iSrcPos = vtx[RPrimVtx::POS0];
                int iPv = -1;
                const int primVtxCount = static_cast<int>(primVtxs.size());
                const int pvStartIdx = pvStartIdxs[iSrcPos]; // ここから検索します。
                if (pvStartIdx == -1)
                {
                    // 初めて出現した頂点座標なら、検索しないで検索開始インデックスを設定します。
                    pvStartIdxs[iSrcPos] = primVtxCount;
                }
                else
                {
                    // すでに出現した頂点座標なら、検索開始インデックスから検索します。
                    for (int iOtherPv = pvStartIdx; iOtherPv < primVtxCount; ++iOtherPv)
                    {
                        if (primVtxs[iOtherPv].IsSameAttr(vtx, compareFlags)) // m_Mtx は比較しません。
                        {
                            iPv = iOtherPv;
                            break;
                        }
                    }
                }

                //-----------------------------------------------------------------------------
                // 新しい頂点なら追加します。
                if (iPv == -1)
                {
                    iPv = static_cast<int>(primVtxs.size());
                    primVtxs.push_back(vtx);
                    outPosIdxs.push_back(iSrcPos);
                    //cerr << "pv: " << vtx[RPrimVtx::POS0] << " " << vtx[RPrimVtx::NRM0] << " " << vtx[RPrimVtx::TAN0] << " " << vtx[RPrimVtx::BIN0] << endl;
                    if (m_VtxAttrFlag[RPrimVtx::NRM0])
                    {
                        outNrmIdxs.push_back(vtx[RPrimVtx::NRM0]);
                    }

                    for (int iTanSet = 0; iTanSet < tanSetCount; ++iTanSet)
                    {
                        outTanIdxss[iTanSet].push_back(vtx[RPrimVtx::TAN0 + iTanSet]);
                    }
                    for (int iBinSet = 0; iBinSet < binSetCount; ++iBinSet)
                    {
                        outBinIdxss[iBinSet].push_back(vtx[RPrimVtx::BIN0 + iBinSet]);
                    }

                    if (outVtxColCount != 0)
                    {
                        for (int iCol = 0; iCol < RPrimVtx::VTX_COL_MAX; ++iCol)
                        {
                            const int iAttr = RPrimVtx::COL0 + iCol;
                            if (m_VtxAttrFlag[iAttr])
                            {
                                outColIdxss[iCol].push_back(vtx[iAttr]);
                            }
                        }
                    }
                    for (int iTex = 0; iTex < outVtxTexCount; ++iTex)
                    {
                        const int iAttr = RPrimVtx::TEX0 + iTex;
                        if (m_VtxAttrFlag[iAttr])
                        {
                            outTexss[iTex].push_back((*pVtxTexss[iTex])[vtx[iAttr]]);
                        }
                    }
                    if (m_VtxAttrFlag[RPrimVtx::IDX0])
                    {
                        outBlendIdxss[0].push_back(m_VtxBlendIdxs[0][vtx[RPrimVtx::IDX0]]);
                    }
                    if (m_VtxAttrFlag[RPrimVtx::WGT0])
                    {
                        outBlendWgtss[0].push_back(m_VtxBlendWgts[0][vtx[RPrimVtx::WGT0]]);
                        if (m_OutBoneCountMax > 4)
                        {
                            outBlendIdxss[1].push_back(m_VtxBlendIdxs[1][vtx[RPrimVtx::IDX1]]);
                            outBlendWgtss[1].push_back(m_VtxBlendWgts[1][vtx[RPrimVtx::WGT1]]);
                        }
                    }
                    if (outUsrFlag)
                    {
                        for (int iUa = 0; iUa < RPrimVtx::VTX_USR_MAX; ++iUa)
                        {
                            const int iAttr = RPrimVtx::USR0 + iUa;
                            if (m_VtxAttrFlag[iAttr])
                            {
                                outUsrss[iUa].push_back((*pVtxUsrss[iUa])[vtx[iAttr]]);
                            }
                        }
                    }
                }

                //-----------------------------------------------------------------------------
                // 頂点インデックスを追加します。
                prim.m_OutIdxs.push_back(iPv);
                if (iPv > iPvMax) iPvMax = iPv;
            }
        }
    }

    //-----------------------------------------------------------------------------
    // begin appending vertex stream
    const int vtxDataIdx = static_cast<int>(vertices.size());
    vertices.push_back(RVertex());
    RVertex& vtxData = vertices[vtxDataIdx];
    vtxData.m_VtxBuffers.push_back(RVtxBuffer());

    //-----------------------------------------------------------------------------
    // キーシェイプ配列を用意します。
    const int keyShapeCount = (!outInfo.m_KeyNames.empty()) ?
        static_cast<int>(outInfo.m_KeyNames.size()) : 1;
    RKeyShapeArray keyShapes;
    for (int iKey = 0; iKey < keyShapeCount; ++iKey)
    {
        const std::string keyName = (!outInfo.m_KeyNames.empty()) ?
            outInfo.m_KeyNames[iKey] : "base";
        RKeyShape keyShape(keyName);
        keyShapes.push_back(keyShape);
    }

    //-----------------------------------------------------------------------------
    // 頂点座標
    AppendKeyShapesVtxStream(vtxData, dataStreams, keyShapes,
        outVtxInfos[RPrimVtx::POS0], outPosIdxs, RVtxAttrib::POSITION, RVec3());

    //-----------------------------------------------------------------------------
    // 法線
    if (m_VtxAttrFlag[RPrimVtx::NRM0])
    {
        AppendKeyShapesVtxStream(vtxData, dataStreams, keyShapes,
            outVtxInfos[RPrimVtx::NRM0], outNrmIdxs, RVtxAttrib::NORMAL, RVec3());
    }

    //-----------------------------------------------------------------------------
    // 接線と従法線
    if (tanSetCount != 0)
    {
        AppendMultiKeyShapesVtxStream(vtxData, dataStreams, keyShapes,
            &outVtxInfos[RPrimVtx::TAN0], outTanIdxss, RVtxAttrib::TANGENT, tanSetCount, RVec4());
    }
    if (binSetCount != 0)
    {
        AppendMultiKeyShapesVtxStream(vtxData, dataStreams, keyShapes,
            &outVtxInfos[RPrimVtx::BIN0], outBinIdxss, RVtxAttrib::BINORMAL, binSetCount, RVec4());
    }

    // 接線・従法線・法線が互いに直交するようにデータ列の接線と従法線の値を調整します。
    if (m_VtxAttrFlag[RPrimVtx::NRM0] &&
        tanSetCount != 0)
    {
        AdjustTangents(dataStreams, vtxData);
    }

    //-----------------------------------------------------------------------------
    // 頂点カラー
    if (outVtxColCount != 0)
    {
        int iHintForKey = maxBaseColSetIdx + 1;
        for (int iKey = 0; iKey < keyShapeCount; ++iKey)
        {
            for (int iCol = 0; iCol < RPrimVtx::VTX_COL_MAX; ++iCol)
            {
                const int iAttr = RPrimVtx::COL0 + iCol;
                const ROutShapeVtxInfo& vtxInfo = outVtxInfos[iAttr];
                const int colKeyCount = static_cast<int>(vtxInfo.m_pArrays.size());
                if (m_VtxAttrFlag[iAttr] && iKey < colKeyCount)
                {
                    const RVec4Array* pSrcValues =
                        reinterpret_cast<const RVec4Array*>(vtxInfo.m_pArrays[iKey]);

                    const RIntArray& outIdxs = outColIdxss[iCol];
                    const int outValueCount = static_cast<int>(outIdxs.size());
                    RVec4Array outValues(outValueCount);
                    for (int iVal = 0; iVal < outValueCount; ++iVal)
                    {
                        outValues[iVal] = (*pSrcValues)[outIdxs[iVal]];
                    }

                    const int iHint = (iKey == 0) ? iCol : iHintForKey++;
                    AppendVec4VtxStream(vtxData, dataStreams,
                        outValues, vtxInfo, RVtxAttrib::COLOR, iHint, NULL);

                    if (colKeyCount >= 2)
                    {
                        keyShapes[iKey].m_AttribNames.push_back(
                            RVtxAttrib::GetNameString(RVtxAttrib::COLOR, iHint));
                    }
                }
            }
        }
    }

    //-----------------------------------------------------------------------------
    // テクスチャ座標
    for (int iTex = 0; iTex < outVtxTexCount; ++iTex)
    {
        const int iHint = (outInfo.m_pUvHintIdxs != NULL) ?
            (*outInfo.m_pUvHintIdxs)[iTex] : iTex;
        const std::string attribName = (outInfo.m_pUvAttribNames != NULL) ?
            (*outInfo.m_pUvAttribNames)[iTex] : "";
        const char* pNameOverride = (!attribName.empty()) ? attribName.c_str() : NULL;
        AppendVtxStream(vtxData, dataStreams,
            outTexss[iTex], outVtxInfos[RPrimVtx::TEX0 + iTex], RVtxAttrib::UV, iHint, pNameOverride);
    }

    //-----------------------------------------------------------------------------
    // blend index & weight
    int skinCompCount = GetSkinningComponentCount(m_OutBoneCountMax, outBlendWgtss);

    // スムーススキニングかつ影響するボーンの最大数が 1 ならダミーのウェイトを追加します。
    // g3d が修正されればこの処理は不要になります。
    if (m_VtxAttrFlag[RPrimVtx::WGT0] && skinCompCount == 1)
    {
        skinCompCount = 2;
    }

    if (m_VtxAttrFlag[RPrimVtx::IDX0])
    {
        ROutShapeVtxInfo info = outVtxInfos[RPrimVtx::IDX0];
        info.m_CompCount = (skinCompCount <= 4) ? skinCompCount : 4;
        AppendVec4VtxStream(vtxData, dataStreams,
            outBlendIdxss[0], info, RVtxAttrib::BLEND_INDEX, 0, NULL);
        if (skinCompCount > 4)
        {
            info.m_CompCount = skinCompCount - 4;
            AppendVec4VtxStream(vtxData, dataStreams,
                outBlendIdxss[1], info, RVtxAttrib::BLEND_INDEX, 1, NULL);
        }
    }
    if (m_VtxAttrFlag[RPrimVtx::WGT0])
    {
        ROutShapeVtxInfo info = outVtxInfos[RPrimVtx::WGT0];
        info.m_CompCount = (skinCompCount <= 4) ? skinCompCount : 4;
        AppendVec4VtxStream(vtxData, dataStreams,
            outBlendWgtss[0], info, RVtxAttrib::BLEND_WEIGHT, 0, NULL);
        if (skinCompCount > 4)
        {
            info.m_CompCount = skinCompCount - 4;
            AppendVec4VtxStream(vtxData, dataStreams,
                outBlendWgtss[1], info, RVtxAttrib::BLEND_WEIGHT, 1, NULL);
        }
    }

    //-----------------------------------------------------------------------------
    // user attribute
    //if (outUsrFlag)
    //{
    //  for (int iUa = 0; iUa < RPrimVtx::VTX_USR_MAX; ++iUa)
    //  {
    //      const int iAttr = RPrimVtx::USR0 + iUa;
    //      if (m_VtxAttrFlag[iAttr])
    //      {
    //          AppendVec4VtxStream(vtxData, dataStreams,
    //              outUsrss[iUa], outVtxInfos[iAttr], RVtxAttrib::USER, iUa, NULL);
    //      }
    //  }
    //}

    //-----------------------------------------------------------------------------
    // 頂点属性名の重複をチェックします。
    if (outInfo.m_pUvAttribNames != NULL)
    {
        RStringArray attribNames;
        const RVtxAttribArray& attribs = vtxData.m_VtxAttribs;
        for (size_t iAttrib = 0; iAttrib < attribs.size(); ++iAttrib)
        {
            const std::string attribName = attribs[iAttrib].GetName();
            if (RFindValueInArray(attribNames, attribName) != -1)
            {
                return RStatus(RStatus::FAILURE, // RShowError: Vertex attribute name is duplicate
                    L"1 つのシェイプで頂点属性名が重複しています: {0} \n"
                    L"「nw4f_fix_」または「nw4f_fixN_（N は整数）」で始まる名前の UV セットを使用する場合、"
                    L"頂点属性名がユニークになるように修正してください。",
                    "The same vertex attribute name appears more than once for a particular shape: {0} \n"
                    "If you use a UV set that has a name starting with nw4f_fix_ or nw4f_fixN_ (where N is an integer), "
                    "edit vertex attribute names so that they are unique.",
                    attribName);
            }
            attribNames.push_back(attribName);
        }
    }

    //-----------------------------------------------------------------------------
    // begin shape
    os << RTab(tc) << "<shape index=\"" << outInfo.m_Index
       << "\" name=\"" << RGetUtf8FromShiftJis(outInfo.m_Name)
       << "\">" << R_ENDL;

    //-----------------------------------------------------------------------------
    // shape info
    os << RTab(tc + 1) << "<shape_info" << R_ENDL;
    os << RTab(tc + 2) << "mat_name=\"" << outInfo.m_MatName << "\"" << R_ENDL;
    os << RTab(tc + 2) << "bone_name=\"" << outInfo.m_BoneName << "\"" << R_ENDL;
    os << RTab(tc + 2) << "original_material_name=\"" << outInfo.m_MatName << "\"" << R_ENDL;
    os << RTab(tc + 2) << "original_bone_name=\"" << outInfo.m_BoneName << "\"" << R_ENDL;
    os << RTab(tc + 2) << "vertex_index=\"" << outInfo.m_Index << "\"" << R_ENDL;
    //os << RTab(tc + 2) << "visibility=\"" << RBoolStr(true) << "\"" << R_ENDL; // 当面廃止
    os << RTab(tc + 2) << "vertex_skinning_count=\"" << skinCompCount << "\"" << R_ENDL;
    os << RTab(tc + 2) << "local_coordinate=\"" << RBoolStr(skinCompCount <= 1) << "\"" << R_ENDL;
    os << RTab(tc + 2) << "original_shape_hash=\"" << "" << "\"" << R_ENDL;
    os << RTab(tc + 2) << "original_shape_count=\"" << 1 << "\"" << R_ENDL;
    os << RTab(tc + 2) << "delete_near_vertex_mode=\"" << "" << "\"" << R_ENDL;
    os << RTab(tc + 2) << "divide_submesh_mode=\"" << "" << "\"" << R_ENDL;
    os << RTab(tc + 2) << "optimize_primitive_mode=\"" << "" << "\"" << R_ENDL;
    os << RTab(tc + 2) << "polygon_reduction_mode=\"" << "" << "\"" << R_ENDL;
    os << RTab(tc + 1) << "/>" << R_ENDL;

    //-----------------------------------------------------------------------------
    // oriented bounding box
    //outInfo.m_OrientedBB.Out(os, tc + 1);

    //-----------------------------------------------------------------------------
    // begin mesh array
    os << RTab(tc + 1) << "<mesh_array length=\"" << 1 << "\">" << R_ENDL;
    const int meshTc = tc + 1;

    //-----------------------------------------------------------------------------
    // loop for primitive set
    result.m_TriangleCount = 0;
    result.m_IndexCount = 0;
    result.m_VertexCount = static_cast<int>(primVtxs.size());
    result.m_ProcessVertexCount = 0;

    for (int iPs = 0; iPs < primSetCount; ++iPs)
    {
        const RPrimSet& ps = *m_pPrimSets[iPs];
        const int primCount = static_cast<int>(ps.m_pPrimitives.size());

        //-----------------------------------------------------------------------------
        // get submeshs & out index stream
        RSubmeshArray submeshs;
        RIntArray outIdxs;
        for (int iPrim = 0; iPrim < primCount; ++iPrim)
        {
            const RPrimitive& prim = *ps.m_pPrimitives[iPrim];
            submeshs.push_back(RSubmesh(static_cast<int>(outIdxs.size()),
                static_cast<int>(prim.m_OutIdxs.size())));
            RAppendArrayToArray(outIdxs, prim.m_OutIdxs);
        }
        const int streamIdx = static_cast<int>(dataStreams.size());
        dataStreams.push_back(RDataStream(outIdxs, 3));

        result.m_TriangleCount += static_cast<int>(outIdxs.size()) / 3;
        result.m_IndexCount    += static_cast<int>(outIdxs.size());
        result.m_ProcessVertexCount += GetProcessVertexCount(outIdxs);

        //-----------------------------------------------------------------------------
        // begin mesh
        os << RTab(meshTc + 1) << "<mesh"
           << " index=\"" << 0 << "\""
           << R_ENDL;
        os << RTab(meshTc + 2) << "mode=\"" << RPrimitive::GetPrimitiveModeStr(RPrimitive::TRIANGLES) << "\"" << R_ENDL;
        os << RTab(meshTc + 2) << "quantize_type=\"" << "none" << "\"" << R_ENDL;
        os << RTab(meshTc + 2) << "count=\"" << static_cast<int>(outIdxs.size()) << "\"" << R_ENDL;
        os << RTab(meshTc + 2) << "stream_index=\"" << streamIdx << "\"" << R_ENDL;
        os << RTab(meshTc + 1) << ">" << R_ENDL;

        //-----------------------------------------------------------------------------
        // out submeshs
        ROutArrayElement(os, meshTc + 2, submeshs, "submesh_array");

        //-----------------------------------------------------------------------------
        // end mesh
        os << RTab(meshTc + 1) << "</mesh>" << R_ENDL;
    }

    //-----------------------------------------------------------------------------
    // end mesh array
    os << RTab(tc + 1) << "</mesh_array>" << R_ENDL;

    //-----------------------------------------------------------------------------
    // out key shapes
    if (!outInfo.m_KeyNames.empty())
    {
        ROutArrayElement(os, tc + 1, keyShapes, "key_shape_array");
    }

    //-----------------------------------------------------------------------------
    // end shape
    os << RTab(tc) << "</shape>" << R_ENDL;

    return RStatus::SUCCESS;
} // NOLINT(impl/function_size)

//=============================================================================
// dcc ネームスペースを終了します。
//=============================================================================
} // namespace dcc
} // namespace tool
} // namespace gfx
} // namespace nn

