﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

// ReadFile

//=============================================================================
// include
//=============================================================================
#include "DccCommon.h"

//=============================================================================
// dcc ネームスペースを開始します。
//=============================================================================
namespace nn {
namespace gfx {
namespace tool {
namespace dcc {

//=============================================================================
//! @brief テクスチャーイメージのクラスです。
//=============================================================================
class RImage
{
public:
    static const int WidthHeightMin = 1; //!< テクスチャーの幅と高さの最小値です。
    static const int WidthHeightMax = 16384; //!< テクスチャーの幅と高さの最大値です。
    static const int MipCountMax = 15; //!< ミップマップのレベル数の最大値です。

    static const int CUBE_HC_COUNT_H = 4; //!< 水平十字キューブマップの水平方向のフェース数です。
    static const int CUBE_HC_COUNT_V = 3; //!< 水平十字キューブマップの垂直方向のフェース数です。
    static const int CUBE_VC_COUNT_H = 3; //!< 垂直十字キューブマップの水平方向のフェース数です。
    static const int CUBE_VC_COUNT_V = 4; //!< 垂直十字キューブマップの垂直方向のフェース数です。

    static const std::string HINT_ALBEDO;     //!< アルベド用のヒント情報です。
    static const std::string HINT_OPACITY;    //!< 不透明度用のヒント情報です。
    static const std::string HINT_EMISSION;   //!< エミッション用のヒント情報です。
    static const std::string HINT_NORMAL;     //!< 法線用のヒント情報です。
    static const std::string HINT_TANGENT;    //!< 接線用のヒント情報です。
    static const std::string HINT_SPECULAR;   //!< スペキュラ用のヒント情報です。
    static const std::string HINT_REFLECTION; //!< 反射用のヒント情報です。

    //! @brief リニア変換フラグを表す列挙型です。
    enum LinearFlag
    {
        LINEAR_NONE    = 0,        //!< sRGB からリニアに変換しません。
        LINEAR_R       = (1 << 3), //!< R 成分を sRGB からリニアに変換します。
        LINEAR_G       = (1 << 2), //!< G 成分を sRGB からリニアに変換します。
        LINEAR_B       = (1 << 1), //!< B 成分を sRGB からリニアに変換します。
        LINEAR_A       = (1 << 0), //!< A 成分を sRGB からリニアに変換します。
        LINEAR_RGB     = LINEAR_R | LINEAR_G | LINEAR_B, //!< RGB 成分を sRGB からリニアに変換します。
        LINEAR_RGBA    = LINEAR_RGB | LINEAR_A,          //!< RGBA 成分を sRGB からリニアに変換します。
        LINEAR_INVALID = -2,       //!< 無効なリニア変換です。
    };

    //! @brief 次元を表す列挙型です。
    enum Dimension
    {
        DIM_1D,             //!< 一次元テクスチャーです。
        DIM_2D,             //!< 二次元テクスチャーです。
        DIM_3D,             //!< 三次元テクスチャーです。
        DIM_CUBE_SEPARATE,  //!< キューブマップ（6 面個別指定）です。
        DIM_CUBE_HC,        //!< キューブマップ（水平十字）です。
        DIM_CUBE_VC,        //!< キューブマップ（垂直十字）です。
        DIM_CUBE_INPUT,     //!< キューブマップ（入力ファイルがキューブマップ）です。
        DIM_1D_ARRAY,       //!< 一次元配列テクスチャーです。
        DIM_2D_ARRAY,       //!< 二次元配列テクスチャーです。
        DIM_COUNT           //!< 次元の総数です。
    };

    //! @brief フォーマットを表す列挙型です。
    enum Format
    {
        Format_Unorm_8,             //!< 1 channel ,  8 bit/pixel
        Format_Snorm_8,             //!< 1 channel ,  8 bit/pixel
        Format_Uint_8,              //!< 1 channel ,  8 bit/pixel
        Format_Sint_8,              //!< 1 channel ,  8 bit/pixel

        Format_Unorm_4_4,           //!< 2 channels,  8 bit/pixel
        Format_Unorm_8_8,           //!< 2 channels, 16 bit/pixel
        Format_Snorm_8_8,           //!< 2 channels, 16 bit/pixel
        Format_Uint_8_8,            //!< 2 channels, 16 bit/pixel
        Format_Sint_8_8,            //!< 2 channels, 16 bit/pixel

        Format_Unorm_5_6_5,         //!< 3 channels, 16 bit/pixel

        Format_Unorm_5_5_5_1,       //!< 4 channels, 16 bit/pixel
        Format_Unorm_4_4_4_4,       //!< 4 channels, 16 bit/pixel

        Format_Unorm_8_8_8_8,       //!< 4 channels, 32 bit/pixel
        Format_Snorm_8_8_8_8,       //!< 4 channels, 32 bit/pixel
        Format_Srgb_8_8_8_8,        //!< 4 channels, 32 bit/pixel
        Format_Uint_8_8_8_8,        //!< 4 channels, 32 bit/pixel
        Format_Sint_8_8_8_8,        //!< 4 channels, 32 bit/pixel

        Format_Float_16,            //!< 1 channels, 16 bit/pixel
        Format_Float_32,            //!< 1 channels, 32 bit/pixel
        Format_Float_16_16,         //!< 2 channels, 16 bit/pixel
        Format_Float_32_32,         //!< 2 channels, 64 bit/pixel
        Format_Float_11_11_10,      //!< 3 channels, 32 bit/pixel
        Format_Float_16_16_16_16,   //!< 4 channels, 64 bit/pixel
        Format_Float_32_32_32_32,   //!< 4 channels,128 bit/pixel

        Format_Unorm_Bc1,           //!< 4 channels,  4 bit/pixel
        Format_Srgb_Bc1,            //!< 4 channels,  4 bit/pixel
        Format_Unorm_Bc2,           //!< 4 channels,  8 bit/pixel
        Format_Srgb_Bc2,            //!< 4 channels,  8 bit/pixel
        Format_Unorm_Bc3,           //!< 4 channels,  8 bit/pixel
        Format_Srgb_Bc3,            //!< 4 channels,  8 bit/pixel
        Format_Unorm_Bc4,           //!< 1 channel ,  4 bit/pixel
        Format_Snorm_Bc4,           //!< 1 channel ,  4 bit/pixel
        Format_Unorm_Bc5,           //!< 2 channels,  8 bit/pixel
        Format_Snorm_Bc5,           //!< 2 channels,  8 bit/pixel

        Format_Ufloat_Bc6,          //!< 3 channels,  8 bit/pixel
        Format_Float_Bc6,           //!< 3 channels,  8 bit/pixel
        Format_Unorm_Bc7,           //!< 4 channels,  8 bit/pixel
        Format_Srgb_Bc7,            //!< 4 channels,  8 bit/pixel

        Format_Unorm_Etc1,          //!< 3 channels,  4 bit/pixel
        Format_Unorm_Etc2,          //!< 3 channels,  4 bit/pixel
        Format_Srgb_Etc2,           //!< 3 channels,  4 bit/pixel
        Format_Unorm_Etc2_Mask,     //!< 4 channels,  4 bit/pixel
        Format_Srgb_Etc2_Mask,      //!< 4 channels,  4 bit/pixel
        Format_Unorm_Etc2_Alpha,    //!< 4 channels,  8 bit/pixel
        Format_Srgb_Etc2_Alpha ,    //!< 4 channels,  8 bit/pixel

        Format_Unorm_Eac_11,        //!< 1 channels,  4 bit/pixel
        Format_Snorm_Eac_11,        //!< 1 channels,  4 bit/pixel
        Format_Unorm_Eac_11_11,     //!< 2 channels,  8 bit/pixel
        Format_Snorm_Eac_11_11,     //!< 2 channels,  8 bit/pixel

        Format_Unorm_Pvrtc1_2Bpp,       //!< 4 channels,  2 bit/pixel
        Format_Srgb_Pvrtc1_2Bpp,        //!< 4 channels,  2 bit/pixel
        Format_Unorm_Pvrtc1_4Bpp,       //!< 4 channels,  4 bit/pixel
        Format_Srgb_Pvrtc1_4Bpp,        //!< 4 channels,  4 bit/pixel
        Format_Unorm_Pvrtc1_Alpha_2Bpp, //!< 4 channels,  2 bit/pixel
        Format_Srgb_Pvrtc1_Alpha_2Bpp,  //!< 4 channels,  2 bit/pixel
        Format_Unorm_Pvrtc1_Alpha_4Bpp, //!< 4 channels,  4 bit/pixel
        Format_Srgb_Pvrtc1_Alpha_4Bpp,  //!< 4 channels,  4 bit/pixel
        Format_Unorm_Pvrtc2_Alpha_2Bpp, //!< 4 channels,  2 bit/pixel
        Format_Srgb_Pvrtc2_Alpha_2Bpp,  //!< 4 channels,  2 bit/pixel
        Format_Unorm_Pvrtc2_Alpha_4Bpp, //!< 4 channels,  4 bit/pixel
        Format_Srgb_Pvrtc2_Alpha_4Bpp,  //!< 4 channels,  4 bit/pixel

        Format_Unorm_Astc_4x4,      //!< 4 channels, 8.0 bit/pixel
        Format_Srgb_Astc_4x4,       //!< 4 channels, 8.0 bit/pixel
        Format_Unorm_Astc_5x4,      //!< 4 channels, 6.4 bit/pixel
        Format_Srgb_Astc_5x4,       //!< 4 channels, 6.4 bit/pixel
        Format_Unorm_Astc_5x5,      //!< 4 channels, 5.1 bit/pixel
        Format_Srgb_Astc_5x5,       //!< 4 channels, 5.1 bit/pixel
        Format_Unorm_Astc_6x5,      //!< 4 channels, 4.3 bit/pixel
        Format_Srgb_Astc_6x5,       //!< 4 channels, 4.3 bit/pixel
        Format_Unorm_Astc_6x6,      //!< 4 channels, 3.6 bit/pixel
        Format_Srgb_Astc_6x6,       //!< 4 channels, 3.6 bit/pixel
        Format_Unorm_Astc_8x5,      //!< 4 channels, 3.2 bit/pixel
        Format_Srgb_Astc_8x5,       //!< 4 channels, 3.2 bit/pixel
        Format_Unorm_Astc_8x6,      //!< 4 channels, 2.7 bit/pixel
        Format_Srgb_Astc_8x6,       //!< 4 channels, 2.7 bit/pixel
        Format_Unorm_Astc_8x8,      //!< 4 channels, 2.0 bit/pixel
        Format_Srgb_Astc_8x8,       //!< 4 channels, 2.0 bit/pixel
        Format_Unorm_Astc_10x5,     //!< 4 channels, 2.6 bit/pixel
        Format_Srgb_Astc_10x5,      //!< 4 channels, 2.6 bit/pixel
        Format_Unorm_Astc_10x6,     //!< 4 channels, 2.1 bit/pixel
        Format_Srgb_Astc_10x6,      //!< 4 channels, 2.1 bit/pixel
        Format_Unorm_Astc_10x8,     //!< 4 channels, 1.6 bit/pixel
        Format_Srgb_Astc_10x8,      //!< 4 channels, 1.6 bit/pixel
        Format_Unorm_Astc_10x10,    //!< 4 channels, 1.3 bit/pixel
        Format_Srgb_Astc_10x10,     //!< 4 channels, 1.3 bit/pixel
        Format_Unorm_Astc_12x10,    //!< 4 channels, 1.1 bit/pixel
        Format_Srgb_Astc_12x10,     //!< 4 channels, 1.1 bit/pixel
        Format_Unorm_Astc_12x12,    //!< 4 channels, 0.9 bit/pixel
        Format_Srgb_Astc_12x12,     //!< 4 channels, 0.9 bit/pixel

        Format_Count,               //!< フォーマットの総数です。
        Format_Invalid,             //!< 無効なフォーマットです。
    };

    static const int INITIAL_SWIZZLE_MASK = 0x7; //!< 初期スウィズル値のマスクです。

    //! @brief 透明モードを表す列挙型です。
    //         OPAQUE は WinGDI.h で定義されているので OPA にしています。
    enum TransparencyMode
    {
        OPA,    //!< 不透明です（アルファがすべて 255）。
        MASK,   //!< 抜きありです（アルファが 0 または 255）。
        XLU,    //!< 半透明です（1 以上 254 以下のアルファが存在）。
    };

    //! @brief キューブマップのフェースを表す列挙型です。
    enum CubeFace
    {
        CUBE_FACE_PX,   //!< +X 面です。
        CUBE_FACE_NX,   //!< -X 面です。
        CUBE_FACE_PY,   //!< +Y 面です。
        CUBE_FACE_NY,   //!< -Y 面です。
        CUBE_FACE_PZ,   //!< +Z 面です。
        CUBE_FACE_NZ,   //!< -Z 面です。
        CUBE_FACE_COUNT //!< フェースの総数です。
    };

    //! @brief 画像ファイルの種類を表す列挙型です。
    enum FileType
    {
        TGA,        //!< TGA ファイルです。
        DDS,        //!< DDS ファイルです。
        EXR,        //!< EXR ファイルです。
        FILE_OTHER  //!< その他のファイルです。
    };

protected:
    // attr
    std::string m_Name; //!< 出力用の名前（オリジナルの名前の使用できない文字を補正）です。
    std::string m_OrgName; //!< オリジナルの名前（画像ファイル名から拡張子を除いたもの）です。

    //! @brief 画像ファイルのパス配列です。
    //!        キューブマップ（6 面個別指定）の場合 +X、-X、+Y、-Y、+Z、-Z 面の順に格納します。
    RStringArray m_FilePaths;

    std::string m_MergePath; //!< マージする ftx ファイルのパスです。

    std::string m_ProjectRootPath; //!< アプリケーションのプロジェクトのルートフォルダのパスです。
    std::string m_DccPreset; //!< DCC プリセット名です。
    std::string m_Hint; //!< ヒント情報です。
    int m_LinearFlag; //!< リニア変換フラグです。
    bool m_UsesSrgbFetch; //!< sRGB フォーマットを使用するなら true です。
    Dimension m_Dimension; //!< 次元です。
    Format m_Format; //!< フォーマットです。
    std::string m_CompSel; //!< 成分選択です。空文字ならフォーマットに応じた成分選択を適用します。
    int m_ImageW; //!< テクスチャーの幅（キューブマップは各フェースの幅）です。
    int m_ImageH; //!< テクスチャーの高さ（キューブマップは各フェースの高さ）です。
    int m_ImageD; //!< テクスチャーの奥行き（キューブマップは 6、他は 1）です。
    int m_MipLevel; //!< ミップマップのレベル数です。
    bool m_WeightedCompress; //!< bc1 / bc2 / bc3 フォーマット圧縮時の RGB 成分の誤差の重み付けです。
    int m_InitialSwizzle; //!< 初期スウィズル値です。
    std::string m_CommentText; //!< 編集用コメント文です。

    // internal attr
    int m_OrgIndex; //!< モデル内でのインデックスです。
    int m_FullW; //!< 元のイメージデータの幅（キューブマップは水平十字全体の幅）です。
    int m_FullH; //!< 元のイメージデータの高さ（キューブマップは水平十字全体の高さ）です。
    bool m_HasAlpha; //!< アルファ成分を持つなら true です。
    bool m_IsFloat; //!< データ型が浮動小数点数なら true、整数なら false です。
    TransparencyMode m_TransparencyMode; //!< 透明モードです。
    FileType m_FileType; //!< 画像ファイルタイプです。

    // add info
    int m_AddInfoOfs; //!< 付加情報の画像ファイルの先頭からのオフセットです。付加情報がなければ 0 です。

    // buffer
    uint8_t* m_pFullBitmap; //!< 元のイメージデータ（RGBA）です。read 時に付加情報があれば使用しません（NULL のまま）。
    uint8_t* m_pTexelData; //!< テクスチャーデータです。
    int m_TexelDataSize; //!< テクスチャーデータのサイズ（バイト数）です。

public:
    //! @brief コンストラクタです。
    //!
    //! @param[in] orgIndex モデル内でのインデックスです。
    //!
    explicit RImage(const int orgIndex = 0)
    : m_OrgIndex(orgIndex),
      m_pFullBitmap(NULL),
      m_pTexelData(NULL)
    {
        Init("", HINT_ALBEDO, LINEAR_NONE, false, DIM_2D, 0, NULL);
    }

    //! コピーコンストラクタです。
    RImage(const RImage& other)
    : m_pFullBitmap(NULL),
      m_pTexelData(NULL)
    {
        operator=(other);
    }

    //! デストラクタです。
    virtual ~RImage()
    {
        FreeMemory();
    }

    //! メモリを解放します。
    void FreeMemory()
    {
        //if (m_pFullBitmap != NULL || m_pTexelData != NULL) std::cerr << "free: " << this << std::endl;
        RFreeAndClearArray(m_pFullBitmap);
        RFreeAndClearArray(m_pTexelData);
        m_TexelDataSize = 0;
    }

    //! @brief 初期化します。
    //!
    //! @param[in] dccPreset DCC プリセット名です。
    //! @param[in] hint ヒント情報です。
    //! @param[in] linearFlag リニア変換フラグです。
    //! @param[in] usesSrgbFetch sRGB フォーマットを使用するなら true です。
    //! @param[in] dimension 次元です。
    //! @param[in] initialSwizzle 初期スウィズル値です。
    //! @param[in] pComment 編集用コメント文です。
    //!
    void Init(
        const std::string& dccPreset,
        const std::string& hint,
        const int linearFlag,
        const bool usesSrgbFetch,
        const Dimension dimension,
        const int initialSwizzle,
        const char* pComment
    )
    {
        FreeMemory();

        // attr
        m_Name.clear();
        m_OrgName.clear();
        m_FilePaths.clear();
        m_MergePath.clear();

        m_ProjectRootPath.clear();
        m_DccPreset = dccPreset;
        m_Hint = hint;
        m_LinearFlag = linearFlag;
        m_UsesSrgbFetch = usesSrgbFetch;
        m_Dimension = dimension;
        m_Format = Format_Unorm_8_8_8_8;
        m_CompSel = "";
        m_ImageW = 0;
        m_ImageH = 0;
        m_ImageD = 1;
        m_MipLevel = 1;
        m_WeightedCompress = false;
        m_InitialSwizzle = initialSwizzle;
        m_CommentText = (pComment != NULL) ? pComment : "";

        // internal attr
        m_FullW = 0;
        m_FullH = 0;
        m_HasAlpha = false;
        m_IsFloat = false;
        m_TransparencyMode = OPA;
        m_FileType = FILE_OTHER;

        // add info
        m_AddInfoOfs = 0;

        // buffer
        m_TexelDataSize = 0;
    }

    //! 代入演算子です。
    RImage& operator=(const RImage& other)
    {
        if (this != &other)
        {
            FreeMemory();

            // attr
            m_Name             = other.m_Name;
            m_OrgName          = other.m_OrgName;
            m_FilePaths        = other.m_FilePaths;

            m_ProjectRootPath  = other.m_ProjectRootPath;
            m_DccPreset        = other.m_DccPreset;
            m_Hint             = other.m_Hint;
            m_LinearFlag       = other.m_LinearFlag;
            m_UsesSrgbFetch    = other.m_UsesSrgbFetch;
            m_Dimension        = other.m_Dimension;
            m_Format           = other.m_Format;
            m_CompSel          = other.m_CompSel;
            m_ImageW           = other.m_ImageW;
            m_ImageH           = other.m_ImageH;
            m_ImageD           = other.m_ImageD;
            m_MipLevel         = other.m_MipLevel;
            m_WeightedCompress = other.m_WeightedCompress;
            m_InitialSwizzle   = other.m_InitialSwizzle;
            m_CommentText      = other.m_CommentText;

            // internal attr
            m_OrgIndex         = other.m_OrgIndex;
            m_FullW            = other.m_FullW;
            m_FullH            = other.m_FullH;
            m_HasAlpha         = other.m_HasAlpha;
            m_IsFloat          = other.m_IsFloat;
            m_TransparencyMode = other.m_TransparencyMode;
            m_FileType         = other.m_FileType;

            // add info
            m_AddInfoOfs  = other.m_AddInfoOfs;

            // buffer
            //if (other.m_pFullBitmap != NULL || other.m_pTexelData != NULL) std::cerr << "copy: " << this << std::endl;
            const int colBytes = (m_IsFloat) ? R_RGBA_FLOAT_BYTES : R_RGBA_BYTES;
            RAllocAndCopyArray(m_pFullBitmap, other.m_pFullBitmap, m_FullW * m_FullH * colBytes);
            m_TexelDataSize = other.m_TexelDataSize;
            RAllocAndCopyArray(m_pTexelData, other.m_pTexelData, m_TexelDataSize);
        }
        return *this;
    }

    //! @brief 画像ファイルをリードしてデータを取得します。
    //!
    //! @param[in] filePaths 画像ファイルのパス配列です。TGA、DDS ファイルに対応しています。
    //!                      キューブマップ（6 面個別指定）の場合 +X、-X、+Y、-Y、+Z、-Z 面の順に指定します。
    //! @param[in] mergePath マージする ftx ファイルのパスです。空文字ならマージしません。
    //! @param[in] dccPreset DCC プリセット名です。
    //! @param[in] hint ヒント情報です。
    //! @param[in] linearFlag リニア変換フラグです。
    //! @param[in] usesSrgbFetch sRGB フォーマットを使用するなら true です。
    //! @param[in] dimension 次元です。
    //! @param[in] initialSwizzle 初期スウィズル値です。
    //! @param[in] comment 編集用コメント文です。
    //!
    //! @return 処理結果を返します。
    //!
    RStatus ReadFile(
        const RStringArray& filePaths,
        const std::string& mergePath,
        const std::string& dccPreset,
        const std::string& hint,
        const int linearFlag,
        const bool usesSrgbFetch,
        const Dimension dimension,
        const int initialSwizzle,
        const std::string& comment
    );

    //! @brief メモリ上のビットマップをリードしてデータを取得します。
    //!
    //! @param[in] filePaths 画像ファイルのパス配列です。テクスチャー名を設定するために使用します。
    //!                      キューブマップ（6 面個別指定）の場合 +X、-X、+Y、-Y、+Z、-Z 面の順に指定します。
    //! @param[in] mergePath マージする ftx ファイルのパスです。空文字ならマージしません。
    //! @param[in] dccPreset DCC プリセット名です。
    //! @param[in] hint ヒント情報です。
    //! @param[in] linearFlag リニア変換フラグです。
    //! @param[in] usesSrgbFetch sRGB フォーマットを使用するなら true です。
    //! @param[in] dimension 次元です。
    //! @param[in] initialSwizzle 初期スウィズル値です。
    //! @param[in] comment 編集用コメント文です。
    //! @param[in] pBitmap 画像全体のビットマップへのポインタです。
    //!                   ビットマップは 1 ピクセルあたり 4 or  16 バイトで、RGBA の順に格納します。
    //!                   画像の左上が先頭です。
    //! @param[in] fullW 画像全体の幅です。
    //! @param[in] fullH 画像全体の高さです。
    //! @param[in] hasAlpha アルファ成分を使用するかどうかです。
    //! @param[in] isFloat データ型が浮動小数点数なら true、整数なら false です。
    //!
    void ReadBitmap(
        const RStringArray& filePaths,
        const std::string& mergePath,
        const std::string& dccPreset,
        const std::string& hint,
        const int linearFlag,
        const bool usesSrgbFetch,
        const Dimension dimension,
        const int initialSwizzle,
        const std::string& comment,
        const uint8_t* pBitmap,
        const int fullW,
        const int fullH,
        const bool hasAlpha,
        const bool isFloat
    );

    //! @brief 3D テクスチャーコンバーターのオプション文字列を返します。
    //!
    //! @param[in] ftxPath ftx ファイルのパスです。
    //!
    //! @return オプション文字列を返します。
    //!
    std::string GetConverterOption(const std::string& ftxPath) const;

    //-----------------------------------------------------------------------------
    // アクセサ

    //! テクスチャー名を返します。
    std::string GetName() const { return m_Name; }

    //! テクスチャー名を設定します。
    void SetName(const std::string& name) { m_Name = name; }

    //! 補正前のテクスチャー名を返します。
    std::string GetOrgName() const { return m_OrgName; }

    //! 画像ファイルのパス配列を返します。
    RStringArray GetFilePaths() const { return m_FilePaths; }

    //! @brief メインの画像ファイルのパスを返します。
    //!        キューブマップ（6 面個別指定）では +Z 面のパスを返します。
    //!        画像ファイルのパス配列と次元を設定してから呼びます。
    //!
    std::string GetMainFilePath() const
    {
        return GetMainFilePath(m_FilePaths, m_Dimension);
    }

    //! @brief 画像ファイルのパス配列を設定します。
    //!        パスからテクスチャー名も設定されます。
    //!        次元を設定してから呼びます。
    //!
    //! @param[in] filePaths パス配列です。
    //!
    void SetFilePaths(const RStringArray& filePaths);

    //! マージする ftx ファイルのパスを返します。
    std::string GetMergePath() const { return m_MergePath; }

    //! プロジェクトのルートフォルダのパスを返します。
    std::string GetProjectRootPath() const
    {
        return m_ProjectRootPath;
    }

    //! プロジェクトのルートフォルダのパスを設定します。
    void SetProjectRootPath(const std::string& projectRootPath)
    {
        m_ProjectRootPath = projectRootPath;
    }

    //! DCC プリセット名を返します。
    std::string GetDccPreset() const { return m_DccPreset; }

    //! DCC プリセット名を設定します。
    void SetDccPreset(const std::string& dccPreset) { m_DccPreset = dccPreset; }

    //! ヒント情報を返します。
    std::string GetHint() const { return m_Hint; }

    //! ヒント情報を設定します。
    void SetHint(const std::string& hint) { m_Hint = hint; }

    //! リニア変換フラグを返します。
    int GetLinearFlag() const { return m_LinearFlag; }

    //! リニア変換フラグを設定します。
    void SetLinearFlag(const int linearFlag) { m_LinearFlag = linearFlag; }

    //! sRGB フォーマットを使用するなら true を返します。
    bool UsesSrgbFetch() const { return m_UsesSrgbFetch; }

    //! sRGB フォーマットを使用するかどうかを設定します。
    void SetUsesSrgbFetch(const bool usesSrgbFetch) { m_UsesSrgbFetch = usesSrgbFetch; }

    //! 次元を返します。
    Dimension GetDimension() const { return m_Dimension; }

    //! 次元を設定します。
    void SetDimension(const Dimension dimension) { m_Dimension = dimension; }

    //! キューブマップなら true を返します。
    bool IsCubeMap() const { return IsCubeDimension(m_Dimension); }

    //! フォーマットを返します。
    Format GetFormat() const { return m_Format; }

    //! フォーマットを設定します。
    void SetFormat(const Format format) { m_Format = format; }

    //! フォーマット名を返します。
    std::string GetFormatName() const { return GetFormatName(m_Format); }

    //! 成分選択を返します。
    std::string GetCompSel() const { return m_CompSel; }

    //! 成分選択を設定します。
    void SetCompSel(const std::string& compSel) { m_CompSel = compSel; }

    //! テクスチャーの幅を返します。
    int GetImageW() const { return m_ImageW; }

    //! テクスチャーの高さを返します。
    int GetImageH() const { return m_ImageH; }

    //! テクスチャーの奥行きを返します。
    int GetImageD() const { return m_ImageD; }

    //! ミップマップのレベル数を返します。
    int GetMipLevel() const { return m_MipLevel; }

    //! bc1 / bc2 / bc3 フォーマット圧縮時の RGB 成分の誤差の重み付けを返します。
    bool GetWeightedCompress() const { return m_WeightedCompress; }

    //! bc1 / bc2 / bc3 フォーマット圧縮時の RGB 成分の誤差の重み付けを設定します。
    void SetWeightedCompress(const bool weightedCompress) { m_WeightedCompress = weightedCompress; }

    //! 初期スウィズル値を返します。
    int GetInitialSwizzle() const { return m_InitialSwizzle; }

    //! 初期スウィズル値を設定します。
    void SetInitialSwizzle(const int initialSwizzle) { m_InitialSwizzle = initialSwizzle; }

    //! 編集用コメント文を返します。
    std::string GetCommentText() const { return m_CommentText; }

    //! 編集用コメント文を設定ます。
    void SetCommentText(const std::string& comment) { m_CommentText = comment; }

    //! モデル内でのインデックスを返します。
    int GetOrgIndex() const { return m_OrgIndex; }

    //! モデル内でのインデックスを設定します。
    void SetOrgIndex(const int orgIndex) { m_OrgIndex = orgIndex; }

    //! @brief 画像全体の幅を返します。
    //!        水平十字キューブマップではテクスチャーの幅の 4 倍、
    //!        垂直十字キューブマップではテクスチャーの幅の 3 倍となります。
    //!
    int GetFullW() const { return m_FullW; }

    //! @brief 画像全体の高さを返します。
    //!        水平十字キューブマップではテクスチャーの幅の 3 倍、
    //!        垂直十字キューブマップではテクスチャーの幅の 4 倍となります。
    //!
    int GetFullH() const { return m_FullH; }

    //! アルファ成分が存在するなら true を返します。
    bool HasAlpha() const { return m_HasAlpha; }

    //! データ型が浮動小数点数なら true、整数なら false を返します。
    bool IsFloat() const { return m_IsFloat; }

    //! 透明モードを返します。
    TransparencyMode GetTransparencyMode() const { return m_TransparencyMode; }

    //! 画像全体のビットマップを返します。
    const uint8_t* GetFullBitmap() const { return m_pFullBitmap; }

    //! 画像全体のビットマップを返します。
    uint8_t* GetFullBitmap() { return m_pFullBitmap; }

    //! テクスチャーデータを返します。
    const uint8_t* GetTexelData() const { return m_pTexelData; }

    //! テクスチャーデータのサイズを返します。
    int GetTexelDataSize() const { return m_TexelDataSize; }

    //! @brief TGA ファイルをライトします（動作確認用）。
    RStatus WriteTga(const std::string& path) const;

    //-----------------------------------------------------------------------------
    // static メンバ関数

    //! @brief メインの画像ファイルのパスを返します。
    //!        キューブマップ（6 面個別指定）では +Z 面のパスを返します。
    //!
    //! @param[in] filePaths 画像ファイルのパス配列です。
    //! @param[in] dimension 次元です。
    //!
    //! @return 水平十字キューブマップの幅と高さなら true を返します。
    //!
    static std::string GetMainFilePath(
        const RStringArray& filePaths,
        const Dimension dimension
    )
    {
        const size_t iPath = (dimension == DIM_CUBE_SEPARATE) ? CUBE_FACE_PZ : 0;
        return (iPath < filePaths.size()) ? filePaths[iPath] : "";
    }

    //! @brief オプション文字列からリニア変換フラグを取得します。
    //!
    //! @param[in] str リニア変換フラグを表すオプション文字列です。
    //!
    //! @return リニア変換フラグを返します。
    //!
    static int GetLinearFlagFromOptString(const std::string& str);

    //! @brief リニア変換フラグからオプション文字列を取得します。
    //!
    //! @param[in] linearFlag リニア変換フラグです。
    //!
    //! @return オプション文字列を返します。
    //!
    static std::string GetOptStringFromLinearFlag(const int linearFlag);

    //! @brief 水平十字キューブマップの幅と高さなら true を返します。
    //!
    //! @param[in] imageW 画像の幅です。
    //! @param[in] imageH 画像の高さです。
    //!
    //! @return 水平十字キューブマップの幅と高さなら true を返します。
    //!
    static bool IsCubeHCWH(const int imageW, const int imageH);

    //! @brief 垂直十字キューブマップの幅と高さなら true を返します。
    //!
    //! @param[in] imageW 画像の幅です。
    //! @param[in] imageH 画像の高さです。
    //!
    //! @return 垂直十字キューブマップの幅と高さなら true を返します。
    //!
    static bool IsCubeVCWH(const int imageW, const int imageH);

    //! @brief 画像ファイル数と幅と高さから次元を決定して返します。
    //!
    //! @param[in] fileCount 画像ファイル数です。
    //! @param[in] checksCubeWH 幅と高さから水平／垂直十字キューブマップか判定するなら true を指定します。
    //! @param[in] imageW 画像の幅です。
    //! @param[in] imageH 画像の高さです。
    //!
    //! @return 次元を返します。
    //!
    static Dimension GetDimension(
        const int fileCount,
        const bool checksCubeWH,
        const int imageW,
        const int imageH
    )
    {
        if (fileCount == RImage::CUBE_FACE_COUNT)
        {
            return DIM_CUBE_SEPARATE;
        }
        else if (checksCubeWH)
        {
            if (IsCubeHCWH(imageW, imageH)) return DIM_CUBE_HC;
            if (IsCubeVCWH(imageW, imageH)) return DIM_CUBE_VC;
        }
        return DIM_2D;
    }

    //! @brief 次元がキューブマップなら true を返します。
    //!
    //! @param[in] dimension 次元です。
    //!
    //! @return 次元がキューブマップなら true を返します。
    //!
    static bool IsCubeDimension(const Dimension dimension)
    {
        return (
            dimension == DIM_CUBE_SEPARATE ||
            dimension == DIM_CUBE_HC       ||
            dimension == DIM_CUBE_VC       ||
            dimension == DIM_CUBE_INPUT);
    }

    //! @brief 次元名を返します。
    //!
    //! @param[in] dimension 次元です。
    //!
    //! @return 次元名を返します。
    //!
    static std::string GetDimensionName(const Dimension dimension);

    //! @brief 次元名から次元を返します。
    //!
    //! @param[in] name 次元名です。
    //!
    //! @return 次元を返します。
    //!
    Dimension GetDimensionFromName(const std::string& name);

    //! @brief フォーマット名を返します。
    //!
    //! @param[in] format フォーマットです。
    //!
    //! @return フォーマット名を返します。
    //!
    static std::string GetFormatName(const Format format);

    //! @brief フォーマット名からフォーマットを返します。
    //!
    //! @param[in] name フォーマット名です。
    //!
    //! @return フォーマットを返します。
    //!
    static Format GetFormatFromName(const std::string& name);

    //! @brief フォーマットが BC1/BC2/BC3 フォーマットなら true を返します。
    //!
    //! @param[in] format フォーマットです。
    //!
    //! @return フォーマットが BC1/BC2/BC3 フォーマットなら true を返します。
    //!
    static bool IsBc123Format(const Format format);

    //! @brief フォーマットが BC フォーマットなら true を返します。
    //!
    //! @param[in] format フォーマットです。
    //!
    //! @return フォーマットが BC フォーマットなら true を返します。
    //!
    static bool IsBcFormat(const Format format);

    //! @brief フォーマットが浮動小数点数型なら true を返します。
    //!
    //! @param[in] format フォーマットです。
    //!
    //! @return フォーマットが浮動小数点数型なら true を返します。
    //!
    static bool IsFloatFormat(const Format format);

    //! @brief フォーマットから成分数を取得します。
    //!
    //! @param[in] format フォーマットです。
    //!
    //! @return 成分数を返します。
    //!
    static int GetCompSize(const Format format);

    //! @brief 有効な成分選択なら true を返します。
    //!
    //! @param[in] compSel 成分選択です。
    //!
    //! @return 有効な成分選択なら true を返します。
    //!
    static bool RImage::IsValidCompSel(const std::string& compSel);

    //! @brief 幅と高さからミップマップの最大レベル数を返します。
    //!
    //! @param[in] imageW 画像の幅です。
    //! @param[in] imageH 画像の高さです。
    //!
    //! @return ミップマップの最大レベル数を返します。
    //!
    static int GetMaxMipLevel(const int imageW, const int imageH)
    {
        int mipMax = 1;
        int maxWH = (imageW > imageH) ? imageW : imageH;
        while (maxWH >= 2)
        {
            ++mipMax;
            maxWH >>= 1;
        }
        return mipMax;
    }

protected:
    //! @brief イメージの幅、高さ、奥行きを設定します。
    //!        パス配列、次元、ビットマップの幅と高さを設定してから呼びます。
    //!
    void SetImageWHD();

    //! @brief ビットマップの内容からテクスチャーイメージデータを設定します。
    //!        次元、用途、ビットマップを設定してから呼びます。
    //!
    //! @return 処理結果を返します。
    //!
    RStatus SetImageData();

    //! @brief ビットマップからテクスチャーイメージのフォーマットを設定します。
    void SetFormatFromBitmap();

    //! @brief マージする ftx ファイルから情報（ヒント情報、リニア変換フラグ、
    //!        次元、フォーマット、ミップマップ、初期スウィズル値）を取得します。
    RStatus GetInfoFromMergeFile();

    RStatus ParseAITexelFormat(const uint8_t* dataBuf, const uint32_t dataSize);

    RStatus ParseAICubeMap(const uint8_t* dataBuf, const uint32_t dataSize);

    RStatus ParseAIMipLevel(const uint8_t* dataBuf, const uint32_t dataSize);

    RStatus ParseAITexelData(const uint8_t* dataBuf, const uint32_t dataSize);

    RStatus DecodeAddInfo(const uint8_t* fileBuf, int fileSize);
};

//! @brief テクスチャーイメージ配列の定義です。
typedef std::vector<RImage> RImageArray;

//! @brief テクスチャーイメージのポインタ配列の定義です。
typedef std::vector<RImage*> RImagePtrArray;

//=============================================================================
// グローバル関数の定義です。
//=============================================================================

//! @brief 画像ファイルの情報を取得します。
//!        現在 TGA、PNG、DDS ファイルに対応しています。
//!
//! @param[out] pImageW 画像の幅を格納する変数へのポインタです。nullptr なら格納しません。
//! @param[out] pImageH 画像の高さを格納する変数へのポインタです。nullptr なら格納しません。
//! @param[in] filePath 画像ファイルのフルパスを指定します。
//!
//! @return 処理結果を返します。
//!
RStatus RGetImageInfo(
    int* pImageW,
    int* pImageH,
    const std::string& filePath
);

//! @brief 画像ファイルがキューブマップなら true を返します。
//!        現在キューブマップの DDS ファイルのみ true を返します。
//!
//! @param[in] filePath 画像ファイルのフルパスを指定します。
//!
//! @return 画像ファイルがキューブマップなら true を返します。
//!
bool RIsCubeMapImage(const std::string& filePath);

//! @brief PNG ファイルの 1 成分あたりビット数を返します。
//!
//! @param[in] filePath 画像ファイルのフルパスを指定します。
//!
//! @return PNG ファイルの 1 成分あたりビット数を返します。
//!         ファイルが存在しない場合や PNG ファイルでない場合は 0 を返します。
//!
int RGetPngBitCount(const std::string& filePath);

//=============================================================================
// dcc ネームスペースを終了します。
//=============================================================================
} // namespace dcc
} // namespace tool
} // namespace gfx
} // namespace nn

