﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

// GetKeys GetStepKeys

// GetInput IsValidAnimCurveValue

//=============================================================================
// 処理選択用マクロです。
//=============================================================================

//=============================================================================
// include
//=============================================================================
#include "Animation.h"

using namespace std;
using namespace nn::gfx::tool::dcc;

//=============================================================================
// constants
//=============================================================================

//-----------------------------------------------------------------------------
//! @brief アニメーションカーブの回転補間がクォータニオン補間なら true を返します。
//-----------------------------------------------------------------------------
bool IsQuaternionAnimCurve(const MObject& curveObj)
{
    const int QuaternionTangentDependent = 3; // 接線依存のクォータニオン
    const int QuaternionSlerp            = 4; // クォータニオン球面リニア補間
    const int QuaternionCubic            = 5; // クォータニオンキュービック

    int rotInterp;
    MFnDependencyNode animDepFn(curveObj);
    animDepFn.findPlug("rotationInterpolation").getValue(rotInterp);
    if (rotInterp == QuaternionTangentDependent ||
        rotInterp == QuaternionSlerp            ||
        rotInterp == QuaternionCubic)
    {
        #ifdef DEBUG_PRINT_SW
        cerr << "Quaternion Interpolation: " << animDepFn.name() << R_ENDL;
        #endif
        return true;
    }
    else
    {
        return false;
    }
}

//-----------------------------------------------------------------------------
//! @brief アニメーションカーブが静的なら true を返します。
//-----------------------------------------------------------------------------
bool IsStaticAnimCurve(const MObject& curveObj)
{
    MFnAnimCurve animFn(curveObj);
    return animFn.isStatic();
}

//-----------------------------------------------------------------------------
//! @brief アニメーションカーブのキーがすべてステップ補間なら true を返します。
//-----------------------------------------------------------------------------
bool IsStepAnimCurve(const MObject& curveObj)
{
    MFnAnimCurve animFn(curveObj);
    const int mayaKeyCount = animFn.numKeys();
    for (int iykey = 0; iykey < mayaKeyCount; ++iykey)
    {
        if (animFn.outTangentType(iykey) != MFnAnimCurve::kTangentStep)
        {
            return false;
        }
    }
    return true;
}

//-----------------------------------------------------------------------------
//! @brief 整数サブフレームから浮動小数点数フレームを取得します。
//-----------------------------------------------------------------------------
float GetFloatFrameFromSubFrame(const int subFrame, const void* pParam)
{
    const YExpOpt& yopt = *reinterpret_cast<const YExpOpt*>(pParam);
    const double floatFrame = static_cast<double>(subFrame) / yopt.m_FramePrecision;
    return static_cast<float>(MTime(floatFrame, yopt.m_UIUnitTime).value());
}

//-----------------------------------------------------------------------------
//! @brief NW が対応しているインフィニティタイプなら true を返します。
//-----------------------------------------------------------------------------
static bool IsValidInfinityType(const MFnAnimCurve::InfinityType infinity)
{
    return (
        infinity == MFnAnimCurve::kConstant  ||
        infinity == MFnAnimCurve::kCycle     ||
        infinity == MFnAnimCurve::kOscillate ||
        infinity == MFnAnimCurve::kCycleRelative);
}

//-----------------------------------------------------------------------------
//! @brief Maya のアニメーションカーブを評価した値が有効なら true を返します。
//!        アニメーションカーブを評価した値がベイクしたデータと等しく、
//!        接線タイプとインフィニティがランタイムで再生可能なら有効とみなします。
//-----------------------------------------------------------------------------
bool IsValidAnimCurveValue(
    const MObject& curveObj,
    const RFloatArray& fullValues,
    const float valueScale,
    const float valueOfs,
    const YExpOpt& yopt
)
{
    MFnAnimCurve animFn(curveObj);
    const int keyCount = animFn.numKeys();

    //-----------------------------------------------------------------------------
    // NW が対応していないインフィニティタイプが設定されていて、
    // 出力するフレーム範囲がインフィニティ領域を含むなら無効とみなします。
    if (!IsValidInfinityType(animFn.preInfinityType()))
    {
        const double fitstFrame = animFn.time(0).as(yopt.m_UIUnitTime);
        if (yopt.m_StartFrame < fitstFrame ||
            yopt.m_EndFrame   < fitstFrame)
        {
            return false;
        }
    }

    if (!IsValidInfinityType(animFn.postInfinityType()))
    {
        const double lastFrame = animFn.time(keyCount - 1).as(yopt.m_UIUnitTime);
        if (yopt.m_StartFrame > lastFrame ||
            yopt.m_EndFrame   > lastFrame)
        {
            return false;
        }
    }

    //-----------------------------------------------------------------------------
    // NW が対応していない接線タイプが設定されていれば無効とみなします。
    for (int iykey = 0; iykey < keyCount; ++iykey)
    {
        const MFnAnimCurve::TangentType otType = animFn.outTangentType(iykey);
        if (otType == MFnAnimCurve::kTangentStepNext)
        {
            //g_Err << "step next (bake): " << animFn.name() << R_ENDL;
            return false;
        }
    }

    //-----------------------------------------------------------------------------
    // アニメーションカーブを評価した値とベイクしたデータを比較します。
    const float tolerance = 1.0e-5f;
    const int subFrameCount = static_cast<int>(fullValues.size());
    for (int iFrame = 0; iFrame < subFrameCount; ++iFrame)
    {
        const float floatFrame = yopt.m_StartFrame + GetFloatFrameFromSubFrame(iFrame, &yopt);
        const float value = static_cast<float>(
            animFn.evaluate(MTime(floatFrame, yopt.m_UIUnitTime)) * valueScale + valueOfs);
        //cerr << "ivacv: " << floatFrame << ", " << fullValues[iFrame] << ", " << value << ", " << RAbs(fullValues[iFrame]-value)/RAbs(value) << R_ENDL;
        if (!RIsSameRelative(fullValues[iFrame], value, tolerance))
        {
            return false;
        }
    }

    return true;
}

//-----------------------------------------------------------------------------
//! @brief int 型のアトリビュートの値が全サブフレームで一定なら true を返します。
//-----------------------------------------------------------------------------
bool IsAttributeConstantInt(const MPlug& plug, const YExpOpt& yopt)
{
    //-----------------------------------------------------------------------------
    // check input connection
    MPlugArray plugArray;
    plug.connectedTo(plugArray, true, false);
    if (plugArray.length() == 0)
    {
        return true;
    }

    //-----------------------------------------------------------------------------
    // check value
    int startValue;
    plug.getValue(startValue);
    //g_Err << "start: " << startValue << R_ENDL;

    MObject inputObj = plugArray[0].node();
    MFn::Type type = inputObj.apiType();
    if (type == MFn::kAnimCurveTimeToAngular ||
        type == MFn::kAnimCurveTimeToDistance ||
        type == MFn::kAnimCurveTimeToUnitless)
    {
        MFnAnimCurve curveFn(inputObj);
        for (int iFrame = 0; iFrame < yopt.m_SubFrameCount; ++iFrame)
        {
            double floatFrame = yopt.m_StartFrame + static_cast<double>(iFrame) / yopt.m_FramePrecision;
            MTime time(floatFrame, yopt.m_UIUnitTime);
            const int value = RRound(curveFn.evaluate(time));
            //g_Err << "val: " << yopt.m_StartFrame + iFrame << ": " << value << R_ENDL;
            if (value != startValue)
            {
                return false;
            }
        }
    }
    else
    {
        for (int iFrame = 0; iFrame < yopt.m_SubFrameCount; ++iFrame)
        {
            double floatFrame = yopt.m_StartFrame + static_cast<double>(iFrame) / yopt.m_FramePrecision;
            MTime time(floatFrame, yopt.m_UIUnitTime);
            int value;
            MDGContext ctx(time);
            #if (MAYA_API_VERSION >= 20180000)
            MDGContextGuard contextGuard(ctx);
            plug.getValue(value);
            #else
            plug.getValue(value, ctx);
            #endif
            if (value != startValue)
            {
                return false;
            }
        }
    }

    return true;
}

//-----------------------------------------------------------------------------
//! @brief アニメーションレイヤのブレンドノードなら true を返します。
//-----------------------------------------------------------------------------
bool IsAnimLayerBlendNode(const MObject& obj)
{
    const MFn::Type type = obj.apiType();
    return (
        type == MFn::kBlendNodeDouble           || // animBlendNodeAdditive
        type == MFn::kBlendNodeDoubleAngle      || // animBlendNodeAdditiveDA
        type == MFn::kBlendNodeDoubleLinear     || // animBlendNodeAdditiveDL
        type == MFn::kBlendNodeFloat            || // animBlendNodeAdditiveF
        type == MFn::kBlendNodeFloatAngle       || // animBlendNodeAdditiveFA
        type == MFn::kBlendNodeFloatLinear      || // animBlendNodeAdditiveFL
        type == MFn::kBlendNodeInt16            || // animBlendNodeAdditiveI16
        type == MFn::kBlendNodeInt32            || // animBlendNodeAdditiveI32
        type == MFn::kBlendNodeAdditiveRotation || // animBlendNodeAdditiveRotation
        type == MFn::kBlendNodeAdditiveScale    || // animBlendNodeAdditiveScale
        type == MFn::kBlendNodeBoolean          || // animBlendNodeBoolean
        type == MFn::kBlendNodeEnum);              // animBlendNodeEnum
}

//-----------------------------------------------------------------------------
//! @brief アニメーションレイヤブレンドのウェイトが 0 でなければ true を返します。
//!
//! @param[in] weightPlug ウェイトアトリビュートのプラグです。
//!
//! @return ウェイトが 0 でなければ true を返します。
//-----------------------------------------------------------------------------
static bool IsNonZeroAnimBlendWeight(const MPlug& weightPlug)
{
    double weight;
    weightPlug.getValue(weight);
    if (weight != 0.0)
    {
        return true;
    }

    if (weightPlug.partialName(false, false, false, false, false, true) == "weightB")
    {
        // ウェイトアトリビュートの値が 0 であっても、
        // B に接続されているアニメーションレイヤノードがミュートでなく、
        // アニメーションレイヤノードのウェイトがアニメーションしていれば
        // アニメーションレイヤブレンドのウェイトが 0 でないとみなします。
        MPlugArray plugArray;
        weightPlug.connectedTo(plugArray, true, false);
        if (plugArray.length() > 0 && plugArray[0].node().apiType() == MFn::kAnimLayer)
        {
            MFnDependencyNode layerFn(plugArray[0].node());
            bool isMuted;
            layerFn.findPlug("outMute").getValue(isMuted);
            if (!isMuted)
            {
                layerFn.findPlug("weight").connectedTo(plugArray, true, false);
                if (plugArray.length() > 0)
                {
                    const MObject inputObj = plugArray[0].node();
                    if (inputObj.hasFn(MFn::kAnimCurve) ||
                        inputObj.apiType() == MFn::kExpression)
                    {
                        //cerr << "anim layer (weight animated): " << layerFn.name() << ": " << weightPlug.name() << endl;
                        return true;
                    }
                }
            }
        }
    }

    return false;
}

//-----------------------------------------------------------------------------
//! @brief アニメーションレイヤブレンドの有効な（ウェイトが 0 でない）入力に
//!        アニメーションが設定されていれば true を返します。
//!        レイヤがミュートされたりソロの対象外になるとウェイトは 0 になります。
//!        再帰的に呼ばれます。
//!
//! @param[in] blendObj アニメーションレイヤブレンドオブジェクトです。
//!
//! @return 有効な入力にアニメーションが設定されていれば true を返します。
//-----------------------------------------------------------------------------
static bool EffectiveAnimLayerInputIsAnimated(const MObject& blendObj)
{
    // アニメーションレイヤが 1 つの場合
    // plug <- animBlendNodeAdditive*.inputB <- layer1Curve
    //                               .inputA <- baseCurve

    // アニメーションレイヤが 2 つの場合
    // plug <- animBlendNodeAdditive*.inputB <- layer1Curve
    //                               .inputA <- animBlendNodeAdditive*.inputB <- layer2Curve
    //                                          animBlendNodeAdditive*.inputA <- baseCurve

    // inputA の階層が深くなっていくので、先に inputB をチェックします。
    const int INPUT_COUNT = 2;
    static const char* const weightAttrs[INPUT_COUNT] = { "weightB", "weightA" };
    static const char* const inputAttrs[INPUT_COUNT]  = { "inputB" , "inputA"  };

    if (IsAnimLayerBlendNode(blendObj))
    {
        MFnDependencyNode blendFn(blendObj);
        for (int iInput = 0; iInput < INPUT_COUNT; ++iInput)
        {
            if (IsNonZeroAnimBlendWeight(blendFn.findPlug(weightAttrs[iInput])))
            {
                MPlugArray plugArray;
                blendFn.findPlug(inputAttrs[iInput]).connectedTo(plugArray, true, false);
                if (plugArray.length() > 0)
                {
                    const MObject inputObj = plugArray[0].node();
                    if (inputObj.hasFn(MFn::kAnimCurve) ||
                        inputObj.apiType() == MFn::kExpression)
                    {
                        //cerr << "effective anim blend: " << blendFn.name() << "." << inputAttrs[iInput] << endl;
                        return true;
                    }
                    else if (EffectiveAnimLayerInputIsAnimated(inputObj))
                    {
                        return true;
                    }
                }
            }
        }
    }
    return false;
}

//-----------------------------------------------------------------------------
//! @brief アニメーションレイヤが有効なら true を返します。
//!        ミュートされていなくてソロの対象外でないレイヤにキーがあれば有効とみなします。
//-----------------------------------------------------------------------------
bool IsEffectiveAnimLayer(const MPlug& plug, const MObject& blendObj)
{
    //-----------------------------------------------------------------------------
    // どのレイヤにもキーがない場合をチェックします。
    //if (!MAnimUtil::isAnimated(plug, false)) // 全レイヤがミュートされている場合でもキーがあれば true を返します。
    //{
    //  return false;
    //}
    R_UNUSED_VARIABLE(plug);

    //-----------------------------------------------------------------------------
    // アニメーションレイヤブレンドの有効な入力にアニメーションが設定されているかチェックします。
    return EffectiveAnimLayerInputIsAnimated(blendObj);
}

#if (MAYA_API_VERSION >= 201700)
//-----------------------------------------------------------------------------
//! @brief タイムエディタのインターポーレーターに接続されたエヴァリュエーター群を取得します。
//!        再帰的に呼ばれます。
//!
//! @param[in,out] pEvaluators 取得したエヴァリュエーターを追加する配列へのポインターです。
//! @param[in] interpolatorObj タイムエディタのインターポーレーターノードです。
//-----------------------------------------------------------------------------
static void GetTimeEditorEvaluators(MObjectArray* pEvaluators, const MObject& interpolatorObj)
{
    const MPlug inputPlug = MFnDependencyNode(interpolatorObj).findPlug("input");
    const int inputCount = static_cast<int>(inputPlug.numElements());
    for (int inputIdx = 0; inputIdx < inputCount; ++inputIdx)
    {
        MPlugArray plugArray;
        inputPlug.elementByPhysicalIndex(inputIdx).connectedTo(plugArray, true, false);
        if (plugArray.length() != 0)
        {
            const MObject inObj = plugArray[0].node();
            if (inObj.apiType() == MFn::kTimeEditorClipEvaluator)
            {
                pEvaluators->append(inObj);
            }
            else if (inObj.apiType() == MFn::kTimeEditorInterpolator)
            {
                GetTimeEditorEvaluators(pEvaluators, inObj);
            }
        }
    }
}

//-----------------------------------------------------------------------------
//! @brief タイムエディタのクリップが有効（ミュートされていない）か取得します。
//!        再帰的に呼ばれます。
//!
//! @param[in] clipObj タイムエディタのクリップノードです。
//!
//! @return クリップが有効なら true を返します。
//-----------------------------------------------------------------------------
static bool IsEffectiveTimeEditorClip(const MObject& clipObj)
{
    static int s_ClipMutedIdx  = -1; // clip[0] 内の clipMuted  プラグのインデックスです。
    static int s_ClipParentIdx = -1; // clip[0] 内の clipParent プラグのインデックスです。

    const MFnDependencyNode clipFn(clipObj);
    const MPlug clip0Plug = clipFn.findPlug("clip").elementByLogicalIndex(0);
    if (s_ClipMutedIdx == -1)
    {
        s_ClipMutedIdx = FindChildPlugIndexByName(clip0Plug, "clipMuted");
    }
    if (s_ClipParentIdx == -1)
    {
        s_ClipParentIdx = FindChildPlugIndexByName(clip0Plug, "clipParent");
    }
    if (clipFn.findPlug("clipTrackMuted").asBool() ||
        clipFn.findPlug("clipSoloMuted").asBool()  ||
        clip0Plug.child(s_ClipMutedIdx).asBool())
    {
        return false;
    }

    MPlugArray plugArray;
    clip0Plug.child(s_ClipParentIdx).connectedTo(plugArray, true, false);
    if (plugArray.length() != 0)
    {
        const MObject parentObj = plugArray[0].node();
        if (parentObj.apiType() == MFn::kTimeEditorClip)
        {
            return IsEffectiveTimeEditorClip(parentObj);
        }
    }
    return true;
}

//-----------------------------------------------------------------------------
//! @brief タイムエディタのインターポーレータに有効なクリップが接続されているか取得します。
//!
//! @param[in] interpolatorObj タイムエディタのインターポーレータノードです。
//!
//! @return 有効なクリップが接続されていれば true を返します。
//-----------------------------------------------------------------------------
static bool TimeEditorInterpolatorHasEffectiveClip(const MObject& interpolatorObj)
{
    //-----------------------------------------------------------------------------
    // インターポーレータに接続されたエヴァリュエータ群を取得します。
    MObjectArray evaluators;
    GetTimeEditorEvaluators(&evaluators, interpolatorObj);

    //-----------------------------------------------------------------------------
    // いずれかのエヴァリュエータに有効なクリップが接続されているか取得します。
    const int evaluatorCount = static_cast<int>(evaluators.length());
    for (int evaluatorIdx = 0; evaluatorIdx < evaluatorCount; ++evaluatorIdx)
    {
        const MPlug pcsPlug = MFnDependencyNode(evaluators[evaluatorIdx]).findPlug("parentContainerState");
        MPlugArray plugArray;
        pcsPlug.connectedTo(plugArray, true, false);
        if (plugArray.length() != 0)
        {
            const MObject clipObj = plugArray[0].node();
            if (clipObj.apiType() == MFn::kTimeEditorClip)
            {
                const bool isEffectiveClip = IsEffectiveTimeEditorClip(clipObj);
                if (isEffectiveClip)
                {
                    return true;
                }
            }
        }
    }
    return false;
}
#endif

//-----------------------------------------------------------------------------
//! @brief プラグからチャンネル入力情報を取得します。
//-----------------------------------------------------------------------------
void ChannelInput::GetInput(const MPlug& plug)
{
    //-----------------------------------------------------------------------------
    // init
    m_CurveFlag = false;
    m_CurveObj = MObject::kNullObj;
    m_CurveWeighted = false;
    m_DrivenKey = false;
    m_Expression = false;
    m_Character = false;
    m_AnimLayer = false;
    m_TimeEditor = false;
    m_Constraint = false;

    //-----------------------------------------------------------------------------
    // エクスプレッション、コンストレイン、アニメーションレイヤが接続されていれば
    // 各フラグを ON にして return します。
    // モーションパス、リジッドボディが接続されていれば何もしないで return します。
    // キャラクタが接続されていればフラグを ON にして次の処理に移行します。
    MPlugArray plugArray;
    plug.connectedTo(plugArray, true, false);
    if (plugArray.length() > 0)
    {
        const MObject obj = plugArray[0].node();
        const MFn::Type type = obj.apiType();

        if (type == MFn::kExpression)
        {
            m_Expression = true;
            return;
        }
        else if (type == MFn::kCharacter)
        {
            m_Character = true;
        }
        else if (type == MFn::kPointConstraint  ||  // constraint
                 type == MFn::kAimConstraint    ||
                 type == MFn::kOrientConstraint ||
                 type == MFn::kScaleConstraint  ||
                 type == MFn::kParentConstraint ||
                 type == MFn::kNormalConstraint ||
                 type == MFn::kTangentConstraint)
        {
            m_Constraint = true;
            return;
        }
        else if (type == MFn::kMotionPath)
        {
            return;
        }
        else if (type == MFn::kAddDoubleLinear ||   // motion path
                 type == MFn::kChoice               // rigid body
                 )
        {
            return;
        }
        else if (IsAnimLayerBlendNode(obj))
        {
            m_AnimLayer = IsEffectiveAnimLayer(plug, obj);
            //if (!m_AnimLayer) cerr << "anim layer is muted: " << plug.info() << endl;
            return;
        }
        #if (MAYA_API_VERSION >= 201700)
        else if (type == MFn::kTimeEditorInterpolator)
        {
            m_TimeEditor = TimeEditorInterpolatorHasEffectiveClip(obj);
            return;
        }
        #endif
        else if (type == MFn::kUnitConversion)
        {
            // unit conversion 経由でエクスプレッションが接続されている場合
            MPlug convPlug = MFnDependencyNode(obj).findPlug("input");
            convPlug.connectedTo(plugArray, true, false);
            if (plugArray.length() > 0 &&
                plugArray[0].node().apiType() == MFn::kExpression)
            {
                m_Expression = true;
                return;
            }
        }
    }

    //-----------------------------------------------------------------------------
    // アニメーションカーブを取得します。
    MObjectArray objArray;
    if (MAnimUtil::findAnimation(plug, objArray))
    {
        const MFn::Type type = objArray[0].apiType();
        if (type == MFn::kAnimCurveTimeToAngular ||
            type == MFn::kAnimCurveTimeToDistance ||
            type == MFn::kAnimCurveTimeToUnitless)
        {
            MObject curveObj = objArray[0];
            MFnAnimCurve animFn(curveObj);

            //-----------------------------------------------------------------------------
            // check node state
            if (!IsNodeStateEffective(curveObj))
            {
                return;
            }

            //-----------------------------------------------------------------------------
            // check curve is connected directory
            //bool directFlag = false;
            //const MObject& plugObj = plug.node();
            //MPlug outputPlug = animFn.findPlug("output");
            //MPlugArray plugArray;
            //outputPlug.connectedTo(plugArray, false, true);
            //if (plugArray.length() > 0)
            //{
            //  int targetCount = plugArray.length();
            //  for (int itarget = 0; itarget < targetCount; ++itarget)
            //  {
            //      MObject targetObj = plugArray[itarget].node();
            //      if (targetObj == plugObj)
            //      {
            //          directFlag = true;
            //          break;
            //      }
            //  }
            //}
            //if (!directFlag)
            //{
            //  #ifdef DEBUG_PRINT_SW
            //  // character を使用していて character に直接キーが打たれている場合もここに
            //  g_Err << "not direct: " << animFn.name() << " -> " << plug.info() << R_ENDL;
            //  #endif
            //  return;
            //}

            //-----------------------------------------------------------------------------
            // curve is OK
            //g_Err << "curve: " << animFn.name() << " -> " << plug.info() << R_ENDL;
            m_CurveFlag = true;
            m_CurveObj = curveObj;
            m_CurveWeighted = animFn.isWeighted();
            #ifdef DEBUG_PRINT_SW
            if (m_CurveWeighted)
            {
                cerr << "weighted: " << animFn.name() << " -> " << plug.info() << R_ENDL;
            }
            #endif
        }
        else
        {
            m_DrivenKey = true;
        }
    }
}

//-----------------------------------------------------------------------------
//! @brief Maya のタンジェントタイプから RAnimKey のタンジェントタイプを取得します。
//-----------------------------------------------------------------------------
static RAnimKey::TangentType GetTangentType(const MFnAnimCurve::TangentType mayaTanType)
{
    if (mayaTanType == MFnAnimCurve::kTangentLinear)
    {
        return RAnimKey::LINEAR;
    }
    else if (mayaTanType == MFnAnimCurve::kTangentStep)
    {
        return RAnimKey::STEP;
    }
    else
    {
        return RAnimKey::HERMITE;
    }
}

//-----------------------------------------------------------------------------
//! @brief キーのスロープ値を取得します。
//-----------------------------------------------------------------------------
static RAnimKey::TangentType GetKeySlope(
    YAnimKey& ykey,
    const MFnAnimCurve& animFn,
    const int iykey,
    const float slopeScale,
    const RAnimKey::TangentType lastOtType
)
{
    ykey.m_InType = GetTangentType(animFn.inTangentType(iykey));
    ykey.m_OtType = GetTangentType(animFn.outTangentType(iykey));

    float inX;
    float inY;
    float otX;
    float otY;
    #if (MAYA_API_VERSION >= 20180000)
    MFnAnimCurve::TangentValue tanInX;
    MFnAnimCurve::TangentValue tanInY;
    MFnAnimCurve::TangentValue tanOtX;
    MFnAnimCurve::TangentValue tanOtY;
    animFn.getTangent(iykey, tanInX, tanInY, true);
    animFn.getTangent(iykey, tanOtX, tanOtY, false);
    inX = static_cast<float>(tanInX);
    inY = static_cast<float>(tanInY);
    otX = static_cast<float>(tanOtX);
    otY = static_cast<float>(tanOtY);
    #else
    animFn.getTangent(iykey, inX, inY, true);
    animFn.getTangent(iykey, otX, otY, false);
    #endif
    ykey.m_InSlope = (inX != 0.0f) ? inY / inX : 0.0f;
    ykey.m_OtSlope = (otX != 0.0f) ? otY / otX : 0.0f;
    ykey.m_InSlope *= slopeScale;
    ykey.m_OtSlope *= slopeScale;

    RAnimKey::TangentType retOtType = lastOtType;
    if (lastOtType == RAnimKey::STEP)
    {
        ykey.m_InType = RAnimKey::STEP; // Maya 上では Fixed になっています。
        ykey.m_InSlope = 0.0f;
    }
    if (lastOtType != ykey.m_InType && lastOtType != RAnimKey::STEP)
    {
        // LINEAR と HERMITE の補間は HERMITE 同士の補間に変更します。
        ykey.m_InType = retOtType = RAnimKey::HERMITE;
    }

    if (ykey.m_OtType == RAnimKey::STEP)
    {
        ykey.m_OtSlope = 0.0f;
    }

    return retOtType;
}

//-----------------------------------------------------------------------------
//! @brief 角度のアニメーションキーの値を絶対値が小さくなるようにシフトします。
//-----------------------------------------------------------------------------
static void ShiftAngleKeyValue(YAnimKeyArray& ykeys, const std::string& curveName)
{
    //-----------------------------------------------------------------------------
    // get value range
    float valueMin = static_cast<float>(LONG_MAX);
    float valueMax = static_cast<float>(LONG_MIN);
    const int keyCount = static_cast<int>(ykeys.size());
    for (int iKey = 0; iKey < keyCount; ++iKey)
    {
        const float value = ykeys[iKey].m_Value;
        if (value < valueMin)
        {
            valueMin = value;
        }
        if (value > valueMax)
        {
            valueMax = value;
        }
    }

    //-----------------------------------------------------------------------------
    // shift
    const float valueCenter = (valueMin + valueMax) / 2.0f;
    if (valueCenter <= -360.0f || valueCenter >= 360.0f)
    {
        const int ishift = RRound(valueCenter / 360.0f);
        const float valueOfs = -360.0f * ishift;
        for (int iKey = 0; iKey < keyCount; ++iKey)
        {
            ykeys[iKey].m_Value += valueOfs;
        }
        #ifdef DEBUG_PRINT_SW
        cerr << "shift angle key: " << curveName << ": " << valueOfs << R_ENDL;
        #else
        R_UNUSED_VARIABLE(curveName);
        #endif
    }
}

//-----------------------------------------------------------------------------
//! @brief Maya から取得したキー配列を出力用のキー配列に変換します。
//-----------------------------------------------------------------------------
static void GetRAnimKeysFromMayaKeys(
    RAnimKeyArray& dstKeys,
    const YAnimKeyArray& srcKeys
)
{
    dstKeys.clear();

    const int keyCount = static_cast<int>(srcKeys.size());
    for (int iKey = 0; iKey < keyCount; ++iKey)
    {
        const YAnimKey& ykey = srcKeys[iKey];
        const RAnimKey::TangentType type = ykey.m_OtType;

        RAnimKey rkey(ykey.m_Frame, ykey.m_Value,
            type, ykey.m_InSlope, ykey.m_OtSlope);
        dstKeys.push_back(rkey);
    }
}

//-----------------------------------------------------------------------------
//! Maya のインフィニティタイプからアニメーションカーブのラップモードを取得します。
//-----------------------------------------------------------------------------
static RAnimCurve::Wrap GetAnimWrapFromInfinity(
    const MFnAnimCurve::InfinityType infinity
)
{
    switch (infinity)
    {
    case MFnAnimCurve::kCycle:          return RAnimCurve::REPEAT;
    case MFnAnimCurve::kOscillate:      return RAnimCurve::MIRROR;
    case MFnAnimCurve::kCycleRelative:  return RAnimCurve::RELATIVE_REPEAT;
    default:                            return RAnimCurve::CLAMP;
    }
}

//-----------------------------------------------------------------------------
//! MFnAnimCurve からアニメーションカーブのラップモードを取得します。
//-----------------------------------------------------------------------------
static void GetAnimWrap(
    RAnimCurve::Wrap& preRepeat,
    RAnimCurve::Wrap& postRepeat,
    const MFnAnimCurve& animFn
)
{
    preRepeat = postRepeat = RAnimCurve::CLAMP;
    const int keyCount = animFn.numKeys();
    if (keyCount >= 2)
    {
        // 現状、出力範囲がインフィニティの領域を含まなくても
        // リピートやミラーとして出力します。
        preRepeat  = GetAnimWrapFromInfinity(animFn.preInfinityType());
        postRepeat = GetAnimWrapFromInfinity(animFn.postInfinityType());
    }
}

//-----------------------------------------------------------------------------
//! @brief 出力するフレーム範囲に対応した Maya のキーのインデックスを取得します。
//-----------------------------------------------------------------------------
static void GetMayaKeyRange(
    int& iykeyStart,
    int& iykeyEnd,
    const MFnAnimCurve& animFn,
    const bool loopFlag,
    const YExpOpt& yopt
)
{
    R_UNUSED_VARIABLE(loopFlag);

    const int keyEndFrame = yopt.m_EndFrame;
    //const int keyEndFrame = (loopFlag) ? yopt.m_EndFrame + 1 : yopt.m_EndFrame;

    const int mayaKeyCount = animFn.numKeys();

    iykeyStart = animFn.findClosest(MTime(static_cast<double>(yopt.m_StartFrame), yopt.m_UIUnitTime));
    iykeyEnd   = animFn.findClosest(MTime(static_cast<double>(keyEndFrame), yopt.m_UIUnitTime));

    if (iykeyStart > 0 &&
        static_cast<int>(animFn.time(iykeyStart).as(yopt.m_UIUnitTime)) > yopt.m_StartFrame)
    {
        --iykeyStart;
    }

    if (iykeyEnd < mayaKeyCount - 1 &&
        static_cast<int>(animFn.time(iykeyEnd).as(yopt.m_UIUnitTime)) < keyEndFrame)
    {
        ++iykeyEnd;
    }
}

//-----------------------------------------------------------------------------
//! @brief Maya からアニメーションキー配列を取得します。
//!        呼ぶ前に m_LoopFlag, m_AngleFlag をセットしておく必要があります。
//-----------------------------------------------------------------------------
void YAnimCurve::GetKeys(
    const MObject& curveObj,
    const float valueScale,
    const float valueOfs,
    const YExpOpt& yopt
)
{
    //-----------------------------------------------------------------------------
    // ラップモードと取得する Maya のキーの範囲を取得します。
    const MFnAnimCurve animFn(curveObj);
    GetAnimWrap(m_PreWrap, m_PostWrap, animFn);

    int iykeyStart, iykeyEnd;
    if (m_PreWrap  == RAnimCurve::CLAMP &&
        m_PostWrap == RAnimCurve::CLAMP)
    {
        GetMayaKeyRange(iykeyStart, iykeyEnd, animFn, m_LoopFlag, yopt);
    }
    else // ラップモードが CLAMP でなければすべてのキーを出力します。
    {
        iykeyStart = 0;
        iykeyEnd = animFn.numKeys() - 1;
    }

    //-----------------------------------------------------------------------------
    // get maya keys
    const float slopeScale = valueScale / yopt.m_FramesPerSecond;

    YAnimKeyArray mayaKeys;
    RAnimKey::TangentType lastOtType = RAnimKey::HERMITE;
    for (int iykey = iykeyStart; iykey <= iykeyEnd; ++iykey)
    {
        YAnimKey ykey(
            static_cast<float>(animFn.time(iykey).as(yopt.m_UIUnitTime) - yopt.m_StartFrame),
            static_cast<float>(animFn.value(iykey) * valueScale + valueOfs), 0.0f);
        const RAnimKey::TangentType retOtType =
            GetKeySlope(ykey, animFn, iykey, slopeScale, lastOtType);
        if (static_cast<int>(mayaKeys.size()) > 0)
        {
            mayaKeys[mayaKeys.size() - 1].m_OtType = retOtType;
        }
        ykey.SnapToZero();
        mayaKeys.push_back(ykey);
        lastOtType = ykey.m_OtType;
    }

    //-----------------------------------------------------------------------------
    // shift angle key value
    if (m_AngleFlag)
    {
        ShiftAngleKeyValue(mayaKeys, m_Name);
    }

    //-----------------------------------------------------------------------------
    // get dcc anim keys
    GetRAnimKeysFromMayaKeys(m_Keys, mayaKeys);
}

//-----------------------------------------------------------------------------
//! @brief Maya からステップ補間用のアニメーションキー配列を取得します。
//-----------------------------------------------------------------------------
void YAnimCurve::GetStepKeys(const MObject& curveObj, const YExpOpt& yopt)
{
    //-----------------------------------------------------------------------------
    // ラップモードと取得する Maya のキーの範囲を取得します。
    const MFnAnimCurve animFn(curveObj);
    GetAnimWrap(m_PreWrap, m_PostWrap, animFn);

    int iykeyStart;
    int iykeyEnd;
    if (m_PreWrap  == RAnimCurve::CLAMP &&
        m_PostWrap == RAnimCurve::CLAMP)
    {
        GetMayaKeyRange(iykeyStart, iykeyEnd, animFn, false, yopt);
    }
    else // ラップモードが CLAMP でなければすべてのキーを出力します。
    {
        iykeyStart = 0;
        iykeyEnd = animFn.numKeys() - 1;
    }

    //-----------------------------------------------------------------------------
    // get step keys
    for (int iykey = iykeyStart; iykey <= iykeyEnd; ++iykey)
    {
        RAnimKey key;
        key.m_Frame = static_cast<float>(animFn.time(iykey).as(yopt.m_UIUnitTime) - yopt.m_StartFrame);
        key.m_Value = static_cast<float>(animFn.value(iykey));
        key.m_Type = RAnimKey::STEP;
        m_Keys.push_back(key);
    }
}

