﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

/******************************************************************************
    include
******************************************************************************/
#include "DccUtilityCommon.h"
#include "DccUtilityMaterial.h"

#include "DccMaterial.h"
#include "DccImage.h"
#include "DccAnimation.h"

using namespace std;

/******************************************************************************
    begin name space utility
******************************************************************************/
namespace nn {
namespace gfx {
namespace tool {
namespace dcc {
namespace utility {

//=============================================================================
//! @brief テクスチャパターンアニメーション関連のデータを保持するクラスです。
//=============================================================================
class TexPatAnim
{
  public:
    // texture list
    //nn::gfx::tool::dcc::RIntArray m_UsedFes;    //!< 使用するフレーム拡張子の配列です。
    nn::gfx::tool::dcc::RIntArray m_TexImgIdxs; //!< m_UsedFes に対応するテクスチャイメージのモデル内のインデックス配列です。

    // anim
    //nn::gfx::tool::dcc::RAnimCurve m_FeAnim;  //!< フレーム拡張子のアニメーションカーブです。
    nn::gfx::tool::dcc::RAnimCurve m_ImgAnim; //!< 出力するイメージのアニメーションカーブです。

    // internal
    //MPlug m_FePlug;        //!< フレーム拡張子プラグです。
    //bool m_ValidCurveFlag; //!< フレーム拡張子プラグに有効なアニメーションカーブ（animCurveTU）が接続されていれば true です。
    //MObject m_InputObj;    //!< フレーム拡張子に接続されたアニメーションカーブまたはエクスプレッションのノードです。
    std::string m_ObjName; //!< Maya のテクスチャオブジェクト名です。
};


//! @brief テクスチャパターンアニメーション配列の定義です。
typedef std::vector<TexPatAnim> TexPatAnimArray;

//=============================================================================
//! @brief テクスチャSRTアニメーション関連のデータを保持するクラスです。
//=============================================================================
class TexSrtAnim
{
public:
    // anim
    //MPlug m_AnimPlugs[ROriginalTexsrt::PARAM_COUNT]; //!< パラメータに対応する Maya のプラグ配列です。
    //ChannelInput m_AnimChans[ROriginalTexsrt::PARAM_COUNT]; //!< パラメータに対応するチャンネル入力情報配列です。
    nn::gfx::tool::dcc::RAnimCurve m_Anims[nn::gfx::tool::dcc::ROriginalTexsrt::PARAM_COUNT]; //!< パラメータに対応するアニメーションカーブ配列です。

    // internal
    //MObject m_PlaceObj; //!< Maya の place2dTexture ノードです。
    std::string m_ObjName; //!< Maya の place2dTexture ノード名です。

public:
    //! @brief コンストラクタです。
    //!
    //! @param[in] placeObj Maya の place2dTexture ノードです。
    //! @param[in] materialIndex マテリアルのインデックスです。
    //!
#if 0
    TexSrtAnim(const MObject& placeObj)
        : m_PlaceObj(placeObj)
    {
        m_PlaceNodeName = MFnDependencyNode(placeObj).name().asChar();
    }
#endif
};

//! @brief Maya 用のテクスチャ SRT アニメーション配列の定義です。
typedef std::vector<TexSrtAnim> TexSrtAnimArray;

/**
シーンに存在する複数のテクスチャファイルを扱うためのクラス
\par 解説：
　シーンのエクスポート時に RSceneMaterials から テクスチャファイル(CTEX)を出力する。
DCCツールのエキスポータプラグインはマテリアルにファイルを指定することで自動的にこのクラスに
ファイルが登録される。
ファイルの重複を防ぐためにファイルのフルパスをキーに登録される。
*/
class RTextureLibrary
{
public:
    /**
    コンストラクタ
    */
    RTextureLibrary(void)
    {
    };

    /**
    デストラクタ
    */
    ~RTextureLibrary(void)
    {
        init();
    };

    //
    void init(void);

    /**
    ライブラリにテクスチャファイルを登録する。
    \param[out] retIndex 登録されたテクスチャのリスト内のインデックス
    \param[in] テクスチャファイルのパスリスト。キューブマップのときなどに複数になります。
    \param[in] hint テクスチャの関連付けに利用するヒント情報です。
    \return 処理結果を返します。
    \par 解説：
    　テクスチャファイルをライブラリに登録します。
    */
    bool addTextureFile( int& retIndex, const nn::gfx::tool::dcc::RStringArray& filePaths, const nn::gfx::tool::dcc::RSampler& sampler, const int curSwizzle, const nn::gfx::tool::dcc::RExpOpt& rOpt);

    /**
    テクスチャファイルの数を返す。
    \return 登録されたテクスチャファイルの数
    \par 解説：
    　登録されたテクスチャファイルの数を返す。
    */
    const int getNumTexture(void) const
    {
        return static_cast<int>(m_TexImgs.size());
    }

    /**
    テクスチャのインスタンスを取得する。
    \return テクスチャへの参照
    \par 解説：
    　登録されているテクスチャのインスタンスを取得する。
    */
    nn::gfx::tool::dcc::RImage& getTexture(int id)
    {
        return m_TexImgs[id];
    }

private:

    /**
        出力するシーンで使用されているテクスチャリスト
    */
    nn::gfx::tool::dcc::RImageArray	m_TexImgs;



};


/**
シーンに存在する複数のマテリアルを扱うためのクラス
\par 解説：
　シーンのエクスポート時に RSceneMaterials のインスタンスを作成し、全てのマテリアル
の管理と <Materials> <MaterialCtr> タグを出力する。
DCCツールのエキスポータプラグインはマテリアルのエキスポートにこのクラスを利用できる。
DCCツールSDKのマテリアルのポインタをキーにして、
FMaterialとそのインデックスを取得してマテリアルの設定を行う。
*/
class RSceneMaterials
{
public:
    /**
    コンストラクタ
    */
    RSceneMaterials(void);

    /**
    デストラクタ
    */
    ~RSceneMaterials(void);

    /**
    初期化
    */
    void init(void);


    /**
    任意のポインタに対応する FMaterial を返す。
    \param[in] addr 任意のポインタ
    \return 対応する FMaterial のポインタ
    \par 解説：
    　渡された任意のポインタに対応するFMaterialを返す。
    対応するFMaterialが存在しない場合は作成してから返す。
    <br />
    　この任意のポインタは、
    DCCツールSDKのマテリアルを表す構造体もしくはクラスオブジェクトのポインタを意図している。
    この関数はDCCツールSDKのマテリアル(のポインタ)からFMaterialを参照する機能として使える。
    */
    FMaterial* getFMaterial(const void* addr)
    {
        return getMaterialInfo(addr)->material;
    }

    /**
    任意のポインタに対応する FMaterial のインデックスを返す。
    \param[in] addr 任意のポインタ
    \return 対応する FMaterial のインデックス
    \par 解説：
    　渡された任意のポインタに対応するFMaterialのインデックスを返す。
    対応するFMaterialが存在しない場合は作成してから返す。
    インデックスはシーン内で一意であり、他のマテリアルとは重複しない。
    <br />
    　この任意のポインタは、
    DCCツールSDKのマテリアルを表す構造体もしくはクラスオブジェクトのポインタを意図している。
    この関数はDCCツールSDKのマテリアル(のポインタ)からFMaterialを参照する機能として使える。
    */
    const int getFMaterialIndex(const void* addr)
    {
        return getMaterialInfo(addr)->index;
    }

    /**
    任意のポインタに対応する FMaterial が存在するか？
    \param[in] addr 任意のポインタ
    \return 対応する FMaterial の存在を表すbool値
    \par 解説：
    　渡された任意のポインタに対応するFMaterialが存在すればtrue、
    存在しなければfalseを返す。
    <br />
    　この任意のポインタは、
    DCCツールSDKのマテリアルを表す構造体もしくはクラスオブジェクトのポインタを意図している。
    この関数はDCCツールSDKのマテリアル(のポインタ)からFMaterialを参照する機能として使える。
    */
    const bool isExistFMaterial(const void* addr) const;

    /**
    インデックスに対応する FMaterial を返す。
    \param[in] index
    \return 対応する FMaterial のポインタ
    \par 解説：
    　渡されたインデックスに対応するFMaterialを返す。
    登録されていない場合はNULLを返す
    */
    FMaterial* getFMaterialByIndex(int index) const;

    /**
    FMaterialの数を返す。
    \return 登録されたFMaterialの数
    \par 解説：
    　登録されたFMaterialの数を返す。
    */
    const int getNumFMaterials(void) const
    {
        return static_cast<int>(materialMap.size());
    }

    /**
    マテリアルアニメーションの出力準備をします。
    OutMaterialAnimations を呼ぶ前に一度呼び出す必要があります。
    \param[in] rOpt エクスポートオプション
    */
    void PrepareAnimations(const nn::gfx::tool::dcc::RExpOpt& rOpt);

    /**
    ライブラリにテクスチャファイルを登録する。
    \param[out] retIndex 登録されたテクスチャのリスト内のインデックス
    \param[in] テクスチャファイルのパスリスト。キューブマップのときなどに複数になります。
    \param[in] hint テクスチャの関連付けに利用するヒント情報です。
    \return 処理結果を返します。
    */
    bool AddTextureFileForExport( int& retIndex, const nn::gfx::tool::dcc::RStringArray& filePaths, const nn::gfx::tool::dcc::RSampler& sampler, const int curSwizzle, const nn::gfx::tool::dcc::RExpOpt& rOpt )
    {
        return texLibrary.addTextureFile(retIndex, filePaths, sampler,curSwizzle, rOpt);
    }

    /**
    テクスチャファイルの数を返す。
    \return 登録されたテクスチャファイルの数
    \par 解説：
    　登録されたテクスチャファイルの数を返す。
    */
    const int getNumTexture(void) const
    {
        return texLibrary.getNumTexture();
    }

    /**
    テクスチャのインスタンスを取得する。
    \return テクスチャへの参照
    \par 解説：
    　登録されているテクスチャのインスタンスを取得する。
    */
    nn::gfx::tool::dcc::RImage& getTexture(int id)
    {
        return texLibrary.getTexture(id);
    }

    /**
    マテリアル名の重複を解消する。
    \par 解説：
    　マテリアル名の重複を解消する。
    */
    bool SetNamesUnique(void);

    /**
    サンプラ名の重複をチェックする。
    \par 解説：
    　サンプラ名の重複をチェックする。
      falseが返った場合はエクスポートを中止する。
    */
    bool isSamplerNameHintValid(void) const;

    /**
    CTEXを出力するかどうかを決めるフラグをセット
    */
    void SetDoesExportTexture(bool f)
    {
        doesExportTexture= f;
    };

    /**
    CTEXを出力するかどうかを決めるフラグを取得
    */
    bool GetDoesExportTexture(void){ return doesExportTexture;};

    /**
    アニメーションを出力するかどうかを決めるフラグをセット
    */
    void SetDoesExportAnimation(bool f){ mDoesExportAnimation = f;};

    /**
    アニメーションを出力するかどうかを決めるフラグを取得
    */
    bool GetDoesExportAnimation(void){ return mDoesExportAnimation;};

    /**
    テクスチャパターンアニメーションデータをリストへ追加します。
    */
    void addTexPatAnimData( TexPatAnim& data )
    {
        mTexPatAnims.push_back( data );
    }

    /**
    テクスチャパターンアニメーションデータの数を取得します。
    */
    size_t getNumTexPatAnimData( void )
    {
        return mTexPatAnims.size();
    }

    /**
    テクスチャパターンアニメーションデータを取得します。
    */
    TexPatAnim& getTexPatAnimData( int nIndex )
    {
        return mTexPatAnims[nIndex];
    }

    /**
    テクスチャSRTアニメーションデータをリストへ追加します。
    */
    void addTexSrtAnimData( TexSrtAnim& data )
    {
        mTexSrtAnims.push_back( data );
    }

    /**
    テクスチャSRTアニメーションデータの数を取得します。
    */
    size_t getNumTexSrtAnimData( void )
    {
        return mTexSrtAnims.size();
    }

    /**
    テクスチャSRTアニメーションデータを取得します。
    */
    TexSrtAnim& getTexSrtAnimData( int nIndex )
    {
        return mTexSrtAnims[nIndex];
    }

private:

    /**
    マテリアルの情報を保持する構造体
    */
    struct materialInfo
    {
        int index;
        FMaterial* material;
    };
    typedef std::map<const void*, materialInfo> pointerMatinfoMap;


    /**
    ポインタとマテリアルの情報を対応づけるマップ
    */
    pointerMatinfoMap materialMap;

    /**
    CTEXに変換するテクスチャファイルのリスト
    */
    RTextureLibrary texLibrary;

    /**
    CTEXを出力するかどうかを決めるフラグ
    */
    bool doesExportTexture;

    /**
        テクスチャパターンアニメーションで使用されているテクスチャのリスト。
    */
    //PatAnimTextureList	mPatAnimTexList;

    /**
    ポインタ値に対応づけられたマテリアル情報の取得を取得する。
    もし無ければ作成して返す。
    */
    materialInfo* getMaterialInfo(const void* addr);

    /**
    アニメーションを出力するかどうか？
    */
    bool	mDoesExportAnimation;

    /**
    テクスチャパターンアニメーションデータ
    */
    TexPatAnimArray	mTexPatAnims;

    /**
    テクスチャパターンアニメーションデータ
    */
    TexSrtAnimArray	mTexSrtAnims;
};

/******************************************************************************
    end name space utility
******************************************************************************/
}}}}} // namespace utility
