﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/******************************************************************************
    include
******************************************************************************/
#include "DccUtilityNode.h"
#include "DccUtilityModel.h"
#include "DccUtilityLogger.h"

#include "DccOutput.h"

using namespace std;
namespace Dcc = nn::gfx::tool::dcc;

/******************************************************************************
    begin name space utility
******************************************************************************/
namespace nn {
namespace gfx {
namespace tool {
namespace dcc {
namespace utility {

//----------------------------------------------------------------------------
// コンストラクタ
RNodeProcessor::RNodeProcessor(void) :
    m_magnify(1.0f)
    , doesUnifyQuantizeScale(false)
    , m_SmoothSkinningMtxCount(0)
    , m_RigidSkinningMtxCount(0)
{
}

//----------------------------------------------------------------------------
// デストラクタ
RNodeProcessor::~RNodeProcessor(void)
{
    // モデルを削除する
    for(size_t i = 0; i < mModelInstList.size(); i++)
    {
        delete mModelInstList[i].model;
        mModelInstList[i].node->setModel(nullptr);
    }
}

//----------------------------------------------------------------------------
// RNode の階層構造を処理
bool RNodeProcessor::processNodeList(std::vector<RNode*>&	NodeList, const Dcc::RVtxMtxArray& vtxMtxs)
{
    bool ret = true;

    mNodeList = NodeList;

    // ノードのコンバイングループをモデルに伝播
    //transmitCombineGroup();

    // シーン内のモデルに対して、スキニングに関するモードおよび座標系の変換処理
    // を行う
    convertSkinningData( vtxMtxs );

    // ノードのアニメーションフラグを設定
    setAnimationFlag();

    // シーンに登場するモデルの一覧を作成
    buildModelInstanceList();

    //	出力するシェイプのリストに通し番号を振る
    std::vector<FShape*>	OutShapeList;
    collectShapeData( OutShapeList );

#if 0 // 最適化コンバータはポリゴンのない fmd ファイルに対応しているので
    // 出力するシェイプがない場合はエラー
    if(OutShapeList.size() == 0)
    {
        RLogger::LogMessagebyID(RLogger::kLogERR_NoEffectivePolygon, "");
        return false;
    }
#endif

    std::vector<FShape*>::iterator	cur = OutShapeList.begin();
    std::vector<FShape*>::iterator	end = OutShapeList.end();
    int	index = 0;

    while( cur != end )
    {
        (*cur)->m_Index = index;
        ++index;
        ++cur;
    }

    //	スキニングに使用するボーンの行列パレットインデックスを設定します。
    //	mNodeList に対して操作をするため、mNodeList に有効な値が設定されている必要があります。
    setSkinningMtxIdx();

    // スキニングが有効な場合、ボーンとして参照するノード(RNode)にスキニング関連
    // の設定をする。また、モデル描画に必要なノードかどうかのフラグも同時に設定
    // する。mModelInstList に登録されているインスタンスを処理するため、
    // buildModelList() より後で呼び出す必要がある。
    //setupBoneRefNode();

    return ret;
}

//----------------------------------------------------------------------------
// シーン内のモデルに対して、スキニングに関するモードおよび座標系の変換処理を行う
void RNodeProcessor::convertSkinningData(const Dcc::RVtxMtxArray& vtxMtxs)
{
    std::vector<RNode*>::iterator it = mNodeList.begin();
    while(it != mNodeList.end())
    {
        RNode* node = *it;
        RModel* model = node->getModel();
        ++it;
        if(model)
        {
            model->convertSkinningData(node, mNodeList, vtxMtxs);
        }
    }
}

//----------------------------------------------------------------------------
// ノードのアニメーションフラグを設定
void RNodeProcessor::setAnimationFlag(void)
{
    std::vector<RNode*>::iterator it = mNodeList.begin();
    while(it != mNodeList.end())
    {
        RNode* node = *it;
        node->setAnimationFlag();
        ++it;
    }
}


//----------------------------------------------------------------------------
// アニメーション持つノードがあるか
bool RNodeProcessor::hasAnimationNode()
{
    int numnode = 0;
    std::vector<RNode*>::iterator it = mNodeList.begin();
    while(it != mNodeList.end())
    {
        RNode* node = *it;
        if(node->hasAnimation())
        {
            numnode++;
        }
        ++it;
    }
    return (numnode > 0);
}

//----------------------------------------------------------------------------
// ビジビリティアニメーション持つノードがあるか
bool RNodeProcessor::hasVisibilityAnimationNode()
{
    int numnode = 0;
    std::vector<RNode*>::iterator it = mNodeList.begin();
    while(it != mNodeList.end())
    {
        RNode* node = *it;
        if(node->hasVisibilityAnimation())
        {
            numnode++;
        }
        ++it;
    }
    return (numnode > 0);
}


//----------------------------------------------------------------------------
// シーンに登場するモデルの一覧を作成
void RNodeProcessor::buildModelInstanceList(void)
{
    // モデルのインスタンスの配列(mModelInstList)を作成。
    // 「インスタンスの配列」とはノードとモデルのペアの配列で、シーン内にある
    // モデルの総数になる。同時に各インスタンスに対して、そのインスタンスの
    // シェイプが<Shapes>タグ内の何番目に出力されるかの番号を割り当てる。
    // (現在中間フォーマットの仕様により、モデルの多重参照ができないため、イン
    // スタンスごとにシェイプを複製する。)
    int shapeIndex = 0;
    std::set<RModel*> models;
    std::vector<RNode*>::iterator it = mNodeList.begin();
    while(it != mNodeList.end())
    {
        RNode* node = *it;
        RModel* model = node->getModel();
        ++it;

        // モデルを参照していなければ何もしない。
        if(model == nullptr) continue;

        // ノードが無効ならばモデルがあっても無視する
        if(!node->isActive()) continue;

        // このモデルがインスタンスされているかを判定
        // すでにモデルがmodelsセットに登録済みの場合、インスタンスと判定。
        bool instanceModel = (models.find(model) != models.end());

        // ノードから参照されているモデルを列挙し、std::setを使って重複を
        // 除いてモデル配列に代入する。
        if(!instanceModel)
        {
            models.insert(model);
        }

#if	0
        //	TODO: このコードは必要か確認
        // インスタンスされたモデルでは RModel の複製を作成する
        if(instanceModel)
        {
            //!!!! 3dsMAX側からRModelが多重参照されたデータ(インスタンス)が
            //!!!! こないので、ここの処理は未テストです。2010/3/30 nagaya
            model = model->duplicate();
        }
#endif

        // インスタンス配列を作成する。
        RModelInstance inst;
        inst.node = node;
        inst.model = model;
        inst.shapeIndexBase = shapeIndex;

        // インスタンス配列に追加
        mModelInstList.push_back(inst);

        // シェイプの<Shapes>配列内の位置を進める。
        shapeIndex += model->getNumShapes();
    }
}

//-----------------------------------------------------------------------------
//! @brief スキニングに使用するボーンの行列パレットインデックスを設定します。
//!        行列パレットはスムーススキン用配列の後にリジッドスキン用配列が並ぶようにします。
//!
//! @param[in,out] ymodel モデルです。
//-----------------------------------------------------------------------------
void RNodeProcessor::setSkinningMtxIdx( void )
{
    int mtxIdx = 0;
    const int nodeCount = static_cast<int>(mNodeList.size());

    for (int iNode = 0; iNode < nodeCount; ++iNode)
    {
        Dcc::RBone& bone = *mNodeList[iNode];
        if (bone.m_SmoothMtxIdx != -1)
        {
            bone.m_SmoothMtxIdx = mtxIdx++;
            ++m_SmoothSkinningMtxCount;
        }
    }

    for (int iNode = 0; iNode < nodeCount; ++iNode)
    {
        Dcc::RBone& bone = *mNodeList[iNode];
        if (bone.m_RigidMtxIdx != -1)
        {
            bone.m_RigidMtxIdx = mtxIdx++;
            ++m_RigidSkinningMtxCount;
        }
    }
}



//----------------------------------------------------------------------------
// モデルデータの整理を行う
bool RNodeProcessor::optimizeModels(const Dcc::RVtxMtxArray& vtxMtxs, const Dcc::RIntArray& smoothMtxIdxs, const Dcc::RIntArray& rigidMtxIdxs, const RSceneMaterials& sceneMaterials)
{
    std::vector<RModelInstance>::iterator it = mModelInstList.begin();
    while(it != mModelInstList.end())
    {
        RModelInstance& inst = *it;
        // オプティマイズを行うモデルの拡大もここで行う
        if(!inst.model->optimize( vtxMtxs, smoothMtxIdxs, rigidMtxIdxs, sceneMaterials, m_magnify))
        {
            return false;
        }
        ++it;
    }
    return true;
}

//----------------------------------------------------------------------------
//	全ての FShape を集めます。
void RNodeProcessor::collectShapeData( std::vector<FShape*>& ShapeDataList )
{
    std::vector<RModelInstance>::iterator it = mModelInstList.begin();
    while(it != mModelInstList.end())
    {
        const RModelInstance &inst = *it;
        for( int iShape = 0; iShape < inst.model->getNumShapes();iShape++ )
        {
            ShapeDataList.push_back(inst.model->getShape(iShape));
        }
        ++it;
    }
}

//----------------------------------------------------------------------------
// <SkeletalModel><Shapes>タグを出力
bool RNodeProcessor::outShapesData( std::ostream& os, Dcc::RDataStreamArray& dataStreams, int tc, Dcc::RVtxMtxArray vtxMtxs, RSceneMaterials& sceneMaterials )
{
    //	マテリアルインデックスでソートして出力する必要があるので
    //	RModel が持っている FShape を全て集めてリストを作成する。

    std::vector<FShape*>	OutShapes;
    std::vector<Dcc::ROutShapeInfo*>	OutShapeInfos;
    std::vector<RNode*>			ShapeNodes;

    std::vector<RModelInstance>::iterator it = mModelInstList.begin();
    while(it != mModelInstList.end())
    {
        const RModelInstance &inst = *it;
        for( int iShape = 0; iShape < inst.model->getNumShapes();iShape++ )
        {
            FShape*	pShapeData = inst.model->getShape(iShape);
            OutShapes.push_back( pShapeData );
            OutShapeInfos.push_back(pShapeData->getShapeInfo());
            ShapeNodes.push_back(inst.node);
        }
        ++it;
    }

    //	TODO:	SortShapeByMaterialIndex(ymodel);
    //		シェイプリストはマテリアルインデックスでソートしたリストを出力する必要がある。
    const int shapeDataCount = static_cast<int>(OutShapes.size());
    /*
    const char* idxFmt;
    if      (shapeDataCount <    10 + 1) idxFmt = "%d";
    else if (shapeDataCount <   100 + 1) idxFmt = "%02d";
    else if (shapeDataCount <  1000 + 1) idxFmt = "%03d";
    else if (shapeDataCount < 10000 + 1) idxFmt = "%04d";
    else								 idxFmt = "%05d";
    */

    for (int iShape = 0; iShape < shapeDataCount; ++iShape)
    {
        //	TODO: エラー出力にしか使用していないのでとりあえず保留
        // オリジナルの名前は属するノード名 + "_" + 適用されているマテリアル名の形式です。
        //shapeData.m_OrgName += "_" + mat.m_Name;
        OutShapeInfos[iShape]->m_Index = OutShapes[iShape]->m_Index;
        //	このシェイプの属しているノード（ボーン）の名前
        OutShapeInfos[iShape]->m_BoneName = ShapeNodes[iShape]->getName();
        // このシェイプに割り当てられるマテリアルの名前
        FMaterial* fmat = sceneMaterials.getFMaterialByIndex(OutShapes[iShape]->GetMaterialId());
        assert(fmat);
        OutShapeInfos[iShape]->m_MatName = fmat->m_Name;
        // 出力用の名前は "ボーン名__マテリアル名" の形式です。
        //OutShapeInfos[iShape]->m_Name = "shape" + Dcc::RGetNumberString(iShape, idxFmt);
        OutShapeInfos[iShape]->m_Name = OutShapeInfos[iShape]->m_BoneName + "__" + OutShapeInfos[iShape]->m_MatName;
    }

    //-----------------------------------------------------------------------------
    // ボーンに対する行列パレットインデックスの配列を作成します。
    const int boneCount = static_cast<int>(mNodeList.size());
    Dcc::RIntArray smoothMtxIdxs(boneCount);
    Dcc::RIntArray rigidMtxIdxs(boneCount);
    for (int iNode = 0; iNode < boneCount; ++iNode)
    {
        smoothMtxIdxs[iNode] = mNodeList[iNode]->m_SmoothMtxIdx;
        rigidMtxIdxs[iNode]  = mNodeList[iNode]->m_RigidMtxIdx;
    }

    // モデルデータの整理を行う
    optimizeModels( vtxMtxs, smoothMtxIdxs, rigidMtxIdxs, sceneMaterials);

    //-----------------------------------------------------------------------------
    // init
    if (shapeDataCount == 0)
    {
        return true;
    }

    std::ostringstream shapeOs;
    Dcc::RInitOutStreamFormat(shapeOs);

    Dcc::RVertexArray vertices;

    //-----------------------------------------------------------------------------
    // loop for shape data
    for (int iShape = 0; iShape < shapeDataCount; ++iShape)
    {
        Dcc::ROutShapeResult result;
        Dcc::RStatus status = OutShapes[iShape]->Out( shapeOs, vertices, dataStreams, result, tc+1, *OutShapeInfos[iShape] );
        if (status == false)
        {
            utility::RLogger::LogMessage(status.GetMessage(), RLogger::kError);
            return false;
        }
        RNode* rnode =  ShapeNodes[iShape];
        rnode->m_TotalTriangleCount      += result.m_TriangleCount;
        rnode->m_TotalIndexCount         += result.m_IndexCount;
        rnode->m_TotalVertexCount        += result.m_VertexCount;
        rnode->m_TotalProcessVertexCount += result.m_ProcessVertexCount;
    }

    //-----------------------------------------------------------------------------
    // out vertex array
    Dcc::ROutArrayElement(os, 0, vertices, "vertex_array");

    //-----------------------------------------------------------------------------
    // out shape array
    os << Dcc::RTab(tc) << "<shape_array length=\"" << shapeDataCount << "\">" << R_ENDL;
    os << shapeOs.str();
    os << Dcc::RTab(tc) << "</shape_array>" << R_ENDL;

    return true;
}

//----------------------------------------------------------------------------
// <SkeletalModel><Skeleton>タグを出力
void RNodeProcessor::outSkeletonData(std::ostream& os, int tc)
{
    //-----------------------------------------------------------------------------
    // begin skeleton
    const int boneCount = static_cast<int>(mNodeList.size());
    if (boneCount == 0)
    {
        return;
    }
    os << Dcc::RTab(tc) << "<skeleton>" << R_ENDL;

    //-----------------------------------------------------------------------------
    // skeleton info
    //	TODO:	スケルトン情報？を引数で渡されたものにする
    Dcc::RSkeleton skeleton;// = ymodel.m_Skeleton;
    skeleton.m_ScaleMode = mNodeList[0]->getScalingRule();
    skeleton.OutInfo(os, tc + 1);

    //-----------------------------------------------------------------------------
    // loop for bone
    os << Dcc::RTab(tc + 1) << "<bone_array length=\"" << boneCount << "\">" << R_ENDL;

    // 個々のノードに対応する<bone>タグを出力する
    for (int iNode = 0; iNode < boneCount; ++iNode)
    {
        std::string	szParentName = "";
        if( mNodeList[iNode]->getParent() != nullptr )
        {
            szParentName = mNodeList[iNode]->getParent()->getName();
        }

        Dcc::RVec3	t, r, s;

        if( mNodeList[iNode]->mHasBindMatrix )
        {
            Dcc::RMtx44	BindMatLocal = mNodeList[iNode]->getBindMatrix( LocalSpace );
            Dcc::RMtx44	BindMatWorld = mNodeList[iNode]->getBindMatrix( WorldSpace );

            BindMatLocal.GetTransform( s, r, t );

            Dcc::RVec3	wt, wr, ws;
            BindMatWorld.GetTransform(ws, wr, wt);
            //BindMatWorld.SetTransform(ws, wr, wt * m_magnify);
            BindMatWorld[3][0] *= m_magnify;
            BindMatWorld[3][1] *= m_magnify;
            BindMatWorld[3][2] *= m_magnify;

            /*
            char str[1024];
            sprintf(str, "world bind matrix %s\n", mNodeList[iNode]->getName().c_str());
            OutputDebugString(str);
            sprintf(str, "\tScale: %f, %f, %f\n", ws.x, ws.y, ws.z);
            OutputDebugString(str);
            sprintf(str, "\tRot:   %f, %f, %f\n", wr.x, wr.y, wr.z);
            OutputDebugString(str);
            sprintf(str, "\tTrans: %f, %f, %f\n\n", wt.x, wt.y, wt.z);
            OutputDebugString(str);
            */

            Dcc::RMtx44	InvBindMatWorld = BindMatWorld.Inverse();
            InvBindMatWorld.SnapToZero();
            for( int i = 0; i < 4;i++ )
            {
                for( int j = 0; j < 4;j++ )
                {
                    mNodeList[iNode]->m_InvModelMtx.m_Value[i][j] = InvBindMatWorld.m_Value[j][i];
                }
            }
        }
        else
        {
            const Dcc::RMtx44&	LocalMat = mNodeList[iNode]->getLocalMatrix(0);
            LocalMat.GetTransform( s, r, t );
        }

        r.ShiftAngle();
        t *= m_magnify;

        mNodeList[iNode]->m_Scale.x = s.x;
        mNodeList[iNode]->m_Scale.y = s.y;
        mNodeList[iNode]->m_Scale.z = s.z;
        mNodeList[iNode]->m_Scale.SnapToZero();

        mNodeList[iNode]->m_Rotate.x = r.x;
        mNodeList[iNode]->m_Rotate.y = r.y;
        mNodeList[iNode]->m_Rotate.z = r.z;
        mNodeList[iNode]->m_Rotate.SnapToZero();

        mNodeList[iNode]->m_Translate.x = t.x;
        mNodeList[iNode]->m_Translate.y = t.y;
        mNodeList[iNode]->m_Translate.z = t.z;
        mNodeList[iNode]->m_Translate.SnapToZero();

        mNodeList[iNode]->Out(os, tc + 2, iNode, szParentName);
    }
    os << Dcc::RTab(tc + 1) << "</bone_array>" << R_ENDL;

    //-----------------------------------------------------------------------------
    // end skeleton
    os << Dcc::RTab(tc) << "</skeleton>" << R_ENDL;
}

//----------------------------------------------------------------------------
// .fsk の <bone_anim_array> の中身を出力
void RNodeProcessor::outBoneAnim( ostream & os, Dcc::RDataStreamArray& dataStreams )
{
    const int tc = 0;
    const int boneCount = static_cast<int>(mNodeList.size());
    if (boneCount <= 0)
    {
        //return;
    }
    os << Dcc::RTab(tc) << "<bone_anim_array length=\"" << boneCount << "\">" << R_ENDL;

    for (int iNode = 0; iNode < boneCount; ++iNode)
    {
        RNode* rnode = mNodeList[iNode];
        if(!rnode) continue;
        std::string parentName = (rnode->getParent())? rnode->getParent()->getName() : "";

        //-----------------------------------------------------------------------------
        // begin bone anim
        os << Dcc::RTab(tc + 1) << "<bone_anim index=\"" << iNode
            << "\" bone_name=\"" << Dcc::RGetUtf8FromShiftJis(rnode->getName())
            << "\" parent_name=\"" << Dcc::RGetUtf8FromShiftJis(parentName) << "\"" << R_ENDL;
        rnode->OutMtxAttrib(os, tc + 2);
        os << Dcc::RTab(tc + 2) << "scale_compensate=\"" << Dcc::RBoolStr(rnode->m_ScaleCompensate) << "\"" << R_ENDL;
        os << Dcc::RTab(tc + 2) << "binarize_scale=\"" << Dcc::RBoolStr(true) << "\"" << R_ENDL; // 現在は true で固定
        os << Dcc::RTab(tc + 2) << "binarize_rotate=\"" << Dcc::RBoolStr(true) << "\"" << R_ENDL; // 現在は true で固定
        os << Dcc::RTab(tc + 2) << "binarize_translate=\"" << Dcc::RBoolStr(true) << "\"" << R_ENDL; // 現在は true で固定
        os << Dcc::RTab(tc + 2) << "compress_enable=\"" << Dcc::RBoolStr(rnode->m_CompressEnable) << "\"" << R_ENDL;
        os << Dcc::RTab(tc + 1) << ">" << R_ENDL;

        //-----------------------------------------------------------------------------
        // bone anim targets
        // スケール、回転、移動を書き出す
        enum { ScaleELEM, RotateELEM, TransELEM };
        int paramIdx = 0;
        for(int iElem = ScaleELEM; iElem <= TransELEM; ++iElem)
        {
            Dcc::RAnimCurve* curveArray;
            switch(iElem)
            {
            case ScaleELEM: // scale
                curveArray = rnode->mScaleCurve;
                break;
            case RotateELEM: // rotate
                curveArray = rnode->mRotateCurve;
                break;
            case TransELEM: // trans
            default:
                curveArray = rnode->mTransCurve;
                break;
            }

            for(int iAxis = 0; iAxis < 3; ++iAxis)
            {
                Dcc::RAnimCurve& curve = curveArray[iAxis];
                if(curve.m_FullValues.size() > 0)
                {
                    // 移動にはカーブをコピーしてm_magnifyをかける
                    if(iElem == TransELEM)
                    {
                        Dcc::RAnimCurve tempCurve;
                        // 属性
                        tempCurve.m_Scale = curve.m_Scale;
                        tempCurve.m_Offset = curve.m_Offset;
                        tempCurve.m_PreWrap = curve.m_PreWrap;
                        tempCurve.m_PostWrap = curve.m_PostWrap;
                        tempCurve.m_Baked = curve.m_Baked;

                        // キー配列
                        tempCurve.m_Keys = curve.m_Keys;

                        // 内部的なパラメータ
                        tempCurve.m_FullValues.push_back(curve.m_FullValues[0] * m_magnify);

                        tempCurve.m_ConstantFlag = curve.m_ConstantFlag;
                        tempCurve.m_UseFlag = curve.m_UseFlag;
                        tempCurve.m_LoopFlag = curve.m_LoopFlag;
                        tempCurve.m_AngleFlag = curve.m_AngleFlag;
                        tempCurve.m_Tolerance = curve.m_Tolerance;

                        tempCurve.m_Name = curve.m_Name;

                        Dcc::RAnimKeyArray::iterator it;
                        for(it = tempCurve.m_Keys.begin(); it != tempCurve.m_Keys.end(); ++it)
                        {
                            it->m_InSlope *= m_magnify;
                            it->m_OtSlope *= m_magnify;
                            it->m_Value *= m_magnify;
                        }

                        tempCurve.Out(os, dataStreams, tc + 2, "bone_anim_target",
                            Dcc::RBone::GetParamName(paramIdx), false);
                    }
                    else
                    {
                        curve.Out(os, dataStreams, tc + 2, "bone_anim_target",
                            Dcc::RBone::GetParamName(paramIdx), false);
                    }

                }
                paramIdx++;
            }
        }

#if 0
        for (int paramIdx = 0; paramIdx < RBone::ParamMax; ++paramIdx)
        {
            const Dcc::RAnimCurve& curve = rnode.m_Anims[paramIdx];
            curve.Out(os, dataStreams, tc + 1, "bone_anim_target",
                RBone::GetParamName(paramIdx), false);
        }
#endif
        //-----------------------------------------------------------------------------
        // end bone anim
        os << Dcc::RTab(tc + 1) << "</bone_anim>" << R_ENDL;
    }
    os << Dcc::RTab(tc) << "</bone_anim_array>" << R_ENDL;
}

//-----------------------------------------------------------------------------
//! @brief ボーンビジビリティアニメーションを 1 つ出力します。
//!
//! @param[in,out] os 出力ストリームです。
//! @param[in,out] dataStreams データ列配列です。
//! @param[in] tc <bone_vis_bone_anim> 要素のインデントに必要なタブの数です。
//! @param[in] animIdx ボーンビジビリティアニメーションのインデックスです。
//! @param[in] ynode ノードです。
//-----------------------------------------------------------------------------
void RNodeProcessor::outBoneVisAnim(
    std::ostream& os,
    Dcc::RDataStreamArray& dataStreams,
    const int tc,
    const int animIdx,
    const RNode& node
)
{
    const Dcc::RAnimCurve& curve = node.mVisAnim;
    std::string	szParentName = "";
    if( node.getParent() != nullptr )
    {
        szParentName = node.getParent()->getName();
    }

    os << Dcc::RTab(tc) << "<bone_vis_bone_anim index=\"" << animIdx
       << "\" bone_name=\"" << Dcc::RGetUtf8FromShiftJis(node.m_Name)
       << "\" parent_name=\"" << Dcc::RGetUtf8FromShiftJis(szParentName.c_str()) << "\"" << R_ENDL;
        node.OutMtxAttrib(os, tc + 1);
    os << Dcc::RTab(tc + 1) << "binarize_visibility=\"" << Dcc::RBoolStr(true) << "\"" << R_ENDL;
    os << Dcc::RTab(tc + 1) << "compress_enable=\"" << Dcc::RBoolStr(node.m_CompressEnable) << "\"" << R_ENDL;
    float	baseValue = node.isVisible() ? 1.0f : 0.0f;
    if( curve.m_FullValues.size() > 0 )
    {
        baseValue = curve.GetBaseValue();
    }
    os << Dcc::RTab(tc + 1) << "base_value=\"" << baseValue << "\"" << R_ENDL;
    if (curve.m_ConstantFlag)
    {
        os << Dcc::RTab(tc) << "/>" << R_ENDL;
    }
    else
    {
        os << Dcc::RTab(tc) << ">" << R_ENDL;

        curve.Out(os, dataStreams, tc + 1, false);

        os << Dcc::RTab(tc) << "</bone_vis_bone_anim>" << R_ENDL;
    }
}

//----------------------------------------------------------------------------
//	　ボーンビジビリティアニメーションのデータタグを出力します。
void RNodeProcessor::outVisibilityAnimationData(std::ostream &os, Dcc::RDataStreamArray& dataStreams, const Dcc::RExpOpt& rOpt )
{
    //-----------------------------------------------------------------------------
    // ベイクしたデータからアニメーションカーブを作成します。
    const int boneCount = static_cast<int>(mNodeList.size());

    std::vector<RNode*> exportNodes;

    for (int iNode = 0; iNode < boneCount; ++iNode)
    {
        RNode* node = mNodeList[iNode];
        Dcc::RAnimCurve& curve = node->getVisibilityAnimationData();
        curve.m_Name = node->m_Name + ".v";
        curve.m_Tolerance = 0.0f;

        curve.UpdateConstantFlag();
        if (!curve.m_ConstantFlag)
        {
            curve.MakeStepKeys(GetFloatFrameFromSubFrame4f, nullptr);
        }

        if (rOpt.m_ExportsBoneVisAll || !curve.m_Keys.empty())
        {
            exportNodes.push_back(node);
        }
    }

    //-----------------------------------------------------------------------------
    // bone vis bone anim array
    if ( exportNodes.empty() == false )
    {
        const int tc = 0;
        os << Dcc::RTab(tc) << "<bone_vis_bone_anim_array length=\"" << exportNodes.size() << "\">" << R_ENDL;
        for (int iNode = 0; iNode < exportNodes.size(); ++iNode)
        {
            outBoneVisAnim(os, dataStreams, tc + 1, iNode, *exportNodes[iNode]);
        }
        os << Dcc::RTab(tc) << "</bone_vis_bone_anim_array>" << R_ENDL;
    }
}

//----------------------------------------------------------------------------
//	　シェイプビジビリティアニメーションのデータタグを出力します。
void RNodeProcessor::outShapeAnimationData( std::ostream &os, Dcc::RDataStreamArray& dataStreams, const Dcc::RExpOpt& rOpt )
{
    //-----------------------------------------------------------------------------
    // ベイクしたデータからアニメーションカーブを作成します。
    const int boneCount = static_cast<int>(mNodeList.size());
    vector<RNode*> shapeAnimNodes;
    std::vector<Dcc::RAnimCurve>::size_type shapeAnimCount = 0;

    for (int iNode = 0; iNode < boneCount; ++iNode)
    {
        RNode* node = mNodeList[iNode];
        utility::FOutShapeInfo& shapeinfo = node->getModel()->getShapeInfo();

        // シェープアニメを持つノード
        if(node->mShapeAnims.size() > 0 && shapeinfo.m_KeyNames.size() > 0)
        {
            shapeAnimNodes.push_back(node);

            std::vector<Dcc::RAnimCurve>& curves = node->mShapeAnims;
            for(size_t ishape = 0; ishape < curves.size(); ishape++)
            {
                Dcc::RAnimCurve& curve = curves[ishape];

                curve.m_Name      = node->m_Name + "." + shapeinfo.m_KeyNames[ishape];
                curve.m_LoopFlag  = rOpt.m_LoopAnim;
                curve.m_AngleFlag = false;
                curve.m_Tolerance = Dcc::R_MAKE_KEY_TOL_SHAPE_ANIM;

                if ( curve.m_Keys.empty() )
                {
                    curve.MakeKeys(GetFloatFrameFromSubFrame4f, nullptr);
                    curve.UpdateConstantFlag();
                }
            }

            // モデル内のシェイプごとにアニメーションがるかどうか調べる
            utility::RModel* rmodel = node->getModel();
            assert(rmodel);
            for (int iShape = 0; iShape < rmodel->getNumShapes(); iShape++)
            {
                //utility::FOutShapeInfo& shapeinfo = node->getModel()->getShapeInfo();
                utility::FOutShapeInfo* shapeinfo = rmodel->getShape(iShape)->getShapeInfo();
                assert(shapeinfo);
                if(shapeinfo->m_KeyNames.size() > 0)
                {
                    shapeAnimCount += curves.size();
                }
            }
        }
    }

    if(shapeAnimNodes.size() > 0 && shapeAnimCount > 0)
    {
        //-----------------------------------------------------------------------------
        // begin vertex shape anim array
        const int tc = 0;
        os << Dcc::RTab(tc) << "<vertex_shape_anim_array length=\"" << shapeAnimCount << "\">" << R_ENDL;

        int shapeCount = 0;
        for(size_t iNode = 0; iNode < shapeAnimNodes.size(); iNode++)
        {
            RNode* node = shapeAnimNodes[iNode];
            std::vector<Dcc::RAnimCurve>& curves = node->mShapeAnims;
            utility::RModel* rmodel = node->getModel();
            assert(rmodel);

            for (int iShape = 0; iShape < rmodel->getNumShapes(); iShape++)
            {
                //utility::FOutShapeInfo& shapeinfo = node->getModel()->getShapeInfo();
                utility::FOutShapeInfo* shapeinfo = rmodel->getShape(iShape)->getShapeInfo();
                assert(shapeinfo);

                if(shapeinfo->m_KeyNames.size() == 0)
                {
                    continue;
                }

                os << Dcc::RTab(tc + 1) << "<vertex_shape_anim index=\"" << shapeCount
                    << "\" shape_name=\"" << Dcc::RGetUtf8FromShiftJis(shapeinfo->m_BoneName + "__" + shapeinfo->m_MatName)
                    << "\" base_name=\"" << Dcc::RGetUtf8FromShiftJis(shapeinfo->m_KeyNames[0])
                    << "\">" << R_ENDL;

                for(size_t iCurve = 0; iCurve < curves.size(); iCurve++)
                {
                    Dcc::RAnimCurve& curve = curves[iCurve];
                    if (curve.m_UseFlag)
                    {
                        os << Dcc::RTab(tc + 2) << "<shape_anim_target"
                            << " key_shape_name=\"" << Dcc::RGetUtf8FromShiftJis(shapeinfo->m_KeyNames[iCurve + 1]) // key nameにはbase nameが最初に入っている
                            << "\" base_value=\"" << curve.GetBaseValue() << "\"";
                        if (curve.m_ConstantFlag)
                        {
                            os << " />" << R_ENDL;
                        }
                        else
                        {
                            os << ">" << R_ENDL;

                            curve.Out(os, dataStreams, tc + 3, false);

                            os << Dcc::RTab(tc + 2) << "</shape_anim_target>" << R_ENDL;
                        }
                    }
                }
                os << Dcc::RTab(tc+1) << "</vertex_shape_anim>" << R_ENDL;
                shapeCount++;
            }

        }

        //-----------------------------------------------------------------------------
        // end vertex shape anim array
        os << Dcc::RTab(tc) << "</vertex_shape_anim_array>" << R_ENDL;
    }
}

/******************************************************************************
    end name space utility
******************************************************************************/
}}}}} // namespace utility

/******************************************************************************
-------------------------------------------------------------------------------
                end of file
-------------------------------------------------------------------------------
******************************************************************************/
