﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/******************************************************************************
    include
******************************************************************************/
#include "DccUtilityNode.h"
#include "DccUtilityModel.h"

using namespace std;
namespace Dcc = nn::gfx::tool::dcc;

/******************************************************************************
    begin name space utility
******************************************************************************/
namespace nn {
namespace gfx {
namespace tool {
namespace dcc {
namespace utility {

int RNode::mAnimSubSample = 1;

float RNode::mScaleTolerance = 0.01f;
float RNode::mRotateTolerance = 0.01f;
float RNode::mTransTolerance = 0.01f;

//----------------------------------------------------------------------------
// コンストラクタ
RNode::RNode(void)
    : Dcc::RBone()
    , mParent(nullptr)
    , mChild(nullptr)
    , mNext(nullptr)
    , mModel(nullptr)
    , mActive(true)
    , mMatrixArrayFirstFrame(-1)
    , mHasBindMatrix(false)
    , mNodeArrayIndex(-1)
    , mScalingRule(Dcc::RSkeleton::STANDARD)
    , mNoAnimConstScale( 1.0f, 1.0f, 1.0f )
    , mNoAnimConstRotate( 0.0f, 0.0f, 0.0f )
    , mNoAnimConstTrans( 0.0f, 0.0f, 0.0f )
    , mForceExportScaleKey(false)
    , mForceExportRotateKey(false)
    , mForceExportTransKey(false)
    , mForceExportVisibility(false)
    , mHasScaleAnim(false)
    , mHasRotateAnim(false)
    , mHasTransAnim(false)
    , m_TotalTriangleCount(0)
    , m_TotalIndexCount(0)
    , m_TotalVertexCount(0)
    , m_TotalProcessVertexCount(0)
{
    // 行列配列は初期化しておく。
    mLocalMatrix.clear();
}

//----------------------------------------------------------------------------
// デストラクタ
RNode::~RNode(void)
{
    // 子階層を再帰的に破棄
    {
        RNode* node = mChild;
        while(node)
        {
            RNode* next = node->mNext;
            delete node;
            node = next;
        }
    }
    // 親から自分を消す
    {
        RNode* parent = mParent;
        if ( parent )
        {
            RNode* node = parent->mChild;
            if ( node == this )
            {
                parent->mChild = mNext;
            }
            else
            {
                while ( node->mNext )
                {
                    if ( node->mNext == this )
                    {
                        node->mNext = mNext;
                        break;
                    }
                    node = node->mNext;
                }
            }
        }
    }
}

//----------------------------------------------------------------------------
// 階層構造の親を指定
void RNode::setParent(RNode* parent)
{
    // すでにこの関数が呼び出されていたらアサート
    assert(mParent == NULL);

    // parentノードの子階層の先頭に挿入
    mNext = parent->mChild;
    parent->mChild = this;
    mParent = parent;
}

//----------------------------------------------------------------------------
// このノードのモデルを設定
void RNode::setModel(RModel* model)
{
    mModel = model;
}

//----------------------------------------------------------------------------
//	Visibility アニメーションの状態を設定
void RNode::setVisibilityAnim(float visibility)
{
    mVisAnim.m_FullValues.push_back(visibility);
}

//----------------------------------------------------------------------------
//	Shape アニメーションを設定
void RNode::setShapeAnim( UINT shapeID, float weight )
{
    if(shapeID >= mShapeAnims.size())
    {
        mShapeAnims.resize(shapeID + 1);
    }
    mShapeAnims[shapeID].m_FullValues.push_back(weight);
}


//----------------------------------------------------------------------------
// ローカル変換行列を設定
void RNode::setLocalMatrix(int frame, const Dcc::RMtx44 &mtx)
{

    // 最初のフレームを記録
    if(mLocalMatrix.size() == 0)
    {
        mMatrixArrayFirstFrame = frame;
    }

    // 最初に設定したフレームより前のタイミングのフレームを後から
    // 設定することはできない。
    assert(mMatrixArrayFirstFrame <= frame);

    // 最初のフレームからの相対に変換
    frame -= mMatrixArrayFirstFrame;

    // 必要に応じて配列を拡大し、変換行列を記録
    if(static_cast<int>(mLocalMatrix.size()) <= frame)
    {
        mLocalMatrix.resize(frame+1);
    }
    mLocalMatrix[frame] = mtx;
}

//----------------------------------------------------------------------------
// ローカル変換行列を取得
const Dcc::RMtx44& RNode::getLocalMatrix(int frame) const
{
    // 開始フレームのオフセットを引く
    frame -= mMatrixArrayFirstFrame;
    // 開始フレームより前のタイミングでは開始フレームを返す
    if(frame < 0)
    {
        frame = 0;
    }

    // 最終フレームより後のタイミングでは最終フレームを返す
    int lastFrame = static_cast<int>(mLocalMatrix.size()) - 1;
    if(lastFrame < frame)
    {
        frame = lastFrame;
    }

    return mLocalMatrix[frame];
}

//----------------------------------------------------------------------------
// ワールド変換行列を取得
const Dcc::RMtx44 RNode::getWorldMatrix(int frame) const
{
    if(mParent)
    {
        if( mScalingRule != Dcc::RSkeleton::SOFTIMAGE )
        {
            // 親フレームに向かって再帰しながら行列を積算
            return getLocalMatrix(frame) * mParent->getWorldMatrix(frame);
        }
        else
        {
            //	Softimageスケーリング
            Dcc::RVec3	vDummy;
            Dcc::RVec3	vParentRot, vParentScale;
            Dcc::RVec3	vChildRot, vChildScale, vChildTrans;
            Dcc::RVec4	vChildTransTemp;
            Dcc::RMtx44	mMat;
            const Dcc::RMtx44&	mLocal = getLocalMatrix(frame);
            const Dcc::RMtx44	mWorld = mParent->getWorldMatrix(frame);

            vChildTransTemp.x = mLocal[3][0];
            vChildTransTemp.y = mLocal[3][1];
            vChildTransTemp.z = mLocal[3][2];
            vChildTransTemp.w = 1.0f;

            //	ローカルの移動量を計算する
            vChildTransTemp *= mWorld;

            vChildTrans.x = vChildTransTemp.x;
            vChildTrans.y = vChildTransTemp.y;
            vChildTrans.z = vChildTransTemp.z;

            mWorld.GetTransform( vParentScale, vParentRot, vDummy );
            mLocal.GetTransform( vChildScale, vChildRot, vDummy );

            //	回転とスケールの親成分を打ち消してローカルの値にする
            vChildScale.x *= vParentScale.x;
            vChildScale.y *= vParentScale.y;
            vChildScale.z *= vParentScale.z;

            Dcc::RMtx44	mRotP, mRotC;

            mRotP.SetTransform( Dcc::RVec3::kOne, vParentRot, Dcc::RVec3::kZero );
            mRotC.SetTransform( Dcc::RVec3::kOne, vChildRot, Dcc::RVec3::kZero );

            mRotC = mRotC * mRotP;
            mRotC.GetTransform( vDummy, vChildRot, vDummy );

            mMat.SetTransform( vChildScale, vChildRot, vChildTrans );
            return mMat;
        }
    }
    else
    {
        // ルートノードまで来たらこのノードの行列をそのまま返す
        return getLocalMatrix(frame);
    }
}

//----------------------------------------------------------------------------
// スキニング用のバインド行列を取得
const Dcc::RMtx44 RNode::getBindMatrix(ECoordSystem coordSystem) const
{
    Dcc::RMtx44 bindMatrix;
    if(mHasBindMatrix)
    {
        // このノードにバインド行列が設定されている場合。
        if(coordSystem == LocalSpace)
        {
            // 親の座標系で返す。
            // (<SkeletalModel><Skeleton><Bones><Bone>に出力する変換)
            // そのために親階層のワールド座標系での変換の逆行列を取得する。
            // 親階層がボーンでバインドポーズを持っている場合、その変換を使う。
            // 親階層がボーンでない場合には、0フレーム目の変換を使う。
            // ちなみにここでは適当に0フレーム目としているが、厳密には.cmdlの
            // <SkeletalModel><Skeleton>に出力される変換を使う必要がある。
            // もしここに0フレーム目ではないタイミングの姿勢が出力されるのなら
            // ば、そのタイミングでの変換を使う必要がある。 nagaya
            Dcc::RMtx44 ipmtx;
            Dcc::RMtx44 pmtx;

            if(mParent)
            {
                if(mParent->mHasBindMatrix)
                {
                    pmtx = mParent->mBindMatrix;
                }
                else
                {
                    pmtx = mParent->getWorldMatrix(0);//mAnimSubStart);
                }
            }

            if( mScalingRule != Dcc::RSkeleton::SOFTIMAGE )
            {
                ipmtx = pmtx.Inverse();
                bindMatrix = mBindMatrix * ipmtx;
            }
            else
            {
                bindMatrix = cancelMatrix( mBindMatrix, pmtx );
            }
        }
        else
        {
            // ワールド座標系で返す。(そのまま)
            bindMatrix = mBindMatrix;
        }
    }
    // このノードにバインド行列が設定されていなければ、単位行列を返す。
    bindMatrix.SnapToZero();

    return bindMatrix;
}

//----------------------------------------------------------------------------
// ノード名を文字コード(辞書順)で比較
static bool greaterNodeName(RNode* node0, RNode* node1)
{
    return node0->getName() < node1->getName();
}

//----------------------------------------------------------------------------
// ノードの有効/無効を設定
void RNode::setActive(bool active, bool recursive)
{
    // 代入
    mActive = active;

    // 再帰的に処理
    if(recursive)
    {
        RNode* node = mChild;
        while(node)
        {
            node->setActive(active, recursive);
            node = node->mNext;
        }
    }
}

//----------------------------------------------------------------------------
// シーンを構成するノードを配列に並べる(再帰)(内部使用)
void RNode::enumNodeToArray(std::vector<RNode*> &nodeList)
{
    // thisにノードインデックスを設定
    mNodeArrayIndex = static_cast<int>(nodeList.size());

    // thisを配列に追加
    nodeList.push_back(this);

    // 子階層へ再帰
    if(mChild)
    {
        // 基本は深さ優先だが、同じ階層のノードはアルファベット順でソートされ
        // て出力される。ただしノードが無効ならばそれ以下の階層は出力しない。
        std::vector<RNode*> sortBuf;
        RNode* node = mChild;
        while(node)
        {
            if(node->mActive)
            {
                sortBuf.push_back(node);
            }
            node = node->mNext;
        }
        std::sort(sortBuf.begin(), sortBuf.end(), greaterNodeName);

        // アルファベット順でソートした順番で子階層に再帰する
        for(size_t i = 0; i < sortBuf.size(); i++)
        {
            sortBuf[i]->enumNodeToArray(nodeList);
        }
    }
}

/******************************************************************************
    get float frame from sub frame
******************************************************************************/
extern float GetFloatFrameFromSubFrame(const int subFrame)
{
    return static_cast<float>(subFrame) / static_cast<float>(RNode::getAnimSubSample());
}

extern float GetFloatFrameFromSubFrame4f(const int subFrame, const void*)
{
    return static_cast<float>(subFrame) / static_cast<float>(RNode::getAnimSubSample());
}

//設定されたローカル変換行列から移動、回転、スケールの値を取り出す。
void RNode::convertMatrixAnimToSRTAnim()
{
    Dcc::RVec3 scale, rotate, trans;
    int numFrame = static_cast<int>(mLocalMatrix.size());
    Dcc::RFloatArray scaleAnim[3];
    Dcc::RFloatArray rotateAnim[3];
    Dcc::RFloatArray transAnim[3];


    // SRT配列をクリアしフレーム数分を確保
    for(int i = 0; i < 3; i++)
    {
        /*
        mScaleAnim[i].clear();
        mScaleAnim[i].resize(numFrame);
        mRotateAnim[i].clear();
        mRotateAnim[i].resize(numFrame);
        mTransAnim[i].clear();
        mTransAnim[i].resize(numFrame);
        mScaleAnim[i].clear();
        mRotateAnim[i].clear();
        mTransAnim[i].clear();
        */

        scaleAnim[i].resize(numFrame);
        rotateAnim[i].resize(numFrame);
        transAnim[i].resize(numFrame);
    }

    // 行列からSRTを取得して、SRT配列にセットする。
    for(int f = 0; f < numFrame; f++)
    {
        mLocalMatrix[f].GetTransform(scale, rotate, trans);
        scaleAnim[0].at(f) = scale.x;
        scaleAnim[1].at(f) = scale.y;
        scaleAnim[2].at(f) = scale.z;

        rotateAnim[0].at(f) = rotate.x;
        rotateAnim[1].at(f) = rotate.y;
        rotateAnim[2].at(f) = rotate.z;

        transAnim[0].at(f) = trans.x;
        transAnim[1].at(f) = trans.y;
        transAnim[2].at(f) = trans.z;
    }

    // 0に近い値を丸める
    for(int i = 0; i < 3; i++)
    {
        Dcc::RSnapToZero(scaleAnim[i]);
        Dcc::RSnapToZero(rotateAnim[i]);
        Dcc::RSnapToZero(transAnim[i]);
    }

    // 回転を連続的にする。
    //RMakeContinuousXyzAngleArray(rotateAnim[0], rotateAnim[1], rotateAnim[2], true);
    Dcc::RMakeContinuousXyzAngleArray(rotateAnim[0], rotateAnim[1], rotateAnim[2]);
    enum { ScaleELEM, RotateELEM, TransELEM };

    // 各要素がアニメーションしているかどうか調べる。
    bool isAnim[3][3] = {{0,0,0},{0,0,0},{0,0,0}};

    // MINMAXを調べてアニメーションしているかどうか判定
    for(int i = ScaleELEM; i <= TransELEM; i++)
    {
        for(int j = 0; j < 3; j++) //xyz
        {
            Dcc::RFloatArray* anim = nullptr;
            float tolerance = 0.0f;
            switch(i)
            {
            case ScaleELEM:
                anim = &scaleAnim[j];
                tolerance = mScaleTolerance;
                break;
            case RotateELEM:
                anim = &rotateAnim[j];
                tolerance = mRotateTolerance;
                break;
            case TransELEM:
                anim = &transAnim[j];
                tolerance = mTransTolerance;
                break;
            }

            if ( anim )
            {
                float vmin, vmax, v;
                vmin = vmax = anim->at(0);
                for(int f = 1; f < numFrame; f++)
                {
                    v = anim->at(f);
                    vmin = (v < vmin)? v: vmin;
                    vmax = (v > vmax)? v: vmax;
                    if((vmax - vmin) > tolerance)
                    {
                        isAnim[i][j] = true;
                        break;
                    }
                }
            }
        }
    }

    for(int i = ScaleELEM; i <= TransELEM; i++)
    {
        for(int j = 0; j < 3; j++) //xyz
        {
            Dcc::RAnimCurve* curve = nullptr;
            Dcc::RFloatArray* values = nullptr;
            //float tolerance = 0.0f;
            switch(i)
            {
            case ScaleELEM:
                curve = &(mScaleCurve[j]);
                values =  &(scaleAnim[j]);
                curve->m_Tolerance = mScaleTolerance;
                curve->m_AngleFlag = false;
                break;
            case RotateELEM:
                curve = &(mRotateCurve[j]);
                values =  &(rotateAnim[j]);
                curve->m_Tolerance = mRotateTolerance;
                curve->m_AngleFlag = true;
                break;
            case TransELEM:
                curve = &(mTransCurve[j]);
                values =  &(transAnim[j]);
                curve->m_Tolerance = mTransTolerance;
                curve->m_AngleFlag = false;
                break;
            }

            if(curve && values)
            {
                curve->m_FullValues = *values;
                curve->UpdateConstantFlag();

                if (!(curve->m_ConstantFlag))
                {
                    curve->MakeKeys(GetFloatFrameFromSubFrame4f, nullptr, false);
                }
            }
        }
    }

    // マトリクス配列を最初の値のみ残して削除
    std::vector<Dcc::RMtx44> mtx;
    mtx.push_back(mLocalMatrix[0]);
    mtx.swap(mLocalMatrix);

    // アニメしない場合のために、固定値をコピー
    mNoAnimConstScale = Dcc::RVec3( scaleAnim[0][0], scaleAnim[1][0], scaleAnim[2][0] );
    mNoAnimConstRotate = Dcc::RVec3( rotateAnim[0][0], rotateAnim[1][0], rotateAnim[2][0] );
    mNoAnimConstTrans = Dcc::RVec3( transAnim[0][0], transAnim[1][0], transAnim[2][0] );

}

// ノードのビジビリティ、SRT各要素がアニメーションしているかどうかを表すフラグを設定する。
void RNode::setAnimationFlag()
{
    // ビジビリティアニメーションをキーフレームに変換
    mVisAnim.m_Name = m_Name + ".v";
    mVisAnim.m_Tolerance = 0.0f;

    //	TODO:	キーがすでに更新されていないか確認する。
    //	Max では外部からキーが設定されることは無いので現在は無効にしておく。
    //	Softimage に対応する際には必要。
    //if( !mVisAnim.IsKeyUpdated() )
    {
        if(mVisAnim.m_FullValues.size() > 0)
        {
            mVisAnim.MakeStepKeys(GetFloatFrameFromSubFrame4f, nullptr);
            mVisAnim.m_UseFlag = true;
        }
        else if(mForceExportVisibility)
        {
            // 強制出力フラグの時は値を追加する。
            setVisibilityAnim(m_Visibility? 1.0f: 0.0f);
            mVisAnim.MakeStepKeys(GetFloatFrameFromSubFrame4f, nullptr);
            mVisAnim.m_UseFlag = true;
        }
        else
        {
            //mVisAnim.UpdateConstantFlag();
            mVisAnim.m_UseFlag = false;
        }
    }

    // SRTの各要素がアニメーションしているかどうか調べる。
    //bool isAnim[3][3] = {{0,0,0},{0,0,0},{0,0,0}};
    Dcc::RVec3 boneScale, boneRotate, boneTrans;
    enum { ScaleELEM, RotateELEM, TransELEM };

    if(mHasBindMatrix)
    {
        Dcc::RMtx44 bindMatrix = getBindMatrix(LocalSpace);
        bindMatrix.GetTransform(boneScale, boneRotate, boneTrans);
    }

    // アニメーションしないノードやバインドポーズのままのボーンは
    // 出力しない
    mHasScaleAnim = false;
    mHasRotateAnim = false;
    mHasTransAnim = false;

    for(int i = ScaleELEM; i <= TransELEM; i++)
    {
        for(int j = 0; j < 3; j++)
        {
            switch(i)
            {
                case ScaleELEM: // scale
                    //if(mForceExportScaleKey || mScaleKeys[j].size() > 1)
                    if(mForceExportScaleKey || !mScaleCurve->m_ConstantFlag)
                    {
                        mHasScaleAnim = true;
                    }
                    else if(mHasBindMatrix)
                    {
                        if(fabs(mNoAnimConstScale[j] - boneScale[j]) > mScaleTolerance)
                        {
                            mHasScaleAnim = true;
                        }
                    }
                    break;

                case RotateELEM: // rotate
                    //if(mForceExportRotateKey || mRotateKeys[j].size() > 1)
                    if(mForceExportRotateKey || !mRotateCurve->m_ConstantFlag)
                    {
                        mHasRotateAnim = true;
                    }
                    else if(mHasBindMatrix)
                    {
                        if(fabs(mNoAnimConstRotate[j] - boneRotate[j]) > mRotateTolerance)
                        {
                            mHasRotateAnim = true;
                        }
                    }
                    break;

                case TransELEM: // trans
                default:
                    //if(mForceExportTransKey || mTransKeys[j].size() > 1)
                    if(mForceExportTransKey || !mTransCurve->m_ConstantFlag)
                    {
                        mHasTransAnim = true;
                    }
                    else if(mHasBindMatrix)
                    {
                        if(fabs(mNoAnimConstTrans[j] - boneTrans[j]) > mTransTolerance)
                        {
                            mHasTransAnim = true;
                        }
                    }
                    break;
            }
        }
    }
}

const Dcc::RMtx44 cancelMatrix( const Dcc::RMtx44& cmtx, const Dcc::RMtx44& pmtx )
{
    //	グローバル変換行列からローカル変換行列へ変換する。
    //	Softimageスケーリングの場合、
    //	単純に親の逆行列をかけてもローカル変換行列は求まらない。
    //	親の行列に直接影響されるのは移動のみで
    //	回転とスケーリングは行列とは別に値が直接伝播される。
    //	詳しくはSoftimage User's GuideのScalign Objectsを参照のこと。
    Dcc::RMtx44 bindMatrix;
    Dcc::RVec3	vDummy;
    Dcc::RVec3	vParentRot, vParentScale;
    Dcc::RVec3	vChildRot, vChildScale, vChildTrans;
    Dcc::RVec4	vChildTransTemp;

    vChildTransTemp.x = cmtx[3][0];
    vChildTransTemp.y = cmtx[3][1];
    vChildTransTemp.z = cmtx[3][2];
    vChildTransTemp.w = 1.0f;

    //	ローカルの移動量を計算する
    vChildTransTemp *= pmtx.Inverse();

    vChildTrans.x = vChildTransTemp.x;
    vChildTrans.y = vChildTransTemp.y;
    vChildTrans.z = vChildTransTemp.z;

    pmtx.GetTransform( vParentScale, vParentRot, vDummy );
    cmtx.GetTransform( vChildScale, vChildRot, vDummy );

    //	回転とスケールの親成分を打ち消してローカルの値にする
    vChildScale.x /= vParentScale.x;
    vChildScale.y /= vParentScale.y;
    vChildScale.z /= vParentScale.z;

    Dcc::RMtx44	mRotP, mRotC;

    mRotP.SetTransform( Dcc::RVec3::kOne, vParentRot, Dcc::RVec3::kZero );
    mRotC.SetTransform( Dcc::RVec3::kOne, vChildRot, Dcc::RVec3::kZero );

    mRotC = mRotC * mRotP.Inverse();
    mRotC.GetTransform( vDummy, vChildRot, vDummy );

    bindMatrix.SetTransform( vChildScale, vChildRot, vChildTrans );

    return bindMatrix;
}




/******************************************************************************
    end name space utility
******************************************************************************/
}}}}} // namespace utility

/******************************************************************************
-------------------------------------------------------------------------------
                end of file
-------------------------------------------------------------------------------
******************************************************************************/
