﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

/******************************************************************************
    include
******************************************************************************/
#include "DccUtilityShape.h"

/******************************************************************************
    begin name space utility
******************************************************************************/
namespace nn {
namespace gfx {
namespace tool {
namespace dcc {
namespace utility {

class RSceneMaterials;
class RNode;

//----------------------------------------------------------------------------
/**
複数のマテリアルで構成されるモデルを扱うためのクラス
\par 解説：
　マテリアル番号を指定してポリゴンを登録することで、
複数のマテリアルから構成されるモデルを作成する。
ポリゴンの登録には appendPolygon() を使用する。
(内部ではマテリアルが異なる部分毎に RShape のインスタンスを作成する。)
\par
　モデルは座標、法線、テクスチャ座標等の頂点属性データを代入するオブジェクト( mShapeInfo )を内部で保持し、
getShapeInfo() によって参照できる。
DCCプラグインはこのオブジェクトに頂点属性データを代入する。
ポリゴンを登録する際に指定する RPrimitive の RPrimitive::mVtxs[].mAttr[] は、
この頂点属性データオブジェクト内の配列のインデックスを指定する。
<br />
　頂点属性データ( mShapeInfo )は appendPolygon() 内でシェイプ( RShape )に複製される。
よって appendPolygon() の呼び出し前に頂点属性データの設定を完了しておく必要がある。
\par
　スキニングが有効なモデルでは setVertexBoneRefArray() 関数で頂点ボーンウェイト配列( RVtxBoneRefArray )を設定する。
ポリゴンを登録する際に指定する RPrimitive の RPrimitive::mVtxs[].mBoneRefIdx は、
この頂点ボーンウェイト配列へのインデックスを指定する。
スキニングの有効/無効はこの頂点ボーンウェイト配列の設定の有無で決まる。
*/
class RModel
{
public:
/**
\name 操作
\{
*/
    /**
    コンストラクタ
    \param[in] mtxPalSize 行列パレット数
    \param[in] orgName シェイプ名。<EditData> 内へノード名を出力する必要がある。
    \param[in] progShapePtr シェイプ処理の進行状況を伝えるオブジェクト。設定しなくてもかまわない。
    */
    RModel(
        const int mtxPalSize,
        const char* orgName,
        nn::gfx::tool::dcc::RProgShape* progShapePtr = nullptr);

    /**
    デストラクタ
    */
    ~RModel(void);

    /**
    頂点属性が使われているかどうかのフラグを設定
    \param[in] attr 頂点属性を識別する列挙子
    \param[in] used 使われるかどうか
    \par 解説：
    　スキニングのためのブレンドウェイトとインデックスが有効かどうかは
    setVertexBoneRefArray() で頂点ボーンウェイトを設定したかによって決まる。
    そのため本関数での RPrimVtx::IDX と RPrimVtx::WGT の設定は無視される。
    */
    void setVertexAttributeUsed(nn::gfx::tool::dcc::RPrimVtx::VtxAttr attr, bool used);

    /**
    頂点属性が使われているかどうかのフラグを取得
    \param[in] attr 頂点属性を識別する列挙子
    \return 使われるかどうか
    \sa
    setVertexAttributeUsed()
    */
    bool getVertexAttributeUsed(nn::gfx::tool::dcc::RPrimVtx::VtxAttr attr)
    { return mVtxAttrFlag[attr]; }

    /**
    ポリゴンを追加
    \param[in] materialID マテリアルID
    \param[in] rprim プリミティブ
    \param[in] forceTriangulate 強制的に三角形分割するかどうか。
    false の場合、多角形で使用する行列数が行列パレット数を超えるなら三角形分割する。
    \return 三角形分割しても行列パレット数を超えるなら nn::gfx::tool::dcc::RStatus::FAILURE を返す。
    \par 解説：
    　マテリアルIDを指定してポリゴンを追加する。
    クラス内ではマテリアル毎に RShape のインスタンスが作成される。
    マテリアルIDは連続した値でなくてもかまわない。
    \par
    　ポリゴンを表す RPrimitive の RPrimitive::mVtxs[].mAttr[] は、
    getShapeInfo() で参照される頂点属性データオブジェクト内の配列のインデックスを指定する。
    スキニングが有効な場合、RPrimitive の RPrimitive::mVtxs[].mBoneRefIdx は、
    setVertexBoneRefArray() で設定した頂点ボーンウェイト配列へのインデックスを指定する。
    */
    nn::gfx::tool::dcc::RStatus appendPolygon(
        unsigned int materialID,
        const nn::gfx::tool::dcc::RPrimitive& rprim,
        const nn::gfx::tool::dcc::RVtxMtxArray& vtxMtxs,
        const bool forceTriangulate);

    /**
    頂点属性データ(頂点データ)を代入するための ROutShapeInfo オブジェクトを参照
    \return 頂点属性データを代入するための ROutShapeInfo オブジェクト
    \par 解説：
    このモデルの頂点属性データ(頂点データ)を代入するための
    ROutShapeInfo オブジェクトを参照する。
    クラスユーザーはこのオブジェクトに対して座標やベクトルなどの値を設定する。
    ポリゴンを登録する際に指定する RPrimitive の RPrimitive::mVtxs[].mAttr[] は、
    この頂点属性データオブジェクト内の配列のインデックスを指定する。
    */
    FOutShapeInfo &getShapeInfo(void)
    { return mShapeInfo; }

    /**
    頂点属性が使われているかどうかのフラグを頂点属性データから設定
    \par 解説：
    　頂点属性が使われているかどうかのフラグ(mVtxAttrFlag)を
    頂点属性データ(ROutShapeInfo)から設定する。
    頂点属性データの各チャンネルにデータ配列が設定されているかどうかでフラグを決定する。
    この関数を呼ぶ前に、ROutShapeInfoを全て設定しておく必要が有る。
    \par
    　スキニングのためのブレンドウェイトとインデックスが有効かどうかは
    setVertexBoneRefArray() で頂点ボーンウェイトを設定したかによって決まる。
    また参照するボーンが一つだった場合には自動的にスキニングが無効に変更される。
    そのため本関数での RPrimVtx::IDX と RPrimVtx::WGT の設定は行われない。
    */
    void setVtxAttrUsedFromShapeInfo(void);

    /**
    シェイプのスキニングモードを調べる。
    \param[in] NodeIndex	シェイプの属するノードのボーンインデックス。
    \param[in] vtxMtxs		頂点行列リスト。
    \param[in,out] bones	全てのノード（ボーン）のリスト。
    \param[in] adjustsSmoothSkinning	モデル単位でスムーススキニングと判断するか
    \par 解説：
    　シェイプに登録されている頂点情報からシェイプごとのスキニングモードを設定します。
    シェイプ中の頂点において複数のボーンが使用されている場合は SmoothSkinning。
    各頂点において一つだけの場合は RigidSkinning。
    全ての頂点において同じ一つのボーンの場合は NoSkinning。
    ※adjustsSmoothSkinningが有効の場合、モデル単位で上記のチェックします。
    */
    void getShapeSkinningMode( const int NodeIndex, const nn::gfx::tool::dcc::RVtxMtxArray& vtxMtxs, std::vector<RNode*>&	bones, const bool adjustsSmoothSkinning );

    /**
    モデル名を取得
    \return モデル名
    \par 解説：
    　コンストラクタに指定したモデル名を取得する。
    */
    const std::string &getOrgName(void) const
    { return mOrgName; }

    /**
    シェイプ数を取得する
    \return このモデルに含まれるシェイプの数
    \par 解説：
    　マテリアルの数を取得する。
    これはポリゴン登録時に指定されたマテリアルIDの数となり、
    内部で保持する RShape の数と一致する。
    */
    const int getNumShapes(void) const
    { return static_cast<int>(mShapeDataList.size()); }

    /**
    シェイプのインスタンスを取得する。
    \return シェイプインスタンス
    */
    FShape* getShape( int index )
    { return mShapeDataList[index]; }

    /**
    シェイプと頂点情報から法線マップ用のタンジェントを計算する。
    \param texCh 計算に使用するUV座標チャンネル
    \param isUVBackward UV座標が逆向きに貼られている場合(3dsmax)
    \return
    \par 解説：
    　シェイプと頂点情報(ShapeInfo)から法線マップに必要なタンジェントを計算して設定する。
    */
    nn::gfx::tool::dcc::RStatus calcTangent(nn::gfx::tool::dcc::RPrimVtx::VtxAttr texCh, bool isUVBackward = false);

/**
\}
\name 実装(内部使用)：
\{
*/
    /**
    スキニングに関するモードおよび座標系の変換処理を行う
    \param node このモデルを持つノード
    \param vtxMtxs 頂点行列リスト。
    \par 解説：
    */
    void convertSkinningData(RNode* node, const std::vector<RNode*>&	NodeList, const nn::gfx::tool::dcc::RVtxMtxArray& vtxMtxs);

    /**
    モデルデータの整理を行う
    \par 解説：
    　以下の処理を行う。
    - 頂点属性データ(頂点データ)の重複したデータのマージ RShape::mergeSameVertexData()
    - 頂点属性データ(頂点データ)の参照されていない頂点の削除 RShape::deleteUnusedVertexData()
    - プリミティブ(三角形)からプリミティブセットを作成。 RShape::makePrimSets()
    - プリミティブセットの結合と順番の最適化 RShape::optimizePrimSets()
    - ？？？ RPrimSet::MergeSinglePrimitivesInSets()
    - 行列パレット内のボーンの並びを決定 RShape::makeMtxPalette()
    - 頂点属性データの RPrimVtx::IDX と RPrimVtx::WGT を決定 RShape::setSkinningAttr()
    \par
    　 getShapeInfo() から返される ROutShapeInfo オブジェクト自体は変更されない。
    失敗した場合falseを返すが、現在のところ実装バグ以外のエラーは定義されていない。
    */
    bool optimize(
        const nn::gfx::tool::dcc::RVtxMtxArray& vtxMtxs,
        const nn::gfx::tool::dcc::RIntArray& smoothMtxIdxs,
        const nn::gfx::tool::dcc::RIntArray& rigidMtxIdxs,
        const RSceneMaterials& sceneMaterials,
        float magnify = 1.0f);

/** \} */
protected:
    typedef std::vector<FShape*>	ShapeDataList;
protected:
    /**
    マテリアルIDに対応した RShape を取得もしくは作成
    \param[in] materialID マテリアルID
    \return RShape
    \par 解説：
    　マテリアルIDに対応した RShape を取得する。
    このマテリアルIDに対応する RShape がまだ無ければ新規に作成して返す。
    */
    FShape* getRShape(int materialID);

protected:
    /**
    行列パレットのサイズ
    */
    int mMtxPalSize;

    /**
    頂点属性が使われているかどうかのフラグ
    \par
    　スキニングのためのブレンドウェイトとインデックスが有効かどうかは
    setVertexBoneRefArray() で頂点ボーンウェイトを設定したかによって決まる。
    そのため RPrimVtx::IDX と RPrimVtx::WGT の設定は無視される。
    */
    bool mVtxAttrFlag[nn::gfx::tool::dcc::RPrimVtx::VA_COUNT];

    /**
    シェイプ名
    \par 解説：
    　デバッグ用途(node_name + material_name)
    */
    std::string mOrgName;

    /**
    シェイプ処理の進行状況を伝えるオブジェクト
    */
    nn::gfx::tool::dcc::RProgShape* mProgShapePtr;

    /**
    このモデルが持っているシェイプのリスト
    */
    ShapeDataList	mShapeDataList;

    /**
        頂点属性データ(頂点データ)
    */
    FOutShapeInfo mShapeInfo;

//	TODO:	フラグの置き場所を考えたい
public:
    int m_SmoothSkinningShapeCount; //!< スムーススキニングのシェイプ数です。
    int m_RigidSkinningShapeCount; //!< リジッドスキニングのシェイプ数です。
};

/******************************************************************************
    end name space utility
******************************************************************************/
}}}}} // namespace utility
