﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/******************************************************************************
    include
******************************************************************************/
#include "DccUtilityLogger.h"

using namespace std;
namespace Dcc = nn::gfx::tool::dcc;

/******************************************************************************
    begin name space utility
******************************************************************************/
namespace nn {
namespace gfx {
namespace tool {
namespace dcc {
namespace utility {

/******************************************************************************
    static member definition
******************************************************************************/
RLogger* RLogger::mInstance = nullptr;

//----------------------------------------------------------------------------
//	デストラクタ
RLogger::~RLogger(void)
{
}

//----------------------------------------------------------------------------
//	シングルトンアクセッサ
RLogger* RLogger::get(void)
{
    if( mInstance == nullptr )
    {
        mInstance = new RLogger();
    }

    return mInstance;
}


//----------------------------------------------------------------------------
//	メッセージロギングメソッド
RStatus RLogger::logMessage( const std::string&	message, RMessageType type )
{
    if( mCallback != nullptr )
    {
        if(type == kWarning) mWarningCount++;
        return (*mCallback)( message, type );
    }

    return RStatus( RStatus::FAILURE, "Loggin callback is NULL." );
}

RStatus RLogger::LogMessage( const std::string& message, RMessageType type /*= kError */ )
{
    return get()->logMessage(message, type);
}

RStatus RLogger::LogMessageFormat( const char* format, ... )
{
    const int STR_SIZE = 1024;
    char* pStr = new char[STR_SIZE];
    va_list args;
    va_start(args, format);
    const int size = _vsnprintf_s(pStr, STR_SIZE, _TRUNCATE, format, args);
    va_end(args);
    auto ret = get()->logMessage(string(pStr));
    delete[] pStr;
    return ret;
}

//----------------------------------------------------------------------------
//	メッセージロギングメソッド
RStatus RLogger::logMessagebyID(RMessageDetail id, std::string option)
{
    if( mCallback == nullptr )
    {
        return RStatus( RStatus::FAILURE, "Loggin callback is NULL." );
    }

    std::string message;
    RMessageType type;
    switch(id)
    {
        // type error
        case kLogERR_FindCS:
            message = "CreativeStudio cannot be found"; break;
        case kLogERR_FindOptimizer:
            message = "3D intermediate file optimizer cannot be found: "; break;
        case kLogERR_GetTextureSize:
            message = "Can't get texture size"; break;
        case kLogERR_OpenFile:
            message = "Can't open file"; break;
        case kLogERR_StartCS:
            message = "Failed to start CreativeStudio"; break;
        case kLogERR_HoledPolygon:
            message = "Holed polygon exists"; break;
        case kLogERR_InfluenceObj:
            message = "Influence object is not outputted"; break;
        case kLogERR_MultipleBindPoses:
            message = "Multiple bind poses are not supported"; break;
        case kLogERR_NoEffectiveNode:
            message = "No effective node"; break;
        case kLogERR_NoEffectivePolygon:
            message = "No effective polygon"; break;
        case kLogERR_NoOutputFile:
            message = "No output file"; break;
        case kLogERR_NoShadingGroup:
            message = "No shading group"; break;
        case kLogERR_NodeForOutputFileName:
            message = "Node for Output File Name is not selected"; break;
        case kLogERR_OutputFileName:
            message = "Output File Name is wrong"; break;
        case kLogERR_OutputFolder:
            message = "Output Folder is wrong"; break;
        case kLogERR_SameUserAttrib:
            message = "Same user attribute exists"; break;
        case kLogERR_TextureImageName:
            message = "Texture image name is wrong"; break;
        case kLogERR_TextureSize:
            message = "Texture size is wrong"; break;
        case kLogERR_TGA_File:
            message = "TGA file is wrong"; break;
        case kLogERR_NumberOfTexture:
        case kLogWRN_NumberOfTexture:
            message = "The number of composite texture is over 3"; break;
        case kLogERR_UV_Linking:
            message = "UV linking is not identical"; break;
        case kLogERR_VertexNumberIsDifferent:
            message = "Vertex number is different from original mesh"; break;
        case kLogERR_WrongWeightedVertices:
            message = "Wrong weighted vertices exist"; break;
        case kLogERR_PolygonBoneSizeIsOver:
            message = "Polygon bone size is over"; break;
        case kLogERR_MaxReservedUniformRegistersIsWrong:
            message = "Max Reserved Uniform Registers is wrong"; break;
        case kLogERR_InternalError:
            message = "Internal Error"; break;
        case kLogERR_PathOfMergeIsWrong:
            message = "Path of fmd file to merge is wrong"; break;
        case kLogERR_TheNumberOfVerticesExceedsTheLimitation:
            message = "The number of vertices exceeds the limitation"; break;
        case kLogERR_CantExportFtxFile:
            message = "Can't export ftx file (See Script Editor for details)"; break;
        case kLogERR_TextureConverterCantBeFound:
            message = "Texture converter can't be found: "; break;
        case kLogERR_FtxFileToMergeIsNotFound:
        case kLogWRN_FtxFileToMergeIsNotFound:
            message = "ftx file to merge is not found"; break;
        case kLogERR_Optimization:
            message = "Optimization failed (See Script Editor for details)"; break;
        case kLogERR_MergeAnimationFolderIsWrong:
            message = "Merge Animation Folder is wrong"; break;
        case kLogERR_MergeAnimationNameIsWrong:
            message = "Merge Animation Name is wrong"; break;
        case kLogERR_AnimationFileToMergeIsNotFound:
        case kLogWRN_AnimationFileToMergeIsNotFound:
            message = "Animation file to merge is not found"; break;
        case kLogERR_SamplerNameIsWrong:
            message = "Sampler name is wrong"; break;
        case kLogERR_SamplerHintIsWrong:
            message = "Sampler hint is wrong"; break;
        case kLogERR_SamplerNameIsDuplicate:
            message = "Sampler name is duplicate"; break;
        case kLogERR_SamplerHintIsDuplicate:
            message = "Sampler hint is duplicate"; break;
        case kLogERR_CommentIsWrong:
            message = "Comment is wrong (please don't use [;] [double quote])"; break;
        case kLogERR_UvSetSameHintExists:
            message = "UV set for same hint exists"; break;

        //
        // type warning
        case kLogWRN_NotConnectedToFrame:
            message = "Animation curve or expression is not connected to frame extension"; break;
        case kLogWRN_ExportNURBS:
            message = "Can't export NURBS surface"; break;
        case kLogWRN_ExportSubdivision:
            message = "Can't export Subdivision surface"; break;
        case kLogWRN_GetUV:
            message = "Can't get UV"; break;
        case kLogWRN_DeformerIsNotEffective:
            message = "Deformer is not effective"; break;
        case kLogWRN_FileTexIsNotConnected:
            message = "File texture is not connected"; break;
        case kLogWRN_FrameOffsetIsNotZero:
            message = "Frame offset is not zero"; break;
        case kLogWRN_InheritsTransformIsOff:
            message = "Inherits transform of non-skinned object is off"; break;
        case kLogWRN_MaterialNameIsChanged:
            message = "Material name is changed"; break;
        case kLogWRN_NoShader:
            message = "No shader"; break;
        case kLogWRN_NodeNameIsChanged:
            message = "Node name is changed"; break;
        case kLogWRN_FileNameIsChanged:
            message = "Output File Name is changed"; break;
        case kLogWRN_TexIsIgnored:
            message = "Procedural texture is ignored"; break;
        case kLogWRN_ProjectionMappingIsNotSupported:
            message = "Projection mapping is not supported"; break;
        case kLogWRN_PSD_File:
            message = "PSD file texture is ignored"; break;
        case kLogWRN_NonSupportedShader:
            message = "Shader is not supported"; break;
        case kLogWRN_ShearIsNotZero:
            message = "Shear is not zero"; break;
        case kLogWRN_StencilMappingIsNotSupported:
            message = "Stencil mapping is not supported"; break;
        case kLogWRN_TexFileForTexAnimIsNotFound:
            message = "Texture file for texture pattern animation is not found"; break;
        case kLogWRN_TexNameIsChanged:
            message = "Texture name is changed"; break;
        case kLogWRN_MultipleShape:
            message = "There are multiple shape"; break;
        case kLogWRN_MultipleTexWithSameName:
            message = "There are multiple texture files with the same name"; break;
        case kLogWRN_ZeroNormalExist:
            message = "Zero normal exist"; break;
        case kLogWRN_TexFileIsNotFound:
            message = "Texture file doesn't exist"; break;
        case kLogWRN_NonSupportedTex:
            message = "Texture type is not supported"; break;
        case kLogWRN_NonSupportedTexFormat:
            message = "Texture format is not supported"; break;
        case kLogWRN_MapChannelDoesNotExist:
            message = "Map channel does not exist"; break;
        case kLogWRN_OptimizationFailed:
            message = "Optimization failed"; break;
        case kLogWRN_GeneratingTangentFailed:
            message = "Can't get UV data for generating the tangent"; break;
        case kLogWRN_BillboardAndConvertToModel:
            message = "Billboard Type is ignored if 'Convert To Model' option is ON"; break;
        case kLogWRN_NonBoneBetweenBones:
            message = "Node isn't bound to skin, but parent and children are bound."; break;
        case kLogWRN_Optimization:
            message = "Optimization warning (See Script Editor for details)"; break;
        case kLogWRN_ZeroTangentExists:
            message = "Zero tangent or binormal exists"; break;
        case kLogWRN_MorpherMustBeUnderSkinModifier:
            message = "Morpher modifier must be under skin(physique) modifier"; break;
        case kLogWRN_MorpherTargetTopologyIsDifferentFromOriginalMesh:
            message = "Morpher target topology is different from original mesh"; break;
        case kLogWRN_UvSetNameIsWrong:
            message = "UV set name is wrong"; break;
        default:
            message = "Unknown Error"; break;
    }

    // IDでログの種類を決定する。
    if(id < kLogERR_END)
    {
        type = kError;
    }
    else if(id < kLogWRN_END)
    {
        type = kWarning;
        mWarningCount++;
    }
    else
    {
        type = kInfo;
    }

    if(option.size() > 0)
    {
        message += "(";
        message += option + ")";
    }

    return (*mCallback)( message, type );
}

RStatus RLogger::LogMessagebyID( RMessageDetail id, std::string option /*= ""*/ )
{
    return get()->logMessagebyID(id, option);
}

/******************************************************************************
    end name space utility
******************************************************************************/
}}}}} // namespace utility

/******************************************************************************
-------------------------------------------------------------------------------
                end of file
-------------------------------------------------------------------------------
*******************************************************************************/
