﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "NintendoExportMax.h"
#include "DccUtilityOptimizer.h"

#include "DccMaterial.h"

using namespace nn::gfx::tool::dcc::utility;

extern Dcc::RVtxMtxArray	g_VtxMtxs;
extern CStr SearchAnim( BOOL selected );

int	DoExportCafe(const TCHAR* name, Interface* i, BOOL suppressPrompts, DWORD options);

//-----------------------------------------------------------------------------

static bool isInitExportOptions = false;
static NintendoOptions exportOptions;
NintendoOptions* GetNintendoOptions()
{
    if (!isInitExportOptions)
    {
        NintendoExport::Get()->Init();
        isInitExportOptions = true;
    }
    return &exportOptions;
}
//
static NintendoExportClassDesc nw4fmaxDesc;
ClassDesc2* GetNintendoExportDesc() { return &nw4fmaxDesc; }

static NintendoAttribClassDesc nw4fAttribDesc;
ClassDesc2* GetNintendoAttribDesc() {return &nw4fAttribDesc;}

static NintendoNodeAttribClassDesc nw4fNodeAttribDesc;
ClassDesc2* GetNintendoNodeAttribDesc() {return &nw4fNodeAttribDesc;}

static NintendoMtlAttribClassDesc nw4fMtlAttribDesc;
ClassDesc2* GetNintendoMtlAttribDesc() {return &nw4fMtlAttribDesc;}

static NintendoSamplerAttribClassDesc nw4fSamplerAttribDesc;
ClassDesc2* GetNintendoSamplerAttribDesc() {return &nw4fSamplerAttribDesc;}

static NintendoMultiExportAttribClassDesc nw4fMultiExportAttribDesc;
ClassDesc2* GetNintendoMultiExportAttribDesc() {return &nw4fMultiExportAttribDesc;}


static NintendoExport_Imp nw4fmaxInterface
(
    NINTENDOEXPORT_INTERCFACE_ID, _T("iNintendoExport"), IDS_NW4F_INTERFACE, &nw4fmaxDesc, 0,
        NintendoExport_Imp::nw4f_startExport, _T("DoExport"), 0, TYPE_BOOL, 0, 1,
            _T("ExportSelection"), 0, TYPE_BOOL,
        NintendoExport_Imp::nw4f_searchAnimations, _T("SearchAnimations"), 0, TYPE_TSTR_BV, 0, 1,
            _T("ExportSelection"), 0, TYPE_BOOL,
        NintendoExport_Imp::nw4f_resetSetting, _T("ResetSetting"), 0, TYPE_VOID,0,0,
        NintendoExport_Imp::nw4f_loadSetting, _T("LoadSetting"), 0, TYPE_VOID,0,0,
        NintendoExport_Imp::nw4f_saveSetting, _T("SaveSetting"), 0, TYPE_VOID,0,0,
        NintendoExport_Imp::nw4f_getVersion, _T("GetVersion"), 0, TYPE_INT,0,0,
        NintendoExport_Imp::nw4f_getVersionString, _T("GetVersionString"), 0, TYPE_STRING,0,0,
        NintendoExport_Imp::nw4f_getDateTimeString, _T("GetDateTimeString"), 0, TYPE_STRING,0,0,
        NintendoExport_Imp::nw4f_getBuildDateTimeString, _T("GetBuildDateTimeString"), 0, TYPE_STRING,0,0,
        NintendoExport_Imp::nw4f_getMtlEditorMenu, _T("GetMtlEditorMenu"), 0, TYPE_INTERFACE, 0,0,
        NintendoExport_Imp::nw4f_getErrorLog, _T("GetErrorLog"), 0, TYPE_TSTR_BV, 0, 0,
        NintendoExport_Imp::nw4f_setOptionEnvVar, _T("SetOptionEnvVar"), 0, TYPE_VOID,0,2,
            _T("IsPostExport"), 0, TYPE_BOOL,
            _T("IsSucceedExport"), 0, TYPE_BOOL,
        NintendoExport_Imp::nw4f_LayerTest, _T("LayerTest"), 0, TYPE_VOID,0,0,
        NintendoExport_Imp::nw4f_useFclFtsFtp, _T("GetUseFclFtsFtp"), 0, TYPE_BOOL,0,0,
        NintendoExport_Imp::nw4f_separateMaterialAnim, _T("GetSeparateMaterialAnim"), 0, TYPE_BOOL,0,0,

    properties,
        NintendoExport_Imp::nw4f_getPresetName, NintendoExport_Imp::nw4f_setPresetName, _T("PresetName"), 0, TYPE_STRING,
        NintendoExport_Imp::nw4f_getPresetFile, NintendoExport_Imp::nw4f_setPresetFile, _T("PresetFile"), 0, TYPE_STRING,
        NintendoExport_Imp::nw4f_getShowPresetUI, NintendoExport_Imp::nw4f_setShowPresetUI, _T("ShowPresetUI"), 0, TYPE_BOOL,
        NintendoExport_Imp::nw4f_getDoesExportSelected, NintendoExport_Imp::nw4f_setDoesExportSelected,
            _T("doesExportSelected"), 0, TYPE_BOOL,
        NintendoExport_Imp::nw4f_getOutFolder, NintendoExport_Imp::nw4f_setOutFolder,
            _T("outputFolder"), 0, TYPE_STRING,
        NintendoExport_Imp::nw4f_getOutName, NintendoExport_Imp::nw4f_setOutName, _T("filename"), 0, TYPE_STRING,

        NintendoExport_Imp::nw4f_getUseMerge, NintendoExport_Imp::nw4f_setUseMerge, _T("useMerge"), 0, TYPE_BOOL,
        NintendoExport_Imp::nw4f_getMergeFile, NintendoExport_Imp::nw4f_setMergeFile, _T("mergeFilename"), 0, TYPE_STRING,
        NintendoExport_Imp::nw4f_getMergeAuto, NintendoExport_Imp::nw4f_setMergeAuto, _T("useAutoMerge"), 0, TYPE_BOOL,

        NintendoExport_Imp::nw4f_getMagnify, NintendoExport_Imp::nw4f_setMagnify, _T("magnify"), 0, TYPE_FLOAT,
        NintendoExport_Imp::nw4f_getUseFigureMode, NintendoExport_Imp::nw4f_setUseFigureMode, _T("UseFigureMode"), 0, TYPE_BOOL,
        NintendoExport_Imp::nw4f_getCommentText, NintendoExport_Imp::nw4f_setCommentText, _T("commentText"), 0, TYPE_STRING,
        //
        NintendoExport_Imp::nw4f_getUseBinaryFormat, NintendoExport_Imp::nw4f_setUseBinaryFormat, _T("isBinaryFormat"), 0, TYPE_BOOL,

        NintendoExport_Imp::nw4f_getDoesExportModel, NintendoExport_Imp::nw4f_setDoesExportModel, _T("doesExportModel"), 0, TYPE_BOOL,
        NintendoExport_Imp::nw4f_getDoesExportSkelAnim, NintendoExport_Imp::nw4f_setDoesExportSkelAnim, _T("doesExportSkelAnim"), 0, TYPE_BOOL,
        NintendoExport_Imp::nw4f_getDoesExportVisBoneAnim, NintendoExport_Imp::nw4f_setDoesExportVisBoneAnim, _T("doesExportVisBoneAnim"), 0, TYPE_BOOL,
        NintendoExport_Imp::nw4f_getDoesExportMtlColAnim, NintendoExport_Imp::nw4f_setDoesExportMtlColAnim, _T("doesExportMtlColorAnim"), 0, TYPE_BOOL,
        NintendoExport_Imp::nw4f_getDoesExportMtlTexPatAnim, NintendoExport_Imp::nw4f_setDoesExportMtlTexPatAnim, _T("doesExportMtlTexPatternAnim"), 0, TYPE_BOOL,
        NintendoExport_Imp::nw4f_getDoesExportMtlTexSRTAnim, NintendoExport_Imp::nw4f_setDoesExportMtlTexSRTAnim, _T("doesExportMtlTexSRTAnim"), 0, TYPE_BOOL,
        NintendoExport_Imp::nw4f_getDoesExportShapeAnim, NintendoExport_Imp::nw4f_setDoesExportShapeAnim, _T("doesExportShapeAnim"), 0, TYPE_BOOL,

        NintendoExport_Imp::nw4f_getDoesExportCam, NintendoExport_Imp::nw4f_setDoesExportCam, _T("doesExportCamera"), 0, TYPE_BOOL,
        NintendoExport_Imp::nw4f_getDoesExportLight, NintendoExport_Imp::nw4f_setDoesExportLight, _T("doesExportLight"), 0, TYPE_BOOL,
        NintendoExport_Imp::nw4f_getDoesExportSceneAnim, NintendoExport_Imp::nw4f_setDoesExportSceneAnim, _T("doesExportSceneAnim"), 0, TYPE_BOOL,
        NintendoExport_Imp::nw4f_getDoesExportEnv, NintendoExport_Imp::nw4f_setDoesExportEnv, _T("doesExportEnv"), 0, TYPE_BOOL,

        NintendoExport_Imp::nw4f_getDoesExportTex, NintendoExport_Imp::nw4f_setDoesExportTex, _T("doesExportTexture"), 0, TYPE_BOOL,

        NintendoExport_Imp::nw4f_getCompressBone, NintendoExport_Imp::nw4f_setCompressBone, _T("compressNode"), 0, TYPE_INT,
        NintendoExport_Imp::nw4f_getDoesUniteChild, NintendoExport_Imp::nw4f_setDoesUniteChild, _T("doesUniteChild"), 0, TYPE_BOOL,
        NintendoExport_Imp::nw4f_getDoesCompressMtl, NintendoExport_Imp::nw4f_setDoesCompressMtl, _T("doesCompressMaterial"), 0, TYPE_BOOL,
        NintendoExport_Imp::nw4f_getDoesCompressShape, NintendoExport_Imp::nw4f_setDoesCompressShape, _T("doesCompressShape"), 0, TYPE_BOOL,
        NintendoExport_Imp::nw4f_getDoesOptimizePrim, NintendoExport_Imp::nw4f_setDoesOptimizePrim, _T("doesOptimizePrimitive"), 0, TYPE_BOOL,
        NintendoExport_Imp::nw4f_getDoesQuantizationAnalize, NintendoExport_Imp::nw4f_setDoesQuantizationAnalize, _T("doesQuantizationAnalyze"), 0, TYPE_BOOL,

        NintendoExport_Imp::nw4f_getQuantPos,		NintendoExport_Imp::nw4f_setQuantPos, _T("quantPos"), 0, TYPE_INT,
        NintendoExport_Imp::nw4f_getQuantNormal,	NintendoExport_Imp::nw4f_setQuantNormal, _T("quantNormal"), 0, TYPE_INT,
        NintendoExport_Imp::nw4f_getQuantTex,		NintendoExport_Imp::nw4f_setQuantTex, _T("quantTex"), 0, TYPE_INT,

        NintendoExport_Imp::nw4f_getDoesExportAllFrames, NintendoExport_Imp::nw4f_setDoesExportAllFrames, _T("doesExportAllFrames"), 0, TYPE_BOOL,
        NintendoExport_Imp::nw4f_getStartFrame, NintendoExport_Imp::nw4f_setStartFrame, _T("startFrame"), 0, TYPE_INT,
        NintendoExport_Imp::nw4f_getEndFrame, NintendoExport_Imp::nw4f_setEndFrame, _T("endFrame"), 0, TYPE_INT,
        NintendoExport_Imp::nw4f_getFramePrecision, NintendoExport_Imp::nw4f_setFramePrecision, _T("framePrecision"), 0, TYPE_INT,
        NintendoExport_Imp::nw4f_getIsLoop, NintendoExport_Imp::nw4f_setIsLoop, _T("isLoop"), 0, TYPE_BOOL,

        NintendoExport_Imp::nw4f_getToleranceTrans,	NintendoExport_Imp::nw4f_setToleranceTrans, _T("toleranceTrans"), 0, TYPE_FLOAT,
        NintendoExport_Imp::nw4f_getToleranceRotate,	NintendoExport_Imp::nw4f_setToleranceRotate, _T("toleranceRotate"), 0, TYPE_FLOAT,
        NintendoExport_Imp::nw4f_getToleranceScale,	NintendoExport_Imp::nw4f_setToleranceScale, _T("toleranceScale"), 0, TYPE_FLOAT,

        NintendoExport_Imp::nw4f_getToleranceUVTrans,	NintendoExport_Imp::nw4f_setToleranceUVTrans,	_T("toleranceUVTrans"), 0, TYPE_FLOAT,
        NintendoExport_Imp::nw4f_getToleranceUVRotate,	NintendoExport_Imp::nw4f_setToleranceUVRotate, _T("toleranceUVRotate"), 0, TYPE_FLOAT,
        NintendoExport_Imp::nw4f_getToleranceUVScale,	NintendoExport_Imp::nw4f_setToleranceUVScale,	_T("toleranceUVScale"), 0, TYPE_FLOAT,
        NintendoExport_Imp::nw4f_getToleranceColor,	NintendoExport_Imp::nw4f_setToleranceColor,	_T("toleranceColor"), 0, TYPE_FLOAT,

        NintendoExport_Imp::nw4f_getQuantToleranceTrans,	NintendoExport_Imp::nw4f_setQuantToleranceTrans, _T("quantToleranceTrans"), 0, TYPE_FLOAT,
        NintendoExport_Imp::nw4f_getQuantToleranceRotate,	NintendoExport_Imp::nw4f_setQuantToleranceRotate, _T("quantToleranceRotate"), 0, TYPE_FLOAT,
        NintendoExport_Imp::nw4f_getQuantToleranceScale,	NintendoExport_Imp::nw4f_setQuantToleranceScale, _T("quantToleranceScale"), 0, TYPE_FLOAT,

        NintendoExport_Imp::nw4f_getQuantToleranceUVTrans,	NintendoExport_Imp::nw4f_setQuantToleranceUVTrans,	_T("quantToleranceUVTrans"), 0, TYPE_FLOAT,
        NintendoExport_Imp::nw4f_getQuantToleranceUVRotate,NintendoExport_Imp::nw4f_setQuantToleranceUVRotate, _T("quantToleranceUVRotate"), 0, TYPE_FLOAT,
        NintendoExport_Imp::nw4f_getQuantToleranceUVScale,	NintendoExport_Imp::nw4f_setQuantToleranceUVScale,	_T("quantToleranceUVScale"), 0, TYPE_FLOAT,
        NintendoExport_Imp::nw4f_getQuantToleranceFrame,	NintendoExport_Imp::nw4f_setQuantToleranceFrame,	_T("quantToleranceFrame"), 0, TYPE_FLOAT,

        NintendoExport_Imp::nw4f_getUseMergeFtx, NintendoExport_Imp::nw4f_setUseMergeFtx, _T("useMergeFtx"), 0, TYPE_BOOL,
        NintendoExport_Imp::nw4f_getUseMergeAnim, NintendoExport_Imp::nw4f_setUseMergeAnim, _T("useMergeAnim"), 0, TYPE_BOOL,
        NintendoExport_Imp::nw4f_getMergeAnimFolder, NintendoExport_Imp::nw4f_setMergeAnimFolder, _T("mergeAnimFolder"), 0, TYPE_STRING,
        NintendoExport_Imp::nw4f_getMergeAnimName, NintendoExport_Imp::nw4f_setMergeAnimName, _T("mergeAnimName"), 0, TYPE_STRING,

        NintendoExport_Imp::nw4f_getDisableZYAxisConversion, NintendoExport_Imp::nw4f_setDisableZYAxisConversion, _T("DisableZYAxisConversion"), 0, TYPE_BOOL,

        //buf
        NintendoExport_Imp::nw4f_getPreExportScript,	NintendoExport_Imp::nw4f_setPreExportScript,	_T("PreExportScript"), 0, TYPE_STRING,
        NintendoExport_Imp::nw4f_getPostExportScript,	NintendoExport_Imp::nw4f_setPostExportScript,	_T("PostExportScript"), 0, TYPE_STRING,

        //lock
        NintendoExport_Imp::nw4f_getLock_export_target, NintendoExport_Imp::nw4f_setLock_export_target, _T("Lock_export_target"), 0, TYPE_BOOL,
        NintendoExport_Imp::nw4f_getLock_output_file_name, NintendoExport_Imp::nw4f_setLock_output_file_name, _T("Lock_output_file_name"), 0, TYPE_BOOL,
        NintendoExport_Imp::nw4f_getLock_output_folder, NintendoExport_Imp::nw4f_setLock_output_folder, _T("Lock_output_folder"), 0, TYPE_BOOL,
        NintendoExport_Imp::nw4f_getLock_use_figure_mode, NintendoExport_Imp::nw4f_setLock_use_figure_mode, _T("Lock_use_figure_mode"), 0, TYPE_BOOL,
        NintendoExport_Imp::nw4f_getLock_merge_fmd, NintendoExport_Imp::nw4f_setLock_merge_fmd, _T("Lock_merge_fmd"), 0, TYPE_BOOL,
        NintendoExport_Imp::nw4f_getLock_merge_ftx, NintendoExport_Imp::nw4f_setLock_merge_ftx, _T("Lock_merge_ftx"), 0, TYPE_BOOL,
        NintendoExport_Imp::nw4f_getLock_merge_anim, NintendoExport_Imp::nw4f_setLock_merge_anim, _T("Lock_merge_anim"), 0, TYPE_BOOL,
        NintendoExport_Imp::nw4f_getLock_merge_anim_name, NintendoExport_Imp::nw4f_setLock_merge_anim_name, _T("Lock_merge_anim_name"), 0, TYPE_BOOL,
        NintendoExport_Imp::nw4f_getLock_magnify, NintendoExport_Imp::nw4f_setLock_magnify, _T("Lock_magnify"), 0, TYPE_BOOL,
        NintendoExport_Imp::nw4f_getLock_texsrt_mode, NintendoExport_Imp::nw4f_setLock_texsrt_mode, _T("Lock_texsrt_mode"), 0, TYPE_BOOL,
        NintendoExport_Imp::nw4f_getLock_remove_namespace, NintendoExport_Imp::nw4f_setLock_remove_namespace, _T("Lock_remove_namespace"), 0, TYPE_BOOL,
        NintendoExport_Imp::nw4f_getLock_comment, NintendoExport_Imp::nw4f_setLock_comment, _T("Lock_comment"), 0, TYPE_BOOL,
        NintendoExport_Imp::nw4f_getLock_frame_range, NintendoExport_Imp::nw4f_setLock_frame_range, _T("Lock_frame_range"), 0, TYPE_BOOL,
        NintendoExport_Imp::nw4f_getLock_loop_anim, NintendoExport_Imp::nw4f_setLock_loop_anim, _T("Lock_loop_anim"), 0, TYPE_BOOL,
        NintendoExport_Imp::nw4f_getLock_bake_all_anim, NintendoExport_Imp::nw4f_setLock_bake_all_anim, _T("Lock_bake_all_anim"), 0, TYPE_BOOL,
        NintendoExport_Imp::nw4f_getLock_frame_precision, NintendoExport_Imp::nw4f_setLock_frame_precision, _T("Lock_frame_precision"), 0, TYPE_BOOL,
        NintendoExport_Imp::nw4f_getLock_output_fmd, NintendoExport_Imp::nw4f_setLock_output_fmd, _T("Lock_output_fmd"), 0, TYPE_BOOL,
        NintendoExport_Imp::nw4f_getLock_output_ftx, NintendoExport_Imp::nw4f_setLock_output_ftx, _T("Lock_output_ftx"), 0, TYPE_BOOL,
        NintendoExport_Imp::nw4f_getLock_output_fsk, NintendoExport_Imp::nw4f_setLock_output_fsk, _T("Lock_output_fsk"), 0, TYPE_BOOL,
        NintendoExport_Imp::nw4f_getLock_output_fvb, NintendoExport_Imp::nw4f_setLock_output_fvb, _T("Lock_output_fvb"), 0, TYPE_BOOL,
        NintendoExport_Imp::nw4f_getLock_output_fcl, NintendoExport_Imp::nw4f_setLock_output_fcl, _T("Lock_output_fcl"), 0, TYPE_BOOL,
        NintendoExport_Imp::nw4f_getLock_output_fts, NintendoExport_Imp::nw4f_setLock_output_fts, _T("Lock_output_fts"), 0, TYPE_BOOL,
        NintendoExport_Imp::nw4f_getLock_output_ftp, NintendoExport_Imp::nw4f_setLock_output_ftp, _T("Lock_output_ftp"), 0, TYPE_BOOL,
        NintendoExport_Imp::nw4f_getLock_output_fsh, NintendoExport_Imp::nw4f_setLock_output_fsh, _T("Lock_output_fsh"), 0, TYPE_BOOL,
        NintendoExport_Imp::nw4f_getLock_output_fsn, NintendoExport_Imp::nw4f_setLock_output_fsn, _T("Lock_output_fsn"), 0, TYPE_BOOL,
        NintendoExport_Imp::nw4f_getLock_compress_bone, NintendoExport_Imp::nw4f_setLock_compress_bone, _T("Lock_compress_bone"), 0, TYPE_BOOL,
        NintendoExport_Imp::nw4f_getLock_unite_child, NintendoExport_Imp::nw4f_setLock_unite_child, _T("Lock_unite_child"), 0, TYPE_BOOL,
        NintendoExport_Imp::nw4f_getLock_compress_material, NintendoExport_Imp::nw4f_setLock_compress_material, _T("Lock_compress_material"), 0, TYPE_BOOL,
        NintendoExport_Imp::nw4f_getLock_compress_shape, NintendoExport_Imp::nw4f_setLock_compress_shape, _T("Lock_compress_shape"), 0, TYPE_BOOL,
        NintendoExport_Imp::nw4f_getLock_tolerance_scale, NintendoExport_Imp::nw4f_setLock_tolerance_scale, _T("Lock_tolerance_scale"), 0, TYPE_BOOL,
        NintendoExport_Imp::nw4f_getLock_tolerance_rotate, NintendoExport_Imp::nw4f_setLock_tolerance_rotate, _T("Lock_tolerance_rotate"), 0, TYPE_BOOL,
        NintendoExport_Imp::nw4f_getLock_tolerance_translate, NintendoExport_Imp::nw4f_setLock_tolerance_translate, _T("Lock_tolerance_translate"), 0, TYPE_BOOL,
        NintendoExport_Imp::nw4f_getLock_tolerance_color, NintendoExport_Imp::nw4f_setLock_tolerance_color, _T("Lock_tolerance_color"), 0, TYPE_BOOL,
        NintendoExport_Imp::nw4f_getLock_tolerance_tex_scale, NintendoExport_Imp::nw4f_setLock_tolerance_tex_scale, _T("Lock_tolerance_tex_scale"), 0, TYPE_BOOL,
        NintendoExport_Imp::nw4f_getLock_tolerance_tex_rotate, NintendoExport_Imp::nw4f_setLock_tolerance_tex_rotate, _T("Lock_tolerance_tex_rotate"), 0, TYPE_BOOL,
        NintendoExport_Imp::nw4f_getLock_tolerance_tex_translate, NintendoExport_Imp::nw4f_setLock_tolerance_tex_translate, _T("Lock_tolerance_tex_translate"), 0, TYPE_BOOL,
        NintendoExport_Imp::nw4f_getLock_quantize_tolerance_scale, NintendoExport_Imp::nw4f_setLock_quantize_tolerance_scale, _T("Lock_quantize_tolerance_scale"), 0, TYPE_BOOL,
        NintendoExport_Imp::nw4f_getLock_quantize_tolerance_rotate, NintendoExport_Imp::nw4f_setLock_quantize_tolerance_rotate, _T("Lock_quantize_tolerance_rotate"), 0, TYPE_BOOL,
        NintendoExport_Imp::nw4f_getLock_quantize_tolerance_translate, NintendoExport_Imp::nw4f_setLock_quantize_tolerance_translate, _T("Lock_quantize_tolerance_translate"), 0, TYPE_BOOL,
        NintendoExport_Imp::nw4f_getLock_quantize_tolerance_tex_scale, NintendoExport_Imp::nw4f_setLock_quantize_tolerance_tex_scale, _T("Lock_quantize_tolerance_tex_scale"), 0, TYPE_BOOL,
        NintendoExport_Imp::nw4f_getLock_quantize_tolerance_tex_rotate, NintendoExport_Imp::nw4f_setLock_quantize_tolerance_tex_rotate, _T("Lock_quantize_tolerance_tex_rotate"), 0, TYPE_BOOL,
        NintendoExport_Imp::nw4f_getLock_quantize_tolerance_tex_translate, NintendoExport_Imp::nw4f_setLock_quantize_tolerance_tex_translate, _T("Lock_quantize_tolerance_tex_translate"), 0, TYPE_BOOL,
        NintendoExport_Imp::nw4f_getLock_pre_export_script, NintendoExport_Imp::nw4f_setLock_pre_export_script, _T("Lock_pre_export_script"), 0, TYPE_BOOL,
        NintendoExport_Imp::nw4f_getLock_post_export_script, NintendoExport_Imp::nw4f_setLock_post_export_script, _T("Lock_post_export_script"), 0, TYPE_BOOL,


#if (MAX_VERSION_MAJOR >= 15) // max2013
    p_end
#else
    end
#endif

);

std::string AppendSlash(std::string path)
{
    if (path.find_last_of('/') != path.length() - 1)
    {
        path += "/";
    }

    return path;
}

//-----------------------------------------------------------------------------
//! @brief NintendoSDK ルートパスを返します。
//!
//! @return NintendoSDK ルートパスを返します。
//!         環境変数が定義されていなければ "" を返します。
//-----------------------------------------------------------------------------
std::string GetNintendoRootPath(void)
{
    std::string path;
    // nintendo sdk 対応
    path = Dcc::RGetEnvVar("NINTENDO_SDK_ROOT");
    if (!path.empty())
    {
        path = Dcc::RGetUnixFilePath(path);
        path = AppendSlash(path);
        if (Dcc::RFolderExists(path))
        {
            return path;
        }
    }

    path = Dcc::RGetEnvVar("NW4F_ROOT");
    if (!path.empty())
    {
        path = Dcc::RGetUnixFilePath(path);
        path = AppendSlash(path);
        if (Dcc::RFolderExists(path))
        {
            return path;
        }
    }
    return "";
}

std::string GetNw4fConfigPath()
{
    std::string path;
    // nintendo sdk 対応
    path = Dcc::RGetEnvVar("NINTENDO_SDK_ROOT");
    if (!path.empty())
    {
        path = Dcc::RGetUnixFilePath(path);
        path = AppendSlash(path);
        path += "Tools/Graphics/3dsMaxPlugins/Config/";
        if (Dcc::RFolderExists(path))
        {
            return path;
        }
    }

    path = Dcc::RGetEnvVar("NW4F_ROOT");
    if (!path.empty())
    {
        path = Dcc::RGetUnixFilePath(path);
        path = AppendSlash(path);
        path += "Tool/DccPlugin/3dsMax/Config/";
        if (Dcc::RFolderExists(path))
        {
            return path;
        }
    }
    return "";
}

std::string Get3dCommandLineToolsPath()
{
    std::string path;
    // nintendo sdk 対応
    path = Dcc::RGetEnvVar("NINTENDO_SDK_ROOT");
    if (!path.empty())
    {
        path = Dcc::RGetUnixFilePath(path);
        path = AppendSlash(path);
        path += "Tools/Graphics/3dTools/";
        if (Dcc::RFolderExists(path))
        {
            return path;
        }
    }

    path = Dcc::RGetEnvVar("NW4F_ROOT");
    if (!path.empty())
    {
        path = Dcc::RGetUnixFilePath(path);
        path = AppendSlash(path);
        path += "Tool/G3dTool/";
#ifdef _WIN64
        path += "win64/";
#else
        path += "win32/";
#endif
        if (Dcc::RFolderExists(path))
        {
            return path;
        }
    }
    return "";
}
/*static nw4f_nodecomp_Imp nw4fmaxNodecompInterface
(
    NW4F_NODECOMP_INTERCFACE_ID, _T("in4wc_nodecomp"), IDS_NW4F_NODECOMP_INTERFACE, &nw4fNodeAttribDesc, FP_MIXIN,
    properties,
        nw4f_nodecomp_Imp::nw4f_noComp_get, nw4f_nodecomp_Imp::nw4f_noComp_set, _T("nw4f_no_compress_node"), 0, TYPE_BOOL,
    end
);
*/

// 出力ファイルネームのボディ部分を取得
// オプションで指定がない場合はシーン名から、シーン名もない場合は
// undefinedを返す
static string getOutFileName()
{
    string namebody = exportOptions.m_Opt.m_OutFileName;
    // ファイル名が指定されていない場合はシーン名から
    if (namebody.length() == 0)
    {
        MSTR scenename = GetCOREInterface()->GetCurFileName();
        if(scenename.length() > 0)
        {
            string s = M_2_A(scenename);
            s = s.substr(0, s.length() - 4);
            namebody = s;
        }
        else
        {
            namebody = "undefined";
        }
    }
    return namebody;
}


//プログレスバー用ダミー関数
DWORD WINAPI progDummyfn(LPVOID arg)
{
    return(0);
}

////////////////////////////////////////////////////////////////////////////
//
BOOL NintendoExport_Imp::DoExport(BOOL selected)
{
    // エラーログバッファを初期化する
    NintendoExport::Get()->ClearErrorLog();
    // 出力するフォルダをチェック
    IPathConfigMgr* iPath = IPathConfigMgr::GetPathConfigMgr();
    TSTR fullpath(A_2_M(exportOptions.m_Opt.m_OutFolderPath.c_str()));
    iPath->AppendSlash(fullpath);
    MaxSDK::Util::Path pathUtil(fullpath);
    // フォルダが存在しなければエラー
    if(!iPath->DoesFileExist(pathUtil))
    {
        //mprintf("ERROR: Output folder(%s) is not exist\n", fullpath.data());
        RLogger::LogMessagebyID(RLogger::kLogERR_OutputFolder, std::string(M_2_A(fullpath.data())));
        NintendoExport::Get()->FinishExport(false);
        return FALSE;
    }

    std::string namebody = getOutFileName();

    std::string sfullpath = M_2_A(fullpath);
    sfullpath += namebody;
    sfullpath += ".fmdb";
    pathUtil = A_2_M(sfullpath.data());
    // 作成したファイル名に使用できない文字が使われていたらエラー
    if(!pathUtil.IsLegal())
    {
        //mprintf("ERROR: Output filename(%s) is not legal\n", fullpath.data());
        RLogger::LogMessagebyID(RLogger::kLogERR_OutputFileName, sfullpath);
        NintendoExport::Get()->FinishExport(false);
        return FALSE;
    }
    // エクスポートを実行
    Interface* ip = GetCOREInterface();
    //LPVOID arg;
    //ip->ProgressStart("Export", FALSE, progDummyfn, nullptr);

    DWORD options = selected? SCENE_EXPORT_SELECTED: 0;
    NintendoExport* ninexport = NintendoExport::Get();
    int ret = ninexport->DoExport(A_2_M(sfullpath.c_str()), nullptr, ip, true, options);

    return ret;
}

//MCHAR* NintendoExport_Imp::SearchAnimations( BOOL selected )
TSTR NintendoExport_Imp::SearchAnimations( BOOL selected )
{
    return TSTR(A_2_M(SearchAnim(selected).data()));
}

TCHAR* NintendoExport_Imp::returnString( std::string& str )
{
    // ここのconst_castは気持ち悪いが、max側では文字列を参照するだけで
    // 変更しないはずなので多分問題ないはず。
#ifndef UNICODE
    CStr* buf = new CStr(str.c_str());
    return buf->data();
    /*
    char* buf = new char[str.length() + 2];
    strcpy_s(buf, str.length()+2, str.c_str());
    return buf;
    */
    //return const_cast<char*>(str.c_str());
#else
    /*
    std::wstring wstr = Dcc::RGetUnicodeFromShiftJis(str);
    WStr* buf = new WStr(wstr.c_str());
    return buf->data();
    */
    std::wstring* wstr = new std::wstring;
    *wstr = Dcc::RGetUnicodeFromShiftJis(str);
    return const_cast<wchar_t*>(wstr->c_str());
#endif
}

#include <ILayerProperties.h>

void NintendoExport_Imp::LayerTest(void)
{
    /*
    IFPLayerManager* lman = static_cast<IFPLayerManager*>(GetCOREInterface(LAYERMANAGER_INTERFACE));
    if(lman == nullptr) return;
    int count = lman->getCount();
    ILayerProperties* layer;
    for(int i = 0; i < count; ++i)
    {
        layer =  lman->getLayer(i);
        MSTR str(layer->getName());
        str += _T("AAA");
        layer->setName(const_cast<const MCHAR*>(str.ToMCHAR()));
    }
    */
}

// ロギング用のコールバック
Dcc::RStatus maxLogging( const std::string& message, RLogger::RMessageType type )
{
    string out;

    switch(type)
    {
        case	RLogger::kError:
            out = "ERROR: ";
            break;
        case	RLogger::kWarning:
            out = "WARNING: ";
            break;
        case	RLogger::kInfo:
            out = "INFO: ";
            break;
        case	RLogger::kDump:
        default:
            out = "";
            break;
    }
    out += message;
    out += "\n";

    NintendoExport::Get()->AddErrorLog(out);
    auto mes = std::wstring(A_2_M(out));
    the_listener->edit_stream->wputs(mes.c_str());
    return Dcc::RStatus::SUCCESS;
}


INT_PTR CALLBACK nw4fmaxOptionsDlgProc(HWND hWnd,UINT message,WPARAM wParam,LPARAM lParam) {
    static NintendoExport* imp = nullptr;

    switch(message) {
        case WM_INITDIALOG:
            imp = (NintendoExport*)lParam;
            CenterWindow(hWnd,GetParent(hWnd));
            return TRUE;

        case WM_CLOSE:
            EndDialog(hWnd, 0);
            return 1;
    }
    return 0;
}

static bool DeleteDirectoryDeep(string basepath)
{
    string findpath = basepath + "\\*.*";
    WIN32_FIND_DATAA fd;

    HANDLE hdl = FindFirstFileA(findpath.data(), &fd);
    if(hdl == INVALID_HANDLE_VALUE) return false;

    do
    {
        unsigned char* fn = (unsigned char*)fd.cFileName;
        //std::string fn = fd.cFileName;

        if (strcmp(fd.cFileName, ".") != 0 && strcmp(fd.cFileName, "..") != 0)
        //if (strcmp(fn, ".") != 0 && strcmp(fn, "..") != 0)
        {
            string path;
            path = basepath + "\\";
            path += fd.cFileName;

            if (fd.dwFileAttributes & FILE_ATTRIBUTE_DIRECTORY)
            {
                DeleteDirectoryDeep(path);
            }
            else
            {
                // ファイルの削除
                ::DeleteFileA(path.data());
            }
        }
    }
    while (FindNextFileA(hdl, &fd));

    FindClose(hdl);

    // ディレクトリの削除
    bool ret = (RemoveDirectoryA(basepath.data()) == TRUE);

    return ret;
}

static bool CopyDirectoryDeep(string frompath, string topath)
{
    string findpath = frompath + "\\*.*";
    WIN32_FIND_DATAA fd;

    HANDLE hdl = FindFirstFileA(findpath.data(), &fd);
    if(hdl == INVALID_HANDLE_VALUE) return false;

    do
    {
        if (strcmp(fd.cFileName, ".") != 0 && strcmp(fd.cFileName, "..") != 0)
        {
            string from, to;
            from = frompath + "\\";
            to = topath + "\\";
            from += fd.cFileName;
            to += fd.cFileName;

            if (fd.dwFileAttributes & FILE_ATTRIBUTE_DIRECTORY)
            {
                CopyDirectoryDeep(from, to);
            }
            else
            {
                // ファイルの移動
                CopyFileA(from.c_str(), to.c_str(), FALSE);
            }
        }
    }
    while (FindNextFileA(hdl, &fd));

    FindClose(hdl);

    return true;
}


// \ to /
/*
CStr BSlashToSlash(CStr path)
{
#if (MAX_VERSION_MAJOR >= 15) // max2013
    WStr wpath = WStr::FromCStr(path);
    wpath.Replace(L'\\', L'/');
    CStr mbpath = wpath.ToCStr();
#else
    WStr wpath(path.data());
    for(int i = 0; i < wpath.Length(); i++)
    {
        if(wpath[i] == L'\\') wpath[i] = L'/';
    }
    CStr mbpath(wpath.data());
#endif


    return mbpath;
}
*/
string ReplaceAmpersand(const string str)
{
    wstring src = Dcc::RGetUnicodeFromShiftJis(str);
    wstring ret = L"";
    for(wstring::iterator it = src.begin(); it != src.end(); it++)
    {
        if(*it != L'&')
        {
            ret += *it;
        }
        else
        {
            ret += L"&amp;";
        }
    }
    return Dcc::RGetShiftJisFromUnicode(ret);

}

//--- NintendoExport -------------------------------------------------------
NintendoExport* NintendoExport::m_pInstance_ = nullptr;

NintendoExport* NintendoExport::Get(void)
{
    if( m_pInstance_ == nullptr )
    {
        RLogger::SetLoggingCallback( maxLogging );
        m_pInstance_ = new NintendoExport();
    }

    return m_pInstance_;
}

int NintendoExport::DoPreExportScript()
{
    SetOptionEnvVarForPrePostExportScript(false, false);

    std::string& script = exportOptions.m_Opt.m_PreExpScript;
    if(script.size() > 0)
    {
        char* buf = new char[script.size()+2];
        strcpy_s(buf, script.size()+2, script.c_str());
        if(!ExecuteMAXScriptScript(A_2_M(buf)))
        {
            RLogger::LogMessage("Pre-Export Script failed.", RLogger::kError);
            FinishExport(false);
            return false;
        }
    }
    return true;
}

int	NintendoExport::DoExport(const TCHAR* outFileFullPath, ExpInterface* ei,Interface* ip, BOOL suppressPrompts, DWORD options)
{
    mprintf(A_2_M("\n========== NW4F_Export ==========\n"));
    suppressPrompts = true;
    if(!suppressPrompts)
        DialogBoxParam(hInstance,
                MAKEINTRESOURCE(IDD_PANEL),
                GetActiveWindow(),
                nw4fmaxOptionsDlgProc, (LPARAM)this);

    // max interface
    m_pIp = ip;

    // 共通モジュールのオプション
    Dcc::RExpOpt& rOpt = exportOptions.m_Opt;

    RLogger::ClearWarningCount();

    //-----------------------------------------------------------------------------
    // root path
    const std::string root_path = GetNintendoRootPath();

    // 環境変数NINTENDO_SDK_ROOT  が指定されていない場合
    if (root_path.size() == 0)
    {
        RLogger::LogMessage("System Environment(NINTENDO_SDK_ROOT) isn't valid.", RLogger::kError);
        FinishExport(false);
        return false;
    }
    rOpt.m_G3dToolPath      = Get3dCommandLineToolsPath();

    // コンフィグファイルを読み込む
    exportOptions.InitConfigOptions();
    loadConfigFile();

    // Pre-Exportスクリプトの実行
    if(!DoPreExportScript())
    {
        FinishExport(false);
        return false;
    }

    // アプリケーションのプロジェクトのルートフォルダのパスをフルパスにします。
    if (!rOpt.m_ProjectRootPath.empty())
    {
        rOpt.m_ProjectRootPath = Dcc::RGetFilePathWithEndingSlash(Dcc::RGetFullFilePath(rOpt.m_ProjectRootPath, true));
    }


    TCHAR path[2048], nameBody[2048], ext[2048];
    BMMSplitFilename(outFileFullPath, path, nameBody, ext);
    m_OutFolder = M_2_A(path);

    {
        std::string outExt;
        if(rOpt.m_OutFileFlag[Dcc::RExpOpt::FMD]) outExt += "fmdb ";
        if(rOpt.m_OutFtxFlag) outExt += "ftxb ";
        if(rOpt.m_OutFileFlag[Dcc::RExpOpt::FSK]) outExt += "fskb ";
        if (rOpt.UsesSingleFma())
        {
            if(	rOpt.m_OutFileFlag[Dcc::RExpOpt::FCL] ||
                rOpt.m_OutFileFlag[Dcc::RExpOpt::FTS] ||
                rOpt.m_OutFileFlag[Dcc::RExpOpt::FTP]) outExt += "fmab ";
        }
        else if (rOpt.UsesSeparatedFma())
        {
            if(rOpt.m_OutFileFlag[Dcc::RExpOpt::FCL]) outExt += "_fcl.fmab ";
            if(rOpt.m_OutFileFlag[Dcc::RExpOpt::FTS]) outExt += "_fts.fmab ";
            if(rOpt.m_OutFileFlag[Dcc::RExpOpt::FTP]) outExt += "_ftp.fmab ";
        }
        else
        {
            if(rOpt.m_OutFileFlag[Dcc::RExpOpt::FCL]) outExt += "fclb ";
            if(rOpt.m_OutFileFlag[Dcc::RExpOpt::FTS]) outExt += "ftsb ";
            if(rOpt.m_OutFileFlag[Dcc::RExpOpt::FTP]) outExt += "ftpb ";
        }
        if(rOpt.m_OutFileFlag[Dcc::RExpOpt::FVB]) outExt += "fvbb ";
        if(rOpt.m_OutFileFlag[Dcc::RExpOpt::FSH]) outExt += "fshb ";
        if(rOpt.m_OutFileFlag[Dcc::RExpOpt::FSN]) outExt += "fsnb ";

        CStr mes;

        mes.printf("File Name       : %s ( %s)", M_2_A(nameBody), outExt.c_str());
        RLogger::LogMessage(mes.data(), RLogger::kDump);

        mes.printf("Output Folder   : %s", Dcc::RGetUnixFilePath(m_OutFolder.data()).c_str());
        RLogger::LogMessage(mes.data(), RLogger::kDump);

        //
    }

    // テンポラリから移動するファイルのリストをクリアする
    m_TmpFileMoves.clear();

    // テンポラリフォルダ
    {
        // テンポラリフォルダの作成
        auto tempFolderPath = Dcc::RGetTempFolderPath();
        if (tempFolderPath.empty())
        {
            RLogger::LogMessage("Temporary Path not found.");
            FinishExport(false);
            return false;
        }
        tempFolderPath += "Nintendo_3dsMaxPlugins_" +
                Dcc::RGetNumberString(static_cast<int>(GetCurrentProcessId())) + "/";
        m_TmpOutFolderPath = tempFolderPath;
        CreateDirectoryA(m_TmpOutFolderPath.c_str(), nullptr);
    }

    // 拡張子が大文字になっているのを小文字にする
    std::string outname, outModelName, outAnimName;
    std::string outnameTmp, outModelNameTemp;

    // テンポラリファイル名、出力ファイル名
    outname = m_OutFolder;
    outname += M_2_A(nameBody);
    outnameTmp  = m_TmpOutFolderPath;
    outnameTmp += M_2_A(nameBody);

    // テクスチャのマージ元フォルダを指定(固定)
    rOpt.m_MergeFtxFolderPath = m_OutFolder + Dcc::RExpOpt::TexFolderName + "/";

    // 出力する形式が一つも選択されていない場合はエラー
    if( !rOpt.ExportsModelFile() && !rOpt.ExportsTexture() &&
        !rOpt.ExportsAnim())
    {
        RLogger::LogMessagebyID(RLogger::kLogERR_NoOutputFile, outModelName.data());
        FinishExport(false);
        return false;
    }

    //-----------------------------------------------------------------------------
    // check optimizer
    // 最適化コンバーターが存在するかチェックします。
    // 見つからない場合はエラー終了させます。
    if (!Dcc::RFolderExists(rOpt.m_G3dToolPath))
    {
        RLogger::LogMessagebyID(RLogger::kLogERR_FindOptimizer, rOpt.m_G3dToolPath.c_str());
        FinishExport(false);
        return false;
    }

    //-----------------------------------------------------------------------------
    // check comment
    // コメントに使用できない文字が使われていないかどうかチェック
    if(!rOpt.m_CommentText.empty())
    {
        const string& com = rOpt.m_CommentText;

        if(com.find("\"") != string::npos || com.find(";") != string::npos)
        {
            RLogger::LogMessagebyID(RLogger::kLogERR_CommentIsWrong, com.c_str());
            FinishExport(false);
            return false;
        }
    }

    // マージ使用時にファイル名が正しいかどうか判定する
    if(rOpt.MergesFmd())
    {
        // マージ先が指定されていないときは上書きする出力ファイルを指定。
        if(rOpt.m_MergeFmdPath.length() == 0)
        {
            rOpt.m_MergeFmdPath = outname + "." + rOpt.GetExtension(Dcc::RExpOpt::FMD).data();
        }

        if(DoesFileExist(A_2_M(rOpt.m_MergeFmdPath.c_str())) == FALSE)
        {
            RLogger::LogMessagebyID(RLogger::kLogERR_PathOfMergeIsWrong, rOpt.m_MergeFmdPath.c_str());
            FinishExport(false);
            return false;
        }
    }

    // アニメーションのマージを行うとき、フォルダなどを設定
    if(rOpt.MergesAnim())
    {
        bool isMergeAnimFolderEmpty = false;
        if (rOpt.m_MergeAnimFolder.empty())
        {
            rOpt.m_MergeAnimFolder = rOpt.m_OutFolderPath;
            isMergeAnimFolderEmpty = true;
        }
        if ((rOpt.m_MergeAnimFolder.find_last_of('/') != rOpt.m_MergeAnimFolder.length() - 1) &&
            (rOpt.m_MergeAnimFolder.find_last_of('\\') != rOpt.m_MergeAnimFolder.length() - 1))
        {
            rOpt.m_MergeAnimFolder += "/"; // 最後にスラッシュがなければ追加
        }

        if (rOpt.m_MergeAnimName.empty())
        {
            rOpt.m_MergeAnimName = rOpt.m_OutFileName;
        }
        if (!Dcc::RIsValidFilePathString(rOpt.m_MergeAnimFolder))
        {
            RLogger::LogMessagebyID(RLogger::kLogERR_MergeAnimationFolderIsWrong, rOpt.m_MergeAnimFolder.c_str());
            FinishExport(false);
            return false;
        }
        if (!Dcc::RIsValidFileNameString(rOpt.m_MergeAnimName))
        {
            RLogger::LogMessagebyID(RLogger::kLogERR_MergeAnimationNameIsWrong, rOpt.m_MergeAnimName.c_str());
            FinishExport(false);
            return false;
        }

        // マージ対象のアニメーション中間ファイルが 1 つも存在しない場合はエラーにします。
        bool mergeAnimFileExists = false;
        const std::string mergeAnimPathTop = rOpt.m_MergeAnimFolder + rOpt.m_MergeAnimName + ".";
        std::string mergeAnimPath = "";
        for (int fileType = 0; fileType < Dcc::RExpOpt::FILE_TYPE_COUNT; ++fileType)
        {
            if (fileType != Dcc::RExpOpt::FMD && rOpt.m_OutFileFlag[fileType])
            {
                mergeAnimPath = mergeAnimPathTop +
                    rOpt.GetExtension(static_cast<Dcc::RExpOpt::FileType>(fileType));
                if (Dcc::RFileExists(mergeAnimPath))
                {
                    mergeAnimFileExists = true;
                    break;
                }
            }
        }
        if (!mergeAnimFileExists)
        {
            // マージアニメーションのフォルダが空欄の場合は、マージ対象のファイルが無くてもワーニングのみでエラーにしない。
            if(isMergeAnimFolderEmpty)
            {
                RLogger::LogMessagebyID(RLogger::kLogWRN_AnimationFileToMergeIsNotFound, mergeAnimPath.c_str());
            }
            else
            {
                RLogger::LogMessagebyID(RLogger::kLogERR_AnimationFileToMergeIsNotFound, mergeAnimPath.c_str());
                FinishExport(false);
                return false;
            }
        }
    }

#if 0
    // モデル用ファイルの準備
    ofstream ofs;
    if(rOpt.ExportsModelFile())
    {
        outModelName = outnameTmp + "." +rOpt.GetExtension(Dcc::RExpOpt::FMD).data();
        m_TmpFileMoves.push_back(strStrPair(outModelName.data(), (outname + "." + rOpt.GetExtension(Dcc::RExpOpt::FMD).data()).data() ));

        ofs.open(outModelName.c_str(), ios_base::out | ios_base::binary);

        // ファイルのオープンに成功したかどうかチェック。
        if(!ofs.is_open())
        {
            RLogger::LogMessagebyID(RLogger::kLogERR_OpenFile, outModelName.data());
            FinishExport(false);
            return false;
        }
    }

    // アニメーション用ファイルの準備
    ofstream oafs;
    if(rOpt.m_OutFileFlag[Dcc::RExpOpt::FSK])
    {
        outAnimName = outnameTmp + "." + rOpt.GetExtension(Dcc::RExpOpt::FSK).data();
        m_TmpFileMoves.push_back(strStrPair(outAnimName.data(), (outname + "." + rOpt.GetExtension(Dcc::RExpOpt::FSK).data()).data() ));

        oafs.open(outAnimName.c_str(), ios_base::out | ios_base::binary);
        if(!oafs.is_open())
        {
            if(ofs.is_open()) { ofs.close();	}
            //mprintf("ERROR: Can't open animation file(%s)\n", outAnimName.data());
            RLogger::LogMessagebyID(RLogger::kLogERR_OpenFile, outAnimName.data());
            FinishExport(false);
            return false;
        }
    }

    // ボーンビジビリティアニメーション用ファイルの準備
    ofstream ovafs;
    if(rOpt.m_OutFileFlag[Dcc::RExpOpt::FVB])
    {
        outAnimName = outnameTmp + "." + rOpt.GetExtension(Dcc::RExpOpt::FVB).data();
        m_TmpFileMoves.push_back(strStrPair(outAnimName.data(), (outname + "." + rOpt.GetExtension(Dcc::RExpOpt::FVB).data()).data() ));

        ovafs.open(outAnimName.c_str(), ios_base::out | ios_base::binary);
        if(!ovafs.is_open())
        {
            if(oafs.is_open()) { oafs.close();	}
            if(ofs.is_open()) { ofs.close();	}
            //mprintf("ERROR: Can't open animation file(%s)\n", outAnimName.data());
            RLogger::LogMessagebyID(RLogger::kLogERR_OpenFile, outAnimName.data());
            FinishExport(false);
            return false;
        }
    }

    // ライト、カメラアニメーション用ファイルの準備
    ofstream osafs;
    if(rOpt.ExportsSceneAnim())
    {
        outAnimName = outnameTmp + "." + rOpt.GetExtension(Dcc::RExpOpt::FSN).data();
        m_TmpFileMoves.push_back(strStrPair(outAnimName.data(), (outname + "." + rOpt.GetExtension(Dcc::RExpOpt::FSN).data()).data() ));

        osafs.open(outAnimName.c_str(), ios_base::out | ios_base::binary);
        if(!osafs.is_open())
        {
            if(ovafs.is_open()) { ovafs.close();}
            if(oafs.is_open()) { oafs.close();	}
            if(ofs.is_open()) { ofs.close();	}
            //mprintf("ERROR: Can't open animation file(%s)\n", outAnimName.data());
            RLogger::LogMessagebyID(RLogger::kLogERR_OpenFile, outAnimName.data());
            FinishExport(false);
            return false;
        }
    }

    // シェイプアニメーション用ファイルの準備
    ofstream oshafs;
    if(rOpt.m_OutFileFlag[Dcc::RExpOpt::FSH])
    {
        outAnimName = outnameTmp + "." + rOpt.GetExtension(Dcc::RExpOpt::FSH).data();
        m_TmpFileMoves.push_back(strStrPair(outAnimName.data(), (outname + "." + rOpt.GetExtension(Dcc::RExpOpt::FSH).data()).data() ));

        oshafs.open(outAnimName.c_str(), ios_base::out | ios_base::binary);
        if(!oshafs.is_open())
        {
            if(osafs.is_open()) { ovafs.close();}
            if(ovafs.is_open()) { ovafs.close();}
            if(oafs.is_open()) { oafs.close();	}
            if(ofs.is_open()) { ofs.close();	}
            RLogger::LogMessagebyID(RLogger::kLogERR_OpenFile, outAnimName.data());
            FinishExport(false);
            return false;
        }
    }
#endif

    // シーン全体のマテリアルを初期化
    m_SceneMaterials.init();
    m_SceneMaterials.SetDoesExportTexture(rOpt.ExportsTexture());
    MtlInfo::exportTexture = (rOpt.ExportsTexture());

    // プロファイル
    m_ModelClocks = 0;
    m_TexClocks = 0;
    m_AnimClocks = 0;
    clock_t startClock = clock();
    clock_t lapClock = startClock;

    // FigureModeにする
    CNodeTree::SetUseFigureMode(exportOptions.m_UseFigureMode == TRUE);
    if(exportOptions.m_UseFigureMode)
    {
        BeginFigureMode();
    }

    // まずノードをトラバースして、出力するノードの情報を集める
    m_SceneNode.Init();
    if(options & SCENE_EXPORT_SELECTED) // 選択をエクスポート
    {
        Interface11* ip11 = GetCOREInterface11();
        INodeTab selNodes;
        ip11->GetSelNodeTab(selNodes);
        if(selNodes.Count() == 0)
        {
            // 有効なノードが一つも選択されない場合はエラー
            RLogger::LogMessagebyID(RLogger::kLogERR_NoEffectiveNode, "");
            FinishExport(false);
            return false;
        }
        else if(selNodes.Count() == 1)
        {
            m_SceneNode.SetRootNode(selNodes[0]);
            CNodeTree* cNode = m_SceneNode.AddNodeInTree(selNodes[0], true);
            if(cNode)
            {
                cNode->SetNode(selNodes[0]);
                cNode->InquiryAsModel();
                cNode->m_DoesExportModel = true;
                cNode->AddChildrenRecursive();
            }
        }
        else
        {
            m_SceneNode.SetRootNode(m_pIp->GetRootNode());
            // 選択されたノードを順番に追加
            for (int idx=0; idx < selNodes.Count(); idx++)
            {
                INode* selnode = selNodes[idx];
                if(!selnode)
                    {	continue;	}

                // 親ノードが選択されていれば出力しない。
                bool isParentSelected = false;
                INode* parent = selnode->GetParentNode();
                while(parent != nullptr && !isParentSelected)
                {
                    isParentSelected = selNodes.Contains(parent);
                    parent = parent->GetParentNode();
                }
                if(isParentSelected)
                    {	continue;	}
                CNodeTree* cNode = m_SceneNode.AddNodeInTree(selnode, true);
                if(cNode)
                {
                    cNode->InquiryAsModel();
                    cNode->m_DoesExportModel = true;
                    cNode->AddChildrenRecursive();
                }
            }
        }
    }
    else // 全てエクスポート対象にする
    {
        int numRootChildren = 0;
        INode* sceneRoot = m_pIp->GetRootNode();

        numRootChildren = sceneRoot->NumberOfChildren();

        if(numRootChildren == 0)
        {
            // 有効なノードが一つも選択されない場合はエラー
            RLogger::LogMessagebyID(RLogger::kLogERR_NoEffectiveNode, "");
            FinishExport(false);
            return false;
        }
        else
        if(numRootChildren == 1)
        {
            INode* node = sceneRoot->GetChildNode(0);
            m_SceneNode.SetRootNode(node);
            CNodeTree* cNode = m_SceneNode.AddNodeInTree(node, true);
            if(cNode)
            {
                cNode->InquiryAsModel();
                cNode->m_DoesExportModel = true;
                cNode->SetNode(node);
                cNode->AddChildrenRecursive();
            }
        }
        else
        {
            m_SceneNode.RecursiveAddNodes(m_pIp->GetRootNode());
        }
    }

    // アニメーション範囲の設定
    int start = 0, end = 0, ticks, length;
    //if(exportOptions.exportAnim)
    {
        Interval range = m_pIp->GetAnimRange();
        ticks = GetTicksPerFrame() / rOpt.m_FramePrecision;
        int maxSt, maxEd;
        maxSt = range.Start() / GetTicksPerFrame();
        maxEd = range.End() / GetTicksPerFrame();
        if(rOpt.m_FrameRange != Dcc::RExpOpt::FRAME_RANGE_SPECIFY)
        {
            start = maxSt;
            end = maxEd;
        }
        else
        {
            int usrSt, usrEd;
            bool isResized = false;
            usrSt = rOpt.m_StartFrame;
            usrEd = rOpt.m_EndFrame;
            // StartよりEndが小さい場合はスワップ
            if(usrEd < usrSt)
            {
                int tmpEnd = usrEd;
                usrEd = usrSt;
                usrSt = tmpEnd;
            }

            start = usrSt;
            end = usrEd;
        }

        length = end - start;
        rOpt.m_StartFrame = start;
        rOpt.m_EndFrame = end;
        rOpt.UpdateFrameCount();
    }

    // サブサンプリングのフレーム
    int subStart = start * rOpt.m_FramePrecision;
    int subEnd = end * rOpt.m_FramePrecision;

    MtlInfo::setAnimRange(start, end, subStart, subEnd, ticks);
    m_SceneMaterials.SetDoesExportAnimation(rOpt.ExportsMaterialAnim());

    bool ret = true;
    bool isBinary = rOpt.m_IsBinaryFormat; //asciiモードは廃止


    // RNodeProcessorに渡せるように、RNode, RModelを作成する。
    ret = m_SceneNode.PrepareExport();
    // スキニングに必要なボーンが選択されていない場合
    if(!ret)
    {
//        if(ofs.is_open()) { ofs.close();}
//        if(oafs.is_open()) { oafs.close();}
        FinishExport(false);
        return false;
    }

    //	シーン中のノードを共通モジュールのデータ構造へ変換する
    ret = m_SceneNode.ProcessNodes( start );
    if(!ret)
    {
//        if(ofs.is_open()) { ofs.close();}
//        if(oafs.is_open()) { oafs.close();}
        FinishExport(false);
        return false;
    }

    std::vector<RNode*>	NodeList;
    //	シーンを構成するノードを配列に並べる
    //
    if(m_SceneNode.GetRootNode()->m_pRNode)
    {
        m_SceneNode.GetRootNode()->m_pRNode->enumNodeToArray(NodeList);
    }

    // シェイプアニメーションのターゲットのメッシュは出力しない
    m_SceneNode.HideMorphTargets(NodeList);

    //	シーン中のモデル（メッシュ）を処理する
    ret = m_SceneNode.ProcessModels( m_SceneMaterials, NodeList, rOpt);
    if(!ret)
    {
//        if(ofs.is_open()) { ofs.close();}
//        if(oafs.is_open()) { oafs.close();}
        FinishExport(false);
        return false;
    }

    // ボーンに階層の上下を挟まれた非ボーンを探す
    if(rOpt.m_OutFileFlag[Dcc::RExpOpt::FSK] || rOpt.ExportsModelFile())
    {
        nodeTreeList nonBoneList;
        m_SceneNode.CheckNonBoneIsExistBetweenBones(nonBoneList);
        nodeTreeList::iterator nit;
        for(nit = nonBoneList.begin(); nit != nonBoneList.end(); nit++)
        {
            RLogger::LogMessagebyID(
                RLogger::kLogWRN_NonBoneBetweenBones, (*nit)->GetName());
        }
    }


    // FigureModeから戻す
    if(exportOptions.m_UseFigureMode)
    {
        m_SceneNode.GetRootNode()->SetInitMtxFromCurrentTM();
        EndFigureMode();
    }

    // プロファイル
    const clock_t getModelClocks = clock() - lapClock;
    lapClock = clock();

    // サブサンプリングの設定
    RNode::setAnimSubSample(rOpt.m_FramePrecision);
    // アニメーション圧縮の許容値を設定
    RNode::setTolerances(rOpt.m_TolT, rOpt.m_TolR, rOpt.m_TolS);

    // 出力する形式がTextureだけ選択されていて、テクスチャが使われていない場合はエラー
    if( rOpt.ExportsTexture() && !rOpt.ExportsModelFile() &&
        !rOpt.ExportsAnim())
    {
        if(m_SceneMaterials.getNumTexture() == 0)
        {
            RLogger::LogMessagebyID(
                RLogger::kLogERR_NoOutputFile, outModelName.data());
            FinishExport(false);
            return false;
        }
    }

    // SRTアニメーションを取得
    if(rOpt.m_OutFileFlag[Dcc::RExpOpt::FSK])
    {
        m_SceneNode.SetAnimations(subStart, subEnd, ticks);
    }

    // モデルアニメーション(ビジビリティ)
    if(rOpt.m_OutFileFlag[Dcc::RExpOpt::FVB])
    {
        m_SceneNode.SetVisibilityAnimations(subStart, subEnd, ticks);
    }

    // シェイプアニメーション(モーフ)
    if(rOpt.m_OutFileFlag[Dcc::RExpOpt::FSH])
    {
        m_SceneNode.SetShapeAnimations(subStart, subEnd, ticks);
    }

    // マテリアルアニメーション
    if(rOpt.ExportsMaterialAnim())
    {
        m_SceneMaterials.PrepareAnimations(rOpt);
    }

    m_SceneCameras.init();
    m_SceneLights.init();
    // ライト、カメラ
    if(rOpt.ExportsSceneAnim())
    {
        // カメラ
        m_SceneCameras.SetMagnify(static_cast<float>(rOpt.m_Magnify));
        m_SceneNode.SetCameras(m_SceneCameras, subStart, subEnd, ticks);
        m_SceneCameras.PrepareAnimations(rOpt);

        // ライト
        m_SceneLights.SetMagnify(static_cast<float>(rOpt.m_Magnify));
        m_SceneNode.setLights(m_SceneLights, subStart, subEnd, ticks);
        m_SceneLights.PrepareAnimations(rOpt);
    }


    // プロファイル
    const clock_t getAnimationClocks = clock() - lapClock;
    lapClock = clock();

    m_NodeProcessor.Init();
    // シーン全体のスケール値を設定。
    m_NodeProcessor.setMaginify((float)rOpt.m_Magnify);

    // RNodeProcessorでシーンを処理する。
    if(rOpt.ExportsModelFile() || rOpt.ExportsModelAnim()
        || rOpt.ExportsMaterialAnim() || rOpt.ExportsShapeAnim())
    {
        if(!m_NodeProcessor.processNodeList(NodeList, g_VtxMtxs ))
        {
//            if(ofs.is_open()) { ofs.close();}
//            if(oafs.is_open()) { oafs.close();}
            FinishExport(false);
            return false;
        }
    }

    // マテリアルのサンプラの名前が重複していないかチェックする。
    if(!m_SceneMaterials.isSamplerNameHintValid())
    {
//        if(ofs.is_open()) { ofs.close();}
//        if(oafs.is_open()) { oafs.close();}
        FinishExport(false);
        return false;
    }

    //-----------------------------------------------------------------------------
    // 中間ファイルに出力するデータを出力順にソートします。
    SetOutTexImgPtrs();
    SortMaterialByName();

    // プロファイル
    const clock_t processNodesClocks = clock() - lapClock;
    lapClock = clock();

    int fileCount;
    OutputFiles(fileCount, rOpt);

    lapClock = clock();

    // 最適化コンバーターを呼び出し、オプティマイズを行う
    {
        ROptimizer optimizer;

        std::string sPath = Dcc::RGetUnixFilePath(m_TmpOutFolderPath);
        bool ret = optimizer.Optimize(sPath, rOpt);
        // オプティマイザがエラーを返した場合は起動に失敗した場合のはず（エラーメッセージは出力済み）
        if(!ret)
        {
            FinishExport(false);
            return false;
        }
    }

    // プロファイル
    const clock_t optimizeClocks = clock() - lapClock;
    //startClock = clock();


    // プロファイル出力
    if(rOpt.m_DisplaysProfile)
    {
        RLogger::LogMessage("\n### Display Profile ###", RLogger::kDump);
                                        DisplayProfileInfo("Get Model        ", getModelClocks      );
        if (rOpt.ExportsAnim()		  ) DisplayProfileInfo("Get Animations   ", getAnimationClocks	);
                                        DisplayProfileInfo("Process Nodes    ", processNodesClocks	);
        if (rOpt.ExportsModelFile()   ) DisplayProfileInfo("Export Models    ", m_ModelClocks		);
        if (rOpt.ExportsTexture()     ) DisplayProfileInfo("Export Textures  ", m_TexClocks			);
        if (rOpt.ExportsAnim()		  ) DisplayProfileInfo("Export Animations", m_AnimClocks		);
        if (rOpt.Uses3dOptimizer()    ) DisplayProfileInfo("Optimize         ", optimizeClocks	    );
        RLogger::LogMessage("", RLogger::kDump);
    }


    {
        CStr mes;
        mes.printf("Texture Count   : %d", m_pOutTexImgs.size());
        RLogger::LogMessage(mes.data(), RLogger::kDump);

        clock_t totalClocks = clock() - startClock;
        const float sec = static_cast<float>(totalClocks) / CLOCKS_PER_SEC;
        const std::string secStr = Dcc::RGetNumberString(Dcc::RRound(sec), "%4d") + " sec.";
        mes.printf("Finished ( Time = %s , Warning = %d )", secStr.c_str(), RLogger::GetWarningCount());
        RLogger::LogMessage(mes.data(), RLogger::kDump);
    }

    // テンポラリフォルダから移動する
    CopyDirectoryDeep(m_TmpOutFolderPath.c_str(), m_OutFolder.data());

    FinishExport(true);

    // シーンマテリアルを初期化してメモリを解放
    m_SceneMaterials.init();
    m_pOutTexImgs.clear();

    return true;
}

std::string NintendoExport::GetTmpFilePath( const Dcc::RExpOpt::FileType fileType ) const
{
    return m_TmpOutFolderPath + GetNintendoOptions()->m_Opt.GetFileName(fileType);
}

////////////////////////////////////////////////////////////////////////////
//
void NintendoExport::BeginFigureMode(INode* node)
{
    // 初期化
    if(!node)
    {
        node = m_pIp->GetRootNode();
        m_FigureModeNodes.Init();
        m_FigureModeNoRestoreNodes.Init();
        m_SkinPoseNodes.Init();
        m_SkinPoseNoRestoreNodes.Init();
    }

    // biped?
    Control* c = node->GetTMController();
    if (c && (c->ClassID() == BIPBODY_CONTROL_CLASS_ID))
    {
        IBipMaster* bipMaster = (IBipMaster*)c->GetInterface(I_BIPMASTER);
        IBipedExport* bipExport= (IBipedExport*) c->GetInterface(I_BIPINTERFACE);
        Control* iMaster = (Control*) c->GetInterface(I_MASTER);
        assert(bipMaster);
        assert(bipExport);
        assert(iMaster);

        // すでにFigureModeか？
        if(bipMaster->GetActiveModes() & BMODE_FIGURE)
        {
            m_FigureModeNodes.AppendNode(node);
        }
        else
        {
            bipExport->BeginFigureMode(0);
        }
        iMaster->NotifyDependents(FOREVER, PART_TM, REFMSG_CHANGE);

        c->ReleaseInterface(I_MASTER,iMaster);
        c->ReleaseInterface( I_BIPINTERFACE, bipExport);
        c->ReleaseInterface( I_BIPMASTER, bipMaster);
    }

    // bone?
    //Object* obj = node->GetObjectRef();
    //if(obj && (obj->ClassID() == BONE_OBJ_CLASSID))
    {
        ISkinPose* skpose = ISkinPose::GetISkinPose(*node);
        if(skpose)
        {
            // すでにSkinPoseか？
            if(skpose->SkinPoseMode())
            {
                m_SkinPoseNodes.AppendNode(node);
            }
            else
            {
                skpose->SetSkinPoseMode(true);
            }
        }
    }

    // recursive child
    for (int i = 0; i < node->NumberOfChildren(); i++)
    {
        INode* child = node->GetChildNode(i);
        BeginFigureMode(child);
    }

}

void NintendoExport::EndFigureMode(INode* node)
{
    // 初期化
    if(!node)
    {
        node = m_pIp->GetRootNode();
    }

    if(	(m_FigureModeNodes.IndexOf(node) == -1) &&
        (m_SkinPoseNodes.IndexOf(node) == -1))
    {

    }

    // biped
    Control* c = node->GetTMController();
    if (c && (c->ClassID() == BIPBODY_CONTROL_CLASS_ID)
        && (m_FigureModeNodes.IndexOf(node) == -1) )
    {
        IBipedExport* bipExport= (IBipedExport*) c->GetInterface(I_BIPINTERFACE);
        Control* iMaster = (Control*) c->GetInterface(I_MASTER);
        assert(bipExport);
        assert(iMaster);
        bipExport->EndFigureMode(0);

        iMaster->NotifyDependents(FOREVER, PART_TM, REFMSG_CHANGE);

        c->ReleaseInterface( I_BIPINTERFACE, bipExport);
        c->ReleaseInterface(I_MASTER,iMaster);
    }

    // node skinpose
    ISkinPose* skinpose = ISkinPose::GetISkinPose(*node);
    if(skinpose && (m_SkinPoseNodes.IndexOf(node) == -1))
    {
        skinpose->SetSkinPoseMode(false);
    }

    // recursive child
    for (int i = 0; i < node->NumberOfChildren(); i++)
    {
        INode* child = node->GetChildNode(i);
        EndFigureMode(child);
    }
}

//-----------------------------------------------------------------------------
//! @brief マテリアルのポインタを名前でソートするための比較関数です。
//-----------------------------------------------------------------------------
static bool RMaterialPtrNameLess(FMaterial*& r1, FMaterial*& r2)
{
    return (strcmp(r1->m_Name.c_str(), r2->m_Name.c_str()) < 0);
}

//-----------------------------------------------------------------------------
//! @brief マテリアルを名前でソートします。
//!        m_SceneMaterials に登録されているマテリアルを名前順でソートした結果を m_pOutMaterials 配列に格納します。
//!        そして YMaterial::m_Index に ymodel.m_pOutYMaterials へのインデックスを設定します。
//-----------------------------------------------------------------------------
void NintendoExport::SortMaterialByName( void )
{
    const int matCount = m_SceneMaterials.getNumFMaterials();
    if (matCount > 0)
    {
        // 出力用マテリアルのポインタ配列を初期化します。
        m_pOutMaterials.resize(matCount);
        for (int iMat = 0; iMat < matCount; ++iMat)
        {
            m_pOutMaterials[iMat] = m_SceneMaterials.getFMaterialByIndex( iMat );
        }

        // 名前でソートします。
        std::sort(m_pOutMaterials.begin(), m_pOutMaterials.end(),
            RMaterialPtrNameLess);

        // ソート後に ymodel.m_pOutYMaterials へのインデックスを設定
        for (int iMat = 0; iMat < matCount; ++iMat)
        {
            m_pOutMaterials[iMat]->m_Index = iMat;
        }
    }
}

//-----------------------------------------------------------------------------
//! @brief テクスチャイメージをテクスチャ名のアルファベット順にソートするための比較関数です。
//-----------------------------------------------------------------------------
static bool TexImgPtrLess(Dcc::RImage*& r1, Dcc::RImage*& r2)
{
    return (strcmp(r1->GetName().c_str(), r2->GetName().c_str()) < 0);
}

//-----------------------------------------------------------------------------
//! @brief テクスチャイメージの出力用ポインタ配列を設定し、
//!        テクスチャ名のアルファベット順にソートします。
//!
//! @param[out] pDstTexImgs テクスチャイメージの出力用ポインタ配列です。
//! @param[in] srcTexImgs テクスチャイメージ配列です。
//-----------------------------------------------------------------------------
void NintendoExport::SetOutTexImgPtrs( void )
{
    m_pOutTexImgs.clear();

    const int imgCount = m_SceneMaterials.getNumTexture();
    if (imgCount > 0)
    {
        // 出力用ポインタ配列を設定します。
        for (int iimg = 0; iimg < imgCount; ++iimg)
        {
            m_pOutTexImgs.push_back(&m_SceneMaterials.getTexture(iimg));
        }

        // 出力用ポインタ配列をテクスチャ名のアルファベット順にソートします。
        std::sort(m_pOutTexImgs.begin(), m_pOutTexImgs.end(), TexImgPtrLess);
    }
}



//-----------------------------------------------------------------------------
//! @brief fmd ファイル（モデルデータ）を出力します。
//!
//! @param[in,out] os 出力ストリームです。
//!
//! @return 処理結果を返します。
//-----------------------------------------------------------------------------
bool NintendoExport::OutputFmdFile( std::ostream& os, Dcc::RDataStreamArray& dataStreams)
{
    const int tc = 0;

    //-----------------------------------------------------------------------------
    // materials
    OutputMaterials( os, 0 );

    //-----------------------------------------------------------------------------
    // skeleton
    m_NodeProcessor.outSkeletonData(os, 0);

    //-----------------------------------------------------------------------------
    // shapes
    if(!m_NodeProcessor.outShapesData(os, dataStreams, 0, g_VtxMtxs, m_SceneMaterials))
    {
        return false;
    }

    //-----------------------------------------------------------------------------
    // original materials
    OutputOriginalMaterials( os, 0 );

    //-----------------------------------------------------------------------------
    // user datas

    return true;
}

//-----------------------------------------------------------------------------
//! @brief fsk ファイル（アニメーションデータ）を出力します。
//!
//! @param[in,out] os 出力ストリームです。
//!
//! @return 処理結果を返します。
//-----------------------------------------------------------------------------
bool NintendoExport::OutputFskFile( ostream & os, Dcc::RDataStreamArray& dataStreams)
{
    m_NodeProcessor.outBoneAnim(os, dataStreams);
    return true;
}

//-----------------------------------------------------------------------------
//! @brief fcl ファイル（マテリアルカラーアニメーション）を出力します。
//!
//! @param[in,out] os 出力ストリームです。
//! @param[in,out] dataStreams データ列配列です。
//!
//! @return 処理結果を返します。
//-----------------------------------------------------------------------------
bool NintendoExport::OutputFclFile( ostream & os, Dcc::RDataStreamArray& dataStreams, const Dcc::RExpOpt& rOpt )
{
    const int tc = 0;

    const int animatedCount = GetColorAnimatedMaterialCount();
    if (animatedCount != 0)
    {
        os << Dcc::RTab(tc) << "<original_material_anim_array length=\"" << animatedCount
            << "\">" << R_ENDL;
        int matAnimIdx = 0;
        const int matCount = static_cast<int>(m_pOutMaterials.size());
        for (int iMat = 0; iMat < matCount; ++iMat)
        {
            const FMaterial& mat = *m_pOutMaterials[iMat];
            if (mat.IsContainingColorAnimData())
            {
                os << Dcc::RTab(tc + 1) << "<original_material_anim index=\"" << matAnimIdx
                    << "\" mat_name=\"" << Dcc::RGetUtf8FromShiftJis(mat.m_Name) << "\">" << R_ENDL;
                mat.OutputColorAnims(os, dataStreams, tc + 2);
                os << Dcc::RTab(tc + 1) << "</original_material_anim>" << R_ENDL;
                ++matAnimIdx;
            }
        }
        os << Dcc::RTab(tc) << "</original_material_anim_array>" << R_ENDL;
    }

    return true;
}
//-----------------------------------------------------------------------------
//! @brief テクスチャ SRT アニメーションを 1 つ出力します。
//!
//! @param[in,out] os 出力ストリームです。
//! @param[in,out] dataStreams データ列配列です。
//! @param[in] tc テクスチャ SRT アニメーション配列要素のインデントに必要なタブの数です。
//! @param[in] mat マテリアルです。
//-----------------------------------------------------------------------------
void NintendoExport::OutputTexSrtAnims(std::ostream& os, Dcc::RDataStreamArray& dataStreams, const int tc, const FMaterial& mat)
{
    //-----------------------------------------------------------------------------
    // begin original texsrt anim array
    int animatedTexSrtCount = 0;
    for (int iSampler = 0; iSampler < static_cast<int>(mat.m_Samplers.size()); ++iSampler)
    {
        if (mat.m_Samplers[iSampler].m_TexSrtAnimIndex != -1)
        {
            ++animatedTexSrtCount;
        }
    }
    os << Dcc::RTab(tc) << "<original_texsrt_anim_array length=\"" << animatedTexSrtCount << "\">" << R_ENDL;

    //-----------------------------------------------------------------------------
    // loop for tex sampler
    int outAnimIdx = 0;
    for (int iSampler = 0; iSampler < static_cast<int>(mat.m_Samplers.size()); ++iSampler)
    {
        const int texSrtAnimIdx = mat.m_Samplers[iSampler].m_TexSrtAnimIndex;
        if (texSrtAnimIdx != -1)
        {
            const TexSrtAnim& texSrtAnim = m_SceneMaterials.getTexSrtAnimData(texSrtAnimIdx);
            const Dcc::RSampler& sampler = mat.m_Samplers[iSampler];
            os << Dcc::RTab(tc + 1) << "<original_texsrt_anim index=\"" << outAnimIdx
                << "\" hint=\"" << sampler.GetHintString() << "\"" << R_ENDL;
            os << Dcc::RTab(tc + 2) << "mode=\"" << sampler.m_OriginalTexsrt.GetModeString() << "\"" << R_ENDL;
            os << Dcc::RTab(tc + 1) << ">" << R_ENDL;
            for (int paramIdx = 0; paramIdx < Dcc::ROriginalTexsrt::PARAM_COUNT; ++paramIdx)
            {
                const Dcc::RAnimCurve& curve = texSrtAnim.m_Anims[paramIdx];
                if (curve.m_UseFlag)
                {
                    curve.Out(os, dataStreams, tc + 2, "original_texsrt_anim_target",
                        Dcc::ROriginalTexsrt::GetParamName(paramIdx), true);
                }
            }
            os << Dcc::RTab(tc + 1) << "</original_texsrt_anim>" << R_ENDL;
            ++outAnimIdx;
        }
    }

    //-----------------------------------------------------------------------------
    // end original texsrt anim array
    os << Dcc::RTab(tc) << "</original_texsrt_anim_array>" << R_ENDL;
}

//-----------------------------------------------------------------------------
//! @brief fts ファイル（テクスチャ SRT アニメーション）を出力します。
//!
//! @param[in,out] os 出力ストリームです。
//! @param[in,out] dataStreams データ列配列です。
//! @param[in] ymodel モデルです。
//!
//! @return 処理結果を返します。
//-----------------------------------------------------------------------------
bool NintendoExport::OutputFtsFile(
                          std::ostream& os,
                          Dcc::RDataStreamArray& dataStreams,
                          const Dcc::RExpOpt& rOpt
                          )
{
    const int tc = 0;

    const int animatedCount = GetTexSrtAnimatedMaterialCount();
    if (animatedCount != 0)
    {
        os << Dcc::RTab(tc) << "<original_material_anim_array length=\"" << animatedCount
            << "\">" << R_ENDL;

        int matAnimIdx = 0;
        const int matCount = static_cast<int>(m_pOutMaterials.size());
        for (int iMat = 0; iMat < matCount; ++iMat)
        {
            const FMaterial& mat = *m_pOutMaterials[iMat];
            if (mat.IsContainingTexSRTAnimData())
            {
                os << Dcc::RTab(tc + 1) << "<original_material_anim index=\"" << matAnimIdx
                    << "\" mat_name=\"" << Dcc::RGetUtf8FromShiftJis(mat.m_Name) << "\">" << R_ENDL;
                OutputTexSrtAnims(os, dataStreams, tc + 2, mat);
                os << Dcc::RTab(tc + 1) << "</original_material_anim>" << R_ENDL;
                ++matAnimIdx;
            }
        }
        os << Dcc::RTab(tc) << "</original_material_anim_array>" << R_ENDL;
    }

    return true;
}

bool NintendoExport::OutputTexPatterns(ostream& os, const Dcc::RExpOpt& rOpt, const int tc)
{
    Dcc::RStringArray texPatRefs;
    if( !CreateTexPatAnimData(texPatRefs, rOpt) )
    {
        return false;
    }

    const int texPatCount = static_cast<int>(texPatRefs.size());
    os << Dcc::RTab(tc) << "<tex_pattern_array length=\"" << texPatCount
        << "\">" << R_ENDL;
    for (int iTexPat = 0; iTexPat < texPatCount; ++iTexPat)
    {
        os << Dcc::RTab(tc + 1) << "<tex_pattern"
            << " pattern_index=\"" << iTexPat
            << "\" tex_name=\"" << Dcc::RGetUtf8FromShiftJis(Dcc::REncodeXmlString(texPatRefs[iTexPat]))
            << "\" />" << R_ENDL;
    }
    os << Dcc::RTab(tc) << "</tex_pattern_array>" << R_ENDL;
    return true;
}

//-----------------------------------------------------------------------------
//! @brief ftp ファイル（テクスチャパターンアニメーション）を出力します。
//!
//! @param[in,out] os 出力ストリームです。
//! @param[in,out] dataStreams データ列配列です。
//!
//! @return 処理結果を返します。
//-----------------------------------------------------------------------------
bool NintendoExport::OutputFtpFile( ostream& os, Dcc::RDataStreamArray& dataStreams, const Dcc::RExpOpt& rOpt)
{
    const int tc = 0;

    const int animatedCount = GetTexPatAnimatedMaterialCount();
    if (animatedCount != 0)
    {
        //-----------------------------------------------------------------------------
        // テクスチャパターン配列
        if (!OutputTexPatterns(os, rOpt, tc)) return false;

        //-----------------------------------------------------------------------------
        // tex pattern mat anim array
        os << Dcc::RTab(tc) << "<tex_pattern_mat_anim_array length=\"" << animatedCount
           << "\">" << R_ENDL;
        int matAnimIdx = 0;
        const int matCount = static_cast<int>(m_pOutMaterials.size());
        for (int iMat = 0; iMat < matCount; ++iMat)
        {
            const FMaterial& mat = *m_pOutMaterials[iMat];
            if (mat.IsContainingTexPatAnimData())
            {
                os << Dcc::RTab(tc + 1) << "<tex_pattern_mat_anim index=\"" << matAnimIdx
                    << "\" mat_name=\"" << Dcc::RGetUtf8FromShiftJis(mat.m_Name) << "\">" << R_ENDL;
                OutputTexPatAnims(os, dataStreams, false, tc + 2, mat);
                os << Dcc::RTab(tc + 1) << "</tex_pattern_mat_anim>" << R_ENDL;
                ++matAnimIdx;
            }
        }
        os << Dcc::RTab(tc) << "</tex_pattern_mat_anim_array>" << R_ENDL;
    }

    return true;
}
//-----------------------------------------------------------------------------
//! @brief fma ファイル（マテリアルアニメーション）を出力します。
//!
//! @param[in,out] os 出力ストリームです。
//! @param[in,out] dataStreams データ列配列です。
//! @param[in,out] ymodel モデルです。
//! @param[in] outputsColor カラーアニメーションを含めるなら true です。
//! @param[in] outputsTexSrt テクスチャ SRT アニメーションを含めるなら true です。
//! @param[in] outputsTexPat テクスチャパターンアニメーションを含めるなら true です。
//!
//! @return 処理結果を返します。
//-----------------------------------------------------------------------------
bool NintendoExport::OutputFmaFile(
    std::ostream& os,
    Dcc::RDataStreamArray& dataStreams,
    const Dcc::RExpOpt& rOpt,
    const bool outputsColor,
    const bool outputsTexSrt,
    const bool outputsTexPat
    )
{
    const int tc = 0;

    //-----------------------------------------------------------------------------
    // テクスチャパターン配列
    const int texPatAnimatedCount =
        (outputsTexPat) ? GetTexPatAnimatedMaterialCount() : 0;
    if (texPatAnimatedCount != 0)
    {
        if (!OutputTexPatterns(os, rOpt, tc))
        {
            return false;
        }
    }

    //-----------------------------------------------------------------------------
    // マテリアル単位アニメーション配列
    const int matCount = static_cast<int>(m_pOutMaterials.size());
    const int animatedCount = texPatAnimatedCount;
    if (animatedCount != 0)
    {
        os << Dcc::RTab(tc) << "<per_material_anim_array length=\"" << animatedCount
            << "\">" << R_ENDL;
        int matAnimIdx = 0;
        for (int matIdx = 0; matIdx < matCount; ++matIdx)
        {
            const FMaterial& mat = *m_pOutMaterials[matIdx];
            const bool hasTexPatAnim = (outputsTexPat && mat.IsContainingTexPatAnimData());
            if (hasTexPatAnim)
            {
                os << Dcc::RTab(tc + 1) << "<per_material_anim index=\"" << matAnimIdx
                    << "\" mat_name=\"" << Dcc::RGetUtf8FromShiftJis(mat.m_Name) << "\">" << R_ENDL;
                OutputTexPatAnims(os, dataStreams, true, tc + 2, mat);
                os << Dcc::RTab(tc + 1) << "</per_material_anim>" << R_ENDL;
                ++matAnimIdx;
            }
        }
        os << Dcc::RTab(tc) << "</per_material_anim_array>" << R_ENDL;
    }

    //-----------------------------------------------------------------------------
    // オリジナルのマテリアル単位アニメーション配列
    const int orgAnimatedCount =
        ((outputsColor ) ? GetColorAnimatedMaterialCount()  : 0) +
        ((outputsTexSrt) ? GetTexSrtAnimatedMaterialCount() : 0);
    if (orgAnimatedCount != 0)
    {
        os << Dcc::RTab(tc) << "<original_per_material_anim_array length=\"" << orgAnimatedCount
            << "\">" << R_ENDL;
        int matAnimIdx = 0;
        for (int matIdx = 0; matIdx < matCount; ++matIdx)
        {
            const FMaterial& mat = *m_pOutMaterials[matIdx];
            const bool hasColorAnim  = (outputsColor  && mat.IsContainingColorAnimData()   );
            const bool hasTexSrtAnim = (outputsTexSrt && mat.IsContainingTexSRTAnimData());
            if (hasColorAnim || hasTexSrtAnim)
            {
                os << Dcc::RTab(tc + 1) << "<original_per_material_anim index=\"" << matAnimIdx
                    << "\" mat_name=\"" << Dcc::RGetUtf8FromShiftJis(mat.m_Name) << "\">" << R_ENDL;
                if (hasColorAnim)
                {
                    mat.OutputColorAnims(os, dataStreams, tc + 2);
                }
                if (hasTexSrtAnim)
                {
                    OutputTexSrtAnims(os, dataStreams, tc + 2, mat);
                }
                os << Dcc::RTab(tc + 1) << "</original_per_material_anim>" << R_ENDL;
                ++matAnimIdx;
            }
        }
        os << Dcc::RTab(tc) << "</original_per_material_anim_array>" << R_ENDL;
    }

    return true;
}

//-----------------------------------------------------------------------------
//! @brief テクスチャSRTアニメーションが設定されたマテリアルの数を返します。
//-----------------------------------------------------------------------------
int NintendoExport::GetTexSrtAnimatedMaterialCount(void)
{
    int animatedCount = 0;
    const int matCount = static_cast<int>(m_pOutMaterials.size());
    for (int iMat = 0; iMat < matCount; ++iMat)
    {
        const FMaterial& mat = *m_pOutMaterials[iMat];
        if (mat.IsContainingTexSRTAnimData())
        {
            ++animatedCount;
        }
    }
    return animatedCount;
}

//-----------------------------------------------------------------------------
//! @brief テクスチャパターンアニメーションが設定されたマテリアルの数を返します。
//-----------------------------------------------------------------------------
int NintendoExport::GetTexPatAnimatedMaterialCount(void)
{
    int animatedCount = 0;
    const int matCount = static_cast<int>(m_pOutMaterials.size());
    for (int iMat = 0; iMat < matCount; ++iMat)
    {
        const FMaterial& mat = *m_pOutMaterials[iMat];
        if (mat.IsContainingTexPatAnimData())
        {
            ++animatedCount;
        }
    }
    return animatedCount;
}

//-----------------------------------------------------------------------------
//! @brief マテリアルカラーアニメーションが設定されたマテリアルの数を返します。
//-----------------------------------------------------------------------------
int NintendoExport::GetColorAnimatedMaterialCount(void)
{
    int animatedCount = 0;
    const int matCount = static_cast<int>(m_pOutMaterials.size());
    for (int iMat = 0; iMat < matCount; ++iMat)
    {
        const FMaterial& mat = *m_pOutMaterials[iMat];
        if (mat.IsContainingColorAnimData())
        {
            ++animatedCount;
        }
    }
    return animatedCount;
}


//-----------------------------------------------------------------------------
//! @brief テクスチャパターンアニメーションデータを作成します。
//!        全テクスチャを取得してから処理するので ftp ファイル出力時に呼ばれます。
//!
//! @param[out] texPatRefs モデルの全テクスチャパターンアニメーションで参照する
//!                        テクスチャ名の配列を格納します。
//!
//! @return 処理結果を返します。
//-----------------------------------------------------------------------------
bool NintendoExport::CreateTexPatAnimData(Dcc::RStringArray& texPatRefs, const Dcc::RExpOpt& rOpt)
{
    //-----------------------------------------------------------------------------
    // check size
    const int texPatAnimCount = static_cast<int>(m_SceneMaterials.getNumTexPatAnimData());
    if (texPatAnimCount == 0)
    {
        return true;
    }

    //-----------------------------------------------------------------------------
    // get full value & tex index list

    // init
    const int allTexImgCount = m_SceneMaterials.getNumTexture();
    Dcc::RIntArray texImgIdxs(allTexImgCount, -1);

    // get full value & tex used flag
    for (int ianim = 0; ianim < texPatAnimCount; ++ianim)
    {
        if( !AnalyzeTexPatFullAnim(m_SceneMaterials.getTexPatAnimData(ianim), texImgIdxs, rOpt) )
        {
            return false;
        }
    }

    // set out tex index
    int outTexImgCount = 0;
    for (int iTexImg = 0; iTexImg < allTexImgCount; ++iTexImg)
    {
        const Dcc::RImage& texImg = *m_pOutTexImgs[iTexImg];
        if (texImgIdxs[texImg.GetOrgIndex()] != -1)
        {
            texImgIdxs[texImg.GetOrgIndex()] = outTexImgCount;
            ++outTexImgCount;
        }
    }

    //-----------------------------------------------------------------------------
    // get key
    for (int ianim = 0; ianim < texPatAnimCount; ++ianim)
    {
        TexPatAnim& texPatAnim = m_SceneMaterials.getTexPatAnimData(ianim);
        GetTexPatKeyAnim(texPatAnim, texImgIdxs, rOpt);
    }

    //-----------------------------------------------------------------------------
    // create texture references
    if (outTexImgCount > 0)
    {
        for (int iTexImg = 0; iTexImg < allTexImgCount; ++iTexImg)
        {
            const Dcc::RImage& texImg = *m_pOutTexImgs[iTexImg];
            if (texImgIdxs[texImg.GetOrgIndex()] != -1)
            {
                texPatRefs.push_back(texImg.GetName());
            }
        }
    }

    return true;
}

//-----------------------------------------------------------------------------
//! @brief 全サブフレームにおけるテクスチャパターンアニメーション値を分析します。
//!        参照されるテクスチャイメージの texImgIdxs に 0 を設定します。
//!        全テクスチャを取得してから処理するので ftp ファイル出力時に呼ばれます。
//!
//! @param[in,out] texPatAnim テクスチャパターンアニメーションです。
//! @param[in,out] texImgIdxs モデルの全テクスチャイメージの
//!                           <tex_pattern_array> 内のインデックスです。
//!                           テクスチャパターンアニメーションで参照されていなければ -1 です。
//!
//! @return 処理結果を返します。
//-----------------------------------------------------------------------------
bool NintendoExport::AnalyzeTexPatFullAnim(
        TexPatAnim& texPatAnim,
        Dcc::RIntArray& texImgIdxs,
        const Dcc::RExpOpt& rOpt )
{
    texPatAnim.m_ImgAnim.m_Name = texPatAnim.m_ObjName + ".img";

#if	0
    //	以下は Maya のオリジナルコード
    //	すでに m_FullValues へ値が設定済みのため処理しない。
    const RFloatArray& fullValues = texPatAnim.m_FeAnim.m_FullValues;
    //RPrintArray(cerr, texPatAnim.m_FeAnim.m_FullValues, "tp");

    const int texCount = texPatAnim.m_TexImgIdxs.length();
    bool warnFlag = false;
    for (int iframe = 0; iframe < yopt.m_SubFrameCount; ++iframe)
    {
        //-----------------------------------------------------------------------------
        // get index in fe list
        const int fe = RRound(fullValues[iframe]);
        int itexNear = 0;
        int diffMin = INT_MAX;
        int itex;
        for (itex = 0; itex < texCount; ++itex)
        {
            if (texPatAnim.m_UsedFes[itex] == fe)
            {
                break;
            }
            const int diff = RAbs(texPatAnim.m_UsedFes[itex] - fe);
            if (diff < diffMin)
            {
                diffMin = diff;
                itexNear = itex;
            }
        }
        if (itex >= texCount)
        {
            itex = itexNear;
            if (!warnFlag)
            {
                warnFlag = true;
                YShowWarning(yscene, "Frame extension is wrong: (use nearest): %s: %d -> %d",
                    texPatAnim.m_ObjName.c_str(), fe, texPatAnim.m_UsedFes[itexNear]);
            }
        }

        //-----------------------------------------------------------------------------
        // append tex img index
        const int iTexImg = texPatAnim.m_TexImgIdxs[itex];
        texPatAnim.m_ImgAnim.m_FullValues.push_back(static_cast<float>(iTexImg));

        //-----------------------------------------------------------------------------
        // get tex used flags
        texImgIdxs[iTexImg] = 0; // 仮の値を格納します。
    }
#else

    //	全てのテクスチャリストの中で、テクスチャパターンアニメーションで
    //	使用しているもののみマークをつけて後の処理で使用している。
    const int texCount = static_cast<int>(texPatAnim.m_TexImgIdxs.size());
    for ( int iTexIndex = 0; iTexIndex < texCount; iTexIndex++ )
    {
        const int iTexImg = texPatAnim.m_TexImgIdxs[iTexIndex];

        //-----------------------------------------------------------------------------
        // get tex used flags
        texImgIdxs[iTexImg] = 0; // 仮の値を格納します。
    }

#endif

    return true;
}

//-----------------------------------------------------------------------------
//! @brief テクスチャパターンのキーアニメーションを取得します。
//!        全テクスチャを取得してから処理するので ftp ファイル出力時に呼ばれます。
//!
//! @param[in,out] texPatAnim テクスチャパターンアニメーションです。
//! @param[in] texImgIdxs モデルの全テクスチャイメージの
//!                       <tex_pattern_array> 内のインデックスです。
//!                       テクスチャパターンアニメーションで参照されていなければ -1 です。
//! @param[in] yopt Export オプションです。
//-----------------------------------------------------------------------------
void NintendoExport::GetTexPatKeyAnim(
    TexPatAnim& texPatAnim,
    const Dcc::RIntArray& texImgIdxs,
    const Dcc::RExpOpt& rOpt
)
{
    Dcc::RAnimCurve& curve = texPatAnim.m_ImgAnim;
#if	0
    //	TODO:
    //-----------------------------------------------------------------------------
    // check constant
    curve.m_Tolerance = 0.0f;
    curve.UpdateConstantFlag();
    if (curve.m_ConstantFlag)
    {
        // convert constant value (first full value)
        int value = Dcc::RRound(curve.m_FullValues[0]);
        if (value != -1)
        {
            value = texImgIdxs[value];
        }
        curve.m_FullValues[0] = static_cast<float>(value);
        return;
    }
#endif

#if	0
    //	キーは全て必ずベイクされる＆キーはデータ取得時に作成済みのため
    //	ここでは何も処理しない
    //-----------------------------------------------------------------------------
    // get & make key
    const bool getKeyFlag = (
        !yopt.m_BakeAllAnim &&
        texPatAnim.m_ValidCurveFlag &&
        IsStepAnimCurve(texPatAnim.m_InputObj));

    if (getKeyFlag)
    {
        curve.GetStepKeys(texPatAnim.m_InputObj, yopt);
        ConvertTexPatAnimImgKey(curve, texPatAnim);
    }
    else
    {
        //cerr << "make keys: " << curve.m_Name << endl;
        curve.MakeStepKeys(GetFloatFrameFromSubFrame, &yopt);
    }
#endif

    //-----------------------------------------------------------------------------
    // キーの値をテクスチャイメージのインデックスから
    // <tex_pattern_array> 内のインデックスに変換します。
    const int keyCount = static_cast<int>(curve.m_Keys.size());
    for (int ikey = 0; ikey < keyCount; ++ikey)
    {
        Dcc::RAnimKey& key = curve.m_Keys[ikey];
        int value = Dcc::RRound(key.m_Value);
        if (value != -1)
        {
            value = texImgIdxs[value];
        }
        key.m_Value = static_cast<float>(value);
    }
}

//-----------------------------------------------------------------------------
//! @brief テクスチャパターンアニメーションを 1 つ出力します。
//!
//! @param[in,out] os 出力ストリームです。
//! @param[in,out] dataStreams データ列配列です。
//! @param[in] tc <tex_pattern_mat_anim> 要素のインデントに必要なタブの数です。
//! @param[in] matAnimIdx オリジナルマテリアルアニメーションのインデックスです。
//! @param[in] mat マテリアルです。
//! @param[in] ymodel モデルです。
//-----------------------------------------------------------------------------
void NintendoExport::OutputTexPatAnims(std::ostream& os, Dcc::RDataStreamArray& dataStreams, const bool isFma, const int tc, const FMaterial& mat)
{
    int animatedSamplerCount = 0;
    for (size_t samplerIdx = 0; samplerIdx < mat.m_Samplers.size(); ++samplerIdx)
    {
        const Dcc::RSampler& sampler = mat.m_Samplers[samplerIdx];
        if (sampler.m_TexPatAnimIndex != -1)
        {
            ++animatedSamplerCount;
        }
    }

    if (isFma)
    {
        os << Dcc::RTab(tc) << "<tex_pattern_anim_array length=\""
            << animatedSamplerCount << "\">" << R_ENDL;
    }
    const int texPatAnimTc = (isFma) ? tc + 1 : tc;
    const char* texPatAnimElemName = (isFma) ?
        "pattern_anim" : "pattern_anim_target";
    for (size_t samplerIdx = 0; samplerIdx < mat.m_Samplers.size(); ++samplerIdx)
    {
        const Dcc::RSampler& sampler = mat.m_Samplers[samplerIdx];
        if (sampler.m_TexPatAnimIndex != -1)
        {
            const TexPatAnim& texPatAnim = m_SceneMaterials.getTexPatAnimData(sampler.m_TexPatAnimIndex);
            const Dcc::RAnimCurve& curve = texPatAnim.m_ImgAnim;
            os << Dcc::RTab(texPatAnimTc) << "<" << texPatAnimElemName
               << " sampler_name=\"" << sampler.GetName()
               << "\" hint=\"" << sampler.GetHintString()
               << "\" base_value=\"" << curve.GetBaseValue() << "\"";
            if (curve.m_ConstantFlag)
            {
                os << " />" << R_ENDL;
            }
            else
            {
                os << ">" << R_ENDL;
                curve.Out(os, dataStreams, texPatAnimTc + 1, false);
                os << Dcc::RTab(texPatAnimTc) << "</" << texPatAnimElemName << ">" << R_ENDL;
            }
        }
    }
    if (isFma)
    {
        os << Dcc::RTab(tc) << "</tex_pattern_anim_array>" << R_ENDL;
    }
}



//-----------------------------------------------------------------------------
//! @brief テクスチャ中間ファイル群を出力します。
//!
//! @return 処理結果を返します。
//-----------------------------------------------------------------------------
bool NintendoExport::OutputTextureFiles(string tmpFolder, const Dcc::RExpOpt& rOpt)
{
    //-----------------------------------------------------------------------------
    // テクスチャイメージ数をチェックします。
    const int texImgCount = static_cast<int>(m_pOutTexImgs.size());
    if (texImgCount == 0)
    {
        return true;
    }

    // CTEX出力TEMPフォルダを作成
    //string texFolder = m_OutFolder;
    string texFolder = m_OutFolder;
    texFolder += Dcc::RExpOpt::TexFolderName;
    texFolder += "/";
    string texTmpFolder = tmpFolder.data();
    texTmpFolder += Dcc::RExpOpt::TexFolderName;
    texTmpFolder += "/";
    if(DoesFileExist(A_2_M(texTmpFolder.c_str())) == FALSE)
    {
        if(!CreateDirectoryA(texTmpFolder.c_str(), nullptr))
        {
            RLogger::LogMessage("texTmpFolder create error");
            RLogger::LogMessagebyID(RLogger::kLogERR_OpenFile, texTmpFolder.c_str());
            DeleteDirectoryDeep(tmpFolder.data());
            return false;
        }
    }

    //-----------------------------------------------------------------------------
    // 画像ファイルをリードして テクスチャコンバータの変換リストに項目を作成します。
    const clock_t startClock = clock();
    int	mergeCount = 0;

    std::ostringstream cvtrOss;
    for (int iTexImg = 0; iTexImg < texImgCount; ++iTexImg)
    {
        Dcc::RImage& texImg = *m_pOutTexImgs[iTexImg];
#if	0
        //	TODO:	必要？
        //-----------------------------------------------------------------------------
        // 同じテクスチャ名を持つテクスチャイメージがすでに出力されているか判定します。
        RImage& texImg = *ymodel.m_pOutTexImgs[iTexImg];
        bool newTexFlag = true;
        const int otherCount = static_cast<int>(yscene.m_AllTexImgs.size());
        for (int iOther = 0; iOther < otherCount; ++iOther)
        {
            const RImage& other = yscene.m_AllTexImgs[iOther];
            if (other.GetName() == texImg.GetName())
            {
                newTexFlag = false;
                // 同じテクスチャ名を持つテクスチャイメージがすでに出力されていた場合、
                // テクスチャファイルパスを比較して異なれば警告を表示します。
                CheckTextureIdentical(yscene, texImg, other);
                break;
            }
        }
#endif

        //-----------------------------------------------------------------------------
        // 画像ファイルをリードしてテクスチャイメージの属性を設定します。
        const std::string ftxName = texImg.GetName() + "." + rOpt.GetExtension(Dcc::RExpOpt::FTX);
        const std::string ftxPath = texTmpFolder + ftxName;
        std::string mergePath = (rOpt.m_MergeFtxFlag) ? rOpt.m_MergeFtxFolderPath + ftxName : "";
        mergePath = (rOpt.m_MergeFtxFlag && Dcc::RFileExists(mergePath)) ? mergePath : "";

        const bool setTexelDataFlag = false;
        const bool forceGetFullBitmap = true;

        Dcc::RStatus rstatus = texImg.ReadFile( texImg.GetFilePaths(), mergePath, ""
            , texImg.GetHint()
            , texImg.GetLinearFlag(), rOpt.m_UsesSrgbFetch, texImg.GetDimension()
            , texImg.GetInitialSwizzle(), rOpt.m_CommentText);

        //-----------------------------------------------------------------------------
        // テクスチャコンバータの変換リストに項目を追加します。
        //if (status && outFlag)
        if (rstatus)
        {
            texImg.SetProjectRootPath(rOpt.m_ProjectRootPath);
            texImg.SetWeightedCompress(rOpt.m_EnablesWeightedCompress);
            std::string cvtrOpt = texImg.GetConverterOption(ftxPath);
            cvtrOss << cvtrOpt << R_ENDL;
        }
        else
        {
            RLogger::LogMessage(rstatus.GetMessage().c_str(), RLogger::kError );
            return false;
        }

        //-----------------------------------------------------------------------------
        // マージしたテクスチャ数を記録します
        //if (newTexFlag && !mergePath.empty())
        if (!mergePath.empty())
        {
            mergeCount++;
        }

        //-----------------------------------------------------------------------------
        // ファイル移動情報を追加します。
        //if (outFlag && newTexFlag)
        {
            m_TmpFileMoves.push_back(Dcc::RFileMove( ftxPath, texFolder + ftxName ));
        }
    }

    //-----------------------------------------------------------------------------
    // マージ対象のテクスチャが 1 つも無い場合は警告を出力します。
    // TODO: アニメーションレンジ出力に対応するなら全モデル出力後に判定
    if (rOpt.MergesFtx() && mergeCount == 0)
    {
        RLogger::LogMessagebyID(RLogger::kLogWRN_FtxFileToMergeIsNotFound);
        //return false;
    }

    // CTEX出力フォルダを作成
    if(!CreateDirectoryA(texFolder.c_str(), nullptr))
    {
        //	4C でもコメントアウトされている
        //RLogger::LogMessagebyID(RLogger::kLogERR_OpenFile, texFolder.c_str());
        //DeleteDirectoryDeep(m_TmpOutFolderPath.data());
        //return false;
    }

    //-----------------------------------------------------------------------------
    // テクスチャコンバータを実行します。
    const std::string cvtrStr = cvtrOss.str();
    if (rOpt.ExportsTexture() && !cvtrStr.empty())
    {
        //-----------------------------------------------------------------------------
        // テクスチャコンバータの exe ファイルが存在するかチェックします。
        const std::string toolsPath = Get3dCommandLineToolsPath();
        // 新しいパスに存在するかどうかをチェック
        std::string converterPath = toolsPath + "3dTextureConverter.exe";
        if (!Dcc::RFileExists(converterPath))
        {
            converterPath = toolsPath + "NW4F_g3dtexcvtr.exe";
        }
        if (!Dcc::RFileExists(converterPath))
        {
            RLogger::LogMessagebyID(RLogger::kLogERR_TextureConverterCantBeFound, "at " +  converterPath);
            return false;
        }

        //-----------------------------------------------------------------------------
        // create option file
        //cerr << "cvtrOss:" << endl << cvtrStr << endl;
        const std::string convertListFilePath = texTmpFolder +
            "Nintendo_TextureConverter_" +
            Dcc::RGetNumberString(static_cast<int>(GetCurrentProcessId())) +
            ".txt";
        std::ofstream ofs;
        ofs.open(convertListFilePath.c_str(), ios_base::out | ios_base::binary);
        if (!ofs)
        {
            RLogger::LogMessagebyID(RLogger::kLogERR_OpenFile, convertListFilePath);
            return false;
        }
        ofs.write(cvtrStr.c_str(), cvtrStr.size());
        ofs.close();

        //-----------------------------------------------------------------------------
        // do
        std::string cmd = "\"" + Dcc::RGetWindowsFilePath(converterPath) + "\"";
        cmd += " -s"; // 処理時間を表示する場合もテクスチャコンバータの情報表示は抑制します。
        cmd += " --final-folder=" + texFolder + "";
        cmd += " --job-list=\"" + convertListFilePath + "\"";

        //	TODO:	プロファイルを出力するかどうかを判定する
        //	Maya では NW4F_MAYA_PROFILE という環境変数が定義されているかどうかで判断している
        //if (yopt.m_DisplaysProfile) // 処理時間を表示する場合もテクスチャコンバータのコマンドライン表示は抑制します。

        std::string outMsg;
        std::string errMsg;
        const int exitCode = Dcc::RExecProcessWithPipe(nullptr, nullptr, &errMsg, cmd.c_str(), SW_HIDE);

        if (!outMsg.empty())
        {
            //	バッチエクスポートでなかったらメッセージを表示する
            RLogger::LogMessage(std::string(outMsg.c_str()), RLogger::kInfo );
        }

        if (!errMsg.empty())
        {
            //	バッチエクスポートでなかったらメッセージを表示する
            RLogger::LogMessage(std::string(errMsg.c_str()), RLogger::kError);
        }

        if (exitCode != 0)
        {
            RLogger::LogMessagebyID(RLogger::kLogERR_CantExportFtxFile);
            return false;
        }

        //-----------------------------------------------------------------------------
        // delete option file
        ::DeleteFileA(convertListFilePath.c_str());

        //-----------------------------------------------------------------------------
        // profile
        m_TexClocks += clock() - startClock;
    }

    return true;
}


#if 0
//-----------------------------------------------------------------------------
//! @brief レンダーセット配列を出力します。
//!
//! @param[in,out] os 出力ストリームです。
//! @param[in] tc <renderset_array> 要素のインデントに必要なタブの数です。
//! @param[in] ymodel モデルです。
//-----------------------------------------------------------------------------
void NintendoExport::OutRenderSets(std::ostream& os, const int tc, RNodeProcessor& np)
{
    std::vector<FShape*>	Shapes;
    np.collectShapeData( Shapes );

    const int shapeCount = static_cast<int>(Shapes.size());
    if (shapeCount > 0)
    {
        Dcc::RRendersetArray rendersets;
        for (int iShape = 0; iShape < shapeCount; ++iShape)
        {
            const FShape& shapeData = *Shapes[iShape];
            const FMaterial& mat = *m_SceneMaterials.getFMaterialByIndex( shapeData.GetMaterialId() );
            rendersets.push_back(
                Dcc::RRenderset(mat.m_Index, shapeData.m_Index, shapeData.m_Priority));
        }
        Dcc::ROutArrayElement(os, tc, rendersets, "renderset_array");
    }
}
#endif

//-----------------------------------------------------------------------------
//! @brief マテリアル群を出力します。
//!
//! @param[in,out] os 出力ストリームです。
//! @param[in] tc <material_array> 要素のインデントに必要なタブの数です。
//-----------------------------------------------------------------------------
void NintendoExport::OutputMaterials(std::ostream& os, const int tc)
{
    //-----------------------------------------------------------------------------
    // set attr after out tex
    // テクスチャを出力するまで決定しないアトリビュートはここで設定します。
    SetMaterialAttrAfterOutTex();

    //-----------------------------------------------------------------------------
    // out material array
    const int matCount = static_cast<int>(m_pOutMaterials.size());
    if (matCount > 0)
    {
        os << Dcc::RTab(tc) << "<material_array length=\"" << matCount << "\">" << R_ENDL;

        // loop for material
        for (int iMat = 0; iMat < matCount; ++iMat)
        {
            const FMaterial& mat = *m_pOutMaterials[iMat];
            mat.Out(os, tc + 1, iMat);
        }

        os << Dcc::RTab(tc) << "</material_array>" << R_ENDL;
    }
}

//-----------------------------------------------------------------------------
//! @brief オリジナルマテリアル群を出力します。
//!
//! @param[in,out] os 出力ストリームです。
//! @param[in] tc <original_material_array> 要素のインデントに必要なタブの数です。
//-----------------------------------------------------------------------------
void NintendoExport::OutputOriginalMaterials(std::ostream& os, const int tc)
{
    const int matCount = static_cast<int>(m_pOutMaterials.size());
    if (matCount > 0)
    {
        os << Dcc::RTab(tc) << "<original_material_array length=\"" << matCount << "\">" << R_ENDL;

        // loop for material
        for (int iMat = 0; iMat < matCount; ++iMat)
        {
            const FMaterial& mat = *m_pOutMaterials[iMat];
            mat.OutOriginal(os, tc + 1, iMat);
        }

        os << Dcc::RTab(tc) << "</original_material_array>" << R_ENDL;
    }
}

//-----------------------------------------------------------------------------
//! @brief テクスチャ出力後に設定可能なマテリアル属性を設定します。
//!
//! @param[in,out] ymodel モデルです。
//-----------------------------------------------------------------------------
void NintendoExport::SetMaterialAttrAfterOutTex(void)
{
    //-----------------------------------------------------------------------------
    // loop for material
    const int matCount = static_cast<int>(m_pOutMaterials.size());
    for (int iMat = 0; iMat < matCount; ++iMat)
    {
        FMaterial& mat = *m_pOutMaterials[iMat];

        //-----------------------------------------------------------------------------
        // loop for sampler
        for (int iSampler = 0; iSampler < static_cast<int>(mat.m_Samplers.size()); ++iSampler)
        {
            Dcc::RSampler& sampler = mat.m_Samplers[iSampler];
            const Dcc::RImage& texImg = m_SceneMaterials.getTexture(sampler.m_ImageIndex);

            //-----------------------------------------------------------------------------
            // ミップマップならミップマップフィルタを変更します。
            if (texImg.GetMipLevel() >= 2)
            {
                sampler.m_FilterMip = (exportOptions.m_Opt.m_IsFilterMipLinear) ? Dcc::RSampler::LINEAR : Dcc::RSampler::POINT;
            }

#if	0
            //	Maya のコードでは m_AutoRenderStateFlag を true に設定している箇所が無いため
            //	ここへは絶対に入らない
            //-----------------------------------------------------------------------------
            // レンダーステートを設定します。
            if (mat.m_AutoRenderStateFlag)
            {
#if	0
                //	同じテクスチャが違う用途で使用されていると
                //	Texture is used for different usage
                //	になるのでここへは入らないと思う
                if (sampler.IsAlbedo() &&
                    mat.m_TransparencyTexObj == mat.m_ColorTexObj)
                {
                    if (texImg.GetTransparencyMode() == RImage::Mask)
                    {
                        if (mat.GetRenderStateMode() == RRenderState::Opaque)
                        {
                            mat.SetRenderStateMode(RRenderState::Mask);
                        }
                    }
                    else if (texImg.GetTransparencyMode() == RImage::Translucent)
                    {
                        mat.SetRenderStateMode(RRenderState::Translucent);
                    }
                }
                else if (sampler.IsOpacity())
#endif
                if (sampler.IsOpacity())
                {
                    mat.SetRenderStateMode(RRenderState::XLU);
                }
            }
#endif

            //-----------------------------------------------------------------------------
            // キューブマップが貼られていたらマッピング方法を変更します。
            // TODO: キューブマップ用のシェーダを使用するように設定する？
            if (texImg.IsCubeMap())
            {
                //sampler.m_TexSrc.m_Semantic = RTexSrc::Normal;
            }
        }
    }

}

//-----------------------------------------------------------------------------
//! @brief 中間ファイルのヘッダを出力します。
//!
//! @param[in,out] os 出力ストリームです。
//! @param[in] tc 要素のインデントに必要なタブの数です。
//! @param[in] fileType 中間ファイルタイプです。
//! @param[in] ymodel モデルです。
//! @param[in] yopt Export オプションです。
//!
//! @return 処理結果を返します。
//-----------------------------------------------------------------------------
void NintendoExport::OutputHeader( std::ostream& os, const int tc, const Dcc::RExpOpt::FileType fileType, const Dcc::RExpOpt& rOpt)
{
    //-----------------------------------------------------------------------------
    // xml header
    os << Dcc::RTab(tc) << "<?xml version=\"1.0\" encoding=\"utf-8\"?>" << R_ENDL;

    //-----------------------------------------------------------------------------
    // begin intermediate file
    os << Dcc::RTab(tc) << "<nw4f_3dif version=\"" << Dcc::RExpOpt::GetRootElementVersion()
       << "\">" << R_ENDL;

    //-----------------------------------------------------------------------------
    // file info
    {
        os << Dcc::RTab(tc) << "<file_info>" << R_ENDL;
        os << Dcc::RTab(tc + 1) << "<create tool_name=\""
            << Dcc::RGetUtf8FromShiftJis(Dcc::REncodeXmlString(GetMaxVersionString()))
            << "\" tool_version=\"" << std::string(GetPluginVersionString()) << "\"" << R_ENDL;
        string maxFullPath(Dcc::RGetUnixFilePath(std::string(M_2_A(m_pIp->GetCurFilePath().data()))));
        string srcPath = ReplaceAmpersand(maxFullPath.data());
        srcPath = Dcc::RGetFileInfoSourceFilePath(srcPath, rOpt.m_ProjectRootPath);
        os << Dcc::RTab(tc + 2) << "src_path=\"" << Dcc::RGetUtf8FromShiftJis(Dcc::REncodeXmlString(srcPath)) << "\"" << R_ENDL;
        os << Dcc::RTab(tc + 1) << "/>" << R_ENDL;
        os << Dcc::RTab(tc) << "</file_info>" << R_ENDL;
    }

    //-----------------------------------------------------------------------------
    // begin root element
    os << Dcc::RTab(tc) << "<" << rOpt.GetTypeElementName(fileType)
        << " version=\"" << rOpt.GetTypeElementVersion(fileType) << "\">" << R_ENDL;
    //-----------------------------------------------------------------------------
    // model info
    if (fileType == Dcc::RExpOpt::FMD)
    {
        os << Dcc::RTab(tc) << "<model_info" << R_ENDL;
        os << Dcc::RTab(tc + 1) << "material_count=\"" << m_pOutMaterials.size() << "\"" << R_ENDL;
        os << Dcc::RTab(tc + 1) << "bone_count=\"" << m_NodeProcessor.getNumBones() << "\"" << R_ENDL;
        os << Dcc::RTab(tc + 1) << "shape_count=\"" << m_NodeProcessor.getNumShapes() << "\"" << R_ENDL;
        os << Dcc::RTab(tc + 1) << "smooth_skinning_shape=\"" << m_NodeProcessor.getNumSmoothSkinningShapes() << "\"" << R_ENDL;
        os << Dcc::RTab(tc + 1) << "rigid_skinning_shape=\"" << m_NodeProcessor.getNumRigidSkinningShapes() << "\"" << R_ENDL;
        os << Dcc::RTab(tc + 1) << "smooth_skinning_matrix=\"" << m_NodeProcessor.getNumSmoothSkinningMtxs() << "\"" << R_ENDL;
        os << Dcc::RTab(tc + 1) << "rigid_skinning_matrix=\"" << m_NodeProcessor.getNumRigidSkinningMtxs() << "\"" << R_ENDL;
        os << Dcc::RTab(tc + 1) << "total_process_vertex=\"" << m_NodeProcessor.getTotalProcessVertexCount() << "\"" << R_ENDL;
        os << Dcc::RTab(tc + 1) << "unite_pos_quantize=\"" << Dcc::RBoolStr(false) << "\"" << R_ENDL;
        os << Dcc::RTab(tc + 1) << "dcc_preset=\"" << Dcc::RGetUtf8FromShiftJis(Dcc::REncodeXmlString(rOpt.m_PresetName)) << "\"" << R_ENDL;
        os << Dcc::RTab(tc + 1) << "dcc_magnify=\"" << rOpt.m_Magnify << "\"" << R_ENDL;
        os << Dcc::RTab(tc + 1) << "dcc_start_frame=\"" << rOpt.m_StartFrame << "\"" << R_ENDL;
        os << Dcc::RTab(tc) << "/>" << R_ENDL;
    }

    //-----------------------------------------------------------------------------
    // animation info
    const bool isFma = (!rOpt.m_UsesFclFtsFtp && rOpt.IsFclFtsFtp(fileType));
    const char* animInfoElem = nullptr;
    if (isFma)
    {
        animInfoElem = "material_anim_info";
    }
    else
    {
        switch (fileType)
        {
        case Dcc::RExpOpt::FSK: animInfoElem = "skeletal_anim_info"       ; break;
        case Dcc::RExpOpt::FVB: animInfoElem = "bone_visibility_anim_info"; break;
        case Dcc::RExpOpt::FCL: animInfoElem = "shader_param_anim_info"   ; break;
        case Dcc::RExpOpt::FTS: animInfoElem = "shader_param_anim_info"   ; break;
        case Dcc::RExpOpt::FTP: animInfoElem = "tex_pattern_anim_info"    ; break;
        case Dcc::RExpOpt::FSH: animInfoElem = "shape_anim_info"          ; break;
        case Dcc::RExpOpt::FSN: animInfoElem = "scene_anim_info"          ; break;
        }
    }

    if (animInfoElem != nullptr)
    {
        os << Dcc::RTab(tc) << "<" << animInfoElem;
        if (fileType != Dcc::RExpOpt::FSN)
        {
            os << " frame_count=\"" << rOpt.m_OutFrameCount
                << "\" loop=\"" << Dcc::RBoolStr(rOpt.m_LoopAnim)
                << "\"" << R_ENDL;
        }
        else
        {
            os << R_ENDL;
        }
        os << Dcc::RTab(tc + 1) << "frame_resolution=\"" << rOpt.m_FramePrecision << "\"" << R_ENDL;
        os << Dcc::RTab(tc + 1) << "dcc_preset=\"" << Dcc::RGetUtf8FromShiftJis(Dcc::REncodeXmlString(rOpt.m_PresetName)) << "\"" << R_ENDL;
        if (fileType == Dcc::RExpOpt::FSK ||
            fileType == Dcc::RExpOpt::FSN)
        {
            os << Dcc::RTab(tc + 1) << "dcc_magnify=\"" << rOpt.m_Magnify << "\"" << R_ENDL;
        }
        os << Dcc::RTab(tc + 1) << "dcc_start_frame=\"" << rOpt.m_StartFrame << "\"" << R_ENDL;
        os << Dcc::RTab(tc + 1) << "dcc_end_frame=\"" << rOpt.m_EndFrame << "\"" << R_ENDL;
        os << Dcc::RTab(tc + 1) << "dcc_fps=\"" << GetFrameRate() << "\"" << R_ENDL;
        os << Dcc::RTab(tc + 1) << "bake_all=\"" << Dcc::RBoolStr(true) << "\"" << R_ENDL;

        if (fileType == Dcc::RExpOpt::FSK)
        {
            os << Dcc::RTab(tc + 1) << "scale_mode=\"" << "standard" << "\"" << R_ENDL;
            os << Dcc::RTab(tc + 1) << "rotate_mode=\"" << "euler_xyz" << "\"" << R_ENDL;
            os << Dcc::RTab(tc + 1) << "bake_tolerance_scale=\""     << rOpt.m_TolS << "\"" << R_ENDL;
            os << Dcc::RTab(tc + 1) << "bake_tolerance_rotate=\""    << rOpt.m_TolR << "\"" << R_ENDL;
            os << Dcc::RTab(tc + 1) << "bake_tolerance_translate=\"" << rOpt.m_TolT << "\"" << R_ENDL;
        }
        else if (isFma						||
            fileType == Dcc::RExpOpt::FCL	||
            fileType == Dcc::RExpOpt::FTS)
        {
            os << Dcc::RTab(tc + 1) << "bake_tolerance_color=\""         << rOpt.m_TolC    << "\"" << R_ENDL;
            os << Dcc::RTab(tc + 1) << "bake_tolerance_tex_scale=\""     << rOpt.m_TolTexS << "\"" << R_ENDL;
            os << Dcc::RTab(tc + 1) << "bake_tolerance_tex_rotate=\""    << rOpt.m_TolTexR << "\"" << R_ENDL;
            os << Dcc::RTab(tc + 1) << "bake_tolerance_tex_translate=\"" << rOpt.m_TolTexT << "\"" << R_ENDL;
        }
        else if (fileType == Dcc::RExpOpt::FSN)
        {
            os << Dcc::RTab(tc + 1) << "bake_tolerance_rotate=\""    << rOpt.m_TolR << "\"" << R_ENDL;
            os << Dcc::RTab(tc + 1) << "bake_tolerance_translate=\"" << rOpt.m_TolT << "\"" << R_ENDL;
            os << Dcc::RTab(tc + 1) << "bake_tolerance_color=\""     << rOpt.m_TolC << "\"" << R_ENDL;
        }

        if (fileType == Dcc::RExpOpt::FSK)
        {
            os << Dcc::RTab(tc + 1) << "quantize_tolerance_scale=\""     << rOpt.m_QuantTolS << "\"" << R_ENDL;
            os << Dcc::RTab(tc + 1) << "quantize_tolerance_rotate=\""    << rOpt.m_QuantTolR << "\"" << R_ENDL;
            os << Dcc::RTab(tc + 1) << "quantize_tolerance_translate=\"" << rOpt.m_QuantTolT << "\"" << R_ENDL;
        }
        else if (isFma						||
            fileType == Dcc::RExpOpt::FCL	||
            fileType == Dcc::RExpOpt::FTS)
        {
            os << Dcc::RTab(tc + 1) << "quantize_tolerance_tex_scale=\""     << rOpt.m_QuantTolTexS << "\"" << R_ENDL;
            os << Dcc::RTab(tc + 1) << "quantize_tolerance_tex_rotate=\""    << rOpt.m_QuantTolTexR << "\"" << R_ENDL;
            os << Dcc::RTab(tc + 1) << "quantize_tolerance_tex_translate=\"" << rOpt.m_QuantTolTexT << "\"" << R_ENDL;
        }
        else if (fileType == Dcc::RExpOpt::FSN)
        {
            os << Dcc::RTab(tc + 1) << "quantize_tolerance_rotate=\""    << rOpt.m_QuantTolR << "\"" << R_ENDL;
            os << Dcc::RTab(tc + 1) << "quantize_tolerance_translate=\"" << rOpt.m_QuantTolT << "\"" << R_ENDL;
        }

        os << Dcc::RTab(tc) << "/>" << R_ENDL;
    }
}

void NintendoExport::OutputFooter(std::ostream& os, const int tc, const nn::gfx::tool::dcc::RExpOpt::FileType fileType, const Dcc::RExpOpt& rOpt)
{
    //-----------------------------------------------------------------------------
    // 編集用コメント
    if (!exportOptions.m_Opt.m_CommentText.empty())
    {
        os << Dcc::RTab(tc) << "<comment label=\"" << ""
            << "\" color=\"" << ""
            << "\" text=\"" << Dcc::RGetUtf8FromShiftJis(Dcc::REncodeXmlString(exportOptions.m_Opt.m_CommentText))
            << "\" />" << R_ENDL;
    }

    //-----------------------------------------------------------------------------
    // end root element
    os << Dcc::RTab(tc) << "</" << rOpt.GetTypeElementName(fileType) << ">" << R_ENDL;

    //-----------------------------------------------------------------------------
    // end intermediate file
    os << Dcc::RTab(tc) << "</nw4f_3dif>" << R_ENDL;
}
//-----------------------------------------------------------------------------
//! @brief check no output file
//!
//! @param[in,out] yscene シーンです。
//! @param[in] yopt エクスポートオプションです。
//! @param[in] texCount 出力した ftx ファイル数です。
//!
//! @return 処理結果を返します。
//-----------------------------------------------------------------------------
static bool CheckNoOutputFile(
    const Dcc::RExpOpt& opt,
    const int texCount
    )
{
    if (!opt.ExportsModelFile()   &&
        !opt.ExportsAnim()        &&
        texCount == 0)
    {
        RLogger::LogMessagebyID(RLogger::kLogERR_NoOutputFile);
        return false;
    }
    return true;
}

//-----------------------------------------------------------------------------
//! @brief モデルについての各タイプの中間ファイル群を出力します。
//!
//! @param[out] fileCount 出力した中間ファイル数を格納します。
//! @param[in,out] ymodel モデルです。
//!
//! @return 処理結果を返します。
//-----------------------------------------------------------------------------
bool NintendoExport::OutputFiles(int& fileCount, const Dcc::RExpOpt& rOpt)
{
    //-----------------------------------------------------------------------------
    // check no output file
    if(!CheckNoOutputFile(rOpt, static_cast<int>(m_pOutTexImgs.size())))
    {
        return false;
    }

    //-----------------------------------------------------------------------------
    // warn no anim
    //WarnNoAnim(rOpt);

    //-----------------------------------------------------------------------------
    // テクスチャ中間ファイル群を出力します。
    if (rOpt.ExportsModelFile() ||
        rOpt.ExportsTexture())
    {
        if( !OutputTextureFiles( m_TmpOutFolderPath, rOpt ) )
        {
            FinishExport(false);
            return false;
        }
        fileCount += static_cast<int>(m_pOutTexImgs.size());
    }

    //-----------------------------------------------------------------------------
    // テクスチャ以外の中間ファイル群を出力します。
    bool isSingleFmaOutput = false;
    for (int fileTypeIdx = 0; fileTypeIdx < Dcc::RExpOpt::FILE_TYPE_COUNT; ++fileTypeIdx)
    {
        const Dcc::RExpOpt::FileType fileType = static_cast<Dcc::RExpOpt::FileType>(fileTypeIdx);
        if (rOpt.m_OutFileFlag[fileType])
        {
            if (rOpt.UsesSingleFma() && rOpt.IsFclFtsFtp(fileType))
            {
                if (isSingleFmaOutput)
                {
                    continue;
                }
                isSingleFmaOutput = true;
            }


            const std::string ext = rOpt.GetExtension(fileType);
            const std::string suffix = rOpt.GetSuffix(fileType);
            const std::string extMsg = (!suffix.empty()) ? suffix + "." + ext : ext;
            RLogger::LogMessage(string ("Making ... ") +  extMsg, RLogger::kInfo);
            const std::string tmpFilePath = GetTmpFilePath(fileType);
            if(!OutputOneFile(rOpt, tmpFilePath, fileType))
            {
                return false;
            }
            m_TmpFileMoves.push_back(
                Dcc::RFileMove(tmpFilePath, rOpt.GetFilePath(fileType)));
            ++fileCount;
        }
    }

    return true;
}
//-----------------------------------------------------------------------------
//! @brief 出力する中間ファイルをオープンします。
//!
//! @param[in,out] ofs 出力ファイルストリームです。
//! @param[in,out] yscene シーンです。
//! @param[in] filePath 中間ファイルのパスです。
//! @param[in] fileType 中間ファイルタイプです。
//!
//! @return 処理結果を返します。
//-----------------------------------------------------------------------------
static bool OpenOutputFile(
    std::ofstream& ofs,
    const std::string& filePath)
{
    ofs.open(filePath.c_str(), ios_base::out | ios_base::binary);
    if (!ofs)
    {
        RLogger::LogMessageFormat("The file cannot be opened. Confirm whether the file can be overwritten: %s", filePath);
        return false;
    }

    return true;
}

//-----------------------------------------------------------------------------
//! @brief マテリアル関連のアニメーション中間ファイルを 1 つ出力します。
//!
//! @param[in,out] os 出力ストリームです。
//! @param[in,out] dataStreams データ列配列です。
//! @param[in,out] ymodel モデルです。
//! @param[in] fileType 中間ファイルタイプです。
//!
//! @return 処理結果を返します。
//-----------------------------------------------------------------------------
bool NintendoExport::OutputOneMatAnimFile(
    std::ostream& os,
    Dcc::RDataStreamArray& dataStreams,
    const Dcc::RExpOpt& rOpt,
    const Dcc::RExpOpt::FileType fileType
    )
{
    if (rOpt.UsesSingleFma())
    {
        return OutputFmaFile(os, dataStreams, rOpt,
            rOpt.m_OutFileFlag[Dcc::RExpOpt::FCL],
            rOpt.m_OutFileFlag[Dcc::RExpOpt::FTS],
            rOpt.m_OutFileFlag[Dcc::RExpOpt::FTP]);
    }
    else if (rOpt.UsesSeparatedFma())
    {
        switch (fileType)
        {
        case Dcc::RExpOpt::FCL:
            return OutputFmaFile(os, dataStreams, rOpt, true , false, false);
        case Dcc::RExpOpt::FTS:
            return OutputFmaFile(os, dataStreams, rOpt, false, true , false);
        case Dcc::RExpOpt::FTP:
            return OutputFmaFile(os, dataStreams, rOpt, false, false, true );
        }
    }
    else
    {
        switch (fileType)
        {
        case Dcc::RExpOpt::FCL:
            return OutputFclFile(os, dataStreams, rOpt);
        case Dcc::RExpOpt::FTS:
            return OutputFtsFile(os, dataStreams, rOpt);
        case Dcc::RExpOpt::FTP:
            return OutputFtpFile(os, dataStreams, rOpt);
        }
    }

    return false; // 通常はここに来ません。
}

//-----------------------------------------------------------------------------
//! @brief 中間ファイルを 1 つ出力します。
//!
//! @param[in,out] ymodel モデルです。
//! @param[in] filePath 出力する中間ファイルのパスです。
//! @param[in] fileType 中間ファイルタイプです。
//!
//! @return 処理結果を返します。
//-----------------------------------------------------------------------------
bool NintendoExport::OutputOneFile(
    const Dcc::RExpOpt& rOpt,
    const std::string& filePath,
    const Dcc::RExpOpt::FileType fileType
    )
{
    //const YEnvObjs& yenvObjs = ymodel.GetEnvObjs();

    //-----------------------------------------------------------------------------
    // ファイルを開きます。
    const clock_t startClock = clock();
    const bool isBinary = rOpt.IsBinaryFormat(fileType);
    std::ofstream ofs;
    if(!OpenOutputFile(ofs, filePath))
    {
        return false;
    }

    std::ostream& os = ofs;

    Dcc::ROutUtf8Bom(os);
    Dcc::RInitOutStreamFormat(os);

    //-----------------------------------------------------------------------------
    // 各中間ファイルの内容を文字列ストリームに出力します。
    // （内容を出力しないと確定できないヘッダの情報があるため）
    bool ret = false;
    std::ostringstream contentOs;
    Dcc::RInitOutStreamFormat(contentOs);
    Dcc::RDataStreamArray dataStreams;
    switch (fileType)
    {
    case Dcc::RExpOpt::FMD:
        ret = OutputFmdFile(contentOs, dataStreams);
        break;
    case Dcc::RExpOpt::FSK:
        ret = OutputFskFile(contentOs, dataStreams);
        break;
    case Dcc::RExpOpt::FVB:
        ret = OutputFvbFile(contentOs, dataStreams, rOpt);
        break;
    case Dcc::RExpOpt::FCL:
    case Dcc::RExpOpt::FTS:
    case Dcc::RExpOpt::FTP:
        ret = OutputOneMatAnimFile(contentOs, dataStreams, rOpt, fileType);
        break;
    case Dcc::RExpOpt::FSH:
        ret = OutputFshFile(contentOs, dataStreams, rOpt);
        break;
    case Dcc::RExpOpt::FSN:
        ret = OutputFsnFile(contentOs, dataStreams, rOpt);
        break;
    }

    //-----------------------------------------------------------------------------
    // ヘッダを出力します。
    OutputHeader(os, 0, fileType, rOpt);

    //-----------------------------------------------------------------------------
    // 各中間ファイルの内容を出力します。
    os << contentOs.str();

    //-----------------------------------------------------------------------------
    // アスキー形式の場合、ここでデータ列配列を出力します。
    if (ret && !isBinary)
    {
        ROutDataStreams(os, dataStreams);
    }

    //-----------------------------------------------------------------------------
    // フッタを出力します。
    if (ret)
    {
        OutputFooter(os, 0, fileType, rOpt);
    }

    //-----------------------------------------------------------------------------
    // バイナリ形式の場合、ここでデータ列配列を出力します。
    if (ret && isBinary)
    {
        ROutBinaryDataStreams(os, dataStreams, true);
    }

    //-----------------------------------------------------------------------------
    // ファイルを閉じます。
    ofs.close();

    //-----------------------------------------------------------------------------
    // プロファイル情報を設定します。
    const clock_t clocks = clock() - startClock;
    if (fileType == Dcc::RExpOpt::FMD)
    {
        m_ModelClocks += clocks;
    }
    else if (Dcc::RExpOpt::FSK <= fileType && fileType <= Dcc::RExpOpt::FSN)
    {
        m_AnimClocks += clocks;
    }

    return ret;
}



void NintendoExport::OutputSkeletalModelTag(ostream& os, std::string name)
{
    int ts = 3;

//	int neededBonesNum = RShape::GetStandardMtxPalSize(exportOptions.nonUniformScale == TRUE, exportOptions.reservedUniform);
    int neededBonesNum  = 0;

    //-----------------------------------------------------------------------------
    // begin model
    os << Dcc::RTab(ts) << "<SkeletalModel"
       << " Name=\"" << Dcc::RGetUtf8FromShiftJis(name) << "\""
       << " IsBranchVisible=\"" << Dcc::RBoolStr(true) << "\""
       << " IsVisible=\"" << Dcc::RBoolStr(true) << "\""
       << " CullingMode=\"" << "Dynamic" << "\"";

    os << " LayerId=\"0\" NeededBoneCapacity=\"" << neededBonesNum << "\""
       //<< " IsTranslateAnimationEnabled=\"" << Dcc::RBoolStr(true) << "\""
       << ">" << R_ENDL;
#if 0
    {
        // AnimationGroup
        os	<<	Dcc::RTab(ts+1) << "<AnimationGroupDescriptions>" << R_ENDL
                <<	Dcc::RTab(ts+2) << "<GraphicsAnimationGroupDescription Name=\"SkeletalAnimation\" EvaluationTiming=\"BeforeWorldUpdate\">" << R_ENDL
                    <<	Dcc::RTab(ts+3) << "<MemberInformationSet>" << R_ENDL
                        <<	Dcc::RTab(ts+4) << "<AnimationMemberDescription BlendOperationName=\"CalculatedTransform\">" << R_ENDL
                            <<	Dcc::RTab(ts+5) << "<Path>Skeleton.Bones[\"*\"].AnimatedTransform</Path>" << R_ENDL
                        <<	Dcc::RTab(ts+4) << "</AnimationMemberDescription>" << R_ENDL
                    <<	Dcc::RTab(ts+3) << "</MemberInformationSet>" << R_ENDL
                <<	Dcc::RTab(ts+2) << "</GraphicsAnimationGroupDescription>" << R_ENDL
                <<	Dcc::RTab(ts+2) << "<GraphicsAnimationGroupDescription Name=\"MaterialAnimation\" EvaluationTiming=\"AfterSceneCulling\">" << R_ENDL
                    <<	Dcc::RTab(ts+3) << "<MemberInformationSet>" << R_ENDL
                        <<	Dcc::RTab(ts+4) << "<AnimationMemberDescription BlendOperationName=\"RgbaColor\">" << R_ENDL
                            <<	Dcc::RTab(ts+5) << "<Path>Materials[\"*\"].MaterialColor.Emission</Path>" << R_ENDL
                        <<	Dcc::RTab(ts+4) << "</AnimationMemberDescription>" << R_ENDL
                        <<	Dcc::RTab(ts+4) << "<AnimationMemberDescription BlendOperationName=\"RgbaColor\">" << R_ENDL
                            <<	Dcc::RTab(ts+5) << "<Path>Materials[\"*\"].MaterialColor.Ambient</Path>" << R_ENDL
                        <<	Dcc::RTab(ts+4) << "</AnimationMemberDescription>" << R_ENDL
                        <<	Dcc::RTab(ts+4) << "<AnimationMemberDescription BlendOperationName=\"RgbaColor\">" << R_ENDL
                            <<	Dcc::RTab(ts+5) << "<Path>Materials[\"*\"].MaterialColor.Diffuse</Path>" << R_ENDL
                        <<	Dcc::RTab(ts+4) << "</AnimationMemberDescription>" << R_ENDL
                        <<	Dcc::RTab(ts+4) << "<AnimationMemberDescription BlendOperationName=\"RgbaColor\">" << R_ENDL
                            <<	Dcc::RTab(ts+5) << "<Path>Materials[\"*\"].MaterialColor.Specular0</Path>" << R_ENDL
                        <<	Dcc::RTab(ts+4) << "</AnimationMemberDescription>" << R_ENDL
                        <<	Dcc::RTab(ts+4) << "<AnimationMemberDescription BlendOperationName=\"RgbaColor\">" << R_ENDL
                            <<	Dcc::RTab(ts+5) << "<Path>Materials[\"*\"].MaterialColor.Specular1</Path>" << R_ENDL
                        <<	Dcc::RTab(ts+4) << "</AnimationMemberDescription>" << R_ENDL
                        <<	Dcc::RTab(ts+4) << "<AnimationMemberDescription BlendOperationName=\"RgbaColor\">" << R_ENDL
                            <<	Dcc::RTab(ts+5) << "<Path>Materials[\"*\"].MaterialColor.Constant0</Path>" << R_ENDL
                        <<	Dcc::RTab(ts+4) << "</AnimationMemberDescription>" << R_ENDL
                        <<	Dcc::RTab(ts+4) << "<AnimationMemberDescription BlendOperationName=\"RgbaColor\">" << R_ENDL
                            <<	Dcc::RTab(ts+5) << "<Path>Materials[\"*\"].MaterialColor.Constant1</Path>" << R_ENDL
                        <<	Dcc::RTab(ts+4) << "</AnimationMemberDescription>" << R_ENDL
                        <<	Dcc::RTab(ts+4) << "<AnimationMemberDescription BlendOperationName=\"RgbaColor\">" << R_ENDL
                            <<	Dcc::RTab(ts+5) << "<Path>Materials[\"*\"].MaterialColor.Constant2</Path>" << R_ENDL
                        <<	Dcc::RTab(ts+4) << "</AnimationMemberDescription>" << R_ENDL
                        <<	Dcc::RTab(ts+4) << "<AnimationMemberDescription BlendOperationName=\"RgbaColor\">" << R_ENDL
                            <<	Dcc::RTab(ts+5) << "<Path>Materials[\"*\"].MaterialColor.Constant3</Path>" << R_ENDL
                        <<	Dcc::RTab(ts+4) << "</AnimationMemberDescription>" << R_ENDL
                        <<	Dcc::RTab(ts+4) << "<AnimationMemberDescription BlendOperationName=\"RgbaColor\">" << R_ENDL
                            <<	Dcc::RTab(ts+5) << "<Path>Materials[\"*\"].MaterialColor.Constant4</Path>" << R_ENDL
                        <<	Dcc::RTab(ts+4) << "</AnimationMemberDescription>" << R_ENDL
                        <<	Dcc::RTab(ts+4) << "<AnimationMemberDescription BlendOperationName=\"RgbaColor\">" << R_ENDL
                            <<	Dcc::RTab(ts+5) << "<Path>Materials[\"*\"].MaterialColor.Constant5</Path>" << R_ENDL
                        <<	Dcc::RTab(ts+4) << "</AnimationMemberDescription>" << R_ENDL
                        <<	Dcc::RTab(ts+4) << "<AnimationMemberDescription BlendOperationName=\"RgbaColor\">" << R_ENDL
                            <<	Dcc::RTab(ts+5) << "<Path>Materials[\"*\"].TextureMappers[\"*\"].Sampler.BorderColor</Path>" << R_ENDL
                        <<	Dcc::RTab(ts+4) << "</AnimationMemberDescription>" << R_ENDL
                        <<	Dcc::RTab(ts+4) << "<AnimationMemberDescription BlendOperationName=\"Int\">" << R_ENDL
                            <<	Dcc::RTab(ts+5) << "<Path>Materials[\"*\"].TextureMappers[\"*\"].Texture</Path>" << R_ENDL
                        <<	Dcc::RTab(ts+4) << "</AnimationMemberDescription>" << R_ENDL
                        <<	Dcc::RTab(ts+4) << "<AnimationMemberDescription BlendOperationName=\"RgbaColor\">" << R_ENDL
                            <<	Dcc::RTab(ts+5) << "<Path>Materials[\"*\"].FragmentOperation.BlendOperation.BlendColor</Path>" << R_ENDL
                        <<	Dcc::RTab(ts+4) << "</AnimationMemberDescription>" << R_ENDL
                        <<	Dcc::RTab(ts+4) << "<AnimationMemberDescription BlendOperationName=\"Vector2\">" << R_ENDL
                            <<	Dcc::RTab(ts+5) << "<Path>Materials[\"*\"].TextureCoordinators[\"*\"].Scale</Path>" << R_ENDL
                        <<	Dcc::RTab(ts+4) << "</AnimationMemberDescription>" << R_ENDL
                        <<	Dcc::RTab(ts+4) << "<AnimationMemberDescription BlendOperationName=\"Float\">" << R_ENDL
                            <<	Dcc::RTab(ts+5) << "<Path>Materials[\"*\"].TextureCoordinators[\"*\"].Rotate</Path>" << R_ENDL
                        <<	Dcc::RTab(ts+4) << "</AnimationMemberDescription>" << R_ENDL
                        <<	Dcc::RTab(ts+4) << "<AnimationMemberDescription BlendOperationName=\"Vector2\">" << R_ENDL
                            <<	Dcc::RTab(ts+5) << "<Path>Materials[\"*\"].TextureCoordinators[\"*\"].Translate</Path>" << R_ENDL
                        <<	Dcc::RTab(ts+4) << "</AnimationMemberDescription>" << R_ENDL
                    <<	Dcc::RTab(ts+3) << "</MemberInformationSet>" << R_ENDL
                <<	Dcc::RTab(ts+2) << "</GraphicsAnimationGroupDescription>" << R_ENDL
                <<	Dcc::RTab(ts+2) << "<GraphicsAnimationGroupDescription Name=\"VisibilityAnimation\" EvaluationTiming=\"BeforeWorldUpdate\">" << R_ENDL
                    <<	Dcc::RTab(ts+3) << "<MemberInformationSet>" << R_ENDL
                        <<	Dcc::RTab(ts+4) << "<AnimationMemberDescription BlendOperationName=\"Bool\">" << R_ENDL
                            <<	Dcc::RTab(ts+5) << "<Path>IsVisible</Path>" << R_ENDL
                        <<	Dcc::RTab(ts+4) << "</AnimationMemberDescription>" << R_ENDL
                        <<	Dcc::RTab(ts+4) << "<AnimationMemberDescription BlendOperationName=\"Bool\">" << R_ENDL
                            <<	Dcc::RTab(ts+5) << "<Path>Meshes[\"*\"].IsVisible</Path>" << R_ENDL
                        <<	Dcc::RTab(ts+4) << "</AnimationMemberDescription>" << R_ENDL
                        <<	Dcc::RTab(ts+4) << "<AnimationMemberDescription BlendOperationName=\"Bool\">" << R_ENDL
                            <<	Dcc::RTab(ts+5) << "<Path>MeshNodeVisibilities[\"*\"].IsVisible</Path>" << R_ENDL
                        <<	Dcc::RTab(ts+4) << "</AnimationMemberDescription>" << R_ENDL
                    <<	Dcc::RTab(ts+3) << "</MemberInformationSet>" << R_ENDL
                <<	Dcc::RTab(ts+2) << "</GraphicsAnimationGroupDescription>" << R_ENDL
            <<	Dcc::RTab(ts+1) << "</AnimationGroupDescriptions>" << R_ENDL;
        // transform
        os << Dcc::RTab(ts+1) << "<Transform>" << R_ENDL;
        {
            os << Dcc::RTab(ts+2) << "<Scale X=\"1\" Y=\"1\" Z=\"1\" />" << R_ENDL;
            os << Dcc::RTab(ts+2) << "<Rotate X=\"0\" Y=\"0\" Z=\"0\" />" << R_ENDL;
            os << Dcc::RTab(ts+2) << "<Translate X=\"0\" Y=\"0\" Z=\"0\" />" << R_ENDL;
        }
        os << Dcc::RTab(ts+1) << "</Transform>" << R_ENDL;
    }
#endif
}

void NintendoExport::OutputMaterialAnimTag(ostream& osMat, CStr loop, int start, int end, int length)
{
    osMat	<< Dcc::RTab(2) << "<Animations>" << R_ENDL;
        osMat <<  Dcc::RTab(3) << "<MaterialAnimationData Name=\"\" FrameSize=\"" << length
            << "\" LoopMode=\"" << loop
                << "\" IsAdditiveAnimation=\"false\">" << R_ENDL;
        // edit data
        {
            //const char* const adjustSkinningStrs[] = { "None", "RigidSkinning" };
            //const char* const meshVisibilityModeStrs[] = { "BindByIndex", "BindByName" };
            osMat	<< Dcc::RTab(4) << "<EditData>" << R_ENDL;
                osMat	<< Dcc::RTab(5) << "<GenericMetaData>" << R_ENDL;
                    osMat	<< Dcc::RTab(6) << "<Key>DccToolExportOption</Key>" << R_ENDL;
                    osMat	<< Dcc::RTab(6) << "<MaterialAnimationDccToolExportOption"
                            << " StartFrame=\"" << start
                            << "\" EndFrame=\"" << end
                            << "\" Magnify=\"" <<exportOptions.m_Opt.m_Magnify
                            << "\">" << R_ENDL;

                        osMat	<< Dcc::RTab(7) << "<MaterialAnimationBakeOption"
                                << " IsBakeAllEnabled=\"true"
                                << "\" FramePrecision=\"_1_" << exportOptions.m_Opt.m_FramePrecision
                                << "\" TextureScaleTolerance=\"" << exportOptions.m_Opt.m_TolS
                                << "\" TextureRotateTolerance=\"" << exportOptions.m_Opt.m_TolR * Dcc::R_M_DEG_TO_RAD
                                << "\" TextureTranslateTolerance=\"" << exportOptions.m_Opt.m_TolT
                                << "\" ColorTolerance=\"" << exportOptions.m_Opt.m_TolC
                                << "\" />" << R_ENDL;

                    osMat	<< Dcc::RTab(6) << "</MaterialAnimationDccToolExportOption>" << R_ENDL;
                osMat	<< Dcc::RTab(5) << "</GenericMetaData>" << R_ENDL;
            osMat	<< Dcc::RTab(4) << "</EditData>" << R_ENDL;
        }
}

bool NintendoExport::loadConfigFile( void )
{
    IPathConfigMgr* iPath = IPathConfigMgr::GetPathConfigMgr();
    const MCHAR* oldConfigFolderPath = iPath->GetDir(APP_PLUGCFG_DIR);
    const std::string oldConfigPath = Dcc::RGetUnixFilePath(std::string(M_2_A(oldConfigFolderPath))) + "/NW4F_Config.ini";

    const std::string configPath = 	GetNw4fConfigPath() + "Export.ini";
    const std::string sampleConfigPath = 	GetNw4fConfigPath() + "Export_Sample.ini";

    // コンフィグファイルがなければコピーします。
    if (!Dcc::RFileExists(configPath))
    {
        if (Dcc::RFileExists(oldConfigPath))	// 旧バージョンで保存されたパスからコピーします。
        {
            CopyFileA(oldConfigPath.c_str(), configPath.c_str(), FALSE);
        }
        else if (Dcc::RFileExists(sampleConfigPath)) // サンプルファイルからコピー
        {
            CopyFileA(sampleConfigPath.c_str(), configPath.c_str(), FALSE);
        }
    }

    if (!Dcc::RFileExists(configPath))
    {
        ofstream ofs;
        ofs.open(configPath.c_str(), ios_base::out | ios_base::binary);
        if(ofs.is_open())
        {
            ostream &os = ofs;
            os << "# NW4F_Export configuration file" << R_ENDL;
            os << "display_profile=\"false\"" << R_ENDL;
            os << "project_root=\"\"" << R_ENDL;
            os << "max_vertex_skinning_count=\"4\"" << R_ENDL;
            os << "adjust_smooth_skinning=\"false\"" << R_ENDL;
            os << "compress_ignoring_vertex_skinning_count = \"false\"" << R_ENDL;
            os << "optimize_primitive=\"forsyth\"" << R_ENDL;
            os << "quantization_analysis=\"true\"" << R_ENDL;
            os << "texture_albedo    =\"1110\"" << R_ENDL;
            os << "texture_emission  =\"1110\"" << R_ENDL;
            os << "texture_specular  =\"1110\"" << R_ENDL;
            os << "texture_reflection=\"1110\"" << R_ENDL;
            os << "use_srgb_fetch=\"true\"" << R_ENDL;
            os << "enable_weighted_compress = \"false\"" << R_ENDL;
            os << "filter_mip=\"linear\"" << R_ENDL;
            os << "use_fcl_fts_ftp=\"true\"" << R_ENDL;
            os << "separate_material_anim=\"false\"" << R_ENDL;
            os << "bone_visibility_merge_priority = \"false\"" << R_ENDL;
            os << "export_bone_visibility_all=\"true\"" << R_ENDL;
            os << "warning_node_name_changed=\"true\"" << R_ENDL;
            ofs.close();
        }
    }

    if (Dcc::RFileExists(configPath))
    {
        Dcc::RStatus rstatus = exportOptions.m_Opt.ParseConfigFile(configPath);

        if(!rstatus)
        {
            RLogger::LogMessage(rstatus.GetMessage(), RLogger::kWarning);
        }

        return rstatus;
    }

    return false;
}

void NintendoExport::DisplayProfileInfo( std::string name, const clock_t clocks )
{
    const float sec = static_cast<float>(clocks) / CLOCKS_PER_SEC;
    const std::string msg = "PROFILE: " + name + ": " +
        Dcc::RGetNumberString(Dcc::RRound(sec * 1000.0f), "%6d") + "ms";
    RLogger::LogMessage(msg, RLogger::kDump);
}


//-----------------------------------------------------------------------------
//! @brief 出力前後のスクリプトにエクスポートオプションの値を伝達するための環境変数を設定します。
//!
static void MySetEnvironmentVariable(string envname, const char* lpValue, bool outOld = true)
{
    if (envname.size() == 0)
    {
        return;
    }
    auto value = lpValue;
    auto oldname=  envname;


    if(lpValue != nullptr)
    {
        if(*lpValue == 0) // 空文字のときはnullptrにする
        {
            value = nullptr;
        }
    }
    ::SetEnvironmentVariableA(envname.c_str(), value);

    // NW4F_ 形式の環境変数も出力
    if (outOld)
    {
        const static string NINTENDO_PREFIX = "NINTENDO_";
        if (envname.find(NINTENDO_PREFIX) == 0)
        {
            oldname = "NW4F_";
            oldname.append(envname.substr(NINTENDO_PREFIX.size()));
            ::SetEnvironmentVariableA(oldname.c_str(), value);
        }
    }
}

//-----------------------------------------------------------------------------
void NintendoExport::SetOptionEnvVarForPrePostExportScript( bool isPost, bool exportSucceeded )
{

    //nw4fOptions
    Dcc::RExpOpt& rOpt = exportOptions.m_Opt;

    //-----------------------------------------------------------------------------
    // 出力ファイル名（拡張子なし）、出力フォルダ、テクスチャ出力フォルダです。
    std::string cmd;
    MySetEnvironmentVariable("NINTENDO_EXPORT_OUTPUT_FILE_NAME", rOpt.m_OutFileName.c_str());
    MySetEnvironmentVariable("NINTENDO_EXPORT_OUTPUT_FOLDER", rOpt.m_OutFolderPath.c_str());
    MySetEnvironmentVariable("NINTENDO_EXPORT_FTX_FOLDER", rOpt.m_TexFolderPath.c_str());

    //-----------------------------------------------------------------------------
    // 各中間ファイルのフルパスです（出力しない場合は空文字）。
    const std::string fullFilePathTop = rOpt.m_OutFolderPath + rOpt.m_OutFileName + ".";
    for (int fileType = Dcc::RExpOpt::FMD; fileType < Dcc::RExpOpt::FILE_TYPE_COUNT; ++fileType)
    {
        const std::string ext = rOpt.GetExtension(static_cast<Dcc::RExpOpt::FileType>(fileType));
        std::string env = "NINTENDO_EXPORT_";
        std::string val = "";

        env += Dcc::RGetUpperCaseString(ext.substr(0, 3));
        env += "_FILE_PATH";
        if (rOpt.m_OutFileFlag[fileType])
        {
            val = fullFilePathTop + ext;
        }
        MySetEnvironmentVariable(env.c_str(), val.c_str());
    }

    //-----------------------------------------------------------------------------
    // ftx ファイルの出力指定が ON かどうかです。
    MySetEnvironmentVariable("NINTENDO_EXPORT_FTX_ENABLED", Dcc::RGetNumberString(rOpt.ExportsTexture() ? 1 : 0).c_str());

    //-----------------------------------------------------------------------------
    // 3dTools のコマンドラインツールフォルダです。
    MySetEnvironmentVariable("NINTENDO_EXPORT_G3D_TOOL_FOLDER", rOpt.m_G3dToolPath.c_str());

    //-----------------------------------------------------------------------------
    // 出力に成功したかどうかです（出力後のスクリプト用）。
    {
        std::string val = "";
        if (isPost)
        {
            val = Dcc::RGetNumberString(exportSucceeded ? 1 : 0);
        }
        MySetEnvironmentVariable("NINTENDO_EXPORT_SUCCEEDED", val.c_str());
    }

    //-----------------------------------------------------------------------------
    // 出力した ftx ファイル名をスペースで区切って並べた文字列です。
    // （出力後のスクリプト用）（出力しない場合は空文字）
    std::string ftxNamesStr;
    if (isPost)
    {
        const std::string ext = rOpt.GetExtension(Dcc::RExpOpt::FTX);
        const int imgCount = static_cast<int>(m_pOutTexImgs.size());
        for (int iTexImg = 0; iTexImg < imgCount; ++iTexImg)
        {
            if (iTexImg > 0)
            {
                ftxNamesStr += " ";
            }
            const std::string ftxName =
                m_pOutTexImgs[iTexImg]->GetName() + "." + ext;
            ftxNamesStr += ftxName;
        }
    }

    {
        std::string val = "";
        if (rOpt.ExportsTexture())
        {
            val = ftxNamesStr;
        }
        MySetEnvironmentVariable("NINTENDO_EXPORT_FTX_FILE_NAMES", val.c_str());
    }

    MySetEnvironmentVariable("NINTENDO_EXPORT_USED_FTX_FILE_NAMES", ftxNamesStr.c_str());


    //-----------------------------------------------------------------------------
    // header
    MySetEnvironmentVariable("NINTENDO_EXPORT_OPTION_tool_name"
        , std::string(std::string(GetMaxVersionString())).c_str());
    MySetEnvironmentVariable("NINTENDO_EXPORT_OPTION_tool_version", GetPluginVersionString());

    //-----------------------------------------------------------------------------
    // preset
    MySetEnvironmentVariable("NINTENDO_EXPORT_OPTION_preset", rOpt.m_PresetName.c_str());

    //-----------------------------------------------------------------------------
    // output
    MySetEnvironmentVariable("NINTENDO_EXPORT_OPTION_export_target", (rOpt.m_Target == Dcc::RExpOpt::EXPORT_TARGET_ALL) ? "all" : "selection");

    MySetEnvironmentVariable("NINTENDO_EXPORT_OPTION_merge_fmd", Dcc::RBoolStr(rOpt.m_MergeFmdFlag));
    MySetEnvironmentVariable("NINTENDO_EXPORT_OPTION_merge_fmd_path", rOpt.m_MergeFmdPath.c_str());

    MySetEnvironmentVariable("NINTENDO_EXPORT_OPTION_merge_ftx", Dcc::RBoolStr(rOpt.m_MergeFtxFlag));

    MySetEnvironmentVariable("NINTENDO_EXPORT_OPTION_merge_anim", Dcc::RBoolStr(rOpt.m_MergeAnimFlag));
    MySetEnvironmentVariable("NINTENDO_EXPORT_OPTION_merge_anim_folder", rOpt.m_MergeAnimFolder.c_str());
    MySetEnvironmentVariable("NINTENDO_EXPORT_OPTION_merge_anim_name", rOpt.m_MergeAnimName.c_str());

    //-----------------------------------------------------------------------------
    // general
    MySetEnvironmentVariable("NINTENDO_EXPORT_OPTION_magnify", Dcc::RGetFloatStringForSettingFile(rOpt.m_Magnify).c_str());

    static const char* const texSrtModeStrs[] = { "maya", "3dsmax", "softimage" };
    MySetEnvironmentVariable("NINTENDO_EXPORT_OPTION_texsrt_mode", texSrtModeStrs[rOpt.m_TexSrtMode]);

    //SetEnvironmentVariableA("NW4F_EXPORT_OPTION_remove_namespace", Dcc::RBoolStr(rOpt.m_RemoveNamespace));

    MySetEnvironmentVariable("NINTENDO_EXPORT_OPTION_comment", rOpt.m_CommentText.c_str());

    //-----------------------------------------------------------------------------
    // animation
    static const char* const frameRangeStrs[] = { "all", "playback", "range" };
    MySetEnvironmentVariable("NINTENDO_EXPORT_OPTION_frame_range", frameRangeStrs[rOpt.m_FrameRange]);
    MySetEnvironmentVariable("NINTENDO_EXPORT_OPTION_start_frame", Dcc::RGetNumberString(rOpt.m_StartFrame).c_str());
    MySetEnvironmentVariable("NINTENDO_EXPORT_OPTION_end_frame", Dcc::RGetNumberString(rOpt.m_EndFrame).c_str());
    MySetEnvironmentVariable("NINTENDO_EXPORT_OPTION_loop_anim", Dcc::RBoolStr(rOpt.m_LoopAnim));
    MySetEnvironmentVariable("NINTENDO_EXPORT_OPTION_bake_all_anim", Dcc::RBoolStr(rOpt.m_BakeAllAnim));
    const float frameStep = (rOpt.m_FramePrecision != 0) ? 1.0f / rOpt.m_FramePrecision : 1.0f;
    MySetEnvironmentVariable("NINTENDO_EXPORT_OPTION_frame_precision", Dcc::RGetFloatStringForSettingFile(frameStep).c_str());

    //-----------------------------------------------------------------------------
    // optimization
    static const char* const compressBoneStrs[] = { "none", "cull", "merge", "unite", "unite_all" };
    MySetEnvironmentVariable("NINTENDO_EXPORT_OPTION_compress_bone", compressBoneStrs[rOpt.m_CompressBone]);
    MySetEnvironmentVariable("NINTENDO_EXPORT_OPTION_unite_child", Dcc::RBoolStr(rOpt.m_UniteChild));
    MySetEnvironmentVariable("NINTENDO_EXPORT_OPTION_compress_material", Dcc::RBoolStr(rOpt.m_CompressMaterial));
    MySetEnvironmentVariable("NINTENDO_EXPORT_OPTION_compress_shape", Dcc::RBoolStr(rOpt.m_CompressShape));

    //-----------------------------------------------------------------------------
    // animation bake tolerance
    MySetEnvironmentVariable("NINTENDO_EXPORT_OPTION_tolerance_scale", Dcc::RGetFloatStringForSettingFile(rOpt.m_TolS).c_str());
    MySetEnvironmentVariable("NINTENDO_EXPORT_OPTION_tolerance_rotate", Dcc::RGetFloatStringForSettingFile(rOpt.m_TolR).c_str());
    MySetEnvironmentVariable("NINTENDO_EXPORT_OPTION_tolerance_translate", Dcc::RGetFloatStringForSettingFile(rOpt.m_TolT).c_str());
    MySetEnvironmentVariable("NINTENDO_EXPORT_OPTION_tolerance_color", Dcc::RGetFloatStringForSettingFile(rOpt.m_TolC).c_str());
    MySetEnvironmentVariable("NINTENDO_EXPORT_OPTION_tolerance_tex_scale", Dcc::RGetFloatStringForSettingFile(rOpt.m_TolTexS).c_str());
    MySetEnvironmentVariable("NINTENDO_EXPORT_OPTION_tolerance_tex_rotate", Dcc::RGetFloatStringForSettingFile(rOpt.m_TolTexR).c_str());
    MySetEnvironmentVariable("NINTENDO_EXPORT_OPTION_tolerance_tex_translate", Dcc::RGetFloatStringForSettingFile(rOpt.m_TolTexT).c_str());

    //-----------------------------------------------------------------------------
    // animation quantization tolerance
    MySetEnvironmentVariable("NINTENDO_EXPORT_OPTION_quantize_tolerance_scale", Dcc::RGetFloatStringForSettingFile(rOpt.m_QuantTolS).c_str());
    MySetEnvironmentVariable("NINTENDO_EXPORT_OPTION_quantize_tolerance_rotate", Dcc::RGetFloatStringForSettingFile(rOpt.m_QuantTolR).c_str());
    MySetEnvironmentVariable("NINTENDO_EXPORT_OPTION_quantize_tolerance_translate", Dcc::RGetFloatStringForSettingFile(rOpt.m_QuantTolT).c_str());
    MySetEnvironmentVariable("NINTENDO_EXPORT_OPTION_quantize_tolerance_tex_scale", Dcc::RGetFloatStringForSettingFile(rOpt.m_QuantTolTexS).c_str());
    MySetEnvironmentVariable("NINTENDO_EXPORT_OPTION_quantize_tolerance_tex_rotate", Dcc::RGetFloatStringForSettingFile(rOpt.m_QuantTolTexR).c_str());
    MySetEnvironmentVariable("NINTENDO_EXPORT_OPTION_quantize_tolerance_tex_translate", Dcc::RGetFloatStringForSettingFile(rOpt.m_QuantTolTexT).c_str());
}

void NintendoExport::Init()
{
    exportOptions.InitConfigOptions();
}

bool NintendoExport::OutputFsnFile(ostream& contentOs, Dcc::RDataStreamArray& dataStreams, const Dcc::RExpOpt& rOpt )
{
    // カメラ
    m_SceneCameras.OutAnims(contentOs, dataStreams, rOpt);

    // ライト
    m_SceneLights.OutAnims(contentOs, dataStreams, rOpt);

    return true;
}

bool NintendoExport::OutputFvbFile(ostream& contentOs, Dcc::RDataStreamArray& dataStreams, const Dcc::RExpOpt& rOpt)
{
    m_NodeProcessor.outVisibilityAnimationData( contentOs, dataStreams, rOpt );
    return true;
}

bool NintendoExport::OutputFshFile(ostream& contentOs, Dcc::RDataStreamArray& dataStreams, const Dcc::RExpOpt& rOpt)
{
    m_NodeProcessor.outShapeAnimationData(contentOs, dataStreams, rOpt);
    return true;
}

const char* NintendoExport::GetMaxVersionString()
{
    char* max_version_platform;
#ifndef WIN64
    max_version_platform = "x86";
#else
    max_version_platform = "x64";
#endif
    static char* verString = nullptr;
    if(verString == nullptr)
    {
        verString = new char[1024];
        sprintf( verString, "NintendoExport for 3ds Max %d %s", MAX_PRODUCT_YEAR_NUMBER, max_version_platform );
    }
    return verString;
}

const char* NintendoExport::GetPluginVersionString()
{
    static char* verString = nullptr;
    if(verString == nullptr)
    {
        verString = new char[128];
        //char verBuf[10];
        //sprintf_s(verBuf, "%.3d", this->Version());
        //sprintf_s(verString, 128, "%c.%c.%c", verBuf[0], verBuf[1], verBuf[2]);
        //char buf[64];
        sprintf_s(verString, 128, "%d.%d.%d", NINTENDOEXPORT_DISPVERSION_MAJOR, NINTENDOEXPORT_DISPVERSION_MINOR, NINTENDOEXPORT_DISPVERSION_PATCH);
    }
    return verString;
}

void NintendoExport::FinishExport(bool succeed)
{
    if(!m_TmpOutFolderPath.empty())
    {
        DeleteDirectoryDeep(m_TmpOutFolderPath.c_str());
    }
    SetOptionEnvVarForPrePostExportScript(true, succeed);
    std::string& script = exportOptions.m_Opt.m_PostExpScript;
    if(script.size() > 0)
    {
        char* buf = new char[script.size()+2];
        strcpy_s(buf, script.size()+2, script.c_str());
        if(!ExecuteMAXScriptScript(A_2_M(buf)))
        {
            RLogger::LogMessage("Post-Export Script failed.", RLogger::kError);
        }
    }
}


//-----------------------------------------------------------------------------
// ユーザーデータを文字列から設定する
bool SetUserDataFromString(std::string str, FMaterial* fmat, RNode* rnode)
{
    UINT wsize = ::MultiByteToWideChar(CP_ACP, 0, str.c_str(), -1, nullptr, 0);
    if(wsize <= 2) return false;
    WCHAR* wstr;
    wstr = new WCHAR[wsize];
    ::MultiByteToWideChar(CP_ACP, 0, str.c_str(), -1, wstr, wsize);

    UINT p = 0, sp, ep;
    while (p < wsize)
    {
        // スペースをスキップ
        while (wstr[p] == L' ' && (p < wsize)) p++;
        if (p >= wsize) break;
        sp = p;
        // データの始まり(名前の終わり)を探す
        while (wstr[sp] != L'\"' && (sp < wsize)) sp++;
        if (sp >= wsize) break;
        int namesize = sp - p;
        WCHAR* name = new WCHAR[namesize + 1]; // add null
        memcpy(name, &wstr[p] , sizeof(WCHAR) * namesize);
        name[namesize] = L'\0';

        // データの終わりを探す
        sp++;
        ep = sp;
        while (wstr[ep] != L'\"' && (ep < wsize)) ep++;
        if (ep >= wsize) break;
        int datasize = ep - sp; // add null
        WCHAR* data = new WCHAR[datasize + 1]; // add null
        memcpy(data, &wstr[sp] , sizeof(WCHAR) * datasize);
        data[datasize] = L'\0';

        Dcc::RUserData ud;
        wstring wdata = &data[2], wsub;
        ud.m_Name = Dcc::RGetUtf8FromShiftJis(Dcc::RGetShiftJisFromUnicode(name));
        size_t dsp = 0, dep = 0;
        switch(data[0])
        {
            case L'i':
                {
                    ud.m_Type = Dcc::RUserData::INT;
                    while(dep != string::npos)
                    {
                        dep = wdata.find(L" ", dsp);
                        if(dep == string::npos)
                        {
                            wsub = wdata.substr(dsp, string::npos);
                        }
                        else
                        {
                            wsub = wdata.substr(dsp, dep - dsp);
                            dsp = dep + 1;
                        }
                        ud.m_IntValues.push_back(_wtoi(wsub.c_str()));
                    }
                }
                break;

            case L'f':
                {
                    ud.m_Type = Dcc::RUserData::FLOAT;
                    while(dep != string::npos)
                    {
                        dep = wdata.find(L" ", dsp);
                        if(dep == string::npos)
                        {
                            wsub = wdata.substr(dsp, string::npos);
                        }
                        else
                        {
                            wsub = wdata.substr(dsp, dep - dsp);
                            dsp = dep + 1;
                        }
                        ud.m_FloatValues.push_back((float)_wtof(wsub.c_str()));
                    }
                }
                break;

            case L'w':
                {
                    ud.m_Type = Dcc::RUserData::WSTRING;
                    while(dep != string::npos)
                    {
                        dep = wdata.find(L"\n", dsp);
                        if(dep == string::npos)
                        {
                            wsub = wdata.substr(dsp, string::npos);
                        }
                        else
                        {
                            wsub = wdata.substr(dsp, dep - dsp);
                            dsp = dep + 1;
                        }
                        ud.m_StringValues.push_back(Dcc::RGetShiftJisFromUnicode(wsub.c_str()));
                    }
                }
                break;

            case L's':
            default:
                {
                    ud.m_Type = Dcc::RUserData::STRING;
                    while(dep != string::npos)
                    {
                        dep = wdata.find(L"\n", dsp);
                        if(dep == string::npos)
                        {
                            wsub = wdata.substr(dsp, string::npos);
                        }
                        else
                        {
                            wsub = wdata.substr(dsp, dep - dsp);
                            dsp = dep + 1;
                        }
                        ud.m_StringValues.push_back(Dcc::RGetShiftJisFromUnicode(wsub.c_str()));
                    }
                }
                break;

        }
        if(fmat)
        {
            fmat->addUserData(ud);
        }
        else if (rnode)
        {
            rnode->addUserData(ud);
        }

        p = ep + 1;
        delete[] name;
        delete[] data;
    }

    delete[] wstr;

    return true;
}

