﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/TargetConfigs/build_Base.h>
#include <nn/nn_SdkLog.h>
#include <nn/nn_SdkAssert.h>
#include <nn/util/util_Constant.h>
#include <nn/util/util_Arithmetic.h>
#include <nn/util/util_Vector.h>
#include <nn/util/util_Matrix.h>
#include <nn/util/util_Quaternion.h>

#include <nns/nac/nac_Pad.h>
#include <nns/nac/nac_Mouse.h>


//---------------------------------------------------------------------------
//! @brief  デモカメラ
//---------------------------------------------------------------------------
class Camera
{
    //---------------------------------------------------------
public:

    //---------------------------------------------------------------------------
    //! @brief  移動用のイベントです。
    //---------------------------------------------------------------------------
    class Event
    {
    public:
        //---------------------------------------------------------------------------
        //! @brief  フラグ
        //---------------------------------------------------------------------------
        enum FlagType
        {
            ROTATE_BASED_ON_TARGET   = 0x00000001, //!< ターゲットを基準に回転をおこなう
            ROTATE_VERTICAL_360      = 0x00000002, //!< 縦にも360度回転する
            EXPAND_BASED_ON_TARGET   = 0x00000004, //!< ターゲットを基準に伸縮をおこなう
            POSITION_RESET           = 0x00000008, //!< 位置のリセット
            MOVE_ALONG_AXIS          = 0x00000010, //!< 座標軸に沿って並行移動する             (デフォルトはXZ平面上を視線ベクトルに沿って、Y方向は軸に沿って移動)
            MOVE_ALONG_VIEW          = 0x00000020, //!< XYZ共に視線ベクトルに沿って並行移動する(デフォルトはXZ平面上を視線ベクトルに沿って、Y方向は軸に沿って移動)
            REVERSE_ON_VIEW_CAMERA   = 0x00000040, //!< ビューカメラの操作の場合に反転する
        };

        //---------------------------------------------------------------------------
        //! @brief  コンストラクタです。
        //---------------------------------------------------------------------------
        Event()
            : m_Flags( 0 )
            , m_RotateH( 0.0f )
            , m_RotateV( 0.0f )
            , m_Zoom( 0.0f )
            , m_Twist( 0.0f )
        {
            nn::util::VectorSet( &m_Move, 0.0f, 0.0f, 0.0f );
        }

        //---------------------------------------------------------------------------
        //! @brief        設定をクリアします。
        //---------------------------------------------------------------------------
        void Clear() { *this = Event(); }

        //---------------------------------------------------------------------------
        //! @brief        ズーム値を取得します。
        //! @return       ズーム値
        //---------------------------------------------------------------------------
        float Zoom() const { return m_Zoom; }
        //---------------------------------------------------------------------------
        //! @brief        ズーム値を設定します。
        //! @param[in]  v ズーム値
        //---------------------------------------------------------------------------
        void SetZoom( float v ) { m_Zoom = v; }

        //---------------------------------------------------------------------------
        //! @brief        水平方向の回転値を取得します。
        //! @return       水平方向の回転値
        //---------------------------------------------------------------------------
        float RotateH() const { return m_RotateH; }
        //---------------------------------------------------------------------------
        //! @brief        水平方向の回転値を設定します。
        //! @param[in]  v 水平方向の回転値
        //---------------------------------------------------------------------------
        void SetRotateH( float v ) { m_RotateH = v; }

        //---------------------------------------------------------------------------
        //! @brief        垂直方向の回転値を取得します。
        //! @return       垂直方向の回転値
        //---------------------------------------------------------------------------
        float RotateV() const { return m_RotateV; }
        //---------------------------------------------------------------------------
        //! @brief        垂直方向の回転値を設定します。
        //! @param[in]  v 垂直方向の回転値
        //---------------------------------------------------------------------------
        void SetRotateV( float v ) { m_RotateV = v; }

        //---------------------------------------------------------------------------
        //! @brief        ビュー方向を軸にした回転値を取得します。
        //! @return       ビュー方向を軸にした回転値
        //---------------------------------------------------------------------------
        float Twist() const { return m_Twist; }

        //---------------------------------------------------------------------------
        //! @brief        平行移動値を取得します。
        //! @return       平行移動値
        //---------------------------------------------------------------------------
        const nn::util::Vector3fType& Move() const { return m_Move; }
        //---------------------------------------------------------------------------
        //! @brief        平行移動値を設定します。
        //! @param[in]  v 平行移動値
        //---------------------------------------------------------------------------
        void SetMove( const nn::util::Vector3fType& v ) { m_Move = v; }

        //---------------------------------------------------------------------------
        //! @brief        移動イベントのオプションフラグを取得します。
        //! @return       移動イベントのオプションフラグ
        //---------------------------------------------------------------------------
        uint32_t Flags() const { return m_Flags; }
        //---------------------------------------------------------------------------
        //! @brief        移動イベントのオプションフラグを設定します。
        //! @param[in]  v 移動イベントのオプションフラグ
        //---------------------------------------------------------------------------
        void EnableFlags( uint32_t flags ) { m_Flags |= flags; }

        //---------------------------------------------------------------------------
        //! @brief        前方への移動値を取得します。
        //! @return       前方への移動値
        //---------------------------------------------------------------------------
        float MoveForward() const { return nn::util::VectorGetZ( m_Move ); }
        //---------------------------------------------------------------------------
        //! @brief        前方への移動値を設定します。
        //! @param[in]  v 前方への移動値
        //---------------------------------------------------------------------------
        void SetMoveForward( float v ) { nn::util::VectorSetZ( &m_Move, v ); }

        //---------------------------------------------------------------------------
        //! @brief        アップベクトル方向への移動値を取得します。
        //! @return       アップベクトル方向への移動値
        //---------------------------------------------------------------------------
        float MoveUp() const { return nn::util::VectorGetY( m_Move ); }
        //---------------------------------------------------------------------------
        //! @brief        アップベクトル方向への移動値を設定します。
        //! @param[in]  v アップベクトル方向への移動値
        //---------------------------------------------------------------------------
        void SetMoveUp( float v ) { nn::util::VectorSetY( &m_Move, v ); }

        //---------------------------------------------------------------------------
        //! @brief        ライトベクトル方向への移動値を取得します。
        //! @return       ライトベクトル方向への移動値
        //---------------------------------------------------------------------------
        float MoveRight() const { return nn::util::VectorGetX( m_Move ); }
        //---------------------------------------------------------------------------
        //! @brief        ライトベクトル方向への移動値を設定します。
        //! @param[in]  v ライトベクトル方向への移動値
        //---------------------------------------------------------------------------
        void SetMoveRight( float v ) { nn::util::VectorSetX( &m_Move, v ); }

    private:
        uint32_t  m_Flags;              //!< フラグ

        //-----------------
        // 回転オプション
        //   ROTATE_BASED_ON_TARGET     ：ターゲットを基準に回転 (デフォルトは首振り回転)
        //   ROTATE_VERTICAL_360        ：縦方向にも360度回転    (デフォルトは真上/真下まで)
        //   REVERSE_ON_VIEW_CAMERA     ：ビューカメラ操作の場合に反転
        float m_RotateH;                //!< 垂直回転係数 標準値[-1.0f, 1.0f]
        float m_RotateV;                //!< 水平回転係数 標準値[-1.0f, 1.0f]

        //-----------------
        // 移動オプション
        //   MOVE_ALONG_AXIS            ：座標軸に沿って並行移動する              (デフォルトはXZ平面上を視線ベクトルに沿って、Y方向は軸に沿って移動)
        //   MOVE_ALONG_VIEW            ：XYZ共に視線ベクトルに沿って並行移動する (デフォルトはXZ平面上を視線ベクトルに沿って、Y方向は軸に沿って移動)
        //   REVERSE_ON_VIEW_CAMERA     ：ビューカメラ操作の場合に反転
        //     (※MOVE_ALONG_AXISとMOVE_ALONG_VIEWフラグは共存不可)

        nn::util::Vector3fType m_Move;  //!< 移動量

        //-----------------
        // ズームオプション
        //   EXPAND_BASED_ON_TARGET     ：ターゲットを基準に伸縮 (デフォルトはターゲットが動く)
        float m_Zoom;                   //!< 伸縮係数 標準値[-1.0f, 1.0f]

        //-----------------
        // ツイストオプション
        float m_Twist;                  //!< ツイスト係数  標準値[-1.0f, 1.0f]
    };

    //---------------------------------------------------------------------------
    //! @brief        コンストラクタです。
    //---------------------------------------------------------------------------
    Camera();

    //---------------------------------------------------------------------------
    //! @brief        デストラクタです。
    //---------------------------------------------------------------------------
    ~Camera() {}

    //---------------------------------------------------------------------------
    //! @brief        カメラ更新。
    //---------------------------------------------------------------------------
    void UpdateCamera( nns::nac::Pad* pad, nns::nac::Mouse* mouse, bool isMouseAvailable, bool isAltModified );

    //---------------------------------------------------------------------------
    //! @brief       入力に従ってイベントの更新を行います。
    //---------------------------------------------------------------------------
    void MakeCameraEventClassic( Event* events, nns::nac::Pad* pad, nns::nac::Mouse* mouse, bool isMouseAvailable, bool isAltModified, float frameSpeed = 1.0f );


    //---------------------------------------------------------------------------
    //! @brief       他のLookAtObjectの設定をコピー
    //!
    //! @param[out]  dirObj コピー元の LookAtObjec です。
    //---------------------------------------------------------------------------
    void SetDirection( Camera* dirObj );

    //---------------------------------------------------------------------------
    //! @brief       オブジェクトの座標を設定します。
    //---------------------------------------------------------------------------
    void SetPos( float x, float y, float z );
    void SetPos( const nn::util::Vector3fType& vec );

    //---------------------------------------------------------------------------
    //! @brief       オブジェクトの座標を取得します。
    //---------------------------------------------------------------------------
    const nn::util::Vector3fType* GetPos() const;
    void GetPos( nn::util::Vector3fType* pos ) const;

    //---------------------------------------------------------------------------
    //! @brief       ターゲットの座標を設定します。
    //!
    //! @param[in]    x       設定する注視点のX座標
    //! @param[in]    y       設定する注視点のY座標
    //! @param[in]    z       設定する注視点のZ座標
    //---------------------------------------------------------------------------
    void SetLookAtPos( float x, float y, float z );

    //---------------------------------------------------------------------------
    //! @param[in]    vec     設定する注視点の座標
    //---------------------------------------------------------------------------
    void SetLookAtPos( const nn::util::Vector3fType& vec );

    //---------------------------------------------------------------------------
    //! @brief       ターゲットの座標を取得します。
    //!
    //! @return       注視点座標の const ポインタ
    //---------------------------------------------------------------------------
    const nn::util::Vector3fType*   GetLookAtPos()     const;

    //---------------------------------------------------------------------------
    //! @param[in]    pos     注視点の座標を取得するためのポインタ
    //---------------------------------------------------------------------------
    void GetLookAtPos( nn::util::Vector3fType* pos ) const;

    //---------------------------------------------------------------------------
    //! @brief       Up ベクトルを設定します。
    //!
    //! @param[in]    x       設定するUp ベクトルのX座標
    //! @param[in]    y       設定するUp ベクトルのY座標
    //! @param[in]    z       設定するUp ベクトルのZ座標
    //---------------------------------------------------------------------------
    void SetUpVec( float x, float y, float z );

    //---------------------------------------------------------------------------
    //! @param[in]    vec     設定するUp ベクトル
    //---------------------------------------------------------------------------
    void SetUpVec( const nn::util::Vector3fType& vec );

    //---------------------------------------------------------------------------
    //! @brief       Up ベクトルの取得
    //!
    //! @return       Up ベクトルを示す const ポインタ
    //---------------------------------------------------------------------------
    const nn::util::Vector3fType* GetUpVec() const;

    //---------------------------------------------------------------------------
    //! @param[in]    pos     Up ベクトルを取得するためのポインタ
    //---------------------------------------------------------------------------
    void GetUpVec( nn::util::Vector3fType* pos ) const;

    //---------------------------------------------------------------------------
    //! @brief       オブジェクトとターゲットの距離を取得します。
    //!
    //! @return       オブジェクト位置から注視点の距離を返します。
    //---------------------------------------------------------------------------
    float GetDistance () const;

    //---------------------------------------------------------------------------
    //! @brief       オブジェクトとターゲットの距離を設定します。
    //!
    //! @details     注視点を固定し、オブジェクトの位置を移動します。
    //!
    //! @param[in]    dist    オブジェクト位置から注視点の距離を設定します。
    //---------------------------------------------------------------------------
    void SetDistance ( float dist );

    //---------------------------------------------------------------------------
    //! @brief       オブジェクトをターゲットへ近づけます。
    //---------------------------------------------------------------------------
    void AddDistanceByMovingObj( float length );

    //---------------------------------------------------------------------------
    //! @brief       ターゲットをオブジェクトへ近づけます。
    //---------------------------------------------------------------------------
    void AddDistanceByMovingTarget( float length );

    //---------------------------------------------------------------------------
    //! @brief       距離の範囲を指定します。
    //---------------------------------------------------------------------------
    void SetDistanceRange( float min, float max ) { m_DistanceMin = min; m_DistanceMax = max; }

    //---------------------------------------------------------------------------
    //! @brief       距離の範囲の最大値を取得します。
    //---------------------------------------------------------------------------
    float  GetDistanceMax() const { return m_DistanceMax; }

    //---------------------------------------------------------------------------
    //! @brief       距離の範囲の最小値を取得します。
    //---------------------------------------------------------------------------
    float  GetDistanceMin() const { return m_DistanceMin; }

    //---------------------------------------------------------------------------
    //! @brief       垂直方向の角度を取得します。
    //---------------------------------------------------------------------------
    float  GetAngleV() const { RecalcIf_( RECALC_ANGLE ); return m_AngleV; }

    //---------------------------------------------------------------------------
    //! @brief       水平方向の角度を取得します。
    //---------------------------------------------------------------------------
    float  GetAngleH() const { RecalcIf_( RECALC_ANGLE ); return m_AngleH; }

    //---------------------------------------------------------------------------
    //! @brief       軸の回転角度を取得します。
    //---------------------------------------------------------------------------
    float  GetTwist() const { return m_Twist; }

    //---------------------------------------------------------------------------
    //! @brief       軸の回転角度を設定します。
    //---------------------------------------------------------------------------
    void SetTwist( float twist ) { m_Twist = twist; }

    //---------------------------------------------------------------------------
    //! @brief       LookAt オブジェクトを平行移動します。
    //!
    //! @param[in]    x       X 方向の移動距離です。
    //! @param[in]    y       Y 方向の移動距離です。
    //! @param[in]    z       Z 方向の移動距離です。
    //---------------------------------------------------------------------------
    void Move( float x, float y, float z );

    //---------------------------------------------------------------------------
    //! @brief       ビュー方向に移動します。
    //!
    //! @param[in]   right   ビューのライトベクトル方向への移動距離です。
    //! @param[in]   forward ビューの正面方向への移動距離です。
    //! @param[in]   up      ビューのアップベクトル方向への移動距離です。
    //! @param[in]   view    ビューカメラ位置を指定します。
    //---------------------------------------------------------------------------
    void MoveView( float right, float forward, float up, const Camera& view );

    //---------------------------------------------------------------------------
    //! @brief       ビューベクトルを XZ 平面に投影したベクトルを基準に平行移動します。
    //!
    //! @param[in]   right   ビューを XZ 平面に投影したライトベクトル方向への移動距離です。
    //! @param[in]   forward ビューを XZ 平面に投影した正面方向への移動距離です。
    //! @param[in]   view    ビューカメラ位置を指定します。
    //---------------------------------------------------------------------------
    void MoveViewHorizontal( float right, float forward, const Camera& view );

    //---------------------------------------------------------------------------
    //! @brief       オブジェクトを中心に回転します。
    //!
    //! @param[in]    angleV  垂直回転角度
    //! @param[in]    angleH  水平回転角度
    //---------------------------------------------------------------------------
    void Rotate( float angleV, float angleH );

    //---------------------------------------------------------------------------
    //! @brief       ターゲットを中心に回転します。
    //!
    //! @param[in]    angleV  垂直回転角度
    //! @param[in]    angleH  水平回転角度
    //---------------------------------------------------------------------------
    void Revolve( float angleV, float angleH );

    //---------------------------------------------------------------------------
    //! @brief       オブジェクトを中心に360度回転します。
    //!
    //! @param[in]    angleV  垂直回転角度
    //! @param[in]    angleH  水平回転角度
    //---------------------------------------------------------------------------
    void Rotate360( float angleV, float angleH );

    //---------------------------------------------------------------------------
    //! @brief       ターゲットを中心に360度回転します。
    //!
    //! @param[in]    angleV  垂直回転角度
    //! @param[in]    angleH  水平回転角度
    //---------------------------------------------------------------------------
    void Revolve360( float angleV, float angleH );

    //---------------------------------------------------------------------------
    //! @brief       位置を最後にプリセットした位置へ戻します。
    //---------------------------------------------------------------------------
    void Reset();

    //---------------------------------------------------------------------------
    //! @brief       Reset 時に状態を戻せるように現在の設定を記憶します。
    //---------------------------------------------------------------------------
    void Preset();

    //---------------------------------------------------------------------------
    //! @brief       Twist の状態のみを Preset します。
    //---------------------------------------------------------------------------
    void PresetTwist();

    //---------------------------------------------------------------------------
    //! @brief       Bind されているオブジェクトに追従します。
    //---------------------------------------------------------------------------
    void Follow( const Camera* boundObj );

    //---------------------------------------------------------------------------
    //! @brief       Bindされているオブジェクトとの相対位置を更新します。
    //---------------------------------------------------------------------------
    void UpdateFollowingBase( const Camera* boundObj );

    //---------------------------------------------------------------------------
    //! @brief       現在の座標・アングルをプリント出力します。
    //---------------------------------------------------------------------------
    void Dump() const;

    //---------------------------------------------------------------------------
    //! @brief       設定された目的位置まで、TIME_TO_GOALのフレーム数だけ掛けてシームレスに移動させます。
    //---------------------------------------------------------------------------
    void MoveToDestination();

    //---------------------------------------------------------------------------
    //! @brief       イベントクラスの状態を元に移動します。
    //---------------------------------------------------------------------------
    void MoveByEvent( const Event& refEvent, const Camera* view = nullptr );

    //---------------------------------------------------------------------------
    //! @brief       姿勢行列を取得します。
    //---------------------------------------------------------------------------
    const nn::util::Matrix4x3fType&  GetMatrix() const { this->GetTargetMtx_( &m_Matrix ); return m_Matrix; }


protected:
    mutable nn::util::Vector3fType      m_Pos;               //!< カメラ位置
    mutable nn::util::Vector3fType      m_LookAtPos;         //!< ターゲット位置
    mutable float                       m_Distance;          //!< 注視点からの距離
    mutable float                       m_AngleV;            //!< カメラの水平回転角度
    mutable float                       m_AngleH;            //!< カメラの垂直回転角度
    float                               m_Twist;             //!< カメラのツイスト角度
    mutable nn::util::Vector3fType      m_LookVec;           //!< 視線方向ベクトル
    mutable nn::util::Vector3fType      m_UpVec;             //!< 視線上方向ベクトル
    float                               m_DistanceMax;       //!< 距離最大値
    float                               m_DistanceMin;       //!< 距離最小値
    nn::util::Vector3fType              m_PosReset;          //!< リセット時の位置
    nn::util::Vector3fType              m_LookAtPosReset;    //!< リセット時のターゲット位置
    float                               m_TwistReset;        //!< リセット時のツイスト角度
    mutable nn::util::Matrix4x3fType    m_Matrix;            //!< TBD

    //! @briefprivate TBD
    enum
    {
        RECALC_NONE = 0,    //!< 再計算の必要なし
        RECALC_OBJECT,      //!< カメラ位置を再計算
        RECALC_TARGET,      //!< ターゲット位置を再計算
        RECALC_ANGLE        //!< 回転角度を再計算
    };

    mutable uint32_t         m_RecalcFlg;         //!< 再計算フラグ

    //! @briefprivate 特定の条件の場合に再計算します。
    void RecalcIf_( uint32_t recalcFlg ) const;

    //! @briefprivate 再計算します。
    void Recalc_()          const;

    //! @briefprivate オブジェクトと注視点の座標から角度と距離を計算します。
    void RecalcAngle_()     const;

    //! @briefprivate 注視点座標と角度からカメラの座標を計算します。
    void RecalcPos_()       const;

    //! @briefprivate オブジェクト座標と角度から注視点座標を計算します。
    void RecalcLookAtPos_() const;

    //! @briefprivate 位置と注視点の距離が設定されたMin/Maxの範囲から外れていないかをチェックし、範囲内にクランプします。
    void ClampDistance_();

    //! @briefprivate 注視点を中心とした座標系への変換行列を取得
    void GetTargetMtx_( nn::util::Matrix4x3fType* pMatrix ) const;

    //! @briefprivate 垂直角度を設定
    void SetAngleV_( float angleV ) const
    {
        m_AngleV = angleV;
    }

    //! @briefprivate 水平角度を設定
    void SetAngleH_( float angleH )
    {
        m_AngleH = angleH;
    }

    bool           m_KeyP; // キーボード入力キー
    bool           m_KeyA; // キーボード入力キー
};
