﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <util/UtilXMLParser.h>

#if defined(USE_PUGIXML)
#   undef USE_PUGIXML
#   undef USE_RAPIDXML
#   undef USE_TINYXML
#   define USE_PUGIXML 1
#elif defined(USE_RAPIDXML)
#   undef USE_PUGIXML
#   undef USE_RAPIDXML
#   undef USE_TINYXML
#   define USE_RAPIDXML 1
#elif defined(USE_TINYXML)
#   undef USE_PUGIXML
#   undef USE_RAPIDXML
#   undef USE_TINYXML
#   define USE_TINYXML 1
#else
#   define USE_PUGIXML 1
#endif

#if USE_PUGIXML
#   define PUGIXML_NO_XPATH
#   define PUGIXML_NO_EXCEPTIONS
#   include <pugixml.hpp>
#   include <pugixml.cpp>
    using namespace pugi;
#elif USE_RAPIDXML
#   define RAPIDXML_NO_EXCEPTIONS
#   include <rapidxml.hpp>
    using namespace rapidxml;
#elif USE_TINYXML
#   define TIXML_USE_STL
#   include <tinyxml.h>
#   include <tinyxmlerror.cpp>
#   include <tinyxmlparser.cpp>
#   include <tinyxml.cpp>
#endif

#if USE_RAPIDXML && defined(RAPIDXML_NO_EXCEPTIONS)
void rapidxml::parse_error_handler(const char *what, void *where)
{
    std::abort();
}
#endif

#define DEFINE_CAST(handle, impl)                                                                  \
    impl* Impl(handle* ptr)                                                                        \
    {                                                                                              \
        return reinterpret_cast<impl*>(ptr);                                                       \
    }                                                                                              \
    const impl* Impl(const handle* ptr)                                                            \
    {                                                                                              \
        return reinterpret_cast<const impl*>(ptr);                                                 \
    }                                                                                              \
    handle* Handle(impl* ptr)                                                                      \
    {                                                                                              \
        return reinterpret_cast<handle*>(ptr);                                                     \
    }                                                                                              \
    const handle* Handle(const impl* ptr)                                                          \
    {                                                                                              \
        return reinterpret_cast<const handle*>(ptr);                                               \
    }                                                                                              \


namespace nw { namespace g3d { namespace tool {

namespace util {

namespace {

#if USE_PUGIXML
struct PredStringRef
{
    PredStringRef(const StringRef& name) : name(name) {}

    bool operator()(pugi::xml_attribute attr) const
    {
        return 0 == strncmp(attr.name(), name.Data(), name.Len()) && 0 == attr.name()[name.Len()];
    }

    bool operator()(pugi::xml_node node) const
    {
        return 0 == strncmp(node.name(), name.Data(), name.Len()) && 0 == node.name()[name.Len()];
    }

    const StringRef& name;

private:
    PredStringRef operator=( const PredStringRef & );
};
#endif // USE_PUGIXML

#if USE_PUGIXML

xml_document* Doc(void* ptr) { return static_cast<xml_document*>(ptr); }

xml_node Impl(XMLElement* ptr)
{
    return *reinterpret_cast<xml_node*>(&ptr);
}

const xml_node Impl(const XMLElement* ptr)
{
    return *reinterpret_cast<const xml_node*>(&ptr);
}

XMLElement* Handle(xml_node impl)
{
    return *reinterpret_cast<XMLElement**>(&impl);
}

#elif USE_RAPIDXML

DEFINE_CAST(XMLElement, xml_node<>)

xml_document<>* Doc(void* ptr) { return static_cast<xml_document<>*>(ptr); }

#elif USE_TINYXML

DEFINE_CAST(XMLElement, TiXmlElement)

TiXmlDocument* Doc(void* ptr) { return static_cast<TiXmlDocument*>(ptr); }

#endif

} // anonymous namespace

//--------------------------------------------------------------------------------------------------

XMLParser::XMLParser()
{
#if USE_PUGIXML
    m_pImpl = new xml_document();
#elif USE_RAPIDXML
    m_pImpl = new xml_document<>();
#elif USE_TINYXML
    m_pImpl = new TiXmlDocument();
#else
    m_pImpl = nullptr;
#endif
}

XMLParser::~XMLParser()
{
    if (m_pImpl)
    {
        delete Doc(m_pImpl);
        m_pImpl = nullptr;
    }
}

bool XMLParser::ParseTree(char* doc, size_t size)
{
    assert(doc);
#if USE_PUGIXML
    xml_parse_result result =
        Doc(m_pImpl)->load_buffer_inplace(doc, size, parse_default, encoding_utf8);
    return result.status == status_ok;
#elif USE_RAPIDXML
    (void)size;
    Doc(m_pImpl)->parse<0>(doc);
    return true;
#elif USE_TINYXML
    (void)size;
    return nullptr != Doc(m_pImpl)->Parse(doc);
#else
    return false;
#endif
}

void XMLParser::Clear()
{
    if (m_pImpl)
    {
#if USE_PUGIXML
        Doc(m_pImpl)->reset();
#elif USE_RAPIDXML
        Doc(m_pImpl)->clear();
#elif USE_TINYXML
        delete Doc(m_pImpl);
        m_pImpl = new TiXmlDocument();
#endif
    }
}


XMLElement* XMLParser::Root()
{
#if USE_PUGIXML
    return Handle(Doc(m_pImpl)->document_element());
#elif USE_RAPIDXML
    return Handle(Doc(m_pImpl)->first_node());
#elif USE_TINYXML
    return Handle(Doc(m_pImpl)->RootElement());
#else
    return nullptr;
#endif
}

//--------------------------------------------------------------------------------------------------

XMLElement::operator bool() const
{
#if USE_PUGIXML
    return !!Impl(this);
#else
    return this != nullptr;
#endif
}

int XMLElement::AttributeIndex(const StringRef& id) const
{
#if USE_PUGIXML
    int index = 0;
    for (auto iter = Impl(this).attributes_begin(); iter != Impl(this).attributes_end(); ++iter, ++index)
    {
        if (0 == strncmp(iter->name(), id.Data(), id.Len()) && 0 == iter->name()[id.Len()])
        {
            return index;
        }
    }
    return -1;
#elif USE_RAPIDXML
    // do not implementation
    return -1;
#elif USE_TINYXML
    // do not implementation
    return -1;
#else
    return -1;
#endif
}

const char* XMLElement::Attribute(const StringRef& id) const
{
#if USE_PUGIXML
    xml_attribute attrib = Impl(this).find_attribute(PredStringRef(id));
    return attrib ? attrib.value() : nullptr;
#elif USE_RAPIDXML
    xml_attribute<>* pAttrib = Impl(this)->first_attribute(id.Data(), id.Len());
    return pAttrib ? pAttrib->value() : nullptr;
#elif USE_TINYXML
    return Impl(this)->Attribute(id.Str())->c_str();
#else
    return nullptr;
#endif
}

const char* XMLElement::AttributeId(int index) const
{
#if USE_PUGIXML
    xml_attribute firstAttrib = Impl(this).first_attribute();
    xml_attribute attrib = firstAttrib;
    int i = index;
    do
    {
        if (i <= 0)
        {
            return attrib ? attrib.name() : nullptr;
        }

        attrib = attrib.next_attribute();
        --i;
    } while (i < index);

    return nullptr;
#elif USE_RAPIDXML
    // do not implementation
    return nullptr;
#elif USE_TINYXML
    // do not implementation
    return nullptr;
#else
    return nullptr;
#endif
}

const char* XMLElement::Text() const
{
#if USE_PUGIXML
    return Impl(this).child_value();
#elif USE_RAPIDXML
    return Impl(this)->value();
#elif USE_TINYXML
    return Impl(this)->GetText();
#else
    return nullptr;
#endif
}

XMLElement* XMLElement::Child(const StringRef& id)
{
#if USE_PUGIXML
    return Handle(Impl(this).find_child(PredStringRef(id)));
#elif USE_RAPIDXML
    return Handle(Impl(this)->first_node(id.Data(), id.Len()));
#elif USE_TINYXML
    return Handle(Impl(this)->FirstChildElement(id.Str()));
#else
    return nullptr;
#endif
}

const XMLElement* XMLElement::Child(const StringRef& id) const
{
#if USE_PUGIXML
    return Handle(Impl(this).find_child(PredStringRef(id)));
#elif USE_RAPIDXML
    return Handle(Impl(this)->first_node(id.Data(), id.Len()));
#elif USE_TINYXML
    return Handle(Impl(this)->FirstChildElement(id.Str()));
#else
    return nullptr;
#endif
}

XMLElement* XMLElement::NextSibling()
{
#if USE_PUGIXML
    return Handle(Impl(this).next_sibling(Impl(this).name()));
#elif USE_RAPIDXML
    return Handle(Impl(this)->next_sibling(Impl(this)->name(), Impl(this)->name_size()));
#elif USE_TINYXML
    return Handle(Impl(this)->NextSiblingElement(Impl(this)->ValueStr()));
#else
    return nullptr;
#endif
}

const XMLElement* XMLElement::NextSibling() const
{
#if USE_PUGIXML
    return Handle(Impl(this).next_sibling(Impl(this).name()));
#elif USE_RAPIDXML
    return Handle(Impl(this)->next_sibling(Impl(this)->name(), Impl(this)->name_size()));
#elif USE_TINYXML
    return Handle(Impl(this)->NextSiblingElement(Impl(this)->ValueStr()));
#else
    return nullptr;
#endif
}

XMLElement* XMLElement::NextSibling(const LiteralStr& id)
{
#if USE_PUGIXML
    return Handle(Impl(this).next_sibling(id.str));
#elif USE_RAPIDXML
    return Handle(Impl(this)->next_sibling(id.str, id.len));
#elif USE_TINYXML
    return Handle(Impl(this)->NextSiblingElement(id.str));
#else
    return nullptr;
#endif
}

const XMLElement* XMLElement::NextSibling(const LiteralStr& id) const
{
#if USE_PUGIXML
    return Handle(Impl(this).next_sibling(id.str));
#elif USE_RAPIDXML
    return Handle(Impl(this)->next_sibling(id.str, id.len));
#elif USE_TINYXML
    return Handle(Impl(this)->NextSiblingElement(id.str));
#else
    return nullptr;
#endif
}

} // namespace util

} // namespace tool
} // namespace g3d
} // namespace nw
