﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <util/UtilCmdArgs.h>
#include <util/UtilError.h>
#include <util/UtilErrorCode.h>

#include <shellapi.h>

namespace
{
bool ReplaceStringAll(std::wstring *pStr, const wchar_t* pOld, const wchar_t* pNew)
{
    bool ret = false;
    int length = lstrlen(pOld);
    for (size_t pos = pStr->find(pOld); pos != std::wstring::npos; pos = pStr->find(pOld, pos))
    {
        pStr->replace(pos, length, pNew);
        ret = true;
    }
    return ret;
}
}

namespace nw { namespace g3d { namespace tool {
CmdOption::CmdOption(const wchar_t* arg) : name(), value()
{
    const std::wstring str(arg);
    auto pos = str.find(L'=');
    if (std::wstring::npos == pos)
    {
        name.assign(arg);
    }
    else
    {
        name.assign(str.substr(0, pos));
        std::wstring optionValue = str.substr(pos + 1);
        DWORD strSize = ExpandEnvironmentStrings(optionValue.c_str(), NULL, 0);
        wchar_t* tempValue = new wchar_t[strSize];
        ExpandEnvironmentStrings(optionValue.c_str(), tempValue, strSize);
        value.assign(tempValue);
        delete[] tempValue;
    }
}

int CmdOption::FindIndex(const std::vector<CmdOption>& options, const wchar_t* name)
{
    int index = 0;
    for (auto iter = options.cbegin(); iter != options.cend(); ++iter, ++index)
    {
        if (0 == wcscmp(name, iter->name.c_str()))
        {
            return index;
        }
    }
    return -1;
}

CmdInput::CmdInput(const wchar_t* path) : fullpath(), fname(), ext(), options()
{
    wchar_t fullpathBuf[_MAX_PATH];
    wchar_t fnameBuf[_MAX_FNAME];
    wchar_t extBuf[_MAX_EXT];
    DWORD strSize = ExpandEnvironmentStrings(path, NULL, 0);
    wchar_t* tempValue = new wchar_t[strSize];
    ExpandEnvironmentStrings(path, tempValue, strSize);
    _wfullpath(fullpathBuf, tempValue, _MAX_PATH);
    delete[] tempValue;
    fullpath.assign(fullpathBuf);
    _wsplitpath_s(path, nullptr, 0, nullptr, 0, fnameBuf, _MAX_FNAME, extBuf, _MAX_EXT);
    fname.assign(fnameBuf);
    ext.assign(extBuf);
}

std::vector<std::wstring> FileToArgs(const wchar_t* fileName)
{
    std::vector<std::wstring> rval;
    std::vector<char> buf;
    std::vector<wchar_t> wbuf;
    wbuf.reserve(8192);

    // フルパス取得
    wchar_t fullpathBuf[_MAX_PATH];
    _wfullpath(fullpathBuf, fileName, _MAX_PATH);

    if (fullpathBuf == NULL)
    {
        THROW_ERROR(ERRCODE_INVALID_OPTIONFILE, "@filename is too long.");
    }

    // バイナリモードでオープン
    std::ifstream binaryFile(fullpathBuf, std::ios::binary);

    if (!binaryFile)
    {
        THROW_ERROR(ERRCODE_INVALID_OPTIONFILE, "Cannot open option file. Please check @filename [%s].", fullpathBuf);
    }

    std::vector<char> bomBuf;
    bomBuf.resize(3);
    char* binaryChar = &bomBuf[0];
    binaryFile.read(binaryChar, 3);

    if (binaryFile.bad())
    {
        THROW_ERROR(ERRCODE_INVALID_OPTIONFILE, "Cannot read option file. Please check @filename.");
    }

    bool isUtf8 = false;
    unsigned char* uBinaryChar = reinterpret_cast<unsigned char*>(binaryChar);
    if (uBinaryChar[0] == 0xEF && uBinaryChar[1] == 0xBB && uBinaryChar[2] == 0xBF)
    {
        // UTF8 でない場合、デフォルトのロケールを使うように変更。
        isUtf8 = true;
        //THROW_ERROR(ERRCODE_INVALID_OPTIONFILE, "Option file is not UTF-8 BOM. @filename have to be UTF-8 BOM.");
    }

    binaryFile.close();

    std::ifstream file(fileName);

    // とりあえず文字列にする
    bool isComment = false;
    char c;
    while(file.get(c))
    {
        if (c == '\n')
        {
            if (isComment)
            {
                isComment = false;
            }
            else
            {
                buf.push_back(' ');
            }
        }
        // # 以降をコメントアウトする。
        else if (c == '#')
        {
            isComment = true;
        }
        else if (c != '\r' && !isComment)
        {
            buf.push_back(c);
        }
    }
    buf.push_back(0);
    file.close();

    //
    // CommandLineToArgvWがワイド文字列版しかないので
    // おもむろにワイド文字列に変換
    //
    int wLen = MultiByteToWideChar(isUtf8 ? CP_UTF8 : CP_ACP,
                                   MB_ERR_INVALID_CHARS,
                                   isUtf8 ? &buf[3] : &buf[0],
                                   -1,
                                   NULL,
                                   0);
    wbuf.resize(wLen);
    int error = MultiByteToWideChar(isUtf8 ? CP_UTF8 : CP_ACP,
                        MB_ERR_INVALID_CHARS,
                        isUtf8 ? &buf[3] : &buf[0],
                        -1,
                        &wbuf[0],
                        wLen);

    if (error == 0)
    {
        THROW_ERROR_INTERNAL(ERRCODE_INTERNAL, "InternalError.");
    }

    std::wstring wstr;
    wstr.resize(wbuf.size());
    memcpy(&wstr[0], &wbuf[0], sizeof(wchar_t) * wbuf.size());

    // "=" を発見したら空白に置き換える。
    // また、直前に目印として DEAD@@BEAF を挿入する。
    const wchar_t* const DUMMY_STRING = L"DEAD@@BEAF";
    const wchar_t* const DUMMY_STRING_ = L"DEAD@@BEAF ";
    const wchar_t* const TARGET_STRING = L"=";
    ReplaceStringAll(&wstr, TARGET_STRING, DUMMY_STRING_);

    // 正確にコマンドライン通りに分割したいのでこのAPIを使う
    // Windows依存になるのはとりあえず黙認
    int argc;
    LPWSTR* argv = CommandLineToArgvW(&wstr[0], &argc);

    // 各argを配列に格納する。
    for (int i = 0; i < argc; ++i)
    {
        std::wstring outString = *(argv+i);
        // DEAD@@BEAF が入っていた場合は除去する。
        if (ReplaceStringAll(&outString, DUMMY_STRING, TARGET_STRING))
        {
            std::wstring strValue = *(argv + (++i));
            ReplaceStringAll(&strValue, DUMMY_STRING_, TARGET_STRING);
            outString += strValue;
        }
        rval.push_back(outString);
    }

    LocalFree(argv);

    return rval;
}

static bool
Unrolling(std::list<std::wstring>& cmd, std::set<std::wstring>& file_set)
{
    bool isReplaced = false;
    for (auto it = cmd.begin(); it != cmd.end(); ++it)
    {
        if ((*it)[0] == '@')
        {
            isReplaced = true;
            std::wstring filename = it->substr(1);

            if (file_set.find(filename) != file_set.end())
            {
                THROW_ERROR(ERRCODE_INVALID_OPTIONFILE, "Option file is recursively unrolling.");
            }

            // 無限ループチェック用セットに追加
            file_set.insert(filename);

            {
                // @つきの引数があった場所に展開する
                std::vector<std::wstring> args = FileToArgs(filename.c_str());
                std::list<std::wstring> tmp(args.begin(), args.end());

                // 再帰的に展開
                Unrolling(tmp, file_set);

                // @つきの引数を取り除く
                it = cmd.erase(it);

                // その場所に展開された引数を挿入
                cmd.splice(it, tmp);

                // 次の要素を指しているので1つ戻る
                --it;
            }

            // 無限ループチェック用セットから削除
            file_set.erase(filename);
        }
    }

    return isReplaced;
}

CmdArgs::CmdArgs(int argc, const wchar_t* argv[]) : cmd(), options(), inputs()
{
    std::list<std::wstring> args;

    for (int cmdIndex = 1; cmdIndex < argc; ++ cmdIndex)
    {
        args.push_back(argv[cmdIndex]);
    }

    std::set<std::wstring> file_set;
    Unrolling(args, file_set);

    auto argIter = args.begin();
    auto parse = [&](std::list<std::wstring>::iterator argIter, std::vector<CmdOption>& options) -> std::list<std::wstring>::iterator
    {
        while (argIter != args.end())
        {
            const std::wstring arg = *argIter;
            if (arg[0] != L'-')
            {
                break; // オプションではない。
            }
            options.push_back(CmdOption(arg.c_str()));
            ++argIter;
        }

        return argIter;
    };

    // グローバルオプション。
    argIter = parse(argIter, options);

    while (argIter != args.end())
    {
        const std::wstring arg = *argIter;
        CmdInput input(arg.c_str());
        ++argIter;

        // ローカルオプション。
        argIter = parse(argIter, input.options);

        inputs.push_back(input);
    }
}

} // namespace tool
} // namespace g3d
} // namespace nw
