﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once
#include <util/UtilStringRef.h>

#ifndef WIN32_LEAN_AND_MEAN
#define WIN32_LEAN_AND_MEAN
#endif
#ifndef NOMINMAX
#define NOMINMAX
#endif
#include <Windows.h>
#include <Shlwapi.h>
#include <eh.h>

#pragma warning(push)
#pragma warning(disable:4091)
#include <dbghelp.h>
#pragma warning(pop)

namespace nw { namespace g3d { namespace tool {
namespace util {

const std::string CRLF = "\r\n";
const std::wstring WCRLF = L"\r\n";
const std::string LF = "\n";
const std::wstring WLF = L"\n";

std::string ToUTF8(const wchar_t* str);

std::string ToAscii(const wchar_t* str);

// ロケールの文字列に変換します。
std::string ToLocale(const std::string& str);

// ワイド文字列に変換します。
std::wstring ToWstr(const std::string& str);

// delSep を指定すると sep が連続した場合に要素を抽出しない（空白での区切りに利用）
void Split(std::vector<std::string>& v, const std::string& str, const std::string& sep, bool delSep = true);

std::string Trim(const std::string& line, const std::string blank = " \t\r\n\0");

std::string TrimLeft(const std::string& line, const std::string blank = " \t\r\n\0");

std::string TrimRight(const std::string& line, const std::string blank = " \t\r\n\0");

std::string TrimString(const std::string& str, const std::string& trim);

std::string EraseToken(const std::string& str, const std::vector<std::string>& tokens);

std::string EraseNull(const std::string& str);

// 文字列を float として正規化する
bool ModifyFloatValue(std::string& str);

// 文字列を int として正規化する
bool ModifyIntValue(std::string& str);

// 配列を展開して、空白で区切る
std::string ModifyBlank(const std::vector<std::string>& v);

template<typename String>
inline String ReplaceAll(const String& str, const String pre, const String post)
{
    String newString(str);
    for (auto comp = newString.find(pre); comp != String::npos; comp = newString.find(pre, post.length() + comp))
    {
        newString.replace(comp, pre.length(), post);
    }

    return newString;
}

std::string EscapeString(const std::string& str);

bool TryParse(int& value, const std::string& str);
bool TryParse(int& value, const std::wstring& str);

std::string GetModuleName(HINSTANCE hInstance);

template<UINT codePage>
class TPath
{
public:
    static bool FileExists(const std::string& path)
    {
        return PathFileExists(ToWideChar(path.c_str()).c_str()) == TRUE;
    }

    static bool IsRelative(const std::string& path)
    {
        return PathIsRelative(ToWideChar(path.c_str()).c_str()) == TRUE;
    }

    static std::string Canonicalize(const std::string& path)
    {
        wchar_t buf[_MAX_PATH];
        return PathCanonicalize(buf, ToWideChar(path.c_str()).c_str()) ? ToMultiByte(buf) : "";
    }

    static std::string ToFull(const std::string& path)
    {
        wchar_t buf[_MAX_PATH];
        return _wfullpath(buf, ToWideChar(path.c_str()).c_str(), _MAX_PATH) ? ToMultiByte(buf) : "";
    }

    static std::string Combine(const std::string& dir, const std::string& file)
    {
        wchar_t buf[_MAX_PATH];
        return PathCombine(buf, ToWideChar(dir.c_str()).c_str(),
            ToWideChar(file.c_str()).c_str()) ? ToMultiByte(buf) : "";
    }

    static std::string ExpandEnvironment(const std::string& path)
    {
        std::wstring wpath = ToWideChar(path.c_str());
        if (int len = ExpandEnvironmentStrings(wpath.c_str(), nullptr, 0))
        {
            std::unique_ptr<wchar_t> buf(new wchar_t[len + 1]);
            if (ExpandEnvironmentStrings(wpath.c_str(), buf.get(), len))
            {
                return ToMultiByte(buf.get());
            }
        }
        return "";
    }

    static bool Split(const std::string& path, std::string* drive, std::string* dir, std::string* fname, std::string* ext)
    {
        wchar_t wdrive[_MAX_DRIVE];
        wchar_t wdir[_MAX_DIR];
        wchar_t wfname[_MAX_FNAME];
        wchar_t wext[_MAX_EXT];
        if (_wsplitpath_s(ToWideChar(path.c_str()).c_str(), wdrive, _MAX_DRIVE,
            wdir, _MAX_DIR, wfname, _MAX_FNAME, wext, _MAX_EXT))
        {
            return false;
        }

        if (drive)
        {
            *drive = ToMultiByte(wdrive);
        }
        if (dir)
        {
            *dir = ToMultiByte(wdir);
        }
        if (fname)
        {
            *fname = ToMultiByte(wfname);
        }
        if (ext)
        {
            *ext = ToMultiByte(wext);
        }

        return true;
    }

    static std::string ToMultiByte(const wchar_t* str)
    {
        int len = WideCharToMultiByte(codePage, 0, str, -1, nullptr, 0, nullptr, nullptr);
        std::unique_ptr<char> buf(new char[len]);
        return WideCharToMultiByte(codePage, 0, str, -1, buf.get(), len, nullptr, nullptr) ? buf.get() : "";
    }

    static std::wstring ToWideChar(const char* str)
    {
        int len = MultiByteToWideChar(codePage, 0, str, -1, nullptr, 0);
        std::unique_ptr<wchar_t> buf(new wchar_t[len]);
        return MultiByteToWideChar(codePage, 0, str, -1, buf.get(), len) ? buf.get() : L"";
    }
};

typedef TPath<CP_UTF8> Path;

} // namespace util

} // namespace tool
} // namespace g3d
} // namespace nw
