﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <memory>
#include <type_traits>

namespace nw { namespace g3d { namespace tool {
namespace util {

template <typename T>
class ShaderErrorLog
{
public:
    typedef T SourceType;

    ShaderErrorLog()
    {
        this->m_Success = 0;
        this->m_InfoLength  = 0;
        this->m_LogLength = 0;

        m_VertexShader = m_GeometryShader = m_FragmentShader = m_ComputeShader = NULL;
    }

    void SetLogLength(int logLength)
    {
        this->m_LogLength = logLength;
        this->m_ErrorMessage.reset(new char[logLength]);
    }

    int m_Success;
    int m_LogLength;
    int m_InfoLength;
    std::unique_ptr<char[]> m_ErrorMessage;

    const typename std::remove_const<T>::type* m_VertexShader;
    const typename std::remove_const<T>::type* m_GeometryShader;
    const typename std::remove_const<T>::type* m_FragmentShader;
    const typename std::remove_const<T>::type* m_ComputeShader;
};

class Exception
{
public:
    Exception(int code, const wchar_t* path, int line)
        : code(code)
        , line(line)
    {
        wchar_t fname[_MAX_FNAME];
        wchar_t ext[_MAX_EXT];
        _wsplitpath_s(path, nullptr, 0, nullptr, 0, fname, _MAX_FNAME, ext, _MAX_EXT);
        filename.assign(fname);
        filename += ext;
    }

    int code;
    int line;
    std::wstring filename;
};

template <typename T>
class ShaderException : public Exception
{
public:
    explicit ShaderException(int code, const wchar_t* path, int line, std::shared_ptr<ShaderErrorLog<T>> errorLog)
        : Exception(code, path, line)
    {
        m_ErrorLog = errorLog;
    }

    std::shared_ptr<ShaderErrorLog<T>> m_ErrorLog;
};

template<typename T>
bool CheckGEMinLEMax(T value, T min, T max)
{
    if (value <= max && value >= min)
    {
        return true;
    }

    return false;
}

template<typename T>
bool CheckGMinLMax(T value, T min, T max)
{
    if (value < max && value > min)
    {
        return true;
    }

    return false;
}

} // namespace util

} // namespace tool
} // namespace g3d
} // namespace nw
