﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <util/types.h>

namespace nw { namespace g3d { namespace tool {
namespace util {

//---------------------------------------------------------------------------
//! @brief ビットフラグを扱うためのクラスです。
//!
//! @details
//! T は u8/u16/u32 のみ有効です。実際に使用する際は、
//! BitFlag8,BitFlag16,BitFlag32の何れかをご使用ください。
//---------------------------------------------------------------------------
template <typename T>
class BitUtil {
    //static_assert( sizeof(T) <= 4 );
public:
    //! @brief デフォルトコンストラクタ
    //!
    BitUtil() : mBits(0) {}

    //! @brief 引数で指定された値で初期化するコンストラクタです。
    //!
    explicit BitUtil( T bits ) : mBits(bits) {}

    //! @brief Tへのキャスト
    //!
    operator T() { return mBits; }

    //! @brief 全てのbitが0か否かを返します。
    //!
    bool IsZero() const { return mBits == 0; }

    //! @brief bitを全て0にします。
    //!
    void SetAllZero()
    {
        mBits = static_cast<T>(0);
    }

    //! @brief bitを全て1にします。
    //!
    void SetAllOne()
    {
        mBits = static_cast<T>(-1);
    }

    //! @brief 直接bitを設定します。
    //!
    void SetDirect( T bits ) { mBits = bits; }

    //! @brief 直接bitを取得します。
    //!
    T GetDirect() const { return mBits; }

    //! @brief ビットフラグを格納している内部変数のアドレスを直接取得します。
    //!
    T* GetPtr() { return &mBits; }

    //! @brief ビットフラグを格納している内部変数のconstアドレスを直接取得します。
    //!
    const T* GetPtr() const { return &mBits; }

    //! @brief フラグ変数の大きさを返します。(バイト)
    //!
    size_t GetByteSize() const { return sizeof(T); }

    //----------------------------------------
    //! @name mask による操作
    //@{
    //! @brief maskで指定されたbitを1にします。
    //!
    void SetMaskOn( T mask ) { mBits |= mask; }

    //! @brief maskで指定されたbitを0にします。
    //!
    void SetMaskOff( T mask ) { mBits &= ~mask; }

    //! @brief maskで指定されたbitを反転します。
    //!
    void ToggleMask( T mask ) { mBits ^= mask; }

    //! @brief maskで指定されたbitをbの値にします。
    //!
    void ChangeMask( T mask, bool b )
    {
        if ( b ) { SetMaskOn(mask); } else { SetMaskOff(mask); }
    }

    //! @brief  与えられたマスクの何れかのbitが立っているか調べます。
    //!
    bool IsMaskOn( T mask ) const { return static_cast<bool>((mBits & mask) != 0); }

    //! @brief  与えられたマスクのbitの何れかが立っていないかを調べます。
    //!
    bool IsMaskOff( T mask ) const { return ! IsMaskOn(mask); }

    //! @brief  与えられたビットマスクとのビット積を返します。
    //!
    T GetAndMask( T mask ) const { return static_cast<T>(mBits & mask); }

    //! @brief  下位から数えて、bit目のbitが1となるマスクを作成します。
    //!
    static T MakeMask( int bit )
    {
        return static_cast<T>(1 << bit);
    }

    //! @brief  上位から数えて、bit目のbitが1となるマスクを作成します。
    //!
    static T MakeHiMask( int bit )
    {
        return static_cast<T>(1 << (sizeof(T)*8 - bit - 1));
    }

    //! @brief  上位から数えて、lhsからrhsが1となるマスクを作成します。
    //!
    static T MakeHiMask( int lhs, int rhs )
    {
        T mask = 0;
        for (int i = lhs; i <= rhs; ++i)
        {
            mask |= static_cast<T>(1 << (sizeof(T)*8 - i - 1));
        }
        return mask;
    }

    //@}
    //----------------------------------------
    //! @name bit による操作
    //@{

    //! @brief  下位から数えて、bit目のbitを1にする。
    //!
    void SetBitOn( int bit ) { SetMaskOn( MakeMask( bit ) ); }

    //! @brief  下位から数えて、bit目のbitを0にする。
    //!
    void SetBitOff( int bit ) { SetMaskOff( MakeMask( bit ) ); }

    //! @brief  下位から数えて、bit目のbitを反転する。
    //!
    void ToggleBit( int bit ) { ToggleMask( MakeMask( bit ) ); }

    //! @brief  下位から数えて、bit目のbitを指定した値にする。
    //!
    void ChangeBit( int bit, bool b ) { ChangeMask( MakeMask( bit ), b ); }

    //! @brief  下位から数えて、bit目のbitが1ならtrueを返す。
    //!
    bool IsBitOn( int bit ) const { return IsMaskOn( MakeMask( bit ) ); }

    //! @brief  下位から数えて、bit目のbitが0ならtrueを返す。
    //!
    bool IsBitOff( int bit ) const { return IsMaskOff( MakeMask( bit ) ); }

    //! @brief 上位から数えて、lhsビットからrhsビットにvalueを書き込む。
    void SetHiBitOn( int lhs, int rhs, int value )
    {
        T mask = MakeHiMask(lhs, rhs);
        SetMaskOff(mask);
        mBits |= (value << (32 - rhs - 1)) & mask;
    }

    //@}

protected:
    T mBits;

};

/// 8bitまでのフラグを扱うことのできるBitFlagです。
typedef BitUtil<u8>  BitFlag8;
/// 16bitまでのフラグを扱うことのできるBitFlagです。
typedef BitUtil<u16> BitFlag16;
/// 32bitまでのフラグを扱うことのできるBitFlagです。
typedef BitUtil<u32> BitFlag32;

} // namespace util

} // namespace tool
} // namespace g3d
} // namespace nw
