﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once


namespace nw { namespace g3d { namespace tool {

class BackTrace
{
    //初期化
    BackTrace()
    {
        m_IsSymbolEngineReady = false;

        this->SymSetOptionsProc = NULL;
        this->SymInitializeProc = NULL;
        this->SymCleanupProc = NULL;
#if (defined(_WIN64) || defined(__x86_64__))
        this->SymGetModuleInfoProc64 = NULL;
        this->SymGetSymFromAddrProc64 = NULL;
        this->SymGetLineFromAddrProc64 = NULL;
#else
        this->SymGetModuleInfoProc = NULL;
        this->SymGetSymFromAddrProc = NULL;
        this->SymGetLineFromAddrProc = NULL;
#endif

        HINSTANCE hDbgDLL = LoadLibrary(L"DbgHelp.dll");

#if (defined(_WIN64) || defined(__x86_64__))
        this->SymGetModuleInfoProc64 = (SymGetModuleInfoDef64)GetProcAddress(hDbgDLL, "SymGetModuleInfo64");
        this->SymGetSymFromAddrProc64 = (SymGetSymFromAddrDef64)GetProcAddress(hDbgDLL, "SymGetSymFromAddr64");
        this->SymGetLineFromAddrProc64 = (SymGetLineFromAddrDef64)GetProcAddress(hDbgDLL, "SymGetLineFromAddr64");
#else
        this->SymGetModuleInfoProc = (SymGetModuleInfoDef)GetProcAddress(hDbgDLL, "SymGetModuleInfo");
        this->SymGetSymFromAddrProc = (SymGetSymFromAddrDef)GetProcAddress(hDbgDLL, "SymGetSymFromAddr");
        this->SymGetLineFromAddrProc = (SymGetLineFromAddrDef)GetProcAddress(hDbgDLL, "SymGetLineFromAddr");
#endif

        this->SymSetOptionsProc = (SymSetOptionsDef)GetProcAddress(hDbgDLL, "SymSetOptions");
        this->SymInitializeProc = (SymInitializeDef)GetProcAddress(hDbgDLL, "SymInitialize");
        this->SymCleanupProc = (SymCleanupDef)GetProcAddress(hDbgDLL, "SymCleanup");

        m_Process = GetCurrentProcess();

        //シンボルエンジンの初期化.
        SymSetOptionsProc(SYMOPT_DEFERRED_LOADS | SYMOPT_LOAD_LINES | SYMOPT_UNDNAME);
        if(SymInitializeProc(m_Process, NULL, TRUE))
        {
            m_IsSymbolEngineReady = true;
        }
    }

#if (defined(_WIN64) || defined(__x86_64__))
    //アドレスからモジュールを求める
    typedef BOOL (WINAPI *SymGetModuleInfoDef64) (HANDLE hProcess, DWORD64 dwAddr,  PIMAGEHLP_MODULE64 ModuleInfo );
    SymGetModuleInfoDef64 SymGetModuleInfoProc64;
    //アドレスからシンボルを求める
    typedef BOOL (WINAPI *SymGetSymFromAddrDef64) (HANDLE hProcess,DWORD64 Address,PDWORD64 Displacement,PIMAGEHLP_SYMBOL64 Symbol);
    SymGetSymFromAddrDef64 SymGetSymFromAddrProc64;
    //アドレスからファイルと行番号を求める
    typedef BOOL (WINAPI *SymGetLineFromAddrDef64) (HANDLE hProcess,  DWORD64 Address,  PDWORD64 Displacement,  PIMAGEHLP_LINE64 Line);
    SymGetLineFromAddrDef64 SymGetLineFromAddrProc64;
#else
    //アドレスからモジュールを求める
    typedef BOOL (WINAPI *SymGetModuleInfoDef) (HANDLE hProcess, DWORD dwAddr,  PIMAGEHLP_MODULE ModuleInfo );
    SymGetModuleInfoDef SymGetModuleInfoProc;
    //アドレスからシンボルを求める
    typedef BOOL (WINAPI *SymGetSymFromAddrDef) (HANDLE hProcess,DWORD Address,PDWORD Displacement,PIMAGEHLP_SYMBOL Symbol);
    SymGetSymFromAddrDef SymGetSymFromAddrProc;
    //アドレスからファイルと行番号を求める
    typedef BOOL (WINAPI *SymGetLineFromAddrDef) (HANDLE hProcess,  DWORD dwAddr,  PDWORD pdwDisplacement,  PIMAGEHLP_LINE Line);
    SymGetLineFromAddrDef SymGetLineFromAddrProc;
#endif

    //シンボルエンジンのオプション
    typedef BOOL (WINAPI *SymSetOptionsDef) ( DWORD SymOptions );
    SymSetOptionsDef SymSetOptionsProc;

    //シンボルエンジンの初期化
    typedef BOOL (WINAPI *SymInitializeDef) (HANDLE hProcess, PSTR UserSearchPath, BOOL fInvadeProcess  );
    SymInitializeDef SymInitializeProc;

    //シンボルエンジンの終了
    typedef BOOL (WINAPI *SymCleanupDef) (HANDLE hProcess);
    SymCleanupDef SymCleanupProc;


    HANDLE m_Process;
    bool m_IsSymbolEngineReady;
public:
    //解放
    virtual ~BackTrace()
    {
        if (m_IsSymbolEngineReady)
        {
            SymCleanupProc(m_Process);
            m_IsSymbolEngineReady = false;
        }
    }

    //シンボルの解決
    void AddressToSymbolString(void* address ,char * outBuffer , int len) const
    {
        if ( ! m_IsSymbolEngineReady )
        {
            //シンボルエンジンが準備できていない
            _snprintf_s(outBuffer ,len , _TRUNCATE , "Address: 0x%p\n" , address);
            return ;
        }

        // モジュール名取得
#if (defined(_WIN64) || defined(__x86_64__))
        IMAGEHLP_MODULE64 imageModule = { sizeof(IMAGEHLP_MODULE64) };
        BOOL r = SymGetModuleInfoProc64(m_Process ,(DWORD64) address , &imageModule);
#else
        IMAGEHLP_MODULE imageModule = { sizeof(IMAGEHLP_MODULE) };
        BOOL r = SymGetModuleInfoProc(m_Process ,(DWORD) address , &imageModule);
#endif
        if (!r)
        {
            _snprintf_s(outBuffer ,len , _TRUNCATE , "Address: 0x%p\n" ,address );
            return ;
        }

        //関数名の取得
#if (defined(_WIN64) || defined(__x86_64__))
        //シンボル情報格納バッファ.
        IMAGEHLP_SYMBOL64 * imageSymbol;
        char buffer[MAX_PATH + sizeof(IMAGEHLP_SYMBOL64) ] = {0};
        imageSymbol = (IMAGEHLP_SYMBOL64*)buffer;
        imageSymbol->SizeOfStruct = sizeof(IMAGEHLP_SYMBOL64);
        imageSymbol->MaxNameLength = MAX_PATH;

        DWORD64 disp = 0;
        r = SymGetSymFromAddrProc64(m_Process , (DWORD64)address , &disp , imageSymbol );
#else
        //シンボル情報格納バッファ.
        IMAGEHLP_SYMBOL * imageSymbol;
        char buffer[MAX_PATH + sizeof(IMAGEHLP_SYMBOL) ] = {0};
        imageSymbol = (IMAGEHLP_SYMBOL*)buffer;
        imageSymbol->SizeOfStruct = sizeof(IMAGEHLP_SYMBOL);
        imageSymbol->MaxNameLength = MAX_PATH;

        DWORD disp = 0;
        r = SymGetSymFromAddrProc(m_Process , (DWORD)address , &disp , imageSymbol );
#endif
        if (!r)
        {
            _snprintf_s(outBuffer ,len , _TRUNCATE , "Address: 0x%p\nModule: %s" , address, imageModule.ModuleName );
            return ;
        }

        //行番号の取得
#if (defined(_WIN64) || defined(__x86_64__))
        IMAGEHLP_LINE64 line ={sizeof(IMAGEHLP_LINE64)};
        r = SymGetLineFromAddrProc64(m_Process ,(DWORD64) address , &disp , &line);
#else
        IMAGEHLP_LINE line ={sizeof(IMAGEHLP_LINE)};
        r = SymGetLineFromAddrProc(m_Process ,(DWORD) address , &disp , &line);
#endif
        if (!r)
        {
            _snprintf_s(outBuffer ,len , _TRUNCATE , "Address: 0x%p\nModule: %s\nSymbol: %s\nAddress: %d" ,address,
            imageModule.ModuleName , imageSymbol->Name, (int) ((char*)address - (char*)line.Address) );
            return ;
        }

        _snprintf_s(outBuffer , len ,_TRUNCATE, "Address: 0x%p\nModule: %s\nSymbol: %s\nFile(Line): %s(%d)" ,address,imageModule.ModuleName , imageSymbol->Name , line.FileName , line.LineNumber);
    }

    static const BackTrace* Get()
    {
        static BackTrace s;
        return &s;
    }
};

} // namespace tool
} // namespace g3d
} // namespace nw
