﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <g3dif/Vertex.h>

namespace nw { namespace g3d { namespace tool {
namespace g3dif {

G3DIF_DEFINE_ENUM_TABLE(
    vtx_attrib, semantic_name,
    S11N_DEFINE_ENUM_ID(generic),
    S11N_DEFINE_ENUM_ID(position),
    S11N_DEFINE_ENUM_ID(normal),
    S11N_DEFINE_ENUM_ID(tangent),
    S11N_DEFINE_ENUM_ID(binormal),
    S11N_DEFINE_ENUM_ID(color),
    S11N_DEFINE_ENUM_ID(blendindex),
    S11N_DEFINE_ENUM_ID(blendweight),
    S11N_DEFINE_ENUM_ID(texcoord)
);

G3DIF_DEFINE_ENUM_TABLE(
    vtx_attrib, type,
    S11N_DEFINE_ENUM_ID(int),
    S11N_DEFINE_ENUM_ID(int2),
    S11N_DEFINE_ENUM_ID(int3),
    S11N_DEFINE_ENUM_ID(int4),
    S11N_DEFINE_ENUM_ID(uint),
    S11N_DEFINE_ENUM_ID(uint2),
    S11N_DEFINE_ENUM_ID(uint3),
    S11N_DEFINE_ENUM_ID(uint4),
    S11N_DEFINE_ENUM_ID(float),
    S11N_DEFINE_ENUM_ID(float2),
    S11N_DEFINE_ENUM_ID(float3),
    S11N_DEFINE_ENUM_ID(float4)
);

G3DIF_DEFINE_ENUM_TABLE(
    vtx_attrib, quantize_type,
    S11N_DEFINE_ENUM_ID(none),
    S11N_DEFINE_ENUM_ID(unorm_8),
    S11N_DEFINE_ENUM_ID(uint_8),
    S11N_DEFINE_ENUM_ID(snorm_8),
    S11N_DEFINE_ENUM_ID(sint_8),
    S11N_DEFINE_ENUM_ID(uint_to_float_8),
    S11N_DEFINE_ENUM_ID(sint_to_float_8),
    S11N_DEFINE_ENUM_ID(unorm_4_4),
    S11N_DEFINE_ENUM_ID(unorm_16),
    S11N_DEFINE_ENUM_ID(uint_16),
    S11N_DEFINE_ENUM_ID(snorm_16),
    S11N_DEFINE_ENUM_ID(sint_16),
    S11N_DEFINE_ENUM_ID(float_16),
    S11N_DEFINE_ENUM_ID(uint_to_float_16),
    S11N_DEFINE_ENUM_ID(sint_to_float_16),
    S11N_DEFINE_ENUM_ID(unorm_8_8),
    S11N_DEFINE_ENUM_ID(uint_8_8),
    S11N_DEFINE_ENUM_ID(snorm_8_8),
    S11N_DEFINE_ENUM_ID(sint_8_8),
    S11N_DEFINE_ENUM_ID(uint_to_float_8_8),
    S11N_DEFINE_ENUM_ID(sint_to_float_8_8),
    S11N_DEFINE_ENUM_ID(uint_32),
    S11N_DEFINE_ENUM_ID(sint_32),
    S11N_DEFINE_ENUM_ID(float_32),
    S11N_DEFINE_ENUM_ID(unorm_16_16),
    S11N_DEFINE_ENUM_ID(uint_16_16),
    S11N_DEFINE_ENUM_ID(snorm_16_16),
    S11N_DEFINE_ENUM_ID(sint_16_16),
    S11N_DEFINE_ENUM_ID(float_16_16),
    S11N_DEFINE_ENUM_ID(uint_to_float_16_16),
    S11N_DEFINE_ENUM_ID(sint_to_float_16_16),
    S11N_DEFINE_ENUM_ID(float_10_11_11),
    S11N_DEFINE_ENUM_ID(unorm_8_8_8_8),
    S11N_DEFINE_ENUM_ID(uint_8_8_8_8),
    S11N_DEFINE_ENUM_ID(snorm_8_8_8_8),
    S11N_DEFINE_ENUM_ID(sint_8_8_8_8),
    S11N_DEFINE_ENUM_ID(uint_to_float_8_8_8_8),
    S11N_DEFINE_ENUM_ID(sint_to_float_8_8_8_8),
    S11N_DEFINE_ENUM_ID(unorm_10_10_10_2),
    S11N_DEFINE_ENUM_ID(uint_10_10_10_2),
    S11N_DEFINE_ENUM_ID(snorm_10_10_10_2),
    S11N_DEFINE_ENUM_ID(sint_10_10_10_2),
    S11N_DEFINE_ENUM_ID(uint_32_32),
    S11N_DEFINE_ENUM_ID(sint_32_32),
    S11N_DEFINE_ENUM_ID(float_32_32),
    S11N_DEFINE_ENUM_ID(unorm_16_16_16_16),
    S11N_DEFINE_ENUM_ID(uint_16_16_16_16),
    S11N_DEFINE_ENUM_ID(snorm_16_16_16_16),
    S11N_DEFINE_ENUM_ID(sint_16_16_16_16),
    S11N_DEFINE_ENUM_ID(float_16_16_16_16),
    S11N_DEFINE_ENUM_ID(uint_to_float_16_16_16_16),
    S11N_DEFINE_ENUM_ID(sint_to_float_16_16_16_16),
    S11N_DEFINE_ENUM_ID(uint_32_32_32),
    S11N_DEFINE_ENUM_ID(sint_32_32_32),
    S11N_DEFINE_ENUM_ID(float_32_32_32),
    S11N_DEFINE_ENUM_ID(uint_32_32_32_32),
    S11N_DEFINE_ENUM_ID(sint_32_32_32_32),
    S11N_DEFINE_ENUM_ID(float_32_32_32_32)
);

enum
{
    attrib_none = 0x0,
    attrib_int = 0x1 << 0,
    attrib_int2 = 0x1 << 1,
    attrib_int3 = 0x1 << 3,
    attrib_int4 = 0x1 << 4,
    attrib_uint = 0x1 << 5,
    attrib_uint2 = 0x1 << 6,
    attrib_uint3 = 0x1 << 7,
    attrib_uint4 = 0x1 << 8,
    attrib_float = 0x1 << 9,
    attrib_float2 = 0x1 << 10,
    attrib_float3 = 0x1 << 11,
    attrib_float4 = 0x1 << 12
};

G3DIF_DEFINE_ENUM_CAST_TABLE(
    vtx_attrib, type, flag,
    attrib_int,
    attrib_int2,
    attrib_int3,
    attrib_int4,
    attrib_uint,
    attrib_uint2,
    attrib_uint3,
    attrib_uint4,
    attrib_float,
    attrib_float2,
    attrib_float3,
    attrib_float4
);

G3DIF_DEFINE_ENUM_CAST_TABLE(
    vtx_attrib, quantize_type, flag,
    attrib_none,
    attrib_float,
    attrib_uint,
    attrib_float,
    attrib_int,
    attrib_float,
    attrib_float,
    attrib_float2,
    attrib_float,
    attrib_uint,
    attrib_float,
    attrib_int,
    attrib_float,
    attrib_float,
    attrib_float,
    attrib_float2,
    attrib_uint2,
    attrib_float2,
    attrib_int2,
    attrib_float2,
    attrib_float2,
    attrib_uint,
    attrib_int,
    attrib_float,
    attrib_float2,
    attrib_uint2,
    attrib_float2,
    attrib_int2,
    attrib_float2,
    attrib_float2,
    attrib_float2,
    attrib_float3,
    attrib_float3 | attrib_float4,
    attrib_uint3 | attrib_uint4,
    attrib_float3 | attrib_float4,
    attrib_int3 | attrib_int4,
    attrib_float3 | attrib_float4,
    attrib_float3 | attrib_float4,
    attrib_float3 | attrib_float4, // unorm_10_10_10_2
    attrib_uint3 | attrib_uint4,   // uint_10_10_10_2
    attrib_float3 | attrib_float4, // snorm_10_10_10_2
    attrib_int3 | attrib_int4,     // sint_10_10_10_2
    attrib_uint2,
    attrib_int2,
    attrib_float2,
    attrib_float3 | attrib_float4,
    attrib_uint3 | attrib_uint4,
    attrib_float3 | attrib_float4,
    attrib_int3 | attrib_int4,
    attrib_float3 | attrib_float4,
    attrib_float3 | attrib_float4,
    attrib_float3 | attrib_float4,
    attrib_uint3,
    attrib_int3,
    attrib_float3,
    attrib_uint4,
    attrib_int4,
    attrib_float4
);

enum
{
    attrib_priority_0 = 0x0,
    attrib_priority_1 = 0x1 << 0,
    attrib_priority_2 = 0x1 << 1,
    attrib_priority_3 = 0x1 << 2,
    attrib_priority_4 = 0x1 << 3,
    attrib_priority_5 = 0x1 << 4,

    attrib_priority_unite = 0x1 << 5
};

G3DIF_DEFINE_ENUM_CAST_TABLE(
    vtx_attrib, quantize_type, priority,
    attrib_priority_0 | attrib_priority_unite,
    attrib_priority_4,
    attrib_priority_4,
    attrib_priority_4,
    attrib_priority_4,
    attrib_priority_4,
    attrib_priority_4,
    attrib_priority_5,
    attrib_priority_1,
    attrib_priority_1,
    attrib_priority_1,
    attrib_priority_1,
    attrib_priority_1,
    attrib_priority_1,
    attrib_priority_1,
    attrib_priority_4,
    attrib_priority_4,
    attrib_priority_4,
    attrib_priority_4,
    attrib_priority_4,
    attrib_priority_4,
    attrib_priority_0,
    attrib_priority_0,
    attrib_priority_0,
    attrib_priority_1,
    attrib_priority_1,
    attrib_priority_1,
    attrib_priority_1,
    attrib_priority_1,
    attrib_priority_1,
    attrib_priority_1,
    attrib_priority_2,
    attrib_priority_4,
    attrib_priority_4,
    attrib_priority_4,
    attrib_priority_4,
    attrib_priority_4,
    attrib_priority_4,
    attrib_priority_3,
    attrib_priority_3,
    attrib_priority_3,
    attrib_priority_3,
    attrib_priority_0,
    attrib_priority_0,
    attrib_priority_0,
    attrib_priority_1,
    attrib_priority_1,
    attrib_priority_1,
    attrib_priority_1,
    attrib_priority_1,
    attrib_priority_1,
    attrib_priority_1,
    attrib_priority_0,
    attrib_priority_0,
    attrib_priority_0,
    attrib_priority_0,
    attrib_priority_0,
    attrib_priority_0
);

elem_vtx_attrib::enum_type elem_vtx_attrib::GetType(std::string str)
{
    elem_vtx_attrib::enum_type type;
    Read(str.c_str(), type);
    return type;
}

std::string elem_vtx_attrib::GetType(elem_vtx_attrib::enum_type type)
{
    return elem_vtx_attrib::tbl_type[type].str;
}

elem_vtx_attrib::enum_quantize_type
elem_vtx_attrib::Compare(elem_vtx_attrib::enum_quantize_type lhs, elem_vtx_attrib::enum_quantize_type rhs)
{
    int lPriority = tbl_vtx_attrib_quantize_type_priority[lhs];
    int rPriority = tbl_vtx_attrib_quantize_type_priority[rhs];
    if (lPriority < rPriority)
    {
        return lhs;
    }
    return rhs;
}

elem_vtx_attrib::enum_semantic_name GetSemanticName(std::string hint)
{
    if (hint.find("position") != std::string::npos)
    {
        return elem_vtx_attrib::position;
    }
    else if (hint.find("normal") != std::string::npos)
    {
        return elem_vtx_attrib::normal;
    }
    else if (hint.find("tangent") != std::string::npos)
    {
        return elem_vtx_attrib::tangent;
    }
    else if (hint.find("binormal") != std::string::npos)
    {
        return elem_vtx_attrib::binormal;
    }
    else if (hint.find("color") != std::string::npos)
    {
        return elem_vtx_attrib::color;
    }
    else if (hint.find("blendindex") != std::string::npos)
    {
        return elem_vtx_attrib::blendindex;
    }
    else if (hint.find("blendweight") != std::string::npos)
    {
        return elem_vtx_attrib::blendweight;
    }
    else if (hint.find("texcoord") != std::string::npos)
    {
        return elem_vtx_attrib::texcoord;
    }

    return elem_vtx_attrib::generic;
}

//--------------------------------------------------------------------------------------------------

void elem_lod_offset::operator<<(const util::XMLElement* pElem)
{
    VerifyElement(pElem, Id());

    try
    {
        const char* textData = pElem->Text();
        std::vector<std::string> splits;
        util::Split(splits, textData, " ");
        lodOffset.reserve(splits.size());
        for (auto iter = splits.cbegin(); iter != splits.cend(); ++iter)
        {
            int value = 0;
            util::TryParse(value, *iter);
            lodOffset.emplace_back(value);
        }
    }
    CATCH_THROW_XML_ERROR()
}

void elem_vtx_attrib::operator<<(const util::XMLElement* pElem)
{
    VerifyElement(pElem, Id());

    try
    {
        name << pElem;
        hint << pElem;
        type << pElem;
        quantize_type << pElem;
        count << pElem;
        stream_index << pElem;
    }
    CATCH_THROW_XML_ERROR()
}

void elem_input::operator<<(const util::XMLElement* pElem)
{
    VerifyElement(pElem, Id());

    try
    {
        attrib_index << pElem;
    }
    CATCH_THROW_XML_ERROR()
}

void elem_vtx_buffer::operator<<(const util::XMLElement* pElem)
{
    VerifyElement(pElem, Id());

    try
    {
        input_array << pElem->Child(elem_input::IdArray());
    }
    CATCH_THROW_XML_ERROR()
}

void elem_vertex::operator<<(const util::XMLElement* pElem)
{
    VerifyElement(pElem, Id());

    try
    {
        lod_offset << pElem->Child(elem_lod_offset::Id());
        vtx_attrib_array << pElem->Child(elem_vtx_attrib::IdArray());
        vtx_buffer_array << pElem->Child(elem_vtx_buffer::IdArray());
    }
    CATCH_THROW_XML_ERROR()
}

void elem_vertex::CheckType()
{
    // 変換可能な量子化タイプが指定されているかを確認します。
    for (auto iterBuffer = vtx_buffer_array.cbegin(); iterBuffer != vtx_buffer_array.cend(); ++iterBuffer)
    {
        for (auto iter = iterBuffer->input_array.cbegin(); iter != iterBuffer->input_array.cend(); ++iter)
        {
            if (iter->attrib_index.value >= static_cast<int>(vtx_attrib_array.size ()))
            {
                THROW_ERROR(ERRCODE_XML_OUT_OF_RANGE,
                    "Out of range. attrib_index: %d vtx_attrib_array.size(): %d",
                    iter->attrib_index.value, vtx_attrib_array.size());
            }
        }
    }

    for (auto iterAttrib = vtx_attrib_array.cbegin(); iterAttrib != vtx_attrib_array.cend(); ++iterAttrib)
    {
        if (iterAttrib->quantize_type.value != elem_vtx_attrib::qtype_none)
        {
            if (!(ToEnum_flag(iterAttrib->quantize_type.value) & ToEnum_flag(iterAttrib->type.value)))
            {
                THROW_ERROR(ERRCODE_XML_INVALID_QUANTIZE_TYPE,
                    "Invalid quantize type. Vertex attrib index is %d.", iterAttrib->index);
            }
        }
    }
}

void elem_vertex::PostProcess(std::vector<elem_stream>& stream_array)
{
    int stream_array_size = static_cast<int>(stream_array.size());

    for (auto iter = vtx_attrib_array.begin(); iter != vtx_attrib_array.end(); ++iter)
    {
        if (stream_array_size <= iter->stream_index.value)
        {
            THROW_ERROR(ERRCODE_XML_OUT_OF_RANGE,
                "Out of range. stream_array: %d vtx_attrib_array->stream_index: %d",
                stream_array_size, iter->stream_index.value);
        }

        // テキストデータを解析します。
        elem_stream& xmlStream = stream_array[iter->stream_index.value];
        iter->stream.rawdata = AnalizeAndCopyData(xmlStream.textData, xmlStream.count.value, xmlStream.GetStreamType());
        iter->stream.count = xmlStream.count.value;
        iter->stream.type = static_cast<StreamType>(xmlStream.type.value);

        // セマンティックを入れておきます。
        iter->semantic_name = GetSemanticName(iter->hint.value);
    }

    CheckType();
}
void elem_vertex::PostBinaryProcess(StreamArray& streamArray)
{
    int stream_array_size = static_cast<int>(streamArray.streamChunk.size());
    for (auto iter = vtx_attrib_array.begin(); iter != vtx_attrib_array.end(); ++iter)
    {
        if (stream_array_size <= iter->stream_index.value)
        {
            THROW_ERROR(ERRCODE_XML_OUT_OF_RANGE,
                "Out of range. stream_array: %d vtx_attrib_array->stream_index: %d",
                stream_array_size, iter->stream_index.value);
        }

        void* rawdata = CopyRawData(streamArray.streamChunk[iter->stream_index.value]);
        iter->stream.rawdata.reset(rawdata, free);
        iter->stream.count = streamArray.streamChunk[iter->stream_index.value].count;
        iter->stream.type = streamArray.streamChunk[iter->stream_index.value].type;

        // セマンティックを入れておきます。
        iter->semantic_name = GetSemanticName(iter->hint.value);
    }

    CheckType();
}

} // namespace g3dif

} // namespace tool
} // namespace g3d
} // namespace nw
