﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <g3dif/UserData.h>

namespace nw { namespace g3d { namespace tool {
namespace g3dif {

void elem_user_int::operator<<(const util::XMLElement* pElem)
{
    VerifyElement(pElem, Id());

    try
    {
        count << pElem;

        // データへのポインタを入れておく
        textData = pElem->Text();
    }
    CATCH_THROW_XML_ERROR()
}

void elem_user_float::operator<<(const util::XMLElement* pElem)
{
    VerifyElement(pElem, Id());

    try
    {
        count << pElem;

        // データへのポインタを入れておく
        textData = pElem->Text();
    }
    CATCH_THROW_XML_ERROR()
}

void elem_string::operator<<(const util::XMLElement* pElem)
{
    VerifyElement(pElem, Id());

    try
    {
        textData = pElem->Text();
    }
    CATCH_THROW_XML_ERROR()
}

void elem_user_string::operator<<(const util::XMLElement* pElem)
{
    VerifyElement(pElem, Id());

    try
    {
        count << pElem;
        string_array << pElem;
    }
    CATCH_THROW_XML_ERROR()
}

void elem_wstring::operator<<(const util::XMLElement* pElem)
{
    VerifyElement(pElem, Id());

    try
    {
        textData = pElem->Text();
    }
    CATCH_THROW_XML_ERROR()
}

void elem_user_wstring::operator<<(const util::XMLElement* pElem)
{
    VerifyElement(pElem, Id());

    try
    {
        count << pElem;
        wstring_array << pElem;
    }
    CATCH_THROW_XML_ERROR()
}

void elem_user_stream::operator<<(const util::XMLElement* pElem)
{
    VerifyElement(pElem, Id());

    try
    {
        size << pElem;
        stream_index << pElem;
    }
    CATCH_THROW_XML_ERROR()
}

void elem_user_data::operator<<(const util::XMLElement* pElem)
{
    VerifyElement(pElem, Id());

    try
    {
        name << pElem;
        user_int << pElem->Child(elem_user_int::Id());
        user_float << pElem->Child(elem_user_float::Id());
        user_string << pElem->Child(elem_user_string::Id());
        user_wstring << pElem->Child(elem_user_wstring::Id());
        user_stream << pElem->Child(elem_user_stream::Id());

        int dataCount = 0;
        if (user_int)
        {
            ++dataCount;
            type = StreamTypeInt;
        }
        if (user_float)
        {
            ++dataCount;
            type = StreamTypeFloat;
        }
        if (user_string)
        {
            ++dataCount;
            type = StreamTypeString;
            if (user_string->count.value != static_cast<int>(user_string->string_array.size()))
            {
                THROW_ERROR(ERRCODE_XML_INVALID_USERDATA,
                    "Invalid user_string(%hs). user_string count: %d, string count: %d.",
                    name.value.c_str(), user_string->count.value, user_string->string_array.size());
            }
        }
        if (user_wstring)
        {
            ++dataCount;
            type = StreamTypeWString;
            if (user_wstring->count. value != static_cast<int>(user_wstring->wstring_array.size()))
            {
                THROW_ERROR(ERRCODE_XML_INVALID_USERDATA,
                    "Invalid user_wstring(%hs). user_wstring count: %d, wstring count: %d.",
                    name.value.c_str(), user_wstring->count.value, user_wstring->wstring_array.size());
            }
        }

        if (user_stream)
        {
            ++dataCount;
            type = StreamTypeByte;
        }

        if (dataCount >= 2)
        {
            THROW_ERROR(ERRCODE_XML_INVALID_USERDATA, "user_data have only one type member.");
        }
    }
    CATCH_THROW_XML_ERROR()
}

void elem_user_data::PostProcess(std::vector<elem_stream>& stream_array)
{
    if (user_int)
    {
        // テキストデータを解析します。
        user_int->rawdata = AnalizeAndCopyData(user_int->textData, user_int->count.value, type);
    }

    if (user_float)
    {
        // テキストデータを解析します。
        user_float->rawdata = AnalizeAndCopyData(user_float->textData, user_float->count.value, type);
    }

    if (user_string)
    {
        for (int i = 0; i < user_string->count.value; ++i)
        {
            elem_string& el_string = user_string->string_array[i];
            el_string.stream = std::string(static_cast<char*>(AnalizeAndCopyData(el_string.textData, 1, StreamTypeString).get()));
        }
    }

    if (user_wstring)
    {

        for (int i = 0; i < user_wstring->count.value; ++i)
        {
            elem_wstring& el_wstring = user_wstring->wstring_array[i];
            el_wstring.stream = std::wstring(static_cast<wchar_t*>(AnalizeAndCopyData(el_wstring.textData, 1, StreamTypeWString).get()));
        }
    }

    if (user_stream)
    {
        int stream_array_size = static_cast<int>(stream_array.size());

        if (stream_array_size <= user_stream->stream_index.value)
        {
            THROW_ERROR(ERRCODE_XML_OUT_OF_RANGE,
                "Out of range. stream_array: %d user_data->use_int->stream_index: %d",
                stream_array_size, user_stream->stream_index.value);
        }

        // テキストデータを解析します。
        elem_stream& xmlStream = stream_array[user_stream->stream_index.value];
        user_stream->stream.rawdata = AnalizeAndCopyData(xmlStream.textData, xmlStream.count.value, xmlStream.GetStreamType());
        user_stream->stream.count = xmlStream.count.value;
        user_stream->stream.type = static_cast<StreamType>(xmlStream.type.value);

        if (user_stream->size.value != user_stream->stream.count)
        {
            THROW_ERROR(ERRCODE_XML_INVALID_USERDATA,
                "Invalid user_stream user_stream count: %d, stream count: %d",
                user_stream->size.value, user_stream->stream.count);
        }
    }
}

void elem_user_data::PostBinaryProcess(StreamArray& streamArray)
{
    if (user_int)
    {
        // テキストデータを解析します。
        user_int->rawdata = AnalizeAndCopyData(user_int->textData, user_int->count.value, type);
    }

    if (user_float)
    {
        // テキストデータを解析します。
        user_float->rawdata = AnalizeAndCopyData(user_float->textData, user_float->count.value, type);
    }

    if (user_string)
    {
        for (int i = 0; i < user_string->count.value; ++i)
        {
            elem_string& el_string = user_string->string_array[i];
            el_string.stream = std::string(static_cast<char*>(AnalizeAndCopyData(el_string.textData, 1, StreamTypeString).get()));
        }
    }

    if (user_wstring)
    {

        for (int i = 0; i < user_wstring->count.value; ++i)
        {
            elem_wstring& el_wstring = user_wstring->wstring_array[i];
            el_wstring.stream = std::wstring(static_cast<wchar_t*>(AnalizeAndCopyData(el_wstring.textData, 1, StreamTypeWString).get()));
        }
    }

    if (user_stream)
    {
        int stream_array_size = static_cast<int>(streamArray.streamChunk.size());
        if (stream_array_size <= user_stream->stream_index.value)
        {
            THROW_ERROR(ERRCODE_XML_OUT_OF_RANGE,
                "Out of range. stream_array: %d user_data->use_int->stream_index: %d",
                stream_array_size, user_stream->stream_index.value);
        }

        void* rawdata = CopyRawData(streamArray.streamChunk[user_stream->stream_index.value]);
        user_stream->stream.rawdata.reset(rawdata, free);
        user_stream->stream.count = streamArray.streamChunk[user_stream->stream_index.value].count;
        user_stream->stream.type = streamArray.streamChunk[user_stream->stream_index.value].type;

        if (user_stream->size.value != user_stream->stream.count)
        {
            THROW_ERROR(ERRCODE_XML_INVALID_USERDATA,
                "Invalid user_stream user_stream count: %d, stream count: %d",
                user_stream->size.value, user_stream->stream.count);
        }
    }
}

} // namespace g3dif

} // namespace tool
} // namespace g3d
} // namespace nw
