﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <g3dif/ShaderDefinition.h>
#include <algorithm>
#include <fstream>
#include <shlwapi.h>

#include <util/UtilBit.h>
#include <util/UtilFile.h>

namespace nw { namespace g3d { namespace tool {
namespace g3dif {

G3DIF_DEFINE_ENUM_TABLE(
    option_var, type,
    S11N_DEFINE_ENUM_ID(static),
    S11N_DEFINE_ENUM_ID(dynamic)
    );

G3DIF_DEFINE_ENUM_TABLE(
    sampler_var, type,
    S11N_DEFINE_ENUM_ID(isampler_1d),
    S11N_DEFINE_ENUM_ID(isampler_2d),
    S11N_DEFINE_ENUM_ID(isampler_2drect),
    S11N_DEFINE_ENUM_ID(isampler_3d),
    S11N_DEFINE_ENUM_ID(isampler_cube),
    S11N_DEFINE_ENUM_ID(isampler_buffer),
    S11N_DEFINE_ENUM_ID(isampler_2dms),
    S11N_DEFINE_ENUM_ID(isampler_1d_array),
    S11N_DEFINE_ENUM_ID(isampler_2d_array),
    S11N_DEFINE_ENUM_ID(isampler_cube_array),
    S11N_DEFINE_ENUM_ID(isampler_2dms_array),
    S11N_DEFINE_ENUM_ID(usampler_1d),
    S11N_DEFINE_ENUM_ID(usampler_2d),
    S11N_DEFINE_ENUM_ID(usampler_2drect),
    S11N_DEFINE_ENUM_ID(usampler_3d),
    S11N_DEFINE_ENUM_ID(usampler_cube),
    S11N_DEFINE_ENUM_ID(usampler_buffer),
    S11N_DEFINE_ENUM_ID(usampler_2dms),
    S11N_DEFINE_ENUM_ID(usampler_1d_array),
    S11N_DEFINE_ENUM_ID(usampler_2d_array),
    S11N_DEFINE_ENUM_ID(usampler_cube_array),
    S11N_DEFINE_ENUM_ID(usampler_2dms_array),
    S11N_DEFINE_ENUM_ID(sampler_1d),
    S11N_DEFINE_ENUM_ID(sampler_2d),
    S11N_DEFINE_ENUM_ID(sampler_2drect),
    S11N_DEFINE_ENUM_ID(sampler_3d),
    S11N_DEFINE_ENUM_ID(sampler_cube),
    S11N_DEFINE_ENUM_ID(sampler_buffer),
    S11N_DEFINE_ENUM_ID(sampler_2dms),
    S11N_DEFINE_ENUM_ID(sampler_1d_array),
    S11N_DEFINE_ENUM_ID(sampler_2d_array),
    S11N_DEFINE_ENUM_ID(sampler_cube_array),
    S11N_DEFINE_ENUM_ID(sampler_2dms_array),
    S11N_DEFINE_ENUM_ID(sampler_1d_shadow),
    S11N_DEFINE_ENUM_ID(sampler_2d_shadow),
    S11N_DEFINE_ENUM_ID(sampler_2drect_shadow),
    S11N_DEFINE_ENUM_ID(sampler_cube_shadow),
    S11N_DEFINE_ENUM_ID(sampler_1d_array_shadow),
    S11N_DEFINE_ENUM_ID(sampler_2d_array_shadow),
    S11N_DEFINE_ENUM_ID(sampler_cube_array_shadow)
    );

G3DIF_DEFINE_ENUM_TABLE(
    block_var, type,
    S11N_DEFINE_ENUM_ID(none),
    S11N_DEFINE_ENUM_ID(material),
    S11N_DEFINE_ENUM_ID(shape),
    S11N_DEFINE_ENUM_ID(skeleton),
    S11N_DEFINE_ENUM_ID(option)
    );

G3DIF_DEFINE_ENUM_TABLE(
    ssbo_block_var, type,
    S11N_DEFINE_ENUM_ID(none),
    S11N_DEFINE_ENUM_ID(material),
    S11N_DEFINE_ENUM_ID(shape),
    S11N_DEFINE_ENUM_ID(skeleton),
    S11N_DEFINE_ENUM_ID(option)
    );

G3DIF_DEFINE_ENUM_TABLE(
    render_info_slot, type,
    S11N_DEFINE_ENUM_ID(int),
    S11N_DEFINE_ENUM_ID(float),
    S11N_DEFINE_ENUM_ID(string),
    );

namespace
{

const std::string PARENTHESIS("( )");
const std::string SYSTEM_ID = "system_id";
const std::string NW_SYSTEM_MACRO = "NW_SYSTEM_MACRO";

#define TAG_END_LINE util::CRLF
#define TAG_OPEN(elem, indent) Indent(indent) << "<" << elem
#define TAG_CLOSE " />" << TAG_END_LINE;
#define TAG_CLOSE_LINE "/>" << TAG_END_LINE;
#define TAG_START ">" << TAG_END_LINE;
#define TAG_END(elem, indent) Indent(indent) << "</" << elem << ">" << TAG_END_LINE;
#define TAG_ARRAY(elem) elem << "_array"

const util::StringRef INDENT_MAX                 = "\t\t\t\t\t\t\t\t\t\t\t\t\t\t\t\t"; // とりあえず。必要なら増やす。
const util::StringRef str_wild_card              = "*";

struct Indent
{
    Indent(int indent) : indent(indent) {}

    int indent;
};

const int INDENT_0(0);
const int INDENT_1(1);
const int INDENT_2(2);
const int INDENT_3(3);
const int INDENT_4(4);
const int INDENT_5(5);
const int INDENT_6(6);

inline std::ostream& operator<<(std::ostream& lhs, const Indent& rhs)
{
    if(rhs.indent > sizeof(INDENT_MAX))
    {
        THROW_ERROR_INTERNAL(ERRCODE_INTERNAL, "Internal error.");
    }

    lhs.write(INDENT_MAX.Data(), rhs.indent);
    return lhs;
}

template <typename T>
struct AttrManip
{
    AttrManip(int indent, const char* name, const T& value)
        : indent(indent), name(name), value(value)
    {
    }

    int indent;
    const char* name;
    const T& value;

private:
    AttrManip operator =( const AttrManip & );
};

template <typename T>
inline
std::ostream& operator<<(std::ostream& lhs, const AttrManip<T>& rhs)
{
    if (rhs.indent > 0)
    {
        lhs << TAG_END_LINE;
        lhs << Indent(rhs.indent);
    }
    else
    {
        lhs << ' ';
    }
    lhs << rhs.name << '=' << '"' << rhs.value << '"';
    return lhs;
}

template <typename T>
inline AttrManip<T> Attr(int indent, const char* name, const T& value)
{
    return AttrManip<T>(indent, name, value);
}

class SystemOptionFinder
{
public:
    SystemOptionFinder() {}

    bool operator()(const elem_option_var& option)
    {
        return option.id.value == SYSTEM_ID;
    }
};

class OptionUniformFinder
{
public:
    OptionUniformFinder(util::StringRef id)
        : m_Id(id)
    {}

    bool operator()(const elem_uniform_var& uniform)
    {
        return uniform.id.value == m_Id;
    }

private:
    util::StringRef m_Id;
};

class ChoiceFinder
{
public:
    ChoiceFinder(const std::string& target)
        : m_Target(target)
    {}

    bool operator()(const std::pair<std::string, std::string>& choice)
    {
        return choice.first == m_Target;
    }
private:
    std::string m_Target;
};

} // anonymous namespace

elem_sampler_var::enum_type elem_sampler_var::GetType(std::string str)
{
    elem_sampler_var::enum_type type;
    Read(str.c_str(), type);
    return type;
}

elem_option_var::enum_type elem_option_var::GetType(std::string str)
{
    elem_option_var::enum_type type;
    Read(str.c_str(), type);
    return type;
}

elem_block_var::enum_type elem_block_var::GetType(std::string str)
{
    elem_block_var::enum_type type;
    Read(str.c_str(), type);
    return type;
}

elem_ssbo_block_var::enum_type elem_ssbo_block_var::GetType(std::string str)
{
    elem_ssbo_block_var::enum_type type;
    Read(str.c_str(), type);
    return type;
}

elem_render_info_slot::enum_type elem_render_info_slot::GetType(std::string str)
{
    elem_render_info_slot::enum_type type;
    Read(str.c_str(), type);
    return type;
}

bool elem_ssbo_block_var::IsValidType( elem_ssbo_block_var::enum_type type )
{
    if( ( type != nw::g3d::tool::g3dif::elem_ssbo_block_var::none ) &&
        ( type != nw::g3d::tool::g3dif::elem_ssbo_block_var::skeleton ) )
    {
        return false;
    }
    return true;
}

std::string elem_uniform_var::GetDefault(elem_shader_param::enum_type type)
{
    static const char* fMatDefault[] = {
        "1.0", "0.0", "0.0", "0.0",
        "0.0", "1.0", "0.0", "0.0",
        "0.0", "0.0", "1.0", "0.0",
        "0.0", "0.0", "0.0", "1.0"
    };

    static const char* fVecDefault[] = {"0.0", "0.0", "0.0", "1.0"};

    static const char* iVecDefault[] = {"0", "0", "0", "1"};

    static const char* srt3dDefault[] = {"1.0", "1.0", "1.0", "0.0", "0.0", "0.0", "0.0", "0.0", "0.0"};

    static const char* srt2dDefault[] = {"1.0", "1.0", "0.0", "0.0", "0.0"};

    static const char* texsrtDefault[] = {"0.0", "1.0", "1.0", "0.0", "0.0", "0.0"};

    std::string str;
    if (type <= elem_shader_param::type_uint4)
    {
        for (int i = 0; i < GetDefaultCount(type); ++i)
        {
            str.append(iVecDefault[i]);
            str.append(" ");
        }
        return util::TrimRight(str);
    }
    else if (type <= elem_shader_param::type_float4)
    {
        for (int i = 0; i < GetDefaultCount(type); ++i)
        {
            str.append(fVecDefault[i]);
            str.append(" ");
        }
        return util::TrimRight(str);
    }
    else if (type <= elem_shader_param::type_float4x4)
    {
        int offsetType = (type - elem_shader_param::type_float2x2);
        for (int i = 0; i < offsetType / 3 + 2; ++i)
        {
            for (int j = 0; j < offsetType % 3 + 2; ++j)
            {
                str.append(fMatDefault[i * 3 + j]);
                str.append(" ");
            }
        }
        return util::TrimRight(str);
    }
    else if (type == elem_shader_param::type_srt2d)
    {
        for (int i = 0; i < GetDefaultCount(type); ++i)
        {
            str.append(srt2dDefault[i]);
            str.append(" ");
        }
        return util::TrimRight(str);
    }
    else if (type == elem_shader_param::type_srt3d)
    {
        for (int i = 0; i < GetDefaultCount(type); ++i)
        {
            str.append(srt3dDefault[i]);
            str.append(" ");
        }
        return util::TrimRight(str);
    }
    else
    {
        // texsrt 系
        for (int i = 0; i < GetDefaultCount(type); ++i)
        {
            str.append(texsrtDefault[i]);
            str.append(" ");
        }
        return util::TrimRight(str);
    }
}

std::string elem_ssbo_uniform_var::GetDefault(elem_shader_param::enum_type type)
{
    return elem_uniform_var::GetDefault(type);
}

int elem_uniform_var::GetDefaultCount(elem_shader_param::enum_type type)
{
    if (type <= elem_shader_param::type_float4)
    {
        return type % 4 + 1;
    }
    else if (type <= elem_shader_param::type_float4x4)
    {
        int offsetType = (type - elem_shader_param::type_float2x2);
        return (offsetType / 3 + 2) * (offsetType % 3 + 2);
    }
    else if (type == elem_shader_param::type_srt2d)
    {
        return 5; // s(2) + r(1) + t(2)
    }
    else if (type == elem_shader_param::type_srt3d)
    {
        return 9; // s(3) + r(3) + t(3)
    }
    else
    {
        return 6; // mode(1) + s(2) + r(1) + t(2)
    }
}

int elem_ssbo_uniform_var::GetDefaultCount(elem_shader_param::enum_type type)
{
    return elem_uniform_var::GetDefaultCount(type);
}

//--------------------------------------------------------------------------------------------------

void elem_shader_symbol::operator<<(const util::XMLElement* pElem)
{
    VerifyElement(pElem, Id());

    try
    {
        name << pElem;
    }

    CATCH_THROW_XML_ERROR()
}

void elem_ui_label::operator<<(const util::XMLElement* pElem)
{
    VerifyElement(pElem, Id());

    try
    {
        value << pElem;
    }

    CATCH_THROW_XML_ERROR()
}

void elem_ui_comment::operator<<(const util::XMLElement* pElem)
{
    VerifyElement(pElem, Id());

    try
    {
        value << pElem;
    }

    CATCH_THROW_XML_ERROR()
}

void elem_ui_group::operator<<(const util::XMLElement* pElem)
{
    VerifyElement(pElem, Id());

    try
    {
        group_name << pElem;
    }

    CATCH_THROW_XML_ERROR()
}

void elem_ui_order::operator<<(const util::XMLElement* pElem)
{
    VerifyElement(pElem, Id());

    try
    {
        value << pElem;
    }

    CATCH_THROW_XML_ERROR()
}

void elem_ui_alias::operator<<(const util::XMLElement* pElem)
{
    VerifyElement(pElem, Id());

    try
    {
        value << pElem;
    }

    CATCH_THROW_XML_ERROR()
}

void elem_ui_item::operator<<(const util::XMLElement* pElem)
{
    VerifyElement(pElem, Id());

    try
    {
        value << pElem;
    }

    CATCH_THROW_XML_ERROR()
}

void elem_ui_min::operator<<(const util::XMLElement* pElem)
{
    VerifyElement(pElem, Id());

    try
    {
        value << pElem;
    }

    CATCH_THROW_XML_ERROR()
}

void elem_ui_max::operator<<(const util::XMLElement* pElem)
{
    VerifyElement(pElem, Id());

    try
    {
        value << pElem;
    }

    CATCH_THROW_XML_ERROR()
}

void elem_ui_visible::operator<<(const util::XMLElement* pElem)
{
    VerifyElement(pElem, Id());

    try
    {
        value << pElem;
    }

    CATCH_THROW_XML_ERROR()
}

void elem_ui_editable::operator<<(const util::XMLElement* pElem)
{
    VerifyElement(pElem, Id());

    try
    {
        value << pElem;
    }

    CATCH_THROW_XML_ERROR()
}

void elem_ui_anim_editable::operator<<(const util::XMLElement* pElem)
{
    VerifyElement(pElem, Id());

    try
    {
        value << pElem;
    }

    CATCH_THROW_XML_ERROR()
}

void elem_option_var::operator<<(const util::XMLElement* pElem)
{
    VerifyElement(pElem, Id());

    try
    {
        id << pElem;
        type << pElem;
        choice << pElem;
        default_value << pElem;
        branch << pElem;
        for (int stageIndex = 0; stageIndex < ShaderStage_StageCount; ++stageIndex)
        {
            try
            {
                shader_symbol[stageIndex] << pElem->Child(elem_shader_symbol::Id(stageIndex));
            }
            catch (nw::g3d::tool::util::Exception&)
            {
                PRINT_ERROR_LOG("<%hs>", elem_shader_symbol::Id(stageIndex).str);
                PRINT_TRACE();
                throw;
            }
        }
        ui_label << pElem->Child(elem_ui_label::Id());
        ui_comment << pElem->Child(elem_ui_comment::Id());
        ui_group << pElem->Child(elem_ui_group::Id());
        ui_order << pElem->Child(elem_ui_order::Id());
        ui_alias << pElem->Child(elem_ui_alias::Id());
    }

    CATCH_THROW_XML_ERROR()
}

void elem_attrib_var::operator<<(const util::XMLElement* pElem)
{
    VerifyElement(pElem, Id());

    try
    {
        id << pElem;
        hint << pElem;
        type << pElem;
        vertex_symbol << pElem->Child(elem_shader_symbol::Id(ShaderStage_Vertex));
        ui_label << pElem->Child(elem_ui_label::Id());
        ui_comment << pElem->Child(elem_ui_comment::Id());
        ui_group << pElem->Child(elem_ui_group::Id());
        ui_editable << pElem->Child(elem_ui_editable::Id());
    }

    CATCH_THROW_XML_ERROR()
}

void elem_sampler_var::operator<<(const util::XMLElement* pElem)
{
    VerifyElement(pElem, Id());

    try
    {
        id << pElem;
        hint << pElem;
        type << pElem;
        alt << pElem;
        nostrip << pElem;
        for (int stageIndex = 0; stageIndex < ShaderStage_StageCount; ++stageIndex)
        {
            try
            {
                shader_symbol[stageIndex] << pElem->Child(elem_shader_symbol::Id(stageIndex));
            }
            catch (nw::g3d::tool::util::Exception&)
            {
                PRINT_ERROR_LOG("<%hs>", elem_shader_symbol::Id(stageIndex).str);
                PRINT_TRACE();
                throw;
            }
        }
        ui_label << pElem->Child(elem_ui_label::Id());
        ui_comment << pElem->Child(elem_ui_comment::Id());
        ui_group << pElem->Child(elem_ui_group::Id());
        ui_order << pElem->Child(elem_ui_order::Id());
        ui_visible << pElem->Child(elem_ui_visible::Id());
    }

    CATCH_THROW_XML_ERROR()
}

void elem_uniform_var::operator<<(const util::XMLElement* pElem)
{
    VerifyElement(pElem, Id());

    try
    {
        id << pElem;
        hint << pElem;
        type << pElem;
        default_value << pElem;
        nostrip << pElem;
        converter << pElem;
        depend << pElem;
        for (int stageIndex = 0; stageIndex < ShaderStage_StageCount; ++stageIndex)
        {
            try
            {
                shader_symbol[stageIndex] << pElem->Child(elem_shader_symbol::Id(stageIndex));
            }
            catch (nw::g3d::tool::util::Exception&)
            {
                PRINT_ERROR_LOG("<%hs>", elem_shader_symbol::Id(stageIndex).str);
                PRINT_TRACE();
                throw;
            }
        }
        ui_label << pElem->Child(elem_ui_label::Id());
        ui_comment << pElem->Child(elem_ui_comment::Id());
        ui_group << pElem->Child(elem_ui_group::Id());
        ui_order << pElem->Child(elem_ui_order::Id());
        ui_item << pElem->Child(elem_ui_item::Id());
        ui_min << pElem->Child(elem_ui_min::Id());
        ui_max << pElem->Child(elem_ui_max::Id());
        ui_visible << pElem->Child(elem_ui_visible::Id());
    }

    CATCH_THROW_XML_ERROR()
}

void elem_ssbo_uniform_var::operator<<(const util::XMLElement* pElem)
{
    VerifyElement(pElem, Id());

    try
    {
        id << pElem;
        nostrip << pElem;
        for (int stageIndex = 0; stageIndex < ShaderStage_StageCount; ++stageIndex)
        {
            try
            {
                shader_symbol[stageIndex] << pElem->Child(elem_shader_symbol::Id(stageIndex));
            }
            catch (nw::g3d::tool::util::Exception&)
            {
                PRINT_ERROR_LOG("<%hs>", elem_shader_symbol::Id(stageIndex).str);
                PRINT_TRACE();
                throw;
            }
        }
        ui_label << pElem->Child(elem_ui_label::Id());
        ui_comment << pElem->Child(elem_ui_comment::Id());
        ui_group << pElem->Child(elem_ui_group::Id());
        ui_order << pElem->Child(elem_ui_order::Id());
        ui_item << pElem->Child(elem_ui_item::Id());
        ui_min << pElem->Child(elem_ui_min::Id());
        ui_max << pElem->Child(elem_ui_max::Id());
        ui_visible << pElem->Child(elem_ui_visible::Id());
    }

    CATCH_THROW_XML_ERROR()
}

void elem_block_var::operator<<(const util::XMLElement* pElem)
{
    VerifyElement(pElem, Id());

    try
    {
        id << pElem;
        type << pElem;
        for (int stageIndex = 0; stageIndex < ShaderStage_StageCount; ++stageIndex)
        {
            try
            {
                shader_symbol[stageIndex] << pElem->Child(elem_shader_symbol::Id(stageIndex));
            }
            catch (nw::g3d::tool::util::Exception&)
            {
                PRINT_ERROR_LOG("<%hs>", elem_shader_symbol::Id(stageIndex).str);
                PRINT_TRACE();
                throw;
            }
        }
        uniform_var_array << pElem->Child(elem_uniform_var::IdArray());
    }

    CATCH_THROW_XML_ERROR()
}

void elem_ssbo_block_var::operator<<(const util::XMLElement* pElem)
{
    VerifyElement(pElem, Id());

    try
    {
        id << pElem;
        type << pElem;
        for (int stageIndex = 0; stageIndex < ShaderStage_StageCount; ++stageIndex)
        {
            try
            {
                shader_symbol[stageIndex] << pElem->Child(elem_shader_symbol::Id(stageIndex));
            }
            catch (nw::g3d::tool::util::Exception&)
            {
                PRINT_ERROR_LOG("<%hs>", elem_shader_symbol::Id(stageIndex).str);
                PRINT_TRACE();
                throw;
            }
        }
        uniform_var_array << pElem->Child(elem_ssbo_uniform_var::IdArray());
    }

    CATCH_THROW_XML_ERROR()
}

void elem_render_info_slot::operator<<(const util::XMLElement* pElem)
{
    VerifyElement(pElem, Id());

    try
    {
        name << pElem;
        type << pElem;
        count << pElem;
        optional << pElem;
        choice << pElem;
        default_value << pElem;
        ui_label << pElem->Child(elem_ui_label::Id());
        ui_comment << pElem->Child(elem_ui_comment::Id());
        ui_group << pElem->Child(elem_ui_group::Id());
        ui_order << pElem->Child(elem_ui_order::Id());
        ui_item << pElem->Child(elem_ui_item::Id());
        ui_visible << pElem->Child(elem_ui_visible::Id());
    }

    CATCH_THROW_XML_ERROR()
}

void elem_textblock::operator<<(const util::XMLElement* pElem)
{
    VerifyElement(pElem, Id());

    try
    {
        id << pElem;
        text << pElem;
        ui_group << pElem->Child(elem_ui_group::Id());
        ui_order << pElem->Child(elem_ui_order::Id());
    }

    CATCH_THROW_XML_ERROR()
}

void elem_interleave::operator<<(const util::XMLElement* pElem)
{
    VerifyElement(pElem, Id());

    try
    {
        id_set << pElem;
        binarize << pElem;
    }

    CATCH_THROW_XML_ERROR()
}

void elem_group::operator<<(const util::XMLElement* pElem)
{
    VerifyElement(pElem, Id());

    try
    {
        name << pElem;
        condition << pElem;
        ui_label << pElem->Child(elem_ui_label::Id());
        ui_comment << pElem->Child(elem_ui_comment::Id());
        ui_order << pElem->Child(elem_ui_order::Id());
        page_name << pElem;
    }

    CATCH_THROW_XML_ERROR()
}

void elem_page::operator<<(const util::XMLElement* pElem)
{
    VerifyElement(pElem, Id());

    try
    {
        name << pElem;
        ui_label << pElem->Child(elem_ui_label::Id());
        ui_comment << pElem->Child(elem_ui_comment::Id());
        ui_order << pElem->Child(elem_ui_order::Id());
    }

    CATCH_THROW_XML_ERROR()
}

void elem_shader_stage::operator<<(const util::XMLElement* pElem)
{
    VerifyElement(pElem, Id());

    try
    {
        src_index << pElem;
    }
    CATCH_THROW_XML_ERROR()
}

void elem_shader_src::operator<<(const util::XMLElement* pElem)
{
    VerifyElement(pElem, Id());

    try
    {
        include_path << pElem;
        path << pElem;
        stream_index << pElem;
    }
    CATCH_THROW_XML_ERROR()
}

void elem_streamout::operator<<(const util::XMLElement* pElem)
{
    VerifyElement(pElem, Id());

    try
    {
        varying << pElem;
    }
    CATCH_THROW_XML_ERROR()
}

void elem_shading_model::operator<<(const util::XMLElement* pElem)
{
    VerifyElement(pElem, Id());

    try
    {
        name << pElem;
        material_shader << pElem;
        revision << pElem;
        for (int stageIndex = 0; stageIndex < ShaderStage_StageCount; ++stageIndex)
        {
            try
            {
                shader_stage[stageIndex] << pElem->Child(elem_shader_stage::Id(stageIndex));
            }
            catch (nw::g3d::tool::util::Exception&)
            {
                PRINT_ERROR_LOG("<%hs>", elem_shader_stage::Id(stageIndex).str);
                PRINT_TRACE();
                throw;
            }
        }
        macro_array << pElem->Child(elem_macro::IdArray());
        option_var_array << pElem->Child(elem_option_var::IdArray());
        attrib_var_array << pElem->Child(elem_attrib_var::IdArray());
        sampler_var_array << pElem->Child(elem_sampler_var::IdArray());
        block_var_array << pElem->Child(elem_block_var::IdArray());
        ssbo_block_var_array << pElem->Child(elem_ssbo_block_var::IdArray());
        render_info_slot_array << pElem->Child(elem_render_info_slot::IdArray());
        textblock_array << pElem->Child(elem_textblock::IdArray());
        interleave_array << pElem->Child(elem_interleave::IdArray());
        group_array << pElem->Child(elem_group::IdArray());
        page_array << pElem->Child(elem_page::IdArray());
        streamout << pElem->Child(elem_streamout::Id());
    }
    CATCH_THROW_XML_ERROR()
}

void elem_shader_definition_info::operator<<(const util::XMLElement* pElem)
{
    VerifyElement(pElem, Id());

    try
    {
        code_page << pElem;
        config_path << pElem;
    }
    CATCH_THROW_XML_ERROR()
}

void elem_force_include::operator<<(const util::XMLElement* pElem)
{
    VerifyElement(pElem, Id());

    try
    {
        src_index << pElem;
    }
    CATCH_THROW_XML_ERROR()
}

void elem_shader_definition::operator<<(const util::XMLElement* pElem)
{
    VerifyElement(pElem, Id());

    try
    {
        version << pElem;
        shader_definition_info << pElem->Child(elem_shader_definition_info::Id());
        force_include_array << pElem->Child(elem_force_include::IdArray());
        shading_model_array << pElem->Child(elem_shading_model::IdArray());
        shader_src_array << pElem->Child(elem_shader_src::IdArray());
        stream_array << pElem->Child(elem_stream::IdArray());
    }
    CATCH_THROW_XML_ERROR()
}

void elem_shader_definition::PostProcess()
{
    try
    {
        int stream_array_size = static_cast<int>(stream_array.size());
        for (auto el_shader_src = shader_src_array.begin(); el_shader_src != shader_src_array.end(); ++el_shader_src)
        {
            if (stream_array_size <= el_shader_src->stream_index.value)
            {
                THROW_ERROR(ERRCODE_XML_OUT_OF_RANGE,
                    "Out of range. stream_array: %d shader_src.stream_index: %d",
                    stream_array_size, el_shader_src->stream_index.value);
            }
            // テキストデータを解析します。
            elem_stream& xmlStream = stream_array[el_shader_src->stream_index.value];
            el_shader_src->stream.rawdata = AnalizeAndCopyData(xmlStream.textData, xmlStream.count.value, xmlStream.GetStreamType());
            el_shader_src->stream.count = xmlStream.count.value;
            el_shader_src->stream.type = StreamTypeString;

            // テキストデータの改行コードを CRLF から LF に変換します。
            std::string data(static_cast<char*>(el_shader_src->stream.rawdata.get()));
            data = util::ReplaceAll(data, util::CRLF, util::LF);

            void* cbuf = malloc(data.size());
            memcpy(cbuf, data.data(), data.size());

            el_shader_src->stream.rawdata.reset(cbuf, free);
        }

        for (auto el_force_include = force_include_array.begin(); el_force_include != force_include_array.end(); ++el_force_include)
        {
            el_force_include->path = shader_src_array[el_force_include->src_index.value].path.value;
        }

        for (auto el_shading_model = shading_model_array.begin(); el_shading_model != shading_model_array.end(); ++el_shading_model)
        {
            el_shading_model->AddSystemOption();
            el_shading_model->Expand();
        }
    }
    CATCH_THROW_XML_ERROR()
}

void elem_shader_definition::PostBinaryProcess(void* data)
{
    try
    {
        StreamArray streamArray;
        AnalizeBinaryData(streamArray, data);

        int stream_array_size = static_cast<int>(streamArray.streamChunk.size());
        for (auto el_shader_src = shader_src_array.begin(); el_shader_src != shader_src_array.end(); ++el_shader_src)
        {
            if (stream_array_size <= el_shader_src->stream_index.value)
            {
                THROW_ERROR(ERRCODE_XML_OUT_OF_RANGE,
                    "Out of range. stream_array: %d shader_src.stream_index: %d",
                    stream_array_size, el_shader_src->stream_index.value);
            }

            void* rawdata = CopyRawData(streamArray.streamChunk[el_shader_src->stream_index.value]);

            std::wstring stream(static_cast<wchar_t*>(rawdata));

            // ワイド文字からマルチバイト文字へ変換する。
            int cLen = WideCharToMultiByte(CP_UTF8,
                0,
                stream.c_str(),
                -1,
                NULL,
                0,
                NULL,
                NULL);
            void* cbuf = malloc(cLen);
            WideCharToMultiByte(CP_UTF8,
                0,
                stream.c_str(),
                -1,
                static_cast<char*>(cbuf),
                cLen,
                NULL,
                NULL);

            el_shader_src->stream.rawdata.reset(cbuf, free);
            el_shader_src->stream.count = streamArray.streamChunk[el_shader_src->stream_index.value].count;
            el_shader_src->stream.type = StreamTypeString;

            // ワイド文字ストリームは解放する。
            free(rawdata);
        }

        for (auto el_force_include = force_include_array.begin(); el_force_include != force_include_array.end(); ++el_force_include)
        {
            el_force_include->path = shader_src_array[el_force_include->src_index.value].path.value;
        }

        for (auto el_shading_model = shading_model_array.begin(); el_shading_model != shading_model_array.end(); ++el_shading_model)
        {
            el_shading_model->AddSystemOption();
            el_shading_model->Expand();
        }
    }
    CATCH_THROW_XML_ERROR()
}

void elem_shader_definition::CheckData(int /*flag*/)
{
}

void elem_option_var::WriteStream(std::ostream& outStream, int indent) const
{
    for (int stageIndex = 0; stageIndex < ShaderStage_StageCount; ++stageIndex)
    {
        if (shader_symbol[stageIndex])
        {
            outStream << TAG_OPEN(g3dif::GetIdentifierName(elem_shader_symbol::GetIdentifier(stageIndex)), indent)
                << Attr(0, g3dif::GetIdentifierName(g3dif::id_name), shader_symbol[stageIndex]->name.value)
                << TAG_CLOSE;
        }
    }

    if (ui_label)
    {
        outStream << TAG_OPEN( g3dif::GetIdentifierName( g3dif::id_ui_label ), indent)
            << Attr(0, g3dif::GetIdentifierName( g3dif::id_value ), ui_label->value.value)
            << TAG_CLOSE;
    }

    if (ui_comment)
    {
        outStream << TAG_OPEN( g3dif::GetIdentifierName( g3dif::id_ui_comment ), indent)
            << Attr(0, g3dif::GetIdentifierName( g3dif::id_value ), ui_comment->value.value)
            << TAG_CLOSE;
    }

    if (ui_group)
    {
        outStream << TAG_OPEN( g3dif::GetIdentifierName( g3dif::id_ui_group ), indent)
            << Attr(0, g3dif::GetIdentifierName( g3dif::id_group_name ), ui_group->group_name.value)
            << TAG_CLOSE;
    }

    if (ui_order)
    {
        outStream << TAG_OPEN(g3dif::GetIdentifierName( g3dif::id_ui_order), indent)
            << Attr(0, g3dif::GetIdentifierName( g3dif::id_value ), ui_order->value.value)
            << TAG_CLOSE;
    }

    if (ui_alias)
    {
        outStream << TAG_OPEN(g3dif::GetIdentifierName( g3dif::id_ui_alias ), indent)
            << Attr(0, g3dif::GetIdentifierName( g3dif::id_value ), ui_alias->value.value)
            << TAG_CLOSE;
    }

    if (ui_visible)
    {
        outStream << TAG_OPEN(g3dif::GetIdentifierName( g3dif::id_ui_visible ), indent)
            << Attr(0, g3dif::GetIdentifierName( g3dif::id_value ), ui_visible->value.value)
            << TAG_CLOSE;
    }
}

void elem_attrib_var::WriteStream(std::ostream& outStream, int indent) const
{
    outStream << TAG_OPEN(g3dif::GetIdentifierName( g3dif::id_vertex_symbol ), indent)
        << Attr(0, g3dif::GetIdentifierName( g3dif::id_name ), vertex_symbol.name.value)
        << TAG_CLOSE;

    if (ui_label)
    {
        outStream << TAG_OPEN(g3dif::GetIdentifierName( g3dif::id_ui_label ), indent)
            << Attr(0, g3dif::GetIdentifierName( g3dif::id_value ), ui_label->value.value)
            << TAG_CLOSE;
    }

    if (ui_comment)
    {
        outStream << TAG_OPEN(g3dif::GetIdentifierName( g3dif::id_ui_comment ), indent)
            << Attr(0, g3dif::GetIdentifierName( g3dif::id_value ), ui_comment->value.value)
            << TAG_CLOSE;
    }

    if (ui_group)
    {
        outStream << TAG_OPEN(g3dif::GetIdentifierName( g3dif::id_ui_group ), indent)
            << Attr(0, g3dif::GetIdentifierName( g3dif::id_group_name ), ui_group->group_name.value)
            << TAG_CLOSE;
    }

    if (ui_editable)
    {
        outStream << TAG_OPEN(g3dif::GetIdentifierName( g3dif::id_ui_editable ), indent)
            << Attr(0, g3dif::GetIdentifierName( g3dif::id_value ), ui_editable->value.value)
            << TAG_CLOSE;
    }

    if (ui_order)
    {
        outStream << TAG_OPEN(g3dif::GetIdentifierName( g3dif::id_ui_order), indent)
            << Attr(0, g3dif::GetIdentifierName( g3dif::id_value ), ui_order->value.value)
            << TAG_CLOSE;
    }
}

void elem_sampler_var::WriteStream(std::ostream& outStream, int indent) const
{
    for (int stageIndex = 0; stageIndex < ShaderStage_StageCount; ++stageIndex)
    {
        if (shader_symbol[stageIndex])
        {
            outStream << TAG_OPEN(g3dif::GetIdentifierName(elem_shader_symbol::GetIdentifier(stageIndex)), indent)
                << Attr(0, g3dif::GetIdentifierName(g3dif::id_name), shader_symbol[stageIndex]->name.value)
                << TAG_CLOSE;
        }
    }

    if (ui_label)
    {
        outStream << TAG_OPEN(g3dif::GetIdentifierName( g3dif::id_ui_label ), indent)
            << Attr(0, g3dif::GetIdentifierName( g3dif::id_value ), ui_label->value.value)
            << TAG_CLOSE;
    }

    if (ui_comment)
    {
        outStream << TAG_OPEN(g3dif::GetIdentifierName( g3dif::id_ui_comment ), indent)
            << Attr(0, g3dif::GetIdentifierName( g3dif::id_value ), ui_comment->value.value)
            << TAG_CLOSE;
    }

    if (ui_group)
    {
        outStream << TAG_OPEN(g3dif::GetIdentifierName( g3dif::id_ui_group ), indent)
            << Attr(0, g3dif::GetIdentifierName( g3dif::id_group_name ), ui_group->group_name.value)
            << TAG_CLOSE;
    }

    if (ui_order)
    {
        outStream << TAG_OPEN(g3dif::GetIdentifierName( g3dif::id_ui_order), indent)
            << Attr(0, g3dif::GetIdentifierName( g3dif::id_value ), ui_order->value.value)
            << TAG_CLOSE;
    }

    if (ui_visible)
    {
        outStream << TAG_OPEN(g3dif::GetIdentifierName( g3dif::id_ui_visible ), indent)
            << Attr(0, g3dif::GetIdentifierName( g3dif::id_value ), ui_visible->value.value)
            << TAG_CLOSE;
    }
}

void elem_uniform_var::WriteStream(std::ostream& outStream, int indent) const
{
    for (int stageIndex = 0; stageIndex < ShaderStage_StageCount; ++stageIndex)
    {
        if (shader_symbol[stageIndex])
        {
            outStream << TAG_OPEN(g3dif::GetIdentifierName(elem_shader_symbol::GetIdentifier(stageIndex)), indent)
                << Attr(0, g3dif::GetIdentifierName(g3dif::id_name), shader_symbol[stageIndex]->name.value)
                << TAG_CLOSE;
        }
    }

    if (ui_label)
    {
        outStream << TAG_OPEN(g3dif::GetIdentifierName( g3dif::id_ui_label ), indent)
            << Attr(0, g3dif::GetIdentifierName( g3dif::id_value ), ui_label->value.value)
            << TAG_CLOSE;
    }

    if (ui_comment)
    {
        outStream << TAG_OPEN(g3dif::GetIdentifierName( g3dif::id_ui_comment ), indent)
            << Attr(0, g3dif::GetIdentifierName( g3dif::id_value ), ui_comment->value.value)
            << TAG_CLOSE;
    }

    if (ui_group)
    {
        outStream << TAG_OPEN(g3dif::GetIdentifierName( g3dif::id_ui_group ), indent)
            << Attr(0, g3dif::GetIdentifierName( g3dif::id_group_name ), ui_group->group_name.value)
            << TAG_CLOSE;
    }

    if (ui_order)
    {
        outStream << TAG_OPEN(g3dif::GetIdentifierName( g3dif::id_ui_order), indent)
            << Attr(0, g3dif::GetIdentifierName( g3dif::id_value ), ui_order->value.value)
            << TAG_CLOSE;
    }

    if (ui_item)
    {
        outStream << TAG_OPEN(g3dif::GetIdentifierName( g3dif::id_ui_item ), indent)
            << Attr(0, g3dif::GetIdentifierName( g3dif::id_value ), ui_item->value.value)
            << TAG_CLOSE;
    }

    if (ui_min)
    {
        outStream << TAG_OPEN(g3dif::GetIdentifierName( g3dif::id_ui_min ), indent)
            << Attr(0, g3dif::GetIdentifierName( g3dif::id_value ), ui_min->value.value)
            << TAG_CLOSE;
    }

    if (ui_max)
    {
        outStream << TAG_OPEN(g3dif::GetIdentifierName( g3dif::id_ui_max ), indent)
            << Attr(0, g3dif::GetIdentifierName( g3dif::id_value ), ui_max->value.value)
            << TAG_CLOSE;
    }

    if (ui_visible)
    {
        outStream << TAG_OPEN(g3dif::GetIdentifierName( g3dif::id_ui_visible ), indent)
            << Attr(0, g3dif::GetIdentifierName( g3dif::id_value ), ui_visible->value.value)
            << TAG_CLOSE;
    }

    if (ui_default_min)
    {
        outStream << TAG_OPEN(g3dif::GetIdentifierName( g3dif::id_ui_default_min ), indent)
            << Attr(0, g3dif::GetIdentifierName( g3dif::id_value ), ui_default_min->value.value)
            << TAG_CLOSE;
    }

    if (ui_default_max)
    {
        outStream << TAG_OPEN(g3dif::GetIdentifierName( g3dif::id_ui_default_max ), indent)
            << Attr(0, g3dif::GetIdentifierName( g3dif::id_value ), ui_default_max->value.value)
            << TAG_CLOSE;
    }
}

void elem_ssbo_uniform_var::WriteStream(std::ostream& outStream, int indent) const
{
    for (int stageIndex = 0; stageIndex < ShaderStage_StageCount; ++stageIndex)
    {
        if (shader_symbol[stageIndex])
        {
            outStream << TAG_OPEN(g3dif::GetIdentifierName(elem_shader_symbol::GetIdentifier(stageIndex)), indent)
                << Attr(0, g3dif::GetIdentifierName(g3dif::id_name), shader_symbol[stageIndex]->name.value)
                << TAG_CLOSE;
        }
    }

    if (ui_label)
    {
        outStream << TAG_OPEN(g3dif::GetIdentifierName( g3dif::id_ui_label ), indent)
            << Attr(0, g3dif::GetIdentifierName( g3dif::id_value ), ui_label->value.value)
            << TAG_CLOSE;
    }

    if (ui_comment)
    {
        outStream << TAG_OPEN(g3dif::GetIdentifierName( g3dif::id_ui_comment ), indent)
            << Attr(0, g3dif::GetIdentifierName( g3dif::id_value ), ui_comment->value.value)
            << TAG_CLOSE;
    }

    if (ui_group)
    {
        outStream << TAG_OPEN(g3dif::GetIdentifierName( g3dif::id_ui_group ), indent)
            << Attr(0, g3dif::GetIdentifierName( g3dif::id_group_name ), ui_group->group_name.value)
            << TAG_CLOSE;
    }

    if (ui_order)
    {
        outStream << TAG_OPEN(g3dif::GetIdentifierName( g3dif::id_ui_order), indent)
            << Attr(0, g3dif::GetIdentifierName( g3dif::id_value ), ui_order->value.value)
            << TAG_CLOSE;
    }

    if (ui_item)
    {
        outStream << TAG_OPEN(g3dif::GetIdentifierName( g3dif::id_ui_item ), indent)
            << Attr(0, g3dif::GetIdentifierName( g3dif::id_value ), ui_item->value.value)
            << TAG_CLOSE;
    }

    if (ui_min)
    {
        outStream << TAG_OPEN(g3dif::GetIdentifierName( g3dif::id_ui_min ), indent)
            << Attr(0, g3dif::GetIdentifierName( g3dif::id_value ), ui_min->value.value)
            << TAG_CLOSE;
    }

    if (ui_max)
    {
        outStream << TAG_OPEN(g3dif::GetIdentifierName( g3dif::id_ui_max ), indent)
            << Attr(0, g3dif::GetIdentifierName( g3dif::id_value ), ui_max->value.value)
            << TAG_CLOSE;
    }

    if (ui_visible)
    {
        outStream << TAG_OPEN(g3dif::GetIdentifierName( g3dif::id_ui_visible ), indent)
            << Attr(0, g3dif::GetIdentifierName( g3dif::id_value ), ui_visible->value.value)
            << TAG_CLOSE;
    }

    if (ui_default_min)
    {
        outStream << TAG_OPEN(g3dif::GetIdentifierName( g3dif::id_ui_default_min ), indent)
            << Attr(0, g3dif::GetIdentifierName( g3dif::id_value ), ui_default_min->value.value)
            << TAG_CLOSE;
    }

    if (ui_default_max)
    {
        outStream << TAG_OPEN(g3dif::GetIdentifierName( g3dif::id_ui_default_max ), indent)
            << Attr(0, g3dif::GetIdentifierName( g3dif::id_value ), ui_default_max->value.value)
            << TAG_CLOSE;
    }
}

void elem_render_info_slot::WriteStream(std::ostream& outStream, int indent) const
{
    if (ui_label)
    {
        outStream << TAG_OPEN(g3dif::GetIdentifierName( g3dif::id_ui_label ), indent)
            << Attr(0, g3dif::GetIdentifierName( g3dif::id_value ), ui_label->value.value)
            << TAG_CLOSE;
    }

    if (ui_comment)
    {
        outStream << TAG_OPEN(g3dif::GetIdentifierName( g3dif::id_ui_comment ), indent)
            << Attr(0, g3dif::GetIdentifierName( g3dif::id_value ), ui_comment->value.value)
            << TAG_CLOSE;
    }

    if (ui_group)
    {
        outStream << TAG_OPEN(g3dif::GetIdentifierName( g3dif::id_ui_group ), indent)
            << Attr(0, g3dif::GetIdentifierName( g3dif::id_group_name ), ui_group->group_name.value)
            << TAG_CLOSE;
    }

    if (ui_order)
    {
        outStream << TAG_OPEN(g3dif::GetIdentifierName( g3dif::id_ui_order), indent)
            << Attr(0, g3dif::GetIdentifierName( g3dif::id_value ), ui_order->value.value)
            << TAG_CLOSE;
    }

    if (ui_item)
    {
        outStream << TAG_OPEN(g3dif::GetIdentifierName( g3dif::id_ui_item ), indent)
            << Attr(0, g3dif::GetIdentifierName( g3dif::id_value ), ui_item->value.value)
            << TAG_CLOSE;
    }

    if (ui_visible)
    {
        outStream << TAG_OPEN(g3dif::GetIdentifierName(g3dif::id_ui_visible), indent)
            << Attr(0, g3dif::GetIdentifierName(g3dif::id_value), ui_visible->value.value)
            << TAG_CLOSE;
    }
}

void elem_textblock::WriteStream(std::ostream& outStream, int indent) const
{
    if (ui_group)
    {
        outStream << TAG_OPEN(g3dif::GetIdentifierName(g3dif::id_ui_group), indent)
            << Attr(0, g3dif::GetIdentifierName(g3dif::id_group_name), ui_group->group_name.value)
            << TAG_CLOSE;
    }

    if (ui_order)
    {
        outStream << TAG_OPEN(g3dif::GetIdentifierName(g3dif::id_ui_order), indent)
            << Attr(0, g3dif::GetIdentifierName(g3dif::id_value), ui_order->value.value)
            << TAG_CLOSE;
    }
}

void elem_group::WriteStream(std::ostream& outStream, int indent) const
{
    if (ui_label)
    {
        outStream << TAG_OPEN(g3dif::GetIdentifierName( g3dif::id_ui_label ), indent)
            << Attr(0, g3dif::GetIdentifierName( g3dif::id_value ), ui_label->value.value)
            << TAG_CLOSE;
    }

    if (ui_comment)
    {
        outStream << TAG_OPEN(g3dif::GetIdentifierName( g3dif::id_ui_comment ), indent)
            << Attr(0, g3dif::GetIdentifierName( g3dif::id_value ), ui_comment->value.value)
            << TAG_CLOSE;
    }

    if (ui_order)
    {
        outStream << TAG_OPEN(g3dif::GetIdentifierName( g3dif::id_ui_order), indent)
            << Attr(0, g3dif::GetIdentifierName( g3dif::id_value ), ui_order->value.value)
            << TAG_CLOSE;
    }

    if (ui_group)
    {
        outStream << TAG_OPEN(g3dif::GetIdentifierName( g3dif::id_ui_group ), indent)
            << Attr(0, g3dif::GetIdentifierName( g3dif::id_group_name ), ui_group->group_name.value)
            << TAG_CLOSE;
    }
}

void elem_page::WriteStream(std::ostream& outStream, int indent) const
{
    if (ui_label)
    {
        outStream << TAG_OPEN(g3dif::GetIdentifierName( g3dif::id_ui_label ), indent)
            << Attr(0, g3dif::GetIdentifierName( g3dif::id_value ), ui_label->value.value)
            << TAG_CLOSE;
    }

    if (ui_comment)
    {
        outStream << TAG_OPEN(g3dif::GetIdentifierName( g3dif::id_ui_comment ), indent)
            << Attr(0, g3dif::GetIdentifierName( g3dif::id_value ), ui_comment->value.value)
            << TAG_CLOSE;
    }

    if (ui_order)
    {
        outStream << TAG_OPEN(g3dif::GetIdentifierName( g3dif::id_ui_order), indent)
            << Attr(0, g3dif::GetIdentifierName( g3dif::id_value ), ui_order->value.value)
            << TAG_CLOSE;
    }
}

template<typename TBlockVarType>
void BuildBlockVarArray( std::ostream& outStream, const std::vector<TBlockVarType>& block_var_array, g3dif::Identifier identifier_block_var, g3dif::Identifier identifier_uniform_var )
{
    if (block_var_array.size())
    {
        const char* str_block_var = g3dif::GetIdentifierName(identifier_block_var);
        const char* str_uniform_var = g3dif::GetIdentifierName(identifier_uniform_var);

        outStream << TAG_OPEN(TAG_ARRAY(str_block_var), INDENT_2)
            << Attr(0, g3dif::GetIdentifierName( g3dif::id_length ), block_var_array.size())
            << TAG_START;

        int block_index = 0;
        for (auto el_block = block_var_array.cbegin(); el_block != block_var_array.cend(); ++el_block, ++block_index)
        {
            outStream << TAG_OPEN(str_block_var, INDENT_3)
                << Attr(0, g3dif::GetIdentifierName( g3dif::id_index ), block_index)
                << Attr(0, g3dif::GetIdentifierName( g3dif::id_id ), el_block->id.value)
                << Attr(INDENT_4, g3dif::GetIdentifierName( g3dif::id_type ), TBlockVarType::tbl_type[el_block->type.value].str) << TAG_END_LINE;
            outStream << Indent(INDENT_3) << TAG_START;

            for (int stageIndex = 0; stageIndex < ShaderStage_StageCount; ++stageIndex)
            {
                if (el_block->shader_symbol[stageIndex])
                {
                    outStream << TAG_OPEN(g3dif::GetIdentifierName(elem_shader_symbol::GetIdentifier(stageIndex)), INDENT_4)
                        << Attr(0, g3dif::GetIdentifierName(g3dif::id_name), el_block->shader_symbol[stageIndex]->name.value)
                        << TAG_CLOSE;
                }
            }

            if (el_block->uniform_var_array.size())
            {
                outStream << TAG_OPEN(TAG_ARRAY(str_uniform_var), INDENT_4)
                    << Attr(0, g3dif::GetIdentifierName( g3dif::id_length ), el_block->uniform_var_array.size())
                    << TAG_START;

                int uniform_index = 0;
                for (auto el_uniform = el_block->uniform_var_array.cbegin();
                    el_uniform != el_block->uniform_var_array.cend();
                    ++el_uniform, ++uniform_index)
                {
                    if (identifier_block_var == g3dif::id_block_var)
                    {
                        outStream << TAG_OPEN(str_uniform_var, INDENT_5)
                            << Attr(0, g3dif::GetIdentifierName(g3dif::id_index), uniform_index)
                            << Attr(0, g3dif::GetIdentifierName(g3dif::id_id), el_uniform->id.value)
                            << Attr(INDENT_6, g3dif::GetIdentifierName(g3dif::id_hint), el_uniform->hint.value)
                            << Attr(INDENT_6, g3dif::GetIdentifierName(g3dif::id_type), elem_shader_param::GetType(el_uniform->type.value))
                            << Attr(INDENT_6, g3dif::GetIdentifierName(g3dif::id_default), el_uniform->default_value.value)
                            << Attr(INDENT_6, g3dif::GetIdentifierName(g3dif::id_nostrip), GetBoolString(el_uniform->nostrip.value))
                            << Attr(INDENT_6, g3dif::GetIdentifierName(g3dif::id_converter), el_uniform->converter.value)
                            << Attr(INDENT_6, g3dif::GetIdentifierName(g3dif::id_depend), el_uniform->depend.value)
                            << TAG_END_LINE;
                        outStream << Indent(INDENT_5) << TAG_START;
                    }
                    else if (identifier_block_var == g3dif::id_shader_storage_block_var)
                    {
                        outStream << TAG_OPEN(str_uniform_var, INDENT_5)
                            << Attr(0, g3dif::GetIdentifierName(g3dif::id_index), uniform_index)
                            << Attr(0, g3dif::GetIdentifierName(g3dif::id_id), el_uniform->id.value)
                            << Attr(INDENT_6, g3dif::GetIdentifierName(g3dif::id_nostrip), GetBoolString(el_uniform->nostrip.value))
                            << TAG_END_LINE;
                        outStream << Indent(INDENT_5) << TAG_START;
                    }
                    el_uniform->WriteStream(outStream, INDENT_6);

                    outStream << TAG_END(str_uniform_var, INDENT_5);
                }

                outStream << TAG_END(TAG_ARRAY(str_uniform_var), INDENT_4);
            }

            outStream << TAG_END(str_block_var, INDENT_3);
        }

        outStream << TAG_END(TAG_ARRAY(str_block_var), INDENT_2);
    }
}

void elem_shading_model::WriteStream(std::ostream& outStream) const
{
    for (int stageIndex = 0; stageIndex < ShaderStage_StageCount; ++stageIndex)
    {
        if (shader_stage[stageIndex])
        {
            outStream << TAG_OPEN(g3dif::GetIdentifierName(elem_shader_stage::GetIdentifier(stageIndex)), INDENT_2)
                << Attr(0, g3dif::GetIdentifierName(g3dif::id_src_index), shader_stage[stageIndex]->src_index.value)
                << TAG_CLOSE;
        }
    }

    if (macro_array.size())
    {
        outStream << TAG_OPEN(TAG_ARRAY(g3dif::GetIdentifierName( g3dif::id_macro )), INDENT_2)
            << Attr(0, g3dif::GetIdentifierName( g3dif::id_length ), macro_array.size())
            << TAG_START;

        int macro_index = 0;
        for (auto el_macro = macro_array.cbegin(); el_macro != macro_array.cend(); ++el_macro, ++macro_index)
        {
            outStream << TAG_OPEN(g3dif::GetIdentifierName( g3dif::id_macro ), INDENT_3)
                << Attr(0, g3dif::GetIdentifierName( g3dif::id_index ), macro_index)
                << Attr(0, g3dif::GetIdentifierName( g3dif::id_name ), el_macro->name.value)
                << Attr(0, g3dif::GetIdentifierName( g3dif::id_value ), el_macro->value.value)
                << TAG_CLOSE;
        }

        outStream << TAG_END(TAG_ARRAY(g3dif::GetIdentifierName( g3dif::id_macro )), INDENT_2);
    }

    auto result =
        std::find_if(option_var_array.begin(), option_var_array.end(), SystemOptionFinder());

    size_t option_size = option_var_array.size() - ((result != option_var_array.cend()) ? 1 : 0);

    if (option_size)
    {
        outStream << TAG_OPEN(TAG_ARRAY(g3dif::GetIdentifierName( g3dif::id_option_var )), INDENT_2)
            << Attr(0, g3dif::GetIdentifierName( g3dif::id_length ), option_size)
            << TAG_START;

        int option_index = 0;
        for (auto el_option = option_var_array.cbegin(); el_option != option_var_array.cend(); ++el_option)
        {
            // SYSTEM_ID の option は書き出さない
            if (el_option->id.value == SYSTEM_ID)
            {
                continue;
            }

            outStream << TAG_OPEN(g3dif::GetIdentifierName( g3dif::id_option_var ), INDENT_3)
                << Attr(0, g3dif::GetIdentifierName( g3dif::id_index ), option_index)
                << Attr(0, g3dif::GetIdentifierName( g3dif::id_id ), el_option->id.value)
                << Attr(INDENT_4, g3dif::GetIdentifierName( g3dif::id_type ), elem_option_var::tbl_type[el_option->type.value].str)
                << Attr(INDENT_4, g3dif::GetIdentifierName( g3dif::id_choice ), el_option->choice.value)
                << Attr(INDENT_4, g3dif::GetIdentifierName( g3dif::id_default ), el_option->default_value.value)
                << Attr(INDENT_4, g3dif::GetIdentifierName( g3dif::id_branch ), GetBoolString(el_option->branch.value)) << TAG_END_LINE;
            outStream << Indent(INDENT_3) << TAG_START;

            el_option->WriteStream(outStream, INDENT_4);

            outStream << TAG_END(g3dif::GetIdentifierName( g3dif::id_option_var ), INDENT_3);
            ++option_index;
        }

        outStream << TAG_END(TAG_ARRAY(g3dif::GetIdentifierName( g3dif::id_option_var )), INDENT_2);
    }

    if (attrib_var_array.size())
    {
        outStream << TAG_OPEN(TAG_ARRAY(g3dif::GetIdentifierName( g3dif::id_attrib_var )), INDENT_2)
            << Attr(0, g3dif::GetIdentifierName( g3dif::id_length ), attrib_var_array.size())
            << TAG_START;

        int attrib_index = 0;
        for (auto el_attrib = attrib_var_array.cbegin(); el_attrib != attrib_var_array.cend(); ++el_attrib, ++attrib_index)
        {
            outStream << TAG_OPEN(g3dif::GetIdentifierName( g3dif::id_attrib_var ), INDENT_3)
                << Attr(0, g3dif::GetIdentifierName( g3dif::id_index ), attrib_index)
                << Attr(0, g3dif::GetIdentifierName( g3dif::id_id ), el_attrib->id.value)
                << Attr(INDENT_4, g3dif::GetIdentifierName( g3dif::id_hint ), el_attrib->hint.value)
                << Attr(INDENT_4, g3dif::GetIdentifierName( g3dif::id_type ), elem_vtx_attrib::GetType(el_attrib->type.value)) << TAG_END_LINE;
            outStream << Indent(INDENT_3) << TAG_START;

            el_attrib->WriteStream(outStream, INDENT_4);

            outStream << TAG_END(g3dif::GetIdentifierName( g3dif::id_attrib_var ), INDENT_3);
        }

        outStream << TAG_END(TAG_ARRAY(g3dif::GetIdentifierName( g3dif::id_attrib_var )), INDENT_2);
    }

    if (sampler_var_array.size())
    {
        outStream << TAG_OPEN(TAG_ARRAY(g3dif::GetIdentifierName( g3dif::id_sampler_var )), INDENT_2)
            << Attr(0, g3dif::GetIdentifierName( g3dif::id_length ), sampler_var_array.size())
            << TAG_START;

        int sampler_index = 0;
        for (auto el_sampler = sampler_var_array.cbegin(); el_sampler != sampler_var_array.cend(); ++el_sampler, ++sampler_index)
        {
            outStream << TAG_OPEN(g3dif::GetIdentifierName( g3dif::id_sampler_var ), INDENT_3)
                << Attr(0, g3dif::GetIdentifierName( g3dif::id_index ), sampler_index)
                << Attr(0, g3dif::GetIdentifierName( g3dif::id_id ), el_sampler->id.value)
                << Attr(INDENT_4, g3dif::GetIdentifierName( g3dif::id_hint ), el_sampler->hint.value)
                << Attr(INDENT_4, g3dif::GetIdentifierName( g3dif::id_type ), elem_sampler_var::tbl_type[el_sampler->type.value].str)
                << Attr(INDENT_4, g3dif::GetIdentifierName( g3dif::id_alt  ), el_sampler->alt.value)
                << Attr(INDENT_4, g3dif::GetIdentifierName( g3dif::id_nostrip ), GetBoolString(el_sampler->nostrip.value)) << TAG_END_LINE;
            outStream << Indent(INDENT_3) << TAG_START;

            el_sampler->WriteStream(outStream, INDENT_4);

            outStream << TAG_END(g3dif::GetIdentifierName( g3dif::id_sampler_var ), INDENT_3);
        }

        outStream << TAG_END(TAG_ARRAY(g3dif::GetIdentifierName( g3dif::id_sampler_var )), INDENT_2);
    }
    BuildBlockVarArray( outStream, block_var_array, g3dif::id_block_var, g3dif::id_uniform_var);
    BuildBlockVarArray( outStream, ssbo_block_var_array, g3dif::id_shader_storage_block_var, g3dif::id_buffer_var );

    if (render_info_slot_array.size())
    {
        outStream << TAG_OPEN(TAG_ARRAY(g3dif::GetIdentifierName( g3dif::id_render_info_slot )), INDENT_2)
            << Attr(0, g3dif::GetIdentifierName( g3dif::id_length ), render_info_slot_array.size())
            << TAG_START;

        int render_info_slot_index = 0;
        for (auto el_render_info_slot = render_info_slot_array.cbegin(); el_render_info_slot != render_info_slot_array.cend(); ++el_render_info_slot, ++render_info_slot_index)
        {
            outStream << TAG_OPEN(g3dif::GetIdentifierName( g3dif::id_render_info_slot ), INDENT_3)
                << Attr(0, g3dif::GetIdentifierName( g3dif::id_index ), render_info_slot_index)
                << Attr(0, g3dif::GetIdentifierName( g3dif::id_name ), el_render_info_slot->name.value)
                << Attr(0, g3dif::GetIdentifierName( g3dif::id_type ), elem_render_info_slot::tbl_type[el_render_info_slot->type.value].str)
                << Attr(0, g3dif::GetIdentifierName( g3dif::id_count ), el_render_info_slot->count.value)
                << Attr(0, g3dif::GetIdentifierName( g3dif::id_optional ), GetBoolString(el_render_info_slot->optional.value))
                << Attr(INDENT_4, g3dif::GetIdentifierName( g3dif::id_choice ), el_render_info_slot->choice.value)
                << Attr(INDENT_4, g3dif::GetIdentifierName( g3dif::id_default ), el_render_info_slot->default_value.value) << TAG_END_LINE;
            outStream << Indent(INDENT_3) << TAG_START;

            el_render_info_slot->WriteStream(outStream, INDENT_4);

            outStream << TAG_END(g3dif::GetIdentifierName( g3dif::id_render_info_slot ), INDENT_3);
        }

        outStream << TAG_END(TAG_ARRAY(g3dif::GetIdentifierName( g3dif::id_render_info_slot )), INDENT_2);
    }

    if (textblock_array.size())
    {
        outStream << TAG_OPEN(TAG_ARRAY(g3dif::GetIdentifierName(g3dif::id_textblock)), INDENT_2)
            << Attr(0, g3dif::GetIdentifierName(g3dif::id_length), textblock_array.size())
            << TAG_START;

        int textblock_index = 0;
        for (auto el_textblock = textblock_array.cbegin(); el_textblock != textblock_array.cend(); ++el_textblock, ++textblock_index)
        {
            outStream << TAG_OPEN(g3dif::GetIdentifierName(g3dif::id_textblock), INDENT_3)
                << Attr(0, g3dif::GetIdentifierName(g3dif::id_index), textblock_index)
                << Attr(0, g3dif::GetIdentifierName(g3dif::id_id), el_textblock->id.value)
                << Attr(INDENT_4, g3dif::GetIdentifierName(g3dif::id_text), el_textblock->text.value) << TAG_END_LINE;
            outStream << Indent(INDENT_3) << TAG_START;

            el_textblock->WriteStream(outStream, INDENT_4);

            outStream << TAG_END(g3dif::GetIdentifierName(g3dif::id_textblock), INDENT_3);
        }

        outStream << TAG_END(TAG_ARRAY(g3dif::GetIdentifierName(g3dif::id_textblock)), INDENT_2);
    }

    if (interleave_array.size())
    {
        outStream << TAG_OPEN(TAG_ARRAY(g3dif::GetIdentifierName( g3dif::id_interleave )), INDENT_2)
            << Attr(0, g3dif::GetIdentifierName( g3dif::id_length ), interleave_array.size())
            << TAG_START;

        int interleave_index = 0;
        for (auto el_interleave = interleave_array.cbegin(); el_interleave != interleave_array.cend(); ++el_interleave, ++interleave_index)
        {
            outStream << TAG_OPEN(g3dif::GetIdentifierName( g3dif::id_interleave ), INDENT_3)
                << Attr(0, g3dif::GetIdentifierName( g3dif::id_index ), interleave_index)
                << Attr(0, g3dif::GetIdentifierName( g3dif::id_id_set ), el_interleave->id_set.value)
                << Attr(0, g3dif::GetIdentifierName( g3dif::id_binarize ), el_interleave->binarize.value)
                << TAG_CLOSE;
        }

        outStream << TAG_END(TAG_ARRAY(g3dif::GetIdentifierName( g3dif::id_interleave )), INDENT_2);
    }

    if (group_array.size())
    {
        outStream << TAG_OPEN(TAG_ARRAY(g3dif::GetIdentifierName( g3dif::id_group )), INDENT_2)
            << Attr(0, g3dif::GetIdentifierName( g3dif::id_length ), group_array.size())
            << TAG_START;

        int group_index = 0;
        for (auto el_group = group_array.cbegin(); el_group != group_array.cend(); ++el_group, ++group_index)
        {
            outStream << TAG_OPEN(g3dif::GetIdentifierName( g3dif::id_group ), INDENT_3)
                << Attr(0, g3dif::GetIdentifierName( g3dif::id_index ), group_index)
                << Attr(0, g3dif::GetIdentifierName( g3dif::id_name ), el_group->name.value)
                << Attr(0, g3dif::GetIdentifierName( g3dif::id_page_name ), el_group->page_name.value)
                << Attr(0, g3dif::GetIdentifierName( g3dif::id_condition ), el_group->condition.value)
                << TAG_START;

            el_group->WriteStream(outStream, INDENT_4);

            outStream << TAG_END(g3dif::GetIdentifierName( g3dif::id_group ), INDENT_3);
        }

        outStream << TAG_END(TAG_ARRAY(g3dif::GetIdentifierName( g3dif::id_group )), INDENT_2);
    }

    if (page_array.size())
    {
        outStream << TAG_OPEN(TAG_ARRAY(g3dif::GetIdentifierName( g3dif::id_page )), INDENT_2)
            << Attr(0, g3dif::GetIdentifierName( g3dif::id_length ), page_array.size())
            << TAG_START;

        int page_index = 0;
        for (auto el_page = page_array.cbegin(); el_page != page_array.cend();
            ++el_page, ++page_index)
        {
            outStream << TAG_OPEN(g3dif::GetIdentifierName( g3dif::id_page ), INDENT_3)
                << Attr(0, g3dif::GetIdentifierName( g3dif::id_index ), page_index)
                << Attr(0, g3dif::GetIdentifierName( g3dif::id_name ), el_page->name.value)
                << TAG_START;

            el_page->WriteStream(outStream, INDENT_4);

            outStream << TAG_END(g3dif::GetIdentifierName( g3dif::id_page ), INDENT_3);
        }

        outStream << TAG_END(TAG_ARRAY(g3dif::GetIdentifierName( g3dif::id_page )), INDENT_2);
    }

    if (streamout)
    {
        outStream << TAG_OPEN(g3dif::GetIdentifierName( g3dif::id_streamout ), INDENT_2)
            << Attr(0, g3dif::GetIdentifierName( g3dif::id_varying ), streamout->varying.value)
            << TAG_CLOSE;
    }
}

void elem_shader_definition::WriteStream(std::ostream& outStream) const
{
    outStream << "<?xml version=\"1.0\" encoding=\"utf-8\"?>" << TAG_END_LINE
        << "<nw4f_3dif version=\""
        << NN_G3D_INTERMEDIATE_VERSION_MAJOR << "."
        << NN_G3D_INTERMEDIATE_VERSION_MINOR << "."
        << NN_G3D_INTERMEDIATE_VERSION_MICRO << "\">" << TAG_END_LINE;

    std::stringstream fsdVersion;
    fsdVersion << NN_G3D_INTERMEDIATE_FSD_VERSION_MAJOR << "." << NN_G3D_INTERMEDIATE_FSD_VERSION_MINOR << "." << NN_G3D_INTERMEDIATE_FSD_VERSION_MICRO;

    outStream << TAG_OPEN(g3dif::GetIdentifierName( g3dif::id_shader_definition ), INDENT_0)
        << Attr(0, g3dif::GetIdentifierName(g3dif::id_version), fsdVersion.str()) << TAG_START;
    outStream << TAG_OPEN(g3dif::GetIdentifierName( g3dif::id_shader_definition_info ), INDENT_0)
        << Attr(INDENT_1, g3dif::GetIdentifierName( g3dif::id_code_page ), shader_definition_info.code_page.value)
        << Attr(INDENT_1, g3dif::GetIdentifierName( g3dif::id_config_path ), util::EscapeString(shader_definition_info.config_path.value))
        << TAG_END_LINE;
    outStream << TAG_CLOSE_LINE;

    if (force_include_array.size())
    {
        outStream << TAG_OPEN(TAG_ARRAY(g3dif::GetIdentifierName( g3dif::id_force_include )), INDENT_0)
            << Attr(0, g3dif::GetIdentifierName( g3dif::id_length ), force_include_array.size()) << TAG_START;

        int include_index = 0;
        for (auto el_force_include = force_include_array.cbegin(); el_force_include != force_include_array.cend(); ++el_force_include, ++include_index)
        {
            outStream << TAG_OPEN(g3dif::GetIdentifierName( g3dif::id_force_include ), INDENT_1)
                << Attr(0, g3dif::GetIdentifierName( g3dif::id_index ), include_index)
                << Attr(0, g3dif::GetIdentifierName( g3dif::id_src_index ), el_force_include->src_index.value)
                << TAG_CLOSE;
        }

        outStream << TAG_END(TAG_ARRAY(g3dif::GetIdentifierName( g3dif::id_force_include )), INDENT_0);
    }

    if (shading_model_array.size())
    {
        outStream << TAG_OPEN(TAG_ARRAY(g3dif::GetIdentifierName( g3dif::id_shading_model )), INDENT_0)
            << Attr(0, g3dif::GetIdentifierName( g3dif::id_length ), shading_model_array.size()) << TAG_START;

        int definition_index = 0;
        for (auto el_shading_model = shading_model_array.cbegin(); el_shading_model != shading_model_array.cend(); ++el_shading_model, ++definition_index)
        {
            outStream << TAG_OPEN(g3dif::GetIdentifierName( g3dif::id_shading_model ), INDENT_1)
                << Attr(0, g3dif::GetIdentifierName( g3dif::id_index ), definition_index)
                << Attr(0, g3dif::GetIdentifierName( g3dif::id_name ), el_shading_model->name.value)
                << Attr(0, g3dif::GetIdentifierName( g3dif::id_material_shader ), GetBoolString(el_shading_model->material_shader.value))
                << Attr(0, g3dif::GetIdentifierName( g3dif::id_revision ), el_shading_model->revision.value) << TAG_START;

            el_shading_model->WriteStream(outStream);

            outStream << TAG_END(g3dif::GetIdentifierName( g3dif::id_shading_model ), INDENT_1);
        }

        outStream << TAG_END(TAG_ARRAY(g3dif::GetIdentifierName( g3dif::id_shading_model )), INDENT_0);
    }

    if (shader_src_array.size())
    {
        outStream << TAG_OPEN(TAG_ARRAY(g3dif::GetIdentifierName( g3dif::id_shader_src )), INDENT_0)
            << Attr(0, g3dif::GetIdentifierName( g3dif::id_length ), shader_src_array.size())
            << TAG_START;

        int shader_src_index = 0;
        for (auto el_shader_src = shader_src_array.cbegin(); el_shader_src != shader_src_array.cend(); ++el_shader_src, ++shader_src_index)
        {
            outStream << TAG_OPEN(g3dif::GetIdentifierName( g3dif::id_shader_src ), INDENT_1)
                << Attr(0, g3dif::GetIdentifierName( g3dif::id_src_index ), shader_src_index)
                << Attr(INDENT_2, g3dif::GetIdentifierName( g3dif::id_include_path ), util::EscapeString(el_shader_src->include_path.value))
                << Attr(INDENT_2, g3dif::GetIdentifierName( g3dif::id_path ), util::EscapeString(el_shader_src->path.value))
                << Attr(INDENT_2, g3dif::GetIdentifierName( g3dif::id_stream_index ), el_shader_src->stream_index.value) << TAG_END_LINE;
            outStream << Indent(INDENT_1) << TAG_CLOSE_LINE;
        }

        outStream << TAG_END(TAG_ARRAY(g3dif::GetIdentifierName( g3dif::id_shader_src )), INDENT_0);

        // stream_array の出力
        // バイナリ中間ファイルは WriteBinaryStream で出力する。
        if (ext == ".fsda")
        {
            // NOTE: .fsda は現状ではシェーダコンバータから出力しません。
            //       必要になれば実装します。
            THROW_ERROR(ERRCODE_UNSUPPORTED_FILETYPE, "Unsupported file type.");
        }
        else if (ext == ".fsdb")
        {
            // do nothing.
        }
        else
        {
            THROW_ERROR_INTERNAL(ERRCODE_INTERNAL, "Internal error.");
        }
    }

    outStream << TAG_END(g3dif::GetIdentifierName( g3dif::id_shader_definition ), INDENT_0);
    outStream << "</nw4f_3dif>" << TAG_END_LINE;
}

void elem_shader_definition::WriteBinaryStream(std::ostream& outStream) const
{
    if (shader_src_array.size())
    {
        if (ext == ".fsda")
        {
            // do nothing.
        }
        else if (ext == ".fsdb")
        {
            // 末尾に追記する。
            outStream.seekp(0, std::ios::end);
            size_t imSize = static_cast<size_t>(outStream.tellp());
            size_t alignedSize = util::Align(imSize, STREAM_BINARY_ALIGNMENT);

            int paddingSize = static_cast<int>(alignedSize - imSize);
            paddingSize = (paddingSize == 0) ? STREAM_BINARY_ALIGNMENT : paddingSize;

            // padding
            for (int i = 0; i < paddingSize; ++i)
            {
                outStream.put('\0');
            }

            StreamArray streamArray;
            streamArray.header = STREAM_HEADER_ORDER;
            streamArray.chunkCount = static_cast<u32>(shader_src_array.size());

            for (int i = 0; i < sizeof(streamArray.header); ++i)
            {
                outStream.put(*static_cast<char*>(util::AddOffset(&streamArray.header, i)));
            }

            G3DIF_WRITE_FIXED_BINARY_STREAM(outStream, streamArray.chunkCount);

            size_t baseOffset =
                sizeof(streamArray.header) + sizeof(streamArray.chunkCount) + shader_src_array.size() * sizeof(ChunkOffset);

            size_t alignedBaseOffset = util::Align(baseOffset, STREAM_CHUNK_ALIGNMENT);

            size_t offset = alignedBaseOffset;

            std::vector<std::string> streamData;
            // データを作成
            for (auto el_shader_src = shader_src_array.cbegin(); el_shader_src != shader_src_array.cend(); ++el_shader_src)
            {
                std::string stream(static_cast<char*>(el_shader_src->stream.rawdata.get()));
                streamData.emplace_back(stream);
            }

            int shader_src_index = 0;
            for (auto el_shader_src = shader_src_array.cbegin(); el_shader_src != shader_src_array.cend(); ++el_shader_src, ++shader_src_index)
            {
                StreamChunk streamChunk;
                streamChunk.header = STREAM_CHUNK_HEADER_ORDER;
                streamChunk.type = StreamTypeWString;
                streamChunk.count = 1;
                streamChunk.column = 0;

                size_t wLen = MultiByteToWideChar(CP_UTF8,
                    0,
                    streamData[shader_src_index].c_str(),
                    -1,
                    NULL,
                    0);

                // 文字列長＋終端文字
                streamChunk.size = static_cast<u32>(wLen * sizeof(wchar_t));
                streamChunk.padding = 0;

                size_t chuckSize = util::Align(STREAM_CHUNK_SIZE_ALIGNMENT + streamChunk.size, STREAM_CHUNK_ALIGNMENT);

                ChunkOffset chunkOffset;
                chunkOffset.offset = static_cast<u32>(offset);
                chunkOffset.size = static_cast<u32>(chuckSize);

                offset += chuckSize;

                streamArray.chunkOffset.push_back(chunkOffset);
                streamArray.streamChunk.push_back(streamChunk);

                G3DIF_WRITE_FIXED_BINARY_STREAM(outStream, chunkOffset.offset);
                G3DIF_WRITE_FIXED_BINARY_STREAM(outStream, chunkOffset.size);
            }

            for (int i = 0; i < static_cast<int>(alignedBaseOffset - baseOffset); ++i)
            {
                outStream.put('\0');
            }

            shader_src_index = 0;
            for (auto el_shader_src = shader_src_array.cbegin(); el_shader_src != shader_src_array.cend(); ++el_shader_src, ++shader_src_index)
            {
                StreamChunk& streamChunk = streamArray.streamChunk[shader_src_index];
                ChunkOffset& chunkOffset = streamArray.chunkOffset[shader_src_index];

                for (int i = 0; i < sizeof(streamChunk.header); ++i)
                {
                    outStream.put(*static_cast<char*>(util::AddOffset(&streamChunk.header, i)));
                }

                G3DIF_WRITE_FIXED_BINARY_STREAM(outStream, streamChunk.type);
                G3DIF_WRITE_FIXED_BINARY_STREAM(outStream, streamChunk.count);
                G3DIF_WRITE_FIXED_BINARY_STREAM(outStream, streamChunk.column);
                G3DIF_WRITE_FIXED_BINARY_STREAM(outStream, streamChunk.size);
                G3DIF_WRITE_FIXED_BINARY_STREAM(outStream, streamChunk.padding);

                std::vector<wchar_t> wbuf;
                wbuf.resize(streamChunk.size / sizeof(wchar_t));

                MultiByteToWideChar(CP_UTF8,
                    0,
                    streamData[shader_src_index].c_str(),
                    -1,
                    &wbuf[0],
                    static_cast<int>(wbuf.size()));

                outStream.write(reinterpret_cast<const char*>(&wbuf[0]), streamChunk.size);

                size_t chunkPaddingSize = chunkOffset.size - (STREAM_CHUNK_SIZE_ALIGNMENT + streamChunk.size);
                for (int i = 0; i < static_cast<int>(chunkPaddingSize); ++i)
                {
                    outStream.put(0);
                }
            }
        }
        else
        {
            THROW_ERROR_INTERNAL(ERRCODE_INTERNAL, "Internal error.");
        }
    }
}

const elem_uniform_var* elem_shading_model::GetOptionUniform(const elem_option_var& el_option) const
{
    const std::vector<elem_uniform_var>* el_option_uniform_var_array = nullptr;
    for (auto el_block = block_var_array.begin(); el_block != block_var_array.end(); ++el_block)
    {
        if (el_block->type.value == elem_block_var::option)
        {
            el_option_uniform_var_array = &el_block->uniform_var_array;
            break;
        }
    }

    if (el_option_uniform_var_array != nullptr)
    {
        auto found =
            std::find_if(el_option_uniform_var_array->cbegin(), el_option_uniform_var_array->cend(), OptionUniformFinder(el_option.id.value));
        if (found != el_option_uniform_var_array->cend())
        {
            return &(*found);
        }
    }

    return nullptr;
}

void elem_shading_model::AddSystemOption()
{
    auto result =
        std::find_if(option_var_array.cbegin(), option_var_array.cend(), SystemOptionFinder());

    if (result == option_var_array.cend())
    {
        g3dif::elem_option_var el_system_option_var;
        el_system_option_var.id.value = SYSTEM_ID;
        el_system_option_var.type = g3dif::elem_option_var::type_dynamic;
        el_system_option_var.choice.value = "0";
        el_system_option_var.default_value.value = "0";
        el_system_option_var.branch = false;
        for (int stageIndex = 0; stageIndex < ShaderStage_StageCount; ++stageIndex)
        {
            el_system_option_var.shader_symbol[stageIndex].Validate();
            el_system_option_var.shader_symbol[stageIndex]->name.value = NW_SYSTEM_MACRO;
        }
        this->option_var_array.insert(this->option_var_array.begin(), el_system_option_var);
    }
}

void elem_shading_model::Expand()
{
    if (option_var_array.size() == 0)
    {
        THROW_ERROR_INTERNAL(ERRCODE_INTERNAL, "Internal error.");
    }

    // option_var の順序を入れ替える。
    // dynamic な option を後ろに移動させる。
    int optionCount = static_cast<int>(option_var_array.size());
    int optionIndex = 0;
    for (int i = 0; i < optionCount; ++i)
    {
        elem_option_var& el_option_var = option_var_array[optionIndex];
        if (el_option_var.type.value == elem_option_var::type_dynamic)
        {
            option_var_array.push_back(el_option_var);
            option_var_array.erase(option_var_array.begin() + optionIndex);
        }
        else
        {
            ++optionIndex;
        }
    }

    // さらに、system_id を最後に移動させる。
    optionIndex = 0;
    for (auto el_option = option_var_array.begin(); el_option != option_var_array.end(); ++el_option, ++optionIndex)
    {
        if (el_option->id.value == SYSTEM_ID)
        {
            option_var_array.push_back(*el_option);
            option_var_array.erase(option_var_array.begin() + optionIndex);
            break;
        }
    }

    // option_var に index を入れておく。
    // 以降、順序入れ替え禁止。
    for (int i = 0; i < optionCount; ++i)
    {
        option_var_array[i].index = i;
    }

    // material の uniform block のデフォルト値を expand
    // ついでに、option の uniform_block を取得しておく。
    std::vector<elem_uniform_var>* el_option_uniform_var_array = nullptr;
    for (auto el_block = block_var_array.begin(); el_block != block_var_array.end(); ++el_block)
    {
        if (el_block->type.value == elem_block_var::material)
        {
            el_block->Expand();
        }
        else if (el_block->type.value == elem_block_var::option)
        {
            el_option_uniform_var_array = &el_block->uniform_var_array;
        }
    }

    size_t keyLengthBit = 0;
    keyLength = 1;
    bool isStaticOption = false;
    for (auto el_option = option_var_array.begin(); el_option != option_var_array.end(); ++el_option)
    {
        const elem_uniform_var* el_option_uniform = nullptr;
        if (el_option_uniform_var_array != nullptr)
        {
            for (auto el_uniform = el_option_uniform_var_array->cbegin(); el_uniform != el_option_uniform_var_array->cend(); ++el_uniform)
            {
                if (el_uniform->id.value == el_option->id.value)
                {
                    el_option_uniform = &(*el_uniform);
                    break;
                }
            }
        }

        el_option->Expand(el_option_uniform);

        // キーの長さを計算します。
        el_option->keyOffset = keyLengthBit + (keyLength - 1) * 32;
        keyLengthBit += el_option->bitWidth;
        bool isDynamicOption = el_option->type.value == elem_option_var::type_dynamic;

        // 32 ビット境界を跨ぐ場合はパディングを挿入します。
        // また、static と dynamic オプションは異なる 32 ビットに入るようパディングを挿入します。
        if (keyLengthBit > 32 || (isStaticOption && isDynamicOption))
        {
            ++keyLength;
            keyLengthBit = el_option->bitWidth;
            el_option->keyOffset = (keyLength - 1) * 32;
        }
        isStaticOption = !isDynamicOption;
    }

    // デフォルトキーを生成する
    defaultKey.reset(malloc(keyLength * sizeof(u32)));
    std::vector<int> defaultOptionArray;
    defaultOptionArray.resize(option_var_array.size(), 0);
    optionIndex = 0;
    for (auto el_option = option_var_array.cbegin(); el_option != option_var_array.cend(); ++el_option, ++optionIndex)
    {
        defaultOptionArray[optionIndex] = el_option->defaultValueIndex;
    }
    CreateKey(static_cast<u32*>(defaultKey.get()), defaultOptionArray);
}

void elem_shading_model::CreateKey(u32* key, const std::vector<int>& choiceArray) const
{
    const int BIT_WIDTH = 32;
    memset(key, 0, sizeof(u32) * keyLength);

    int optionIndex = 0;
    for (auto option_var = option_var_array.cbegin(); option_var != option_var_array.cend(); ++option_var, ++optionIndex)
    {
        u32 keyByte  = static_cast<u32>(option_var->keyOffset / BIT_WIDTH);
        u32 keyBit   = static_cast<u32>(option_var->keyOffset % BIT_WIDTH);
        u32 bitWidth = static_cast<u32>(option_var->bitWidth);

        // 書き込み先バッファを取得
        u32* dst = &key[keyByte];
        util::BitUtil<u32> bit(*dst);
        bit.SetHiBitOn(keyBit, keyBit + bitWidth - 1, choiceArray[optionIndex]);
        *dst = bit;
    }
}

void elem_option_var::Filter(util::StringRef choice_filter, util::StringRef default_filter)
{
    std::string tokenFilter(choice_filter.Str());

    choice.value = util::Trim(choice.value);
    tokenFilter = util::Trim(tokenFilter);

    std::vector<std::pair<std::string, std::string>> shaderChoiceArray;
    ExpandChoice(shaderChoiceArray, choice.value);

    std::vector<std::pair<std::string, std::string>> configChoiceArray;
    ExpandChoice(configChoiceArray, tokenFilter);

    // シェーダで定義されたデフォルト値を fsc で補正する
    if (!default_filter.Empty())
    {
        default_value.value = default_filter.Str();
    }

    if (choice_filter.Empty())
    {
        // option_filter が指定されてない場合は choice にデフォルト値を設定する。
        // デフォルト値が存在しない場合は最初の choice とする。

        int defaultValue = IncludeValue(shaderChoiceArray, default_value.value);

        if (defaultValue < 0)
        {
            THROW_ERROR(ERRCODE_SHADER_CONVERTER_INVALID_OPTION_DEFAULT_VALUE_NOT_FOUND,
                "A default value is not found in filtered choices: %hs (%hs)", default_value.value.c_str(), id.value.c_str());
        }

        defaultValueIndex = 0;
        choice.value = shaderChoiceArray[defaultValue].first;
        if (!shaderChoiceArray[defaultValue].second.empty())
        {
            choice.value += ":" + shaderChoiceArray[defaultValue].second;
        }
    }
    else if (choice_filter == str_wild_card)
    {
        // choice_filter に * が指定された場合は
        // choice には変更を加えず、シェーダの指定通り扱う
    }
    else
    {
        bool hasAlias = false;
        std::vector<std::pair<std::string, std::string>> validArray;
        // configChoiceArray のすべての値が shaderChoiceArray に含まれることを確認する。
        for (auto config = configChoiceArray.cbegin(); config != configChoiceArray.cend(); ++config)
        {
            auto comp = std::find_if(shaderChoiceArray.cbegin(), shaderChoiceArray.cend(), ChoiceFinder(config->first));
            if (comp == shaderChoiceArray.cend())
            {
                THROW_ERROR(ERRCODE_SHADER_CONVERTER_INVALID_OPTION_FILTER,
                    "Invalid variation choice: %hs (%hs)", config->first.c_str(), id.value.c_str());
            }
            else
            {
                if (!comp->second.empty())
                {
                    hasAlias = true;
                }
                validArray.emplace_back(*comp);
            }
        }

        // alias が存在している場合は個々の要素をバラバラに展開する。
        if (hasAlias)
        {
            choice.value = "";
            for (auto valid = validArray.cbegin(); valid != validArray.cend(); ++valid)
            {
                if (valid != validArray.cbegin())
                {
                    choice.value += ", ";
                }

                choice.value += valid->first;
                if (!valid->second.empty())
                {
                    choice.value += ":" + valid->second;
                }
            }
        }
        else
        {
            // 全て存在していることが確認できたら filter で choice を置き換える。
            choice.value = choice_filter.Str();
        }

        // default_filter が空でかつ default が存在しない場合、その default は採用しない
        if (default_filter.Empty() && IncludeValue(configChoiceArray, default_value.value) < 0)
        {
            default_value.value = std::string();
        }
    }
}

void elem_option_var::Expand(const elem_uniform_var* el_option_uniform)
{
    choice.value = util::Trim(choice.value);
    default_value.value = util::Trim(default_value.value);

    ExpandChoice(choiceArray, choice.value);

    // choice を表すのに必要なビット長を求める。
    int choiceBitWidth = 1;
    for (int i = 2; i < static_cast<int>(choiceArray.size()); i<<=1, ++choiceBitWidth)
    {
        ;
    }

    this->bitWidth = choiceBitWidth;

    // デフォルト値が choiceArray に含まれているか確認する。
    if (!default_value.value.empty())
    {
        defaultValueIndex = IncludeValue(choiceArray, default_value.value);

        if (defaultValueIndex < 0)
        {
            THROW_ERROR(ERRCODE_SHADER_CONVERTER_INVALID_OPTION_DEFAULT_VALUE_NOT_FOUND,
                "A default value is not found in choice: %hs (%hs)", default_value.value.c_str(), id.value.c_str());
        }
    }
    else
    {
        // デフォルト値が空の場合は choice の最初の要素を設定する。
        default_value.value = choiceArray[0].first;
        defaultValueIndex = 0;
    }

    // uniform による代替文字列が存在している場合は入れておく
    // branch=ture かつ static オプションは uniformblock での別定義が必要
    if (branch.value && type.value == elem_option_var::type_static)
    {
        ExpandChoiceUint(choiceUintArray, choiceArray);
        if (choiceArray.size() != choiceUintArray.size())
        {
            THROW_ERROR(ERRCODE_SHADER_CONVERTER_INVALID_BRACH_CHOICE,
                "A choice using branch must be \"int\" or \"bool\" values (%hs).", id.value.c_str());
        }

        for (int stageIndex = 0; stageIndex < ShaderStage_StageCount; ++stageIndex)
        {
            if (stageIndex == ShaderStage_Compute)
            {
                continue;
            }
            if (shader_symbol[stageIndex])
            {
                if (el_option_uniform == nullptr || !(el_option_uniform->shader_symbol[stageIndex]))
                {
                    THROW_TRANSLATED_ERROR(ERRCODE_SHADER_CONVERTER_ALTERNATIVE_OPTION_NOT_FOUND, "Identifier_NoAlternativeOption", id.value.c_str());
                }

                altUniformSymbol[stageIndex] = el_option_uniform->shader_symbol[stageIndex]->name.value;
            }
        }
    }
}

void elem_block_var::Expand()
{
    // material の場合のみデフォルト値が入る
    if (type.value == material)
    {
        for (auto el_uniform = uniform_var_array.begin(); el_uniform != uniform_var_array.end(); ++el_uniform)
        {
            el_uniform->Expand();
        }
    }
}

void elem_ssbo_block_var::Expand()
{
    // do nothing.
}

void elem_uniform_var::Expand()
{
    if (default_value.value.empty())
    {
        dataSize = 0;
    }
    else
    {
        std::pair<void*, size_t> data;
        try
        {
            data = elem_shader_param::TextToValue(default_value->c_str(), type.value);
        }
        CATCH_THROW_WITH_MSG("Error occurred in uniform_var \"%hs\"", this->id.value.c_str());

        rawdata.reset(data.first, free);
        dataSize = data.second;
    }
}

void elem_ssbo_uniform_var::Expand()
{
    dataSize = 0;
}

} // namespace g3dif

} // namespace tool
} // namespace g3d
} // namespace nw
