﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.IO;
using System.Globalization;
using System.Text;

namespace nw.g3d.toollib
{
    // 引数パーサ
    public class ArgumentPerser
    {
        // オプション値のモード
        public enum OptionValueMode
        {
            // 必須
            Require,
            // 省略可
            Option,
            // 無し
            None,
        }

        // コンストラクタ
        public ArgumentPerser()
        {
            string[] args = Environment.GetCommandLineArgs();
            for (int i = 1; i < args.Length; i++)
            {
                this.Arguments.Add(args[i]);
            }
        }

        // プログラムオプション登録
        public void RegisterProgramOption(string name, string shortName, OptionValueMode valueMode)
        {
            Nintendo.Foundation.Contracts.Assertion.Argument.True((name != null) && (shortName != null));
            Nintendo.Foundation.Contracts.Assertion.Argument.True(name.StartsWith("--") && shortName.StartsWith("-"));

            _programOptionNameDic.Add(shortName, name);
            _programOptionValueDic.Add(name, valueMode);
        }

        // ファイルオプション登録
        public void RegisterFileOption(string name, string shortName, OptionValueMode valueMode)
        {
            Nintendo.Foundation.Contracts.Assertion.Argument.True((name != null) && (shortName != null));
            Nintendo.Foundation.Contracts.Assertion.Argument.True(name.StartsWith("--") && shortName.StartsWith("-"));

            _fileOptionNameDic.Add(shortName, name);
            _fileOptionValueDic.Add(name, valueMode);
        }

        // パース
        // @ による複数 ProgramOption 対応は、PerseMulti() を別途作成する
        public ProgramOption Perse()
        {
            // 引数リストの構築
            List<string> argList = new List<string>();

            foreach (string arg in this.Arguments)
            {
                if (arg.StartsWith("@"))
                {
                    // arg.Length == 1 でプログラムオプション分割に対応可能
                    ParseArgFile(arg.Substring(1), argList);
                }
                else
                {
                    argList.Add(arg);
                }
            }

            // 引数リストの解析
            return AnalyzeArgumentList(argList);
        }

        //---------------------------------------------------------------------
        // 引数ファイルのパース
        // 行を跨ぐダブルクォーテーションエスケープは非サポート
        private void ParseArgFile(string argFilePath, List<string> argList)
        {
            char[] delimiter = new char[] { ' ', '\t', '\r', '\n' };

            // システムのロケールに対応するには、起動時にロケールを退避する必要がある
            Encoding encoding = Encoding.GetEncoding(CultureInfo.CurrentCulture.TextInfo.ANSICodePage);
            if (argFilePath.StartsWith("~"))
            {
                // HACK: ファイルイベントから出力される引数ファイルが UTF-8 BOM 無しであるため
                // 日本語パスがある場合に正しく読み込みできない問題があります。
                // 引数ファイルを @~argfile.txt と指定すると UTF-8 BOM 無しで読み込むようにします。
                encoding = new UTF8Encoding(false);
                argFilePath = argFilePath.Substring(1);
            }

            string[] lines = File.ReadAllLines(argFilePath, encoding);

            foreach (string originalLine in lines)
            {
                // トリムしておく
                string line = originalLine.Trim();

                // コメントの読み飛ばし
                if (line.StartsWith("#")) { continue; }

                while (line.Length != 0)
                {
                    if (line.StartsWith("-"))
                    {
                        // オプション
                        int quotIndex = line.IndexOf('"');
                        int delIndex = line.IndexOfAny(delimiter);
                        if (delIndex == -1)
                        {
                            if (quotIndex == -1)
                            {
                                argList.Add(line);
                                line = string.Empty;
                            }
                            else
                            {
                                argList.Add(line.Replace("\"", string.Empty));
                                line = string.Empty;
                            }
                        }
                        else if ((quotIndex != -1) && (quotIndex < delIndex))
                        {
                            // ダブルクォーテーションをエスケープしてオプション取り出し
                            int index = line.IndexOf('"', quotIndex + 1);
                            if (index == -1)
                            {
                                TlStrings.Throw(
                                    "ArgumentPerser_DoubleQuotNotClose", originalLine);
                            }
                            argList.Add(line.Substring(0, index).Replace("\"", string.Empty));
                            line = line.Substring(index + 1).TrimStart();
                        }
                        else
                        {
                            // オプション取り出し
                            argList.Add(line.Substring(0, delIndex));
                            line = line.Substring(delIndex + 1).TrimStart();
                        }
                    }
                    else
                    {
                        // パス
                        if (line.StartsWith("\""))
                        {
                            // ダブルクォーテーションをエスケープしてパス取り出し
                            line = line.Substring(1);
                            int index = line.IndexOf('"');
                            if (index == -1)
                            {
                                TlStrings.Throw(
                                    "ArgumentPerser_DoubleQuotNotClose", originalLine);
                            }
                            argList.Add(line.Substring(0, index));
                            line = line.Substring(index + 1).TrimStart();
                        }
                        else
                        {
                            // パス取り出し
                            int index = line.IndexOfAny(delimiter);
                            if (index == -1)
                            {
                                argList.Add(line);
                                line = string.Empty;
                            }
                            else
                            {
                                argList.Add(line.Substring(0, index));
                                line = line.Substring(index + 1).TrimStart();
                            }
                        }
                    }
                }
            }
        }

        //---------------------------------------------------------------------
        // 引数リストの解析
        private ProgramOption AnalyzeArgumentList(List<string> argList)
        {
            ProgramOption programOption = new ProgramOption();
            FileOption fileOption = null;
            bool programMode = true;
            foreach (string arg in argList)
            {
                if (programMode)
                {
                    // プログラムオプションの解析
                    if (arg.StartsWith("-"))
                    {
                        programOption.Options.Add(AnalyzeArgumentOption(arg, true));
                        continue;
                    }
                    else
                    {
                        programMode = false;
                    }
                }

                // ファイルオプションの解析
                if (arg.StartsWith("-"))
                {
                    fileOption.Options.Add(AnalyzeArgumentOption(arg, false));
                }
                else
                {
                    fileOption = new FileOption(arg);
                    programOption.FileOptions.Add(fileOption);
                }
            }
            return programOption;
        }

        // 引数オプションの解析
        private ArgumentOption AnalyzeArgumentOption(string arg, bool isProgramOption)
        {
            string optName = arg;
            string optValue = null;

            // 値の取り出し
            int index = arg.IndexOf('=');
            if (index != -1)
            {
                optName = arg.Substring(0, index);
                optValue = arg.Substring(index + 1);
            }

            // ショートネームを通常名に変換
            if (!optName.StartsWith("--"))
            {
                if (isProgramOption)
                {
                    if (!_programOptionNameDic.ContainsKey(optName))
                    {
                        TlStrings.Throw(
                            "ArgumentPerser_UnknownProgramOption", arg);
                    }
                    optName = _programOptionNameDic[optName];
                }
                else
                {
                    if (!_fileOptionNameDic.ContainsKey(optName))
                    {
                        TlStrings.Throw(
                            "ArgumentPerser_UnknownFileOption", arg);
                    }
                    optName = _fileOptionNameDic[optName];
                }
            }

            // 値のモード取得
            OptionValueMode valueMode;
            if (isProgramOption)
            {
                if (!_programOptionValueDic.ContainsKey(optName))
                {
                    TlStrings.Throw(
                        "ArgumentPerser_UnknownProgramOption", arg);
                }
                valueMode = _programOptionValueDic[optName];
            }
            else
            {
                if (!_fileOptionValueDic.ContainsKey(optName))
                {
                    TlStrings.Throw(
                        "ArgumentPerser_UnknownFileOption", arg);
                }
                valueMode = _fileOptionValueDic[optName];
            }

            // 値のエラーチェック
            if ((valueMode == OptionValueMode.Require) && (optValue == null))
            {
                TlStrings.Throw(
                    "ArgumentPerser_RequireOptionValue", arg);
            }
            if ((valueMode == OptionValueMode.None) && (optValue != null))
            {
                TlStrings.Throw(
                    "ArgumentPerser_NoneOptionValue", arg);
            }

            ArgumentOption argOpt = new ArgumentOption(arg, optName);
            if (optValue != null) { argOpt.Value = optValue; }
            return argOpt;
        }

        //---------------------------------------------------------------------
        // 引数
        public readonly List<string> Arguments = new List<string>();

        // プログラムオプション辞書
        private Dictionary<string, string> _programOptionNameDic =
            new Dictionary<string, string>();
        private Dictionary<string, OptionValueMode> _programOptionValueDic =
            new Dictionary<string, OptionValueMode>();

        // ファイルオプション辞書
        private Dictionary<string, string> _fileOptionNameDic =
            new Dictionary<string, string>();
        private Dictionary<string, OptionValueMode> _fileOptionValueDic =
            new Dictionary<string, OptionValueMode>();
    }
}
