﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

// DecodeFtx
// OutputFtxFile OutputFtxXml ROriginalImage_Output
// RMergeInfo

//=============================================================================
// include
//=============================================================================
#include <Sources/ImageConvert.h>

#include <wincrypt.h> // ハッシュ計算に必要
#pragma comment(lib, "Advapi32.lib") // ハッシュ計算に必要（Advapi32.dll 使用）

using namespace std;

//=============================================================================
// texcvtr ネームスペースを開始します。
//=============================================================================
namespace nn {
namespace gfx {
namespace tool {
namespace texcvtr {

//-----------------------------------------------------------------------------
// 無名名前空間を開始します。
namespace
{

//=============================================================================
// constants
//=============================================================================
const char* IntermediateFileVersion = "4.0.0"; //!< <nw4f_3dif> 要素のバージョンです。
const char* TextureElementVersion = "4.0.0"; //!< <texture> 要素のバージョンです。

const std::string FtxFileInfoNone = "<none>"; //!< 入力 ftx に <file_info> がなかった場合の文字列です。

//-----------------------------------------------------------------------------
//! @brief メモリー上のデータ群からハッシュ値を計算します。
//!
//! @param[in] pValue ハッシュ値を格納するバッファへのポインターです。
//! @param[in] valueSize ハッシュ値を格納するバッファのサイズです。
//! @param[in] pDatas データへのポインター配列です。
//! @param[in] dataSizes pDatas に対応したデータサイズ配列です。
//! @param[in] algorithm ハッシュのアルゴリズム ID です。
//!                      （CALG_MD5 / CALG_SHA1 / CALG_SHA_256 / CALG_SHA_512 など）
//!
//! @return 処理成功なら true を返します。
//-----------------------------------------------------------------------------
bool CalculateHashValue(
    uint8_t* pValue,
    const size_t valueSize,
    std::vector<const void*> pDatas,
    std::vector<size_t> dataSizes,
    const ALG_ID algorithm
)
{
    const size_t dataCount = pDatas.size();
    if (dataCount == 0 || dataSizes.size() != dataCount)
    {
        cerr << "Error: The parameter to calculate hash value is wrong" << endl;
        return false;
    }

    bool isSucceeded = false;
    const DWORD provType = PROV_RSA_AES;
    HCRYPTPROV hCryptProv;
    if (CryptAcquireContextA(&hCryptProv, nullptr, nullptr, provType, CRYPT_VERIFYCONTEXT))
    {
        HCRYPTHASH hHash;
        if (CryptCreateHash(hCryptProv, algorithm, 0, 0, &hHash))
        {
            bool isDataOk = true;
            for (size_t dataIdx = 0; dataIdx < dataCount; ++dataIdx)
            {
                if (!CryptHashData(hHash, reinterpret_cast<const BYTE*>(pDatas[dataIdx]),
                    static_cast<DWORD>(dataSizes[dataIdx]), 0))
                {
                    cerr << "Error: Cannot add crypt hash data" << endl;
                    isDataOk = false;
                    break;
                }
            }
            if (isDataOk)
            {
                DWORD valueSizeRet = static_cast<DWORD>(valueSize);
                if (CryptGetHashParam(hHash, HP_HASHVAL, pValue, &valueSizeRet, 0))
                {
                    isSucceeded = true;
                }
                else
                {
                    cerr << "Error: Cannot get crypt hash value: data length = " << valueSize << " / " << valueSizeRet << endl;
                }
            }
            if (!CryptDestroyHash(hHash))
            {
                cerr << "Error: Cannot destroy crypt hash" << endl;
            }
        }
        else
        {
            cerr << "Error: Cannot create crypt hash: ALG_ID = 0x" << hex << algorithm << dec << endl;
        }
        if (!CryptReleaseContext(hCryptProv, 0))
        {
            cerr << "Error: Cannot release crypt context" << endl;
        }
    }
    else
    {
        cerr << "Error: Cannot acquire crypt context: provType = " << provType << endl;
    }
    return isSucceeded;
}

//-----------------------------------------------------------------------------
//! @brief データ列群からハッシュ値文字列（SHA-512 の Base64 形式）を取得します。
//!
//! @param[in] dataStreams データ列配列です。
//! @param[in] streamIdxs ハッシュ値計算の対象となるデータ列の dataStreams 内インデックス配列です。
//!
//! @return ハッシュ値文字列を返します。
//-----------------------------------------------------------------------------
std::string GetHashStringFromStreams(
    const RDataStreamArray& dataStreams,
    const std::vector<int>& streamIdxs
)
{
    std::vector<const void*> pDatas;
    std::vector<size_t> dataSizes;
    for (size_t blockIdx = 0; blockIdx < streamIdxs.size(); ++blockIdx)
    {
        const RDataStream& dataStream = dataStreams[streamIdxs[blockIdx]];
        const void* pData = dataStream.GetDataPtr();
        if (pData != nullptr)
        {
            pDatas.push_back(pData);
            dataSizes.push_back(dataStream.GetSize());
        }
    }

    std::string hashStr;
    uint8_t sha512[512 / 8];
    if (CalculateHashValue(sha512, sizeof(sha512), pDatas, dataSizes, CALG_SHA_512))
    {
        const size_t base64BufSize = GetBase64EncodedSize(sizeof(sha512));
        char* base64Buf = new char[base64BufSize];
        const size_t base64Size = EncodeToBase64(base64Buf, sha512, sizeof(sha512));
        hashStr = std::string(base64Buf, base64Size);
        delete[] base64Buf;
    }
    return hashStr;
}

//-----------------------------------------------------------------------------
//! @brief 中間ファイルの文字列からリニア変換フラグを取得します。
//!
//! @param[in] str リニア変換フラグを表す文字列です。
//!
//! @return リニア変換フラグを返します。
//-----------------------------------------------------------------------------
int GetLinearFlagFromString(const std::string& str)
{
    RStringArray tokens;
    if (RTokenizeString(tokens, str) == R_RGBA_COUNT)
    {
        int linearFlag = FtxOpt::LINEAR_NONE;
        for (int iRgba = 0; iRgba < R_RGBA_COUNT; ++iRgba)
        {
            if (tokens[iRgba] == "true")
            {
                linearFlag |= LinearRgbaFlags[iRgba];
            }
        }
        return linearFlag;
    }
    return FtxOpt::LINEAR_NONE;
}

//-----------------------------------------------------------------------------
//! @brief リニア変換フラグから中間ファイルの文字列を取得します。
//-----------------------------------------------------------------------------
std::string GetLinearFlagString(const int linearFlag)
{
    std::string str;
    for (int iRgba = 0; iRgba < R_RGBA_COUNT; ++iRgba)
    {
        const bool linear = (linearFlag >= 0 && (linearFlag & LinearRgbaFlags[iRgba]) != 0);
        if (iRgba >= 1)
        {
            str += " ";
        }
        str += RBoolStr(linear);
    }
    return str;
}

//-----------------------------------------------------------------------------
//! @brief 中間ファイルの文字列から BC 圧縮重み付けを取得します。
//!
//! @param[in] str BC 圧縮重み付けを表す文字列です。
//!
//! @return BC 圧縮重み付けを返します。
//-----------------------------------------------------------------------------
int GetWeightedCompressFromString(const std::string& str)
{
    return (str == "true") ?
        FtxOpt::WEIGHTED_COMPRESS_ENABLE : FtxOpt::WEIGHTED_COMPRESS_DISABLE;
}

//-----------------------------------------------------------------------------
//! @brief ftx ファイルの編集用コメントを取得します。
//!
//! @param[out] pCommentLabel 編集用コメントラベルへのポインターです。
//! @param[out] pCommentCol 編集用コメントカラー文字列へのポインターです。
//! @param[out] pCommentText 編集用コメント文へのポインターです。
//! @param[in] textureElem <texture> XML 要素です。
//-----------------------------------------------------------------------------
void GetFtxComment(
    std::string* pCommentLabel,
    std::string* pCommentCol,
    std::string* pCommentText,
    const RXMLElement* textureElem
)
{
    const RXMLElement* commentElem = textureElem->FindElement("comment", false);
    if (commentElem != nullptr)
    {
        *pCommentLabel = RDecodeXmlString(
            RGetShiftJisFromUtf8(commentElem->GetAttribute("label", "", false))); // 4.0.0 で追加
        *pCommentCol = commentElem->GetAttribute("color");
        *pCommentText = RDecodeXmlString(
            RGetShiftJisFromUtf8(commentElem->GetAttribute("text")));
    }
}

//-----------------------------------------------------------------------------
//! @brief ftx ファイルのツールデータとユーザーツールデータを取得します。
//!
//! @param[out] toolData <tool_data> 要素の内容を格納します。
//! @param[out] userToolData <user_tool_data> 要素の内容を格納します。
//! @param[in] xmlStr ftx ファイルの XML 部分の文字列です。
//-----------------------------------------------------------------------------
void GetFtxToolData(
    std::string& toolData,
    std::string& userToolData,
    const std::string& xmlStr
)
{
    //-----------------------------------------------------------------------------
    // get tool data
    const std::string TOOL_DATA_BEGIN = "<tool_data";
    const std::string TOOL_DATA_END = "</tool_data>";
    const size_t iToolData = xmlStr.find(TOOL_DATA_BEGIN);
    if (iToolData != std::string::npos)
    {
        const size_t iEndToolData = xmlStr.find(TOOL_DATA_END, iToolData);
        if (iEndToolData != std::string::npos)
        {
            toolData = RGetShiftJisFromUtf8(xmlStr.substr(
                iToolData, iEndToolData + TOOL_DATA_END.size() - iToolData));
            //cerr << "tool data: [" << toolData << "]" << endl;
        }
    }

    //-----------------------------------------------------------------------------
    // get user tool data
    const std::string USER_TOOL_DATA_BEGIN = "<user_tool_data";
    const std::string USER_TOOL_DATA_END = "</user_tool_data>";
    const size_t iUserToolData = xmlStr.find(USER_TOOL_DATA_BEGIN);
    if (iUserToolData != std::string::npos)
    {
        const size_t iEndUserToolData = xmlStr.find(USER_TOOL_DATA_END, iUserToolData);
        if (iEndUserToolData != std::string::npos)
        {
            userToolData = RGetShiftJisFromUtf8(xmlStr.substr(
                iUserToolData, iEndUserToolData + USER_TOOL_DATA_END.size() - iUserToolData));
            //cerr << "user tool data: [" << userToolData << "]" << endl;
        }
    }
}

//-----------------------------------------------------------------------------
//! @brief ファイル情報に出力する元ファイルのパスを取得します。
//!
//! @param[in] fullPath 元ファイルのフルパスです。
//! @param[in] projectRootPath プロジェクトのルートフォルダーのパスです。
//!                            空文字でなければ元ファイルのパスはルートフォルダーからの相対パスとなります。
//!
//! @return 元ファイルのパスを返します。
//-----------------------------------------------------------------------------
std::string GetSourceFilePath(
    const std::string& fullPath,
    const std::string& projectRootPath
)
{
    std::string srcPath(fullPath);
    if (!projectRootPath.empty())
    {
        char relativePath[MAX_PATH];
        if (::PathRelativePathToA(relativePath,
            RGetWindowsFilePath(projectRootPath).c_str(), FILE_ATTRIBUTE_DIRECTORY,
            RGetWindowsFilePath(fullPath       ).c_str(), FILE_ATTRIBUTE_NORMAL))
        {
            srcPath = RGetUnixFilePath(relativePath);
            if (srcPath.find("./") == 0)
            {
                srcPath = srcPath.substr(2);
            }
        }
    }
    return srcPath;
}

//-----------------------------------------------------------------------------
//! @brief ftx ファイルの XML 部分を出力し、データ列配列を作成します。
//!
//! @param[in,out] os 出力ストリームです。
//! @param[in,out] pDataStreams データ列配列へのポインタです。
//! @param[in,out] pImage 画像へのポインタです。
//! @param[in,out] pOriginalImages 元画像配列へのポインタです。
//! @param[in] isBinary バイナリー形式なら true を指定します。
//! @param[in] opt ftx 変換オプションです。
//! @param[in] toolVersionStr ツールバージョン文字列です。
//! @param[in] overwrites 入力ファイルと出力ファイルが同じなら true です。
//! @param[in] freesMemory メモリーの最大使用量を減らすために内部で画像と元画像の
//!                        メモリーを解放するなら true です。
//!
//! @return 処理結果を返します。
//-----------------------------------------------------------------------------
RStatus OutputFtxXml(
    std::ostream& os,
    RDataStreamArray* pDataStreams,
    RImage* pImage,
    ROriginalImageArray* pOriginalImages,
    const bool isBinary,
    const FtxOpt& opt,
    const std::string& toolVersionStr,
    const bool overwrites,
    const bool freesMemory
)
{
    //-----------------------------------------------------------------------------
    // 元画像データの出力フラグを設定します。
    // ftx ファイルから ftx ファイルへの変換で、入力ファイルに元画像データがない場合、
    // --disable-original-image を指定しなくても元画像データを出力しません。
    RImage& rimg = *pImage;
    ROriginalImageArray& originalImages = *pOriginalImages;
    const bool outputsOriginal = !opt.m_DisablesOriginalImage && !rimg.IsDecodedOriginal();

    //-----------------------------------------------------------------------------
    // write UTF-8 BOM
    ROutUtf8Bom(os);

    //-----------------------------------------------------------------------------
    // init stream format
    RInitOutStreamFormat(os);

    //-----------------------------------------------------------------------------
    // xml header
    const int tc = 0;
    os << RTab(tc) << "<?xml version=\"1.0\" encoding=\"utf-8\"?>" << R_ENDL;

    //-----------------------------------------------------------------------------
    // begin intermediate file
    os << RTab(tc) << "<nw4f_3dif version=\"" << IntermediateFileVersion
       << "\">" << R_ENDL;

    //-----------------------------------------------------------------------------
    // file info
    std::string orgCreateInfo;
    std::string orgModifyInfo;
    if (!originalImages.empty())
    {
        if (overwrites || opt.m_DisablesFileInfo)
        {
            orgCreateInfo = originalImages[0].m_CreateInfo;
        }
        if (opt.m_DisablesFileInfo)
        {
            orgModifyInfo = originalImages[0].m_ModifyInfo;
        }
    }

    const std::string inputExt = RGetExtensionFromFilePath(opt.m_InputPaths[0]);
    const bool isFtxInput = (inputExt == FtxaExtension || inputExt == FtxbExtension);
    if (orgCreateInfo != FtxFileInfoNone &&
        !(!isFtxInput && opt.m_DisablesFileInfo))
    {
        os << RTab(tc) << "<file_info>" << R_ENDL;
        if (!orgCreateInfo.empty())
        {
            os << RGetUtf8FromShiftJis(orgCreateInfo);
        }
        if (orgCreateInfo.empty() || !opt.m_DisablesFileInfo)
        {
            const std::string toolName = (!opt.m_ToolName.empty()) ?
                opt.m_ToolName : "TextureConverter";
            os << RTab(tc + 1) << "<" << (orgCreateInfo.empty() ? "create" : "modify")
               << " tool_name=\"" << RGetUtf8FromShiftJis(REncodeXmlString(toolName))
               << "\" tool_version=\"" << toolVersionStr << "\"";
            if (orgCreateInfo.empty())
            {
                os << R_ENDL;
                const std::string srcPath = GetSourceFilePath(opt.m_InputPaths[0], opt.m_ProjectRootPath);
                os << RTab(tc + 2) << "src_path=\"" << RGetUtf8FromShiftJis(REncodeXmlString(srcPath)) << "\"" << R_ENDL;
                os << RTab(tc + 1) << "/>" << R_ENDL;
            }
            else
            {
                os << " />" << R_ENDL;
            }
        }
        else if (!orgModifyInfo.empty())
        {
            os << RGetUtf8FromShiftJis(orgModifyInfo);
        }
        os << RTab(tc) << "</file_info>" << R_ENDL;
    }

    //-----------------------------------------------------------------------------
    // begin texture
    os << RTab(tc) << "<texture version=\"" << TextureElementVersion << "\">" << R_ENDL;

    //-----------------------------------------------------------------------------
    // add texture data stream
    // 一時的な最大メモリー確保量を減らすため、
    // 先にデータ列配列に長さ 1 のデータ列を追加してからリサイズします。
    RDataStreamArray& dataStreams = *pDataStreams;
    const int encodedStreamIdx = static_cast<int>(dataStreams.size());

    const uint8_t dummyArray[1] = { 0 };
    dataStreams.push_back(RDataStream(dummyArray, sizeof(dummyArray), 16));
    RDataStream& texStream = dataStreams.back();
    texStream.m_ByteValues.clear();

    texStream.Append(rimg.GetImageData(), rimg.GetImageDataSize());

    //-----------------------------------------------------------------------------
    // texture info
    os << RTab(tc) << "<texture_info" << R_ENDL;
    os << RTab(tc + 1) << "dcc_preset=\"" << RGetUtf8FromShiftJis(REncodeXmlString(rimg.GetDccPreset())) << "\"" << R_ENDL;
    os << RTab(tc + 1) << "hint=\"" << rimg.GetHint() << "\"" << R_ENDL;
    os << RTab(tc + 1) << "linear=\"" << GetLinearFlagString(rimg.GetLinearFlag()) << "\"" << R_ENDL;
    os << RTab(tc + 1) << "dimension=\"" << RGetTextureDimensionString(
        static_cast<FtxDimension>(rimg.GetDimension())) << "\"" << R_ENDL;
    os << RTab(tc + 1) << "quantize_type=\"" << RGetTextureFormatString(static_cast<FtxFormat>(rimg.GetFormat())) << "\"" << R_ENDL;
    os << RTab(tc + 1) << "width=\"" << rimg.GetImageW()
       << "\" height=\"" << rimg.GetImageH()
       << "\" depth=\"" << rimg.GetImageD() << "\"" << R_ENDL;
    os << RTab(tc + 1) << "mip_level=\"" << rimg.GetMipCount() << "\"" << R_ENDL;
    os << RTab(tc + 1) << "mip_gen_filter=\"" << FtxOpt::GetMipGenFilterString(rimg.GetMipGenFilter()) << "\"" << R_ENDL;
    os << RTab(tc + 1) << "size=\"" << texStream.GetCount() << "\"" << R_ENDL;
    os << RTab(tc + 1) << "comp_sel=\"" << RGetCompSelString(rimg.GetCompSel()) << "\"" << R_ENDL;
    os << RTab(tc + 1) << "weighted_compress=\"" << RBoolStr(rimg.GetWeightedCompress() == FtxOpt::WEIGHTED_COMPRESS_ENABLE) << "\"" << R_ENDL;
    os << RTab(tc + 1) << "initial_swizzle=\"" << rimg.GetInitialSwizzle() << "\"" << R_ENDL;
    std::string levelOffsetStr;
    for (int level = 0; level < rimg.GetMipCount(); ++level)
    {
        if (level > 0)
        {
            levelOffsetStr += " ";
        }
        levelOffsetStr += RGetNumberString(static_cast<uint32_t>(rimg.GetLevelOffset(level)));
    }
    os << RTab(tc + 1) << "level_offset=\"" << levelOffsetStr << "\"" << R_ENDL;

    //-----------------------------------------------------------------------------
    // 必要なら画像のメモリーを解放します。
    // ユーザーデータ、編集用コメント、ツールデータ、ユーザーツールデータは
    // 後で出力するのでコピーします。
    const std::string srcOriginalImageHash = rimg.GetOriginalImageHash();
    const RUserDataArray userDatas = rimg.GetUserDatas();
    const std::string commentLabel = rimg.GetCommentLabel();
    const std::string commentCol   = rimg.GetCommentCol();
    const std::string commentText  = rimg.GetCommentText();
    const std::string toolData     = rimg.GetToolData();
    const std::string userToolData = rimg.GetUserToolData();
    if (freesMemory)
    {
        rimg.Clear();
    }

    //-----------------------------------------------------------------------------
    // 元画像配列の XML 文字列を作成します。
    // 必要なら元画像のメモリーを解放します。
    std::ostringstream orgOss;
    RInitOutStreamFormat(orgOss);
    std::vector<int> originalStreamIdxs;
    const int originalImageCount = static_cast<int>(originalImages.size());
    orgOss << RTab(tc) << "<original_image_array length=\"" << originalImageCount << "\">" << R_ENDL;
    for (int orgImgIdx = 0; orgImgIdx < originalImageCount; ++orgImgIdx)
    {
        ROriginalImage& originalImage = originalImages[orgImgIdx];
        const int originalStreamIdx = static_cast<int>(dataStreams.size());
        originalStreamIdxs.push_back(originalStreamIdx);
        originalImage.AppendStream(dataStreams);
        originalImage.Output(orgOss, tc + 1, orgImgIdx, originalStreamIdx);
        if (freesMemory)
        {
            originalImage.Clear();
        }
    }
    orgOss << RTab(tc) << "</original_image_array>" << R_ENDL;

    //-----------------------------------------------------------------------------
    // 元画像から計算したハッシュ値を出力して、<texture_info> を終了します。
    const std::string originalImageHash = (!originalStreamIdxs.empty()) ?
        GetHashStringFromStreams(dataStreams, originalStreamIdxs) :
        srcOriginalImageHash;
    os << RTab(tc + 1) << "original_image_hash=\"" << originalImageHash << "\"" << R_ENDL;
    os << RTab(tc + 1) << "stream_index=\"" << encodedStreamIdx << "\"" << R_ENDL;
    os << RTab(tc) << "/>" << R_ENDL;

    //-----------------------------------------------------------------------------
    // 元画像配列を出力します。
    if (!originalStreamIdxs.empty())
    {
        if (outputsOriginal)
        {
            os << orgOss.str();
        }
        else
        {
            // 元画像を出力しない場合は、元画像のデータ列を削除します。
            dataStreams.erase(dataStreams.begin() + encodedStreamIdx + 1, dataStreams.end());
        }
    }

    //-----------------------------------------------------------------------------
    // user data array
    const int userDataCount = static_cast<int>(userDatas.size());
    if (userDataCount != 0)
    {
        os << RTab(tc) << "<user_data_array length=\"" << userDataCount << "\">" << R_ENDL;
        for (int iData = 0; iData < userDataCount; ++iData)
        {
            userDatas[iData].Out(os, dataStreams, tc + 1, iData);
        }
        os << RTab(tc) << "</user_data_array>" << R_ENDL;
    }

    //-----------------------------------------------------------------------------
    // アスキー形式ならデータ列配列を XML 要素として出力します。
    if (!isBinary)
    {
        ROutArrayElement(os, tc, dataStreams, "stream_array");
    }

    //-----------------------------------------------------------------------------
    // 編集用コメント
    if (!commentText.empty())
    {
        os << "<comment label=\"" << RGetUtf8FromShiftJis(REncodeXmlString(commentLabel))
           << "\" color=\"" << commentCol
           << "\" text=\"" << RGetUtf8FromShiftJis(REncodeXmlString(commentText))
           << "\" />" << R_ENDL;
    }

    //-----------------------------------------------------------------------------
    // tool data & user tool data
    if (!toolData.empty())
    {
        os << RGetUtf8FromShiftJis(toolData) << R_ENDL;
    }

    if (!userToolData.empty())
    {
        os << RGetUtf8FromShiftJis(userToolData) << R_ENDL;
    }

    //-----------------------------------------------------------------------------
    // end texture
    os << RTab(tc) << "</texture>" << R_ENDL;

    //-----------------------------------------------------------------------------
    // end intermediate file
    os << RTab(tc) << "</nw4f_3dif>" << R_ENDL;

    return RStatus::SUCCESS;
} // NOLINT(impl/function_size)

//-----------------------------------------------------------------------------
// 無名名前空間を終了します。
} // unnamed namespace

//-----------------------------------------------------------------------------
//! @brief 元画像を出力します。
//-----------------------------------------------------------------------------
void ROriginalImage::Output( // ROriginalImage_Output
    std::ostream& os,
    const int tabCount,
    const int index,
    const int streamIndex
) const
{
    static const char* const s_FaceStrs[] =
    {
        "none",
        "positive_x",
        "negative_x",
        "positive_y",
        "negative_y",
        "positive_z",
        "negative_z"
    };
    static const char* const s_FormatStrs[] =
    {
        "rgba8",
        "rgb8",
        "rgba32f",
        "rgb32f"
    };

    const int& tc = tabCount;
    os << RTab(tc) << "<original_image index=\"" << index << "\"" << R_ENDL;
    os << RTab(tc + 1) << "slice_index=\"" << m_SliceIndex << "\"" << R_ENDL;
    os << RTab(tc + 1) << "face=\"" << s_FaceStrs[m_Face] << "\"" << R_ENDL;
    os << RTab(tc + 1) << "format=\"" << s_FormatStrs[m_Format] << "\"" << R_ENDL;
    os << RTab(tc + 1) << "width=\"" << m_Width
       << "\" height=\"" << m_Height << "\"" << R_ENDL;
    os << RTab(tc + 1) << "size=\"" << GetSize() << "\"" << R_ENDL;
    os << RTab(tc + 1) << "original_path=\"" << RGetUtf8FromShiftJis(REncodeXmlString(m_OriginalPath)) << "\"" << R_ENDL;
    os << RTab(tc + 1) << "stream_index=\"" << streamIndex << "\"" << R_ENDL;
    os << RTab(tc) << "/>" << R_ENDL;
}

//-----------------------------------------------------------------------------
//! @brief 元画像のデータ列をデータ列配列に追加します。
//-----------------------------------------------------------------------------
void ROriginalImage::AppendStream(RDataStreamArray& dataStreams) const
{
    //-----------------------------------------------------------------------------
    // データ列の列数を決定します。
    const int column =
        (m_Format == RGBA8  ) ? 16 :
        (m_Format == RGB8   ) ? 12 :
        (m_Format == RGBA32F) ?  4 :
        (m_Format == RGB32F ) ?  3 :
        16;

    //-----------------------------------------------------------------------------
    // データ列にコピーします。
    // 一時的なメモリー確保量を減らすため、
    // 先にデータ列配列に長さ 1 のデータ列を追加してからリサイズします。
    const size_t outSize = GetSize();
    if (!IsFloat())
    {
        const uint8_t dummyBytes[1] = { 0 };
        dataStreams.push_back(RDataStream(dummyBytes, sizeof(dummyBytes), column));

        RUCharArray& dstArray = dataStreams[dataStreams.size() - 1].m_ByteValues;
        dstArray.resize(outSize);
        uint8_t* pDataBuf = &dstArray[0];

        if (m_Format == RGBA8)
        {
            memcpy(pDataBuf, m_pImageData, outSize);
        }
        else // RGB8
        {
            const uint8_t* pSrc = m_pImageData;
            uint8_t* pDst = pDataBuf;
            const int pixCount = m_Width * m_Height;
            for (int iPix = 0; iPix < pixCount; ++iPix)
            {
                memcpy(pDst, pSrc, R_RGB_BYTES);
                pDst += R_RGB_COUNT;
                pSrc += R_RGBA_COUNT;
            }
        }
    }
    else
    {
        RFloatArray dummyFloats;
        dummyFloats.push_back(0.0f);
        dataStreams.push_back(RDataStream(dummyFloats, column));

        RFloatArray& dstArray = dataStreams[dataStreams.size() - 1].m_FloatValues;
        dstArray.resize(outSize / sizeof(float));
        float* pDataBuf = &dstArray[0];

        if (m_Format == RGBA32F)
        {
            memcpy(pDataBuf, m_pImageData, outSize);
        }
        else // RGB32F
        {
            const float* pSrcF32 = reinterpret_cast<const float*>(m_pImageData);
            float* pDstF32 = pDataBuf;
            const int pixCount = m_Width * m_Height;
            for (int iPix = 0; iPix < pixCount; ++iPix)
            {
                memcpy(pDstF32, pSrcF32, R_RGB_FLOAT_BYTES);
                pDstF32 += R_RGB_COUNT;
                pSrcF32 += R_RGBA_COUNT;
            }
        }
    }
}

//-----------------------------------------------------------------------------
//! @brief ftx ファイルのデータをデコードします。
//-----------------------------------------------------------------------------
RStatus RImage::DecodeFtx(
    REncoder* pEncoder,
    const uint8_t* fileBuf,
    const size_t fileSize,
    const bool isBinary,
    const int readFlags
)
{
    //-----------------------------------------------------------------------------
    // parse XML
    const size_t binOfs = (isBinary) ?
        RGetBinaryOffset(fileBuf, fileSize) : fileSize;
    //cerr << "binOfs: " << binOfs << endl;
    const std::string xmlStr(reinterpret_cast<const char*>(fileBuf), binOfs);

    RXMLElement xml;
    xml.LoadDocument(nullptr, xmlStr);

    //-----------------------------------------------------------------------------
    // get file info
    const RXMLElement* intermediateElem = xml.FindElement("nw4f_3dif");
    const int intermediateVer = RGetNw4f3difVerAsInt(intermediateElem->GetAttribute("version"));
    const RXMLElement* fileInfoElem = intermediateElem->FindElement("file_info", false);
    if (fileInfoElem != nullptr)
    {
        const RXMLElement* createElem = fileInfoElem->FindElement("create", false);
        if (createElem != nullptr)
        {
            m_CreateInfo = "\t" +
                std::string("<create tool_name=\"") + RGetShiftJisFromUtf8(createElem->GetAttribute("tool_name")) +
                "\" tool_version=\"" + createElem->GetAttribute("tool_version") +
                "\"" + R_ENDL;
            const std::string srcPath = createElem->GetAttribute("src_path", "", false); // 2.2.0 で追加
            m_CreateInfo += "\t\t" + std::string("src_path=\"") + RGetShiftJisFromUtf8(srcPath) +
                "\"" + R_ENDL;
            m_CreateInfo += "\t" + std::string("/>") + R_ENDL;
        }

        const RXMLElement* modifyElem = fileInfoElem->FindElement("modify", false);
        if (modifyElem != nullptr)
        {
            m_ModifyInfo = "\t" +
                std::string("<modify tool_name=\"") + RGetShiftJisFromUtf8(modifyElem->GetAttribute("tool_name")) +
                "\" tool_version=\"" + modifyElem->GetAttribute("tool_version") +
                "\" />" + R_ENDL;
        }
    }
    else
    {
        m_CreateInfo = FtxFileInfoNone;
    }

    //-----------------------------------------------------------------------------
    // get texture info
    const RXMLElement* textureElem = intermediateElem->FindElement("texture");
    const RXMLElement* infoElem = textureElem->FindElement("texture_info");
    m_Dimension = RGetTextureDimensionFromString(infoElem->GetAttribute("dimension"));
    m_Format = RGetTextureFormatFromString(infoElem->GetAttribute("quantize_type"));
    const FtxFormat ftxFormat = static_cast<FtxFormat>(m_Format);
    m_ImageW = atoi(infoElem->GetAttribute("width").c_str());
    m_ImageH = atoi(infoElem->GetAttribute("height").c_str());
    m_ImageD = atoi(infoElem->GetAttribute("depth").c_str());
    m_MipCount = atoi(infoElem->GetAttribute("mip_level").c_str());

    m_MipGenFilter = FtxOpt::GetMipGenFilterFromString(infoElem->GetAttribute("mip_gen_filter", "", false), true); // 3.0.0 で追加

    const std::string compSel = infoElem->GetAttribute("comp_sel", "", false); // 1.0.0 で追加
    m_CompSel          = (!compSel.empty()) ?
        RGetCompSelFromString(compSel) : RGetDefaultCompSel(ftxFormat);

    const std::string weightedCompress = infoElem->GetAttribute("weighted_compress", "", false); // 1.0.0 で追加
    m_WeightedCompress = (!weightedCompress.empty()) ?
        GetWeightedCompressFromString(weightedCompress) : FtxOpt::WEIGHTED_COMPRESS_DISABLE;

    m_DccPreset = RGetShiftJisFromUtf8(infoElem->GetAttribute("dcc_preset", "", false)); // 3.0.0 で追加

    const std::string defaultHint = RGetDefaultHint(ftxFormat);
    m_Hint = infoElem->GetAttribute("hint", defaultHint.c_str(), false); // 2.2.0 で追加

    const std::string linearStr = infoElem->GetAttribute("linear", "", false); // 2.2.0 で追加
    m_LinearFlag = (!linearStr.empty()) ? GetLinearFlagFromString(linearStr) : FtxOpt::LINEAR_NONE;

    const std::string initialSwizzle = infoElem->GetAttribute("initial_swizzle", "", false); // 2.2.0 で追加
    m_InitialSwizzle = (!initialSwizzle.empty()) ? atoi(initialSwizzle.c_str()) : 0;

    m_TileMode  = FtxTileMode_Linear;
    m_Swizzle   = 0;
    m_Alignment = 1;
    m_Pitch     = 0;

    ClearLevelOffsets();
    const std::string levelOffsetStr = infoElem->GetAttribute("level_offset", "", false); // 4.0.0 で追加
    if (!levelOffsetStr.empty())
    {
        std::istringstream levelOffsetIss(levelOffsetStr);
        for (int level = 0; level < m_MipCount; ++level)
        {
            levelOffsetIss >> m_LevelOffsets[level];
        }
    }
    else
    {
        std::istringstream mipOffsetIss(infoElem->GetAttribute("mip_offset")); // 4.0.0 で削除
        const int MipOffsetCount = 13;
        size_t mipOffsets[MipOffsetCount];
        memset(mipOffsets, 0x00, sizeof(mipOffsets));
        for (int level = 0; level < MipOffsetCount && mipOffsetIss; ++level)
        {
            mipOffsetIss >> mipOffsets[level];
        }
        if (m_MipCount >= 2)
        {
            m_LevelOffsets[1] = mipOffsets[0];
            for (int level = 1; level < m_MipCount - 1; ++level)
            {
                m_LevelOffsets[level + 1] = m_LevelOffsets[1] + mipOffsets[level];
            }
        }
    }
    m_OriginalImageHash = infoElem->GetAttribute("original_image_hash", "", false); // 4.0.0 で追加
    m_HasAlpha = (RGetComponentCount(ftxFormat) >= 4);
    m_IsFloat = RIsRealNumberTextureFormat(ftxFormat);
    const bool is1d = (m_Dimension == FtxDimension_1d || m_Dimension == FtxDimension_1dArray);

    //-----------------------------------------------------------------------------
    // エンコードされたピクセルデータを取得します。
    const bool getsOriginal = ((readFlags & ReadFlag_Original) != 0);
    const bool getsEncoded  = ((readFlags & ReadFlag_Encoded ) != 0);
    const RXMLElement* streamsElem = (isBinary) ? nullptr : textureElem->FindElement("stream_array");
    const RXMLElement* originalsElem = textureElem->FindElement("original_image_array", false);
    const bool decodesEncoded = (getsOriginal && originalsElem == nullptr);
    const void* binTop = (isBinary) ? fileBuf + binOfs : nullptr;
    if (getsEncoded || decodesEncoded)
    {
        std::istringstream(infoElem->GetAttribute("size")) >> m_EncodedDataSize;
        m_pEncodedData = new uint8_t[m_EncodedDataSize];
        const int iEncodedStream = atoi(infoElem->GetAttribute("stream_index").c_str());
        if (!isBinary)
        {
            const RXMLElement* streamElem = &(streamsElem->nodes[iEncodedStream]);
            RDecodeDataStringU8(m_pEncodedData, streamElem->text, m_EncodedDataSize);
        }
        else
        {
            const RBinDataStream* binStream = RBinDataStream::Get(binTop, iEncodedStream);
            memcpy(m_pEncodedData, binStream->m_Data, m_EncodedDataSize);
        }
    }

    //-----------------------------------------------------------------------------
    // 元画像のピクセルデータを取得します。
    m_IsDecodedOriginal = false;
    if (getsOriginal)
    {
        if (originalsElem != nullptr)
        {
            //-----------------------------------------------------------------------------
            // <original_image_array> から元画像のピクセルデータを取得します。
            const RXMLElement* firstOriginalElem = &(originalsElem->nodes[0]);
            const int originalW = atoi(firstOriginalElem->GetAttribute("width").c_str());
            const int originalH = atoi(firstOriginalElem->GetAttribute("height").c_str());
            if (getsEncoded)
            {
                // エンコードされたピクセルデータも取得する場合、
                // 幅と高さが元画像と同じかチェックします。
                // 2 次元画像から変換した 1 次元の ftx ファイルの場合、
                // texture_info の height は 1 ですが、
                // original_image の height は 1 でないので特別処理します。
                const int usedH = (is1d) ? 1 : originalH;
                if (originalW != m_ImageW || usedH != m_ImageH)
                {
                    return RStatus(RStatus::FAILURE, "Original image size is wrong: " + m_FilePath); // RShowError
                }
            }
            m_ImageW = originalW;
            m_ImageH = originalH;

            m_HasAlpha = false;
            const std::string firstOriginalFormat = firstOriginalElem->GetAttribute("format");
            m_IsFloat = (firstOriginalFormat == "rgba32f" || firstOriginalFormat == "rgb32f");
            const size_t pixelBytes = (m_IsFloat) ? R_RGBA_FLOAT_BYTES : R_RGBA_BYTES;
            const size_t faceBytes = pixelBytes * m_ImageW * m_ImageH;

            m_ImageDataSize = faceBytes * m_ImageD;
            m_pImageData = new uint8_t[m_ImageDataSize];
            m_OriginalPaths.resize(m_ImageD);
            for (int iDepth = 0; iDepth < m_ImageD; ++iDepth)
            {
                //-----------------------------------------------------------------------------
                // get stream
                const RXMLElement* originalElem = &(originalsElem->nodes[iDepth]);
                int iDst = atoi(originalElem->GetAttribute("slice_index").c_str());
                if (RIsCubMapDimension(static_cast<FtxDimension>(m_Dimension)))
                {
                    iDst *= RImage::CUBE_FACE_COUNT;
                    const std::string faceName = originalElem->GetAttribute("face");
                    if      (faceName == "negative_x")
                    {
                        iDst += 1;
                    }
                    else if (faceName == "positive_y")
                    {
                        iDst += 2;
                    }
                    else if (faceName == "negative_y")
                    {
                        iDst += 3;
                    }
                    else if (faceName == "positive_z")
                    {
                        iDst += 4;
                    }
                    else if (faceName == "negative_z")
                    {
                        iDst += 5;
                    }
                }
                uint8_t* pDst = m_pImageData + iDst * faceBytes;
                m_OriginalPaths[iDst] = RDecodeXmlString(
                    RGetShiftJisFromUtf8(originalElem->GetAttribute("original_path")));

                const std::string originalFormat = originalElem->GetAttribute("format");
                const size_t originalSize = atoi(originalElem->GetAttribute("size").c_str());
                uint8_t* pStreamData = new uint8_t[originalSize];
                const int iStream = atoi(originalElem->GetAttribute("stream_index").c_str());

                if (!isBinary)
                {
                    //-----------------------------------------------------------------------------
                    // parse ascii stream
                    const RXMLElement* streamElem = &(streamsElem->nodes[iStream]);
                    //const int count = atoi(streamElem->GetAttribute("count").c_str());
                    if (originalFormat == "rgba8" || originalFormat == "rgb8")
                    {
                        RDecodeDataStringU8(pStreamData, streamElem->text, originalSize);
                    }
                    else
                    {
                        RDecodeDataStringF32(pStreamData, streamElem->text, originalSize / sizeof(float));
                    }
                }
                else
                {
                    //-----------------------------------------------------------------------------
                    // get binary stream
                    const RBinDataStream* binStream = RBinDataStream::Get(binTop, iStream);
                    memcpy(pStreamData, binStream->m_Data, originalSize);
                }

                //-----------------------------------------------------------------------------
                // copy to stream to image
                if (originalFormat == "rgba8" || originalFormat == "rgba32f")
                {
                    memcpy(pDst, pStreamData, faceBytes);
                    m_HasAlpha = true;
                }
                else if (originalFormat == "rgb8")
                {
                    const uint8_t* pSrc = pStreamData;
                    for (size_t ofs = 0; ofs < faceBytes; ofs += R_RGBA_BYTES)
                    {
                        pDst[0] = pSrc[0];
                        pDst[1] = pSrc[1];
                        pDst[2] = pSrc[2];
                        pDst[3] = 0xff;
                        pSrc += R_RGB_COUNT;
                        pDst += R_RGBA_COUNT;
                    }
                }
                else if (originalFormat == "rgb32f")
                {
                    const float* pSrcF32 = reinterpret_cast<const float*>(pStreamData);
                    float* pDstF32 = reinterpret_cast<float*>(pDst);
                    for (size_t ofs = 0; ofs < faceBytes; ofs += R_RGBA_FLOAT_BYTES)
                    {
                        pDstF32[0] = pSrcF32[0];
                        pDstF32[1] = pSrcF32[1];
                        pDstF32[2] = pSrcF32[2];
                        pDstF32[3] = 1.0f;
                        pSrcF32 += R_RGB_COUNT;
                        pDstF32 += R_RGBA_COUNT;
                    }
                }
                delete[] pStreamData;
            }
        }
        else if (m_pEncodedData != nullptr)
        {
            //-----------------------------------------------------------------------------
            // エンコードされたピクセルデータをデコードして元画像のピクセルデータとします。
            DecodeEncodedDataToImageData(pEncoder, false);
            m_IsDecodedOriginal = true;

            // エンコードされたピクセルデータが不要なら解放します。
            if (!getsEncoded)
            {
                RFreeAndClearArray(m_pEncodedData);
                m_EncodedDataSize = 0;
            }
        }
    }

    //-----------------------------------------------------------------------------
    // 元画像を取得しない場合、エンコードされたピクセルデータをイメージデータとします。
    if (!getsOriginal && getsEncoded)
    {
        MoveEncodedDataToImageData();
    }

    //-----------------------------------------------------------------------------
    // get user data
    // 中間ファイル 1.0.0 以前のユーザーデータは取得しません。
    const RXMLElement* userDatasElem = textureElem->FindElement("user_data_array", false);
    if (intermediateVer >= 101 && userDatasElem != nullptr)
    {
        for (size_t iData = 0; iData < userDatasElem->nodes.size(); ++iData)
        {
            const RXMLElement* userDataElem = &(userDatasElem->nodes[iData]);
            m_UserDatas.push_back(RUserData(userDataElem, streamsElem, binTop));
        }
    }

    //-----------------------------------------------------------------------------
    // 編集用コメントを取得します。
    GetFtxComment(&m_CommentLabel, &m_CommentCol, &m_CommentText, textureElem);

    //-----------------------------------------------------------------------------
    // get tool data
    GetFtxToolData(m_ToolData, m_UserToolData, xmlStr);

    return RStatus::SUCCESS;
} // NOLINT(impl/function_size)

//-----------------------------------------------------------------------------
//! @brief ftx ファイルを解析してマージ情報を取得します。
//-----------------------------------------------------------------------------
RStatus RMergeInfo::ParseFtx(
    const uint8_t* fileBuf,
    const size_t fileSize,
    const bool isBinary
)
{
    //-----------------------------------------------------------------------------
    // parse XML
    const size_t binOfs = (isBinary) ?
        RGetBinaryOffset(fileBuf, fileSize) : fileSize;
    const std::string xmlStr(reinterpret_cast<const char*>(fileBuf), binOfs);

    RXMLElement xml;
    xml.LoadDocument(nullptr, xmlStr);

    //-----------------------------------------------------------------------------
    // get info
    RStatus status;
    while (status) // エラー時に break するための while で、ループではありません。
    {
        //-----------------------------------------------------------------------------
        // get parameter
        const RXMLElement* intermediateElem = xml.FindElement("nw4f_3dif");
        if (intermediateElem == nullptr)
        {
            break;
        }
        const int intermediateVer = RGetNw4f3difVerAsInt(intermediateElem->GetAttribute("version"));
        const RXMLElement* textureElem = intermediateElem->FindElement("texture");
        if (textureElem == nullptr)
        {
            break;
        }
        const RXMLElement* infoElem = textureElem->FindElement("texture_info");
        if (infoElem == nullptr)
        {
            break;
        }

        m_Dimension        = RGetTextureDimensionFromString(infoElem->GetAttribute("dimension"));
        m_Format           = RGetTextureFormatFromString(infoElem->GetAttribute("quantize_type"));
        m_MipCount         = atoi(infoElem->GetAttribute("mip_level").c_str());

        m_MipGenFilter = FtxOpt::GetMipGenFilterFromString(infoElem->GetAttribute("mip_gen_filter", "", false), true); // 3.0.0 で追加

        const std::string compSel = infoElem->GetAttribute("comp_sel", "", false); // 1.0.0 で追加
        m_CompSel          = (!compSel.empty()) ?
            RGetCompSelFromString(compSel) : RGetDefaultCompSel(static_cast<FtxFormat>(m_Format));

        const std::string weightedCompress = infoElem->GetAttribute("weighted_compress", "", false); // 1.0.0 で追加
        m_WeightedCompress = (!weightedCompress.empty()) ?
            GetWeightedCompressFromString(weightedCompress) : FtxOpt::WEIGHTED_COMPRESS_DISABLE;

        const std::string defaultHint = RGetDefaultHint(static_cast<FtxFormat>(m_Format));
        m_Hint = infoElem->GetAttribute("hint", defaultHint.c_str(), false); // 2.2.0 で追加

        const std::string linearStr = infoElem->GetAttribute("linear", "", false); // 2.2.0 で追加
        m_LinearFlag = (!linearStr.empty()) ? GetLinearFlagFromString(linearStr) : FtxOpt::LINEAR_NONE;

        const std::string initialSwizzle = infoElem->GetAttribute("initial_swizzle", "", false); // 2.2.0 で追加
        m_InitialSwizzle = (!initialSwizzle.empty()) ? atoi(initialSwizzle.c_str()) : 0;

        m_Width            = atoi(infoElem->GetAttribute("width").c_str());
        m_Height           = atoi(infoElem->GetAttribute("height").c_str());

        //-----------------------------------------------------------------------------
        // get user data
        // 中間ファイル 1.0.0 以前のユーザーデータは取得しません。
        const RXMLElement* streamsElem = (isBinary) ? nullptr : textureElem->FindElement("stream_array");
        const void* binTop = (isBinary) ? fileBuf + binOfs : nullptr;

        const RXMLElement* userDatasElem = textureElem->FindElement("user_data_array", false);
        if (intermediateVer >= 101 && userDatasElem != nullptr)
        {
            for (size_t iData = 0; iData < userDatasElem->nodes.size(); ++iData)
            {
                const RXMLElement* userDataElem = &(userDatasElem->nodes[iData]);
                m_UserDatas.push_back(RUserData(userDataElem, streamsElem, binTop));
            }
        }

        //-----------------------------------------------------------------------------
        // 編集用コメントを取得します。
        GetFtxComment(&m_CommentLabel, &m_CommentCol, &m_CommentText, textureElem);

        //-----------------------------------------------------------------------------
        // get tool data
        GetFtxToolData(m_ToolData, m_UserToolData, xmlStr);

        m_IsEnable  = true;
        break;
    }

    if (status && !m_IsEnable)
    {
        status = RStatus(RStatus::FAILURE, "Ftx file is wrong: " + m_FilePath); // RShowError
    }

    return status;
}

//-----------------------------------------------------------------------------
//! @brief ftx ファイルを出力します。
//-----------------------------------------------------------------------------
RStatus RImage::OutputFtxFile(
    std::string* pOutputData,
    ROriginalImageArray* pOriginalImages,
    const std::string& filePath,
    const FtxOpt& opt,
    const std::string& toolVersionStr,
    const bool freesMemory
)
{
    //-----------------------------------------------------------------------------
    // 必要ならファイルをオープンします。
    std::ofstream ofs;
    if (pOutputData == nullptr)
    {
        ofs.open(filePath.c_str(), std::ios::binary);
        if (!ofs)
        {
            return RStatus(RStatus::FAILURE, "Cannot open the file: " + filePath); // RShowError
        }
    }
    std::ostringstream oss;
    std::ostream* pOs = (pOutputData == nullptr) ? &ofs : reinterpret_cast<std::ostream*>(&oss);
    std::ostream& os = *pOs;

    //-----------------------------------------------------------------------------
    // XML 部分を出力し、データ列配列を作成します。
    const bool isBinary = (RGetExtensionFromFilePath(filePath) == FtxbExtension);
    const bool overwrites = (opt.m_InputPaths[0] == filePath);
    RDataStreamArray dataStreams;
    RStatus status = OutputFtxXml(os, &dataStreams, this, pOriginalImages,
        isBinary, opt, toolVersionStr, overwrites, freesMemory);

    //-----------------------------------------------------------------------------
    // バイナリー形式ならデータ列配列をバイナリー形式で出力します。
    if (status && isBinary)
    {
        ROutBinaryDataStreams(os, dataStreams, true);
    }
    dataStreams.clear();

    //-----------------------------------------------------------------------------
    // ファイルに出力しない場合はバッファにデータを格納します。
    if (pOutputData != nullptr)
    {
        *pOutputData = (status) ? oss.str() : "";
    }

    return status;
}

//=============================================================================
// texcvtr ネームスペースを終了します。
//=============================================================================
} // texcvtr
} // tool
} // gfx
} // nn

