﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <ShdrConverter.h>

#include <iostream>
#include <memory>
#include <regex>

#include <util/UtilFlag.h>
#include <util/UtilString.h>
#include <util/UtilXMLParser.h>
#include <util/UtilProcess.h>
#include <util/UtilBuffer.h>
#include <util/UtilFile.h>
#include <shdrlib.h>

#include <nn/util/util_StringUtil.h>
#include <nn/gfxTool/gfxTool_CommandLineParser.h>
#include "../../3dShaderConverter/Include/CommandLineParserWrapper.h"
#include <util/UtilCmdArgs.h>

#include <shlwapi.h>

#define OPTION_EQUALS(name, option) \
    (wcscmp((name), (option)) == 0)

#define OPTION_EQUALS_FULL(name, option) \
    (OPTION_EQUALS(name,option##_FULL))

#define OPTION_EQUALS_SHORT_FULL(name, option) \
    (OPTION_EQUALS(name,option) || OPTION_EQUALS(name, option##_FULL))



namespace nn { namespace g3dTool {


namespace {

const wchar_t* const s_NameRegexString = L"[0-9A-Za-z\\-\\._]+";
const wchar_t* const s_ZLibName = L"zlibwapi.dll";

} // anonymous namespace

bool Converter::Initialize(std::wstring path)
{
    try
    {
        char cfullpathBuf[_MAX_PATH];
        BOOL usedDefaultChar = false;

        if (path.empty())
        {
            THROW_ERROR(ERRCODE_SHADER_CONVERTER_INVALID_TEMP_DIRECTORY_PATH, "Invalid temp directory path.");
        }

        WideCharToMultiByte(CP_UTF8,
            0,
            path.c_str(),
            -1,
            cfullpathBuf,
            _MAX_PATH,
            NULL,
            0);

        // TODO: 変換できない文字列がないかエラーチェックをする。
        if (usedDefaultChar)
        {
            THROW_ERROR(ERRCODE_SHADER_CONVERTER_INVALID_TEMP_DIRECTORY_PATH, "Invalid temp directory path.");
        }

        // gfx tool の dll をロードします。
        if( !ShaderCompilerManager::InitializeDll() )
        {
            return false;
        }

        return Clear();
    }
    CATCH_ERROR_ALL(false)
}

bool Converter::Clear()
{
    try
    {
        m_File.reset(new ShdrFile());
        m_Ctx.reset(new Context());
        m_FscRootPathUtf.clear();
        m_FsdRootPathUtf.clear();
        m_ImOutStream.clear();
        std::ostringstream().swap(m_ImOutStream);
        m_ImSize = 0;
        m_Check = CHECK_SHADER_DEFAULT;
        m_Behavior = BEHAVIOR_DEFAULT;
        m_ConvertMode = ConvertMode_None;
        m_Macros.clear();
        tool::LogConfig::s_Level = tool::LogConfig::Verbose;
        m_SkipConvert = false;

        return true;
    }
    CATCH_ERROR_ALL(false)
}

bool Converter::Shutdown()
{
    try
    {
        m_File.reset();
        m_Ctx.reset();

        // gfx tool の dll をアンロードします。
        ShaderCompilerManager::FinalizeDll();

        return true;
    }
    CATCH_ERROR_ALL(false)
}

std::string ParseOutputName(const wchar_t* value)
{
    std::wstring fileName;
    std::wstring ext;
    wchar_t nameBuf[_MAX_FNAME];
    wchar_t extBuf[_MAX_EXT];
    // 拡張子抽出
    if (_wsplitpath_s(value, nullptr, 0, nullptr, 0, nameBuf, _MAX_FNAME, extBuf, _MAX_EXT) == 0)
    {
        ext.assign(extBuf);
    }

    fileName.assign(value);
    // 拡張子削除
    std::wstring::size_type pos = fileName.rfind(ext);
    if(pos != std::wstring::npos)
    {
        fileName.resize(pos);
    }

    // セパレータ変換
    std::replace(fileName.begin(), fileName.end(), L'\\', L'/');
    // ベース名取得
    fileName = nw::g3d::tool::util::GetBaseName(fileName, L'/');

    // 名前の命名規則に合っていなかった場合例外
    if (!std::regex_match(fileName, std::wregex(s_NameRegexString)))
    {
        THROW_ERROR(ERRCODE_INVALID_FILENAME, "Invalid output filename. %ls",fileName.c_str());
    }

    // wstring から string に変換する
    std::string cStr = nw::g3d::tool::util::ToAscii(&fileName[0]);

    return cStr;
}

std::string ConvertOptionPath(const wchar_t* pPath)
{
    wchar_t fullpath[_MAX_PATH];
    _wfullpath(fullpath, pPath, _MAX_PATH);
    std::wstring dumpFolder;
    dumpFolder.assign(fullpath);
    if (L'\\' != dumpFolder.back())
    {
        dumpFolder.push_back(L'\\');
    }
    if (!PathIsDirectory(dumpFolder.c_str()))
    {
        THROW_ERROR(ERRCODE_INVALID_DIRECTORY_NAME, "Directory is not exist (%ls)", dumpFolder.c_str());
    }
    std::replace(dumpFolder.begin(), dumpFolder.end(), L'\\', L'/');
    return nw::g3d::tool::util::ToUTF8(dumpFolder.c_str());
}

void Converter::SetCodeTypeOption(std::string value)
{
    if ("Binary" == value ||
        "Binary_Ir" == value ||
        "Binary_Source" == value)
    {
        m_File->SetShaderCodeTypeBinary(true);
    }
    if ("Source" == value ||
        "Binary_Source" == value ||
        "Ir_Source" == value)
    {
        m_File->SetShaderCodeTypeSource(true);
    }
    if ("Binary_Ir" == value ||
        "Ir" == value ||
        "Ir_Source" == value)
    {
        m_File->SetShaderCodeTypeIr(true);
    }

    // Ir, Ir_Sourceの場合、シェーダーリフレクションの取得先をIrに設定します。
    if ("Ir" == value ||
        "Ir_Source" == value)
    {
        m_File->SetShaderReflectionCodeType(ShaderCompilerManager::ShaderCodeType_Intermediate);
    }
    else
    {
        m_File->SetShaderReflectionCodeType(ShaderCompilerManager::ShaderCodeType_Binary);
    }
}

// Parser で未定義オプションを無視するコールバック。--gfx-shader-converter-options の解析に使用。
bool IgnoreUndefinedOptionCallback(int argc, const char* const* argv, int argIndex, void* pCallbackParam)
{
    NN_UNUSED(argc);
    NN_UNUSED(argv);
    NN_UNUSED(argIndex);
    NN_UNUSED(pCallbackParam);
    return true;
};

bool Converter::SetOptions(const wchar_t* options[])
{
    try
    {
        // ログのサイレント設定を先に行っておく
        {
            int index = 0;
            while (options[index])
            {
                const wchar_t* name = options[index++];
                const wchar_t* value = options[index++];
                NN_UNUSED(value);
                if (OPTION_EQUALS_SHORT_FULL(name, OPT_SILENT))
                {
                    tool::LogConfig::s_Level = tool::LogConfig::Silent;
                    std::string gfxToolOption("--silent");
                    m_File->AddGfxToolConverterOptions(gfxToolOption);
                }
            }
        }

        // マージオプションの検出を先に行っておく。SetOption(arg) への移行で先に検出する処理は不要になるはず。
        {
            int index = 0;
            while (options[index])
            {
                const wchar_t* name = options[index++];
                const wchar_t* value = options[index++];

                if (OPTION_EQUALS(name, OPT_MERGE_SHADER_ARCHIVE_FILE_FULL))
                {
                    nw::g3d::tool::CmdInput shaderArchiveInput(value);
                    if (0 != wcscmp(L".bfsha", shaderArchiveInput.ext.c_str()))
                    {
                        THROW_ERROR(ERRCODE_OPEN_FILE, "Invalid file extension.: %s", shaderArchiveInput.fullpath.c_str());
                    }
                    nw::g3d::tool::util::Buffer file;
                    bool success = file.LoadFile(shaderArchiveInput.fullpath.c_str());
                    if (!success)
                    {
                        THROW_ERROR(ERRCODE_OPEN_FILE, "Load file failed. File: %s", shaderArchiveInput.fullpath.c_str());
                    }

                    nn::g3d::ResShaderFileData* pSrcData = static_cast<nn::g3d::ResShaderFileData*>(file.Get());
                    size_t fileSize = pSrcData->fileHeader.GetFileSize();
                    void* pDstData = malloc(fileSize);
                    memcpy(pDstData, pSrcData, fileSize);
                    std::shared_ptr<nn::g3d::ResShaderFile> pResShaderFile(static_cast<nn::g3d::ResShaderFile*>(pDstData));
                    if (!pResShaderFile->GetFileHeader()->IsRelocated())
                    {
                        pResShaderFile->ToData().fileHeader.GetRelocationTable()->Relocate();
                    }

                    // どこまでバージョンをチェックすればよいか不明。とりあえず変なコンバート結果にならないように全一致をチェックしておく。
                    if (pResShaderFile->GetFileHeader()->version.GetPacked() !=
                        NN_UTIL_CREATE_BINVERSION(NN_G3D_SHADER_BINARY_VERSION_MAJOR, NN_G3D_SHADER_BINARY_VERSION_MINOR, NN_G3D_SHADER_BINARY_VERSION_MICRO))
                    {
                        THROW_ERROR(ERRCODE_SHADER_CONVERTER_VERSION_MISMATCH, "ShaderArchiveFile version mismatch.\nPlease re-convert fsd files to bfsha files.: %s", shaderArchiveInput.fullpath.c_str());
                    }
                    m_File->Add(pResShaderFile);
                    m_File->SetName(pResShaderFile->GetFileHeader()->GetFileName().data());
                    if (m_ConvertMode == ConvertMode_FscToFsd || ConvertMode_MergeBfsha == ConvertMode_FsdToBfsha)
                    {
                        THROW_ERROR(ERRCODE_INVALID_OPTION_SET, "Cannot use --merge-shader-archive-file with fsd or fsd files input.");
                    }
                    m_ConvertMode = ConvertMode_MergeBfsha;
                    m_File->SetConvertMode(m_ConvertMode);
                }
            }
        }

        // bfsha のマージ処理に、不要なオプションが指定されていないかをチェックする
        // fsc->fsd, fsd->bfsha の変換もオプションチェックしたいが、チェックするとユーザーの対応が大変なので未対応。
        if (m_ConvertMode == ConvertMode_MergeBfsha)
        {
            const wchar_t* validOptionArray[] =
            {
                OPT_OUTPUT_FILE_NAME,
                OPT_OUTPUT_FILE_NAME_FULL,
                OPT_VERSION,
                OPT_VERSION_FULL,
                OPT_HELP,
                OPT_HELP_FULL,
                OPT_SILENT,
                OPT_SILENT_FULL,
                OPT_GFX_SHADER_CONVERTER_OPTIONS_FULL,
                OPT_MERGE_SHADER_ARCHIVE_FILE_FULL
            };

            int index = 0;
            while (options[index])
            {
                const wchar_t* name = options[index++];
                const wchar_t* value = options[index++];
                NN_UNUSED(value);
                bool isValidOption = false;
                for (int validOptionIndex = 0; validOptionIndex < sizeof(validOptionArray) / sizeof(validOptionArray[0]); ++validOptionIndex)
                {
                    if (OPTION_EQUALS(name, validOptionArray[validOptionIndex]))
                    {
                        isValidOption = true;
                        break;
                    }
                }
                if (!isValidOption)
                {
                    THROW_ERROR(ERRCODE_INVALID_OPTION_SET, "Cannot use --merge-shader-archive-file with %s.", name);
                }
            }
            // gfx へのダミーオプションを追加する
            if (m_ConvertMode == ConvertMode_MergeBfsha)
            {
                m_File->AddGfxToolConverterOptions("-a=Gl");
                m_File->AddGfxToolConverterOptions("--code-type=Binary");
            }
        }

        int index = 0;
        int cacheDirectoryOptionFlag = 0;
        while (options[index])
        {
            const wchar_t* name = options[index++];
            const wchar_t* value = options[index++];

            if (OPTION_EQUALS_SHORT_FULL(name, OPT_OUTPUT_FILE_NAME))
            {
                std::wstring extension;
                wchar_t extensionBuff[_MAX_EXT];
                if (_wsplitpath_s( value, nullptr, 0, nullptr, 0, nullptr, 0, extensionBuff, _MAX_EXT) == 0)
                {
                    extension.assign(extensionBuff);
                }
                if( extension == OUTPUT_DEFINITION_BINARY_FILE_EXT )
                {
                    std::string cStr = ParseOutputName(value);
                    m_File->SetName(cStr);
                    PRINT_LOG("fsd file name: %hs", cStr.c_str());
                }
                else if( extension == OUTPUT_BINARY_FILE_EXT )
                {
                    std::string cStr = ParseOutputName(value);
                    PRINT_LOG("bfsha name: %hs", cStr.c_str());
                }
                else
                {
                    THROW_ERROR(ERRCODE_INVALID_FILENAME, "Unknown extension name: %ls(%ls)", name, value);
                }
            }
            else if (OPTION_EQUALS_FULL(name, OPT_ROOT_CONFIG_PATH))
            {
                wchar_t fullpath[_MAX_PATH];
                _wfullpath(fullpath, value, _MAX_PATH);
                std::wstring crootPath;
                crootPath.assign(fullpath);
                if (L'\\' != crootPath.back())
                {
                    // フォルダ名が部分一致しないように確実に終端する。
                    crootPath.push_back(L'\\');
                }

                std::replace(crootPath.begin(), crootPath.end(), L'\\', L'/');
                m_FscRootPathUtf = nw::g3d::tool::util::ToUTF8(crootPath.c_str());

                PRINT_LOG("fsc root path: %hs", m_FscRootPathUtf.c_str());
            }
            else if (OPTION_EQUALS_FULL(name, OPT_ROOT_DEFINITION_PATH))
            {
                wchar_t fullpath[_MAX_PATH];
                _wfullpath(fullpath, value, _MAX_PATH);
                std::wstring drootPath;
                drootPath.assign(fullpath);
                if (L'\\' != drootPath.back())
                {
                    // フォルダ名が部分一致しないように確実に終端する。
                    drootPath.push_back(L'\\');
                }

                std::replace(drootPath.begin(), drootPath.end(), L'\\', L'/');
                m_FsdRootPathUtf = nw::g3d::tool::util::ToUTF8(drootPath.c_str());

                PRINT_LOG("fsd root path: %hs", m_FsdRootPathUtf.c_str());
            }
            else if (OPTION_EQUALS_FULL(name, OPT_UNIFIED_ANNOTATION))
            {
                m_Behavior = nw::g3d::tool::util::EnableFlag(m_Behavior, BEHAVIOR_UNIFIED_ANNOTATION);
            }
            else if (OPTION_EQUALS_FULL(name, OPT_AUTO_EXTRACT))
            {
                m_Behavior = nw::g3d::tool::util::EnableFlag(m_Behavior, BEHAVIOR_AUTO_EXTRACT);
            }
            else if (OPTION_EQUALS_FULL(name, OPT_DEFINE_MACRO))
            {
                std::string macro = nw::g3d::tool::util::ToUTF8(value);
                m_Macros.push_back(macro);
            }
            else if (OPTION_EQUALS_FULL(name, OPT_FORCE_VARIATION))
            {
                m_File->SetForceVariation(true);
            }
            else if (OPTION_EQUALS_FULL(name, OPT_DUMP_SHADER_SOURCE) ||
                OPTION_EQUALS(name, OPT_DUMP_STATISTICS_INFORMATION_FULL))
            {
                if (OPTION_EQUALS_FULL(name, OPT_DUMP_SHADER_SOURCE))
                {
                    m_File->SetDumpAll(true);
                }
                std::string path = ConvertOptionPath(value);
                m_File->SetDumpFolder(path);
                PRINT_LOG("source dump path: %hs", path.c_str());

                std::string gfxOptDumpDirectory( "--dump-directory=" );
                gfxOptDumpDirectory += path;
                m_File->AddGfxToolConverterOptions( gfxOptDumpDirectory );
            }
            else if (OPTION_EQUALS_FULL(name, OPT_GLSL_VERSION))
            {
                m_File->SetGlslVersion(nw::g3d::tool::util::ToUTF8(value));

                std::string gfxToolOption( "--glsl-version=" );
                gfxToolOption += nw::g3d::tool::util::ToUTF8(value);
                m_File->AddGfxToolConverterOptions( gfxToolOption );
            }
            else if ( OPTION_EQUALS( name, OPT_CODE_TYPE_FULL ) )
            {
                SetCodeTypeOption(nw::g3d::tool::util::ToUTF8(value));

                std::string gfxToolOption("-c=");
                gfxToolOption += nw::g3d::tool::util::ToUTF8(value);
                m_File->AddGfxToolConverterOptions( gfxToolOption );
            }
            else if ( OPTION_EQUALS( name, OPT_API_TYPE_FULL) )
            {
                // --api-type を追加
                std::string gfxToolOption( "-a=" );
                gfxToolOption += nw::g3d::tool::util::ToUTF8(value);
                m_File->AddGfxToolConverterOptions( gfxToolOption );
                // オプションに NVN が指定された場合 BinaryAvailable フラグを立てる
                if(nn::util::Strnicmp<wchar_t>(value, L"nvn", 3) == 0)
                {
                    m_File->SetBinaryAvailable(true);
                }
            }
            else if( OPTION_EQUALS( name, OPT_PREPROCESS_FULL ) )
            {
                // --preprocess を追加
                std::string gfxToolOption( "--preprocess" );
                m_File->AddGfxToolConverterOptions( gfxToolOption );
            }
            else if( OPTION_EQUALS( name, OPT_DEBUG_INFO_LEVEL_FULL ) )
            {
                std::string gfxToolOpt( "--debug-info-level=" );
                gfxToolOpt += nw::g3d::tool::util::ToUTF8( value );
                m_File->AddGfxToolConverterOptions( gfxToolOpt );
            }
            else if ( OPTION_EQUALS(name, OPT_DEBUG_INFO_DIR_FULL) )
            {
                std::string path = ConvertOptionPath(value);
                std::string gfxToolOpt("--debug-info-directory=");
                gfxToolOpt += path;
                m_File->AddGfxToolConverterOptions(gfxToolOpt);
                PRINT_LOG("Graphics debugger debug info path: %hs", path.c_str());
            }
            else if((OPTION_EQUALS(name, OPT_SHADER_CACHE_DIR_FULL)) ||
                (OPTION_EQUALS(name, OPT_SHADER_CACHE_WRITE_DIR_FULL)) ||
                (OPTION_EQUALS(name, OPT_SHADER_CACHE_READ_DIR_FULL)))
            {
                std::string path = ConvertOptionPath(value);
                if (OPTION_EQUALS(name, OPT_SHADER_CACHE_DIR_FULL))
                {
                    cacheDirectoryOptionFlag |= CacheDirectoryOptionFlag_Both;
                    m_File->SetShaderCacheAccessFlag(ShaderCompilerManager::ShaderCacheAccessType_Write);
                    m_File->SetShaderCacheAccessFlag(ShaderCompilerManager::ShaderCacheAccessType_Read);
                    m_File->SetShaderCacheWriteDirectory(path);
                    m_File->SetShaderCacheReadDirectory(path);
                    PRINT_LOG("Shader binary caching directory path: %hs", path.c_str());
                }
                else
                {
                    cacheDirectoryOptionFlag |= CacheDirectoryOptionFlag_Separate;
                    if (OPTION_EQUALS(name, OPT_SHADER_CACHE_WRITE_DIR_FULL))
                    {
                        m_File->SetShaderCacheAccessFlag(ShaderCompilerManager::ShaderCacheAccessType_Write);
                        m_File->SetShaderCacheWriteDirectory(path);
                        PRINT_LOG("Shader binary caching write directory path: %hs", path.c_str());
                    }
                    else if (OPTION_EQUALS(name, OPT_SHADER_CACHE_READ_DIR_FULL))
                    {
                        m_File->SetShaderCacheAccessFlag(ShaderCompilerManager::ShaderCacheAccessType_Read);
                        m_File->AddShaderCacheReadDirectory(path);
                        PRINT_LOG("Shader binary caching read directory path: %hs", path.c_str());
                    }
                }
                std::string gfxToolOpt("--shader-cache-directory=");
                gfxToolOpt += path;
                m_File->AddGfxToolConverterOptions(gfxToolOpt);
            }
            else if( OPTION_EQUALS( name, OPT_DECOMPOSE_BINARY_FULL ) )
            {
                std::string gfxToolOpt( "--decompose-binary" );
                m_File->AddGfxToolConverterOptions( gfxToolOpt );
            }
            else if( OPTION_EQUALS( name, OPT_GLSL_EXTENSION_FULL ) )
            {
                std::string gfxToolOpt( "--glsl-extension=" );
                gfxToolOpt += nw::g3d::tool::util::ToUTF8(value);
                m_File->AddGfxToolConverterOptions( gfxToolOpt );
            }
            else if( OPTION_EQUALS( name, OPT_JOBS_FULL ) )
            {
                std::string gfxToolOpt( "--jobs=" );
                gfxToolOpt += nw::g3d::tool::util::ToUTF8( value );
                m_File->AddGfxToolConverterOptions( gfxToolOpt );
            }
            else if( OPTION_EQUALS( name, OPT_USE_PREPRCESSOR_VARIATION ) )
            {
                m_File->SetUsePreprocessorVariation( true );
            }
            else if (OPTION_EQUALS( name, OPT_GLSLC_OPTION_FLAGS_FULL ) )
            {
                std::string gfxToolOpt( "--glslc-option-flags=" );
                gfxToolOpt += nw::g3d::tool::util::ToUTF8( value );
                m_File->AddGfxToolConverterOptions( gfxToolOpt );
            }
            else if (OPTION_EQUALS(name, OPT_GFX_SHADER_CONVERTER_OPTIONS_FULL))
            {
                std::string gfxToolOpt = nw::g3d::tool::util::ToUTF8(value);
                // ' を " に置き換える
                size_t position = gfxToolOpt.find("'");
                while (position != std::string::npos)
                {
                    gfxToolOpt.replace(position, 1, "\"");
                    position = gfxToolOpt.find("'", position + 1);
                }
                nn::gfxTool::CommandLineParser::CommandLineArg args( nn::util::string_view(gfxToolOpt.c_str(), gfxToolOpt.size()) );

                // 3dShaderConverter レイヤーでも扱うオプション、もしくは共存できないオプションは g3d 側でチェックする。
                // 該当オプションだけ SetOption() し、それ以外のオプションを無視して Parse する。
                nn::gfxTool::CommandLineParser parser;
                nn::gfxTool::Custom< std::vector< nn::gfxTool::Custom< std::string >::Type > >::Type inputArgArray;
                nn::gfxTool::CommandLineOption< nn::gfxTool::Custom< std::string >::Type > glslVersion;
                nn::gfxTool::CommandLineOption< nn::gfxTool::Custom< std::string >::Type > codeType;
                nn::gfxTool::CommandLineOption< nn::gfxTool::Custom< std::string >::Type > dumpDirectory;
                nn::gfxTool::CommandLineOption< nn::gfxTool::Custom< std::string >::Type > shaderCacheDirectory;
                nn::gfxTool::CommandLineOption< nn::gfxTool::Custom< std::string >::Type > mergeShaderFile;
                parser.SetArgs(&inputArgArray);
                parser.SetOption(glslVersion.Define(false, "glsl-version", ""));
                parser.SetOption(codeType.Define(false, 'c', "code-type", ""));
                parser.SetOption(dumpDirectory.Define(false, "dump-directory", ""));
                parser.SetOption(shaderCacheDirectory.Define(false, "shader-cache-directory", ""));
                parser.SetOption(mergeShaderFile.Define(false, "merge-shader-file", ""));
                parser.SetUndefinedOptionCallback(IgnoreUndefinedOptionCallback, NULL);
                parser.Parse(args.GetArgC(), args.GetArgv());

                if (glslVersion.IsExisting())
                {
                    m_File->SetGlslVersion(glslVersion.GetValue());
                }
                if (codeType.IsExisting())
                {
                    SetCodeTypeOption(codeType.GetValue());
                }
                if (dumpDirectory.IsExisting())
                {
                    std::string path = ConvertOptionPath(nw::g3d::tool::util::ToWstr(dumpDirectory.GetValue()).c_str());
                    m_File->SetDumpFolder(path);
                    PRINT_LOG("source dump path: %hs", path.c_str());
                }
                if (shaderCacheDirectory.IsExisting())
                {
                    std::string path = ConvertOptionPath(nw::g3d::tool::util::ToWstr(shaderCacheDirectory.GetValue()).c_str());
                    cacheDirectoryOptionFlag |= CacheDirectoryOptionFlag_Both;
                    m_File->SetShaderCacheAccessFlag(ShaderCompilerManager::ShaderCacheAccessType_Write);
                    m_File->SetShaderCacheAccessFlag(ShaderCompilerManager::ShaderCacheAccessType_Read);
                    m_File->SetShaderCacheWriteDirectory(path);
                    m_File->SetShaderCacheReadDirectory(path);
                    PRINT_LOG("Shader binary caching directory path: %hs", path.c_str());
                }
                if (mergeShaderFile.IsExisting())
                {
                    THROW_ERROR(ERRCODE_INVALID_OPTION_SET, "--merge-shader-file is unusable from 3dShaderConverter. Please use --merge-shader-archive-file.");
                }

                for(int argvIndex = 0; argvIndex < args.GetArgC(); argvIndex++)
                {
                    m_File->AddGfxToolConverterOptions(args.GetArgv()[argvIndex]);
                }
            }
            else if( OPTION_EQUALS( name, OPT_VARIATION_BEGIN_RATIO_FULL ) )
            {
                std::string variationBeginRatioStr = nw::g3d::tool::util::ToUTF8( value );
                float variationBeginRatio = nn::gfxTool::LexicalCastAuto( variationBeginRatioStr );
                m_File->SetVariationBeginRatio( variationBeginRatio );
            }
            else if( OPTION_EQUALS( name, OPT_VARIATION_END_RATIO_FULL ) )
            {
                std::string variationEndRatioStr = nw::g3d::tool::util::ToUTF8( value );
                float variationEndRatio = nn::gfxTool::LexicalCastAuto( variationEndRatioStr );
                m_File->SetVariationEndRatio( variationEndRatio );
            }
            else if( OPTION_EQUALS( name, OPT_SKIP_CONVERT_FULL ) )
            {
                m_SkipConvert = true;
                m_File->SetSkipConvert( true );
            }
            else if (
                OPTION_EQUALS_SHORT_FULL(name, OPT_SILENT) ||
                OPTION_EQUALS_FULL(name, OPT_VERIFY) ||
                OPTION_EQUALS_SHORT_FULL(name, OPT_VERSION) ||
                OPTION_EQUALS_FULL(name, OPT_OUTPUT_FILE_NAME) ||
                OPTION_EQUALS(name, OPT_MERGE_SHADER_ARCHIVE_FILE_FULL))
            {
                // do nothing.
            }
            else
            {
                THROW_ERROR(ERRCODE_UNKNOWN_OPTION, "Unknown global option: %ls(%ls)", name, value);
            }
        }

        // --shader-cache-directory は --shader-cache-write/read-directory を同時指定できない
        if( (cacheDirectoryOptionFlag & CacheDirectoryOptionFlag_Both) && (cacheDirectoryOptionFlag & CacheDirectoryOptionFlag_Separate) )
        {
            THROW_ERROR(ERRCODE_INVALID_OPTION_SET, "Cannot use --shader-cache-directory option with --shader-cache-write/read-directory option");
        }

        m_File->SetMacros(m_Macros);

        // 以下の変更により Concurrency のスケジューラ作成が条件次第で落ちるようになるため注意
        // 32 ビット版の精度を 64 ビット版相当にします。
        // 32 ビット版でも SIMD の精度にする。
#ifndef _WIN64
        unsigned int control_word;
        int err = _controlfp_s(&control_word, _PC_24, MCW_PC);
        if (err != 0)
        {
            return FALSE;
        }
#endif

        return true;
    }
    CATCH_ERROR_ALL(false)
}

bool Converter::SetOptions(int argc, const wchar_t* argv[])
{
    try
    {
        // 引数の解析
        nn::g3dTool::CommandLineParserWrapper	parserWrapper( argc, argv );
        std::shared_ptr< nw::g3d::tool::CmdArgs > pCmdArgs = parserWrapper.GetCmdArgs();

        nw::g3d::tool::CmdArgs& args = *pCmdArgs;

        // CmdArgs の生成
        {
            // 出力パスの取得。
            int outputIndex		= nw::g3d::tool::CmdOption::FindIndex(args.options, OPT_OUTPUT_FILE_NAME_FULL);

            if( outputIndex < 0 )
            {
                if (args.inputs.size() == 1)
                {
                    // 入力ファイルが1つの場合に限り出力ファイルの指定を省略できる。
                    const nw::g3d::tool::CmdInput& input = args.inputs.front();
                    auto lenExt = input.ext.size();
                    const std::wstring& fullpath = input.fullpath;
                    wchar_t fullpathBuf[_MAX_PATH];

                    // 入力が fsc の時は fsd を出力する
                    if (L".fsdb" != input.ext && L".fsda" != input.ext)
                    {
                        std::wstring doutpath; // 定義出力パス。
                        doutpath.assign(fullpath.substr(0, fullpath.size() - lenExt));
                        doutpath	+= OUTPUT_DEFINITION_BINARY_FILE_EXT;
                        if(_wfullpath(fullpathBuf, doutpath.c_str(), _MAX_PATH))
                        {
                            doutpath.assign(fullpathBuf);
                        }
                        else
                        {
                            PRINT_ERROR_LOG("Output definition path is too long.");
                            return EXIT_FAILURE;
                        }
                        std::wstringstream dss;
                        dss << OPT_OUTPUT_FILE_NAME_FULL << L"=";
                        dss << doutpath;
                        args.options.push_back(nw::g3d::tool::CmdOption(dss.str().c_str()));
                    }
                    else
                    {
                        // 入力が fsd の時の処理
                        std::wstring boutpath; // バイナリ出力パス
                        boutpath.assign(fullpath.substr(0, fullpath.size() - lenExt));
                        boutpath	+= OUTPUT_BINARY_FILE_EXT;
                        if(_wfullpath(fullpathBuf, boutpath.c_str(), _MAX_PATH))
                        {
                            boutpath.assign(fullpathBuf);
                        }
                        else
                        {
                            PRINT_ERROR_LOG("Output definition path is too long.");
                            return EXIT_FAILURE;
                        }
                        std::wstringstream bss;
                        bss << OPT_OUTPUT_FILE_NAME_FULL << L"=";
                        bss << boutpath;
                        args.options.push_back(nw::g3d::tool::CmdOption(bss.str().c_str()));
                    }
                }
                else
                {
                    PRINT_ERROR_LOG("Not found output path.");
                    return EXIT_FAILURE;
                }
            }
        }

        // Option の登録
        {
            auto conv = [](const std::vector<nw::g3d::tool::CmdOption>& options) -> const wchar_t**
            {
                auto count = options.size();
                const wchar_t** optionArray = new const wchar_t*[count * 3 + 1];
                int index = 0;
                for (auto iter = options.cbegin(); iter != options.cend(); ++iter)
                {
                    optionArray[index++] = iter->name.c_str();
                    optionArray[index++] = iter->value.c_str();
                }
                optionArray[index] = nullptr;
                return optionArray;
            };

            std::unique_ptr<const wchar_t*[]> options(conv(args.options));
            // オプションの登録。
            if ( SetOptions(options.get()) )
            {
                PRINT_ERROR_LOG("SetOptions() failed.");
                return EXIT_FAILURE;
            }
        }
        return true;
    }
    CATCH_ERROR_ALL(false)
}

// 各中間ファイルバージョンをチェックする
template<typename T>
void Converter::CheckVersion(T pElem, int major, int minor, int micro, const char* elementName)
{
    if (nw::g3d::tool::util::CheckFlag(m_Check, CHECK_VERSION))
    {
        if (pElem->version.value.major != major ||
            pElem->version.value.minor != minor ||
            pElem->version.value.micro != micro)
        {
            THROW_ERROR(
                ERRCODE_XML_VERSION_MISMATCH,
                "Xml version mismatch.\n%hs supports <%hs> version: %d.%d.%d.\nBut input <%hs> version: %d.%d.%d.\nPlease re-convert fsc files to fsd or bfsha files.",
                nw::g3d::tool::util::GetModuleName(nullptr).c_str(),
                elementName,
                major,
                minor,
                micro,
                elementName,
                pElem->version.value.major,
                pElem->version.value.minor,
                pElem->version.value.micro);
        }
    }
}

bool Converter::AddFile(void* pData, size_t dataSize, const wchar_t* path[3])
{
    // 今は AddFile は fsc と fsd にしか対応していないため、merge オプションと fsd, fsc の入力が同時に行われていないかをここでチェックする。
    if (m_ConvertMode == ConvertMode_MergeBfsha)
    {
        THROW_ERROR(ERRCODE_INVALID_OPTION_SET, "Cannot use --merge-shader-archive-file with fsd or fsd files input.");
    }
    PRINT_LOG("%ls", path[0]);

    enum
    {
        PATH_FULLPATH,
        PATH_FNAME,
        PATH_EXT
    };
    const wchar_t* fname = path[PATH_FNAME];
    const wchar_t* ext = path[PATH_EXT];

    if (!std::regex_match(fname, std::wregex(s_NameRegexString)))
    {
        THROW_ERROR(ERRCODE_INVALID_FILENAME, "Invalid input filename. %ls",fname);
    }

    // ファイル名を char に変換する。
    std::string fnameAscii = nw::g3d::tool::util::ToAscii(&fname[0]);

    std::wstring wpath(path[PATH_FULLPATH]);
    // ワイド文字でセパレータを置換してから utf-8 に変換
    std::replace(wpath.begin(), wpath.end(), L'\\', L'/');
    std::string fullpathUtf = nw::g3d::tool::util::ToUTF8(wpath.c_str());

    try
    {
        nw::g3d::tool::util::XMLParser parser;
        // バイナリ中間ファイルのために NULL 終端を含んだサイズを計算する。
        size_t textSize = strnlen_s(static_cast<const char*>(pData), dataSize) + sizeof(char);
        bool success = parser.ParseTree(static_cast<char*>(pData), textSize);
        if (!success)
        {
            THROW_ERROR(ERRCODE_XML_CONVERT_PARSE_FAILURE, "XML parse failure.");
        }
        const nw::g3d::tool::util::XMLElement* pRoot = parser.Root();

        std::wstring imExt(ext);
        // データサイズ＋１とテキスト部分のサイズが同一の場合、テキスト中間ファイルとして扱う。
        if ((dataSize + 1) == textSize)
        {
            imExt.pop_back();
            imExt.push_back(L'a');
        }
        std::transform(imExt.begin(), imExt.end(), imExt.begin(), ::tolower);

        std::shared_ptr<nw::g3d::tool::g3dif::elem_nw4f_3dif> nw4f_3dif(new nw::g3d::tool::g3dif::elem_nw4f_3dif());
        *nw4f_3dif << pRoot;

        // 全中間ファイル拡張子で統一されたコンテナバージョンをチェックします。
        CheckVersion(nw4f_3dif, NN_G3D_INTERMEDIATE_VERSION_MAJOR, NN_G3D_INTERMEDIATE_VERSION_MINOR, NN_G3D_INTERMEDIATE_VERSION_MICRO, "nw4f_3dif");

        // シェーダコンフィグ
        if (0 == wcscmp(L".fsca", imExt.c_str()) || 0 == wcscmp(L".fscb", imExt.c_str()))
        {
            std::string trimString = nw::g3d::tool::util::TrimString(fullpathUtf, m_FscRootPathUtf);

            std::shared_ptr<nw::g3d::tool::g3dif::elem_shader_config> elem =
                nw::g3d::tool::util::AddElem<nw::g3d::tool::g3dif::elem_shader_config>(pRoot, m_Check);
            CheckVersion(elem, NN_G3D_INTERMEDIATE_FSC_VERSION_MAJOR, NN_G3D_INTERMEDIATE_FSC_VERSION_MINOR, NN_G3D_INTERMEDIATE_FSC_VERSION_MICRO, "shader_config");
            nw::g3d::tool::util::AddPath(elem, fnameAscii.c_str(), fullpathUtf.c_str());
            if (!m_FscRootPathUtf.empty())
            {
                // root_path がある場合は root からの相対パス
                elem->SetRelativePath(trimString.c_str());
            }
            else
            {
                // root_path がない場合はファイル名
                std::stringstream ss;
                ss << fnameAscii << nw::g3d::tool::util::ToAscii(ext);
                elem->SetRelativePath(ss.str().c_str());
            }
            m_File->Add(elem);
            m_ConvertMode = ConvertMode_FscToFsd;
        }
#if 0 // fsda のコンバートはできません。必要になったらここを有効にして、各種対応をします。
        // シェーダ定義
        else if (0 == wcscmp(L".fsda", imExt.c_str()))
        {
            // root_path が設定されていない場合は path を入れない。
            std::string trimString = nw::g3d::tool::util::TrimString(fullpathUtf, m_FsdRootPathUtf);
            if (m_FsdRootPathUtf.empty())
            {
                trimString = "";
            }

            std::shared_ptr<nw::g3d::tool::g3dif::elem_shader_definition> elem =
                nw::g3d::tool::util::AddElem<nw::g3d::tool::g3dif::elem_shader_definition>(pRoot, m_Check);
            CheckVersion(elem, NN_G3D_INTERMEDIATE_FSD_VERSION_MAJOR, NN_G3D_INTERMEDIATE_FSD_VERSION_MINOR, NN_G3D_INTERMEDIATE_FSD_VERSION_MICRO, "shader_definition");
            nw::g3d::tool::util::AddPath(elem, fnameAscii.c_str(), trimString.c_str());
            m_File->Add(elem);
        }
#endif
        else if (0 == wcscmp(L".fsdb", imExt.c_str()))
        {
            // root_path が設定されていない場合は path を入れない。
            std::string trimString = nw::g3d::tool::util::TrimString(fullpathUtf, m_FsdRootPathUtf);
            if (m_FsdRootPathUtf.empty())
            {
                trimString = "";
            }

            std::shared_ptr<nw::g3d::tool::g3dif::elem_shader_definition> elem =
                nw::g3d::tool::util::AddBinaryElem<nw::g3d::tool::g3dif::elem_shader_definition>(pRoot, nw::g3d::tool::util::AddOffset(pData, nw::g3d::tool::util::Align(textSize, static_cast<int>(nw::g3d::tool::g3dif::STREAM_BINARY_ALIGNMENT))), m_Check);
            CheckVersion(elem, NN_G3D_INTERMEDIATE_FSD_VERSION_MAJOR, NN_G3D_INTERMEDIATE_FSD_VERSION_MINOR, NN_G3D_INTERMEDIATE_FSD_VERSION_MICRO, "shader_definition");
            nw::g3d::tool::util::AddPath(elem, fnameAscii.c_str(), trimString.c_str());
            m_File->Add(elem);
            m_File->SetName(fnameAscii);
            m_ConvertMode = ConvertMode_FsdToBfsha;
        }
        // シェーダバリエーション
        else if (0 == wcscmp(L".fsva", imExt.c_str()) || 0 == wcscmp(L".fsvb", imExt.c_str()))
        {
            std::shared_ptr<nw::g3d::tool::g3dif::elem_shader_variation> elem =
                nw::g3d::tool::util::AddElem<nw::g3d::tool::g3dif::elem_shader_variation>(pRoot, m_Check);
            CheckVersion(elem, NN_G3D_INTERMEDIATE_FSV_VERSION_MAJOR, NN_G3D_INTERMEDIATE_FSV_VERSION_MINOR, NN_G3D_INTERMEDIATE_FSV_VERSION_MICRO, "shader_variation");
            nw::g3d::tool::util::AddPath(elem, fnameAscii.c_str(), fullpathUtf.c_str());
            m_File->Add(elem);
        }
        else
        {
            THROW_ERROR(ERRCODE_UNSUPPORTED_FILETYPE, "Unsupported file type.");
        }
        m_File->SetConvertMode(m_ConvertMode);
        return true;
    }
    CATCH_ERROR_ALL(false)
}

bool Converter::Make()
{
    try
    {
        if (m_ConvertMode == ConvertMode_FscToFsd)
        {
            m_File->Make(nw::g3d::tool::util::CheckFlag(m_Behavior, BEHAVIOR_UNIFIED_ANNOTATION), nw::g3d::tool::util::CheckFlag(m_Behavior, BEHAVIOR_AUTO_EXTRACT));
        }

        return true;
    }
    CATCH_ERROR_ALL(false)
}

size_t Converter::CalculateDefinitionSize(const wchar_t* fullpath)
{
    try
    {
        m_ImSize = 0;
        if (m_File->DefinitionFromConfig() != nullptr)
        {
            m_ImOutStream.setstate(std::ios::binary);

            static const u8 BOM[] = { 0xEF, 0xBB, 0xBF };
            m_ImOutStream.write(reinterpret_cast<const char*>(BOM), sizeof(BOM));

            std::wstring wpath(fullpath);
            // ワイド文字でセパレータを置換してから utf-8 に変換
            std::replace(wpath.begin(), wpath.end(), L'\\', L'/');
            std::string fullpathUtf = nw::g3d::tool::util::ToUTF8(wpath.c_str());

            if (!m_FsdRootPathUtf.empty())
            {
                std::string temppath = nw::g3d::tool::util::TrimString(fullpathUtf, m_FsdRootPathUtf);
                m_File->DefinitionFromConfig()->SetPath(temppath.c_str());
            }

            std::string fname;
            std::string ext;
            nw::g3d::tool::util::Path::Split(fullpathUtf, nullptr, nullptr, &fname, &ext);
            m_File->DefinitionFromConfig()->SetName(fname.c_str());
            m_File->DefinitionFromConfig()->SetExt(ext.c_str());
            m_File->DefinitionFromConfig()->WriteStream(m_ImOutStream);
            m_File->DefinitionFromConfig()->WriteBinaryStream(m_ImOutStream);

            m_ImSize = static_cast<size_t>(m_ImOutStream.tellp());
        }

        return m_ImSize;
    }
    CATCH_ERROR_ALL(0)
}

bool Converter::Write(void* pBuffer, size_t size)
{
    try
    {
        if (m_File->DefinitionFromConfig() != nullptr)
        {
            if (m_ImSize < size)
            {
                THROW_ERROR_INTERNAL(ERRCODE_INTERNAL, "Internal error.");
            }
            m_ImOutStream.seekp(0);
            memcpy(pBuffer, m_ImOutStream.str().data(), m_ImSize);
        }

        return true;
    }
    CATCH_ERROR_ALL(false)
}

size_t Converter::CalculateArchiveSize()
{
    try
    {
        if (m_ConvertMode != ConvertMode_FscToFsd)
        {
            if (m_ConvertMode == ConvertMode_MergeBfsha)
            {
                // bfsha を shader_definition_elem に変換して追加
                m_File->ConstructFsdFromBfsha();
            }

            // Builder を作成
            // オプションに対応するユニフォーム変数を設定
            // 有効な choice の配列を作る
            // 分散コンパイルの設定
            m_File->MakeVariation();

            // ShaderProgram の作成
            // force include とステージマクロを追加
            // インクルード用コールバックパラメーター
            // オプションマクロの追加
            // バリエーションマクロの追加
            // コンパイル
            m_File->CalculateShaderFileSize(m_Ctx);
            if( m_SkipConvert )
            {
                return 0;
            }
            m_File->Build(m_Ctx);
            //m_Ctx->pool.Finalize();

            auto execCalcSize = [&](BinaryBlock* pBlock)
            {
                pBlock->CalculateSize();
            };
            std::for_each(m_Ctx->blocks.begin(), m_Ctx->blocks.end(), execCalcSize);

            // 並列処理をする場合はこちらを用います。
            //parallel_for_each(blocks.begin(), blocks.end(), execCalcSize);

            auto execCalcOffset = [&](BinaryBlock* pBlock)
            {
                pBlock->CalculateOffset(m_Ctx);
            };
            std::for_each(m_Ctx->blocks.begin(), m_Ctx->blocks.end(), execCalcOffset); // 並列化不可。

            if( m_Ctx->GetMemBlockSize( Context::MemBlockType_Main ) != 0 )
            {
                m_Ctx->SetAlignment( Context::MemBlockType_Main, BinaryAliment_Default );
            }

            if (m_Ctx->GetMemBlockSize(Context::MemBlockType_ShaderFile) != 0)
            {
                // 全ての ShaderFile の中の最大アラインメントを取得する(SDK6.0.0 時では全てのアラインメントは同じ)
                int maxAlignmentShaderFile = 0;
                for (auto shaderContext = m_Ctx->m_ShaderContexts.cbegin(); shaderContext != m_Ctx->m_ShaderContexts.cend(); ++shaderContext)
                {
                    if (shaderContext->m_pPrograms.size() > 0)
                    {
                        int shaderStreamAlignment = static_cast<int>(shaderContext->m_ResShaderFileAlignment);
                        maxAlignmentShaderFile = (maxAlignmentShaderFile < shaderStreamAlignment) ? shaderStreamAlignment : maxAlignmentShaderFile;
                    }
                }

                m_Ctx->SetAlignment( Context::MemBlockType_ShaderFile, maxAlignmentShaderFile);
            }
        }

        // シリアライズコンテキストを初期化します。
        m_Ctx->Build( nn::util::string_view( m_File->GetName().c_str() ) );
        m_File->Compile(m_Ctx);

        size_t resourceSize = m_Ctx->GetBaseSize();
        return resourceSize;	// リロケーションテーブルを含まないサイズを返す。
    }
    CATCH_ERROR_ALL(0)
}

bool Converter::Convert(void* pBuffer, size_t size)
{
    NN_UNUSED( pBuffer );
    NN_UNUSED( size );
    try
    {
        if (m_ConvertMode != ConvertMode_FscToFsd)
        {
            auto execConvert = [&](BinaryBlock* pBlock)
            {
                pBlock->Convert(m_Ctx);
            };
            std::for_each(m_Ctx->blocks.begin(), m_Ctx->blocks.end(), execConvert);

            // 並列処理をする場合はこちらを用います。
            //parallel_for_each(blocks.begin(), blocks.end(), execConvert);

            // コンバート後に補正を行います。
            auto execAdjust = [&](BinaryBlock* pBlock)
            {
                pBlock->Adjust(m_Ctx);
            };
            std::for_each(m_Ctx->blocks.begin(), m_Ctx->blocks.end(), execAdjust);

            // relocation table のサイズ計算と作成
            m_Ctx->Convert();
        }
    }
    CATCH_ERROR_ALL(false)

    return true;
}

bool Converter::SwapEndian(void* pBuffer, size_t /*size*/)
{
    NN_UNUSED( pBuffer );

    try
    {
#if !defined( NN_GFX_TOOL_ENDIAN_SWAP_IS_NOT_IMPLEMENTED )
        nn::g3d::ResShaderArchive* resShaderArchive = nn::g3d::ResShaderArchive::ResCast(pBuffer);

        if (nw::g3d::tool::util::CheckFlag(m_Check, CHECK_ENDIAN_SWAP))
        {
            u8* copyBuffer = new u8[m_Ctx->bufSize];
            memcpy(copyBuffer, pBuffer, m_Ctx->bufSize);

            // エンディアン反転
#if 0	//!< TODO: nn g3d の endian 反転を使用する
            nw::g3d::res::Endian<false>::Swap(resShaderArchive->ptr());

            // 正しくエンディアンスワップできているか確認する。
            nw::g3d::res::Endian<true>::Swap(resShaderArchive->ptr());
            if (memcmp(copyBuffer, resShaderArchive->ptr(), m_Ctx->bufSize) != 0)
            {
                THROW_ERROR(ERRCODE_ENDIAN_SWAP_MISMATCH, "Mismatch endian swap.");
            }
#endif

            // 0xDEADBEAF が存在しないか確認する。
            for (int i = 0; i < static_cast<int>(m_Ctx->bufSize) / 4; ++i)
            {
                if (copyBuffer[i*4+0] == 0xde &&
                    copyBuffer[i*4+1] == 0xad &&
                    copyBuffer[i*4+2] == 0xbe &&
                    copyBuffer[i*4+3] == 0xaf)
                {
                    THROW_ERROR(ERRCODE_UNINITIALIZED_MEMORY, "Exist uninitialized memory.");
                }
            }

            delete[] copyBuffer;
        }

        // エンディアン反転
#if 0	//!< TODO: nn g3d を swap を使う
        nw::g3d::res::Endian<false>::Swap(resShaderArchive->ptr());
#endif
#endif
        return true;
    }
    CATCH_ERROR_ALL(false)
}

} // namespace g3dTool
} // namespace nn
