﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <conio.h>
#include <filesystem>
#include <iostream>
#include <algorithm>
#include <iterator>
#include <mutex>
#include <nn/gfxTool/gfxTool_ApiCommon.h>
#include <nn/gfxTool/gfxTool_ShaderCompilerApi.h>
#include <util/UtilError.h>
#include <util/UtilErrorCode.h>
#include <ShaderCompilerManager.h>
#include <nn/gfxTool/gfxTool_ShaderCompilerApi-nvn.h>
#include <nvnTool/nvnTool_GlslcInterface.h>
#include <pugixml.hpp>

namespace {

// シェーダプログラムの出力を取得
const nngfxToolShaderCompilerShaderProgramOutput* GetShaderCompilerShaderProgramOutput(
    const nngfxToolShaderCompilerCompileOutput* pOutput,
    int idxVariation, ShaderCompilerManager::ShaderCodeType idxProgram)
{
    static nngfxToolShaderCompilerShaderProgramOutput*
        nngfxToolShaderCompilerVariationOutput::*s_pProgramOutputs[] =
    {
        &nngfxToolShaderCompilerVariationOutput::pSourceOutput,
        &nngfxToolShaderCompilerVariationOutput::pIntermediateLanguageOutput,
        &nngfxToolShaderCompilerVariationOutput::pBinaryOutput
    };

    auto variationCount = static_cast< int >( pOutput->variationCount );
    if( idxVariation >= variationCount )
    {
        THROW_ERROR(ERRCODE_INTERNAL, "Variation index should be in %d. Index is %d", variationCount, idxVariation);
    }

    auto& variationOutput = pOutput->pVariationOutputArray[ idxVariation ];
    if( idxProgram >= static_cast<int>( ARRAYSIZE( s_pProgramOutputs ) ) )
    {
        THROW_ERROR(ERRCODE_INTERNAL, "Program index should be in %d. Index is %d", static_cast<int>( ARRAYSIZE( s_pProgramOutputs ) ), idxProgram);
    }

    if( auto pProgramOutput = variationOutput.*( s_pProgramOutputs[ idxProgram ] ) )
    {
        return pProgramOutput;
    }

    return nullptr;
}

bool ReadIncludeFileCallback( void** ppOutFileData,		/* out: シェーダコードのストリームへのポインタ */
                              size_t* pStrLength,		/* out: シェーダコードの文字数、終端コードは含まない */
                              const char* pFileName,	/* in: gfx の ShaderConverter から相対パス付きのファイル名が入力される。*/
                              void* pParam				/* in: 3d が gfxTool にセットした FileReadCallbackParam へのポインタ */
                            )
{
    auto pCbParam = static_cast<ShaderCompilerManager::FileReadCallbackParam*>( pParam );
    auto findShaderInfoByName = [&]( const ShaderCompilerManager::ShaderFileInfo& shaderFileInfo )->bool
    {
        if( *shaderFileInfo.pRelativeFilePathName == pFileName )
        {
            return true;
        }
        return false;
    };
    auto shaderInfo = std::find_if( pCbParam->shaderFileInfoArray.cbegin(), pCbParam->shaderFileInfoArray.cend(), findShaderInfoByName );
    if( shaderInfo == pCbParam->shaderFileInfoArray.cend() )
    {
        std::string includeFileDiscardedMessage( "// 3dShaderConverter discarded include file: " );
        includeFileDiscardedMessage = includeFileDiscardedMessage + pFileName;

        size_t buffSize = includeFileDiscardedMessage.size();
        void* buff = malloc( buffSize + 1 );
        if( buff == nullptr )
        {
            NN_GFXTOOL_PRINT( "3dShaderConverter Error: Cannot allocate buffer for message of discarded include file.\n" );
            return false;
        }
        memcpy( buff, includeFileDiscardedMessage.c_str(), buffSize );
        reinterpret_cast<char*>( buff )[buffSize] = '\0';
        ShaderCompilerManager::DataStream includeFileDiscardedMessageStringDataStream;
        includeFileDiscardedMessageStringDataStream.data.reset( buff, free );
        includeFileDiscardedMessageStringDataStream.size = buffSize + 1;
        pCbParam->includeFileDiscardedMessageStringDataStreamArray.push_back( includeFileDiscardedMessageStringDataStream );
        *ppOutFileData = buff;
        *pStrLength	= includeFileDiscardedMessage.length();
        return true;
    }

    *ppOutFileData		= shaderInfo->pFileData;
    *pStrLength	= shaderInfo->strLength;
    return true;
}

}	// no namespace

void WriteFileCallback::SetDirectory(const char* directory)
{
    m_Directory = directory;
    std::replace(m_Directory.begin(), m_Directory.end(), '/', '\\');
    if (m_Directory.length() > 0 && m_Directory.back() != '\\')
    {
        m_Directory.push_back('\\');
    }
}

bool WriteFileCallback::WriteFileFromDirectory(const void* pFileData, size_t fileDataSize,
    const char* pFilename, void* pParam)
{
    auto pThis = static_cast< WriteFileCallback* >(pParam);
    std::string path = pThis->m_Directory;
    path.append(pFilename);
    return pThis->WriteFile(pFileData, fileDataSize, path.c_str(), nullptr);
}

bool WriteFileCallback::WriteFile(const void* pFileData,
    size_t fileDataSize,
    const char* pFilename,
    void* pParam
)
{
    NN_UNUSED(pParam);
    std::ofstream ofs(pFilename, std::ios::out | std::ios::binary);
    if (ofs.is_open())
    {
        ofs.write(static_cast< const char* >(pFileData), fileDataSize);
        ofs.close();
        return true;
    }
    return true;
}

void ReadFileCallback::AddDirectory(const char* directory)
{
    m_Directories.push_back(directory);
    std::replace(m_Directories.back().begin(), m_Directories.back().end(), '/', '\\');
    if (m_Directories.back().length() > 0 && m_Directories.back().back() != '\\')
    {
        m_Directories.back().push_back('\\');
    }
}

bool ReadFileCallback::ReadFileFromDirectories(void** ppOutFileData,
    size_t* pOutFileDataSize,
    const char* pFileName,
    void* pParam
)
{
    auto pThis = static_cast< ReadFileCallback* >(pParam);

    if (pThis->m_Directories.size() < 1)
    {
        return pThis->ReadFile(ppOutFileData, pOutFileDataSize, pFileName, pThis->GetCallbackParam());
    }

    for (auto& directory : pThis->m_Directories)
    {
        auto path = directory + pFileName;
        if (pThis->ReadFile(ppOutFileData, pOutFileDataSize, path.data(), pThis->GetCallbackParam()))
        {
            return true;
        }
    }
    return false;
}

bool ReadFileCallback::ReadFile(void** ppOutFileData,
    size_t* pOutFileDataSize,
    const char* pFileName,
    void* pParam
)
{
    static std::mutex s_Mutex;
    std::lock_guard< decltype(s_Mutex) > lock(s_Mutex);

    auto& param = *reinterpret_cast<CallbackParam*>(pParam);

    HANDLE hFile = CreateFileA( pFileName, GENERIC_READ,
        FILE_SHARE_READ, nullptr, OPEN_EXISTING, 0, nullptr );
    if (hFile == INVALID_HANDLE_VALUE)
    {
        return false;
    }

    auto fileSize = GetFileSize( hFile, nullptr );
    std::vector< char > buffer( fileSize );
    DWORD read = 0;
    BOOL result = TRUE;
    if( fileSize > 0 )
    {
        ::ReadFile( hFile, &buffer[ 0 ], fileSize, &read, nullptr );
    }
    CloseHandle( hFile );
    if( result == FALSE || read != fileSize )
    {
        return false;
    }

    param.pFileArray->emplace_back();
    auto& iter = param.pFileArray->back();

    iter.first = pFileName;
    iter.second = std::move( buffer );

    *ppOutFileData = iter.second.size() > 0 ? &iter.second[ 0 ] : nullptr;
    *pOutFileDataSize = iter.second.size();

    return true;
};

ShaderConverterDll	ShaderCompilerManager::m_sShaderConverterDll;

bool ShaderCompilerManager::InitializeDll( void )
{
    bool result = false;
    std::string path = "..\\GraphicsTools\\x86\\ShaderConverter.dll";
    result = m_sShaderConverterDll.Initialize( path.c_str() );
    if( !result )
    {
        path = "..\\GraphicsTools\\ShaderConverter.dll";
        result = m_sShaderConverterDll.Initialize( path.c_str() );
    }
    return result;
}

void ShaderCompilerManager::FinalizeDll( void )
{
    m_sShaderConverterDll.Finalize();
}

bool ShaderCompilerManager::SetLogStream(const nngfxToolSetLogStreamArg* pArg)
{
    if (m_sShaderConverterDll.SetLogStream(pArg) != nngfxToolResultCode_Succeeded)
    {
        return false;
    }
    return true;
}

void ShaderCompilerManager::InitializeGfxToolConverterOptions( const std::vector<std::string>& options )
{
    // グローバルオプションのセット
    const char* defaultOptions[] =
    {
        "--reflection",
        "-o""dummy",			// -o option を指定しないと GfxToolConverter のパーサで弾かれる。ConvertArg 生成時に出力しないようにする。
        "--print-progress",		// デフォルトでコンパイル進捗を表示するようにする。
        "-sGlsl",				// 今は Glsl しか対応していないのでデフォルトオプションとする。
        "--include-directory=""",	// ダミーのインクルードディレクトリです。本オプションを追加しないと FileReadCb が登録されません。
    };

    const int defaultOptionCount = sizeof( defaultOptions ) / sizeof( const char* );
    for( int defaultOptionIdx = 0; defaultOptionIdx < defaultOptionCount; ++defaultOptionIdx )
    {
        m_OptionNames.push_back( defaultOptions[ defaultOptionIdx ] );
    }

    for( auto externalOptionNameIter = options.begin(); externalOptionNameIter != options.end(); ++externalOptionNameIter )
    {
        m_OptionNames.push_back( *externalOptionNameIter );	// 文字列をコピーして持つ。
    }

    for( auto& optionName : m_OptionNames )
    {
        m_OptionNamePtrArray.push_back( optionName.c_str() );
        if( optionName == "-c=Source" )
        {
            m_IsSourceOnly = true;
            m_OptionNamePtrArraySourceBinary.push_back( "-c=Binary_Source" );
        }
        else
        {
            m_OptionNamePtrArraySourceBinary.push_back( optionName.c_str() );
        }
    }
}

void ShaderCompilerManager::SetShaderSource( const std::string& shaderSource, ShaderType shaderType )
{
    if( shaderType >= ShaderType_Count )
    {
        THROW_ERROR(ERRCODE_UNKNOWN_ENUM, "Unknown enum value. %d", shaderType);
    }

    static nngfxToolString nngfxToolShaderCompilerCompileArg::*
        s_SourceString[] =
        {
            &nngfxToolShaderCompilerCompileArg::vertexShaderSource,
            &nngfxToolShaderCompilerCompileArg::geometryShaderSource,
            &nngfxToolShaderCompilerCompileArg::pixelShaderSource,
            &nngfxToolShaderCompilerCompileArg::computeShaderSource,
            &nngfxToolShaderCompilerCompileArg::hullShaderSource,
            &nngfxToolShaderCompilerCompileArg::domainShaderSource,
        };

    nngfxToolShaderCompilerCompileArg* pCompileArg =
        const_cast<nngfxToolShaderCompilerCompileArg*>( m_pConvertArg->pCompileArg );

    ( pCompileArg->*( s_SourceString[ shaderType ] ) ).pValue = shaderSource.c_str();
    ( pCompileArg->*( s_SourceString[ shaderType ] ) ).length = static_cast<uint32_t>( shaderSource.length() );

    if( m_IsSourceOnly )
    {
        nngfxToolShaderCompilerCompileArg* pCompileArgSourceBinary =
            const_cast<nngfxToolShaderCompilerCompileArg*>( m_pConvertArgSourceBinary->pCompileArg );
        ( pCompileArgSourceBinary->*( s_SourceString[ shaderType ] ) ).pValue = ( pCompileArg->*( s_SourceString[ shaderType ] ) ).pValue;
        ( pCompileArgSourceBinary->*( s_SourceString[ shaderType ] ) ).length = ( pCompileArg->*( s_SourceString[ shaderType ] ) ).length;
    }
}

nngfxToolResultCode ShaderCompilerManager::InitializeConvertArg( void )
{
    nngfxToolResultCode result;

    result = m_sShaderConverterDll.CreateHandle( &m_GfxToolHandle, &m_CreateHandleArg );
    if( result != nngfxToolResultCode_Succeeded )
    {
        NN_GFXTOOL_PRINT( "Failed to delete handle.\n" );
        return result;
    }

    if( m_IsSourceOnly )
    {
        result = m_sShaderConverterDll.CreateHandle( &m_GfxToolHandleSourceBinary, &m_CreateHandleArgSourceBinary );
        if( result != nngfxToolResultCode_Succeeded )
        {
            NN_GFXTOOL_PRINT( "Failed to delete handle.\n" );
            return result;
        }
    }

    m_ConvertArgArg.optionCount = static_cast<uint32_t>( m_OptionNamePtrArray.size() );
    m_ConvertArgArg.ppOptionArray = &m_OptionNamePtrArray[ 0 ];
    m_ConvertArgArg.pReadInputFileCallback = nullptr;
    m_ConvertArgArg.pReadInputFileCallbackParam = nullptr;
    m_ConvertArgArg.pReadIncludeFileCallback = ReadIncludeFileCallback;
    m_ConvertArgArg.pReadIncludeFileCallbackParam = &m_FileReadCbParam;
    m_ConvertArgArg.pWriteDebugInfoFileCallback = WriteFileCallback::WriteFile;
    m_ConvertArgArg.pWriteDebugInfoFileCallbackParam = nullptr;

    // 後で設定するため初期値に null を入れておく
    m_ConvertArgArg.pReadShaderCacheCallback = nullptr;
    m_ConvertArgArg.pReadShaderCacheCallbackParam = nullptr;
    m_ConvertArgArg.pWriteShaderCacheCallback = nullptr;
    m_ConvertArgArg.pWriteShaderCacheCallbackParam = nullptr;

    if( m_IsSourceOnly )
    {
        m_ConvertArgArgSourceBinary = m_ConvertArgArg;
        m_ConvertArgArgSourceBinary.optionCount = static_cast<uint32_t>( m_OptionNamePtrArraySourceBinary.size() );
        m_ConvertArgArgSourceBinary.ppOptionArray = &m_OptionNamePtrArraySourceBinary[ 0 ];
    }

    result = m_sShaderConverterDll.CreateConvertArg( &m_pConvertArg, &m_ConvertArgArg );
    if( result != nngfxToolResultCode_Succeeded )
    {
        NN_GFXTOOL_PRINT( "CreateConvertArg() Failure Error Code: %d\n", result );
        m_sShaderConverterDll.DeleteHandle( m_GfxToolHandle );
        return result;
    }

    if( m_IsSourceOnly )
    {
        result = m_sShaderConverterDll.CreateConvertArg( &m_pConvertArgSourceBinary, &m_ConvertArgArgSourceBinary );
        if( result != nngfxToolResultCode_Succeeded )
        {
            NN_GFXTOOL_PRINT( "CreateConvertArg() Failure Error Code: %d\n", result );
            m_sShaderConverterDll.DeleteHandle( m_GfxToolHandleSourceBinary );
            return result;
        }
    }

    // マージファイルが存在する場合は pCompileArg を NULL にして終了
    if (m_pMergeResShaderFileArray.size() > 0)
    {
        m_pConvertArg->pCompileArg = NULL;
        m_pConvertArg->mergeShaderFileCount = static_cast<int32_t>(m_pMergeResShaderFileArray.size());
        m_pConvertArg->pMergeShaderFiles = &m_pMergeResShaderFileArray[0];
        return result;
    }

    // 外部から受け取ったコンバート設定をセットします。
    {

        if (m_FileReadCbParam.shaderFileInfoArray.size() == 0)
        {
            THROW_ERROR_INTERNAL(ERRCODE_INTERNAL, "Internal error. shaderFileInfoArray.size() == 0.");
        }

        if( m_pVariationValueArgArray.size() == 0 || m_pVariationDefinitionArg == nullptr )
        {
            NN_GFXTOOL_PRINT( "There is no preprocessor value definition or preprocessor definition.\n" );
            m_sShaderConverterDll.DeleteHandle( m_GfxToolHandle );
            if( m_IsSourceOnly )
            {
                m_sShaderConverterDll.DeleteHandle( m_GfxToolHandleSourceBinary );
            }
            return result;
        }

        const_cast<nngfxToolShaderCompilerCompileArg*>
            ( m_pConvertArg->pCompileArg )->variationCount = static_cast<uint32_t>( m_pVariationValueArgArray.size() );

        if( !m_pVariationValueArgArray.empty() )
        {
            const_cast<nngfxToolShaderCompilerCompileArg*>
                ( m_pConvertArg->pCompileArg )->pVariationValueArray = &m_pVariationValueArgArray[ 0 ];
        }

        if( m_pVariationDefinitionArg != nullptr )
        {
            const_cast<nngfxToolShaderCompilerCompileArg*>
                ( m_pConvertArg->pCompileArg )->pVariationDefinitionArg = m_pVariationDefinitionArg;
        }

        if( m_IsSourceOnly )
        {
            const_cast<nngfxToolShaderCompilerCompileArg*>
                ( m_pConvertArgSourceBinary->pCompileArg )->variationCount = m_pConvertArg->pCompileArg->variationCount;
            const_cast<nngfxToolShaderCompilerCompileArg*>
                ( m_pConvertArgSourceBinary->pCompileArg )->pVariationValueArray = m_pConvertArg->pCompileArg->pVariationValueArray;
            const_cast<nngfxToolShaderCompilerCompileArg*>
                ( m_pConvertArgSourceBinary->pCompileArg )->pVariationDefinitionArg = m_pConvertArg->pCompileArg->pVariationDefinitionArg;
        }

        // キャッシュディレクトリの読み込みを行うコールバック関数とディレクトリを上書き設定
        if (m_ShaderCacheOptionInfo.shaderCacheAccessFlag & ShaderCacheAccessType_Write)
        {
            m_WriteFile.SetDirectory(m_ShaderCacheOptionInfo.shaderCacheWriteDirectory.c_str());
            const_cast<nngfxToolShaderCompilerCompileArg*>
                (m_pConvertArg->pCompileArg)->pWriteShaderCacheCallback = m_ShaderCacheOptionInfo.shaderCacheWriteDirectory.empty() ? nullptr : WriteFileCallback::WriteFileFromDirectory;
            const_cast<nngfxToolShaderCompilerCompileArg*>
                (m_pConvertArg->pCompileArg)->pWriteShaderCacheCallbackParam = &m_WriteFile;
        }

        if (m_ShaderCacheOptionInfo.shaderCacheAccessFlag & ShaderCacheAccessType_Read)
        {
            std::for_each(m_ShaderCacheOptionInfo.shaderCacheReadDirectoryArray.cbegin(), m_ShaderCacheOptionInfo.shaderCacheReadDirectoryArray.cend(), [&](const std::string& Directory)
            {
                m_ReadFile.AddDirectory(Directory.c_str());
            });
            const_cast<nngfxToolShaderCompilerCompileArg*>
                (m_pConvertArg->pCompileArg)->pReadShaderCacheCallback = m_ShaderCacheOptionInfo.shaderCacheReadDirectoryArray.empty() ? nullptr : ReadFileCallback::ReadFileFromDirectories;
            const_cast<nngfxToolShaderCompilerCompileArg*>
                (m_pConvertArg->pCompileArg)->pReadShaderCacheCallbackParam = &m_ReadFile;
        }
    }

    return result;
}

nngfxToolResultCode ShaderCompilerManager::FinalizeConvertArg( void )
{
    nngfxToolResultCode result;

    result = m_sShaderConverterDll.DeleteConvertArg(m_pConvertArg);
    if (result != nngfxToolResultCode_Succeeded)
    {
        NN_GFXTOOL_PRINT("Failed to delete conversion arguments.\n");
        m_sShaderConverterDll.DeleteHandle(m_GfxToolHandle);
        return result;
    }

    if (m_IsSourceOnly)
    {
        result = m_sShaderConverterDll.DeleteConvertArg(m_pConvertArgSourceBinary);
        if (result != nngfxToolResultCode_Succeeded)
        {
            NN_GFXTOOL_PRINT("Failed to delete conversion arguments.\n");
            m_sShaderConverterDll.DeleteHandle(m_GfxToolHandleSourceBinary);
            return result;
        }
    }
    return result;
}

bool ShaderCompilerManager::Convert( size_t* pOutShaderFileSize, size_t* pOutShaderFileAlignment )
{
    if (pOutShaderFileSize == NULL || pOutShaderFileAlignment == NULL)
    {
        THROW_ERROR_INTERNAL(ERRCODE_INTERNAL, "Internal error. pOut variable is NULL.");
    }

    nngfxToolResultCode result;

    nngfxToolShaderConverterConvertOutput convertOutput;
    result = m_sShaderConverterDll.Convert( &convertOutput, m_GfxToolHandle, m_pConvertArg );
    if( result != nngfxToolResultCode_Succeeded )
    {
        NN_GFXTOOL_PRINT( "Conversion failed.\n" );
        m_sShaderConverterDll.DeleteConvertArg( m_pConvertArg );
        m_sShaderConverterDll.DeleteHandle( m_GfxToolHandle );
        return false;
    }
    if (convertOutput.binarySize == 0)
    {
        return false;
    }

    *pOutShaderFileSize = convertOutput.binarySize;
    *pOutShaderFileAlignment = convertOutput.binaryAlignment;

    // マージモードの時はコンパイル結果が無いのでここで返す。
    if (m_pMergeResShaderFileArray.size() > 0)
    {
        return true;
    }

    const int variationCount = convertOutput.pCompileOutput->variationCount;

    // シェーダコードの設定
    for( int idxCodeType = 0; idxCodeType < ShaderCodeType_Count; ++idxCodeType )
    {
        m_AttribInfoSetArray[ idxCodeType ].resize( variationCount );
        m_UniformBlockInfoSetArray[ idxCodeType ].resize( variationCount );
        m_SsboBlockInfoSetArray[idxCodeType].resize( variationCount );
        m_UniformInfoSetArray[ idxCodeType ].resize( variationCount );
        m_SsboInfoSetArray[idxCodeType].resize( variationCount );
        m_SamplerInfoSetArray[ idxCodeType ].resize( variationCount );
        m_PipelineInfoArray[ idxCodeType ].resize( variationCount );
        for( int idxVariation = 0; idxVariation < variationCount; ++idxVariation )
        {
            const nngfxToolShaderCompilerShaderProgramOutput* pShaderCompilerShaderProgramOutput = nullptr;
            pShaderCompilerShaderProgramOutput =
                GetShaderCompilerShaderProgramOutput( convertOutput.pCompileOutput, idxVariation, static_cast<ShaderCodeType>( idxCodeType ) );

            SetShaderInfo( *pShaderCompilerShaderProgramOutput, idxVariation, static_cast<ShaderCodeType>( idxCodeType ) );
        }
    }

    // code-type=Source の場合は Binary_Source 用ハンドラからリフレクション情報を取得する
    if( m_IsSourceOnly )
    {
        memset( &convertOutput, 0, sizeof( nngfxToolShaderConverterConvertOutput ) );
        result = m_sShaderConverterDll.Convert( &convertOutput, m_GfxToolHandleSourceBinary, m_pConvertArgSourceBinary );
        if( result != nngfxToolResultCode_Succeeded )
        {
            NN_GFXTOOL_PRINT( "Conversion failed.\n" );
            m_sShaderConverterDll.DeleteConvertArg( m_pConvertArgSourceBinary );
            m_sShaderConverterDll.DeleteHandle( m_GfxToolHandleSourceBinary );
            return false;
        }
    }

    // リフレクション情報の設定
    for( int idxCodeType = 0; idxCodeType < ShaderCodeType_Count; ++idxCodeType )
    {
        for( int idxVariation = 0; idxVariation < variationCount; ++idxVariation )
        {
            const nngfxToolShaderCompilerShaderProgramOutput* pShaderCompilerShaderProgramOutput = nullptr;
            pShaderCompilerShaderProgramOutput =
                GetShaderCompilerShaderProgramOutput(convertOutput.pCompileOutput, idxVariation, static_cast<ShaderCodeType>(idxCodeType));

            SetReflectionData( *pShaderCompilerShaderProgramOutput, idxVariation, static_cast<ShaderCodeType>( idxCodeType ) );
        }
    }
    return true;
}

nngfxToolResultCode ShaderCompilerManager::Serialize( void* pBuff, size_t buffSize )
{
    nngfxToolResultCode result = nngfxToolResultCode_Succeeded;

    char* pBinary = reinterpret_cast<char*>( pBuff );
    result = m_sShaderConverterDll.Serialize( pBinary, buffSize, m_GfxToolHandle );
    if( result != nngfxToolResultCode_Succeeded )
    {
        NN_GFXTOOL_PRINT( "	Serialization failed.\n" );
        m_sShaderConverterDll.DeleteHandle( m_GfxToolHandle );
        return result;
    }

    result = m_sShaderConverterDll.DeleteHandle( m_GfxToolHandle );
    if( result != nngfxToolResultCode_Succeeded )
    {
        NN_GFXTOOL_PRINT( "	Failed to delete handle.\n" );
        return result;
    }
    if (m_IsSourceOnly)
    {
        result = m_sShaderConverterDll.DeleteHandle( m_GfxToolHandleSourceBinary );
        if (result != nngfxToolResultCode_Succeeded)
        {
            NN_GFXTOOL_PRINT("	Failed to delete handle.\n");
            return result;
        }
    }

    return result;
}

void ShaderCompilerManager::SetReflectionData(
    const nngfxToolShaderCompilerShaderProgramOutput& shaderCompilerShaderProgramOutput,
    int idxVariation, ShaderCodeType idxShaderCodeType )
{
    const nngfxToolShaderCompilerShaderProgramOutput* pOutput = &shaderCompilerShaderProgramOutput;
    if( pOutput == nullptr )
    {
        return;
    }

    const nngfxToolShaderCompilerOptionOutputProgramCommon* pOptionOutputProgramCommon = nullptr;
    for (int idxOption = 0; idxOption < static_cast< int >(pOutput->optionOutputCount); ++idxOption)
    {
        auto& option = pOutput->pOptionOutputArray[idxOption];
        switch (option.optionOutputType)
        {
        case nngfxToolShaderCompilerOptionOutputType_ProgramCommon:
            {
                pOptionOutputProgramCommon = nn::gfxTool::StaticCastAuto(option.pOutput);
            }
            break;
        default:
            break;
        }
    }

    if (pOptionOutputProgramCommon == nullptr)
    {
        return;
    }

    if (auto pReflection = pOptionOutputProgramCommon->pReflection)
    {
        for( int idxConstantBuffer = 0; idxConstantBuffer < static_cast< int >(
            pReflection->constantBufferCount ); ++idxConstantBuffer )
        {
            auto& target = pReflection->pConstantBufferArray[ idxConstantBuffer ];

            UniformBlockInfo	uniformBlockInfo;
            uniformBlockInfo.index		= idxConstantBuffer;
            uniformBlockInfo.shaderSlot = target.shaderSlot;
            uniformBlockInfo.size		= target.size;
            uniformBlockInfo.stageBit	= target.stages;

            m_UniformBlockInfoSetArray[ idxShaderCodeType ][ idxVariation ].insert( std::make_pair( std::string( target.name.pValue ), uniformBlockInfo ) );
        }
        for( int idxBuffer = 0; idxBuffer < static_cast<int>( pReflection->unorderedAccessBufferCount ); ++idxBuffer )
        {
            auto& target = pReflection->pUnorderedAccessBufferArray[idxBuffer];

            SsboBlockInfo	ssboBlockInfo;
            ssboBlockInfo.index		= idxBuffer;
            ssboBlockInfo.shaderSlot = target.shaderSlot;
            ssboBlockInfo.size		= target.size;
            ssboBlockInfo.stageBit	= target.stages;

            m_SsboBlockInfoSetArray[idxShaderCodeType][idxVariation].insert( std::make_pair( std::string( target.name.pValue ), ssboBlockInfo ) );
        }
        for( int idxShaderInput = 0; idxShaderInput < static_cast< int >( pReflection->shaderInputCount ); ++idxShaderInput )
        {
            auto& target = pReflection->pShaderInputArray[ idxShaderInput ];
            AttribInfo	attribInfo;
            attribInfo.index = idxShaderInput;
            attribInfo.type = target.type;
            attribInfo.arrayCount = target.arrayCount;
            attribInfo.stageBit = target.stages;
            attribInfo.slot = target.shaderSlot;

            m_AttribInfoSetArray[ idxShaderCodeType ][ idxVariation ].insert( std::make_pair( std::string( target.name.pValue ), attribInfo ) );
        }
        for( int idxSampler = 0; idxSampler < static_cast< int >( pReflection->samplerCount ); ++idxSampler )
        {
            auto& target = pReflection->pSamplerArray[ idxSampler ];
            SamplerInfo	samplerInfo;
            samplerInfo.index = idxSampler;
            samplerInfo.name = std::string( target.name.pValue );
            samplerInfo.type = target.type;
            samplerInfo.shaderSlot = target.shaderSlot;
            samplerInfo.stageBit = target.stages;
            m_SamplerInfoSetArray[ idxShaderCodeType ][ idxVariation ].insert(
                std::make_pair( std::string( target.name.pValue ), samplerInfo ) );
        }
        for( int idxConstantBufferVariable = 0; idxConstantBufferVariable < static_cast< int >(
            pReflection->constantBufferVariableCount ); ++idxConstantBufferVariable )
        {
            auto& target = pReflection->pConstantBufferVariableArray[ idxConstantBufferVariable ];
            UniformInfo	uniformInfo;
            uniformInfo.blockIndex = target.blockIndex;
            if( ( target.blockIndex ) != -1 && target.blockIndex < static_cast< decltype( target.blockIndex ) >( pReflection->constantBufferCount ) )
            {
                auto& block = pReflection->pConstantBufferArray[ target.blockIndex ];
                uniformInfo.blockName = std::string( block.name.pValue );
                uniformInfo.type = target.type;
                uniformInfo.offset = target.offset;
                uniformInfo.arrayCount = target.arrayCount;
                uniformInfo.stageBit = target.stages;
            }
            else	// uniformIndex = -1 の時
            {
                uniformInfo.type = -1;
                uniformInfo.offset = -1;
                uniformInfo.arrayCount = -1;
                uniformInfo.stageBit = target.stages;
            }
            m_UniformInfoSetArray[ idxShaderCodeType ][ idxVariation ].insert(
                std::make_pair( std::string( target.name.pValue ), uniformInfo ) );
        }
        // ssbo,uboで同名のメンバ名が存在し得るので分けて管理します。
        for( int idxBufferVariable = 0; idxBufferVariable < static_cast<int>( pReflection->unorderedAccessBufferVariableCount ); ++idxBufferVariable )
        {
            auto& target = pReflection->pUnorderedAccessBufferVariableArray[idxBufferVariable];
            SsboInfo	ssboInfo;
            ssboInfo.blockIndex = target.blockIndex;
            if( target.blockIndex != -1 && ( target.blockIndex < static_cast<decltype( target.blockIndex )>( pReflection->unorderedAccessBufferCount ) ) )
            {
                auto& block = pReflection->pUnorderedAccessBufferArray[ target.blockIndex ];
                ssboInfo.blockName = std::string( block.name.pValue );
                ssboInfo.type = target.type;
                ssboInfo.offset = target.offset;
                ssboInfo.arrayCount = target.arrayCount;
                ssboInfo.stageBit = target.stages;
            }
            else	// uniformIndex = -1 の時
            {
                ssboInfo.type = -1;
                ssboInfo.offset = -1;
                ssboInfo.arrayCount = -1;
                ssboInfo.stageBit = target.stages;
            }
            m_SsboInfoSetArray[idxShaderCodeType][idxVariation].insert( std::make_pair( std::string( target.name.pValue ), ssboInfo ) );
        }
    }
}

void ShaderCompilerManager::SetShaderInfo(
    const nngfxToolShaderCompilerShaderProgramOutput& shaderCompilerShaderProgramOutput,
    int idxVariation, ShaderCodeType idxCodeType )
{
    const nngfxToolShaderCompilerShaderProgramOutput* pOutput = &shaderCompilerShaderProgramOutput;
    if (pOutput == nullptr)
    {
        return;
    }

    const nngfxToolShaderCompilerOptionOutputProgramCommon* pOptionOutputProgramCommon = nullptr;
    const nngfxToolShaderCompilerOptionOutputProgramNvn* pOptionOutputProgramNvn = nullptr;
    for (int idxOption = 0; idxOption < static_cast< int >(pOutput->optionOutputCount); ++idxOption)
    {
        auto& option = pOutput->pOptionOutputArray[idxOption];
        switch (option.optionOutputType)
        {
        case nngfxToolShaderCompilerOptionOutputType_ProgramCommon:
            {
                pOptionOutputProgramCommon = nn::gfxTool::StaticCastAuto(option.pOutput);
            }
            break;
        case nngfxToolShaderCompilerOptionOutputType_ProgramNvn:
            {
                pOptionOutputProgramNvn = nn::gfxTool::StaticCastAuto(option.pOutput);
            }
            break;
        default:
            break;
        }
    }

    if (pOptionOutputProgramCommon == nullptr)
    {
        return;
    }

    if (pOptionOutputProgramCommon->pVertexStageOutput != nullptr)
    {
        m_PipelineInfoArray[idxCodeType][idxVariation].vertexShaderInfo.pSource =
            pOptionOutputProgramCommon->pVertexStageOutput->dump.pValue;
        m_PipelineInfoArray[idxCodeType][idxVariation].vertexShaderInfo.pStats =
            pOptionOutputProgramCommon->pVertexStageOutput->pShaderStatistics;

        if (pOptionOutputProgramNvn != nullptr && pOptionOutputProgramNvn->pVertexStageOutput != nullptr)
        {
            m_PipelineInfoArray[idxCodeType][idxVariation].vertexShaderInfo.scratchMemoryStats.requiredScratchMemorySizePerWarp =
                pOptionOutputProgramNvn->pVertexStageOutput->requiredScratchMemorySizePerWarp;
            m_PipelineInfoArray[idxCodeType][idxVariation].vertexShaderInfo.scratchMemoryStats.recommendedScratchMemorySizePerWarp =
                pOptionOutputProgramNvn->pVertexStageOutput->recommendedScratchMemorySizePerWarp;
        }
    }

    if (pOptionOutputProgramCommon->pGeometryStageOutput != nullptr)
    {
        m_PipelineInfoArray[idxCodeType][idxVariation].geometryShaderInfo.pSource =
            pOptionOutputProgramCommon->pGeometryStageOutput->dump.pValue;
        m_PipelineInfoArray[idxCodeType][idxVariation].geometryShaderInfo.pStats =
            pOptionOutputProgramCommon->pGeometryStageOutput->pShaderStatistics;
        if (pOptionOutputProgramNvn != nullptr && pOptionOutputProgramNvn->pGeometryStageOutput != nullptr)
        {
            m_PipelineInfoArray[idxCodeType][idxVariation].geometryShaderInfo.scratchMemoryStats.requiredScratchMemorySizePerWarp =
                pOptionOutputProgramNvn->pGeometryStageOutput->requiredScratchMemorySizePerWarp;
            m_PipelineInfoArray[idxCodeType][idxVariation].geometryShaderInfo.scratchMemoryStats.recommendedScratchMemorySizePerWarp =
                pOptionOutputProgramNvn->pGeometryStageOutput->recommendedScratchMemorySizePerWarp;
        }
    }

    if (pOptionOutputProgramCommon->pPixelStageOutput != nullptr)
    {
        m_PipelineInfoArray[idxCodeType][idxVariation].pixelShaderInfo.pSource =
            pOptionOutputProgramCommon->pPixelStageOutput->dump.pValue;
        m_PipelineInfoArray[idxCodeType][idxVariation].pixelShaderInfo.pStats =
            pOptionOutputProgramCommon->pPixelStageOutput->pShaderStatistics;
        if (pOptionOutputProgramNvn != nullptr && pOptionOutputProgramNvn->pPixelStageOutput != nullptr)
        {
            m_PipelineInfoArray[idxCodeType][idxVariation].pixelShaderInfo.scratchMemoryStats.requiredScratchMemorySizePerWarp =
                pOptionOutputProgramNvn->pPixelStageOutput->requiredScratchMemorySizePerWarp;
            m_PipelineInfoArray[idxCodeType][idxVariation].pixelShaderInfo.scratchMemoryStats.recommendedScratchMemorySizePerWarp =
                pOptionOutputProgramNvn->pPixelStageOutput->recommendedScratchMemorySizePerWarp;
        }
    }

    if (pOptionOutputProgramCommon->pComputeStageOutput != nullptr)
    {
        m_PipelineInfoArray[idxCodeType][idxVariation].computeShaderInfo.pSource =
            pOptionOutputProgramCommon->pComputeStageOutput->dump.pValue;
        m_PipelineInfoArray[idxCodeType][idxVariation].computeShaderInfo.pStats =
            pOptionOutputProgramCommon->pComputeStageOutput->pShaderStatistics;
        if (pOptionOutputProgramNvn != nullptr && pOptionOutputProgramNvn->pComputeStageOutput != nullptr)
        {
            m_PipelineInfoArray[idxCodeType][idxVariation].computeShaderInfo.scratchMemoryStats.requiredScratchMemorySizePerWarp =
                pOptionOutputProgramNvn->pComputeStageOutput->requiredScratchMemorySizePerWarp;
            m_PipelineInfoArray[idxCodeType][idxVariation].computeShaderInfo.scratchMemoryStats.recommendedScratchMemorySizePerWarp =
                pOptionOutputProgramNvn->pComputeStageOutput->recommendedScratchMemorySizePerWarp;
        }
    }

    if (pOptionOutputProgramCommon->pHullStageOutput != nullptr)
    {
        m_PipelineInfoArray[idxCodeType][idxVariation].hullShaderInfo.pSource =
            pOptionOutputProgramCommon->pHullStageOutput->dump.pValue;
        m_PipelineInfoArray[idxCodeType][idxVariation].hullShaderInfo.pStats =
            pOptionOutputProgramCommon->pHullStageOutput->pShaderStatistics;
        if (pOptionOutputProgramNvn != nullptr && pOptionOutputProgramNvn->pHullStageOutput != nullptr)
        {
            m_PipelineInfoArray[idxCodeType][idxVariation].hullShaderInfo.scratchMemoryStats.requiredScratchMemorySizePerWarp =
                pOptionOutputProgramNvn->pHullStageOutput->requiredScratchMemorySizePerWarp;
            m_PipelineInfoArray[idxCodeType][idxVariation].hullShaderInfo.scratchMemoryStats.recommendedScratchMemorySizePerWarp =
                pOptionOutputProgramNvn->pHullStageOutput->recommendedScratchMemorySizePerWarp;
        }
    }

    if (pOptionOutputProgramCommon->pDomainStageOutput != nullptr)
    {
        m_PipelineInfoArray[idxCodeType][idxVariation].domainShaderInfo.pSource =
            pOptionOutputProgramCommon->pDomainStageOutput->dump.pValue;
        m_PipelineInfoArray[idxCodeType][idxVariation].domainShaderInfo.pStats =
            pOptionOutputProgramCommon->pDomainStageOutput->pShaderStatistics;
        if (pOptionOutputProgramNvn != nullptr && pOptionOutputProgramNvn->pDomainStageOutput != nullptr)
        {
            m_PipelineInfoArray[idxCodeType][idxVariation].domainShaderInfo.scratchMemoryStats.requiredScratchMemorySizePerWarp =
                pOptionOutputProgramNvn->pDomainStageOutput->requiredScratchMemorySizePerWarp;
            m_PipelineInfoArray[idxCodeType][idxVariation].domainShaderInfo.scratchMemoryStats.recommendedScratchMemorySizePerWarp =
                pOptionOutputProgramNvn->pDomainStageOutput->recommendedScratchMemorySizePerWarp;
        }
    }
}

void ShaderCompilerManager::DumpStatistics(std::ostream* pOs, const ShaderInfo* pShader) const
{
    void* pStats = pShader->pStats;

    if (pStats != nullptr)
    {
        auto& os = *pOs;

        if (m_pConvertArg->pCompileArg->targetLowLevelApiType == nngfxToolShaderCompilerLowLevelApiType_Nvn)
        {
            auto& statistics = *static_cast<const GLSLCperfStatsData*>(pStats);
            if (statistics.magic != GLSLC_PERF_STATS_SECTION_MAGIC_NUMBER)
            {
                return;
            }
            const char* const separator = "------------------------";
            os << separator << "ShaderScratchMemory Information" << separator << "\n";
            os << "requiredScratchMemorySizePerWarp: " << pShader->scratchMemoryStats.requiredScratchMemorySizePerWarp;
            os << "\nrecommendedScratchMemorySizePerWarp: " << pShader->scratchMemoryStats.recommendedScratchMemorySizePerWarp;
            os << "\n\n";

            os << separator << "ShaderStatistics Information" << separator << "\n";
            os << "Latency: " << statistics.latency;
            os << "\nSpillMem";
            os << "\n    NumLmemSpillBytes: " << statistics.spillMem.numLmemSpillBytes;
            os << "\n    NumLmemRefillBytes: " << statistics.spillMem.numLmemRefillBytes;
            os << "\n    NumSmemSpillBytes: " << statistics.spillMem.numSmemSpillBytes;
            os << "\n    NumSmemRefillBytes: " << statistics.spillMem.numSmemRefillBytes;
            os << "\n    Size: " << statistics.spillMem.size;
            os << "\nNonSpillLMem";
            os << "\n    LoadBytes: " << statistics.nonSpillLMem.loadBytes;
            os << "\n    StoreBytes: " << statistics.nonSpillLMem.storeBytes;
            os << "\n    Size: " << statistics.nonSpillLMem.size;
            os << "\nOccupancy: " << statistics.occupancy;
            os << "\nNumDivergentBranches: " << statistics.numDivergentBranches;
            os << "\nAttributeMemUsage: " << statistics.attributeMemUsage;
            os << "\nProgramSize: " << statistics.programSize;
            os << "\nRequiresGlobalLoadUniformEmulation: " <<
                (statistics.requiresGlobalLoadUniformEmulation ? "true" : "false");
            os << "\nThroughputLimiter";
            os << "\n    Issue: " << statistics.throughputLimiter.issue;
            os << "\n    Fp: " << statistics.throughputLimiter.fp;
            os << "\n    Half: " << statistics.throughputLimiter.half;
            os << "\n    Trancedental: " << statistics.throughputLimiter.trancedental;
            os << "\n    Ipa: " << statistics.throughputLimiter.ipa;
            os << "\n    Shared: " << statistics.throughputLimiter.shared;
            os << "\n    ControlFlow: " << statistics.throughputLimiter.controlFlow;
            os << "\n    TexLoadStore: " << statistics.throughputLimiter.texLoadStore;
            os << "\n    Reg: " << statistics.throughputLimiter.reg;
            os << "\n    Warp: " << statistics.throughputLimiter.warp;
            os << "\nLoopData";
            os << "\n    PartiallyUnrolled: " << statistics.loopData.partiallyUnrolled;
            os << "\n    NonUnrolled: " << statistics.loopData.nonUnrolled;
            os << "\n\n";
        }
    }
}

void ShaderCompilerManager::DumpXmlStatistics(pugi::xml_node* pRoot, const ShaderInfo* pShader) const
{
    void* pStats = pShader->pStats;

    if (pStats == nullptr)
    {
        return;
    }

    if (m_pConvertArg->pCompileArg->targetLowLevelApiType == nngfxToolShaderCompilerLowLevelApiType_Nvn)
    {
        auto& statistics = *static_cast<const GLSLCperfStatsData*>(pStats);
        if (statistics.magic != GLSLC_PERF_STATS_SECTION_MAGIC_NUMBER)
        {
            return;
        }

        pugi::xml_node shaderStatistics = pRoot->append_child("GLSLCperfStatsData");
        {
            shaderStatistics.append_child("latency").append_child(pugi::node_pcdata).set_value(std::to_string(statistics.latency).c_str());
            pugi::xml_node spillLMem = shaderStatistics.append_child("spillMem");
            {
                spillLMem.append_child("numLmemSpillBytes").append_child(pugi::node_pcdata).set_value(std::to_string(statistics.spillMem.numLmemSpillBytes).c_str());
                spillLMem.append_child("numLmemRefillBytes").append_child(pugi::node_pcdata).set_value(std::to_string(statistics.spillMem.numLmemRefillBytes).c_str());
                spillLMem.append_child("numSmemSpillBytes").append_child(pugi::node_pcdata).set_value(std::to_string(statistics.spillMem.numSmemSpillBytes).c_str());
                spillLMem.append_child("numSmemRefillBytes").append_child(pugi::node_pcdata).set_value(std::to_string(statistics.spillMem.numSmemRefillBytes).c_str());
                spillLMem.append_child("size").append_child(pugi::node_pcdata).set_value(std::to_string(statistics.spillMem.size).c_str());
            }

            pugi::xml_node nonSpillLMem = shaderStatistics.append_child("nonSpillLMem");
            {
                nonSpillLMem.append_child("loadBytes").append_child(pugi::node_pcdata).set_value(std::to_string(statistics.nonSpillLMem.loadBytes).c_str());
                nonSpillLMem.append_child("storeBytes").append_child(pugi::node_pcdata).set_value(std::to_string(statistics.nonSpillLMem.storeBytes).c_str());
                nonSpillLMem.append_child("size").append_child(pugi::node_pcdata).set_value(std::to_string(statistics.nonSpillLMem.size).c_str());
            }

            shaderStatistics.append_child("occupancy").append_child(pugi::node_pcdata).set_value(std::to_string(statistics.occupancy).c_str());
            shaderStatistics.append_child("numDivergentBranches").append_child(pugi::node_pcdata).set_value(std::to_string(statistics.numDivergentBranches).c_str());
            shaderStatistics.append_child("attributeMemUsage").append_child(pugi::node_pcdata).set_value(std::to_string(statistics.attributeMemUsage).c_str());
            shaderStatistics.append_child("programSize").append_child(pugi::node_pcdata).set_value(std::to_string(statistics.programSize).c_str());
            shaderStatistics.append_child("requiresGlobalLoadUniformEmulation").append_child(pugi::node_pcdata).set_value((statistics.requiresGlobalLoadUniformEmulation ? "true" : "false"));

            pugi::xml_node throughputLimiter = shaderStatistics.append_child("throughputLimiter");
            {
                throughputLimiter.append_child("issue").append_child(pugi::node_pcdata).set_value(std::to_string(statistics.throughputLimiter.issue).c_str());
                throughputLimiter.append_child("fp").append_child(pugi::node_pcdata).set_value(std::to_string(statistics.throughputLimiter.fp).c_str());
                throughputLimiter.append_child("half").append_child(pugi::node_pcdata).set_value(std::to_string(statistics.throughputLimiter.half).c_str());
                throughputLimiter.append_child("trancedental").append_child(pugi::node_pcdata).set_value(std::to_string(statistics.throughputLimiter.trancedental).c_str());
                throughputLimiter.append_child("ipa").append_child(pugi::node_pcdata).set_value(std::to_string(statistics.throughputLimiter.ipa).c_str());
                throughputLimiter.append_child("shared").append_child(pugi::node_pcdata).set_value(std::to_string(statistics.throughputLimiter.shared).c_str());
                throughputLimiter.append_child("controlFlow").append_child(pugi::node_pcdata).set_value(std::to_string(statistics.throughputLimiter.controlFlow).c_str());
                throughputLimiter.append_child("texLoadStore").append_child(pugi::node_pcdata).set_value(std::to_string(statistics.throughputLimiter.texLoadStore).c_str());
                throughputLimiter.append_child("reg").append_child(pugi::node_pcdata).set_value(std::to_string(statistics.throughputLimiter.reg).c_str());
                throughputLimiter.append_child("warp").append_child(pugi::node_pcdata).set_value(std::to_string(statistics.throughputLimiter.warp).c_str());
            }

            pugi::xml_node loopData = shaderStatistics.append_child("loopData");
            {
                loopData.append_child("partiallyUnrolled").append_child(pugi::node_pcdata).set_value(std::to_string(statistics.loopData.partiallyUnrolled).c_str());
                loopData.append_child("nonUnrolled").append_child(pugi::node_pcdata).set_value(std::to_string(statistics.loopData.nonUnrolled).c_str());
            }
        }
    }
}

void VariationValue::CreateVariationValueArg( const std::vector<const std::string*>& preprocessorValueNameArray,
                                             std::vector<VariationConstantValueInfo>& variationConstantValueInfoArray,
                                             ShaderCompilerManager::ShaderType shaderType )
{
    if( shaderType == ShaderCompilerManager::ShaderType_Count )
    {
        THROW_ERROR(ERRCODE_UNKNOWN_ENUM, "Unknown enum value. %d", shaderType);
    }

    static const nngfxToolShaderCompilerVariationValue*
        nngfxToolShaderCompilerVariationValueArg::* s_pVariationValueDefinition[] =
    {
        &nngfxToolShaderCompilerVariationValueArg::pVertexShaderVariationValue,
        &nngfxToolShaderCompilerVariationValueArg::pGeometryShaderVariationValue,
        &nngfxToolShaderCompilerVariationValueArg::pPixelShaderVariationValue,
        &nngfxToolShaderCompilerVariationValueArg::pComputeShaderVariationValue,
        &nngfxToolShaderCompilerVariationValueArg::pHullShaderVariationValue,
        &nngfxToolShaderCompilerVariationValueArg::pDomainShaderVariationValue
    };

    m_VariationValueArg.*s_pVariationValueDefinition[ shaderType ]  = &m_VariationValueArray[ shaderType ];

    nngfxToolShaderCompilerVariationValue* pVariationValue =
        const_cast<nngfxToolShaderCompilerVariationValue*>( m_VariationValueArg.*s_pVariationValueDefinition[ shaderType ]  );

    if( preprocessorValueNameArray.size() )
    {
        m_PreprocessorValueArray[ shaderType ].resize( preprocessorValueNameArray.size() );
        pVariationValue->preprocessorDefinitionValueCount = static_cast<uint32_t>( preprocessorValueNameArray.size() );
        pVariationValue->pPreprocessorDefinitionValueArray = &m_PreprocessorValueArray[ shaderType ][ 0 ];
        nngfxToolString* pToolStr = &m_PreprocessorValueArray[ shaderType ][ 0 ];
        for( auto nameIter = preprocessorValueNameArray.begin(); nameIter != preprocessorValueNameArray.end(); ++nameIter, ++pToolStr )
        {
            pToolStr->pValue = (*nameIter)->data();
            pToolStr->length = static_cast<uint32_t>( (*nameIter)->length() );
        }
    }

    if( variationConstantValueInfoArray.size() )
    {
        m_UniformConstantValueArray[shaderType].resize( variationConstantValueInfoArray.size() );
        pVariationValue->pVariationConstantValueArray = &m_UniformConstantValueArray[shaderType][0];
        pVariationValue->variationConstantValueCount = static_cast<uint32_t>( variationConstantValueInfoArray.size() );
        int idxVariationConstantValueInfo = 0;
        for( auto& variationConstantValueInfo : variationConstantValueInfoArray )
        {
            m_UniformConstantValueArray[shaderType][idxVariationConstantValueInfo].pValue = variationConstantValueInfo.pData;
            m_UniformConstantValueArray[shaderType][idxVariationConstantValueInfo].valueSizeIn32Bit = variationConstantValueInfo.sizeIn32Bit;
            ++idxVariationConstantValueInfo;
        }
    }
}

void VariationDefinition::CreateVariationDefinitionArg( std::vector<const std::string*>& preprocessorDefinitionNameArray,
                                                       std::vector<UniformVariationDefinitionInfo>& uniformConstantDefinitionArray,
                                                       ShaderCompilerManager::ShaderType shaderType )
{
    if( shaderType == ShaderCompilerManager::ShaderType_Count )
    {
        THROW_ERROR(ERRCODE_UNKNOWN_ENUM, "Unknown enum value. %d", shaderType);
    }

    static const nngfxToolShaderCompilerVariationDefinition*
        nngfxToolShaderCompilerVariationDefinitionArg::* s_pVariationDefinition[] =
    {
        &nngfxToolShaderCompilerVariationDefinitionArg::pVertexShaderVariationDefinition,
        &nngfxToolShaderCompilerVariationDefinitionArg::pGeometryShaderVariationDefinition,
        &nngfxToolShaderCompilerVariationDefinitionArg::pPixelShaderVariationDefinition,
        &nngfxToolShaderCompilerVariationDefinitionArg::pComputeShaderVariationDefinition,
        &nngfxToolShaderCompilerVariationDefinitionArg::pHullShaderVariationDefinition,
        &nngfxToolShaderCompilerVariationDefinitionArg::pDomainShaderVariationDefinition
    };

    m_VariationDefinitionArg.*s_pVariationDefinition[ shaderType ] = &m_VariationDefinitionArray[ shaderType ];

    nngfxToolShaderCompilerVariationDefinition* pDefinition = const_cast<nngfxToolShaderCompilerVariationDefinition*>( m_VariationDefinitionArg.*s_pVariationDefinition[ shaderType ] );
    pDefinition = &m_VariationDefinitionArray[ shaderType ];

    if( uniformConstantDefinitionArray.size() )
    {
        m_UniformConsstantDefinitionArray[shaderType].resize( uniformConstantDefinitionArray.size() );
        pDefinition->variationConstantDefinitionCount = static_cast<uint32_t>( uniformConstantDefinitionArray.size() );
        pDefinition->pVariationConstantDefinitionArray = &m_UniformConsstantDefinitionArray[ shaderType ][ 0 ];
        pDefinition->variationConstantBufferName.length = static_cast<uint32_t>( uniformConstantDefinitionArray[0].pUniformBufferName->length() );
        pDefinition->variationConstantBufferName.pValue = uniformConstantDefinitionArray[0].pUniformBufferName->c_str();
        int idxUniformVariation = 0;
        nngfxToolShaderCompilerVariationConstantDefinition* pUniformVariationDefinition = &m_UniformConsstantDefinitionArray[shaderType][ 0 ];
        for( auto nameIter = uniformConstantDefinitionArray.begin(); nameIter != uniformConstantDefinitionArray.end(); ++nameIter, ++pUniformVariationDefinition, ++idxUniformVariation )
        {
            pUniformVariationDefinition->indexInVariationConstantValueArray = idxUniformVariation;
            pUniformVariationDefinition->arrayLength = nameIter->arrayLength;
            pUniformVariationDefinition->vectorComponents = nameIter->vectorComponents;
            pUniformVariationDefinition->matrix.column = nameIter->matrix.column;
            pUniformVariationDefinition->matrix.row = nameIter->matrix.row;
            pUniformVariationDefinition->type = static_cast<int16_t>( nameIter->type );
            pUniformVariationDefinition->name.pValue = nameIter->pName->c_str();
            pUniformVariationDefinition->name.length = static_cast<uint32_t>( nameIter->pName->length() );
        }
    }

    if( preprocessorDefinitionNameArray.size() )
    {
        m_PreprocessorDefinitionArray[ shaderType ].resize( preprocessorDefinitionNameArray.size() );
        pDefinition->preprocessorDefinitionDefinitionCount = static_cast<uint32_t>( preprocessorDefinitionNameArray.size() );
        pDefinition->pPreprocessorDefinitionDefinitionArray = &m_PreprocessorDefinitionArray[ shaderType ][ 0 ];
        int definitionIdx = 0;
        nngfxToolShaderCompilerPreprocessorDefinitionDefinition* pDefinitoinDefinition = &m_PreprocessorDefinitionArray[ shaderType ][ 0 ];
        for( auto nameIter = preprocessorDefinitionNameArray.begin(); nameIter != preprocessorDefinitionNameArray.end(); ++nameIter, ++pDefinitoinDefinition, ++definitionIdx )
        {
            pDefinitoinDefinition->indexInPreprocessorDefinitionValueArray = definitionIdx;	// vector に入っている順番に index を振る。value definition もこの順番で詰める必要がある。
            pDefinitoinDefinition->name.pValue = (*nameIter)->data();
            pDefinitoinDefinition->name.length = static_cast<uint32_t>( (*nameIter)->length() );
        }
    }
}
