﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <BinDictionary.h>

namespace nn {
namespace g3dTool {


void BinDictionary::Build(std::shared_ptr<Context> pCtx, size_t dicSize)
{
    pCtx->blocks.push_back(this);
    m_NameArray.resize(dicSize);
}

void BinDictionary::CalculateSize()
{
    // 要素が無い場合は辞書自体を作らない。
    auto count = m_NameArray.size();
    if (count > 0)
    {
        size_t size = sizeof( nn::util::ResDicData );
        size += sizeof( nn::util::ResDicData::Entry ) * count;
        SetBlockSize(Context::MemBlockType_Main, size);
    }
}

void BinDictionary::Convert( std::shared_ptr<Context> pCtx )
{
    auto count = m_NameArray.size();
    if (count == 0)
    {
        return;
    }

    nn::util::ResDicData* pDicData = GetPtr<nn::util::ResDicData>( pCtx->GetMemBlockPtr( Context::MemBlockType_Main ) );
    memset( pDicData, 0, GetBlockSize(Context::MemBlockType_Main) );
    pDicData->count	= static_cast<int32_t>( count );

    // ResDic では先頭に空文字を入れる必要があります。
    bool result = false;
    {
        nn::util::ResDicData::Entry* entry = pDicData->entries;
        pCtx->LinkStr( &entry->pKey, nn::util::string_view( "" ) );
        entry->pKey.Relocate( pCtx->GetBasePtr() );
        ++entry;

        for (auto iter = m_NameArray.cbegin(); iter != m_NameArray.cend(); ++iter, ++entry )
        {
            const std::string* pName = *iter;
            pCtx->LinkStr( &entry->pKey, nn::util::string_view( pName->c_str() ) );
            entry->pKey.Relocate( pCtx->GetBasePtr() );
        }

        nn::util::ResDic* pDic = GetPtr<nn::util::ResDic>( pCtx->GetMemBlockPtr( Context::MemBlockType_Main ) );
        result = pDic->Build();
    }

    // オフセットに戻す
    for ( auto entryIdx = 0; entryIdx < static_cast< int >( m_NameArray.size() ) + 1; entryIdx++ )
    {
        pDicData->entries[ entryIdx ].pKey.Unrelocate( pCtx->GetBasePtr() );
    }

    if ( !result )
    {
        THROW_ERROR(
            ERRCODE_INCONSISTENT_DICTIONARY,
            "Can not build dictionary. Some node names may be duplicated.\n%s\n%d\n",
            __FILE__,
            __LINE__
            );
    }
}

}
}
