﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <shdrdefs.h>

#pragma warning(push)
#pragma warning(disable:4100)
#pragma warning(disable:4127)
#pragma warning(disable:4245)
#pragma warning(disable:4267)
#pragma warning(disable:4302)
#pragma warning(disable:4311)
#pragma warning(disable:4312)
#pragma warning(disable:4458)
#pragma warning(disable:4459)
#pragma warning(disable:4477)
#pragma warning(disable:4512)
#pragma warning(disable:4702)
#pragma warning(disable:4706)

// wave settings
#ifndef BOOST_WAVE_SUPPORT_THREADING
#define BOOST_WAVE_SUPPORT_THREADING 1
#endif

///////////////////////////////////////////////////////////////////////////////
//  Include Wave itself
#include <boost/wave.hpp>

///////////////////////////////////////////////////////////////////////////////
// Include the lexer stuff
#include <boost/wave/cpplexer/cpp_lex_token.hpp>    // token class
#include <boost/wave/cpplexer/cpp_lex_iterator.hpp> // lexer class

///////////////////////////////////////////////////////////////////////////////

#include <boost/wave/preprocessing_hooks.hpp>

#pragma warning(pop)

#include <ShdrAnnotation.h>
#include <g3dif/ShaderDefinition.h>

#define NW_MODIFY_FILE ".nw_modify"


namespace nw { namespace g3d { namespace tool { namespace g3dif {

class elem_shading_model;

}}}}

namespace nn { namespace g3dTool {

class Extractor : public boost::wave::context_policies::default_preprocessing_hooks
{
public:
    //  This token type is one of the central types used throughout the library,
    //  because it is a template parameter to some of the public classes and
    //  instances of this type are returned from the iterators.
    typedef boost::wave::cpplexer::lex_token<> token_type;

    //  The template cpplexer::lex_iterator<> is the lexer type to
    //  to use as the token source for the preprocessing engine. It is
    //  parametrized with the token type.
    typedef boost::wave::cpplexer::lex_iterator<token_type> lex_iterator_type;

    //  This is the resulting context type to use. The first template parameter
    //  should match the iterator type to be used during construction of the
    //  corresponding context object.
    typedef boost::wave::context<
        const char*,
        lex_iterator_type,
        Extractor,//boost::wave::iteration_context_policies::load_file_to_string,
        Extractor>
        context_type;

    struct SymbolDB
    {
        void Clear();

        std::string path[ShaderStage_StageCount];
        std::vector<SymbolOption> options;
        std::vector<SymbolAttrib> attribs;
        std::vector<SymbolSampler> samplers;
        std::vector<SymbolBlock> blocks;
        std::vector<SymbolSsboBlock> ssboBlocks;
        std::vector<SymbolRenderInfo> renderInfos;
        std::vector<SymbolTextBlock> textBlocks;
        std::vector<SymbolInterleave> interleaves;
        std::vector<SymbolGroup> groups;
        std::vector<SymbolStreamout> streamouts;
        std::vector<SymbolRevision> revisions;
        std::vector<SymbolLayout> layouts;
        std::vector<SymbolPage> pages;
    };

    Extractor();

    // 設定をクリアします。
    void ClearSettings();

    // 結果をクリアします。
    void ClearResults();

    // 固定マクロを追加します。
    bool AddFixedMacro(const StringRef& macro);
    bool AddFixedMacro(const StringRef& name, const StringRef& value);

    // マクロをクリアします。
    void ClearMacro();

    void AddIncludePath(const StringRef& path);
    void AddForceInclude(const StringRef& path);
    void AddGlslVersion(const StringRef& path);
    void SetCodePage(uint32_t codePage);
    uint32_t GetCodePage() const;
    FileDB* GetFileDB();

    // アノテーションの抽出を行います。
    void Extract(const StringRef& fullpath, ShaderStage stage, FileDB* fileDB);
    void Merge(bool unifiedAnnotation);
    void WriteLog(std::ostream& stream) const;

    // シェーダー定義を中間ファイルに書き込みます。
    void WriteDefinition(nw::g3d::tool::g3dif::elem_shading_model& el_shading_model, FileDB* fileDB) const;

protected:

    struct AnnotDB
    {
        void Clear();

        std::vector<AnnotOption> options;
        std::vector<AnnotAttrib> attribs;
        std::vector<AnnotSampler> samplers;
        std::vector<AnnotBlock> blocks;
        std::vector<AnnotSsboBlock> ssboBlocks;
        std::vector<AnnotRenderInfo> renderInfos;
        std::vector<AnnotTextBlock> textBlocks;
        std::vector<AnnotInterleave> interleaves;
        std::vector<AnnotGroup> groups;
        std::vector<AnnotStreamout> streamouts;
        std::vector<AnnotRevision> revisions;
        std::vector<AnnotLayout> layouts;
        std::vector<AnnotPage> pages;
    };

    static StringRef LoadFile(const StringRef& filename, FileDB* fileDB, uint32_t codePage);

    void ExtractOption(const char* str);
    void BeginBlock(int tokenIndex);
    void BeginSsboBlock(int tokenIndex);
    void ExtractLayout(int tokenIndex);
    void EndBlock();
    void EndSsboBlock();
    void ExtractBlock(const char* annotation);
    void ExtractSsboBlock(const char* annotation);
    void ExtractVar(int tokenIndex, const char* annotation);
    void ExtractAny(const char* annotation);

    void CheckAnnotationSyntax(const char* annotation);

    // フックからのコールバック
    void OnDefinedMacro(const token_type& token);
    void OnSkippedComment(const token_type& token);

private:
    struct State;
    std::shared_ptr<State> m_pState; // context 構築時にコピーされるので変数は State 内に持つ。

    StringRef ModifySource(StringRef src, std::string path);
    void ExtractInternal(context_type* pCtx, ShaderStage stage);
    void InsertTextblockAutoId(std::string* annotation, int* id);
public:
    // インプットポリシー

    ///////////////////////////////////////////////////////////////////////////
    //
    //  load_file_to_string
    //
    //      Loads a file into a string and returns the iterators pointing to
    //      the beginning and the end of the loaded string.
    //
    ///////////////////////////////////////////////////////////////////////////
    template <typename IterContextT>
    class inner
    {
    public:
        template <typename PositionT>
        static void init_iterators(IterContextT &iter_ctx,
            PositionT const &act_pos, boost::wave::language_support language)
        {
            // ソースファイルの集計、重複ロードの回避、文字コード変換などのため Extractor 経由。
            typedef typename IterContextT::iterator_type iterator_type;
            Extractor& extractor = iter_ctx.ctx.get_hooks();
            uint32_t codePage = extractor.GetCodePage();
            FileDB* fileDB = extractor.GetFileDB();

            const StringRef src = extractor.LoadFile(iter_ctx.filename.c_str(), fileDB, codePage);
            if (src.Empty())
            {
                BOOST_WAVE_THROW_CTX(iter_ctx.ctx, preprocess_exception,
                    bad_include_file, iter_ctx.filename.c_str(), act_pos);
            }

            iter_ctx.first = iterator_type(src.Begin(), src.End(),
                PositionT(iter_ctx.filename), language);
            iter_ctx.last = iterator_type();
        }
    };

    // フック

    ///////////////////////////////////////////////////////////////////////////
    //
    //  The function 'defined_macro' is called, whenever a macro was defined
    //  successfully.
    //
    //  The parameter 'ctx' is a reference to the context object used for
    //  instantiating the preprocessing iterators by the user.
    //
    //  The parameter 'name' is a reference to the token holding the macro name.
    //
    //  The parameter 'is_functionlike' is set to true, whenever the newly
    //  defined macro is defined as a function like macro.
    //
    //  The parameter 'parameters' holds the parameter tokens for the macro
    //  definition. If the macro has no parameters or if it is a object like
    //  macro, then this container is empty.
    //
    //  The parameter 'definition' contains the token sequence given as the
    //  replacement sequence (definition part) of the newly defined macro.
    //
    //  The parameter 'is_predefined' is set to true for all macros predefined
    //  during the initialization phase of the library.
    //
    ///////////////////////////////////////////////////////////////////////////
    template <
        typename ContextT, typename TokenT, typename ParametersT,
        typename DefinitionT
    >
    void
    defined_macro(ContextT const& /*ctx*/, TokenT const& macro_name,
    bool is_functionlike, ParametersT const& /*parameters*/,
    DefinitionT const& /*definition*/, bool /*is_predefined*/)
    {
        if (!is_functionlike)
        {
            OnDefinedMacro(macro_name);
        }
    }

    ///////////////////////////////////////////////////////////////////////////
    //
    //  The function 'skipped_token' is called, whenever a token is about to be
    //  skipped due to a false preprocessor condition (code fragments to be
    //  skipped inside the not evaluated conditional #if/#else/#endif branches).
    //
    //  The parameter 'ctx' is a reference to the context object used for
    //  instantiating the preprocessing iterators by the user.
    //
    //  The parameter 'token' refers to the token to be skipped.
    //
    ///////////////////////////////////////////////////////////////////////////
    template <typename ContextT, typename TokenT>
    void
    skipped_token(ContextT const& /*ctx*/, TokenT const& token)
    {
        if (boost::wave::T_CCOMMENT == token || boost::wave::T_CPPCOMMENT == token)
        {
            OnSkippedComment(token);
        }
    }

    ///////////////////////////////////////////////////////////////////////////
    //
    //  The function 'found_unknown_directive' is called, whenever an unknown
    //  preprocessor directive was encountered.
    //
    //  The parameter 'ctx' is a reference to the context object used for
    //  instantiating the preprocessing iterators by the user.
    //
    //  The parameter 'line' holds the tokens of the entire source line
    //  containing the unknown directive.
    //
    //  The parameter 'pending' may be used to push tokens back into the input
    //  stream, which are to be used as the replacement text for the whole
    //  line containing the unknown directive.
    //
    //  The return value defines, whether the given expression has been
    //  properly interpreted by the hook function or not. If this function
    //  returns 'false', the library will raise an 'ill_formed_directive'
    //  preprocess_exception. Otherwise the tokens pushed back into 'pending'
    //  are passed on to the user program.
    //
    ///////////////////////////////////////////////////////////////////////////
    template <typename ContextT, typename ContainerT>
    bool
    found_unknown_directive(ContextT const& /*ctx*/, ContainerT const& line,
        ContainerT& pending)
    {
#if 0
        // 変数の解析で邪魔にならないようにすべて消す。
        return true;
#else
        auto iter = line.begin();
        boost::wave::token_id id = boost::wave::util::impl::skip_whitespace(iter, line.end());

        if (id != boost::wave::T_IDENTIFIER)
        {
            return false; // nothing we could do
        }

        if (iter->get_value() == "version" || iter->get_value() == "extension")
        {
            // handle #version and #extension directives
            std::copy(line.begin(), line.end(), std::back_inserter(pending));
            return true;
        }
        return false; // unknown directive
#endif
    }

    template <typename ContextT>
    bool locate_include_file(ContextT& /*ctx*/, std::string& file_path, bool /*is_system*/,
        char const* /*current_name*/, std::string& /*dir_path*/, std::string& native_name)
    {
        for (auto include_path : m_pState->includePaths)
        {
            native_name = nw::g3d::tool::util::Path::Combine(include_path, file_path);
            if (nw::g3d::tool::util::Path::FileExists(native_name))
            {
                return true;
            }
        }
        return false;
    }

    ///////////////////////////////////////////////////////////////////////////
    //
    //  The function 'throw_exception' will be called by the library whenever a
    //  preprocessing exception occurs.
    //
    //  The parameter 'ctx' is a reference to the context object used for
    //  instantiating the preprocessing iterators by the user.
    //
    //  The parameter 'e' is the exception object containing detailed error
    //  information.
    //
    //  The default behavior is to call the function boost::throw_exception.
    //
    ///////////////////////////////////////////////////////////////////////////
    template <typename ContextT, typename ExceptionT>
    void
    throw_exception(ContextT const& /*ctx*/, ExceptionT const& e)
    {
        if (e.get_errorcode() == boost::wave::preprocess_exception::macro_redefinition)
        {
            return;
        }
        boost::throw_exception(e);
    }
};

} // namespace g3dTool
} // namespace nn
