﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <shdrdefs.h>
#include <util/UtilXMLParser.h>
#include <ShdrFile.h>
#include <Context.h>

namespace nn { namespace g3dTool {


class Converter
{
public:
    enum ConvertBehaviorFlag
    {
        BEHAVIOR_NONE                               = 0x0,
        BEHAVIOR_UNIFIED_ANNOTATION                 = 0x1 << 0,
        BEHAVIOR_AUTO_EXTRACT                       = 0x1 << 1,

        BEHAVIOR_DEFAULT =
            BEHAVIOR_NONE
    };

    enum CacheDirectoryOptionFlag
    {
        CacheDirectoryOptionFlag_Undefined = 0x0,
        CacheDirectoryOptionFlag_Both      = 0x1,
        CacheDirectoryOptionFlag_Separate  = 0x2
    };

    Converter() {}

    uint32_t GetCvtrVersion() const
    {
        return NN_G3D_SHADER_CVTR_VERSION;
    }

    uint32_t GetCvtrBinaryVersion() const
    {
        return NN_UTIL_CREATE_BINVERSION( NN_G3D_SHADER_BINARY_VERSION_MAJOR, NN_G3D_SHADER_BINARY_VERSION_MINOR, NN_G3D_SHADER_BINARY_VERSION_MICRO );
    }

    // コンバータ設定
    bool Initialize(std::wstring path);
    bool Clear();
    bool Shutdown();
    bool SetOptions(const wchar_t* options[]);
    bool SetOptions(int argc, const wchar_t* argv[]);
    bool AddFile(void* pData, size_t dataSize, const wchar_t* path[3]);
    bool SetLogStream(const nngfxToolSetLogStreamArg* pArg)
    {
        ShaderCompilerManager::SetLogStream(pArg);
        return true;
    }

    template<typename T>
    void CheckVersion(T pElem, int major, int minor, int micro, const char* elementName);

    // 個別オプション設定
    void SetCodeTypeOption(std::string value);

    // bfsha 作成
    size_t CalculateArchiveSize();
    bool Convert(void* pBuffer, size_t size);
    bool SwapEndian(void* pBuffer, size_t size);

    // fsd 作成
    bool Make();
    size_t CalculateDefinitionSize(const wchar_t* fullpath);
    bool Write(void* pBuffer, size_t size);

    // リロケーションテーブル関連
    void* GetBasePtr( void ) const
    {
        return m_Ctx->GetBasePtr();
    }

    void* GetRelocationTablePtr( void ) const
    {
        return m_Ctx->GetRelocationTblPtr();
    }

    size_t GetBaseSize( void ) const
    {
        return m_Ctx->GetBaseSize();
    }

    size_t GetRelocationTableSize( void ) const
    {
        return m_Ctx->GetRelocationTblSize();
    }

private:
    std::string m_FscRootPathUtf;
    std::string m_FsdRootPathUtf;
    bit32 m_Check;
    bit32 m_Behavior;
    ConvertMode m_ConvertMode;

    // fsd を一時的に格納しておく
    std::ostringstream m_ImOutStream;
    size_t m_ImSize;

    std::shared_ptr<ShdrFile> m_File;
    std::shared_ptr<Context> m_Ctx;

    std::vector<std::string> m_Macros;

    bool m_SkipConvert;
};

} // namespace g3dTool
} // namespace nn
