﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/g3d/g3d_ResShader.h>
#include <g3dif/ShaderDefinition.h>
#include <shdrdefs.h>
#include <BinaryBlock.h>
#include <BinDictionary.h>
#include <BinProgram.h>

namespace nn { namespace g3dTool {

class ShdrFile;
class BinShader;
typedef std::vector<const nw::g3d::tool::g3dif::elem_sampler_var*> sampler_array;
typedef std::vector<const nw::g3d::tool::g3dif::elem_block_var*> block_array;
typedef std::vector<const nw::g3d::tool::g3dif::elem_ssbo_block_var*> ssbo_block_array;
typedef std::vector<const nw::g3d::tool::g3dif::elem_option_var*> option_array;

//! @brief シェーダーデータのバイナリ化を行うクラスです。
class BinShader : public BinaryBlock
{
public:
    BinShader()
        : BinaryBlock()
        , m_pElem(nullptr)
        , m_DicStaticOption()
        , m_DicDynamicOption()
        , m_DicAttrib()
        , m_DicSampler()
        , m_DicBlock()
        , m_ProgramArray()
        , m_ActiveAttribCount(0)
        , m_DefaultMaterialSize(0)
        , m_DefaultOptionSize(0)
        , m_AttribLocationTable()
        , m_BlockArray()
        , m_SamplerArray()
        , m_StaticOptionArray()
        , m_DynamicOptionArray()
        , m_IsForceVariation(false)
        , m_IsShaderSource(false)
        , m_IsShaderBinary(false)
        , m_IsShaderIr(false)
        , m_ShaderFileChunk()
    {
    }

    void Build(std::shared_ptr<Context> pCtx, const nw::g3d::tool::g3dif::elem_shading_model& elem);
    virtual void CalculateSize();
    virtual void CalculateOffset( std::shared_ptr<Context> pCtx );
    virtual void Convert( std::shared_ptr<Context> pCtx );

    void SetShaderContext(Context::ShaderContext* shaderContext)
    {
        m_ShaderContext = shaderContext;
        m_ShaderContext->m_pBinShader = this;
    }

    void SetShaderArchive(const ShdrFile* file)
    {
        m_pFile = file;
    }

    void SetForceVariation(bool value) { m_IsForceVariation = value; }

    int GetAttribLocation(int idxAttrib) const { return m_AttribLocationTable[idxAttrib]; }

    const BinProgram* GetProgram(int index) const { return &m_ProgramArray[index]; }

    nn::gfx::ResShaderFileData* GetResShaderFileDataPtr( std::shared_ptr<Context> pCtx )
    {
        return static_cast<nn::gfx::ResShaderFileData*>( GetPtr(pCtx, Context::MemBlockType_ShaderFile, m_ShaderFileChunk[ShaderFileChunk_ShaderFileData].offset) );
    }

private:
    void ConvertOption(nn::g3d::ResShadingModelData &shadingModel, std::shared_ptr<Context> pCtx);
    void ConvertAttrib(nn::g3d::ResShadingModelData &shadingModel, std::shared_ptr<Context> pCtx);
    void ConvertSampler(nn::g3d::ResShadingModelData &shadingModel, std::shared_ptr<Context> pCtx);
    void ConvertBlock(nn::g3d::ResShadingModelData &shadingModel, std::shared_ptr<Context> pCtx);
    void ConvertUniform(nn::g3d::ResShadingModelData &shadingModel, std::shared_ptr<Context> pCtx);

    const nw::g3d::tool::g3dif::elem_shading_model* m_pElem;
    const ShdrFile* m_pFile;

    BinDictionary m_DicStaticOption;
    BinDictionary m_DicDynamicOption;
    BinDictionary m_DicAttrib;
    BinDictionary m_DicSampler;
    BinDictionary m_DicBlock;
    BinDictionary m_DicSsboBlock;
    std::vector<BinDictionary> m_DicChoices;
    std::vector<BinDictionary> m_DicUniforms;
    std::vector<BinDictionary> m_DicSsboUniforms;
    std::vector<BinProgram> m_ProgramArray;

    // 出来る限りこちらは使わず、m_ProgramArray から引いてくる。
    Context::ShaderContext* m_ShaderContext;

    int m_ActiveAttribCount;
    size_t m_DefaultMaterialSize;	// material ubのサイズ
    size_t m_DefaultOptionSize;

    bool m_IsForceVariation;

    std::vector<int> m_AttribLocationTable;

    block_array m_BlockArray;
    ssbo_block_array m_SsboBlockArray;
    sampler_array m_SamplerArray;
    option_array m_StaticOptionArray;
    option_array m_DynamicOptionArray;

    bool m_IsShaderSource;
    bool m_IsShaderBinary;
    bool m_IsShaderIr;

    enum ChunkType
    {
        //SHADER,	// 親から渡します。
        ChunkType_Option,
        ChunkType_Attribute,
        ChunkType_Sampler,
        ChunkType_Block,
        ChunkType_SsboBlock,
        ChunkType_Uniform,
        ChunkType_Ssbo,
        ChunkType_Program,	//!< 子の ResShaderProgram に渡します。
        ChunkType_Key,
        ChunkType_DefaultValue,	//!< materialのデフォルト値、materialはssbo, uboのどちからしか存在しないので1チャンクだけ存在します。
        ChunkType_ChoiceValue,
        ChunkType_ShaderInfo,
        ChunkType_ChunkCount
    };

    enum ShaderFileChunk
    {
        ShaderFileChunk_ShaderFileData,
        ShaderFileChunk_Padding,
        ShaderFileChunk_ChunkCount
    };

    Chunk m_Chunk[ChunkType_ChunkCount];
    Chunk m_ShaderFileChunk[ ShaderFileChunk_ChunkCount ];
};

} // namespace g3dTool
} // namespace nn
