﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdlib>
#include <cstdio>
#include <clocale>
#include <vector>
#include <algorithm>
#include <fstream>
#include <sstream>

#include <nn/util/util_BinaryFormat.h>
#include <nn/gfxTool/gfxTool_CommandLineParser.h>

// shdrcvtr 名前空間には依存しない。
#include <util/types.h>
#include <util/UtilError.h>
#include <util/UtilPrint.h>
#include <util/UtilException.h>
#include <util/UtilCmdArgs.h>
#include <util/UtilBuffer.h>
#include <util/UtilGLContext.h>
#include <util/UtilTime.h>
#include <util/UtilString.h>
#include "shdrlib.h"

#include "CommandLineParserWrapper.h"

#ifndef WIN32_LEAN_AND_MEAN
#define WIN32_LEAN_AND_MEAN
#endif
#ifndef NOMINMAX
#define NOMINMAX
#endif
#include <Windows.h>
#include <crtdbg.h>

#define CVTR_DLL L"3dShaderConverter.dll"
#define CVTR_DLL32 L"x86\\3dShaderConverter.dll"
#define PROC_ADDRESS(handle, name, type)                                                               \
auto name = reinterpret_cast<type>(GetProcAddress(handle, #name));                                     \
    if (name == nullptr)                                                                               \
    {                                                                                                  \
        PRINT_ERROR_LOG("Not found function %s in %hs.", L#name, util::GetModuleName(handle).c_str()); \
        return EXIT_FAILURE;                                                                           \
    }                                                                                                  \

using namespace nw::g3d::tool;

void se_translator_function(unsigned int /*code*/, struct _EXCEPTION_POINTERS* ep)
{
    throw ep; //標準C++の例外を発生させる。
}

class FindOption
{
public:
    FindOption(std::wstring name)
        : m_Name(name)
    {}

    bool operator()(const CmdOption& opt)
    {
        return opt.name == m_Name;
    }

private:
    std::wstring m_Name;
};

bool FindFsc( const std::vector<CmdInput>& inputs )
{
    for (auto iter = inputs.cbegin(); iter != inputs.cend(); ++iter)
    {
        if (iter->ext == INPUT_CONFIG_ASCII_FILE_EXT)
        {
            return true;
        }
    }
    return false;
}

int Convert(CmdArgs& args, HINSTANCE hDLL);

int wmain(int argc, const wchar_t* argv[])
{
    util::Tick startTime = util::Tick::GetSystemCurrent();

    _CrtSetDbgFlag(_CRTDBG_ALLOC_MEM_DF | _CRTDBG_LEAK_CHECK_DF);
    _CrtSetBreakAlloc(-1);

    // gfxtool によるロガーの初期化
    nn::gfxTool::Logger::SetDefaultStream();

    setlocale(LC_CTYPE, "");

    // 構造化例外の変換関数。
    _set_se_translator(se_translator_function); // このスレッドでのみ有効。
    int result = EXIT_SUCCESS;
    try
    {
#if defined(_M_IX86)
        HINSTANCE hDLL = LoadLibrary(CVTR_DLL32);
#else
        HINSTANCE hDLL = LoadLibrary(CVTR_DLL);
#endif
        if (hDLL == nullptr)
        {
            PRINT_ERROR_LOG("failed to load dll.");
            return EXIT_FAILURE;
        }
        struct Deleter
        {
            HINSTANCE hInstance;
            Deleter(HINSTANCE hInstance) : hInstance(hInstance) {}
            ~Deleter()
            {
                std::string moduleName = util::GetModuleName(hInstance);
                BOOL ret = FreeLibrary(hInstance);
                if (!ret)
                {
                    PRINT_ERROR_LOG("failed to release dll ( %hs ).", moduleName.c_str());
                }
            }
        } deleter(hDLL);

        // 引数の解析
        nn::g3dTool::CommandLineParserWrapper	parserWrapper( argc, argv );
        std::shared_ptr< CmdArgs > pCmdArgs = parserWrapper.GetCmdArgs();

        CmdArgs& args = *pCmdArgs;

        // サイレントモードの設定
        auto silentOpt = find_if(args.options.begin(), args.options.end(), FindOption(OPT_SILENT_FULL));
        if (silentOpt != args.options.end())
        {
            nw::g3d::tool::LogConfig::s_Level = nw::g3d::tool::LogConfig::Silent;
        }

        PRINT_LOG("Convert start.");

        // バージョン取得
        PROC_ADDRESS(hDLL, nng3dToolShaderConverterGetConverterVersion, util::u32(*)());
        uint32_t version = nng3dToolShaderConverterGetConverterVersion();

        PRINT_LOG("Converter version: %d.%d.%d.%d",
            version & 0xFF, (version >> 8) & 0xFF, (version >> 16) & 0xFF, (version >> 24) & 0xFF);

        PROC_ADDRESS(hDLL, nng3dToolShaderConverterGetBinaryVersion, util::u32 (*)());
        version = nng3dToolShaderConverterGetBinaryVersion();
        nn::util::BinVersion binVersion;
        binVersion.SetPacked( version );
        if( version != 0 )
        {
            PRINT_LOG( "Binary version: %d.%d.%d", binVersion.major, binVersion.minor, binVersion.micro );
        }

        // バージョンを表示して終了する。
        auto vOpt = find_if(args.options.begin(), args.options.end(), FindOption(OPT_VERSION));
        auto versionOpt = find_if(args.options.begin(), args.options.end(), FindOption(OPT_VERSION_FULL));
        if (vOpt != args.options.end() || versionOpt != args.options.end())
        {
            return EXIT_SUCCESS;
        }

        auto helpOpt = find_if(args.options.begin(), args.options.end(), FindOption(OPT_HELP_FULL));
        if (helpOpt != args.options.end())
        {
            const char* toolName = "3dShaderConverter";
            parserWrapper.PrintHelp( toolName, nullptr );
            return EXIT_SUCCESS;
        }

        // 入力ファイルが存在しない場合はエラーで終了する。
        if(args.inputs.size() == 0 && find_if(args.options.begin(), args.options.end(), FindOption(OPT_MERGE_SHADER_ARCHIVE_FILE_FULL)) == args.options.end())
        {
            PRINT_ERROR_LOG("Error: No input files. Need to input fsc or fsd file.\n");
            return EXIT_FAILURE;
        }

        result = Convert(args, hDLL);
    }
    CATCH_ERROR_ALL(EXIT_FAILURE)

    if (result == EXIT_SUCCESS)
    {
        // 正常終了した場合のみログ出力
        util::Tick endTime = util::Tick::GetSystemCurrent();
        util::TimeSpan convertSpan = endTime - startTime;

        util::s32 sec = static_cast<util::s32>(convertSpan.GetSeconds());
        util::s32 ms = static_cast<util::s32>(convertSpan.GetMilliSeconds());

        PRINT_LOG("Convert finish: %ld.%ld sec.", sec, ms - (sec * 1000));
    }

    return result;
}

int Convert(CmdArgs& args, HINSTANCE hDLL)
{
    // DLL から関数の読み込み。
    PROC_ADDRESS(hDLL, nng3dToolShaderConverterInitialize, bool (*)());
    PROC_ADDRESS(hDLL, nng3dToolShaderConverterClear, bool (*)());
    PROC_ADDRESS(hDLL, nng3dToolShaderConverterShutdown, bool (*)());
    PROC_ADDRESS(hDLL, nng3dToolShaderConverterSetOptions, bool (*)(const wchar_t*[]));
    PROC_ADDRESS(hDLL, nng3dToolShaderConverterAddFile, bool (*)(const void*, size_t, const wchar_t*[], const wchar_t*[]));
    PROC_ADDRESS(hDLL, nng3dToolShaderConverterCalculateArchiveSize, size_t (*)());
    PROC_ADDRESS(hDLL, nng3dToolShaderConverterConvert, bool (*)(const void*, size_t));
    PROC_ADDRESS(hDLL, nng3dToolShaderConverterSwapEndian, bool (*)(void*, size_t));
    PROC_ADDRESS(hDLL, nng3dToolShaderConverterMake, bool (*)());
    PROC_ADDRESS(hDLL, nng3dToolShaderConverterCalculateDefinitionSize, size_t (*)(const wchar_t*));
    PROC_ADDRESS(hDLL, nng3dToolShaderConverterWrite, bool (*)(const void*, size_t));

    bool success = true;

    // 初期化
    {
        success = nng3dToolShaderConverterInitialize();
        if (!success)
        {
            PRINT_ERROR_LOG("Init failed.");
            return EXIT_FAILURE;
        }
    }

    auto conv = [](const std::vector<CmdOption>& options) -> const wchar_t**
    {
        auto count = options.size();
        const wchar_t** optionArray = new const wchar_t*[count * 3 + 1];
        int index = 0;
        for (auto iter = options.cbegin(); iter != options.cend(); ++iter)
        {
            optionArray[index++] = iter->name.c_str();
            optionArray[index++] = iter->value.c_str();
        }
        optionArray[index] = nullptr;
        return optionArray;
    };

    // グローバルオプション。
    std::wstring fsdbOutputPath; // 定義出力パス。
    std::wstring bfshaOutputPath; // バイナリ出力パス
    PRINT_LOG(LOG_LEVEL_0(L"Global Options"));

    // TODO: nw 版のサポートを切った段階で SetOptions(int argc, const wchar_t* argv[]) に移行する。その時にソースも移動する。
    // 出力パスの取得。
    int outputIndex	= CmdOption::FindIndex(args.options, OPT_OUTPUT_FILE_NAME_FULL);

    if( outputIndex < 0 )
    {
        if (args.inputs.size() == 1 && CmdOption::FindIndex(args.options, OPT_MERGE_SHADER_ARCHIVE_FILE_FULL) < 0)
        {
            // 入力ファイルが1つの場合に限り出力ファイルの指定を省略できる。
            // 入力ファイルの拡張子が fsc の時は fsd を生成、fsd の時は bfsha を生成する。
            const CmdInput& input = args.inputs.front();
            auto lenExt = input.ext.size();
            const std::wstring& fullpath = input.fullpath;
            wchar_t fullpathBuf[_MAX_PATH];

            // 入力が fsc の時は fsd を出力する
            if (L".fsdb" != input.ext && L".fsda" != input.ext)
            {
                fsdbOutputPath.assign(fullpath.substr(0, fullpath.size() - lenExt));
                fsdbOutputPath	+= OUTPUT_DEFINITION_BINARY_FILE_EXT;
                if(_wfullpath(fullpathBuf, fsdbOutputPath.c_str(), _MAX_PATH))
                {
                    fsdbOutputPath.assign(fullpathBuf);
                }
                else
                {
                    PRINT_ERROR_LOG("Output definition path is too long.");
                    return EXIT_FAILURE;
                }
                std::wstringstream dss;
                dss << OPT_OUTPUT_FILE_NAME_FULL << L"=";
                dss << fsdbOutputPath;
                args.options.push_back(CmdOption(dss.str().c_str()));
            }
            else
            {
                // 入力が fsd の時の処理
                bfshaOutputPath.assign(fullpath.substr(0, fullpath.size() - lenExt));
                bfshaOutputPath	+= OUTPUT_BINARY_FILE_EXT;
                if(_wfullpath(fullpathBuf, bfshaOutputPath.c_str(), _MAX_PATH))
                {
                    bfshaOutputPath.assign(fullpathBuf);
                }
                else
                {
                    PRINT_ERROR_LOG("Output definition path is too long.");
                    return EXIT_FAILURE;
                }
                std::wstringstream bss;
                bss << OPT_OUTPUT_FILE_NAME_FULL << L"=";
                bss << bfshaOutputPath;
                args.options.push_back(CmdOption(bss.str().c_str()));
            }
        }
        else
        {
            PRINT_ERROR_LOG("Not found output path.");
            return EXIT_FAILURE;
        }
    }
    else
    {
        // ここで入力ファイルの拡張子を見て出力が fsd か bfsha を判定する。
        const CmdOption& output = args.options[outputIndex];

        // 拡張子抄出
        std::wstring dext;
        std::wstring bext;
        wchar_t nameBuf[_MAX_FNAME];
        wchar_t extBuf[_MAX_EXT];
        wchar_t fullpathBuf[_MAX_PATH];
        std::wstring outputPath;

        if (_wsplitpath_s(output.value.c_str(), nullptr, 0, nullptr, 0, nameBuf, _MAX_FNAME, extBuf, _MAX_EXT) == 0)
        {
            dext.assign(extBuf);
        }

        if( FindFsc( args.inputs ) )
        {
            // input に fsc が見つかった時は fsd を生成する。
            if (dext == OUTPUT_DEFINITION_BINARY_FILE_EXT)
            {
                outputPath = output.value;
            }
            else
            {
                // 出力先ファイルの拡張子が .fsda か .fsdb ではない場合、.fsdb をつけて出力する。
                outputPath = output.value;
                outputPath += OUTPUT_DEFINITION_BINARY_FILE_EXT;
            }
            if(_wfullpath(fullpathBuf, outputPath.c_str(), _MAX_PATH))
            {
                outputPath.assign(fullpathBuf);
            }
            else
            {
                PRINT_ERROR_LOG("Output definition path is too long.");
                return EXIT_FAILURE;
            }
            fsdbOutputPath = outputPath;
        }
        else
        {
            // fsd が入力されている時は bfres を出力する。
            if (dext == OUTPUT_BINARY_FILE_EXT )
            {
                outputPath = output.value;
            }
            else
            {
                // 出力先ファイルの拡張子が .bfsha ではない場合、.bfsha をつけて出力する。
                outputPath = output.value;
                outputPath += OUTPUT_BINARY_FILE_EXT;
            }
            if(_wfullpath(fullpathBuf, outputPath.c_str(), _MAX_PATH))
            {
                outputPath.assign(fullpathBuf);
            }
            else
            {
                PRINT_ERROR_LOG("Output definition path is too long.");
                return EXIT_FAILURE;
            }
            bfshaOutputPath = outputPath;
        }
    }

    std::unique_ptr<const wchar_t*[]> optionsSet(conv(args.options));
    // オプションの登録。
    success = nng3dToolShaderConverterSetOptions(optionsSet.get());
    if (!success)
    {
        PRINT_ERROR_LOG("SetOptions() failed. Check if your options are correct.");
        return EXIT_FAILURE;
    }

    // 各入力ファイル。
    PRINT_LOG(LOG_LEVEL_0(L"Input Files"));
    for (auto iter = args.inputs.cbegin(); iter != args.inputs.cend(); ++iter)
    {
        util::Buffer file;
        success = file.LoadFile(iter->fullpath.c_str());
        if (!success)
        {
            PRINT_ERROR_LOG("Load file failed.");
            PRINT_ERROR_LOG("File: %s", iter->fullpath.c_str());
            return EXIT_FAILURE;
        }

        const wchar_t* path[3];
        path[0] = iter->fullpath.c_str();
        path[1] = iter->fname.c_str();
        path[2] = iter->ext.c_str();
        std::unique_ptr<const wchar_t*[]> options(conv(iter->options));

        success = nng3dToolShaderConverterAddFile(file.Get(), file.Size(), path, options.get());
        if (!success)
        {
            PRINT_ERROR_LOG("AddFile() failed.");
            PRINT_ERROR_LOG("File: %s", iter->fullpath.c_str());
            return EXIT_FAILURE;
        }
    }

    // テストモードが有効な場合は中間ファイルのチェックが終わった段階で終了する。
    if (CmdOption::FindIndex(args.options, OPT_VERIFY_FULL) >= 0)
    {
        return EXIT_SUCCESS;
    }

    PRINT_LOG(LOG_LEVEL_0(L"Make fsd."));
    success = nng3dToolShaderConverterMake();
    if (!success)
    {
        PRINT_ERROR_LOG("Make() failed.");
        return EXIT_FAILURE;
    }

    if (!fsdbOutputPath.empty())
    {
        size_t size = nng3dToolShaderConverterCalculateDefinitionSize(fsdbOutputPath.c_str());
        if (size == 0)
        {
            PRINT_ERROR_LOG("CalcDefinitionSize() failed.");
            return EXIT_FAILURE;
        }

        util::Buffer outfile;
        outfile.Alloc(size, 0x100);

        success = nng3dToolShaderConverterWrite(outfile.Get(), size);
        if (!success)
        {
            PRINT_ERROR_LOG("Write() failed.");
            PRINT_ERROR_LOG("File: %s", fsdbOutputPath.c_str());
            return EXIT_FAILURE;
        }

        // 出力ファイルの書き出し。
        success = outfile.SaveFile(fsdbOutputPath.c_str());
        if (!success)
        {
            PRINT_ERROR_LOG("Output file failed.");
            PRINT_ERROR_LOG("File: %s", fsdbOutputPath.c_str());
            return EXIT_FAILURE;
        }
    }

    if (!bfshaOutputPath.empty())
    {
        PRINT_LOG(LOG_LEVEL_0(L"Convert binary."));

        // 出力ファイルサイズの計算。
        size_t size = nng3dToolShaderConverterCalculateArchiveSize();
        if( CmdOption::FindIndex( args.options, OPT_SKIP_CONVERT_FULL ) < 0 )
        {
            if( size == 0 )
            {
                PRINT_ERROR_LOG( "CalcArchiveSize() failed." );
                return EXIT_FAILURE;
            }
            util::Buffer outfile;
            outfile.Alloc( size, 0x100 );

            // 変換処理。
            success = nng3dToolShaderConverterConvert( outfile.Get(), outfile.Size() );
            if( !success )
            {
                PRINT_ERROR_LOG( "Convert() failed." );
                return EXIT_FAILURE;
            }

            // エンディアンスワップ
            success = nng3dToolShaderConverterSwapEndian( outfile.Get(), outfile.Size() );
            if( !success )
            {
                PRINT_ERROR_LOG( "EndianSwap() failed." );
                return EXIT_FAILURE;
            }

            // 出力ファイルの書き出し。
            success = outfile.SaveFile(bfshaOutputPath.c_str());
            if( !success )
            {
                PRINT_ERROR_LOG( "Output file failed. Check if output path is correct and not being opened." );
                PRINT_ERROR_LOG( "File: %s", bfshaOutputPath.c_str() );
                return EXIT_FAILURE;
            }
        }
    }

    // 終了処理
    {
        success = nng3dToolShaderConverterShutdown();
        if (!success)
        {
            PRINT_ERROR_LOG("Shutdown failed.");
            return EXIT_FAILURE;
        }
    }

    return EXIT_SUCCESS;
}
