﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <Windows.h>
#include <vector>
#include <memory>
#include <util/UtilCmdArgs.h>
#include "GlobalOptionParser.h"
#include <ShaderConverterEnum.h>

namespace nn {
namespace g3dTool {

class CommandLineParserWrapper
{
public:
    //!< @brief dll 関数ポインタの定義と宣言です。
    struct DllFuncs
    {
        typedef int GetCvtrVersion();
        typedef bool Initialize();
        typedef bool Shutdown();
        typedef bool SetOptions(const wchar_t* options[]);
        typedef bool AddFile(void* pData, size_t dataSize, const wchar_t* path[3], const wchar_t* options[]);
        typedef size_t CalculateArchiveSize();
        typedef bool Convert(void* pBuffer, size_t bufferSize);
        typedef bool SwapEndian(void* pBuffer, size_t size);
        typedef bool Make();
        typedef size_t CalculateDefinitionSize(const wchar_t* fullpath);
        typedef bool Write(void* pBuffer, size_t size);

        GetCvtrVersion				pFuncGetCvtrVersion;
        Initialize					pFuncInitialize;
        Shutdown					pFuncShutdown;
        SetOptions					pFuncSetOptions;
        AddFile						pFuncAddFile;
        CalculateArchiveSize		pFuncCalculateArchiveSize;
        Convert						pFuncConvert;
        SwapEndian					pFuncSwapEndian;
        Make						pFuncMake;
        CalculateDefinitionSize		pFuncCalculateDefinitionSize;
        Write						pFuncWrite;
    };

    explicit
    CommandLineParserWrapper( int argc, const wchar_t* argv[] );

    virtual ~CommandLineParserWrapper()
    {
        m_pCommandArgs.reset();
    }

    //!< @brief HMODULE から関数ポンタを取得します。
    void LoadDllFuncs( const HMODULE& hModule );

    //!< @brief 初期化済みの nw コマンドラインパーサを取得します。
    std::shared_ptr< nw::g3d::tool::CmdArgs > GetCmdArgs()
    {
        return m_pCommandArgs;
    }

    //!< @brief dll 関数ポインタを返します。
    const DllFuncs& GetDllFuncs() const
    {
        return m_DllFuncs;
    }

    //!< @brief ヘルプを表示します。
    void PrintHelp( const char* pToolName, const char* pToolDescription )
    {
        m_pGlobalOptionParser->PrintHelp( pToolName, pToolDescription );
    }

private:
    //!< @brief グローバルオプションのパーサです。
    std::unique_ptr<GlobalOptionParserBase>		m_pGlobalOptionParser;

    //!< @brief dll から取得した関数ポインタを保持します。
    DllFuncs							m_DllFuncs;

    //!< @brief char 文字列を保持するバッファです。
    std::vector< std::string >			m_CharBuff;

    //!< @brief wchar_t 文字列を保持するバッファです。
    std::vector< std::wstring >			m_WideCharBuff;

    //!< @brief 旧 nw のパーサです。
    std::shared_ptr< nw::g3d::tool::CmdArgs >	m_pCommandArgs;

};

}
}
