﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "stdafx.h"
#include "Converter.h"
#include <sdk_ver.h>
#if CAFE_OS_SDK_VERSION < 20903
using namespace TexUtils;
#endif
#include "Common.h"
#include "DecodeInternal.h"

#include "PVRTextureDefines.h"
#include "PVRTDecompress.h"

using namespace System::Threading::Tasks;
using namespace nw::g3d::iflib;

namespace nw
{
namespace g3d
{
namespace iftexutil
{
namespace cafe
{

HMODULE hDLL;
bool (*DLL_TC2Initialize)(TC2Config* pConfig);
bool (*DLL_TC2Destroy)(void);
bool (*DLL_TC2GenerateMipLevels)(GX2Surface* pInSurface, s32 iMinSize, TC2MipFilter filter, GX2Surface* pOutSurface);
bool (*DLL_TC2ConvertSurfaceFormat)(GX2Surface *pInSurface, GX2SurfaceFormat dstFormat, GX2Surface *pOutSurface, TC2ConvertOptions* pOptions);
bool (*DLL_TC2ConvertTiling)(GX2Surface *pInSurface, GX2TileMode dstTileMode, u32 initialSwizzle, GX2Surface* pOutSurface);
bool (*DLL_TC2DestroyGX2Surface)(GX2Surface* pGX2Surface);

// モジュールのディレクトリ名を取得
static DWORD GetModuleDirName(HMODULE hModule, TCHAR* lpDirName, DWORD nSize);

class RSurface
{
  public:
    GX2Surface m_Surface; //!< GX2 サーフェスです。
    std::string m_Name; //!< 名前です。
    bool m_IsCreatedByTC2; //!< TC2 で作成されたなら true です。

  public:
    //! コンストラクタです。
    RSurface(const std::string& name, const bool isCreatedByTC2)
    : m_Name(name),
      m_IsCreatedByTC2(isCreatedByTC2)
    {
        //cerr << " RSurface(): " << m_Name << ": " << m_IsCreatedByTC2 << endl;
        memset(&m_Surface, 0, sizeof(m_Surface));
    }

    //! デストラクタです。
    virtual ~RSurface()
    {
        //cerr << "~RSurface(): " << m_Name << ": " << m_IsCreatedByTC2 << endl;
        if (m_IsCreatedByTC2)
        {
            DLL_TC2DestroyGX2Surface(&m_Surface);
        }
        else
        {
            if (m_Surface.imagePtr != NULL)
            {
                delete[] m_Surface.imagePtr;
            }

            if (m_Surface.mipPtr != NULL)
            {
                delete[] m_Surface.mipPtr;
            }
        }
    }
};

namespace
{
const unsigned char gamma22[256] =
{
    #include "gamma22.inc"
};

const unsigned char gamma10[256] =
{
    #include "gamma10.inc"
};

const int blockWidth  = 4;
const int blockHeight = 4;
const int sizeofRGBA  = 4;

const int blockWidthMask  = blockWidth - 1;
const int blockHeightMask = blockHeight - 1;

struct FormatInfo
{
    texture_info_quantize_typeType	type_;				// フォーマットのタイプ
    int								channelCount_;		// 有効チャンネル数 1〜4
    float							bytePerPixel_;		// １ピクセルあたりのバイト数
    bool                            isFloat_;           // floatフォーマットとかどうか？
};

static FormatInfo formatInfos[] =
{
    // http://msdn.microsoft.com/ja-jp/library/bb694531%28v=vs.85%29.aspx

    {texture_info_quantize_typeType::unorm_8,			1,	float((8) / 8)                        , false},
    {texture_info_quantize_typeType::uint_8,			1,	float((8) / 8)                        , false},
    {texture_info_quantize_typeType::snorm_8,			1,	float((8) / 8)                        , false},
    {texture_info_quantize_typeType::sint_8,			1,	float((8) / 8)                        , false},
    {texture_info_quantize_typeType::unorm_8_8,			2,	float((8 + 8) / 8)                    , false},
    {texture_info_quantize_typeType::uint_8_8,			2,	float((8 + 8) / 8)                    , false},
    {texture_info_quantize_typeType::snorm_8_8,			2,	float((8 + 8) / 8)                    , false},
    {texture_info_quantize_typeType::sint_8_8,			2,	float((8 + 8) / 8)                    , false},
    {texture_info_quantize_typeType::unorm_5_6_5,		3,	float((5 + 6 + 5) / 8)                , false},
    {texture_info_quantize_typeType::unorm_5_5_5_1,		4,	float((5 + 5 + 5 + 1) / 8)            , false},
    {texture_info_quantize_typeType::unorm_4_4_4_4,		4,	float((4 + 4 + 4 + 4) / 8)            , false},
    {texture_info_quantize_typeType::unorm_8_8_8_8,		4,	float((8 + 8 + 8 + 8) / 8)            , false},
    {texture_info_quantize_typeType::uint_8_8_8_8,		4,	float((8 + 8 + 8 + 8) / 8)            , false},
    {texture_info_quantize_typeType::snorm_8_8_8_8,		4,	float((8 + 8 + 8 + 8) / 8)            , false},
    {texture_info_quantize_typeType::sint_8_8_8_8,		4,	float((8 + 8 + 8 + 8) / 8)            , false},
    {texture_info_quantize_typeType::unorm_bc1,			4,	0.5f                                  , false},
    {texture_info_quantize_typeType::unorm_bc2,			4,	1.0f                                  , false},
    {texture_info_quantize_typeType::unorm_bc3,			4,	1.0f                                  , false},
    {texture_info_quantize_typeType::unorm_bc4,			1,	0.5f                                  , false},
    {texture_info_quantize_typeType::snorm_bc4,			1,	0.5f                                  , false},
    {texture_info_quantize_typeType::unorm_bc5,			2,	1.0f                                  , false},
    {texture_info_quantize_typeType::snorm_bc5,			2,	1.0f                                  , false},

    // 2.2追加分
    {texture_info_quantize_typeType::srgb_8_8_8_8,		4,	float((8 + 8 + 8 + 8) / 8)            , false},
    {texture_info_quantize_typeType::srgb_bc1,			4,	0.5f                                  , false},
    {texture_info_quantize_typeType::srgb_bc2,			4,	1.0f                                  , false},
    {texture_info_quantize_typeType::srgb_bc3,			4,	1.0f                                  , false},

    // 追加
    {texture_info_quantize_typeType::unorm_4_4,			2,	float((4 + 4) / 8)                    , false},

    // 追加 (float系)
    {texture_info_quantize_typeType::float_16,			1,	float((16) / 8)                       , true},
    {texture_info_quantize_typeType::float_32,			1,	float((32) / 8)                       , true},
    {texture_info_quantize_typeType::float_16_16,		2,	float((16 + 16) / 8)                  , true},
    {texture_info_quantize_typeType::float_32_32,		2,	float((32 + 32) / 8)                  , true},
    {texture_info_quantize_typeType::float_11_11_10,	3,	float((11 + 11 + 10) / 8)             , true},
    {texture_info_quantize_typeType::float_16_16_16_16,	4,	float((16 + 16 + 16 + 16) / 8)        , true},
    {texture_info_quantize_typeType::float_32_32_32_32,	4,	float((32 + 32 + 32 + 32) / 8)        , true},


    {texture_info_quantize_typeType::unorm_pvrtc1_2bpp, 3, float((2 + 2 + 2) / 8), false},
    {texture_info_quantize_typeType::unorm_pvrtc1_4bpp, 3, float((4 + 4 + 4) / 8), false},
    {texture_info_quantize_typeType::unorm_pvrtc1_alpha_2bpp, 4, float((2 + 2 + 2 + 2) / 8), false},
    {texture_info_quantize_typeType::unorm_pvrtc1_alpha_4bpp, 4, float((4 + 4 + 4 + 4) / 8), false},
    {texture_info_quantize_typeType::unorm_pvrtc2_alpha_2bpp, 4, float((2 + 2 + 2 + 2) / 8), false},
    {texture_info_quantize_typeType::unorm_pvrtc2_alpha_4bpp, 4, float((4 + 4 + 4 + 4) / 8), false},
    {texture_info_quantize_typeType::srgb_pvrtc1_2bpp, 3, float((2 + 2 + 2) / 8), false},
    {texture_info_quantize_typeType::srgb_pvrtc1_4bpp, 3, float((4 + 4 + 4) / 8), false},
    {texture_info_quantize_typeType::srgb_pvrtc1_alpha_2bpp, 4, float((2 + 2 + 2 + 2) / 8), false},
    {texture_info_quantize_typeType::srgb_pvrtc1_alpha_4bpp, 4, float((4 + 4 + 4 + 4) / 8), false},
    {texture_info_quantize_typeType::srgb_pvrtc2_alpha_2bpp, 4, float((2 + 2 + 2 + 2) / 8), false},
    {texture_info_quantize_typeType::srgb_pvrtc2_alpha_4bpp, 4, float((4 + 4 + 4 + 4) / 8), false},
};

struct CompareByType
{
    bool operator()(const FormatInfo &l, const FormatInfo &r) const
    {
        return (int)l.type_ < (int)r.type_;
    }
};

void InitializeFormatInfos()
{
    // type_でソートしておく
    int count = sizeof(formatInfos) / sizeof(formatInfos[0]);
    std::sort(formatInfos, formatInfos + count, CompareByType());
}

const FormatInfo &FindFormatInfo(texture_info_quantize_typeType type)
{
    FormatInfo key;
    key.type_ = type;

    int count = sizeof(formatInfos) / sizeof(formatInfos[0]);
    auto i = std::lower_bound(formatInfos, formatInfos + count, key, CompareByType());

    assert(i != formatInfos + count);

    return *i;
}

float GetBytesPerPixel(texture_info_quantize_typeType type)
{
    return FindFormatInfo(type).bytePerPixel_;
}

int GetChannelCount(texture_info_quantize_typeType type)
{
    return FindFormatInfo(type).channelCount_;
}

std::string Cast(System::String ^s){

    msclr::interop::marshal_context ^mc = gcnew msclr::interop::marshal_context;

    std::string d = mc->marshal_as<std::string>(s);

    delete mc;

    return d;
}

int MipmapSize(int srcSize, int mipmapLevel)
{
    return std::max(srcSize >> mipmapLevel, 1);
}

//-----------------------------------------------------------------------------
void* GetLevelImagePtr(GX2Surface& surface, u32 level, u32 iDepth)
{
    void* top;
    u32 size;
    if (level == 0)
    {
        top  = surface.imagePtr;
        size = surface.imageSize;
    }
    else if (level == 1)
    {
        top  = surface.mipPtr;
        size = surface.mipOffset[1];
    }
    else // 2-
    {
        top = reinterpret_cast<u8*>(surface.mipPtr) + surface.mipOffset[level - 1];
        size = (level == surface.numMips - 1) ?
            surface.mipSize          - surface.mipOffset[level - 1] :
            surface.mipOffset[level] - surface.mipOffset[level - 1];
    }

    const u32 curD = (surface.dim == GX2_SURFACE_DIM_3D) ?
        std::max(surface.depth >> level, static_cast<u32>(1)) : surface.depth;
    return reinterpret_cast<u8*>(top) + size / curD * iDepth;
}

void MakeAllSourceSurface(RSurface &dstSurface, textureType ^texture, List<G3dStream ^> ^streams, bool isCubeArray)
{
    // ドキュメントにないがキューブアレイは2Dアレイとして扱うと正常にデタイリングできる。
    dstSurface.m_Surface.dim		=
        isCubeArray ?
            GX2_SURFACE_DIM_2D_ARRAY :
            nwFtx::RGetSurfaceDimensionFromString(NULL, Cast(texture->texture_info->dimension.ToString()));

    dstSurface.m_Surface.width		= texture->texture_info->width;
    dstSurface.m_Surface.height		= texture->texture_info->height;
    dstSurface.m_Surface.depth		= texture->texture_info->depth;
    dstSurface.m_Surface.numMips	= texture->texture_info->mip_level;
    dstSurface.m_Surface.format		= nwFtx::RGetSurfaceFormatFromString(Cast(texture->texture_info->quantize_type.ToString()));
    dstSurface.m_Surface.aa			= GX2_AA_MODE_1X;
    dstSurface.m_Surface.use		= GX2_SURFACE_USE_TEXTURE;
    dstSurface.m_Surface.imageSize	= static_cast<u32>((texture->texture_info->mip_level >= 2) ? texture->texture_info->mip_offset[0] : texture->texture_info->size);
    dstSurface.m_Surface.imagePtr	= new char[dstSurface.m_Surface.imageSize];
    dstSurface.m_Surface.mipSize	= std::max<int>(texture->texture_info->size - dstSurface.m_Surface.imageSize, 0);
    dstSurface.m_Surface.mipPtr		= new char[dstSurface.m_Surface.mipSize];

    for(int i = 0;i != 13;++ i)
    {
        dstSurface.m_Surface.mipOffset[i] = static_cast<u32>(texture->texture_info->mip_offset[i]);
    }

    dstSurface.m_Surface.tileMode	= nwFtx::RGetTileModeFromString(Cast(texture->texture_info->tile_mode.ToString()));
    dstSurface.m_Surface.swizzle	= texture->texture_info->swizzle;
    dstSurface.m_Surface.alignment	= texture->texture_info->alignment;
    dstSurface.m_Surface.pitch		= texture->texture_info->pitch;

    {
        array<byte>	^src = streams[texture->texture_info->stream_index]->ByteData->ToArray();

        Marshal::Copy(src, 0,								(IntPtr)dstSurface.m_Surface.imagePtr,	dstSurface.m_Surface.imageSize);
        Marshal::Copy(src, dstSurface.m_Surface.imageSize,	(IntPtr)dstSurface.m_Surface.mipPtr,	dstSurface.m_Surface.mipSize);
    }
}

void MakeGammaCorrectTable(bool isGammaCorrection, textureType ^texture, const unsigned char *dst[4])
{
    if (isGammaCorrection)
    {
        switch(texture->texture_info->quantize_type)
        {
            // srgb_* はガンマ補正なし
            case texture_info_quantize_typeType::srgb_8_8_8_8:
            case texture_info_quantize_typeType::srgb_bc1:
            case texture_info_quantize_typeType::srgb_bc2:
            case texture_info_quantize_typeType::srgb_bc3:
            {
                dst[2] = gamma10;	// R
                dst[1] = gamma10;	// G
                dst[0] = gamma10;	// B
                dst[3] = gamma10;	// A
                break;
            }

            default:
            {
                // http://www-sdd.zelda.nintendo.co.jp/project/nintendoware3/kagemai/html/user.cgi?project=nw3_3de&action=view_report&id=1010
                // > また、2 チャンネルの実機向けフォーマットについては、
                // > 元画像が RGB であれば R と G が、RGBA であれば R と A が、
                // > ガンマ補正の対象となるチャンネルになります。
                bool linear[4];
                for (int i = 0; i < 4; i++)
                {
                    linear[i] = texture->texture_info->linear[i] == true;
                }

                // 実数型テクスチャはリニア空間なので、常にsRGB変換を行う。
                if (texture->original_image_array->original_image[0]->format == original_image_formatType::rgb32f || texture->original_image_array->original_image[0]->format == original_image_formatType::rgba32f)
                {
                    linear[0] = linear[1] = linear[2] = true;
                    linear[3] = false;
                }

                bool is2ch = GetChannelCount(texture->texture_info->quantize_type) == 2;
                if (is2ch)
                {
                    bool isRgb = (texture->original_image_array->original_image[0]->format == original_image_formatType::rgb8) ||
                                 (texture->original_image_array->original_image[0]->format == original_image_formatType::rgb32f);

                    // RGB
                    if (isRgb)
                    {
                        dst[2] = linear[0] ? gamma22 : gamma10;	// R
                        dst[1] = linear[1] ? gamma22 : gamma10;	// G
                        dst[0] = linear[2] ? gamma22 : gamma10;	// B
                        dst[3] = linear[3] ? gamma22 : gamma10;	// A
                    }
                    // RGBA
                    else
                    {
                        dst[2] = linear[0] ? gamma22 : gamma10;	// R
                        dst[1] = linear[3] ? gamma22 : gamma10;	// G
                        dst[0] = linear[2] ? gamma22 : gamma10;	// B
                        dst[3] = linear[3] ? gamma22 : gamma10;	// A
                    }
                }
                else
                {
                    dst[2] = linear[0] ? gamma22 : gamma10;	// R
                    dst[1] = linear[1] ? gamma22 : gamma10;	// G
                    dst[0] = linear[2] ? gamma22 : gamma10;	// B
                    dst[3] = linear[3] ? gamma22 : gamma10;	// A
                }

                break;
            }
        }
    }
    else
    {
        dst[2] = gamma10;	// R
        dst[1] = gamma10;	// G
        dst[0] = gamma10;	// B
        dst[3] = gamma10;	// A
    }
}

//-----------------------------------------------------------------------------
// 並列実行部
ref class ConvertToBitmapLinear_Decode
{
public:
    ConvertToBitmapLinear_Decode(texture_info_quantize_typeType quantize_type, int colorDataStride, unsigned char *dstColor, const unsigned char *src, int lineSize, const unsigned char *gammaCorrectTable[4], bool isHintNormal) :
        quantize_type_(quantize_type),
        colorDataStride_(colorDataStride),
        dstColor_(dstColor),
        src_(src),
        lineSize_(lineSize),
        gammaCorrectTable_(gammaCorrectTable),
        isHintNormal_(isHintNormal){}

    void Execute(int y)
    {
        const unsigned char *src = src_ + lineSize_ * y;
        unsigned char *dstColor = dstColor_ + colorDataStride_ * y;

        switch(quantize_type_)
        {
            case texture_info_quantize_typeType::unorm_8:			TexUtils::Decode_unorm_8(			dstColor, src, lineSize_, gammaCorrectTable_);					break;
            // unorm_8 と同じ
            case texture_info_quantize_typeType::uint_8:			TexUtils::Decode_unorm_8(			dstColor, src, lineSize_, gammaCorrectTable_);					break;
            case texture_info_quantize_typeType::snorm_8:			TexUtils::Decode_snorm_8(			dstColor, src, lineSize_, gammaCorrectTable_);					break;
            // snorm_8 と同じ
            case texture_info_quantize_typeType::sint_8:			TexUtils::Decode_snorm_8(			dstColor, src, lineSize_, gammaCorrectTable_);					break;
            case texture_info_quantize_typeType::unorm_4_4:			TexUtils::Decode_unorm_4_4(			dstColor, src, lineSize_, gammaCorrectTable_);					break;
            case texture_info_quantize_typeType::unorm_8_8:			TexUtils::Decode_unorm_8_8(			dstColor, src, lineSize_, gammaCorrectTable_);					break;
            // unorm_8_8 と同じ
            case texture_info_quantize_typeType::uint_8_8:			TexUtils::Decode_unorm_8_8(			dstColor, src, lineSize_, gammaCorrectTable_);					break;
            case texture_info_quantize_typeType::snorm_8_8:			TexUtils::Decode_snorm_8_8(			dstColor, src, lineSize_, gammaCorrectTable_, isHintNormal_);	break;
            // snorm_8_8 と同じ
            case texture_info_quantize_typeType::sint_8_8:			TexUtils::Decode_snorm_8_8(			dstColor, src, lineSize_, gammaCorrectTable_, isHintNormal_);	break;
            case texture_info_quantize_typeType::unorm_5_6_5:		TexUtils::Decode_unorm_5_6_5(		dstColor, src, lineSize_, gammaCorrectTable_);					break;
            case texture_info_quantize_typeType::unorm_5_5_5_1:		TexUtils::Decode_unorm_5_5_5_1(		dstColor, src, lineSize_, gammaCorrectTable_);					break;
            case texture_info_quantize_typeType::unorm_4_4_4_4:		TexUtils::Decode_unorm_4_4_4_4(		dstColor, src, lineSize_, gammaCorrectTable_);					break;
            case texture_info_quantize_typeType::unorm_8_8_8_8:		TexUtils::Decode_unorm_8_8_8_8(		dstColor, src, lineSize_, gammaCorrectTable_);					break;
            // unorm_8_8_8_8 と同じ
            case texture_info_quantize_typeType::uint_8_8_8_8:		TexUtils::Decode_unorm_8_8_8_8(		dstColor, src, lineSize_, gammaCorrectTable_);					break;
            case texture_info_quantize_typeType::snorm_8_8_8_8:		TexUtils::Decode_snorm_8_8_8_8(		dstColor, src, lineSize_, gammaCorrectTable_);					break;
            // snorm_8_8_8_8 と同じ
            case texture_info_quantize_typeType::sint_8_8_8_8:		TexUtils::Decode_snorm_8_8_8_8(		dstColor, src, lineSize_, gammaCorrectTable_);					break;
            case texture_info_quantize_typeType::srgb_8_8_8_8:		TexUtils::Decode_unorm_8_8_8_8(		dstColor, src, lineSize_, gammaCorrectTable_);					break;
            case texture_info_quantize_typeType::float_16:			TexUtils::Decode_float_16(			dstColor, src, lineSize_, gammaCorrectTable_);					break;
            case texture_info_quantize_typeType::float_32:			TexUtils::Decode_float_32(			dstColor, src, lineSize_, gammaCorrectTable_);					break;
            case texture_info_quantize_typeType::float_16_16:		TexUtils::Decode_float_16_16(		dstColor, src, lineSize_, gammaCorrectTable_);					break;
            case texture_info_quantize_typeType::float_32_32:		TexUtils::Decode_float_32_32(		dstColor, src, lineSize_, gammaCorrectTable_);					break;
            case texture_info_quantize_typeType::float_11_11_10:	TexUtils::Decode_float_11_11_10(	dstColor, src, lineSize_, gammaCorrectTable_);					break;
            case texture_info_quantize_typeType::float_16_16_16_16:	TexUtils::Decode_float_16_16_16_16(	dstColor, src, lineSize_, gammaCorrectTable_);					break;
            case texture_info_quantize_typeType::float_32_32_32_32:	TexUtils::Decode_float_32_32_32_32(	dstColor, src, lineSize_, gammaCorrectTable_);					break;
            default:												assert(false);																						break;
        }
    }

private:
    texture_info_quantize_typeType	quantize_type_;
    int								colorDataStride_;
    unsigned char *					dstColor_;
    const unsigned char *			src_;
    int								lineSize_;
    const unsigned char **			gammaCorrectTable_;
    bool							isHintNormal_;
};

ref class ConvertToBitmapCompress_Decode
{
public:
    ConvertToBitmapCompress_Decode(texture_info_quantize_typeType quantize_type, int width, int stride, int blockSize, unsigned char *dst, const unsigned char *src, bool isHintNormal) :
        quantize_type_(quantize_type),
        width_(width),
        stride_(stride),
        blockSize_(blockSize),
        dst_(dst),
        src_(src),
        isHintNormal_(isHintNormal){}

    void Execute(int y)
    {
        const unsigned char *src = src_ + blockSize_ * y * width_ / blockWidth;

        y *= blockHeight;

        for(int x = 0;x < width_;x += blockWidth)
        {
            switch(quantize_type_)
            {
                case texture_info_quantize_typeType::unorm_bc1:	TexUtils::Decode_unorm_bc1(dst_, src, x, y, stride_);					break;
                case texture_info_quantize_typeType::unorm_bc2:	TexUtils::Decode_unorm_bc2(dst_, src, x, y, stride_);					break;
                case texture_info_quantize_typeType::unorm_bc3:	TexUtils::Decode_unorm_bc3(dst_, src, x, y, stride_);					break;
                case texture_info_quantize_typeType::unorm_bc4:	TexUtils::Decode_unorm_bc4(dst_, src, x, y, stride_);					break;
                case texture_info_quantize_typeType::snorm_bc4:	TexUtils::Decode_snorm_bc4(dst_, src, x, y, stride_);					break;
                case texture_info_quantize_typeType::unorm_bc5:	TexUtils::Decode_unorm_bc5(dst_, src, x, y, stride_);					break;
                case texture_info_quantize_typeType::snorm_bc5:	TexUtils::Decode_snorm_bc5(dst_, src, x, y, stride_, isHintNormal_);	break;
                case texture_info_quantize_typeType::srgb_bc1:	TexUtils::Decode_unorm_bc1(dst_, src, x, y, stride_);					break;
                case texture_info_quantize_typeType::srgb_bc2:	TexUtils::Decode_unorm_bc2(dst_, src, x, y, stride_);					break;
                case texture_info_quantize_typeType::srgb_bc3:	TexUtils::Decode_unorm_bc3(dst_, src, x, y, stride_);					break;
                default:										assert(false);															break;
            }

            src += blockSize_;
        }
    }

private:
    texture_info_quantize_typeType	quantize_type_;
    int								width_;
    int								stride_;
    int								blockSize_;
    unsigned char *					dst_;
    const unsigned char *			src_;
    bool							isHintNormal_;
};

ref class ConvertToBitmapCompress_Transfer
{
public:
    ConvertToBitmapCompress_Transfer(int colorDataStride, unsigned char *dstColor, const unsigned char *src, int lineSize, int alignLineSize, const unsigned char *gammaCorrectTable[4]) :
        colorDataStride_(colorDataStride),
        dstColor_(dstColor),
        src_(src),
        lineSize_(lineSize),
        alignLineSize_(alignLineSize),
        gammaCorrectTable_(gammaCorrectTable),
        srcStride_(std::max(alignLineSize, blockWidth * sizeofRGBA)){}

    void Execute(int y)
    {
        const unsigned char *src = src_ + srcStride_ * y;
        unsigned char *dstColor = dstColor_ + colorDataStride_ * y;

        TexUtils::Decode_unorm_8_8_8_8(dstColor, src, lineSize_, gammaCorrectTable_);
    }

private:
    int						colorDataStride_;
    unsigned char *			dstColor_;
    const unsigned char *	src_;
    int						lineSize_;
    int						alignLineSize_;
    const unsigned char **	gammaCorrectTable_;
    int						srcStride_;
};


//-----------------------------------------------------------------------------
// 並列実行部
ref class ConvertToStreamLinear_Decode
{
public:
    ConvertToStreamLinear_Decode(texture_info_quantize_typeType quantize_type, int colorDataStride, Byte *dstByteColor, float *dstFloatColor, const unsigned char *src, int lineSize, bool isHintNormal) :
        quantize_type_(quantize_type),
        colorDataStride_(colorDataStride),
        dstByteColor_(dstByteColor),
        dstFloatColor_(dstFloatColor),
        src_(src),
        lineSize_(lineSize),
        isHintNormal_(isHintNormal){}

    void Execute(int y)
    {
        const unsigned char *src = src_ + lineSize_ * y;
        Byte  *dstByteColor  = dstByteColor_  + colorDataStride_ * y;
        float *dstFloatColor = dstFloatColor_ + colorDataStride_ * y;

        switch(quantize_type_)
        {
            case texture_info_quantize_typeType::unorm_8:			TexUtils::Decode_unorm_8(			dstByteColor,  src, lineSize_);					break;
            case texture_info_quantize_typeType::snorm_8:			TexUtils::Decode_snorm_8(			dstByteColor,  src, lineSize_);					break;
            case texture_info_quantize_typeType::unorm_4_4:			TexUtils::Decode_unorm_4_4(			dstByteColor,  src, lineSize_);					break;
            case texture_info_quantize_typeType::unorm_8_8:			TexUtils::Decode_unorm_8_8(			dstByteColor,  src, lineSize_);					break;
            case texture_info_quantize_typeType::snorm_8_8:			TexUtils::Decode_snorm_8_8(			dstByteColor,  src, lineSize_, isHintNormal_);	break;
            case texture_info_quantize_typeType::unorm_5_6_5:		TexUtils::Decode_unorm_5_6_5(		dstByteColor,  src, lineSize_);					break;
            case texture_info_quantize_typeType::unorm_5_5_5_1:		TexUtils::Decode_unorm_5_5_5_1(		dstByteColor,  src, lineSize_);					break;
            case texture_info_quantize_typeType::unorm_4_4_4_4:		TexUtils::Decode_unorm_4_4_4_4(		dstByteColor,  src, lineSize_);					break;
            case texture_info_quantize_typeType::unorm_8_8_8_8:		TexUtils::Decode_unorm_8_8_8_8(		dstByteColor,  src, lineSize_);					break;
            case texture_info_quantize_typeType::snorm_8_8_8_8:		TexUtils::Decode_snorm_8_8_8_8(		dstByteColor,  src, lineSize_);					break;
            case texture_info_quantize_typeType::srgb_8_8_8_8:		TexUtils::Decode_unorm_8_8_8_8(		dstByteColor,  src, lineSize_);					break;
            case texture_info_quantize_typeType::float_16:			TexUtils::Decode_float_16(			dstFloatColor, src, lineSize_);					break;
            case texture_info_quantize_typeType::float_32:			TexUtils::Decode_float_32(			dstFloatColor, src, lineSize_);					break;
            case texture_info_quantize_typeType::float_16_16:		TexUtils::Decode_float_16_16(		dstFloatColor, src, lineSize_);				    break;
            case texture_info_quantize_typeType::float_32_32:		TexUtils::Decode_float_32_32(		dstFloatColor, src, lineSize_);			        break;
            case texture_info_quantize_typeType::float_11_11_10:	TexUtils::Decode_float_11_11_10(	dstFloatColor, src, lineSize_);		            break;
            case texture_info_quantize_typeType::float_16_16_16_16:	TexUtils::Decode_float_16_16_16_16(	dstFloatColor, src, lineSize_);	                break;
            case texture_info_quantize_typeType::float_32_32_32_32:	TexUtils::Decode_float_32_32_32_32(	dstFloatColor, src, lineSize_);                 break;
            case texture_info_quantize_typeType::unorm_pvrtc1_2bpp:	        break;
            case texture_info_quantize_typeType::unorm_pvrtc1_4bpp:	        break;
            case texture_info_quantize_typeType::unorm_pvrtc1_alpha_2bpp:	break;
            case texture_info_quantize_typeType::unorm_pvrtc1_alpha_4bpp:	break;
            case texture_info_quantize_typeType::unorm_pvrtc2_alpha_2bpp:	break;
            case texture_info_quantize_typeType::unorm_pvrtc2_alpha_4bpp:	break;
            case texture_info_quantize_typeType::srgb_pvrtc1_2bpp:      	break;
            case texture_info_quantize_typeType::srgb_pvrtc1_4bpp:      	break;
            case texture_info_quantize_typeType::srgb_pvrtc1_alpha_2bpp:	break;
            case texture_info_quantize_typeType::srgb_pvrtc1_alpha_4bpp:	break;
            case texture_info_quantize_typeType::srgb_pvrtc2_alpha_2bpp:	break;
            case texture_info_quantize_typeType::srgb_pvrtc2_alpha_4bpp:	break;
            default:												assert(false);						    											break;
        }
    }

private:
    texture_info_quantize_typeType	quantize_type_;
    int								colorDataStride_;
    Byte *					        dstByteColor_;
    float *					        dstFloatColor_;
    const unsigned char *			src_;
    int								lineSize_;
    bool							isHintNormal_;
};

ref class ConvertToStreamCompress_Decode
{
public:
    ConvertToStreamCompress_Decode(texture_info_quantize_typeType quantize_type, int width, int stride, int blockSize, unsigned char *dst, const unsigned char *src, bool isHintNormal) :
        quantize_type_(quantize_type),
        width_(width),
        stride_(stride),
        blockSize_(blockSize),
        dst_(dst),
        src_(src),
        isHintNormal_(isHintNormal){}

    void Execute(int y)
    {
        const unsigned char *src = src_ + blockSize_ * y * width_ / blockWidth;

        y *= blockHeight;

        for(int x = 0;x < width_;x += blockWidth)
        {
            switch(quantize_type_)
            {
                case texture_info_quantize_typeType::unorm_bc1:	TexUtils::Decode_unorm_bc1(dst_, src, x, y, stride_);					break;
                case texture_info_quantize_typeType::unorm_bc2:	TexUtils::Decode_unorm_bc2(dst_, src, x, y, stride_);					break;
                case texture_info_quantize_typeType::unorm_bc3:	TexUtils::Decode_unorm_bc3(dst_, src, x, y, stride_);					break;
                case texture_info_quantize_typeType::unorm_bc4:	TexUtils::Decode_unorm_bc4(dst_, src, x, y, stride_);					break;
                case texture_info_quantize_typeType::snorm_bc4:	TexUtils::Decode_snorm_bc4(dst_, src, x, y, stride_);					break;
                case texture_info_quantize_typeType::unorm_bc5:	TexUtils::Decode_unorm_bc5(dst_, src, x, y, stride_);					break;
                case texture_info_quantize_typeType::snorm_bc5:	TexUtils::Decode_snorm_bc5(dst_, src, x, y, stride_, isHintNormal_);	break;
                case texture_info_quantize_typeType::srgb_bc1:	TexUtils::Decode_unorm_bc1(dst_, src, x, y, stride_);					break;
                case texture_info_quantize_typeType::srgb_bc2:	TexUtils::Decode_unorm_bc2(dst_, src, x, y, stride_);					break;
                case texture_info_quantize_typeType::srgb_bc3:	TexUtils::Decode_unorm_bc3(dst_, src, x, y, stride_);					break;
                default:										assert(false);															break;
            }

            src += blockSize_;
        }
    }

private:
    texture_info_quantize_typeType	quantize_type_;
    int								width_;
    int								stride_;
    int								blockSize_;
    unsigned char *					dst_;
    const unsigned char *			src_;
    bool							isHintNormal_;
};

ref class ConvertToStreamCompress_Transfer
{
public:
    ConvertToStreamCompress_Transfer(int colorDataStride, Byte *dstByteColor, const Byte *srcByteColor, int lineSize, int alignLineSize) :
        colorDataStride_(colorDataStride),
        dstByteColor_(dstByteColor),
        srcByteColor_(srcByteColor),
        lineSize_(lineSize),
        alignLineSize_(alignLineSize),
        srcStride_(std::max(alignLineSize, blockWidth * sizeofRGBA)){}

    void Execute(int y)
    {
        const unsigned char *src = srcByteColor_ + srcStride_ * y;
        Byte *dstByteColor = dstByteColor_ + colorDataStride_ * y;

        TexUtils::Decode_unorm_8_8_8_8(dstByteColor, src, lineSize_);
    }

private:
    int						colorDataStride_;
    Byte *			        dstByteColor_;
    const Byte *	        srcByteColor_;
    int						lineSize_;
    int						alignLineSize_;
    int						srcStride_;
};


}	// namespace

Converter::Converter() :
    isInitialized_(false),
    isDllLoaded_(false),
    isGammaCorrection_(true)
{
    ;
}


//-----------------------------------------------------------------------------
/// DLLを読み込みます。
/// @param[in] basePath DLLのベースパス
//-----------------------------------------------------------------------------
bool Converter::LoadDll(array<String^>^ basePath)
{
    TCHAR szTargetPath[_MAX_DIR + _MAX_FNAME + _MAX_EXT];
    _tcscpy_s(szTargetPath, _T(""));

    static const TCHAR* DLL_NAME = _T("NW4F_g3dtexcvtr_texUtils.dll");
    static const TCHAR* DLL_NAME2 = _T("TextureConverterTexUtils.dll");
    const TCHAR* pDLLNames[] = { DLL_NAME2, DLL_NAME };
    int numDLLNames = sizeof(pDLLNames) / sizeof(TCHAR*);
    bool exists = false;

    // ベースパスが指定されているとき、それを使用する
    if (basePath != nullptr)
    {
        for (int i = 0; i < basePath->Length; ++i)
        {
            array<Char>^ warr = basePath[i]->ToCharArray();
            pin_ptr<Char> wptr = &warr[0];

            // DLLがあるかチェック
            for (int j = 0; j < numDLLNames; j++)
            {
                _tcscpy_s(szTargetPath, wptr);
                _tcscat_s(szTargetPath, pDLLNames[j]);
                exists = PathFileExists(szTargetPath) == TRUE;
                if (exists)
                {
                    break;
                }
            }

            if (exists)
            {
                break;
            }
        }
    }

    // DLLがあればロード
    if (exists)
    {
         hDLL = LoadLibrary(szTargetPath);
    }

    return true;
}

//-----------------------------------------------------------------------------
/// 初期化処理を行います。
/// @param[in] basePath DLLのベースパス
//-----------------------------------------------------------------------------
bool Converter::Initialize(array<String^>^ basePath)
{
    // 初期化済みならなにもしない
    if (isInitialized_)
    {
        return true;
    }

    // DLLを読み込む
    LoadDll(basePath);

    if (hDLL != NULL)
    {
        *(void **)&DLL_TC2Initialize			= (void *)GetProcAddress(hDLL, "TC2Initialize");
        *(void **)&DLL_TC2Destroy				= (void *)GetProcAddress(hDLL, "TC2Destroy");
        *(void **)&DLL_TC2GenerateMipLevels		= (void *)GetProcAddress(hDLL, "TC2GenerateMipLevels");
        *(void **)&DLL_TC2ConvertSurfaceFormat	= (void *)GetProcAddress(hDLL, "TC2ConvertSurfaceFormat");
        *(void **)&DLL_TC2ConvertTiling			= (void *)GetProcAddress(hDLL, "TC2ConvertTiling");
        *(void **)&DLL_TC2DestroyGX2Surface		= (void *)GetProcAddress(hDLL, "TC2DestroyGX2Surface");

        // TexUtils を初期化します。
        {
            TC2Config texConfig;
            memset(&texConfig, 0, sizeof(texConfig));
            texConfig.gbTilingConfig = 0;
            texConfig.gpu            = GPU_Cafe;
            DLL_TC2Initialize(&texConfig);
        }
        isDllLoaded_ = true;
    }


    InitializeFormatInfos();

    // 初期化済みに
    isInitialized_ = true;

    return isDllLoaded_;
}

void Converter::Destroy()
{
    // 未初期化ならｍなにもしない
    if (isInitialized_ == false)
    {
        return;
    }

    // 未初期化に
    isInitialized_ = false;
    isDllLoaded_ = false;

    if (hDLL != NULL)
    {
        DLL_TC2Destroy();
        FreeLibrary(hDLL);
        hDLL = NULL;
    }
}

array<Bitmap ^>^ Converter::ConvertTo1d2dBitmap(textureType ^texture, List<G3dStream ^> ^streams)
{
    const unsigned char *gammaCorrectTable[4];
    MakeGammaCorrectTable(isGammaCorrection_, texture, gammaCorrectTable);

    RSurface surface("src", false);
    MakeAllSourceSurface(surface, texture, streams, false);

    RSurface detiledSurface("dedetiled", true);
    bool isOk = DLL_TC2ConvertTiling(&surface.m_Surface, GX2_TILE_MODE_LINEAR_SPECIAL, 0, &detiledSurface.m_Surface);
    assert(isOk);

    switch(texture->texture_info->quantize_type)
    {
        case texture_info_quantize_typeType::unorm_8_8_8_8:
        case texture_info_quantize_typeType::uint_8_8_8_8:
        case texture_info_quantize_typeType::unorm_8:
        case texture_info_quantize_typeType::uint_8:
        case texture_info_quantize_typeType::snorm_8:
        case texture_info_quantize_typeType::sint_8:
        case texture_info_quantize_typeType::unorm_4_4:
        case texture_info_quantize_typeType::unorm_8_8:
        case texture_info_quantize_typeType::uint_8_8:
        case texture_info_quantize_typeType::snorm_8_8:
        case texture_info_quantize_typeType::sint_8_8:
        case texture_info_quantize_typeType::unorm_5_6_5:
        case texture_info_quantize_typeType::unorm_5_5_5_1:
        case texture_info_quantize_typeType::unorm_4_4_4_4:
        case texture_info_quantize_typeType::snorm_8_8_8_8:
        case texture_info_quantize_typeType::sint_8_8_8_8:
        case texture_info_quantize_typeType::srgb_8_8_8_8:
        case texture_info_quantize_typeType::float_16:
        case texture_info_quantize_typeType::float_32:
        case texture_info_quantize_typeType::float_16_16:
        case texture_info_quantize_typeType::float_32_32:
        case texture_info_quantize_typeType::float_11_11_10:
        case texture_info_quantize_typeType::float_16_16_16_16:
        case texture_info_quantize_typeType::float_32_32_32_32:
        {
            return ConvertToBitmapLinear(texture, detiledSurface, 0, gammaCorrectTable);
        }

        case texture_info_quantize_typeType::unorm_bc1:
        case texture_info_quantize_typeType::unorm_bc2:
        case texture_info_quantize_typeType::unorm_bc3:
        case texture_info_quantize_typeType::unorm_bc4:
        case texture_info_quantize_typeType::snorm_bc4:
        case texture_info_quantize_typeType::unorm_bc5:
        case texture_info_quantize_typeType::snorm_bc5:
        case texture_info_quantize_typeType::srgb_bc1:
        case texture_info_quantize_typeType::srgb_bc2:
        case texture_info_quantize_typeType::srgb_bc3:
        {
            return ConvertToBitmapCompress(texture, detiledSurface, 0, gammaCorrectTable);
        }
    }

    assert(false);
    return nullptr;
}

array<array<Bitmap ^> ^>^ Converter::ConvertTo3dBitmap(textureType ^texture, List<G3dStream ^> ^streams)
{
    const unsigned char *gammaCorrectTable[4];
    MakeGammaCorrectTable(isGammaCorrection_, texture, gammaCorrectTable);

    // 最終出力を作るまでのワーク。違いは List<>
    array<List<Bitmap ^> ^> ^bitmapsWork = gcnew array<List<Bitmap ^> ^>(texture->texture_info->depth);
    {
        for(int i = 0;i != texture->texture_info->depth;++ i)
        {
            bitmapsWork[i] = gcnew List<Bitmap ^>();
        }
    }
    {
        RSurface surface("src", false);
        MakeAllSourceSurface(surface, texture, streams, false);

        RSurface detiledSurface("dedetiled", true);
        bool isOk = DLL_TC2ConvertTiling(&surface.m_Surface, GX2_TILE_MODE_LINEAR_SPECIAL, 0, &detiledSurface.m_Surface);
        assert(isOk);

        for(int i = 0;i != texture->texture_info->mip_level;++ i)
        {
            switch(texture->texture_info->quantize_type)
            {
                case texture_info_quantize_typeType::unorm_8_8_8_8:
                case texture_info_quantize_typeType::uint_8_8_8_8:
                case texture_info_quantize_typeType::unorm_8:
                case texture_info_quantize_typeType::uint_8:
                case texture_info_quantize_typeType::snorm_8:
                case texture_info_quantize_typeType::sint_8:
                case texture_info_quantize_typeType::unorm_4_4:
                case texture_info_quantize_typeType::unorm_8_8:
                case texture_info_quantize_typeType::uint_8_8:
                case texture_info_quantize_typeType::snorm_8_8:
                case texture_info_quantize_typeType::sint_8_8:
                case texture_info_quantize_typeType::unorm_5_6_5:
                case texture_info_quantize_typeType::unorm_5_5_5_1:
                case texture_info_quantize_typeType::unorm_4_4_4_4:
                case texture_info_quantize_typeType::snorm_8_8_8_8:
                case texture_info_quantize_typeType::sint_8_8_8_8:
                case texture_info_quantize_typeType::srgb_8_8_8_8:
                case texture_info_quantize_typeType::float_16:
                case texture_info_quantize_typeType::float_32:
                case texture_info_quantize_typeType::float_16_16:
                case texture_info_quantize_typeType::float_32_32:
                case texture_info_quantize_typeType::float_11_11_10:
                case texture_info_quantize_typeType::float_16_16_16_16:
                case texture_info_quantize_typeType::float_32_32_32_32:
                {
                    // 転地して保存する
                    int j = 0;
                    for each(Bitmap ^bitmap in ConvertToBitmapLinear3d(texture, detiledSurface, i, gammaCorrectTable))
                    {
                        bitmapsWork[j ++]->Add(bitmap);
                    }

                    break;
                }

                case texture_info_quantize_typeType::unorm_bc1:
                case texture_info_quantize_typeType::unorm_bc2:
                case texture_info_quantize_typeType::unorm_bc3:
                case texture_info_quantize_typeType::unorm_bc4:
                case texture_info_quantize_typeType::snorm_bc4:
                case texture_info_quantize_typeType::unorm_bc5:
                case texture_info_quantize_typeType::snorm_bc5:
                case texture_info_quantize_typeType::srgb_bc1:
                case texture_info_quantize_typeType::srgb_bc2:
                case texture_info_quantize_typeType::srgb_bc3:
                {
                    // 転置して保存する
                    int j = 0;
                    for each(Bitmap ^bitmap in ConvertToBitmapCompress3d(texture, detiledSurface, i, gammaCorrectTable))
                    {
                        bitmapsWork[j ++]->Add(bitmap);
                    }

                    break;
                }
            }
        }
    }

    // Listから配列に変換する
    array<array<Bitmap ^> ^> ^bitmaps = gcnew array<array<Bitmap ^> ^>(texture->texture_info->depth);
    {
        for(int i = 0;i != texture->texture_info->depth;++ i)
        {
            bitmaps[i] = bitmapsWork[i]->ToArray();
        }
    }

    return bitmaps;
}

array<array<Bitmap ^> ^>^ Converter::ConvertToCubeBitmap(textureType ^texture, List<G3dStream ^> ^streams)
{
    return ConvertToCubeBitmapInternal(texture, streams, false);
}

array<array<Bitmap ^> ^>^ Converter::ConvertTo1dArrayBitmap(textureType ^texture, List<G3dStream ^> ^streams)
{
    return ConvertToCubeBitmapInternal(texture, streams, false);
}

array<array<Bitmap ^> ^>^ Converter::ConvertTo2dArrayBitmap(textureType ^texture, List<G3dStream ^> ^streams)
{
    return ConvertToCubeBitmapInternal(texture, streams, false);
}

array<array<Bitmap ^> ^>^ Converter::ConvertToCubeArrayBitmap(textureType ^texture, List<G3dStream ^> ^streams)
{
    return ConvertToCubeBitmapInternal(texture, streams, true);
}

array<array<Bitmap ^> ^>^ Converter::ConvertToCubeBitmapInternal(textureType ^texture, List<G3dStream ^> ^streams, bool isCubeArray)
{
    const unsigned char *gammaCorrectTable[4];
    MakeGammaCorrectTable(isGammaCorrection_, texture, gammaCorrectTable);

    array<array<Bitmap ^> ^> ^bitmaps = gcnew array<array<Bitmap ^> ^>(texture->texture_info->depth);
    {
        RSurface surface("src", false);
        MakeAllSourceSurface(surface, texture, streams, isCubeArray);

        RSurface detiledSurface("dedetiled", true);
        bool isOk = DLL_TC2ConvertTiling(&surface.m_Surface, GX2_TILE_MODE_LINEAR_SPECIAL, 0, &detiledSurface.m_Surface);
        assert(isOk);

        for(int i = 0;i != texture->texture_info->depth;++ i)
        {
            switch(texture->texture_info->quantize_type)
            {
                case texture_info_quantize_typeType::unorm_8_8_8_8:
                case texture_info_quantize_typeType::uint_8_8_8_8:
                case texture_info_quantize_typeType::unorm_8:
                case texture_info_quantize_typeType::uint_8:
                case texture_info_quantize_typeType::snorm_8:
                case texture_info_quantize_typeType::sint_8:
                case texture_info_quantize_typeType::unorm_4_4:
                case texture_info_quantize_typeType::unorm_8_8:
                case texture_info_quantize_typeType::uint_8_8:
                case texture_info_quantize_typeType::snorm_8_8:
                case texture_info_quantize_typeType::sint_8_8:
                case texture_info_quantize_typeType::unorm_5_6_5:
                case texture_info_quantize_typeType::unorm_5_5_5_1:
                case texture_info_quantize_typeType::unorm_4_4_4_4:
                case texture_info_quantize_typeType::snorm_8_8_8_8:
                case texture_info_quantize_typeType::sint_8_8_8_8:
                case texture_info_quantize_typeType::srgb_8_8_8_8:
                case texture_info_quantize_typeType::float_16:
                case texture_info_quantize_typeType::float_32:
                case texture_info_quantize_typeType::float_16_16:
                case texture_info_quantize_typeType::float_32_32:
                case texture_info_quantize_typeType::float_11_11_10:
                case texture_info_quantize_typeType::float_16_16_16_16:
                case texture_info_quantize_typeType::float_32_32_32_32:
                {
                    bitmaps[i] = ConvertToBitmapLinear(texture, detiledSurface, i, gammaCorrectTable);
                    break;
                }

                case texture_info_quantize_typeType::unorm_bc1:
                case texture_info_quantize_typeType::unorm_bc2:
                case texture_info_quantize_typeType::unorm_bc3:
                case texture_info_quantize_typeType::unorm_bc4:
                case texture_info_quantize_typeType::snorm_bc4:
                case texture_info_quantize_typeType::unorm_bc5:
                case texture_info_quantize_typeType::snorm_bc5:
                case texture_info_quantize_typeType::srgb_bc1:
                case texture_info_quantize_typeType::srgb_bc2:
                case texture_info_quantize_typeType::srgb_bc3:
                {
                    bitmaps[i] = ConvertToBitmapCompress(texture, detiledSurface, i, gammaCorrectTable);
                    break;
                }
            }
        }
    }
    return bitmaps;
}

array<Bitmap ^>^ Converter::ConvertToBitmapLinear(
    textureType ^texture,
    RSurface &detiledSurface,
    int depthLevel,
    const unsigned char *gammaCorrectTable[4]
)
{
    array<Bitmap ^> ^bitmaps = gcnew array<Bitmap ^>(texture->texture_info->mip_level);
    {
        bool	isHintNormal	= texture->texture_info->hint == "normal";
        float	bpp				= GetBytesPerPixel(texture->texture_info->quantize_type);

        for(int mipmapLevel = 0;mipmapLevel != texture->texture_info->mip_level;++ mipmapLevel)
        {
            bitmaps[mipmapLevel] =
                ConvertToBitmapLinearInternal(
                    texture,
                    detiledSurface,
                    depthLevel,
                    gammaCorrectTable,
                    isHintNormal,
                    bpp,
                    mipmapLevel
                );
        }
    }
    return bitmaps;
}

array<Bitmap ^>^ Converter::ConvertToBitmapCompress(
    textureType ^texture,
    RSurface &detiledSurface,
    int depthLevel,
    const unsigned char *gammaCorrectTable[4]
)
{
    array<Bitmap ^> ^bitmaps = gcnew array<Bitmap ^>(texture->texture_info->mip_level);
    {
        bool	isHintNormal	= texture->texture_info->hint == "normal";
        float	bpp				= GetBytesPerPixel(texture->texture_info->quantize_type);

        for(int mipmapLevel = 0;mipmapLevel != texture->texture_info->mip_level;++ mipmapLevel)
        {
            bitmaps[mipmapLevel] =
                ConvertToBitmapCompressInternal(
                texture,
                detiledSurface,
                depthLevel,
                gammaCorrectTable,
                isHintNormal,
                bpp,
                mipmapLevel
            );
        }
    }
    return bitmaps;
}

array<Bitmap ^>^ Converter::ConvertToBitmapLinear3d(
    textureType ^texture,
    RSurface &detiledSurface,
    int mipmapLevel,
    const unsigned char *gammaCorrectTable[4]
)
{
    int depthCount = std::max(texture->texture_info->depth >> mipmapLevel, 1);

    array<Bitmap ^> ^bitmaps = gcnew array<Bitmap ^>(depthCount);
    {
        bool	isHintNormal	= texture->texture_info->hint == "normal";
        float	bpp				= GetBytesPerPixel(texture->texture_info->quantize_type);

        for(int depthLevel = 0;depthLevel != depthCount;++ depthLevel)
        {
            bitmaps[depthLevel] =
                ConvertToBitmapLinearInternal(
                    texture,
                    detiledSurface,
                    depthLevel,
                    gammaCorrectTable,
                    isHintNormal,
                    bpp,
                    mipmapLevel
                );
        }
    }
    return bitmaps;
}

array<Bitmap ^>^ Converter::ConvertToBitmapCompress3d(
    textureType ^texture,
    RSurface &detiledSurface,
    int mipmapLevel,
    const unsigned char *gammaCorrectTable[4]
){
    int depthCount = std::max(texture->texture_info->depth >> mipmapLevel, 1);

    array<Bitmap ^> ^bitmaps = gcnew array<Bitmap ^>(depthCount);
    {
        bool	isHintNormal	= texture->texture_info->hint == "normal";
        float	bpp				= GetBytesPerPixel(texture->texture_info->quantize_type);

        for(int depthLevel = 0;depthLevel != depthCount;++ depthLevel)
        {
            bitmaps[depthLevel] =
                ConvertToBitmapCompressInternal(
                    texture,
                    detiledSurface,
                    depthLevel,
                    gammaCorrectTable,
                    isHintNormal,
                    bpp,
                    mipmapLevel
                );
        }
    }
    return bitmaps;
}

Bitmap^ Converter::ConvertToBitmapLinearInternal(
    textureType ^texture,
    RSurface &detiledSurface,
    int depthLevel,
    const unsigned char *gammaCorrectTable[4],
    bool isHintNormal,
    float bpp,
    int mipmapLevel
)
{
    int	width	= MipmapSize(texture->texture_info->width,  mipmapLevel);
    int	height	= MipmapSize(texture->texture_info->height, mipmapLevel);

    Drawing::Rectangle	rect		= Drawing::Rectangle(0, 0, width, height);

    Bitmap				^detiledColorBitmap	= gcnew Bitmap(width, height, PixelFormat::Format32bppArgb);
    BitmapData			^colorData			= detiledColorBitmap->LockBits(rect, ImageLockMode::WriteOnly, PixelFormat::Format32bppArgb);
    unsigned char		*dstColor			= static_cast<unsigned char *>(colorData->Scan0.ToPointer());
    const unsigned char	*src				= static_cast<const unsigned char *>(GetLevelImagePtr(detiledSurface.m_Surface, mipmapLevel, depthLevel));
    {
        int lineSize = (int)(width * bpp);

        Parallel::For(
            0,
            height,
            gcnew Action<int>(
                gcnew ConvertToBitmapLinear_Decode(
                    texture->texture_info->quantize_type,
                    colorData->Stride,
                    dstColor,
                    src,
                    lineSize,
                    gammaCorrectTable,
                    isHintNormal
                ),
                &ConvertToBitmapLinear_Decode::Execute
            )
        );
    }
    detiledColorBitmap->UnlockBits(colorData);
    delete colorData;

    return detiledColorBitmap;
}

Bitmap^ Converter::ConvertToBitmapCompressInternal(
    textureType ^texture,
    RSurface &detiledSurface,
    int depthLevel,
    const unsigned char *gammaCorrectTable[4],
    bool isHintNormal,
    float bpp,
    int mipmapLevel
)
{
    int	width		= MipmapSize(texture->texture_info->width,  mipmapLevel);
    int	height		= MipmapSize(texture->texture_info->height, mipmapLevel);
    int	blockSize	= int(blockWidth * blockHeight * bpp);
    int	alignWidth	= (width  + blockWidthMask ) & ~blockWidthMask;
    int	alignHeight	= (height + blockHeightMask) & ~blockHeightMask;
    int	alignStride	= int(alignWidth * sizeofRGBA);

    Drawing::Rectangle	rect		= Drawing::Rectangle(0, 0, width, height);

    Bitmap				^detiledColorBitmap	= gcnew Bitmap(width, height, PixelFormat::Format32bppArgb);
    BitmapData			^colorData			= detiledColorBitmap->LockBits(rect, ImageLockMode::WriteOnly, PixelFormat::Format32bppArgb);
    unsigned char		*dstColor			= static_cast<unsigned char *>(colorData->Scan0.ToPointer());
    const unsigned char	*src				= static_cast<const unsigned char *>(GetLevelImagePtr(detiledSurface.m_Surface, mipmapLevel, depthLevel));
    {
        std::vector<unsigned char> rgba(
            std::max(alignWidth,  blockWidth ) *
            std::max(alignHeight, blockHeight) *
            sizeofRGBA
        );
        {
            unsigned char *dst	= &rgba[0];

            Parallel::For(
                0,
                alignHeight / blockHeight,
                gcnew Action<int>(
                    gcnew ConvertToBitmapCompress_Decode(
                        texture->texture_info->quantize_type,
                        alignWidth,
                        alignStride,
                        blockSize,
                        dst,
                        src,
                        isHintNormal
                    ),
                    &ConvertToBitmapCompress_Decode::Execute
                )
            );
        }

        {
            const unsigned char *src	= &rgba[0];

            int lineSize		= width * sizeofRGBA;
            int alignLineSize	= alignWidth * sizeofRGBA;

            Parallel::For(
                0,
                height,
                gcnew Action<int>(
                    gcnew ConvertToBitmapCompress_Transfer(
                        colorData->Stride,
                        dstColor,
                        src,
                        lineSize,
                        alignLineSize,
                        gammaCorrectTable
                    ),
                    &ConvertToBitmapCompress_Transfer::Execute
                )
            );
        }
    }
    detiledColorBitmap->UnlockBits(colorData);
    delete colorData;

    return detiledColorBitmap;
}

array<TextureData ^>^ Converter::ConvertTo1d2dStream(textureType ^texture, List<G3dStream ^> ^streams)
{
    switch (texture->texture_info->quantize_type)
    {
        case texture_info_quantize_typeType::unorm_pvrtc1_2bpp:
        case texture_info_quantize_typeType::unorm_pvrtc1_4bpp:
        case texture_info_quantize_typeType::unorm_pvrtc1_alpha_2bpp:
        case texture_info_quantize_typeType::unorm_pvrtc1_alpha_4bpp:
        case texture_info_quantize_typeType::unorm_pvrtc2_alpha_2bpp:
        case texture_info_quantize_typeType::unorm_pvrtc2_alpha_4bpp:
        case texture_info_quantize_typeType::srgb_pvrtc1_2bpp:
        case texture_info_quantize_typeType::srgb_pvrtc1_4bpp:
        case texture_info_quantize_typeType::srgb_pvrtc1_alpha_2bpp:
        case texture_info_quantize_typeType::srgb_pvrtc1_alpha_4bpp:
        case texture_info_quantize_typeType::srgb_pvrtc2_alpha_2bpp:
        case texture_info_quantize_typeType::srgb_pvrtc2_alpha_4bpp:
        {
            return ConvertToStreamPvrCompress(texture, streams, 0);
        }
    }

    RSurface surface("src", false);
    MakeAllSourceSurface(surface, texture, streams, false);

    RSurface detiledSurface("dedetiled", true);
    if ( DLL_TC2ConvertTiling == NULL )
    {
        return nullptr;
    }

    bool isOk = DLL_TC2ConvertTiling(&surface.m_Surface, GX2_TILE_MODE_LINEAR_SPECIAL, 0, &detiledSurface.m_Surface);
    assert(isOk);

    switch(texture->texture_info->quantize_type)
    {
        case texture_info_quantize_typeType::unorm_8_8_8_8:
        case texture_info_quantize_typeType::unorm_8:
        case texture_info_quantize_typeType::snorm_8:
        case texture_info_quantize_typeType::unorm_4_4:
        case texture_info_quantize_typeType::unorm_8_8:
        case texture_info_quantize_typeType::snorm_8_8:
        case texture_info_quantize_typeType::unorm_5_6_5:
        case texture_info_quantize_typeType::unorm_5_5_5_1:
        case texture_info_quantize_typeType::unorm_4_4_4_4:
        case texture_info_quantize_typeType::snorm_8_8_8_8:
        case texture_info_quantize_typeType::srgb_8_8_8_8:
        case texture_info_quantize_typeType::float_16:
        case texture_info_quantize_typeType::float_32:
        case texture_info_quantize_typeType::float_16_16:
        case texture_info_quantize_typeType::float_32_32:
        case texture_info_quantize_typeType::float_11_11_10:
        case texture_info_quantize_typeType::float_16_16_16_16:
        case texture_info_quantize_typeType::float_32_32_32_32:
        {
            return ConvertToStreamLinear(texture, detiledSurface, 0);
        }

        case texture_info_quantize_typeType::unorm_bc1:
        case texture_info_quantize_typeType::unorm_bc2:
        case texture_info_quantize_typeType::unorm_bc3:
        case texture_info_quantize_typeType::unorm_bc4:
        case texture_info_quantize_typeType::snorm_bc4:
        case texture_info_quantize_typeType::unorm_bc5:
        case texture_info_quantize_typeType::snorm_bc5:
        case texture_info_quantize_typeType::srgb_bc1:
        case texture_info_quantize_typeType::srgb_bc2:
        case texture_info_quantize_typeType::srgb_bc3:
        {
            return ConvertToStreamCompress(texture, detiledSurface, 0);
        }
    }

    assert(false);
    return nullptr;
}

array<array<TextureData ^> ^>^ Converter::ConvertTo3dStream(textureType ^texture, List<G3dStream ^> ^streams)
{
    return nullptr;
}

array<array<TextureData ^> ^>^ Converter::ConvertToCubeStream(textureType ^texture, List<G3dStream ^> ^streams)
{
    return nullptr;
}

array<array<TextureData ^> ^>^ Converter::ConvertTo1dArrayStream(textureType ^texture, List<G3dStream ^> ^streams)
{
    return nullptr;
}

array<array<TextureData ^> ^>^ Converter::ConvertTo2dArrayStream(textureType ^texture, List<G3dStream ^> ^streams)
{
    return nullptr;
}

array<array<TextureData ^> ^>^ Converter::ConvertToCubeArrayStream(textureType ^texture, List<G3dStream ^> ^streams)
{
    return nullptr;
}

array<array<TextureData ^> ^>^ Converter::ConvertToCubeStreamInternal(textureType ^texture, List<G3dStream ^> ^streams, bool isCubeArray)
{
    array<array<TextureData ^> ^> ^outputTextureData = gcnew array<array<TextureData ^> ^>(texture->texture_info->depth);
    {
        RSurface surface("src", false);
        MakeAllSourceSurface(surface, texture, streams, isCubeArray);

        RSurface detiledSurface("dedetiled", true);
        bool isOk = DLL_TC2ConvertTiling(&surface.m_Surface, GX2_TILE_MODE_LINEAR_SPECIAL, 0, &detiledSurface.m_Surface);
        assert(isOk);

        for(int i = 0;i != texture->texture_info->depth;++ i)
        {
            switch(texture->texture_info->quantize_type)
            {
                case texture_info_quantize_typeType::unorm_8_8_8_8:
                case texture_info_quantize_typeType::unorm_8:
                case texture_info_quantize_typeType::snorm_8:
                case texture_info_quantize_typeType::unorm_4_4:
                case texture_info_quantize_typeType::unorm_8_8:
                case texture_info_quantize_typeType::snorm_8_8:
                case texture_info_quantize_typeType::unorm_5_6_5:
                case texture_info_quantize_typeType::unorm_5_5_5_1:
                case texture_info_quantize_typeType::unorm_4_4_4_4:
                case texture_info_quantize_typeType::snorm_8_8_8_8:
                case texture_info_quantize_typeType::srgb_8_8_8_8:
                case texture_info_quantize_typeType::float_16:
                case texture_info_quantize_typeType::float_32:
                case texture_info_quantize_typeType::float_16_16:
                case texture_info_quantize_typeType::float_32_32:
                case texture_info_quantize_typeType::float_11_11_10:
                case texture_info_quantize_typeType::float_16_16_16_16:
                case texture_info_quantize_typeType::float_32_32_32_32:
                {
                    outputTextureData[i] = ConvertToStreamLinear(texture, detiledSurface, i);
                    break;
                }

                case texture_info_quantize_typeType::unorm_bc1:
                case texture_info_quantize_typeType::unorm_bc2:
                case texture_info_quantize_typeType::unorm_bc3:
                case texture_info_quantize_typeType::unorm_bc4:
                case texture_info_quantize_typeType::snorm_bc4:
                case texture_info_quantize_typeType::unorm_bc5:
                case texture_info_quantize_typeType::snorm_bc5:
                case texture_info_quantize_typeType::srgb_bc1:
                case texture_info_quantize_typeType::srgb_bc2:
                case texture_info_quantize_typeType::srgb_bc3:
                {
                    outputTextureData[i] = ConvertToStreamCompress(texture, detiledSurface, i);
                    break;
                }
            }
        }
    }
    return outputTextureData;
}

array<TextureData ^>^ Converter::ConvertToStreamLinear(		textureType ^texture, RSurface &detiledSurface, int depthLevel)
{
    array<TextureData ^> ^outputTextureData = gcnew array<TextureData ^>(texture->texture_info->mip_level);
    {
        bool	isHintNormal	= texture->texture_info->hint == "normal";
        float	bpp				= GetBytesPerPixel(texture->texture_info->quantize_type);

        for(int mipmapLevel = 0;mipmapLevel != texture->texture_info->mip_level;++ mipmapLevel)
        {
            outputTextureData[mipmapLevel] =
                ConvertToStreamLinearInternal(
                    texture,
                    detiledSurface,
                    depthLevel,
                    isHintNormal,
                    bpp,
                    mipmapLevel
                );
        }
    }
    return outputTextureData;
}

array<TextureData ^>^ Converter::ConvertToStreamCompress(	textureType ^texture, RSurface &detiledSurface, int depthLevel)
{
    array<TextureData ^> ^outputTextureData = gcnew array<TextureData ^>(texture->texture_info->mip_level);
    {
        bool	isHintNormal	= texture->texture_info->hint == "normal";
        float	bpp				= GetBytesPerPixel(texture->texture_info->quantize_type);

        for(int mipmapLevel = 0;mipmapLevel != texture->texture_info->mip_level;++ mipmapLevel)
        {
            outputTextureData[mipmapLevel] =
                ConvertToStreamCompressInternal(
                texture,
                detiledSurface,
                depthLevel,
                isHintNormal,
                bpp,
                mipmapLevel
            );
        }
    }
    return outputTextureData;
}

array<TextureData ^>^ Converter::ConvertToStreamPvrCompress(textureType ^texture, List<G3dStream ^> ^streams, int depthLevel)
{
    array<TextureData ^> ^outputTextureData = gcnew array<TextureData ^>(texture->texture_info->mip_level);
    {
        // 圧縮状態のバイト列をアンマネージ配列を確保してコピー
        array<byte, 1>^ orgData = streams[texture->texture_info->stream_index]->ByteData->ToArray();
        void *pCompressedData = malloc(orgData->Length);
        Marshal::Copy(orgData, 0, (System::IntPtr)pCompressedData, orgData->Length);

        // 展開後の配列を、構造体を使わず自前でサイズを計算して確保
        PVRTuint32 levelSize =
            texture->texture_info->width * texture->texture_info->height * texture->texture_info->depth * 4;
        PVRTuint32 totalSize = 0;
        for(int i = 0; i < texture->texture_info->mip_level; ++i)
        {
            totalSize += levelSize;
            levelSize /= 4;
        }
        void *pDecompressedData = malloc(totalSize);

        // ミップマップの初期サイズを取得
        PVRTuint32 uiMIPWidth = texture->texture_info->width;
        PVRTuint32 uiMIPHeight = texture->texture_info->height;

        // スライド用の一時ポインタ
        PVRTuint8* pTempDecompData = (PVRTuint8*)pDecompressedData;
        PVRTuint8* pTempCompData = (PVRTuint8*)pCompressedData;

        // フォーマットの変換と、データサイズ計算用の除算値を決定
        int bpp_inv;
        PVRTuint64 PixelFormat;
        switch(texture->texture_info->quantize_type)
        {
            case texture_info_quantize_typeType::unorm_pvrtc1_2bpp:
            case texture_info_quantize_typeType::srgb_pvrtc1_2bpp:
                PixelFormat = ePVRTPF_PVRTCI_2bpp_RGB;
                bpp_inv = 4;
                break;
            case texture_info_quantize_typeType::unorm_pvrtc1_4bpp:
            case texture_info_quantize_typeType::srgb_pvrtc1_4bpp:
                PixelFormat = ePVRTPF_PVRTCI_4bpp_RGB;
                bpp_inv = 2;
                break;
            case texture_info_quantize_typeType::unorm_pvrtc1_alpha_2bpp:
            case texture_info_quantize_typeType::srgb_pvrtc1_alpha_2bpp:
                PixelFormat = ePVRTPF_PVRTCI_2bpp_RGBA;
                bpp_inv = 4;
                break;
            case texture_info_quantize_typeType::unorm_pvrtc1_alpha_4bpp:
            case texture_info_quantize_typeType::srgb_pvrtc1_alpha_4bpp:
                PixelFormat = ePVRTPF_PVRTCI_4bpp_RGBA;
                bpp_inv = 2;
                break;
            case texture_info_quantize_typeType::unorm_pvrtc2_alpha_2bpp:
            case texture_info_quantize_typeType::srgb_pvrtc2_alpha_2bpp:
                PixelFormat = ePVRTPF_PVRTCII_2bpp;
                bpp_inv = 4;
                break;
            case texture_info_quantize_typeType::unorm_pvrtc2_alpha_4bpp:
            case texture_info_quantize_typeType::srgb_pvrtc2_alpha_4bpp:
                PixelFormat = ePVRTPF_PVRTCII_4bpp;
                bpp_inv = 2;
                break;
        }

        // カラースペース値の変換
        EPVRTColourSpace ColourSpace;
        switch(texture->texture_info->quantize_type)
        {
            case texture_info_quantize_typeType::srgb_pvrtc1_2bpp:
            case texture_info_quantize_typeType::srgb_pvrtc1_4bpp:
            case texture_info_quantize_typeType::srgb_pvrtc1_alpha_2bpp:
            case texture_info_quantize_typeType::srgb_pvrtc1_alpha_4bpp:
            case texture_info_quantize_typeType::srgb_pvrtc2_alpha_2bpp:
            case texture_info_quantize_typeType::srgb_pvrtc2_alpha_4bpp:
                ColourSpace = EPVRTColourSpace::ePVRTCSpacesRGB;
                break;
            default:
                ColourSpace = EPVRTColourSpace::ePVRTCSpacelRGB;
        }

        // Check if it's 2bpp.
        bool bIs2bppPVRTC = ColourSpace != ePVRTCSpacesRGB &&
            (PixelFormat == ePVRTPF_PVRTCI_2bpp_RGB || PixelFormat == ePVRTPF_PVRTCI_2bpp_RGBA);

        // Decompress all the MIP levels.
        for (PVRTuint32 uiMIPMap = 0; uiMIPMap < (PVRTuint32)texture->texture_info->mip_level; ++uiMIPMap)
        {
            // Get the face offset. Varies per MIP level.
            PVRTuint32 decompressedFaceOffset = uiMIPWidth * uiMIPHeight * 4;

            // 2bpp/4bppは全チャンネルを含めたbit per pixelだったので、ピクセル数を4か2で割るだけでオフセットが求まる
            PVRTuint32 compressedFaceOffset = uiMIPWidth * uiMIPHeight / bpp_inv;

            // キューブマップは別関数で処理するからここでは常に1枚ずつで良いはず
            for (PVRTuint32 uiFace = 0; uiFace < 1; ++uiFace)
            {
                //Decompress the texture data.
                PVRTDecompressPVRTC(pTempCompData, bIs2bppPVRTC ? 1 : 0, uiMIPWidth, uiMIPHeight, pTempDecompData);

                outputTextureData[uiMIPMap] = gcnew TextureData();
                outputTextureData[uiMIPMap]->QuantizeType  = texture->texture_info->quantize_type;
                outputTextureData[uiMIPMap]->DimensionType = texture->texture_info->dimension;
                outputTextureData[uiMIPMap]->Width         = uiMIPWidth;
                outputTextureData[uiMIPMap]->Height        = uiMIPHeight;
                outputTextureData[uiMIPMap]->Depth         = texture->texture_info->depth;
                outputTextureData[uiMIPMap]->IsFloat       = false;

                // 長さ0のマネージド配列はpinできないので長さ1にしています。
                outputTextureData[uiMIPMap]->ByteImage     = gcnew array<Byte>(uiMIPWidth * uiMIPHeight * 4);
                outputTextureData[uiMIPMap]->FloatImage    = gcnew array<float>(1);

                // RとBを手動スワップ(本当はcompselでいじるべきかもしれない)
                for(PVRTuint32 i = 0; i < decompressedFaceOffset; i += 4)
                {
                    unsigned char swap = pTempDecompData[i];
                    pTempDecompData[i] = pTempDecompData[i + 2];
                    pTempDecompData[i + 2] = swap;
                }

                // アンマネージ配列からマネージ配列へコピー
                Marshal::Copy(
                    (System::IntPtr)pTempDecompData, outputTextureData[uiMIPMap]->ByteImage,
                    0, outputTextureData[uiMIPMap]->ByteImage->Length);

                // Move forward through the pointers.
                pTempDecompData += decompressedFaceOffset;
                pTempCompData += compressedFaceOffset;
            }

            // Work out the current MIP dimensions.
            uiMIPWidth = PVRT_MAX(1, uiMIPWidth >> 1);
            uiMIPHeight = PVRT_MAX(1, uiMIPHeight >> 1);
        }

        // メモリ解放
        free(pCompressedData);
        free(pDecompressedData);
    }

    return outputTextureData;
}

array<TextureData ^>^ Converter::ConvertToStreamLinear3d(	textureType ^texture, RSurface &detiledSurface, int mipmapLevel)
{
    int depthCount = std::max(texture->texture_info->depth >> mipmapLevel, 1);

    array<TextureData ^> ^outputTextureData = gcnew array<TextureData ^>(depthCount);
    {
        bool	isHintNormal	= texture->texture_info->hint == "normal";
        float	bpp				= GetBytesPerPixel(texture->texture_info->quantize_type);

        for(int depthLevel = 0;depthLevel != depthCount;++ depthLevel)
        {
            outputTextureData[depthLevel] =
                ConvertToStreamLinearInternal(
                    texture,
                    detiledSurface,
                    depthLevel,
                    isHintNormal,
                    bpp,
                    mipmapLevel
                );
        }
    }
    return outputTextureData;
}

array<TextureData ^>^ Converter::ConvertToStreamCompress3d(	textureType ^texture, RSurface &detiledSurface, int mipmapLevel)
{
    int depthCount = std::max(texture->texture_info->depth >> mipmapLevel, 1);

    array<TextureData ^> ^outputTextureData = gcnew array<TextureData ^>(depthCount);
    {
        bool	isHintNormal	= texture->texture_info->hint == "normal";
        float	bpp				= GetBytesPerPixel(texture->texture_info->quantize_type);

        for(int depthLevel = 0;depthLevel != depthCount;++ depthLevel)
        {
            outputTextureData[depthLevel] =
                ConvertToStreamCompressInternal(
                    texture,
                    detiledSurface,
                    depthLevel,
                    isHintNormal,
                    bpp,
                    mipmapLevel
                );
        }
    }
    return outputTextureData;
}

TextureData^ Converter::ConvertToStreamLinearInternal(		textureType ^texture, RSurface &detiledSurface, int depthLevel, bool isHintNormal, float bpp, int mipmapLevel)
{
    int	width    = MipmapSize(texture->texture_info->width,  mipmapLevel);
    int	height   = MipmapSize(texture->texture_info->height, mipmapLevel);
    int lineSize = (int)(width * bpp);

    // 出力するテクスチャデータを作る
    TextureData ^outputTextureData = gcnew TextureData();
    bool isFloat = false;
    {
        const FormatInfo &formatInfo = FindFormatInfo(texture->texture_info->quantize_type);

        outputTextureData->QuantizeType  = texture->texture_info->quantize_type;
        outputTextureData->DimensionType = texture->texture_info->dimension;
        outputTextureData->Width         = texture->texture_info->width;
        outputTextureData->Height        = texture->texture_info->height;
        outputTextureData->Depth         = texture->texture_info->depth;
        outputTextureData->IsFloat       = formatInfo.isFloat_;

        // 長さ0のマネージド配列はpinできないので長さ1にしています。
        outputTextureData->ByteImage     = gcnew array<Byte >(formatInfo.isFloat_ == false ? width * height * 4 : 1);
        outputTextureData->FloatImage    = gcnew array<float>(formatInfo.isFloat_ == true  ? width * height * 4 : 1);

        outputTextureData->OriginalImage = nullptr;

        isFloat = formatInfo.isFloat_;
    }

    array<Byte>^    byteImage  = outputTextureData->ByteImage;
    array<float>^   floatImage = outputTextureData->FloatImage;
    pin_ptr<Byte>   dstByteColorPin  = &byteImage[0];
    pin_ptr<float>  dstFloatColorPin = &floatImage[0];
    Byte  *         dstByteColor  = dstByteColorPin;
    float *         dstFloatColor = dstFloatColorPin;

    const unsigned char	*src		    = static_cast<const unsigned char *>(GetLevelImagePtr(detiledSurface.m_Surface, mipmapLevel, depthLevel));
    {
        //int colorDataStride = isFloat ? lineSize / 4 : lineSize;
        int colorDataStride = width * 4;

        Parallel::For(
            0,
            height,
            gcnew Action<int>(
                gcnew ConvertToStreamLinear_Decode(
                    texture->texture_info->quantize_type,
                    colorDataStride,
                    dstByteColor,
                    dstFloatColor,
                    src,
                    lineSize,
                    isHintNormal
                ),
                &ConvertToStreamLinear_Decode::Execute
            )
        );
    }

    // ピンポインタを外す
    dstByteColorPin  = nullptr;
    dstFloatColorPin = nullptr;

    return outputTextureData;
}

TextureData^ Converter::ConvertToStreamCompressInternal(	textureType ^texture, RSurface &detiledSurface, int depthLevel, bool isHintNormal, float bpp, int mipmapLevel)
{
    int	width		= MipmapSize(texture->texture_info->width,  mipmapLevel);
    int	height		= MipmapSize(texture->texture_info->height, mipmapLevel);
    int	blockSize	= int(blockWidth * blockHeight * bpp);
    int	alignWidth	= (width  + blockWidthMask ) & ~blockWidthMask;
    int	alignHeight	= (height + blockHeightMask) & ~blockHeightMask;
    int	alignStride	= int(alignWidth * sizeofRGBA);

    // 出力するテクスチャデータを作る
    TextureData ^outputTextureData = gcnew TextureData();
    {
        const FormatInfo &formatInfo = FindFormatInfo(texture->texture_info->quantize_type);

        outputTextureData->QuantizeType  = texture->texture_info->quantize_type;
        outputTextureData->DimensionType = texture->texture_info->dimension;
        outputTextureData->Width         = texture->texture_info->width;
        outputTextureData->Height        = texture->texture_info->height;
        outputTextureData->Depth         = texture->texture_info->depth;
        outputTextureData->IsFloat       = formatInfo.isFloat_;

        int pixelCount = alignWidth * alignHeight * 4;

        // 長さ0のマネージド配列はpinできないので長さ1にしています。
        outputTextureData->ByteImage     = gcnew array<Byte >(formatInfo.isFloat_ == false ? pixelCount : 1);
        outputTextureData->FloatImage    = gcnew array<float>(formatInfo.isFloat_ == true  ? pixelCount : 1);
    }

    array<Byte>^    byteImage  = outputTextureData->ByteImage;
    array<float>^   floatImage = outputTextureData->FloatImage;
    pin_ptr<Byte>   dstByteColorPin  = &byteImage[0];
    pin_ptr<float>  dstFloatColorPin = &floatImage[0];
    Byte  *         dstByteColor  = dstByteColorPin;
    float *         dstFloatColor = dstFloatColorPin;

    const unsigned char	*src				= static_cast<const unsigned char *>(GetLevelImagePtr(detiledSurface.m_Surface, mipmapLevel, depthLevel));
    {
        std::vector<unsigned char> rgba(
            std::max(alignWidth,  blockWidth ) *
            std::max(alignHeight, blockHeight) *
            sizeofRGBA
        );
        {
            unsigned char *dst	= &rgba[0];

            Parallel::For(
                0,
                alignHeight / blockHeight,
                gcnew Action<int>(
                    gcnew ConvertToStreamCompress_Decode(
                        texture->texture_info->quantize_type,
                        alignWidth,
                        alignStride,
                        blockSize,
                        dst,
                        src,
                        isHintNormal
                    ),
                    &ConvertToStreamCompress_Decode::Execute
                )
            );
        }

        {
            const unsigned char *src	= &rgba[0];

            int lineSize		= width * sizeofRGBA;
            int alignLineSize	= alignWidth * sizeofRGBA;

            Parallel::For(
                0,
                height,
                gcnew Action<int>(
                    gcnew ConvertToStreamCompress_Transfer(
                        alignLineSize,      //colorData->Stride,                            /// 要調査
                        dstByteColor,
                        src,
                        lineSize,
                        alignLineSize
                    ),
                    &ConvertToStreamCompress_Transfer::Execute
                )
            );
        }
    }

    // ピンポインタを外す
    dstByteColorPin  = nullptr;
    dstFloatColorPin = nullptr;

    return outputTextureData;
}

//------------------------------------------------------------------------------
/// モジュールのディレクトリ名を取得します。
/// @param[in]  hModule   モジュールハンドル
/// @param[out] lpDirName ディレクトリ名
/// @param[in]  nSize     lpDirNameのサイズ
/// @return ディレクトリ名の文字数を返します。
//------------------------------------------------------------------------------
DWORD GetModuleDirName(HMODULE hModule, TCHAR* lpDirName, DWORD nSize)
{
    TCHAR fullPath[MAX_PATH];

    // モジュールのフルパスを取得
    DWORD charRead = GetModuleFileName(hModule, fullPath, MAX_PATH);
    if (charRead == 0 || charRead > MAX_PATH)
    {
        return 0;
    }

    TCHAR drive[MAX_PATH];
    TCHAR dir[MAX_PATH];

    // フルパスをドライブ名、ディレクトリ名に分解
    _tsplitpath_s(fullPath, drive, MAX_PATH, dir, MAX_PATH, NULL, 0, NULL, 0);

    // ドライブ名, ディレクトリ名を連結
    _tmakepath_s(lpDirName, nSize, drive, dir, NULL, NULL);

    return (DWORD)_tcslen(lpDirName);
}

} //namespace cafe
} //namespace texutils
} //namespace g3d
} //namespace nw
